package jp.ac.nii.icpc2010.playfield;

import java.util.Vector;

import jp.ac.nii.icpc2010.ITron;

/** 
 * Read-only interface to the play field.
 * <P>
 * Playfields use a 2D coordinate system. X-values increase from left to right, and Y-values increase from top to bottom.
 * <P>
 * 
 * <pre>
 * 0----------------->x
 * |        ^
 * |        -y 
 * |    <-x    +x>
 * |        +y
 * |        v
 * |
 * v
 * y
 * 
 * </pre>
 * <P>
 * This interface is not meant to be implemented by user classes. It is mainly intended to provide access to the field provided
 * by the framework. If you want to make your own field class, you may consider making a different interface.
 *<P> 
 * It is usually enough to use the various convenience methods provided for interacting with the field, but the below explanation
 * may be useful in some cases.
 * <P>
 * Objects on the playfield are identified by integer codes. getObjectAt(x, y) obtains the object at a given position.
 * This object can be either a coin (FIELD_COIN), a wall (FIELD_WALL), or an empty square (FIELD_FREE).
 * Only FIELD_FREE and FIELD_COIN are safe to pass through.
 * <P>
 * In addition, the object can be the trail of an enemy tron. AbstractPlayer.getTrailIdOf(id) gets the trail code for the player identified
 * by "id". Id numbers range from 0 and upwards.
 * getTrons() in this class can be used to get all trons, and then getId() can be used on each to obtain their id code. 
 * BasePlayer.getEnemyIds() can also be used to obtain enemy tron ids.
 * 
 * @author johan
 *
 *@see jp.ac.nii.icpc2010.players.AbstractPlayer
 *@see jp.ac.nii.icpc2010.players.BasePlayer
 *@see ITron
 */
public interface IPlayField {

	/**
	 * An empty square. Trons can move through walls.
	 */
	public static final int FIELD_FREE = 0;
	
	/**
	 * A wall. Trons cannot move through walls.
	 */
	public static final int FIELD_WALL = 1;
	
	/**
	 * A coin. Trons can move through coins, and they get points for doing so.
	 */
	public static final int FIELD_COIN = 2;
		
	
	/**
	 * Obtain the width (in units) of the field
	 * @return
	 */
	public int getWidth();

	/**
	 * Obtain the height (in units) of the field
	 * @return
	 */
	public int getHeight();

	/** 
	 * Obtain Object ID at a certain position in the field.
	 * @param x
	 * @param y
	 * @return One of the values above (FIELD_FREE ... FIELD_WALL), or a Tron trail ID.
	 */
	public int getObjectAt(int x, int y);

	/**
	 * Obtain Trons on this playfield.
	 * @return
	 */
	public Vector<ITron> getTrons();

	/**
	 * Return the new coordinates of the tron after stepping one step.
	 * This function takes care of wrapping over the edge of the field, etc.
	 *  
	 * @param x Current position X 
	 * @param y Current position Y
	 * @param dir Direction to step in
	 * @return Array of size 2; position 0 is x, position 1 is y
	 */
	public int[] stepOne(int x, int y, FieldDirection dir);

	/**
	 * Get the number of players in the play field.
	 * @return
	 */
	public int getNumOfPlayers();
	
	/**
	 * Return number of current turn. 
	 * @return
	 */
	public int getTurn();

	/**
	 * Return number of current round.
	 * @return
	 */
	public int getRound();
	
	/**
	 * Return remaining turns in this round.
	 * @return
	 */
	public int getRemainingTurns();
	
	/**
	 * Return remaining rounds in this match.
	 * @return
	 */
	public int getRemainingRounds();
}
