/**********************************************************************/
/*                                                                    */
/*	CRISP - Programmable editor                                   */
/*	===========================                                   */
/*                                                                    */
/*  File:          restore.cr                                         */
/*  Author:        P. D. Fox                                          */
/*  Created:       22 Feb 1990                     		      */
/*                                                                    */
/*  Copyright (c) 1990 Paul Fox                                       */
/*                All Rights Reserved.                                */
/*                                                                    */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*  Description:  Save and restore editing session.                   */
/*                                                                    */
/**********************************************************************/

/* SCCS ID: %Z% %M% %R%.%L% */
# include	"crisp.h"

# define HPFS

/**********************************************************************/
/*   Name  of  file  to  store  restore  info in. This should be the  */
/*   pure  file  name.  We  prepend  the  users  home directory (via  */
/*   $HOME).  If  $HOME  doesn't  exist,  then  RESTORE_FILE will be  */
/*   used as it is.						      */
/**********************************************************************/
/**********************************************************************/
/*   Name  of  file for storing the status of the screen. Created in  */
/*   the current directory.					      */
/**********************************************************************/
# if !(defined(OS2) || defined(MSDOS)) || defined(HPFS)
# define	RESTORE_FILE	".crisp"
# define	STATE_FILE		".crstate"
# else
# define	RESTORE_FILE	"crisp.rst"
# define	STATE_FILE		"crisp.sts"
# endif

/**********************************************************************/
/*   If  this  variable  is TRUE, then the restore file is stored in  */
/*   the  current  working  directory.  If  it is FALSE, then we may  */
/*   store it in the path specified by BRESTORE.		      */
/**********************************************************************/
int	res_this_dir = FALSE;

/**********************************************************************/
/*   Set  to  TRUE to save and restore the total editing state. This  */
/*   is  more  expensive  and  litters .crstate files in the current  */
/*   directory, but many people like this feature.		      */
/**********************************************************************/
int	ss_flag = FALSE;

/**********************************************************************/
/*   Filename  computed  at  load  time  where we load and store the  */
/*   restore info.						      */
/**********************************************************************/
string	res_filename;

string	state_file = STATE_FILE;
string	restore_file = RESTORE_FILE;

extern string	search__pattern;
extern string	translate__pattern;
extern string	translate__replacement;
/**********************************************************************/
/*   This macro should be called to turn on full state saving.	      */
/**********************************************************************/
void
save_state()
{
	ss_flag = TRUE;
}
/**********************************************************************/
/*   Macro  executed  at  load  time  to restore the current editing  */
/*   state.  This  macro  is  called  after  command line files have  */
/*   been  read  in  and is passed a flag telling us whether we read  */
/*   files  from  the  command  line.  If  we have then don't bother  */
/*   restoring state.						      */
/**********************************************************************/
void
_startup_complete(int files_read_in)
{	string	cwd, buf, bfile, home;
	string	curr_file;
	int	crisp_buf, cur_buf;
	int	ml;
	list	l;
# if 0
	extern string CRISP_OPSYS;
# endif

	if (! first_time())
		return;

	/***********************************************/
	/*   Set  up  a  trap  for when we exit so we  */
	/*   can save the state.		       */
	/***********************************************/
	register_macro(REG_EXIT, "restore_exit");
	
	/***********************************************/
	/*   If  the  current  buffer  isn't  that of  */
	/*   the   BFILE  environment  variable  then  */
	/*   the  user  has entered crisp with a file  */
	/*   on   the  command  line,  and  hence  we  */
	/*   shouldn't restore the filename info.      */
	/***********************************************/
	bfile = getenv("BFILE");
	inq_names(NULL, NULL, curr_file);

	getwd(NULL, cwd);
	home = getenv("HOME");
	if (substr(restore_file, 1, 1) == "/" || home == "")
		res_filename = restore_file;
	else {
		sprintf(res_filename, "%s/%s", home, restore_file);
		}
	/***********************************************/
	/*   If  files  specified  on  command  don't  */
	/*   bother restoring the state.	       */
	/***********************************************/
	if (files_read_in)
		return;

	/***********************************************/
	/*   Try  and  restore from the .crstate file  */
	/*   in  current  directory.  If  that  fails  */
	/*   we'll  use  the  .crisp file in the home  */
	/*   directory.				       */
	/***********************************************/
	if (restore_full_state())
		return;

	if (!exist(res_filename))
		return;
	cur_buf = inq_buffer();	
	edit_file(res_filename);
	crisp_buf = inq_buffer();

	set_buffer_flags(~BF_BACKUP, 0);
	/***********************************************/
	/*   Restore status line mode.		       */
	/***********************************************/
	top_of_buffer();
	if (re_search(NULL, "^echo_line=") > 0) {
		buf = substr(trim(read()), 11);
		echo_line(atoi(buf));
		}
	
	/***********************************************/
	/*   If   user  specified  files  on  command  */
	/*   line  then  don't  restore the last file  */
	/*   edited.				       */
	/***********************************************/
	top_of_buffer();
	/***********************************************/
	/*   Get last file edited in this directory.   */
	/***********************************************/
	goto_line(1);
	if (re_search(NULL, "^" + quote_regexp(cwd) + ":") > 0) {
		buf = trim(read());
		l = split(buf, " ");
		/***********************************************/
		/*   Make sure user sees progress messages.    */
		/***********************************************/
		ml = inq_msg_level();
		set_msg_level(0);
		if (exist(l[1])) {
			edit_file(l[1]);
			move_abs(atoi(l[2]), atoi(l[3]));
			set_center_of_window();
			}
		else
			edit_file(bfile);
		set_msg_level(ml);
		}
	else {
		ml = inq_msg_level();
		set_msg_level(0);
		edit_file(bfile);
		set_msg_level(ml);
		}
		

	if (inq_buffer() != crisp_buf)
		delete_buffer(crisp_buf);	
}

/**********************************************************************/
/*   Macro  called  on exit to save the current file and position in  */
/*   the .crisp file in the home directory.			      */
/**********************************************************************/
void
restore_exit()
{	string	filename, buf, cwd;
	int	line, col;
	int	flags;

	/***********************************************/
	/*   Try   and   save   full  state  if  user  */
	/*   requested it.			       */
	/***********************************************/
	if (ss_flag || exist(state_file)) {
		if (save_full_state())
			return;
		}

	getwd(NULL, cwd);
	inq_names(filename);
	inq_position(line, col);

	edit_file(res_filename);
	/***********************************************/
	/*   Make  sure  we  don't  keep  a backup of  */
	/*   the state file.			       */
	/***********************************************/
	set_buffer_flags(~BF_BACKUP, BF_SYSBUF | BF_NO_UNDO);

	goto_line(1);
	if (re_search(NULL, "^" + quote_regexp(cwd) + ":") > 0) {
		beginning_of_line();
		delete_line();
		goto_line(1);
		}
	sprintf(buf, "%s: %s %d %d\n", cwd, filename, line, col);
	insert(buf);
	/***********************************************/
	/*   Save state of the echo line options.      */
	/***********************************************/
	flags = echo_line();
	if (re_search(NULL, "^echo_line") > 0) {
		beginning_of_line();
		delete_line();
		}
	sprintf(buf, "echo_line=%d\n", flags);
	insert(buf);

	/***********************************************/
	/*   Now save the buffer.		       */
	/***********************************************/
	write_buffer();
}
/**********************************************************************/
/*   Macro to save the window layout and buffer states.		      */
/**********************************************************************/
int
save_full_state()
{	int	curwin, curbuf;
	int	win, l, c;
	string	name, buf_name;
	string	state_info;
	list	lst;
	int	i, len;
	int	is_system = inq_system();

	/***********************************************/
	/*   Save   size  of  screen  in  state  file  */
	/*   because  we  cannot  attempt  to restore  */
	/*   state  if  we  have  a  different  sized  */
	/*   window.				       */
	/***********************************************/
	inq_screen_size(l, c);
	sprintf(state_info, "screen=%d %d\n", l, c);

	/***********************************************/
	/*   Save list of buffers being edited.	       */
	/***********************************************/
	curbuf = inq_buffer();
	do {
		inq_names(name, NULL, buf_name);
		inq_position(l, c);
		/***********************************************/
		/*   Dont  save  info about our files because  */
		/*   we   may   get  screwed  if  user  exits  */
		/*   whilst looking at these files.	       */
		/***********************************************/
		if (!inq_system() && buf_name != state_file && buf_name != restore_file)
			state_info += "buffer=" + name + " " + l + " " + c + "\n";
		set_buffer(next_buffer(is_system));
		}
	while (inq_buffer() != curbuf);
	/***********************************************/
	/*   Walk   through  all  background  windows  */
	/*   and save the layout information.	       */
	/***********************************************/
	curwin = inq_window();
	while (1) {
		save_window_state();
		win = next_window();
		if (win == curwin)
			break;
		}
		
	edit_file(state_file);
	set_buffer_flags(~BF_BACKUP, BF_SYSBUF | BF_NO_UNDO);
	clear_buffer();
	insert(state_info);
	/***********************************************/
	/*   Save the bookmark information.	       */
	/***********************************************/
	lst = bookmark_list();
	len = length_of_list(lst);
	for (i = 0; i < len; i += 4) {
		inq_names(name, NULL, NULL, lst[i+1]);
		insert("bookmark=" + lst[i] + " " +
				name + " " +
				lst[i+2] + " " +
				lst[i+3] + "\n");
		}
		
	insert("search=" + quote_regexp(search__pattern) + "\n");
	insert("translate=" + quote_regexp(translate__pattern) + "\n");
	insert("repl=" + quote_regexp(translate__replacement) + "\n");
	write_buffer();
	return TRUE;
}

/**********************************************************************/
/*   Macro to save the information about the 'current' window.	      */
/**********************************************************************/
void
save_window_state()
{	int	win_id, buf_id, lx, by, rx, ty;
	int	line, col, top_left;
	string	filename, buf;
	extern string state_info;
	extern int curwin;

	inq_window_info(win_id, buf_id, lx, by, rx, ty);
	inq_position(line, col);
	inq_top_left(top_left);
	inq_names(filename);
	
	sprintf(buf, "window=%d %d %d %d %d %d %d %s %s\n",
		lx, by, rx, ty, line, col, top_left, filename,
		(win_id == curwin ? "CURRENT" : "0"));
	state_info += buf;
}

/**********************************************************************/
/*   Macro  to  restore the state of the screen from the state file.  */
/*   Returns TRUE if we restored the state, otherwise FALSE.	      */
/**********************************************************************/
int
restore_full_state()
{	list	lst;
	string	line;
	int	l, c, i;
	int	buf;
	int	rstbuf;
	int	curwin;
	int	state_buf;
	int	old_curwin = -1;
	int	line_no = 1;
	string	bfile, file;
	int	ml;
	int	partial_restore = FALSE;
	string	current_file;

	buf = inq_buffer();
	/***********************************************/
	/*   Cannot restore state if no file there.    */
	/***********************************************/
	if (!exist(state_file))
		return FALSE;

	curwin = inq_window();
	edit_file(state_file);
	set_buffer_flags(~BF_BACKUP, BF_SYSBUF | BF_NO_UNDO);
	state_buf = inq_buffer();
	top_of_buffer();
	rstbuf = inq_buffer();
	
	/***********************************************/
	/*   First  line  must  tell  us  the  screen  */
	/*   size.  If  this  is  different,  then we  */
	/*   ignore the state file.		       */
	/***********************************************/
	line = compress(read(), TRUE);
	if (substr(line, 1, 7) != "screen=") {
		partial_restore = TRUE;
		}
	else {
		lst = split(substr(line, 8), " ", 1);
		inq_screen_size(l, c);
		if (lst[0] != l || lst[1] != c)
			partial_restore = TRUE;
		}
	line_no++;
	/***********************************************/
	/*   Keep  reading  from  state  buffer until  */
	/*   we get to the end.			       */
	/***********************************************/
	while (1) {
		move_abs(line_no++);
		line = compress(read(), TRUE);
		if (line == "")
			break;
		
		/***********************************************/
		/*   Edit   the  specified  buffers  so  user  */
		/*   sees   them  in  a  buffer  list.  These  */
		/*   entries   should   precede   the  window  */
		/*   entries  otherwise  we'll  end  up  with  */
		/*   the windows containing the wrong info.    */
		/***********************************************/
		i = index(line, "=");
		switch (substr(line, 1, i-1)) {
		  case "buffer":
			line = substr(line, 8);
			l = c = 1;
			/***********************************************/
			/*   If  we  have a line and column specifier  */
			/*   then get that.			       */
			/***********************************************/
			if (i = index(line, " ")) {
				lst = split(line, " ", 1);
				file = lst[0];
				l = lst[1];
				c = lst[2];
				}
			else
				file = line;
			/***********************************************/
			/*   Only edit file if the file exists.	       */
			/***********************************************/
			if (exist(file)) {
				ml = inq_msg_level();
				set_msg_level(0);
				edit_file(file);
				set_msg_level(ml);
				/***********************************************/
				/*   Goto  the  line and column where we left  */
				/*   off.				       */
				/***********************************************/
				move_abs(l, c);
				}
			set_buffer(state_buf);
			break;

		  case "window":
			/***********************************************/
			/*   If  we  have  a  window definition, then  */
			/*   create it.				       */
			/***********************************************/
			lst = split(substr(line, 8), " ", 1);
			set_window(curwin);
			ml = inq_msg_level();
			set_msg_level(0);
			edit_file(lst[7]);
			set_msg_level(ml);
			if (partial_restore) {
				if (lst[8] == "CURRENT")
					current_file = lst[7];
				}
			else {
				i = create_tiled_window(
					lst[0], 
					lst[1],
					lst[2], 
					lst[3], inq_buffer());
				/***********************************************/
				/*   Remember which was the current window.    */
				/***********************************************/
				if (old_curwin < 0)
					old_curwin = i;
				set_top_left(lst[6], NULL, NULL, lst[4], lst[5]);
				}
			set_buffer(rstbuf);
			break;

		  case "bookmark":
			/***********************************************/
			/*   Restore  bookmarks.  These should follow  */
			/*   the   buffer   and   window  definitions  */
			/*   because  we  assume  the buffers are now  */
			/*   loaded.				       */
			/***********************************************/
			lst = split(substr(line, 10), " ", 1);
			drop_bookmark(lst[0], "y", inq_buffer(lst[1]), lst[2], lst[3]);
			break;
			
		  case "search":
		  	search__pattern = substr(line, 8);
		  	break;
		  case "translate":
		  	translate__pattern = substr(line, 11);
		  	break;
		  case "repl":
		  	translate__replacement = substr(line, 6);
		  	break;
		  }
		}
		
	
	if (old_curwin >= 0) {
		delete_window(curwin);
		set_window(old_curwin);
		i = inq_window_buf();
		if (i)
			set_buffer(i);
		else
			edit_file(getenv("BFILE"));
		}
	else {
		set_buffer(buf);
		attach_buffer(buf);
		}

	/***********************************************/
	/*   If  $BFILE  doesn't  exist and we've got  */
	/*   it  loaded  and  its  not displayed in a  */
	/*   window then delete it.		       */
	/***********************************************/
	bfile = getenv("BFILE");
	if (substr(bfile, 1, 1) != "/") {
		string	s;
		getwd(NULL, s);
		bfile = s + "/" + bfile;
		}
	i = inq_buffer(bfile);
	if (!exist(bfile) && i != 0 && inq_views(i) == 0)
		delete_buffer(i);

	if (current_file)
		edit_file(current_file);
	/***********************************************/
	/*   Set  flag  to  make  sure  we  save full  */
	/*   state on exit if not already set.	       */
	/***********************************************/
	ss_flag = TRUE;
	
	return TRUE;
}
