/*
$VerboseHistory: tcl.e$
*/
/*
  Don't modify this code unless defining extension specific
  aliases do not suit your needs.   For example, if you
  want your brace style to be:

       if [] {
          }

  Use the Extension Options dialog box ("Other", "Configuration...",
  "File Extension Setup...") and press the the "Alias" button to
  display the Alias Editor dialog box.  Press the New button, type
  "if" for the name of the alias and press <Enter>.  Enter the
  following text into the upper right editor control:

       if [%\c] {
       %\i}

  The  %\c indicates where the cursor will be placed after the
  "if" alias is expanded.  The %\i specifies to indent by the
  Extension Specific "Syntax Indent" amount define in the
  "Extension Options" dialog box.  Check the "Indent With Tabs"
  check box on the Extension Options dialog box if you want
  the %\i option to indent using tab characters.

*/
/*
  Options for TCL syntax expansion/indenting may be accessed from the
  Extension Options dialog ("Other", "Configuration...",
  "File Extension Setup...").

  The extension specific options is a string of five numbers separated
  with spaces with the following meaning:

    Position       Option
       1             Minimum abbreviation.  Defaults to 1.  Specify large
                     value to avoid abbreviation expansion.
       2             reserved.
       3             begin/end style.  Begin/end style may be 0,1, or 2
                     as show below.  Add 4 to the begin end style if you
                     want braces inserted when syntax expansion occurs
                     (main and do insert braces anyway).  Typing a begin
                     brace, '{', inserts an end brace when appropriate
                     (unless you unbind the key).  If you want a blank
                     line inserted in between, add 8 to the begin end
                     style.  Default is 4.

                      Style 0
                          if [] {
                             ++i;
                          }

                      Style 1
                          if []
                          {
                             ++i;
                          }

                      Style 2
                          if []
                            {
                            ++i;
                            }


       4             Indent first level of code.  Default is 1.
                     Specify 0 if you want first level statements to
                     start in column 1.
*/
#include 'slick.sh'

#define STYLE1_FLAG 1
#define STYLE2_FLAG 2
#define BRACE_INSERT_FLAG 4
#define BRACE_INSERT_LINE_FLAG 8
#define NO_SPACE_BEFORE_PAREN 16   // "if(" or "if ("

#define MODE_NAME 'tcl'
#define EXTENSION 'tcl'

defload()
{
   setup_info='MN='MODE_NAME',TABS=+8,MA=1 74 1,':+
               'KEYTAB='MODE_NAME'-keys,WW=1,IWT=0,ST=0,'
   compile_info='0 tcl *;'
   syntax_info='4 1 1 0 4 1 0'
   be_info=''
   create_ext(kt_index,EXTENSION,'',MODE_NAME,setup_info,compile_info,
              syntax_info,be_info)

   if ( kt_index ) {
      set_eventtab_index(kt_index,event2index(name2event('{')),
                         find_index('tcl-begin',COMMAND_TYPE));
      set_eventtab_index kt_index,event2index(name2event('}')),
                         find_index('tcl-endbrace',COMMAND_TYPE)
   }
   create_ext(kt_index,'tlib',EXTENSION);
   create_ext(kt_index,'itk',EXTENSION);
   create_ext(kt_index,'itcl',EXTENSION);
}

_command void tcl_mode() name_info(','VSARG2_REQUIRES_EDITORCTL|VSARG2_READ_ONLY|VSARG2_ICON)
{
   select_edit_mode('tcl')
}
_command void tcl_enter() name_info(','VSARG2_CMDLINE|VSARG2_ICON|VSARG2_REQUIRES_EDITORCTL)
{
   parse name_info(_edit_window().p_index) with syntax_indent expand . . be_style indent_fl .
   if ( command_state() || p_window_state:=='I' ||
      syntax_indent<0 || p_indent_style!=INDENT_SMART ||
      _in_comment() || tcl_expand_enter(syntax_indent,expand,be_style,indent_fl) ) {
      call_root_key(ENTER)
   } else if (_argument=='') {
      _undo 'S'
   }

}
_command void tcl_space() name_info(','VSARG2_CMDLINE|VSARG2_REQUIRES_EDITORCTL|VSARG2_LASTKEY)
{
   was_space=(last_event():==' ')
   parse name_info(_edit_window().p_index) with syntax_indent expand . . be_style indent_fl .
   if ( command_state() || ! expand || syntax_indent<0 ||
      _in_comment() ||
         tcl_expand_space(syntax_indent,be_style,indent_fl) ) {
      if ( was_space ) {
         if ( command_state() ) {
            call_root_key(' ')
         } else {
            keyin ' '
         }
      }
   } else if (_argument=='') {
      _undo 'S'
   }
}
_command void tcl_begin() name_info(','VSARG2_REQUIRES_EDITORCTL|VSARG2_CMDLINE)
{
   parse name_info(_edit_window().p_index) with syntax_indent expand . . be_style indent_fl .
   if ( command_state() || _in_comment() || !expand ||
       tcl_expand_begin(expand,syntax_indent,be_style,indent_fl) ) {
      call_root_key('{')
   } else if (_argument=='') {
      _undo 'S'
   }

}
_command void tcl_endbrace() name_info(','VSARG2_CMDLINE|VSARG2_REQUIRES_EDITORCTL)
{
   parse name_info(_edit_window().p_index) with syntax_indent expand . . be_style indent_fl .
   keyin '}'
   if ( command_state() || p_window_state:=='I' ||
      syntax_indent<0 || p_indent_style!=INDENT_SMART ||
      _in_comment() ) {
   } else if (_argument=='') {
      get_line line
      if (line=='}') {
         col=tcl_endbrace_col(be_style);
         if (col) {
            replace_line indent_string(col-1):+'}';
            p_col=col+1;
         }
      }
      _undo 'S'
   }
}

/* Returns column where end brace should go.
   Returns 0 if this function does not know the column where the
   end brace should go.
 */
int tcl_endbrace_col(be_style)
{
   if (p_lexer_name=='') {
      return(0);
   }
   save_pos(p);
   --p_col;
   // Find matching begin brace
   status=_find_matching_paren(def_pmatch_max_diff);
   if (status) {
      restore_pos(p);
      return(0);
   }
   // Assume end brace is at level 0
   if (p_col==1) {
      restore_pos(p);
      return(1);
   }
   begin_brace_col=p_col;
   // Check if the first char before open brace is close paren
   col= find_block_col();
   if (!col) {
      restore_pos(p);
      return(0);
   }
   style=(be_style & (STYLE1_FLAG|STYLE2_FLAG));
   if (style!=0) {
      restore_pos(p);
      return(begin_brace_col);
   }
   restore_pos(p);
   return(col);
}

static int find_block_col()
{
   --p_col;
   if (_clex_skip_blanks('-')) return(0);
   if (get_text()!=')') {
      if (_clex_find(0,'g')!=CFG_KEYWORD) {
         return(0);
      }
      word=cur_word(col);
      if (word=='do' || word=='else') {
         first_non_blank();
         return(p_col);
         //return(p_col-length(word)+1);
      }
      return(0);
   }
   status=_find_matching_paren(def_pmatch_max_diff);
   if (status) return(0);
   if (p_col==1) return(1);
   --p_col;

   if (_clex_skip_blanks('-')) return(0);
   if (_clex_find(0,'g')!=CFG_KEYWORD) {
      return(0);
   }
   word=cur_word(col);
   if (pos(' 'word' ',' if while foreach for elsif ')) {
      first_non_blank();
      return(p_col);
      //return(p_col-length(word)+1);
   }
   return(0);
}


/*
   TCL built-ins

AddErrInfo
after
Alloc
AllowExc
append
AppInit
array
AssocData
Async
BackgdErr
Backslash
bgerror
binary
BoolObj
break
CallDel
case
catch
cd
clock
close
CmdCmplt
Concat
concat
continue
CrtChannel
CrtChnlHdlr
CrtCloseHdlr
CrtCommand
CrtFileHdlr
CrtInterp
CrtMathFnc
CrtObjCmd
CrtSlave
CrtTimerHdlr
CrtTrace
DetachPids
DoOneEvent
DoubleObj
DoWhenIdle
DString
eof
error
Eval
eval
EvalObj
exec
Exit
exit
expr
ExprLong
ExprLongObj
fblocked
fconfigure
fcopy
file
fileevent
filename
FindExec
flush
for
foreach
format
GetIndex
GetInt
GetOpnFl
gets
GetStdChan
glob
global
Hash
history
http
if
incr
info
Interp
interp
IntObj
join
lappend
library
license.ter
lindex
LinkVar
linsert
list
ListObj
llength
load
lrange
lreplace
lsearch
lsort
man.macr
namespace
Notifier
Object
ObjectType
ObjSetVar
open
OpenFileChnl
OpenTcp
package
pid
pkgMkIndex
PkgRequire
Preserve
PrintDbl
proc
puts
pwd
read
RecEvalObj
RecordEval
RegExp
regexp
registry
regsub
rename
resource
return
safe
scan
seek
set
SetErrno
SetRecLmt
SetResult
SetVar
Sleep
socket
source
split
SplitList
SplitPath
StaticPkg
string
StringObj
StrMatch
subst
switch
Tcl
Tcl_Main
tclsh
tclvars
tell
time
trace
TraceVar
Translate
unknown
unset
update
uplevel
UpVar
upvar
variable
vwait
while
WrongNumAr
*/



#define EXPAND_WORDS ' for foreach if library namespace package proc switch while '
static _str space_words[] = {
   'array',
   'bgerror',
   'binary',
   'break',
   'case',
   'catch',
   'continue',
   'eof',
   'error',
   'eval',
   'exec',
   'exit',
   'expr',
   'for',
   'foreach',
   'format',
   'global',
   'history',
   'if',
   'incr',
   'info',
   'interp',
   'library',
   'list',
   'load',
   'namespace',
   'package',
   'proc',
   'return',
   'safe',
   'set',
   'switch',
   'tell',
   'trace',
   'unset',
   'variable',
   'while'
};

tcl_get_info(var Noflines,var cur_line,var first_word,var last_word,
              var rest,var non_blank_col,var semi,var prev_semi)
{
   save_pos(old_pos);
   first_word='';last_word='';non_blank_col=p_col;
   if (arg(9)=='') {
      for (j=0; ; ++j) {
         get_line cur_line
         if ( cur_line!='' ) {
            parse cur_line with line '#' /* Strip comment on current line. */
            parse line with before_brace '{' +0 last_word
            parse strip(line,'L') with first_word '[({:; \t]','r' +0 rest
            last_word=strip(last_word)
            parse name_info(_edit_window().p_index) with syntax_indent expand . . be_style indent_fl .
            if (last_word=='{' && !(be_style & STYLE2_FLAG)) {
               save_pos(p2);
               p_col=text_col(before_brace);
               _clex_skip_blanks('-');
               status=1;
               if (get_text()==')') {
                  status=_find_matching_paren(def_pmatch_max_diff);
               }
               if (!status) {
                  status=1;
                  if (p_col==1) {
                     up();_end_line();
                  } else {
                     left
                  }
                  _clex_skip_blanks('-');
                  if (_clex_find(0,'g')==CFG_KEYWORD) {
                     kwd=cur_word(junk);
                     status=!pos(' 'kwd' ',' if while foreach for ');
                  }
               }
               if (status) {
                  non_blank_col=text_col(line,pos('[~ \t]|$',line,1,'r'),'I')
                  restore_pos(p2);
               } else {
                  get_line(line);
                  non_blank_col=text_col(line,pos('[~ \t]|$',line,1,'r'),'I')
                  /* Use non blank of start of if, do, while, foreach, unless, or for. */
               }
            } else {
               non_blank_col=text_col(line,pos('[~ \t]|$',line,1,'r'),'I')
            }
            Noflines=j;
            break;
         }
         if ( up() ) {
            restore_pos(old_pos);
            return(1);
         }
         if (j>=100) {
            restore_pos(old_pos);
            return(1);
         }
      }
   } else {
      orig_col=p_col;
      for (j=0;  ; ++j) {
         get_line cur_line
         _begin_line();
         i=verify(cur_line,' '\t);
         if ( i ) p_col=text_col(cur_line,i,'I');
         if ( cur_line!='' && _clex_find(0,'g')!=CFG_COMMENT) {
            parse cur_line with line '#' /* Strip comment on current line. */
            parse line with before_brace '{' +0 last_word
            parse strip(line,'L') with first_word '[({:; \t]','r' +0 rest
            last_word=strip(last_word)
            parse name_info(_edit_window().p_index) with syntax_indent expand . . be_style indent_fl .
            if (last_word=='{' && !(be_style & STYLE2_FLAG)) {
               save_pos(p2);
               p_col=text_col(before_brace);
               _clex_skip_blanks('-');
               status=1;
               if (get_text()==')') {
                  status=_find_matching_paren(def_pmatch_max_diff);
               }
               if (!status) {
                  status=1;
                  if (p_col==1) {
                     up();_end_line();
                  } else {
                     left
                  }
                  _clex_skip_blanks('-');
                  if (_clex_find(0,'g')==CFG_KEYWORD) {
                     kwd=cur_word(junk);
                     status=!pos(' 'kwd' ',' if while foreach for ');
                  }
               }
               if (status) {
                  non_blank_col=text_col(line,pos('[~ \t]|$',line,1,'r'),'I')
                  restore_pos(p2);
               } else {
                  get_line(line);
                  non_blank_col=text_col(line,pos('[~ \t]|$',line,1,'r'),'I')
                  /* Use non blank of start of if, do, while, unless, foreach, or for. */
               }
            } else {
               non_blank_col=text_col(line,pos('[~ \t]|$',line,1,'r'),'I')
            }
            Noflines=j;
            break;
         }
         if ( up() ) {
            restore_pos(old_pos);
            return(1);
         }
         if (j>=100) {
            restore_pos(old_pos);
            return(1);
         }
      }
      if (!j) p_col=orig_col;
   }
   p='';
   if ( j ) {
      p=1;
   }
   semi=stat_has_semi(p);
   prev_semi=prev_stat_has_semi();
   restore_pos(old_pos);
   return(0);
}
/* Returns non-zero number if pass through to enter key required */
static typeless tcl_expand_enter(syntax_indent,expand,be_style,indent_fl)
{
   status=tcl_get_info(Noflines,cur_line,first_word,last_word,rest,
                        non_blank_col,semi,prev_semi)
   if (status) return(1);
   status=0
   style1=be_style & STYLE1_FLAG;
   style2=be_style & STYLE2_FLAG;
   if ( expand && ! Noflines ) {
      if ( (first_word=='for' || first_word=='foreach') &&
            name_on_key(ENTER):=='nosplit-insert-line' ) {
         if ( name_on_key(ENTER):!='nosplit-insert-line' ) {
            if ( (style1) || semi ) {
               return(1)
            }
            indent_on_enter(syntax_indent)
            return(0)
         }
         /* tab to fields of Perl for statement */
         line=expand_tabs(cur_line)
         semi1_col=pos(';',line,p_col)
         if ( semi1_col>0 && semi1_col>=p_col ) {
            p_col=semi1_col+2
         } else {
            semi2_col=pos(';',line,semi1_col+1)
            if ( (semi2_col>0) && (semi2_col>=p_col) ) {
               p_col=semi2_col+2
            } else {
               if ( style1 || semi ) {
                  return(1)
               }
               indent_on_enter(syntax_indent)
            }
         }
      } else {
         status=1
      }
   } else {
     status=1
   }
   if ( status ) {  /* try some more? Indenting only. */
      status=0;
      col=tcl_indent_col(cur_line,first_word,last_word,non_blank_col,semi,prev_semi,Noflines)
      indent_on_enter('',col)
   }
   return(status)

}

typeless tcl_indent_col(cur_line,first_word,last_word,non_blank_col,semi,prev_semi,Noflines /*,pasting_open_brace2 */)
{
   pasting_open_brace2=arg(8);   // pasting open brace in style2
   parse name_info(p_index) with syntax_indent expand . . be_style indent_fl .
   if ( syntax_indent=='' ) {
      return(non_blank_col);
   }
   style2=be_style & STYLE2_FLAG;
   is_structure=pos(' 'first_word' ',' if do while foreach for ')
   level1_brace=substr(cur_line,1,1)=='{'
   past_non_blank=p_col>non_blank_col || name_on_key(ENTER)=='nosplit-insert-line';
   /* messageNwait('is_struct='is_structure' semi='semi' psemi='prev_semi' firstw='first_word' lastw='last_word) */

   save_pos(p);
   up(Noflines);get_line(line);
   // Check for statement like this
   //
   //   if ( func(a,b,
   //          c,(d),(e) )) return;
   //
   //  look for last paren which matches to paren on different line.
   //
   if (Noflines) {
      i=length(line);
   } else {
      i=text_col(line,p_col,'p')-1;
   }
   //i=text_col(expand_tabs(line,1,p_col-1));
   //messageNwait('line='line' i='i);
   //old_col=p_col;
   pline=point();
   for (;;) {
      if (i<=0) break;
      j=lastpos(')',line,i);
      if (!j) break;
      p_col=text_col(line,j,'I');
      color=_clex_find(0,'g');
      //messageNwait('h1');
      if (color==CFG_COMMENT || color==CFG_STRING) {
         i=j-1;
         continue;
      }
      //messageNwait('try');
      status=_find_matching_paren(def_pmatch_max_diff);
      if (status) break;
      if (pline!=point()) {
         //messageNwait('special case');
         first_non_blank();
         non_blank_col=p_col;
         get_line(line);
         parse line with word .
         is_structure=pos(' 'word' ',' if do while foreach for ')
         //restore_pos(p);
         //return(col);
      }
      i=j-1;
   }
   restore_pos(p);
   if (
      (last_word=='{' && (! style2 || level1_brace) && indent_fl && past_non_blank) ||     /* Line end with '{' ?*/
      (is_structure && ! semi && past_non_blank && pasting_open_brace2!=1) ||
       pos('(\}|)else$',strip(cur_line),1,'r') || (first_word=='else' && !semi) ||
       (is_structure && last_word=='{' && past_non_blank) ) {
      //messageNwait('case1');
      return(non_blank_col+syntax_indent)
      /* Look for spaces, end brace, spaces, comment */
   } else if ( (pos('^([ \t])*\}([ \t]*)(\\|\#|$)',cur_line,1,'r') && style2)|| (semi && ! prev_semi)) {
      // OK we are a little lazy here. If the dangling statement is not indented
      // correctly, then neither will this statement.
      //
      //     if (
      //             )
      //             i=1;
      //         <end up here> and should be aligned with if
      //
      //messageNwait('case2');
      col=non_blank_col-syntax_indent;
      if ( col<=0 ) {
         col=1
      }
      if ( col==1 && indent_fl ) {
         return(non_blank_col)
      }
      return(col)
   }
   return(non_blank_col)

}
static typeless tcl_expand_space(syntax_indent,be_style,indent_fl)
{
   status=0
   get_line orig_line
   line=strip(orig_line,'T')
   orig_word=strip(line)
   if ( p_col!=text_col(line)+1 ) {
      return(1)
   }
   if_special_case=0;
   aliasfilename='';
   word=min_abbrev2(orig_word,space_words,name_info(p_index),aliasfilename)
   if (aliasfilename!=''&&word!='') {
      if (orig_word:==word && orig_word==get_alias(word,mult_line_info,1,aliasfilename)) {
         _insert_text(' ');
         return(0);
      }
      col=p_col-length(orig_word);
      if (col==1) {
         line_prefix='';
      }else{
         line_prefix=indent_string(col-1);
      }
      replace_line(line_prefix);
      p_col=col;
      return(expand_alias(word,'',aliasfilename));
   }
   #if 1
   if ( word=='') {
      // Check for ) unless
      parse orig_line with . '\)|last|next','r' +0 first_word second_word rest
      if ((first_word==')' || first_word=='last' || first_word=='next') &&
           second_word!='' && rest=='' && second_word:==substr('unless',1,length(second_word))) {
         keyin(substr('unless ',length(second_word)+1));
         return(0)
      }
      #if 0
      // Check for else if or } else if
      if (first_word=='else' && orig_word==substr('elseif',1,length(orig_word))) {
         word='elseif'
         if_special_case=1;
      } else if (second_word=='else' && rest!='' && orig_word==substr('} elseif',1,length(orig_word))) {
         word='} elseif'
         if_special_case=1;
      } else if (first_word=='}else' && second_word!='' && orig_word==substr('}elseif',1,length(orig_word))) {
         word='}elseif'
         if_special_case=1;
      } else {
         return(1)
      }
      #endif
   }
   #endif
   if ( word=='') return(1);

   maybespace=(be_style & NO_SPACE_BEFORE_PAREN)?'':' ';
   line=substr(line,1,length(line)-length(orig_word)):+word
   width=text_col(line,length(line)-length(word)+1,'i')-1
   style1=be_style & STYLE1_FLAG
   style2=be_style & STYLE2_FLAG
   e1=' {'
   #if 1
   if (! (word=='do' && !style2 && !style1) ) {
      if ( (be_style & (STYLE1_FLAG|STYLE2_FLAG)) ||
         ! (be_style & BRACE_INSERT_FLAG) ) {
         e1=''
      }
   }
   #endif
   if ( word=='elseif' ) {
      replace_line line:+maybespace:+'()'e1
      maybe_insert_braces(syntax_indent,be_style,width,word)
   } else if ( word=='for' ) {
      replace_line line:+maybespace'()'e1
      maybe_insert_braces(syntax_indent,be_style,width,word)
   #if 0
   } else if ( word=='foreach' ) {
      replace_line line:+maybespace'()'e1
      maybe_insert_braces(syntax_indent,be_style,width,word)
   #endif
   } else if ( word=='if' || if_special_case) {
      replace_line line:+maybespace:+'()'e1
      maybe_insert_braces(syntax_indent,be_style,width,word)
   #if 0
   } else if( word=='local' ) {
      replace_line line:+maybespace:+'();'
      p_col=width+length(word:+maybespace)+2;
   #endif
   } else if ( word=='next' || word=='last' ) {
      if ( orig_word==word ) {
         keyin(' ');
      } else {
         replace_line indent_string(width)word
         _end_line
      }
   } else if ( word=='print' ) {
      if ( orig_word=='print' ) {
         keyin(' ');
      } else {
         replace_line indent_string(width)'print '
         _end_line
      }
   } else if ( word=='private') {
      replace_line line:+':'
      _end_line
   } else if ( word=='return' ) {
      if (orig_word=='return') {
         keyin ' '
      } else {
         replace_line indent_string(width)'return '
         _end_line
      }
   } else if( word=='select' ) {
      replace_line line:+maybespace:+'();'
      p_col=width+length(word:+maybespace)+2;
   } else if ( word=='proc' ) {
      tcl_insert_proc();
   } else if ( word=='while' ) {
      replace_line line:+maybespace'()'e1
      maybe_insert_braces(syntax_indent,be_style,width,word)
   } else if ( pos(' 'word' ',EXPAND_WORDS) ) {
      replace_line indent_string(width)word' '
      _end_line
   } else {
     status=1
   }
   return status
}
static tcl_expand_begin(expand,syntax_indent,be_style,indent_fl)
{
   brace_indent=0
   keyin '{'
   get_line line
   pcol=text_col(line,p_col,'P')
   last_word=''
   if ( pcol-2>0 ) {
      i=lastpos('[~ ]',line,pcol-2,'r')
      if ( i && substr(line,i,1)==')' ) {
         parse substr(line,pcol-1) with  last_word '/\*|//','r'
      }
   }
   if ( line!='{' ) {
      if ( last_word!='{' ) {
         parse line with first_word second_word
         parse line with '}' word '{' +0 last_word '#'
         if ( (last_word!='{' || word!='else') &&
              first_word!='do' && first_word!='for' && first_word!='foreach' ) {
            return(0);
         }
      }
      if ( be_style & STYLE2_FLAG ) {
         brace_indent=syntax_indent
         be_style= be_style & ~(STYLE1_FLAG|STYLE2_FLAG|BRACE_INSERT_FLAG)
      }
   } else if ( ! (be_style & STYLE2_FLAG) ) {
      if ( ! prev_stat_has_semi() ) {
         old_col=p_col
         up
         if ( ! rc ) {
            first_non_blank();p_col=p_col+syntax_indent+1
            down
         }
         col=p_col-syntax_indent-1
         if ( col<1 ) {
            col=1
         }
         if ( col<old_col ) {
            replace_line indent_string(col-1)'{'
         }
      }
   }
   first_non_blank()
   if ( expand ) {
      col=p_col-1;
      if ( (col && (be_style & STYLE2_FLAG)) || (! (indent_fl+col)) ) {
         syntax_indent=0;
      }
      insert_line(indent_string(col+brace_indent));
      tcl_endbrace();
      up;_end_line;
      if ((be_style & BRACE_INSERT_LINE_FLAG) ) {
         tcl_enter();
      }
#if 0
      if ( be_style & BRACE_INSERT_LINE_FLAG ) {
         insert_line indent_string(col+syntax_indent)
      }
      insert_line indent_string(col+brace_indent)'}'
      up;_end_line
#endif
   } else {
      _end_line
   }
   return(0)

}
static typeless prev_stat_has_semi()
{
   status=1
   up;
   if ( ! rc ) {
      col=p_col;_end_line;get_line line
      parse line with line '\#','r'
      /* parse line with line '{' +0 last_word */
      /* parse line with first_word rest */
      /* status=stat_has_semi() or line='}' or line='' or last_word='{' */
      line=strip(line,'T');
      if (last_char(line)==')') {
         save_pos(p);
         p_col=text_col(line);
         status=_find_matching_paren(def_pmatch_max_diff);
         if (!status) {
            status=search('[~( \t]','@-r');
            if (!status) {
               if (!_clex_find(0,'g')==CFG_KEYWORD) {
                  status=1;
               } else {
                  kwd=cur_word(junk);
                  status=!pos(' 'kwd' ',' if do while foreach for ');
               }
            }
         }
         restore_pos(p);
      } else {
         status=last_char(line)!=')' && ! pos('(\}|)else$',line,1,'r')
      }
      down
      p_col=col
   }
   return(status)
}
static typeless stat_has_semi()
{
   get_line line
   parse line with line '#'
   line=strip(line,'T')
   name=name_on_key(ENTER)
   return((last_char(line):==';' || last_char(line):=='}') &&
            (
               ! (( name=='split-insert-line' ||
                     (name=='maybe-split-insert-line' && _insert_state())
                    ) && (p_col<=text_col(line) && arg(1)=='')
                   )
            )
         )

}
static void maybe_insert_braces(syntax_indent,be_style,width,word)
{
   col=width+length(word)+3
   if (be_style & NO_SPACE_BEFORE_PAREN) --col;
   if ( be_style & STYLE2_FLAG ) {
      width=width+syntax_indent
   }
   if ( be_style & BRACE_INSERT_FLAG ) {
      up_count=1
      if ( be_style & (STYLE1_FLAG|STYLE2_FLAG) ) {
         up_count=up_count+1
         insert_line  indent_string(width)'{'
      }
      if ( be_style & BRACE_INSERT_LINE_FLAG ) {
         up_count=up_count+1
         insert_line indent_string(width+syntax_indent)
      }
      insert_line indent_string(width)'}'
      up up_count;
   }
   p_col=col
   if ( ! _insert_state() ) { _insert_toggle }
}
/*
   It is no longer necessary to modify this function to
   create your own sub style.  Just define an extension
   specific alias.  See comment at the top of this file.
*/
static typeless tcl_insert_proc()
{
   parse name_info(p_index) with syntax_indent . . . be_style indent_fl .
   if( !((be_style&BRACE_INSERT_FLAG) && (be_style&STYLE2_FLAG)) ) {
      syntax_indent=0;
   }
   up_count=0;
   if( be_style&BRACE_INSERT_FLAG ) {
      up_count=1;
      if( (be_style&STYLE1_FLAG) || (be_style&STYLE2_FLAG) ) {
         ++up_count;
         replace_line('proc ');
         insert_line(indent_string(syntax_indent):+'{');
      } else {
         replace_line('proc  {');
      }
      if( be_style&BRACE_INSERT_LINE_FLAG ) {
         ++up_count;
         insert_line('');
      }
      insert_line(indent_string(syntax_indent):+'}');
   } else {
      replace_line('proc ');
      _end_line();
   }

   up(up_count);
   p_col=5;   // Put cursor after 'sub ' so user can keyin the name

   return(0);
}


/* =========== Perl Tagging Support ================== */
typeless def_tcl_proto;


typeless tcl_proc_search(var proc_name,find_first)
{
   orig_proc_name=proc_name
   if ( find_first ) {
      /* Pickup Perl prototype in code file. */
      tcl_proto=def_tcl_proto
      _tag_pass=1 " "point() " "point('l')
      if ( proc_name:=='' ) {
         //messageNwait("tcl_proc_search h1");
         //search '^[ \t]@proc[ \t]#[\~A-Za-z0-9_]#((\:\:|\x27)[\~A-Za-z0-9_]#|)\c[ \t]*($|\{)','@er'
         //search '^[ \t]@proc[ \t]#[\~A-Za-z0-9_]#((\:\:|\x27)[\~A-Za-z0-9_]#|)\c[ \t]*($|\{|[\~A-Za-z0-9_]#[ \t]*\{)','@er'
         search '^[ \t]@proc[ \t]#[\~A-Za-z0-9_]#((\:|\x27)[\~A-Za-z0-9_]#|)\c[ \t]*($|\{|[\~A-Za-z0-9_]#[ \t]*\{)','@er'
         //messageNwait("tcl_proc_search h2");
      } else {
         //parse proc_name with name '(' kind ')'
         //parse kind with class_name ':' kind
         //if ( kind=='' ) {
         //   kind=class_name
         //   class_name=''
         //}
         tag_tree_decompose_tag(proc_name, name, class_name, kind, df);
         if ( kind=='proto' ) {
            if ( class_name!='' ) {
               return(tcl_proc_search2(proc_name,1))
            }
            tcl_proto=1
         } else if ( kind=='func' || kind:=='' ) {
            tcl_proto=0
         } else {
            return(tcl_proc_search2(proc_name,1))
         }
         if ( class_name!='' ) {
            name=class_name'(\:\:|\x27)'name
         }
         search name,'@er>w=[\~A-Za-z0-9_]'
      }
   } else {
      if ( _tag_pass==2 ) {
         return(tcl_proc_search2(proc_name,0))
      }
      repeat_search
   }
   for (;;) {
      if ( rc ) {
         parse _tag_pass with pass p l
         if ( pass==1 && (proc_name:=='' || pos(':',proc_name)) ) {
            goto_point p,l
            _tag_pass=2
            return(tcl_proc_search2(proc_name,1))
         }
         return(rc)
      }
      get_line line
      line=expand_tabs(line)
      /* Strip trailing comment.  */
      i=pos('#',line)
      if ( i ) {
         line=substr(line,1,i-1)
      }
      /* The IF below is required for FIND-TAG and not MAKE-TAGS. */
      if ( pos('[A-Za-z_]',substr(line,1,1),1,'r') ) {
         kind='func)'
         if ( last_char(line):==';' ) {
            /* Found a prototype */
            if ( ! tcl_proto ) {
               /* Not looking for proto types. */
               repeat_search
               continue
            }
            kind='proto)'
         } else if ( tcl_proto && orig_proc_name!='' ) {
            /* Looking for Perl prototype of a specific name.  Not a procedure. */
            repeat_search
            continue
         }
         parse line with line '{'
         //messageNwait("tcl_proc_search h2 line="line);
         //i=lastpos('[ *]',strip(translate(line,' ',\t)),'','r')
         i=pos('[ *]',strip(translate(line,' ',\t)),'','r')
         proc_name=strip(substr(line,i+1))
         //messageNwait("tcl_proc_search proc_name="proc_name);
         ii=pos(' ',proc_name);
         if (ii) {
            proc_name=substr(proc_name,1,ii);
         }
         //messageNwait("tcl_proc_search proc_name="proc_name);
         /* x27 =' (apostrophe) */
         //if ( pos('{#1\:\:|\x27}',proc_name,1,'r') ) {
         if ( pos('{#1\:|\x27}',proc_name,1,'r') ) {
            if ( orig_proc_name!='' && ! pos(':',orig_proc_name) ) {
               /* Don't want procedure with class */
               repeat_search
               continue
            }
            delim=substr(proc_name,pos('S1'),pos('1'));
            parse proc_name with class_name (delim) proc_name
            kind=class_name':'kind
         }
         proc_name=proc_name:+'('kind
         return(0)
      }
      repeat_search
   }
}

static typeless tcl_proc_search2(var proc_name,find_first)
{
   orig_proc_name=proc_name
   if ( find_first ) {
      if ( proc_name:=='' ) {
         name_re='{[A-Za-z0-9_]#}'
      } else {
         //parse proc_name with name '(' kind ')'
         //parse kind with class_name ':' rest
         //if ( rest!='' ) {
         //   /* Search  class [class_name] */
         //   name=class_name
         //}
         tag_tree_decompose_tag(proc_name, name, class_name, kind, df);
         name_re='{'name'}'
      }
      after_struct_id='[\t ]*;[\t ]*(\#?*|)[\t ]*$'
      struct_re='(package)[\t ]+':+name_re:+after_struct_id
      if ( proc_name:=='' ) {
         search '('struct_re')','@re'
      } else {
         re=struct_re
         search re,'@ri'
      }
   } else {
      repeat_search
   }
   for (;;) {
      if ( rc ) {
         return rc
      }
      get_line line
      line=expand_tabs(line)
      if ( substr(line,p_col,7):!='package' ||
         (p_col!=1 && substr(line,p_col-1,1)!='') ) {
         /* Found incorrect case */
         repeat_search
         continue
      }
      word=get_text(match_length('0'),match_length('S0'))
      proc_name=word:+'(class)'
      break
   }
   return(0)

}

#define PROC_RE '^[ \t]@sub[ \t]#[\~A-Za-z0-9_]#\c[ \t]*($|\{)'

static typeless tcl_proc_search3(var proc_name,find_first)
{
   orig_proc_name=proc_name
   if ( find_first ) {
      /* Pickup Perl prototype in code file. */
      tcl_proto=def_tcl_proto
      if ( proc_name:=='' ) {
         search PROC_RE,'@mer'
      } else {
         //parse proc_name with name '(' kind ')'
         //parse kind with class_name ':' kind
         //if ( kind=='' ) {
         //   kind=class_name
         //   class_name=''
         //}
         tag_tree_decompose_tag(proc_name, name, class_name, kind, df);
         if ( kind=='proto' ) {
            tcl_proto=1
         } else {
            tcl_proto=0
         }
         search name,'@me>w=[\~A-Za-z0-9_]'
      }
   } else {
      repeat_search
   }
   for (;;) {
      if ( rc ) {
         _deselect();
         return(rc)
      }
      get_line line
      line=expand_tabs(line)
      /* Strip trailing comment.  */
      i=pos('#',line)
      if ( i ) {
         line=substr(line,1,i-1)
      }
      /* Make sure we found a proc */
      if ( proc_name=='' || pos(PROC_RE,line,1,'r') ) {
         kind='func)'
         if ( last_char(line):==';' ) {
            /* Found a prototype */
            if ( ! tcl_proto ) {
               /* Not looking for proto types. */
               repeat_search
               continue
            }
            kind='proto)'
         } else if ( tcl_proto && orig_proc_name!='' ) {
            /* Looking for Perl prototype of a specific name.  Not a procedure. */
            repeat_search
            continue
         }
         i=lastpos('[ *]',strip(translate(line,' ',\t),'T'),'','r')
         proc_name=strip(substr(line,i+1))
         if ( pos('{#1\:\:|\x27}',proc_name,1,'r') ) {
            if ( orig_proc_name!='' && ! pos(':',orig_proc_name) ) {
               /* Don't want procedure with class */
               repeat_search
               continue
            }
            delim=substr(proc_name,pos('S1'),pos('1'));
            parse proc_name with class_name (delim) proc_name
            kind=class_name':'kind
         }
         proc_name=proc_name:+'('kind
         return(0)
      }
      repeat_search
   }
}


