/** ColEm: portable Coleco emulator **************************/
/**                                                         **/
/**                           os2.c                         **/
/**                                                         **/
/** This file contains OS/2 dependent subroutines and       **/
/** drivers.                                                **/
/**                                                         **/
/** Copyright (C) Darrell Spice Jr, 1998                    **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/


/** Standard Unix/X #includes ********************************/
#include "Coleco.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "os2.h"
#include "os2rc.h"
#include "os2dialog.h"

// sound
#include "SN76489.h"
#include "sndos2.h"
#include "dsp.h"

#define INCL_WIN
#define INCL_PM
#define INCL_GPI
#define INCL_DOS
#define INCL_DOSPROCESS
#define INCL_WINSTDFILE   /* OS/2 include for File Dialog */
#define INCL_WINDIALOGS   /* OS/2 include for Window Dialog */
#include <os2.h>
#include "joyos2.h"


/* DIVE required header files  */
#define  _MEERROR_H_
#include <mmioos2.h>
#include <dive.h>
#include <fourcc.h>


/* Dive variables */
DIVE_CAPS          DiveCaps           = {0};
FOURCC             fccFormats[100]    = {0};
HDIVE              hDive              = NULLHANDLE;
BOOL               bDiveHaveBuffer    = FALSE;
ULONG              ulDiveBufferNumber = 0;
HDC                hdcClientHDC = NULLHANDLE;
HPAL               hpalMain;

/* PM Window variables */
HAB          hab ;
HINI         hini;
HMQ          hmq ;
HWND         hwndFrame, hwndClient ;
QMSG         qmsg ;
//BITMAPINFO2 * pbmi;
FONTMETRICS fm;
RECTL       udrcl;

char *Title="ColEm OS/2 1.0"; /* Window/command line title */
int UseSound = 0;

MRESULT EXPENTRY ClientWndProc (HWND, ULONG, MPARAM, MPARAM);
MRESULT EXPENTRY DlgProc(HWND, ULONG, MPARAM, MPARAM);


/** Various variables and short functions ********************/
#define WIDTH  272
#define HEIGHT 208
#define DisplayWIDTH 256       // actual display size is 256 x 192 the outer 8
#define DisplayHEIGHT 192      // pixels on each side are not used and are cropped off
byte XBuf[WIDTH * HEIGHT];           /* display buffer */
POINTL ptl, aptl[4];
int infoheight;

#define Black 0
#define Green 16

int ViewWave;
byte *ViewWaveStart;
extern unsigned char *Buf;

byte Palette[32][3];
byte InitPalette[17][3] =
{
    {0x00,0x00,0x00},{0x00,0x00,0x00},{0x30,0xD0,0x30},{0x70,0xF0,0x70}, //
    {0x30,0x30,0xF0},{0x50,0x70,0xF0},{0xB0,0x30,0x30},{0x50,0xD0,0xF0}, //
    {0xF0,0x30,0x30},{0xF0,0x70,0x70},{0xD0,0xD0,0x30},{0xD0,0xD0,0x90}, //
    {0x30,0x90,0x30},{0xD0,0x50,0xB0},{0xB0,0xB0,0xB0},{0xF0,0xF0,0xF0}, //
    {0x00,0xff,0x00}
//    {0x00,0x00,0x00},{0x00,0x00,0x00},{0x20,0xC0,0x20},{0x60,0xE0,0x60}, //
//    {0x20,0x20,0xE0},{0x40,0x60,0xE0},{0xA0,0x20,0x20},{0x40,0xC0,0xE0}, //
//    {0xE0,0x20,0x20},{0xE0,0x60,0x60},{0xC0,0xC0,0x20},{0xC0,0xC0,0x80}, //
//    {0x20,0x80,0x20},{0xC0,0x40,0xA0},{0xA0,0xA0,0xA0},{0xE0,0xE0,0xE0}, //
};
int Palette2[17];

int IFreq=50; /* used external */

CHAR szTitle[MAXNAMEL + 1];
CHAR* szClientClass;
int screenx, screeny, xborderwidth, yborderwidth, titleheight, menubarheight;
int width, height;

ERRORID err;
HPS     hps, hpsClient;

int game_paused;

/* Joystick Values */
word KB[2] = {0xFFFF, 0xFFFF};  /* keyboard results */
word JS[2] = {0xFFFF, 0xFFFF};  /* joystick results */

int frt = 0; /* frame rate table.  0 = 60 fps, 1 = 50 fps */
double framerate, newrate;
int count = 0;
int titletoggle = 0;
int titlecount = 0;
int autofr = TRUE;
int skip = 8;
int maxskip = 15;
int initskip = 8;
int minskip = 0;
int windowsize = 2;
int framecount;
int throttle;
int colem_active;

CHAR carthelppath[CCHMAXPATH] = "";
CHAR carthelpfile[CCHMAXPATH] = "";
CHAR carthelpfullfile[CCHMAXPATH] = "";
CHAR dragfilename[CCHMAXPATH] = "nada";
extern int newgame;
extern int firstgame;

extern int volume;
extern int buflen;           // 1536
extern ULONG AUDIOfreq;      // = 44100;

char CartName[CCHMAXPATH] = "";
extern char* ColemHelp;
extern char* ColemIni;

// dialog box variables
int db_volume, db_frequency, db_buffer; // audio
int db_frt, db_min, db_max, db_init, db_size; // video
int db_OK; // TRUE if dialog box not cancelled


int different_extension(char* ext, char* file)
{
    char* extpointer = file;
    int i = strlen(file);
    extpointer += i - 4;
    return strcmp(extpointer, ext);
}


int GetCartridge(char *filename)
{
   static int Init = FALSE;
   static FILEDLG fdFileDlg;
   char pszTitle[30] = "Select Cartridge for ColEm/2";
   static char drive[10];
   char* separator;
   char* temp;

   if (Init == FALSE)
   {
     memset(&fdFileDlg, 0, sizeof(FILEDLG));
     fdFileDlg.cbSize = sizeof(FILEDLG);
     fdFileDlg.fl = FDS_CENTER | FDS_PRELOAD_VOLINFO | FDS_OPEN_DIALOG;
     fdFileDlg.pszTitle = pszTitle;
     strcpy(fdFileDlg.szFullFile, "*.ROM");
     Init = TRUE;
   }
    if (firstgame != 0)
    {
       strcpy(dragfilename, filename);
       firstgame = 0;
    }
   if (strcmp(dragfilename,"nada") == 0)
   {
      WinFileDlg(HWND_DESKTOP, hwndClient, &fdFileDlg);
      if (fdFileDlg.lReturn!= DID_OK)
         return FALSE;
      strcpy (filename, fdFileDlg.szFullFile);
   }
   else
   {
      strcpy (filename, dragfilename);
      strcpy (dragfilename,"nada");
   }

   separator = strrchr(filename, PATH_SEPARATOR);


  if(separator != 0)
  {  strncpy(carthelppath, filename, separator - filename);
     carthelppath[separator - filename] = 0;
     strcat(carthelppath, "\\..\\docs\\");
     temp = (separator == 0) ? filename : separator + 1;
     strcpy (carthelpfile, temp);
     strcpy (CartName, temp);
  }
  else
  {
     carthelppath[0] = 0;
     strcpy(carthelpfile, filename);
     strcpy (CartName, temp);
  }

  separator = strrchr(carthelpfile, '.');
  if(separator != 0)
     strcpy(separator, ".doc");

   strcpy(carthelpfullfile,carthelppath);
   strcat(carthelpfullfile,carthelpfile);

   separator = strrchr(CartName, '.');
   if (separator != 0)
      strcpy(separator, "    ");


   sprintf(szTitle, "ColEm/2- %s", CartName); // update the window title
   WinSetWindowText(hwndFrame, szTitle);           // with current game

   /* new cartridge screen inits */
   colem_active = TRUE;
   return TRUE;
}


/** PUTIMAGE: macro copying image buffer into a window ********/
void PutImage(void)
{
   register int I, J;
   static ULONG    ulTime0, ulTime1;     /* output buffer for DosQierySysInfo      */
   static int FramesShown = 0;

   static int tc = 0;
   static pow[9] = {256, 224, 192, 160, 128, 96, 64, 32, 0};

   FillBuffer(); // fill audio buffer

   if ((throttle>0) && (count&0x1))
      DosSleep(((1-skiprate[throttle & 0xf][count]) + (throttle >> 4))*5);

/* if our current skiprate value is a 1, we've got a frame to show */
   if (skiprate[skip][count] == 1)
   {
      if(ViewWave)
         for (I=0;I<8;I++)
            for(J=0;J<256;J++)
               ViewWaveStart[J + I * WIDTH] = ((Buf[J] <= pow[I]) && (Buf[J] > pow[I+1])) ? Green : Black;

      DiveBlitImage (hDive, ulDiveBufferNumber, DIVE_BUFFER_SCREEN ); /* display ColEm frame buffer */

      FramesShown++;
      if (framecount-- <= 0)   /* framecounter should drop to zero once per second */
      {

         // calculate FPS
         DosQuerySysInfo ( QSV_MS_COUNT, QSV_MS_COUNT, &ulTime1, 4L );
         newrate = (double) ((FramesShown * 1000) / (ulTime1 - ulTime0));
         ulTime0 = ulTime1;
         framerate = (framerate * 7 / 8) + (newrate / 8); /* age frame rate */

         FramesShown = 0;

         // update status box
         if (throttle > 0)
            sprintf(szTitle, "%2.0ffps %dT%c %3d%c %dx%d",
               framerate, throttle, ((autofr) ? 'A':' '),
               (int) (100 * (framerate + fps[frt][0] - fps[frt][skip]) / fps[frt][0]), '%', width, height);
         else
            sprintf(szTitle, "%2.0ffps %2.0ft%c %2.0fm %3d%c %dx%d",
               framerate, fps[frt][skip], ((autofr) ? 'A':' '), fps[frt][maxskip],
               (int) (100 * (framerate + fps[frt][0] - fps[frt][skip]) / fps[frt][0]), '%', width, height);
         GpiCharStringPosAt(hpsClient, &ptl, &udrcl, CHS_CLIP | CHS_OPAQUE, (LONG) strlen(szTitle), szTitle, NULL);

         if (autofr)                          /* if autoframerate routines are active */
         {                                    /*   then adjust frame rate */
            if (throttle > 0)
            {
               if ((framerate - fps[frt][0])  > fps[frt][15])
                  throttle += 1;
               else if ((fps[frt][0] - framerate) > fps[frt][15])
                  throttle -= 1;
            }
            else
            {
               if ((fps[frt][skip] - framerate) > fps[frt][15]) /* if emulation is to slow */
               {                                        /* fps[frt][15] is difference between adjacent fps[] values */
                  if (skip < maxskip)                   /* and not at user's slowest rate */
                     skip++;                            /* choose a slower frame rate */
               }
               else
               {
                  if ((framerate - fps[frt][skip]) > fps[frt][15])/* if emulation is to fast        */
                     if (skip > 0)                        /* and not already at highest fps */
                        skip--;                           /* choose a faster frame rate     */
                     else
                        throttle = 1;
               }
            }
         }
         framecount = framerate;              /* init framecount for 1 second of frames */
      } /* end of --> if (framecount-- <= 0) */
   } /* end of --> if (skip[skip][count]) */
   count++;
   count &= 0xf;    /* inc counter by 1 & limit values from 0-15 */
}




/** TrashMachine *********************************************/
/** Deallocate all resources taken by InitMachine().        **/
/*************************************************************/
void TrashMachine(void)
{
   audio_close();
   JoystickOff();
   JoystickSaveCalibration(ColemIni); /* save calibration info */
   DiveUnMap();
   CloseWindow();
}

/** InitMachine **********************************************/
/** Allocate resources needed by Unix/X-dependent code.     **/
/*************************************************************/
int InitMachine(void)
{
   int i, j;
   /*** Palette ***/
   for (i = 0;i < 17;i++)
   {
      for (j = 0;j < 3;j++)
         Palette[i][j] = InitPalette[i][j];
      Palette2[i] = PC_RESERVED * 16777216 + Palette[i][0] * 65536 + Palette[i][1] * 256 + Palette[i][2];
   }
   if (!OpenWindow())   /* open failed */
      return 0;
   DiveMap();
   JoystickInit(0,ColemIni); // Initialize joystick calibration information
   JoystickOn();

   ViewWave = 0;
   ViewWaveStart = XBuf + 8 + 272 * 200;

   if (volume > 0)
      audio_init(AUDIOfreq, buflen);

   return 1;
}

/** Joysticks ************************************************/
/** Check for keyboard events, parse them, and modify       **/
/** joystick controller status                              **/
/*************************************************************/
void Joysticks(void)
{
   static JOYSTICK_STATUS stick;

loop:
   if (WinPeekMsg(hab, &qmsg, NULLHANDLE, 0, 0, PM_NOREMOVE))
      if (WinGetMsg (hab, &qmsg, NULLHANDLE, 0, 0))
         WinDispatchMsg (hab, &qmsg) ;
      else
         ExitNow = 1;

while (game_paused)
   {
      audio_quiet();
      if (WinGetMsg (hab, &qmsg, NULLHANDLE, 0, 0))
         WinDispatchMsg (hab, &qmsg);
      else
      {
         ExitNow = 1;
         game_paused = FALSE;
      }
   }

   if (WinPeekMsg(hab, &qmsg, NULLHANDLE, 0, 0, PM_NOREMOVE)) goto loop;      /* don't let messages queue up */

   JS[0] = JS[1] = 0xFFFF;  /* default to nothing */
   if (JoystickValues(&stick))
   {
      /* process joystick 1 */
      if (stick.Joy1X == -1)     JS[0] &=0xF7FF;  /* Player 1 left   */
      else if (stick.Joy1X == 1) JS[0] &=0xFDFF;  /*          right  */
      if (stick.Joy1Y == -1)     JS[0] &=0xFEFF;  /*          up     */
      else if (stick.Joy1Y == 1) JS[0] &=0xFBFF;  /*          down   */
      if (stick.Joy1A)           JS[0] &=0xBFFF;  /*          fire 1 */
      if (stick.Joy1B)           JS[0] &=0xFFBF;  /*          fire 2 */
      /* process joystick 2 */
      if (stick.Joy2X == -1)     JS[1] &=0xF7FF;  /* Player 2 left   */
      else if (stick.Joy2X == 1) JS[1] &=0xFDFF;  /*          right  */
      if (stick.Joy2Y == -1)     JS[1] &=0xFEFF;  /*          up     */
      else if (stick.Joy2Y == 1) JS[1] &=0xFBFF;  /*          down   */
      if (stick.Joy2A)           JS[1] &=0xBFFF;  /*          fire 1 */
      if (stick.Joy2B)           JS[1] &=0xFFBF;  /*          fire 2 */
      }

   JoyState[0] = (JS[0] == 0xFFFF) ? KB[0] : ((JS[0] & 0xFFF0) | (KB[0] & 0x000F));
   JoyState[1] = (JS[1] == 0xFFFF) ? KB[1] : ((JS[1] & 0xFFF0) | (KB[1] & 0x000F));
}

/** InitScrF *************************************************/
/** Generic function to be called when screen is changed.   **/
/*************************************************************/
int InitScrF(void)
{
    return(1);
}

/** ColorScrF ************************************************/
/** Generic function to be called when colors are changed.  **/
/*************************************************************/
void ColorsScrF(void)
{
   memset(XBuf,BGColor,WIDTH*HEIGHT);
}

/** Sound ****************************************************/
/** Called when something is written to a sound register.   **/
/*************************************************************/
void Sound(byte R,word V)
{
}


void Sprites(void)
{
  register byte N,J,Y,C;
  register byte *P,*T,*S;
  register int L;

  for(N=0,S=SprTab;(N<32)&&(S[0]!=208);N++,S+=4);

  if(Sprites16x16)
    for(S-=4;N;N--,S-=4)
    {
      L=S[3]&0x80? S[1]-32:S[1];C=S[3]&0x0F;
      if((L<=256-16)&&(L>=0)&&C)
      {
        P=XBuf+WIDTH*(HEIGHT-192)/2+(WIDTH-256)/2+L;
        T=SprGen+((long)(S[2]&0xFC)<<3);
        Y=S[0]+1;

        if(Y<192) { P+=WIDTH*Y;Y=(Y>176)? 192-Y:16; }
        else      { T+=256-Y;Y-=(Y>240)? 240:Y; }

        for(;Y;Y--,T++,P+=WIDTH)
        {
          J=*T;
          if(J&0x80) P[0]=C;if(J&0x40) P[1]=C;
          if(J&0x20) P[2]=C;if(J&0x10) P[3]=C;
          if(J&0x08) P[4]=C;if(J&0x04) P[5]=C;
          if(J&0x02) P[6]=C;if(J&0x01) P[7]=C;
          J=*(T+16);
          if(J&0x80) P[8]=C;if(J&0x40) P[9]=C;
          if(J&0x20) P[10]=C;if(J&0x10) P[11]=C;
          if(J&0x08) P[12]=C;if(J&0x04) P[13]=C;
          if(J&0x02) P[14]=C;if(J&0x01) P[15]=C;
        }
      }
    }
  else
    for(S-=4;N;N--,S-=4)
    {
      L=S[3]&0x80? S[1]-32:S[1];C=S[3]&0x0F;
      if((L<=256-8)&&(L>=0)&&C)
      {
        P=XBuf+WIDTH*(HEIGHT-192)/2+(WIDTH-256)/2+L;
        T=SprGen+((long)S[2]<<3);
        Y=S[0]+1;

        if(Y<192) { P+=WIDTH*Y;Y=(Y>184)? 192-Y:8; }
        else      { T+=256-Y;Y-=(Y>248)? 248:Y; }

        for(;Y;Y--,T++,P+=WIDTH)
        {
          J=*T;
          if(J&0x80) P[0]=C;if(J&0x40) P[1]=C;
          if(J&0x20) P[2]=C;if(J&0x10) P[3]=C;
          if(J&0x08) P[4]=C;if(J&0x04) P[5]=C;
          if(J&0x02) P[6]=C;if(J&0x01) P[7]=C;
        }
      }
    }
}


void RefreshScrF(register byte Y1,register byte Y2)
{
  if(Verbose>1)
    printf
    (
      "ScrMODE %d: ChrTab=%X ChrGen=%X ColTab=%X SprTab=%X SprGen=%X\n",
      ScrMode,ChrTab-VRAM,ChrGen-VRAM,ColTab-VRAM,SprTab-VRAM,SprGen-VRAM
    );
  memset(XBuf+(word)Y1*WIDTH,BGColor,(Y2-Y1+1)*WIDTH);
}


void RefreshScr0(register byte Y1,register byte Y2)
{
  register byte X,Y,J,K,FC,BC;
  register byte *P,*S,*T;

  P=XBuf+WIDTH*(HEIGHT-192)/2;

  if(ScreenON)
  {
    BC=BGColor;
    FC=FGColor;
    P+=(WIDTH-240)/2;T=ChrTab;

    for(Y=24;Y;Y--,P+=WIDTH*8-6*40)
      for(X=0;X<40;X++,T++)
      {
        S=ChrGen+((long)*T<<3);
        for(J=0;J<8;J++)
        {
          K=*S++;
          P[0]=K&0x80? FC:BC;P[1]=K&0x40? FC:BC;
          P[2]=K&0x20? FC:BC;P[3]=K&0x10? FC:BC;
          P[4]=K&0x08? FC:BC;P[5]=K&0x04? FC:BC;
          P+=WIDTH;
        }
        P+=6-WIDTH*8;
      }
  }
  else memset(P+(word)Y1*WIDTH,BGColor,(Y2-Y1+1)*WIDTH);
  //WinSetWindowText(hwndFrame, "Screen 0");
  PutImage();
}


void RefreshScr1(register byte Y1,register byte Y2)
{
  register byte X,Y,K,I,FC,BC;
  register byte *P,*S,*T;

  P=XBuf+WIDTH*(HEIGHT-192)/2;

  if(ScreenON)
  {
    P+=(WIDTH-256)/2;T=ChrTab;

    for(Y=24;Y;Y--,P+=WIDTH*8-8*32)
      for(X=0;X<32;X++,T++)
      {
        S=ChrGen+((long)*T<<3);
        BC=*(ColTab+(*T>>3));
        FC=BC>>4;
        BC=BC&0x0F;
        for(I=0;I<8;I++)
        {
          K=*S++;
          P[0]=K&0x80? FC:BC;P[1]=K&0x40? FC:BC;
          P[2]=K&0x20? FC:BC;P[3]=K&0x10? FC:BC;
          P[4]=K&0x08? FC:BC;P[5]=K&0x04? FC:BC;
          P[6]=K&0x02? FC:BC;P[7]=K&0x01? FC:BC;
          P+=WIDTH;
        }
        P+=8-WIDTH*8;
      }
    Sprites();
  }
  else memset(P+(word)Y1*WIDTH,BGColor,(Y2-Y1+1)*WIDTH);
 // WinSetWindowText(hwndFrame, "Screen 1");
  PutImage();
}


void RefreshScr2(register byte Y1,register byte Y2)
{
  register byte X,Y,K,I,N,FC,BC;
  register byte *P,*S,*T,*PGT,*CLT,*C;

  P=XBuf+WIDTH*(HEIGHT-192)/2;

  if(ScreenON)
  {
    P+=(WIDTH-256)/2;
    PGT=ChrGen;CLT=ColTab;T=ChrTab;

    for(N=0;N<3;N++,PGT+=0x0800,CLT+=0x0800)
      for(Y=8;Y;Y--,P+=WIDTH*8-8*32)
        for(X=0;X<32;X++,T++)
        {
          S=PGT+((long)*T<<3);C=CLT+((long)*T<<3);
          for(I=0;I<8;I++)
          {
            BC=*C++;K=*S++;
            FC=BC>>4;
            BC=BC&0x0F;
            P[0]=K&0x80? FC:BC;P[1]=K&0x40? FC:BC;
            P[2]=K&0x20? FC:BC;P[3]=K&0x10? FC:BC;
            P[4]=K&0x08? FC:BC;P[5]=K&0x04? FC:BC;
            P[6]=K&0x02? FC:BC;P[7]=K&0x01? FC:BC;
            P+=WIDTH;
          }
          P+=8-WIDTH*8;
        }
    Sprites();
  }
  else memset(P+(word)Y1*WIDTH,BGColor,(Y2-Y1+1)*WIDTH);
 // WinSetWindowText(hwndFrame, "Screen 2");
  PutImage();
}


void RefreshScr3(byte Y1, byte Y2)
{
  byte X,Y;
  byte *P,*T,*C;
  ulong C1,C2;

  P=XBuf+WIDTH*(HEIGHT-192)/2;

  if(ScreenON)
  {
    P+=(WIDTH-256)/2;
    T=ChrTab;C=ChrGen;

    for(Y=0;Y<24;Y++,P+=WIDTH*8-8*32)
      for(X=0;X<32;X++,T++)
      {
        C=ChrGen+((long)*T<<3)+((Y&3)<<1);
        C1=(*C >> 4)  * 0x01010101L;
        C2=(*C & 0x0F) * 0x01010101L;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;P+=WIDTH;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;P+=WIDTH;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;P+=WIDTH;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;P+=WIDTH;
        C++;
        C1=(*C>>4) * 0x01010101L;
        C2=(*C&0x0F) * 0x01010101L;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;P+=WIDTH;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;P+=WIDTH;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;P+=WIDTH;
        *(ulong *)P=C1;*(ulong *)(P+4)=C2;
        P+=8-7*WIDTH;
      }
    Sprites();
  }
  else memset(P+(word)Y1*WIDTH,BGColor,(Y2-Y1+1)*WIDTH);
 // WinSetWindowText(hwndFrame, "Screen 3");
  PutImage();
}



int DiveMap(void)
{
    ulDiveBufferNumber = 0;
    bDiveHaveBuffer = FALSE;
    /* Associate the canvas with a DIVE handle */
    if (DiveAllocImageBuffer (hDive, &ulDiveBufferNumber, FOURCC_LUT8,
                          WIDTH, HEIGHT, WIDTH,  (PBYTE) XBuf) != NULL)
      {
      /* post error message here.                               */
      /* PostError (hab, hwndFrame, IDS_ERR_CREATE_DIVEBUFFER); */
       return (FALSE);
        };
      bDiveHaveBuffer = TRUE;

          /* Turn on visible region notification. */
         WinSetVisibleRegionNotify (hwndClient, TRUE);

           /* Send an invalidation message to the client. */
          WinPostMsg (hwndFrame, WM_VRNENABLED, 0L, 0L );

    return (TRUE);
}



int DiveUnMap(void)
{
  if (bDiveHaveBuffer)
    {
    DiveFreeImageBuffer (hDive, ulDiveBufferNumber);
    ulDiveBufferNumber = 0;
    bDiveHaveBuffer = FALSE;
    };
  return TRUE;
}

/* set up DIVE - if not possible, display message and quit program */
int InitializeDive(void)
{
    /* Get the screen capabilities, and if the system supports only 16 colors */
    /*  the program should be terminated. */

    DiveCaps.pFormatData    = fccFormats;
    DiveCaps.ulFormatLength = 100;
    DiveCaps.ulStructLen    = sizeof(DIVE_CAPS);

    if ( DiveQueryCaps ( &DiveCaps, DIVE_BUFFER_SCREEN ))
       {
       WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
           (PSZ)"Error: DIVE routines cannot function in this system environment.",
           (PSZ)"   This program is unable to run.", 0, MB_OK | MB_INFORMATION );
       return (FALSE);
       };

    if ( DiveCaps.ulDepth < 8 )
       {
       WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
           (PSZ)"Error: Not enough screen colors to run DIVE.  Must be at least 256 colors.",
           (PSZ)"   This program is unable to run.", 0, MB_OK | MB_INFORMATION );
       return (FALSE);
       };

    /* Get an instance of DIVE APIs. */
    if ( DiveOpen ( &hDive, FALSE, 0 ) )
       {
       WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
           (PSZ)"Error: Unable to open an instance of DIVE.",
           (PSZ)"   This program is unable to run.", 0, MB_OK | MB_INFORMATION );
       return (FALSE);
       };

    return (TRUE);
};

VOID SizeTheWindow(HWND hwnd, int size)
{
   static int lastsize = 1;
   if (size > 0)
   {
      width = DisplayWIDTH * size;
      height = (DisplayHEIGHT + ViewWave) * size;
      lastsize = size;
   }
   else
   {
      width = DisplayWIDTH * lastsize;
      height = (DisplayHEIGHT + ViewWave) * lastsize;
   }

   WinSetWindowPos(hwndFrame, NULLHANDLE,
                     (screenx - width - 2*xborderwidth)/2,
                     (screeny - height - 2*yborderwidth - titleheight - menubarheight)/2,
                     width + 2*xborderwidth, height + 2*yborderwidth + titleheight + menubarheight + infoheight,
                     SWP_SIZE|SWP_MOVE|SWP_ACTIVATE|SWP_SHOW);
}



/* create window */
int OpenWindow(void)
{
   int reg;
   ULONG cb;
   /* joystick support */
   ULONG action; /* only used for DosOpen */
   /* open terminal session */
   static ULONG flFrameFlags = FCF_TITLEBAR      | FCF_SYSMENU |    FCF_MENU   |
                               FCF_SIZEBORDER    | FCF_MINMAX  |
                               FCF_TASKLIST      | FCF_ICON ;

   szClientClass = strdup("ColEmWindow");

   //skip = 8;                   /* set starting rate at 25 or 30fps */
   //framerate = frt ? 25 : 30;  /* set current frame rate at 25 or 30 */
   throttle = 0;               /* set throttle delay to 0 */
   colem_active = FALSE;

   DosSetCurrentDir("ROMS");

   /* divisor value for paddle reading based on mouse's current x position on screen */
   screenx      = WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN);
   screeny      = WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN);
   xborderwidth = WinQuerySysValue(HWND_DESKTOP, SV_CXSIZEBORDER);
   yborderwidth = WinQuerySysValue(HWND_DESKTOP, SV_CYSIZEBORDER);
   titleheight  = WinQuerySysValue(HWND_DESKTOP, SV_CYTITLEBAR);
   menubarheight = 37; /* button graphics */

   hab = WinInitialize(0) ;
   err = WinGetLastError(hab);
   hmq = WinCreateMsgQueue(hab, 0) ;
   err = WinGetLastError(hab);

   if (InitializeDive() == FALSE)     /* bail out if no DIVE support */
      return FALSE;

   // colem user setting defaults
   frt        = 0; // initialize to current settings
   maxskip    = 15; // min frame rate defined by maximum skip value
   minskip    = 0; // maximum frame rate defined by minimum skip value
   initskip   = 8;
   windowsize = 1;
   volume     = 1; // no longer works with new ColEm core
   buflen     = 768;
   AUDIOfreq  = 22050;

   hini = PrfOpenProfile(hab, ColemIni); // load saved settings
   if (hini)
   {
      PrfQueryProfileSize(hini, "colem", "frt", &cb);
      if (cb == sizeof(frt))
         if (!PrfQueryProfileData(hini,"colem","frt", &frt, &cb))
            frt = 0; // default

      PrfQueryProfileSize(hini, "colem", "maxskip", &cb);
      if (cb == sizeof(maxskip))
         if (!PrfQueryProfileData(hini,"colem","maxskip", &maxskip, &cb))
            maxskip = 15; // default

      PrfQueryProfileSize(hini, "colem", "minskip", &cb);
      if (cb == sizeof(minskip))
         if (!PrfQueryProfileData(hini,"colem","minskip", &minskip, &cb))
            minskip = 0; // default

      PrfQueryProfileSize(hini, "colem", "initskip", &cb);
      if (cb == sizeof(initskip))
         if (!PrfQueryProfileData(hini,"colem","initskip", &initskip, &cb))
            initskip = 8; // default

      PrfQueryProfileSize(hini, "colem", "windowsize", &cb);
      if (cb == sizeof(windowsize))
         if (!PrfQueryProfileData(hini,"colem","windowsize", &windowsize, &cb))
            windowsize = 1; // default

      PrfQueryProfileSize(hini, "colem", "volume", &cb);
      if (cb == sizeof(volume))
         if (!PrfQueryProfileData(hini,"colem","volume", &volume, &cb))
            volume = 1; // default

      PrfQueryProfileSize(hini, "colem", "buflen", &cb);
      if (cb == sizeof(buflen))
         if (!PrfQueryProfileData(hini,"colem","buflen", &buflen, &cb))
            buflen = 768; // default

      PrfQueryProfileSize(hini, "colem", "AUDIOfreq", &cb);
      if (cb == sizeof(AUDIOfreq))
         if (!PrfQueryProfileData(hini,"colem","AUDIOfreq", &AUDIOfreq, &cb))
            AUDIOfreq = 22050; // default

      PrfCloseProfile(hini);
   }
   skip = initskip;
   framerate = fps[frt][skip];
   reg = WinRegisterClass (hab, szClientClass, ClientWndProc, CS_SIZEREDRAW, 0) ;
   err = WinGetLastError(hab);
   hwndFrame = WinCreateStdWindow (HWND_DESKTOP, 0,
                                     &flFrameFlags, szClientClass, NULL,
                                     0L, 0, 444, &hwndClient);
   err = WinGetLastError(hab);

   hpsClient = WinGetPS(hwndClient);
   GpiQueryFontMetrics(hpsClient, (LONG) sizeof fm, &fm);
   infoheight = fm.lMaxBaselineExt + 4;

   width = DisplayWIDTH * windowsize;
   height = (DisplayHEIGHT + ViewWave) * windowsize;

   if (hwndFrame)
       WinSetWindowPos(hwndFrame, NULLHANDLE, (screenx - width - 2*xborderwidth)/2,
                                              (screeny - height- 2*yborderwidth - titleheight - menubarheight)/2,
                                              width + 2*xborderwidth,
                                              height + 2*yborderwidth + titleheight + menubarheight + infoheight,
                                              SWP_SIZE|SWP_MOVE|SWP_ACTIVATE|SWP_SHOW);

   SizeTheWindow(hwndFrame, windowsize);

   while (WinPeekMsg(hab, &qmsg, NULLHANDLE, 0, 0, PM_NOREMOVE))
       if (WinGetMsg(hab, &qmsg, NULLHANDLE, 0, 0))
            WinDispatchMsg(hab, &qmsg) ;

   return ((hwndFrame) ? TRUE : FALSE);
}


/* destroy window */
int CloseWindow()
{
   hini = PrfOpenProfile(hab, ColemIni); // save settings
   if (hini) // load saved settings
   {
      PrfWriteProfileData(hini,"colem","frt", &frt, sizeof(frt));
      PrfWriteProfileData(hini,"colem","maxskip", &maxskip, sizeof(maxskip));
      PrfWriteProfileData(hini,"colem","minskip", &minskip, sizeof(minskip));
      PrfWriteProfileData(hini,"colem","initskip", &initskip, sizeof(initskip));
      PrfWriteProfileData(hini,"colem","windowsize", &windowsize, sizeof(windowsize));
      PrfWriteProfileData(hini,"colem","volume", &volume, sizeof(volume));
      PrfWriteProfileData(hini,"colem","buflen", &buflen, sizeof(buflen));
      PrfWriteProfileData(hini,"colem","AUDIOfreq", &AUDIOfreq, sizeof(AUDIOfreq));
      PrfCloseProfile(hini);
   }
   if (hDive)     DiveClose (hDive);
   if (hwndFrame) WinDestroyWindow (hwndFrame) ;
   if (hmq)       WinDestroyMsgQueue (hmq) ;
   if (hab)       WinTerminate (hab) ;

   return TRUE;
}



MRESULT EXPENTRY ClientWndProc (HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
      int x, y;
      int i, i2;
      static HBITMAP hbm;
  //    static BITMAPINFOHEADER2 bmp;
      static RECTL          rcl;
      static RECTL          flip;
      char key;
      char vkey;
      WNDPARAMS *pwprm;
      HWND hwndFrame3;

      /* added w/dive routines */
      SIZEL              sizl;
      ULONG              ulcPaletteColors = 17;
      POINTL             pointl;         /* Point to offset from Desktop      */
      SWP                swp;            /* Window position                   */
      HRGN               hrgn;           /* Region handle                     */
      RECTL              rcls[50];       /* Rectangle coordinates             */
      RGNRECT            rgnCtl;         /* Processing control structure      */
      SETUP_BLITTER      SetupBlitter;   /* structure for DiveSetupBlitter    */
      static HPS         hpsMainHPS;

      PDRAGINFO pDragInfo;
      PDRAGITEM pDragItem;
      char dragname[CCHMAXPATH];

      STARTDATA   sd;
      PID         procID;
      ULONG       sessID;
      APIRET      rc;
      APIRET      arReturn;
      FILESTATUS3 fsStatus;
      ULONG  action; /* only used for DosOpen */

     switch (msg)
     {
        case WM_COMMAND:
           switch (COMMANDMSG(&msg)->cmd)
           {
              case IDM_NEWCART: ExitNow = 1;
                                newgame = TRUE;
                                break;

              case IDM_SETTINGS: // int db_volume, db_frequency, db_buffer; // audio
                                 // int db_frt, db_min, db_max, db_init, db_size; // video;
                                 db_frt = frt; // initialize to current settings
                                 db_min = maxskip; // min frame rate defined by maximum skip value
                                 db_max = minskip; // maximum frame rate defined by minimum skip value
                                 db_init = initskip;
                                 db_size = windowsize;
                                 db_volume = volume;
                                 db_buffer = buflen;
                                 db_frequency = AUDIOfreq;
                                 audio_quiet(); // quiet, we're thinking :-)
                                 WinDlgBox(HWND_DESKTOP, hwnd, DlgProc, NULLHANDLE, DB_MAIN, NULL);
                                 if (db_OK == TRUE)
                                 {
                                    frt = db_frt; // set to new settings
                                    maxskip = db_min;
                                    minskip = db_max;
                                    initskip = db_init;
                                    skip = initskip;
                                    if (windowsize != db_size)
                                    {
                                       SizeTheWindow(hwnd, db_size);
                                       windowsize = db_size;
                                    }
                                    if ((db_volume != volume) || (db_frequency != AUDIOfreq) || (db_buffer != buflen))
                                    {
                                       audio_close();
                                       if (db_volume != 0)
                                          audio_init(db_frequency, db_buffer);
                                       volume = db_volume;
                                       AUDIOfreq = db_frequency;
                                       buflen = db_buffer;
                                    }
                                 }
                                 break;

              case IDM_GAMEHELP:
                                audio_quiet(); // quiet, we're thinking :-)
                                arReturn = DosQueryPathInfo(carthelpfullfile, FIL_STANDARD, &fsStatus, sizeof(fsStatus));
                                if (arReturn != 0)
                                   WinMessageBox( HWND_DESKTOP, HWND_DESKTOP, (PSZ)carthelpfile, (PSZ)"No helpfile found for this game!", 0, MB_OK | MB_INFORMATION );
                                else
                                {
                                   sprintf(dragname, "\"%s\"", carthelpfullfile);
                                   JoystickOff();
                                   procID = 0;
                                   sessID = 0;
                                   memset(&sd, 0, sizeof(STARTDATA));
                                   sd.PgmName = "E.EXE";
                                   sd.PgmInputs = dragname;
                                   sd.SessionType = SSF_TYPE_PM;
                                   sd.Length = sizeof(STARTDATA);
                                   sd.Related = SSF_RELATED_INDEPENDENT;
                                   sd.FgBg = SSF_FGBG_FORE;
                                   sd.TraceOpt = SSF_TRACEOPT_NONE;
                                   sd.TermQ = 0;
                                   sd.Environment = NULL;
                                   sd.InheritOpt = SSF_INHERTOPT_PARENT;
                                   sd.PgmControl = SSF_CONTROL_VISIBLE | SSF_CONTROL_MAXIMIZE;
                                   sd.InitXPos = 10;
                                   sd.InitYPos = 10;
                                   sd.InitXSize = 400;
                                   sd.InitYSize = 600;
                                   sd.Reserved = 0;
                                   sd.ObjectBuffer = NULL;
                                   sd.ObjectBuffLen = 0;
                                   rc = DosStartSession(&sd, &sessID, &procID);
                                   DosSleep(5000);
                                   JoystickOn();
                                }
                                break;
              case IDM_COLEMHELP:
                                audio_quiet(); // quiet, we're thinking :-)
                                arReturn = DosQueryPathInfo(ColemHelp, FIL_STANDARD, &fsStatus, sizeof(fsStatus));
                                if (arReturn != 0)
                                   WinMessageBox( HWND_DESKTOP, HWND_DESKTOP, ColemHelp, (PSZ)"File is missing!", 0, MB_OK | MB_INFORMATION );
                                else
                                {
                                   sprintf(dragname, "\"%s\"", ColemHelp);
                                   JoystickOff();
                                   procID = 0;
                                   sessID = 0;
                                   memset(&sd, 0, sizeof(STARTDATA));
                                   sd.PgmName = "VIEW.EXE";
                                   sd.PgmInputs = dragname;
                                   sd.SessionType = SSF_TYPE_PM;
                                   sd.Length = sizeof(STARTDATA);
                                   sd.Related = SSF_RELATED_INDEPENDENT;
                                   sd.FgBg = SSF_FGBG_FORE;
                                   sd.TraceOpt = SSF_TRACEOPT_NONE;
                                   sd.TermQ = 0;
                                   sd.Environment = NULL;
                                   sd.InheritOpt = SSF_INHERTOPT_PARENT;
                                   sd.PgmControl = SSF_CONTROL_VISIBLE | SSF_CONTROL_MAXIMIZE;
                                   sd.InitXPos = 10;
                                   sd.InitYPos = 10;
                                   sd.InitXSize = 400;
                                   sd.InitYSize = 600;
                                   sd.Reserved = 0;
                                   sd.ObjectBuffer = NULL;
                                   sd.ObjectBuffLen = 0;
                                   rc = DosStartSession(&sd, &sessID, &procID);
                                   DosSleep(5000);
                                   JoystickOn();
                                }
                                break;
           }

          /* here we are creating the new window and setting up some DIVE information */
          case WM_CREATE:

             if (colem_active == FALSE)
             {
               hps = WinGetPS (hwnd) ;
               hbm = GpiLoadBitmap (hps, 0, START_BMP, 0L, 0L);
               WinReleasePS (hps) ;
             }

             sizl.cx = sizl.cy = 0;
             hdcClientHDC = WinOpenWindowDC (hwnd);
             hpsMainHPS = GpiCreatePS (hab, hdcClientHDC, &sizl,
                           PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC);

             hpalMain = GpiCreatePalette (hab,
                                    LCOL_PURECOLOR,
                                    LCOLF_CONSECRGB,
                                    ulcPaletteColors,
                                    (PULONG) Palette2);

             GpiSelectPalette (hpsMainHPS, hpalMain);
             WinRealizePalette (hwnd, hps, &ulcPaletteColors);

             DiveSetSourcePalette (hDive, 0, 17L, (PBYTE) Palette2);
             DiveSetDestinationPalette (hDive, 0, 17, 0 );

             WinQueryWindowRect (hwnd, &rcl);
             aptl[0].x = flip.xLeft;
             aptl[0].y = flip.yTop;
             aptl[1].x = flip.xRight;
             aptl[1].y = flip.yBottom;
             aptl[2].x = 0;
             aptl[2].y = 0;
             aptl[3].x = WIDTH;
             aptl[3].y = HEIGHT + ViewWave;

             WinQueryWindowRect (hwnd, &rcl);

             width = rcl.xRight;
             height = rcl.yTop - infoheight;
             aptl[0].x = 0;aptl[0].y = 0;
             aptl[1].x = width-1;aptl[1].y = 0;
             aptl[2].x = width-1;aptl[2].y = infoheight - 2;
             GpiSetColor(hpsClient, CLR_WHITE);
             GpiMove(hpsClient, &aptl[0]);
             GpiPolyLine(hpsClient, 2, &aptl[1]);

             aptl[1].x = 0;aptl[1].y = infoheight - 2;
             aptl[2].x = 0;aptl[2].y = 0;
             GpiSetColor(hpsClient, CLR_DARKGRAY);
             GpiPolyLine(hpsClient, 2, &aptl[1]);

             aptl[0].x = 0;         aptl[0].y = infoheight - 1;
             aptl[1].x = width - 1; aptl[1].y = infoheight - 1;
             GpiSetColor(hpsClient, CLR_PALEGRAY);
             GpiMove(hpsClient, &aptl[0]);
             GpiLine(hpsClient, &aptl[1]);

             GpiSetBackColor(hpsClient, CLR_PALEGRAY);
             GpiSetColor(hpsClient, CLR_PALEGRAY);

             ptl.x = 2; ptl.y = fm.lMaxDescender + 1;
             udrcl.xLeft = udrcl.yBottom = 1;
             udrcl.xRight = width-2; udrcl.yTop = infoheight-3;

             aptl[0].x = 1; aptl[0].y = 1;
             aptl[1].x = width - 2; aptl[1].y = infoheight - 3;
             GpiMove(hpsClient, &aptl[0]);
             GpiBox(hpsClient, DRO_FILL, &aptl[1], 0L, 0L);
             GpiSetColor(hpsClient, CLR_DARKBLUE);


             return 0 ;


          case WM_PAINT:
             if (colem_active)
             {
                WinBeginPaint (hwnd, hps, NULL);
                DiveBlitImage (hDive, ulDiveBufferNumber, DIVE_BUFFER_SCREEN );
                WinEndPaint (hps) ;
             }
             else
                if (hwnd == hwndClient)
                {
                   hps = WinBeginPaint (hwnd, NULLHANDLE, NULL) ;
                   GpiErase (hps) ;
                   WinQueryWindowRect (hwnd, &rcl) ;
                   rcl.yBottom = infoheight;
                   WinDrawBitmap (hps, hbm, NULL, (PPOINTL) &rcl,
                                CLR_BACKGROUND, CLR_NEUTRAL, DBM_STRETCH) ;
                   DosSleep(500);
                   WinEndPaint (hps) ;
                }
                else
                {
                   hps = WinBeginPaint (hwnd, NULLHANDLE, NULL) ;
                   GpiErase (hps) ;
                   WinEndPaint (hps) ;
                }
             WinQueryWindowRect (hwnd, &rcl);
             width = rcl.xRight;
             height = rcl.yTop - infoheight;
             aptl[0].x = 0;aptl[0].y = 0;
             aptl[1].x = width-1;aptl[1].y = 0;
             aptl[2].x = width-1;aptl[2].y = infoheight - 2;
             GpiSetColor(hpsClient, CLR_WHITE);
             GpiMove(hpsClient, &aptl[0]);
             GpiPolyLine(hpsClient, 2, &aptl[1]);
             aptl[1].x = 0;aptl[1].y = infoheight - 2;
             aptl[2].x = 0;aptl[2].y = 0;
             GpiSetColor(hpsClient, CLR_DARKGRAY);
             GpiPolyLine(hpsClient, 2, &aptl[1]);
             aptl[0].x = 0;         aptl[0].y = infoheight - 1;
             aptl[1].x = width - 1; aptl[1].y = infoheight - 1;
             GpiSetColor(hpsClient, CLR_PALEGRAY);
             GpiMove(hpsClient, &aptl[0]);
             GpiLine(hpsClient, &aptl[1]);
             GpiSetBackColor(hpsClient, CLR_PALEGRAY);
             GpiSetColor(hpsClient, CLR_PALEGRAY);
             ptl.x = 2; ptl.y = fm.lMaxDescender + 1;
             udrcl.xLeft = udrcl.yBottom = 1;
             udrcl.xRight = width-2; udrcl.yTop = infoheight-3;
             aptl[0].x = 1; aptl[0].y = 1;
             aptl[1].x = width - 2; aptl[1].y = infoheight - 3;
             GpiMove(hpsClient, &aptl[0]);
             GpiBox(hpsClient, DRO_FILL, &aptl[1], 0L, 0L);
             GpiSetColor(hpsClient, CLR_DARKBLUE);
             return 0 ;


          case WM_CLOSE:
             ExitNow = 1;
             return 0;

          case WM_SIZE:
             WinQueryWindowRect (hwnd, &rcl);
             width = rcl.xRight;
             height = rcl.yTop - infoheight;
             aptl[0].x = 0; aptl[0].y = 0;
             aptl[1].x = width-1;aptl[1].y = 0;
             aptl[2].x = width-1;aptl[2].y = infoheight - 2;
             GpiSetColor(hpsClient, CLR_WHITE);
             GpiMove(hpsClient, &aptl[0]);
             GpiPolyLine(hpsClient, 2, &aptl[1]);
             aptl[1].x = 0;aptl[1].y = infoheight - 2;
             aptl[2].x = 0;aptl[2].y = 0;
             GpiSetColor(hpsClient, CLR_DARKGRAY);
             GpiPolyLine(hpsClient, 2, &aptl[1]);
             aptl[0].x = 0;         aptl[0].y = infoheight - 1;
             aptl[1].x = width - 1; aptl[1].y = infoheight - 1;
             GpiSetColor(hpsClient, CLR_PALEGRAY);
             GpiMove(hpsClient, &aptl[0]);
             GpiLine(hpsClient, &aptl[1]);
             GpiSetBackColor(hpsClient, CLR_PALEGRAY);
             GpiSetColor(hpsClient, CLR_PALEGRAY);
             ptl.x = 2; ptl.y = fm.lMaxDescender + 1;
             udrcl.xLeft = udrcl.yBottom = 1;
             udrcl.xRight = width-2; udrcl.yTop = infoheight-3;
             GpiSetColor(hpsClient, CLR_DARKBLUE);

             return 0;


          case WM_CHAR:
             if (CHARMSG(&msg)->fs & KC_PREVDOWN)
                return 0;
             key  = SHORT1FROMMP(mp2);
             vkey = SHORT2FROMMP(mp2);
             if (CHARMSG(&msg)->fs & KC_KEYUP)
             {                                  /* key released */
                if (key || vkey)
                   {
                      if (vkey)
                         switch (key)
                         {
                            case '0': case '1': case '2': case '3':     /* player 1 keypad buttons */
                            case '4': case '5': case '6': case '7':
                            case '8': case '9':
                            case '.': case 13: if (key == '.') key = '0' + 10;
                                                else if (key == 13) key = '0' + 11;
                                                if ((KB[0] &0x000F) == (key - '0'))
                                                   KB[0] |= 0x000F; break;
                            default: switch (vkey)
                            {
                               case VK_DOWN:  KB[0] |= 0x0400;  break; /* player 1 move down  */
                               case VK_UP:    KB[0] |= 0x0100;  break; /*               up    */
                               case VK_LEFT:  KB[0] |= 0x0800;  break; /*               left  */
                               case VK_RIGHT: KB[0] |= 0x0200;  break; /*               right */
                               case VK_ESC:   ExitNow = 1;      break; /* Quit */
                            }
                         }
                      else
                         if (CHARMSG(&msg)->fs & KC_CTRL)
                            switch (key)
                            {
                               case '1': case '2': case '3':
                               case '4': case '5': case '6':
                               case '7': case '8': case '9':
                                  SizeTheWindow(hwnd, key - '0');    break;
                            }
                         else
                            switch(key)
                            {
                               case '+': case '=': if (maxskip > 0)  /* select faster framerate */
                                                      maxskip--;
                                                   if ((maxskip < skip) || (autofr == FALSE))
                                                      skip = maxskip;
                                                   /* force update of window title display */
                                                   titlecount = titletoggle = framecount = 0;
                                                   break;
                               case '-': case '_': if (CHAR4FROMMP (mp1) == 0x4A)
                                                   {
                                                      if ((KB[0] &0x000F) == 0x000b)
                                                      KB[0] |= 0x000F; break;
                                                   }
                                                   else
                                                   {
                                                      if (maxskip < 15)  /* select slower framerate */
                                                         maxskip++;
                                                      if (autofr == FALSE)
                                                         skip = maxskip;
                                                      /* force update of window title display */
                                                      titlecount = titletoggle = framecount = 0;
                                                   }
                                                   break;
                               case 'a': case 'A': autofr = (autofr) ? FALSE : TRUE;
                                                   /* force update of window title display */
                                                   titlecount = titletoggle = framecount = 0;
                                                   break;
                               case 'p': case 'P': game_paused = game_paused ? 0 : 1; break;
                               case 'g': case 'G': KB[1] |= 0x0400; break; /* player 2 move down   */
                               case 't': case 'T': KB[1] |= 0x0100; break; /*               up     */
                               case 'f': case 'F': KB[1] |= 0x0800; break; /*               left   */
                               case 'h': case 'H': KB[1] |= 0x0200; break; /*               right  */
                               case 'z': case 'Z': KB[1] |= 0x4000; break; /*               fire 1 */
                               case 'x': case 'X': KB[1] |= 0x0040; break; /*               fire 2 */
                               case 'n': case 'N': ExitNow = 1;
                                                   newgame = TRUE;
                                                   break;
                               case '>': case '.': KB[0] |= 0x4000; break; /* player 1 fire 1 */
                               case '/': case '?': KB[0] |= 0x0040; break; /*          fire 2 */
                               case '0': case '1': case '2': case '3':     /* player 2 keypad buttons */
                               case '4': case '5': case '6': case '7':     /* Q = * */
                               case '8': case '9': case 'q': case 'Q':     /* W = # */
                               case 'w': case 'W': if ((key == 'q') || (key == 'Q')) key = '0' + 10;
                                                   else if ((key == 'w') || (key == 'W')) key = '0' + 11;
                                                   if ((KB[1] &0x000F) == (key - '0'))
                                                      KB[1] |= 0x000F; break;
                               case '*': if ((KB[0] &0x000F) == 0x000a)
                                                      KB[0] |= 0x000F; break;
                               case 'o': case 'O': ViewWave = 8 - ViewWave;
                                                   SizeTheWindow(hwnd, 0);
                                                   break;
                            }
                   }
             }
             else
             {                                  /* key pressed */
                if (key || vkey)
                   {
                      if (vkey)
                         switch(key)
                         {
                            case '0': case '1': case '2': case '3':     /* player 1 keypad buttons */
                            case '4': case '5': case '6': case '7':
                            case '8': case '9':
                            case '.': case 13: if (key == '.') key = '0' + 10;
                                                else if (key == 13) key = '0' + 11;
                                                KB[0] = (KB[0] & 0xFFF0) | (key - '0'); break;
                            default: switch (vkey)
                            {
                               case VK_DOWN:  KB[0] &= 0xFBFF;  break; /* player 1 move down  */
                               case VK_UP:    KB[0] &= 0xFEFF;  break; /*               up    */
                               case VK_LEFT:  KB[0] &= 0xF7FF;  break; /*               left  */
                               case VK_RIGHT: KB[0] &= 0xFDFF;  break; /*               right */
                            }
                         }
                      else
                             if (CHARMSG(&msg)->fs & KC_CTRL)
                            switch (key)
                            {
                               case '1': case '2': case '3':
                               case '4': case '5': case '6':
                               case '7': case '8': case '9':
                                  SizeTheWindow(hwnd, key - '0');    break;
                            }
                         else
                            switch(key)
                            {
                               case 'g': case 'G': KB[1] &= 0xFBFF; break; /* player 2 move down   */
                               case 't': case 'T': KB[1] &= 0xFEFF; break; /*               up     */
                               case 'f': case 'F': KB[1] &= 0xF7FF; break; /*               left   */
                               case 'h': case 'H': KB[1] &= 0xFDFF; break; /*               right  */
                               case 'z': case 'Z': KB[1] &= 0xBFFF; break; /*               fire 1 */
                               case 'x': case 'X': KB[1] &= 0xFFBF; break; /*               fire 2 */
                               case '>': case '.': KB[0] &= 0xBFFF; break; /* player 1 fire 1 */
                               case '/': case '?': KB[0] &= 0xFFBF; break; /*          fire 2 */
                               case '0': case '1': case '2': case '3':     /* player 2 keypad buttons */
                               case '4': case '5': case '6': case '7':
                               case '8': case '9': case 'q': case 'Q':
                               case 'w': case 'W': if ((key == 'q') || (key == 'Q')) key = '0' + 10;
                                                   else if ((key == 'w') || (key == 'W')) key = '0' + 11;
                                                   KB[1] = (KB[1] & 0xFFF0) | (key - '0'); break;
                               case '*': KB[0] = ( KB[0] & 0xFFF0) | 0x000a; break;
                               case '-': if (CHAR4FROMMP (mp1) == 0x4A)
                                            KB[0] = ( KB[0] & 0xFFF0) | 0x000b;
                                         break;

                            }
                   }
             }
             return 0;


          case DM_DRAGOVER:
             pDragInfo = (PDRAGINFO) mp1;
             DrgAccessDraginfo(pDragInfo);
             pDragItem = DrgQueryDragitemPtr(pDragInfo, 0);
             DrgQueryStrName(pDragItem->hstrSourceName, CCHMAXPATH, dragname);
             /* exclude multi select & not snapshot/tap */
             if((pDragInfo->cditem > 1) ||
                (different_extension(".rom", dragname) && different_extension(".ROM", dragname)))
                return (MPFROM2SHORT (DOR_NEVERDROP, DO_UNKNOWN));
             else
                return (MPFROM2SHORT (DOR_DROP, DO_UNKNOWN));
             break;

          case DM_DROP:
            pDragInfo = (PDRAGINFO)mp1;
            DrgAccessDraginfo(pDragInfo);
            pDragItem = DrgQueryDragitemPtr(pDragInfo, 0);
            DrgQueryStrName(pDragItem->hstrContainerName, CCHMAXPATH, dragname);
            DrgQueryStrName(pDragItem->hstrSourceName, CCHMAXPATH, dragname+strlen(dragname));

            strcpy(dragfilename, dragname);
            newgame = TRUE;
            ExitNow = 1;
            WinSetFocus(HWND_DESKTOP, hwnd);
            break;


          case WM_SETWINDOWPARAMS:
               pwprm = (PWNDPARAMS) PVOIDFROMMP (mp1);
               if (pwprm->fsStatus & WPM_TEXT)
                  strcpy(pwprm->pszText, szTitle);
               return MRFROMSHORT(1);



          case WM_ACTIVATE:
              game_paused = (mp1) ? (game_paused & 1) : (game_paused | 2);
               return 0;



          case WM_DESTROY:
             ExitNow = 1;
           //  free (pbmi);
             if (hpsMainHPS  != NULLHANDLE)
               {
               GpiSelectPalette (hpsMainHPS, NULLHANDLE);
               };
             if (hpalMain != NULLHANDLE)
               {
               GpiDeletePalette (hpalMain);
               };
             if (hpsMainHPS != NULLHANDLE)  GpiDestroyPS (hpsMainHPS);
             return 0 ;



          case WM_REALIZEPALETTE:
               /* This tells DIVE that the physical palette may have changed. */
               DiveSetDestinationPalette (hDive, 0, 17, 0 );
               break;



          case WM_VRNDISABLED:
               DiveSetupBlitter (hDive, 0);
               break;



          case WM_VRNENABLED:
               if ( !hpsMainHPS )
                  break;
               hrgn = GpiCreateRegion ( hpsMainHPS, 0L, NULL );
               if ( hrgn )
               {
/*                  // NOTE: If mp1 is zero, then this was just a move message.
                  // Illustrate the visible region on a WM_VRNENABLE.
                  //
*/                WinQueryVisibleRegion ( hwnd, hrgn );
                  rgnCtl.ircStart     = 0;
                  rgnCtl.crc          = 50;
                  rgnCtl.ulDirection  = 1;

                  /* Get the all ORed rectangles */
                  if ( GpiQueryRegionRects ( hpsMainHPS, hrgn, NULL, &rgnCtl, rcls) )
                  {

                     /* Now find the window position and size, relative to parent. */
                     WinQueryWindowPos (hwndClient, &swp );

                     /* Convert the point to offset from desktop lower left. */
                     pointl.x = swp.x;
                     pointl.y = swp.y;
                     WinMapWindowPoints ( hwndFrame, HWND_DESKTOP, &pointl, 1 );

                     /* Tell DIVE about the new settings. */
                     SetupBlitter.ulStructLen = sizeof ( SETUP_BLITTER );
                     SetupBlitter.fccSrcColorFormat = FOURCC_LUT8;
                     SetupBlitter.ulSrcWidth   = DisplayWIDTH;
                     SetupBlitter.ulSrcHeight  = DisplayHEIGHT+ViewWave;
                     SetupBlitter.ulSrcPosX    = 8;
                     SetupBlitter.ulSrcPosY    = 8;
                     SetupBlitter.fInvert      = FALSE;
                     SetupBlitter.ulDitherType = 1;

                     SetupBlitter.fccDstColorFormat = FOURCC_SCRN;
                     SetupBlitter.ulDstWidth        = swp.cx;
                     SetupBlitter.ulDstHeight       = swp.cy - infoheight;
                     SetupBlitter.lDstPosX          = 0;
                     SetupBlitter.lDstPosY          = infoheight;
                     SetupBlitter.lScreenPosX       = pointl.x;
                     SetupBlitter.lScreenPosY       = pointl.y;
                     SetupBlitter.ulNumDstRects     = rgnCtl.crcReturned;
                     SetupBlitter.pVisDstRects      = rcls;
                     DiveSetupBlitter (hDive, &SetupBlitter);
                  }
                  else
                     DiveSetupBlitter (hDive, 0);

                  GpiDestroyRegion (hpsMainHPS, hrgn);
               }
               break;
     }
     return WinDefWindowProc (hwnd, msg, mp1, mp2) ;
}



MRESULT EXPENTRY DlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
      char dragname[CCHMAXPATH];
      STARTDATA   sd;
      PID         procID;
      ULONG       sessID;
      APIRET      rc;
      APIRET      arReturn;
      FILESTATUS3 fsStatus;

   switch(msg)
   {
      case WM_INITDLG:   // initialize all the sliders and numeric displays
         if (db_frt == 0)
            WinCheckButton(hwnd, DB_VR_NTSC, 1);
         else
            WinCheckButton(hwnd, DB_VR_PAL, 1);

         WinCheckButton(hwnd, DB_AF_8 + (int) (db_frequency / 11025), 1);
         WinCheckButton(hwnd, DB_AV_OFF + (int) db_volume, 1);

         WinSetDlgItemShort(hwnd, DB_BS_BT, db_buffer, TRUE);                    // init buffer size text
         WinSendDlgItemMsg(hwnd, DB_BS_B, SLM_SETSLIDERINFO,                     // init buffer size slider position
              MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((db_buffer - 256) / 32) );

         WinSetDlgItemShort(hwnd, DB_FPS_MIN_T, fps[db_frt][db_min] + .5, TRUE); // init min text
         WinSendDlgItemMsg(hwnd, DB_FPS_MIN, SLM_SETSLIDERINFO,                  // init min slider position
              MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_min) * 5) );

         WinSetDlgItemShort(hwnd, DB_FPS_MAX_T, fps[db_frt][db_max] + .5, TRUE); // init min text
         WinSendDlgItemMsg(hwnd, DB_FPS_MAX, SLM_SETSLIDERINFO,                  // init min slider position
              MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_max) * 5) );

         WinSetDlgItemShort(hwnd, DB_FPS_INIT_T, fps[db_frt][db_init] + .5, TRUE);       // init min text
         WinSendDlgItemMsg(hwnd, DB_FPS_INIT, SLM_SETSLIDERINFO,                   // init min slider position
              MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_init) * 5) );

         WinSetDlgItemShort(hwnd, DB_IWS_T, db_size, TRUE);       // init window size text
         WinSendDlgItemMsg(hwnd, DB_IWS, SLM_SETSLIDERINFO,                   // init window size slider position
              MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((db_size - 1) * 7) );

         return 0;

      case WM_COMMAND:
         switch(SHORT1FROMMP(mp1))
         {
            case DB_J_REC: JoystickInit(1,ColemIni);
                         WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
                            "The Joystick calibration information has been reset.  Once back in ColEm you will need to move your joystick to all four corners to recalibrate!",
                            "Joysticks Reset" , 0, MB_OK | MB_INFORMATION );
                         return 0;

            case DB_AF_DFLT: db_volume = 1;
                         db_frequency = 22050;
                         db_buffer = 768;
                         WinCheckButton(hwnd, DB_AF_8 + (int) (db_frequency / 11025), 1);
                         WinCheckButton(hwnd, DB_AV_OFF + (int) db_volume, 1);
                         WinSetDlgItemShort(hwnd, DB_BS_BT, db_buffer, TRUE);                    // init buffer size text
                         WinSendDlgItemMsg(hwnd, DB_BS_B, SLM_SETSLIDERINFO,                     // init buffer size slider position
                            MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((db_buffer - 256) / 32) );
                         return 0;

            case DB_HELP: arReturn = DosQueryPathInfo(ColemHelp, FIL_STANDARD, &fsStatus, sizeof(fsStatus));
                         if (arReturn != 0)
                            WinMessageBox( HWND_DESKTOP, HWND_DESKTOP, ColemHelp, (PSZ)"File is missing!", 0, MB_OK | MB_INFORMATION );
                         else
                         {
                            sprintf(dragname, "\"%s\" \"settings dialog\"", ColemHelp);
                            JoystickOff();
                            procID = 0;
                            sessID = 0;
                            memset(&sd, 0, sizeof(STARTDATA));
                            sd.PgmName = "VIEW.EXE";
                            sd.PgmInputs = dragname; // ColemHelp;
                            sd.SessionType = SSF_TYPE_PM;
                            sd.Length = sizeof(STARTDATA);
                            sd.Related = SSF_RELATED_INDEPENDENT;
                            sd.FgBg = SSF_FGBG_FORE;
                            sd.TraceOpt = SSF_TRACEOPT_NONE;
                            sd.TermQ = 0;
                            sd.Environment = NULL;
                            sd.InheritOpt = SSF_INHERTOPT_PARENT;
                            sd.PgmControl = SSF_CONTROL_VISIBLE | SSF_CONTROL_MAXIMIZE;
                            sd.InitXPos = 10;
                            sd.InitYPos = 10;
                            sd.InitXSize = 400;
                            sd.InitYSize = 600;
                            sd.Reserved = 0;
                            sd.ObjectBuffer = NULL;
                            sd.ObjectBuffLen = 0;
                            rc = DosStartSession(&sd, &sessID, &procID);
                            DosSleep(5000);
                            JoystickOn();
                         }
                         return 0;

            case DB_CAN: db_OK = FALSE;
                         WinDismissDlg(hwnd, TRUE);
                         return 0;

            case DB_OK:  db_OK = TRUE;
                         WinDismissDlg(hwnd, TRUE);
                         return 0;
         }

      case WM_CONTROL:
         switch(SHORT1FROMMP(mp1))
         {
            case DB_VR_NTSC: db_frt = 0;
                             WinSetDlgItemShort(hwnd, DB_FPS_MIN_T, fps[db_frt][db_min] + .5, TRUE);
                             WinSetDlgItemShort(hwnd, DB_FPS_MAX_T, fps[db_frt][db_max] + .5, TRUE);
                             WinSetDlgItemShort(hwnd, DB_FPS_INIT_T, fps[db_frt][db_init] + .5, TRUE);
                             return 0;

            case DB_VR_PAL:  db_frt = 1;
                             WinSetDlgItemShort(hwnd, DB_FPS_MIN_T, fps[db_frt][db_min] + .5, TRUE);
                             WinSetDlgItemShort(hwnd, DB_FPS_MAX_T, fps[db_frt][db_max] + .5, TRUE);
                             WinSetDlgItemShort(hwnd, DB_FPS_INIT_T, fps[db_frt][db_init] + .5, TRUE);
                             return 0;

            case DB_AV_OFF:  db_volume = 0; return 0;
            case DB_AV_ON:   db_volume = 1; return 0;
//            case DB_AV_HI:   db_volume = 2; return 0;
            case DB_AF_8:    db_frequency =  8000; return 0;
            case DB_AF_11:   db_frequency =  11025; return 0;
            case DB_AF_22:   db_frequency =  22050; return 0;
            case DB_AF_33:   db_frequency =  33075; return 0;
            case DB_AF_44:   db_frequency =  44100; return 0;

            case DB_BS_B:    if ((SHORT2FROMMP(mp1) == SLN_CHANGE) || (SHORT2FROMMP(mp1) == SLN_SLIDERTRACK))
                             {
                                db_buffer = (int) mp2 * 32 + 256;
                                WinSetDlgItemShort(hwnd, DB_BS_BT, db_buffer, TRUE);
                             }
                          return 0;

            case DB_FPS_MIN: if ((SHORT2FROMMP(mp1) == SLN_CHANGE) || (SHORT2FROMMP(mp1) == SLN_SLIDERTRACK))
                             {
                                db_min = 15 - (int) mp2 / 5;
                                WinSetDlgItemShort(hwnd, DB_FPS_MIN_T, fps[db_frt][db_min] + .5, TRUE);
                                if (db_min < db_max)
                                {
                                   db_max = db_min;
                                   WinSetDlgItemShort(hwnd, DB_FPS_MAX_T, fps[db_frt][db_max] + .5, TRUE);       // init min text
                                   WinSendDlgItemMsg(hwnd, DB_FPS_MAX, SLM_SETSLIDERINFO,                   // init min slider position
                                      MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_max) * 5) );
                                }
                                if (db_min < db_init)
                                {
                                   db_init = db_min;
                                   WinSetDlgItemShort(hwnd, DB_FPS_INIT_T, fps[db_frt][db_init] + .5, TRUE);       // init min text
                                   WinSendDlgItemMsg(hwnd, DB_FPS_INIT, SLM_SETSLIDERINFO,                   // init min slider position
                                      MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_init) * 5) );
                                }
                             }
                             return 0;

            case DB_FPS_MAX: if ((SHORT2FROMMP(mp1) == SLN_CHANGE) || (SHORT2FROMMP(mp1) == SLN_SLIDERTRACK))
                             {
                                db_max = 15 - (int) mp2 / 5;
                                WinSetDlgItemShort(hwnd, DB_FPS_MAX_T, fps[db_frt][db_max] + .5, TRUE);
                                if (db_max > db_min)
                                {
                                   db_min = db_max;
                                   WinSetDlgItemShort(hwnd, DB_FPS_MIN_T, fps[db_frt][db_min] + .5, TRUE);       // init min text
                                   WinSendDlgItemMsg(hwnd, DB_FPS_MIN, SLM_SETSLIDERINFO,                   // init min slider position
                                      MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_min) * 5) );
                                }
                                if (db_max > db_init)
                                {
                                   db_init = db_max;
                                   WinSetDlgItemShort(hwnd, DB_FPS_INIT_T, fps[db_frt][db_init] + .5, TRUE);       // init min text
                                   WinSendDlgItemMsg(hwnd, DB_FPS_INIT, SLM_SETSLIDERINFO,                   // init min slider position
                                      MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_init) * 5) );
                                }
                             }
                             return 0;

            case DB_FPS_INIT: if ((SHORT2FROMMP(mp1) == SLN_CHANGE) || (SHORT2FROMMP(mp1) == SLN_SLIDERTRACK))
                             {
                                db_init = 15 - (int) mp2 / 5;
                                WinSetDlgItemShort(hwnd, DB_FPS_INIT_T, fps[db_frt][db_init] + .5, TRUE);
                                if (db_max > db_init)
                                {
                                   db_max = db_init;
                                   WinSetDlgItemShort(hwnd, DB_FPS_MAX_T, fps[db_frt][db_max] + .5, TRUE);       // init min text
                                   WinSendDlgItemMsg(hwnd, DB_FPS_MAX, SLM_SETSLIDERINFO,                   // init min slider position
                                      MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_max) * 5) );
                                }
                                if (db_min < db_init)
                                {
                                   db_min = db_init;
                                   WinSetDlgItemShort(hwnd, DB_FPS_MIN_T, fps[db_frt][db_min] + .5, TRUE);       // init min text
                                   WinSendDlgItemMsg(hwnd, DB_FPS_MIN, SLM_SETSLIDERINFO,                   // init min slider position
                                      MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG((15 - db_min) * 5) );
                                }
                             }
                             return 0;

            case DB_IWS:     if ((SHORT2FROMMP(mp1) == SLN_CHANGE) || (SHORT2FROMMP(mp1) == SLN_SLIDERTRACK))
                             {
                                db_size = (int) (mp2) / 7 + 1;
                                WinSetDlgItemShort(hwnd, DB_IWS_T, db_size, TRUE);
                             }
                             return 0;

            }
      return 0;
   }
   return WinDefDlgProc(hwnd, msg, mp1, mp2);
}

#include "common.h"
