/* ices.c
 * - Main Program *
 * Copyright (c) 2000 Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "definitions.h"
#include "sysctrl.h"
#include <debug.h>

ices_config_t ices_config;

static void _setProgDir(char *pszFullFName)
{
  // Change a current directory to the program's directory.
  char *pcSlash = strrchr( pszFullFName, '\\' );

  if ( pcSlash != NULL )
  {
    char acPath[_MAX_PATH];
    int  cbPath = pcSlash - pszFullFName;

    memcpy( acPath, pszFullFName, cbPath );
    acPath[cbPath] = '\0';

    chdir( acPath );
  }
}

/* Global function definitions */

/* Ices starts here. But ends in ices_setup_shutdown(),
 * which calls exit(0) and that's the last you see of it. */


#ifdef __NT__

#include <windows.h>

void serviceMain(int argc, char **argv) {
#else
void main(int argc, char **argv) {

  setbuf( stdout, NULL );
  setbuf( stderr, NULL );

  // Change current directory to ices.exe's directory.
  _setProgDir( argv[0] );
  debugInit();

	/* Make command line arguments available through
	 * ices_util_get_argc() and argv */
	ices_util_set_args(argc, argv);
#endif

	/* Setup all options, and initialize all s`ubmodules */
	ices_setup_initialize();

	/* Connect to server and keep streaming all the good stuff */
	ices_stream_loop(&ices_config);

	/* Not reached */
	return;
}


#ifdef __NT__

SERVICE_STATUS         serviceStatus;
SERVICE_STATUS_HANDLE  hServiceStatus;

void serviceExit(void)
{
  if ( serviceStatus.dwCurrentState != SERVICE_STOPPED )
  {
    serviceStatus.dwCurrentState = SERVICE_STOPPED;
    SetServiceStatus( hServiceStatus, &serviceStatus );
  }
}

void serviceControl(DWORD request)
{
  PSZ        pszMsg = "Shutdown";

  switch( request )
  {
    case SERVICE_CONTROL_STOP:
      pszMsg = "Stop";

    case SERVICE_CONTROL_SHUTDOWN:
      ices_log( "%s service", pszMsg );

      serviceStatus.dwCurrentState = SERVICE_STOP_PENDING;
      SetServiceStatus( hServiceStatus, &serviceStatus );
      ices_sysctrl_post( SEMID_STOP );
      break;

    case ICES_SERVICE_CONTROL_NEXT:
      ices_sysctrl_post( SEMID_NEXT );
      break;

    case ICES_SERVICE_CONTROL_RELOAD:
      ices_sysctrl_post( SEMID_RELOAD );
      break;

    case ICES_SERVICE_CONTROL_ROTATE:
      ices_sysctrl_post( SEMID_ROTATE );
      break;

    default:
    case SERVICE_CONTROL_INTERROGATE: 
      SetServiceStatus( hServiceStatus, &serviceStatus );
      break;
  }

  return;
}

//void serviceRun(int argc, char **argv)
VOID WINAPI serviceRun(DWORD argc, LPTSTR *argv)
{
  serviceStatus.dwServiceType      = SERVICE_WIN32_OWN_PROCESS;
  serviceStatus.dwCurrentState     = SERVICE_START_PENDING;
  serviceStatus.dwControlsAccepted = SERVICE_ACCEPT_STOP |
                                     SERVICE_ACCEPT_SHUTDOWN;
  serviceStatus.dwWin32ExitCode    = 0;
  serviceStatus.dwServiceSpecificExitCode = 0;
  serviceStatus.dwCheckPoint       = 0;
  serviceStatus.dwWaitHint         = 0;
  SetServiceStatus( hServiceStatus, &serviceStatus );

  hServiceStatus = RegisterServiceCtrlHandler( ICES_SERVICE_NAME,
                                          (LPHANDLER_FUNCTION)serviceControl );
  if ( hServiceStatus == (SERVICE_STATUS_HANDLE)0 )
  {
    debug( "RegisterServiceCtrlHandler() failed" );
    return;
  }

  atexit( serviceExit );

  debugCP( "Set service status to SERVICE_RUNNING..." );
  serviceStatus.dwCurrentState = SERVICE_RUNNING;
  SetServiceStatus( hServiceStatus, &serviceStatus );

  serviceMain( argc, argv );
}

void main(int argc, char **argv)
{
  SERVICE_TABLE_ENTRY aDispatcherTable[] =
  { { ICES_SERVICE_NAME, serviceRun },
    { NULL, NULL } };

  _setProgDir( argv[0] );
  debugInit();
	ices_util_set_args(argc, argv);

  if ( !StartServiceCtrlDispatcher( aDispatcherTable ) )
  {
    debug( "StartServiceCtrlDispatcher: Error %ld, run as cli program...",
           GetLastError() );
    serviceMain( argc, argv );
    return;
  }
}

#endif
