/* playlist_text.c
 * - Module for simple playlist
 * Copyright (c) 2000 Alexander Havng
 * Copyright (c) 2001-4 Brendan Cully
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <definitions.h>
#include <direct.h>
#include <time.h>
#include "..\util.h"
#include "..\xpl.h"
#include <debug.h>

#define PLEXT_SCAN_LIST          "sl$"
#define PLEXT_SCAN_RESULT        "sr$"
#define PLEXT_RANDOM             "r$$"

extern ices_config_t ices_config;

static FILE   *fpPlaylist = NULL;
static time_t timePlaylist = 0;
static int    iLineNo;
static HMTX   hmtxScanResultFile = NULLHANDLE;

// When fAutoScan is TRUE, hevStop posted and fReload is TRUE the list of
// files will be updated.
// When fAutoScan is FALSE and fReload is TRUE the configured playlist will
// be reopened.
// In other words, flag fReload is used in thread when we work with local path
// and it is used in plGetNext() when we works with configured playlist.
static BOOL   fAutoScan;            // Path specified instead a playlist file.
static HEV    hevStop = NULLHANDLE; // Reload or exit scan thread.
static BOOL   fReload = FALSE;      // Reload, do not exit scan thread.


static BOOL _getPLFileName(PCHAR pcBuf, ULONG cbBuf, PSZ pszExt)
{
  int        iLen;

  if ( ices_config.instance_id == NULL )
    iLen = _snprintf( pcBuf, cbBuf, "%s\\pl.%s",
                      ices_config.base_directory, pszExt );
  else
    iLen = _snprintf( pcBuf, cbBuf, "%s\\pl_%s.%s",
                      ices_config.base_directory, ices_config.instance_id,
                      pszExt );

  return iLen != -1;
}

// Shuffles lines from fpIn and writes result to pfOut.
static VOID _shuffleList(FILE *fpIn, FILE *pfOut)
{
  struct _SLLINE {
    struct _SLLINE *pNext;
    CHAR           acLine[1];
  }          *pList = NULL,
             *pLine, *pSwap,
             **pTable;
  CHAR       acBuf[1024];
  ULONG      cbBuf;
  ULONG      ulLines = 0;

  while( fgets( acBuf, sizeof(acBuf), fpIn ) != NULL )
  {
    cbBuf = strlen( acBuf );
    pLine = debugMAlloc( sizeof(struct _SLLINE) + cbBuf );
    if ( pLine == NULL )
      break;
    strcpy( pLine->acLine, acBuf );

    pLine->pNext = pList;
    pList = pLine;
    ulLines++;
  }

  pTable = debugMAlloc( ulLines * sizeof(struct _SLLINE *) );
  if ( pTable != NULL )
  {
    ULONG    ulIdx, ulSwapIdx;

    for( pLine = pList, ulIdx = 0; pLine != NULL; pLine = pLine->pNext, ulIdx++ )
      pTable[ulIdx] = pLine;

    srand( time( NULL ) );
    for( ulIdx = 0; ulIdx < ulLines; ulIdx++ )
    {
      ulSwapIdx = ( (rand() << 15) | rand() ) % ulLines;

      pSwap = pTable[ulSwapIdx];
      pTable[ulSwapIdx] = pTable[ulIdx];
      pTable[ulIdx] = pSwap;
    }

    while( ulLines > 0 )
    {
      ulLines--;
      fputs( pTable[ulLines]->acLine, pfOut );
    }

    debugFree( pTable );
  }

  while( pList != NULL )
  {
    pSwap = pList->pNext;
    debugFree( pList );
    pList = pSwap;
  }
}

static BOOL _reopenPlaylist(PSZ pszPlaylist, BOOL fRandomize, int *piLineNo)
{
  ices_log_debug( "Reopen playlist %s", pszPlaylist );

  if ( fpPlaylist != NULL )
    fclose( fpPlaylist );

  fpPlaylist = fopen( pszPlaylist, "r" );
  if ( fpPlaylist == NULL )
  {
		ices_log_error( "Could not open playlist file: %s", pszPlaylist );
    return FALSE;
  }

  if ( fRandomize )
  {
    FILE     *pfRandom;
    CHAR     acRandomFName[_MAX_PATH];

    debug( "Shuffle playlist..." );

    if ( !_getPLFileName( acRandomFName, sizeof(acRandomFName), PLEXT_RANDOM ) )
    {
      debug( "_getPLFileName() failed" );
      return FALSE;
    }

    // Building a shuffled list.

    pfRandom = fopen( acRandomFName, "w+" );
    if ( pfRandom == NULL )
    {
      ices_log_error( "Could not create file: %s", acRandomFName );
      return FALSE;
    }

    _shuffleList( fpPlaylist, pfRandom );

    fclose( fpPlaylist );
    fclose( pfRandom );

    // Open shuffled list.
    return _reopenPlaylist( acRandomFName, FALSE, NULL );
  }

  // Skip playlist up to current pointed line numbmer.

  if ( piLineNo != NULL )
  {
    CHAR     acBuf[_MAX_PATH];
    int      iIdx;

    debug( "Skip %u records of the playlist %s...", *piLineNo, pszPlaylist );
    for( iIdx = 0;
         ( iIdx < *piLineNo ) &&
         ( fgets( acBuf, sizeof(acBuf), fpPlaylist ) != NULL );
         iIdx++ );

    *piLineNo = iIdx;
  }

  return TRUE;
}


/* Directory scan thread */

static VOID _scanDir(FILE *fpOut, PCHAR pcPath, PSZ pszExtList)
{
  DIR           *pDir;
  struct dirent *pDirEnt;
  PCHAR         pcPathEnd = strchr( pcPath, '\0' );

  strcpy( pcPathEnd, "\\*" );
  pcPathEnd++;

  pDir = opendir( pcPath );
  if ( pDir != NULL )
  {
    do
    {
      pDirEnt = readdir( pDir ); 
      if ( pDirEnt == NULL )
        break;

      if ( ( *((PUSHORT)&pDirEnt->d_name) == '\0.' ) ||
           ( ( *((PUSHORT)&pDirEnt->d_name) == '..' ) &&
             ( pDirEnt->d_name[2] == '\0' ) ) )
        continue;

      strcpy( pcPathEnd, pDirEnt->d_name );

      if ( (pDirEnt->d_attr & _A_SUBDIR) != 0 )
        _scanDir( fpOut, pcPath, pszExtList );
      else
      {
        // Store the file name.

        BOOL fStore;
        PSZ  pszFExt = strrchr( pDirEnt->d_name, '.' );
          
        if ( ( pszExtList == NULL ) ||
             ( ices_util_word_index( pszExtList,
                                     pszFExt == NULL ? "." : &pszFExt[1] )
               != -1 ) )
        {
          // List of file extensions is not specified or extension is listed.
          strcpy( pcPathEnd, pDirEnt->d_name );
          fprintf( fpOut, "%s\n", pcPath );
        }
      }
    }
    while( xplEventWait( hevStop, 0 ) == XPL_ERROR_TIMEOUT );

    closedir( pDir ); 
  }
}

static VOID _updatePlaylist(FILE *fpList, PSZ pszListFName)
{
  CHAR                acResultFName[_MAX_PATH];
  CHAR                acBufList[_MAX_PATH];
  CHAR                acBufResult[_MAX_PATH];
  FILE                *fpResult;
  BOOL                fChanged = FALSE;

  xplMutexLock( hmtxScanResultFile, XPL_INDEFINITE_WAIT );

  if ( !_getPLFileName( acResultFName, sizeof(acResultFName),
                        PLEXT_SCAN_RESULT ) )
    debug( "_getPLFileName() failed" );
  else
  {
    struct stat        stStat;

    fseek( fpList, 0, SEEK_END );

    if ( ( stat( acResultFName, &stStat ) != -1 ) &&
         ( ftell( fpList ) != stStat.st_size ) )
    {
      debug( "Size of current list of files %s is not equal to playlist %s",
             pszListFName, acResultFName );
      fChanged = TRUE;
    }
    else
    {
      fpResult = fopen( acResultFName, "r" );
      if ( fpResult == NULL )
        fChanged = TRUE;
      else
      {
        debug( "Compare current list of files %s with playlist %s...",
               pszListFName, acResultFName );
        rewind( fpList );

        while( xplEventWait( hevStop, 0 ) == XPL_ERROR_TIMEOUT )
        {
          if ( fgets( acBufList, sizeof(acBufList), fpList ) == NULL )
          {
            if ( !feof( fpList ) )
              debug( "fgets(,,fpList) failed" );
            break;
          }

          if ( fgets( acBufResult, sizeof(acBufResult), fpResult ) == NULL )
          {
            debug( "fgets(,,fpList) failed" );
            fChanged = TRUE;
            break;
          }

          if ( stricmp( acBufList, acBufResult ) != 0 )
          {
            debug( "Changes found" );
            fChanged = TRUE;
            break;
          }
        }

        fclose( fpResult );
      } // if ( fpList == NULL ) else
    }
  }

  fclose( fpList );

  if ( fChanged )
  {
    ices_log_debug( "Replace scan result playlist %s with current list %s...",
                    acResultFName, pszListFName );

    if ( fpPlaylist != NULL )
    {
      // If randomize playlist is NOT used, current playlist file fpPlaylist is
      // scan-result file and it open now in main thread. It must be closed to
      // rename current direcory list file to scan-result file. Scan-result
      // file will be opened again as current playlist in plGetNext().
      fclose( fpPlaylist );
      fpPlaylist = NULL;
    }

    if ( unlink( acResultFName ) != 0 )
      debug( "unlink(%s) failed", acResultFName );

    if ( rename( pszListFName, acResultFName ) != 0 )
    {
      fChanged = FALSE;
      debug( "rename(%s,%s) failed", pszListFName, acResultFName );
      ices_log_error_output( "Could not rename %s to %s",
                             pszListFName, acResultFName );
    }
  }
  else
    ices_log_debug( "Playlist was not chaged" );

  xplMutexUnlock( hmtxScanResultFile );

  if ( !fChanged && ( unlink( pszListFName ) != 0 ) )
    debug( "unlink(%s) failed", pszListFName );
}

static BOOL _makePlaylist(playlist_module_t *pPLMod)
{
  CHAR                acPath[_MAX_PATH];
  CHAR                acListFName[_MAX_PATH];
  FILE                *fpList;

  if ( !_getPLFileName( acListFName, sizeof(acListFName), PLEXT_SCAN_LIST ) )
  {
    debug( "_getPLFileName() failed" );
    return FALSE;
  }

  fpList = fopen( acListFName, "w+" );
  if ( fpList == NULL )
  {
    ices_log_error_output( "Could not open file: %s", acListFName );
    return FALSE;
  }

  strcpy( acPath, pPLMod->playlist_file );
  ices_log( "Read the list of files: %s", pPLMod->playlist_file );
  _scanDir( fpList, acPath, pPLMod->ext );
  // Update playlist if list of files was changed.
  // _updatePlaylist() closes fpList and removes acListFName.
  _updatePlaylist( fpList, acListFName );

  return TRUE;
}

// Waits the stop event for ulDelay minutes.
static BOOL _waitStopEvent(ULONG ulDelay)
{
  // We can't simply multiple ulDelay to convert it in milliseconds, the result
  // may be greater than the maximum for ULONG. We divide the wait time for
  // hours.
  ULONG      ulHourDelay;

  debug( "Wait for %u minute(s)", ulDelay );
  do
  {
    // Wait semaphore max. 1 hour.

    ulHourDelay = ulDelay > 60 ? 60 : ulDelay; // Left minutes in next hour.
    if ( xplEventWait( hevStop, ulHourDelay * 60 * 1000 ) != XPL_ERROR_TIMEOUT )
      return TRUE;

    ulDelay -= ulHourDelay;
    debug( "%u minutes left", ulDelay );
  }
  while( ulDelay != 0 );

  return FALSE;
}

void threadScan(void *pData)
{
  playlist_module_t    *pPLMod = (playlist_module_t*)pData;

  do
    fReload = FALSE;
  while( ( !_waitStopEvent( pPLMod->scan_per ) || fReload ) &&
         _makePlaylist( pPLMod ) );

  debug( "xplThreadEnd()..." );
  fAutoScan = FALSE;
  xplThreadEnd();
}


/* Shutdown the builtin playlist handler */
static void plShutdown()
{
  CHAR               acFName[_MAX_PATH];

  if ( fAutoScan )
  {
    if ( hevStop != NULLHANDLE )
    {
      fReload = FALSE;
      xplEventPost( hevStop );
      do
        xplSleep( 1 );
      while( fAutoScan );

      xplEventDestroy( hevStop );
    }

    if ( hmtxScanResultFile != NULLHANDLE )
      xplMutexDestroy( hmtxScanResultFile );
  }

  if ( fpPlaylist != NULL )
    fclose( fpPlaylist );

  if ( _getPLFileName( acFName, sizeof(acFName), PLEXT_SCAN_LIST ) )
    unlink( acFName );

  if ( _getPLFileName( acFName, sizeof(acFName), PLEXT_SCAN_RESULT ) )
    unlink( acFName );

  if ( _getPLFileName( acFName, sizeof(acFName), PLEXT_RANDOM ) )
    unlink( acFName );
}

static char *plGetNext(char *pcBuf, int cbBuf)
{
  playlist_module_t* pPLMod = &ices_config.pm;
  CHAR               acPlaylist[_MAX_PATH];
  PSZ                pszPlaylist;
  PCHAR              pcResult = NULL;
  int                iStartLineNo;
  struct stat        stStat;

  // Reopen playlist (configured or scan thread result) file if it was changed.

  if ( fAutoScan )
  {
    // Get name for the scan thread output file.
    if ( !_getPLFileName( acPlaylist, sizeof(acPlaylist), PLEXT_SCAN_RESULT ) )
    {
      debug( "_getPLFileName() failed" );
      return NULL;
    }
    pszPlaylist = acPlaylist;

    xplMutexLock( hmtxScanResultFile, XPL_INDEFINITE_WAIT );
  }
  else
    // No scan thread - get configured name for the playlist file.
    pszPlaylist = pPLMod->playlist_file;

  if ( stat( pszPlaylist, &stStat ) == -1 )
    debug( "File was not found: %s", pszPlaylist );
  // The playlist file has not been opened OR has been changed OR reload
  // flag is set when file instead a path configured for playlist  - reopen it.
  else if ( ( (fpPlaylist != NULL) && (stStat.st_mtime == timePlaylist) &&
              ( fAutoScan || !fReload ) ) ||
            _reopenPlaylist( pszPlaylist, (pPLMod->randomize != 0), &iLineNo ) )
  {
    // Read playlist up to not empty string.

    // Reset reload flag if it was set when playlist configured as file (not
    // a path).
    if ( !fAutoScan )
      fReload = FALSE;

    timePlaylist = stStat.st_mtime;
    iStartLineNo = iLineNo;
    pcBuf[cbBuf - 1] = '\0';
    do
    {
      // Read line.
l00:
      if ( fgets( pcBuf, cbBuf - 1, fpPlaylist ) == NULL )
      {
        if ( feof( fpPlaylist ) )
        {
          debug( "End of playlist..." );

          if ( iStartLineNo == 0 )
          {
            ices_log_error( "Empty playlist" );
            pcResult = NULL;
            break;
          }

          // Jump to top.
          iLineNo = 0;
          iStartLineNo = 0;
          ices_log_debug( "Reached end of playlist, rewinding" );
          rewind( fpPlaylist );
          goto l00;
        }

        ices_log_error( "Got error while reading playlist file: [%s]",
                        ices_util_strerror( errno, pcBuf, cbBuf ) );
        pcResult = NULL;
        break;
      }
      iLineNo++;

      // Remove leading and trailing spaces.
      pcResult = strchr( pcBuf, '\0' );
      while( ( pcResult > pcBuf ) && isspace( *(pcResult - 1) ) )
        pcResult--;
      *pcResult = '\0';
      pcResult = pcBuf;
      while( isspace( *pcResult ) )
        pcResult++;
    }
    while( ( *pcResult == '\0' ) || ( *pcResult == '#' ) );
  }

  if ( fAutoScan )
    xplMutexUnlock( hmtxScanResultFile );

  return pcResult;
}

static int plReload()
{
  fReload = TRUE;

  if ( fAutoScan )
  {
    xplEventPost( hevStop );
    return 1;
  }

  return -1;
}

/* Return the current playlist file line number */
static int plGetLineNo()
{
	return iLineNo;
}

/* Global function definitions */

/* Initialize the builting playlist handler */
int ices_playlist_builtin_initialize(playlist_module_t* pPLMod)
{
  struct stat          stStat;

	ices_log_debug( "Initializing builting playlist handler..." );

	iLineNo = 0;
	pPLMod->get_next     = plGetNext;
	pPLMod->get_metadata = NULL;
	pPLMod->get_lineno   = plGetLineNo;
	pPLMod->shutdown     = plShutdown;
	pPLMod->reload       = plReload;

	if ( ( pPLMod->playlist_file == NULL ) || ( pPLMod->playlist_file[0] == '\0' ) )
  {
		ices_log_error( "Playlist file or path is not set." );
		return -1;
	}

  if ( stat( pPLMod->playlist_file, &stStat ) == -1 )
  {
		ices_log_error( "Playlist file or path does not exist." );
		return -1;
	}

	fAutoScan = S_ISDIR( stStat.st_mode );

  if ( !fAutoScan )
    return 1;

  do
  {
    xplMutexCreate( &hmtxScanResultFile, FALSE );
    if ( hmtxScanResultFile == NULLHANDLE )
    {
      ices_log_error( "Can't create a mutex semaphore." );
      break;
    }

    xplEventCreate( &hevStop, XPL_EV_AUTO_RESET, FALSE );
    if ( hevStop == NULLHANDLE )
    {
      ices_log_error( "Can't create an event semaphore." );
      break;
    }

    if ( !_makePlaylist( pPLMod ) )
    {
      ices_log_error( "Failed to create playlist." );
      break;
    }

    debug( "Start directory scan thread..." );
    if ( xplThreadStart( threadScan, (PVOID)pPLMod ) == XPL_THREAD_START_FAIL )
    {
      ices_log_error( "Can't start a thread." );
      break;
    }

    // Success.
    return 1;
  }
  while( FALSE );

  // Error.

  debug( "Error. Cleaning..." );

  if ( hevStop != NULLHANDLE )
    xplEventDestroy( hevStop );

  if ( hmtxScanResultFile != NULLHANDLE )
      xplMutexDestroy( hmtxScanResultFile );

  hevStop = NULLHANDLE;
  hmtxScanResultFile = NULLHANDLE;

	return -1;
}
