/* Solpos.cmd - compute solar position and intensity from time and place.    */
/* Edit the following lines and then run.  If you don't have values leave    */
/* the existing numbers.                                                     */
                                                 
/* Date & time information.                                                  */
year      = 2000       /* 4-digit year (2-digit is assumed 19xx)             */

/* Enter EITHER daynum by itself or month and day.                           */
daynum    =   ' '      /* day of year; Feb 1 = 32; if =' '  use month & day  */

month     =   11       /* Month number (Jan = 1, Feb = 2, etc.)              */
day       =   29       /* Day of month (May 27 = 27, etc.)                   */


hour      =   20       /* Hour of day,      0 - 23                           */
minute    =   38       /* Minute of hour,   0 - 59                           */
second    =   00       /* Second of minute, 0 - 59                           */

/* To use GMT set the time zone to 0.                                        */
timezone  =     0      /* *STANDARD* time zone. Mountain = -7, Central = -6  */

/* Location of observer/flat surface.                                        */
latitude  =   33.65    /* Latitude, degrees north (south negative)           */
longitude =  -84.43    /* Longitude, degrees east (west negative)            */

/* Atmospheric temperature and pressure.                                     */
temp      =   25.0     /* Ambient dry-bulb temperature, degrees C            */
pressure  = 1006.0     /* Surface air pressure, millibar                     */

/* Orientation of a flat surface (a panel) receiving energy from the sun.    */
tilt      = latitude   /* Degrees tilt from horizontal of panel              */
aspect    = 135.0      /* Azimuth of panel N=0, E=90, S=180, W=270           */


/*============================================================================
*        S_solpos     (computes solar position and intensity
*                      from time and place)
*            INPUTS:     S_month, S_day, S_latitude, S_longitude, S_timezone
*            OPTIONAL:   S_year, S_hour, S_minute, S_second,
*                            S_press, S_temp, S_tilt, S_aspect
*            OUTPUTS:    EVERY variable beginning with "S_"
*                            (defined beneath this comment block)
*
*    Martin Rymes
*    National Renewable Energy Laboratory
*    25 March 1998
*
*    27 April 1999 REVISION:  Corrected leap year in S_date.
*    14 January 2000 REVISION:  Corrected leap year in S_solpos.
*    http://rredc.nrel.gov/solar/codes_algs/solpos/                           
*
*    Converted to REXX and I/O modifications by 
*    Doug Rickman, GHCC/MAFS/NASA Nov. 29, 2000        
*----------------------------------------------------------------------------*/

/*  Major Variable Definitions:
* S_day       /* Day of month (May 27 = 27, etc.)                            */
* S_daynum    /* Day number (day of year; Feb 1 = 32 )                       */
* S_hour      /* Hour of day, 0 - 23                                         */
* S_minute    /* Minute of hour, 0 - 59                                      */
* S_month     /* Month number (Jan = 1, Feb = 2, etc.)                       */
* S_second    /* Second of minute, 0 - 59                                    */
* S_year      /* 4-digit year (2-digit is assumed 19xx)                      */
* S_amass;    /* Relative optical airmass                                    */
* S_ampress;  /* Pressure-corrected airmass                                  */
* S_aspect    /* Azimuth of panel surface N=0, E=90, S=180, W=270            */
* S_azim;     /* Solar azimuth angle:  N=0, E=90, S=180, W=270               */
* S_cosinc;   /* Cosine of solar incidence angle on panel                    */
* S_dayang;   /* Day angle (daynum*360/year-length) degrees                  */
* S_declin;   /* Declination-zenith angle of solar noon at equator, deg NORTH*/
* S_eclong;   /* Ecliptic longitude, degrees                                 */
* S_ecobli;   /* Obliquity of ecliptic                                       */
* S_ectime;   /* Time of ecliptic calculations                               */
* S_elevref;  /* Solar elevation angle, deg. from horizon, refracted         */
* S_eqntim;   /* Equation of time (TST - LMT), minutes                       */
* S_erv;      /* Earth radius vector (multiplied to solar constant)          */
* S_etr;      /* global horizontal solar irradiance, No atmos. W/sq m^2      */
* S_etrn;     /* Direct normal solar irradiance, No atmos. W/sq m^2          */
* S_etrtilt;  /* Global irradiance on a tilted surface, No atmos. W/sq m^2   */
* S_gmst;     /* Greenwich mean sidereal time, hours                         */
* S_hrang;    /* Hour angle--hour of sun from solar noon, degrees WEST       */
* S_julday;   /* Julian Day of 1 JAN 2000 minus 2,400,000 days (in order to regain single precision) */
* S_latitude  /* Latitude, degrees north (south negative)                    */
* S_longitude /* Longitude, degrees east (west negative)                     */
* S_lmst;     /* Local mean sidereal time, degrees                           */
* S_mnanom;   /* Mean anomaly, degrees                                       */
* S_mnlong;   /* Mean longitude, degrees                                     */
* S_rascen;   /* Right ascension, degrees                                    */
* S_press     /* Surface pressure, millibars                                 */
* S_prime;    /* Factor that normalizes Kt, Kn, etc.                         */
* S_sbcf;     /* Shadow-band correction factor                               */
* S_solcon    /* Solar constant, 1367 W/sq m                                 */
* S_ssha;     /* Sunset(/rise) hour angle, degrees                           */
* S_sunrise;  /* Sunrise time, minutes from midnight, local, no refraction   */
* S_sunset;   /* Sunset time, minutes from midnight,  local, no refraction   */
* S_temp      /* Ambient dry-bulb temperature, degrees C                     */
* S_tilt      /* Degrees tilt from horizontal of panel                       */
* S_timezone  /* Time zone, (west negative). Mountain =-7, Central=-6,etc.   */
* S_tst;      /* True solar time, minutes from midnight                      */
* S_tstfix;   /* True solar time - local standard time                       */
* S_unprime;  /* Factor that denormalizes Kt', Kn', etc.                     */
* S_utime;    /* Universal (Greenwich) standard time                         */
* S_zenetr;   /* Solar zenith angle, no atmospheric correction (= ETR)       */
* S_zenref;   /* Solar zenith angle, deg. from zenith, refracted             */
* 
*----------------------------------------------------------------------------*/

S_solcon    = 1367.0   /* Solar constant, 1367 W/sq m**2 */

/* length of year; 0 = non-leap, 1 = leap year */
lenyr.0 = 365          
lenyr.1 = 366

/* cumulative number of days prior to beginning of month */
month_days.0.00 = 0
month_days.0.01 = 0
month_days.0.02 = 31
month_days.0.03 = 59
month_days.0.04 = 90
month_days.0.05 = 120
month_days.0.06 = 151
month_days.0.07 = 181
month_days.0.08 = 212
month_days.0.09 = 243
month_days.0.10 = 273
month_days.0.11 = 304
month_days.0.12 = 334

month_days.1.00 = 0
month_days.1.01 = 0
month_days.1.02 = 31
month_days.1.03 = 60
month_days.1.04 = 91
month_days.1.05 = 121
month_days.1.06 = 152
month_days.1.07 = 182
month_days.1.08 = 213
month_days.1.09 = 244
month_days.1.10 = 274
month_days.1.11 = 305
month_days.1.12 = 335

degrad = 57.295779513  /* converts from radians to degrees */
raddeg = 0.0174532925  /* converts from degrees to radians */

/* convert to variable names used by program. */
S_year      = year
S_daynum    = daynum
S_month     = month
S_day       = day
S_hour      = hour
S_minute    = minute
S_second    = second
S_timezone  = timezone
S_latitude  = latitude
S_longitude = longitude
S_temp      = temp
S_press     = pressure
S_tilt      = tilt
S_aspect    = aspect

/* Load libraries. */
if rxfuncquery('rexxlibregister') then do         /* this will start rexxlib */
   call rxfuncadd 'rexxlibregister', 'rexxlib', 'rexxlibregister'  
   call rexxlibregister
   end
if rxfuncquery('sysloadfuncs') then do           /* this will start rexxutil */
   CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs' 
   CALL SysLoadFuncs
   end


if S_daynum \= ' ' then
   rc = s_date()

/*============================================================================
*    Integer function S_solpos, adapted from the VAX solar libraries
*
*    This function calculates the apparent solar position and the
*    intensity of the sun (theoretical maximum solar energy) from
*    time and place on Earth.
*
*    Requires:
/* Date and time:  S_year S_month S_day S_hour S_minute S_second             */
/* Location:       S_latitude S_longitude                                    */
/* Time adjuster:  S_timezone                                                */
/* Atmospheric pressure and temperature:  S_press S_temp                     */
/* Tilt of flat surface that receives solar energy: S_aspect S_tilt          */
*----------------------------------------------------------------------------*/

S_solpos:

S_month = right(S_month,2,'0')

/* No absurd dates, please. */
if (S_month < 1) | (S_month > 12) then do
   say "Please fix the month:" S_month
   return -1
   end
if (S_day < 1) | (S_day > 31 ) then do
   say "Please fix the day of month:" S_day
   return -1
   end

/* No absurd times, please. */
if (S_hour < 0) | (S_hour > 23) then do
   say  "Please fix the hour:" S_hour  
   return -1
   end
if (S_minute < 0) | (S_minute > 59) then do
   say  "Please fix the minute:" S_minute  
   return -1 
   end
if (S_second < 0) | (S_second > 59) then do
   say  "Please fix the second:" S_second  
   return -1 
   end
if (S_timezone < -12.0) | (S_timezone > 12.0) then do 
   say  "Please fix the time zone:" S_timezone  
   return -1
   end

/* No absurd locations, please. */
if abs(S_longitude) > 180.0 then do
   say "Please fix the longitude: " S_longitude
   return -1 
   end
if abs(S_latitude) > 90.0 then do
   say  "Please fix the latitude:" S_latitude  
   return -1
   end

/* No silly temperatures or pressures, please. */
if abs(S_temp) > 100.0 then do
   say  "Please fix the temperature:" S_temp  
   return -1
   end
if (S_press < 0.0) | (S_press > 2000.0) then do
   say  "Please fix the pressure:" S_press  
   return -1
   end


/* Day number */
/* (convert day-of-month to day-of-year (non-leap)) */
S_daynum = S_day + month_days.0.S_month

/*  If no year was entered, give it an arbitrary 2001.  Also, turn 2-digit   */
/* years into 19xx.  Note that if the year > 1999, it had better be 4-digit. */
if S_year < 0 then
   S_year  = 2001
if S_year < 100 then 
   S_year = S_year + 1900

/* (now adjust for leap year) */
if (S_year%4 = 0) & ((S_year%100 \= 0) | (S_year % 400 = 0)) & (S_month > 2) then
   S_daynum = S_daynum + 1

/* Day angle  ref I1983 */
S_dayang = 360.0 * ( S_daynum - 1 ) / 365.0


/* Earth radius vector * solar constant = solar energy    ref S1971          */
sd     = sin(raddeg * S_dayang)
cd     = cos(raddeg * S_dayang)
d2     = 2.0 * S_dayang
c2     = cos(raddeg * d2)
s2     = sin(raddeg * d2)

S_erv  = 1.000110 + 0.034221 * cd + 0.001280 * sd
S_erv  = S_erv + (0.000719 * c2 + 0.000077 * s2)

/* Universal Coordinated (Greenwich standard) time ref M1988 */
S_utime = S_hour * 3600.0 + S_minute * 60.0 + S_second
S_utime = S_utime / 3600.0 - S_timezone

/* Julian Day minus 2,400,000 days (to eliminate roundoff errors) ref M1988  */
delta    = S_year - 1949
leap     = trunc(delta / 4.0)
S_julday = 32916.5 + delta * 365.0 + leap + S_daynum + S_utime / 24.0

if (S_year%100 = 0) & (S_year%400 \= 0) then 
   S_julday = S_julday -1

/* Time used in the calculation of ecliptic coordinates                      */
/* Noon 1 JAN 2000 = 2,400,000 + 51,545 days Julian Date  ref M1988          */
S_ectime = S_julday - 51545.0

/* Mean longitude ref M1988 */
S_mnlong  = 280.460 + 0.9856474 * S_ectime

/* Dump the multiples of 360, so the answer is between 0 and 360.            */
S_mnlong = S_mnlong -(360.0 * trunc(S_mnlong / 360.0 ))
if ( S_mnlong < 0.0 ) then
   S_mnlong = S_mnlong + 360.0

/* Mean anomaly  ref M1988 */
S_mnanom  = 357.528 + 0.9856003 * S_ectime

/* Dump the multiples of 360, so the answer is between 0 and 360. */
S_mnanom = S_mnanom - (360.0 * trunc(S_mnanom / 360.0 ))
if ( S_mnanom < 0.0 ) then
   S_mnanom = S_mnanom + 360.0

/* Ecliptic longitude ref M1988 */
S_eclong  = S_mnlong + 1.915 * sin(S_mnanom * raddeg) + ,
   0.020 * sin(2.0 * S_mnanom * raddeg)

/* Dump the multiples of 360, so the answer is between 0 and 360). */
S_eclong = S_eclong - (360.0 * trunc(S_eclong / 360.0 ))
if ( S_eclong < 0.0 ) then
   S_eclong = S_eclong + 360.0

/* Obliquity of the ecliptic ref M1988 */
S_ecobli = 23.439 + 4.0e-07 * S_ectime

/* Declination ref M1988 */
S_declin = degrad * asin(sin(S_ecobli * raddeg) * sin(S_eclong * raddeg) )

/* Right ascension ref M1988 */
top      =  cos(raddeg * S_ecobli) * sin(raddeg * S_eclong)
bottom   =  cos(raddeg * S_eclong)

S_rascen =  degrad * atan2(top,bottom)

/* Make it a positive angle. */
if S_rascen < 0.0 then
   S_rascen = S_rascen + 360.0 

/* Greenwich mean sidereal time  ref M1988 */
S_gmst  = 6.697375 + 0.0657098242 * S_ectime + S_utime

/* Dump the multiples of 24, so the answer is between 0 and 24. */
S_gmst = S_gmst - (24.0 * trunc(S_gmst / 24.0 ))
if S_gmst < 0.0 then 
   S_gmst = S_gmst + 24.0

/* Local mean sidereal time ref M1988 */
S_lmst  = S_gmst * 15.0 + S_longitude

/* Dump the multiples of 360, so the answer is between 0 and 360. */
S_lmst = S_lmst - (360.0 * trunc(S_lmst / 360.0 ))
if S_lmst < 0 then
   S_lmst = S_lmst + 360.0

/* Hour angle ref M1988 */
S_hrang = S_lmst - S_rascen

/* Force it between -180 and 180 degrees. */
select
   when S_hrang < -180.0 then
      S_hrang = S_hrang + 360.0
   when S_hrang > 180.0 then 
      S_hrang -= S_hrang - 360.0
   otherwise
      nop
   end /* select */

/* ETR solar zenith angle ref I1983 */
cd = cos(raddeg * S_declin)
ch = cos(raddeg * S_hrang)
cl = cos(raddeg * S_latitude)
sd = sin(raddeg * S_declin)
sl = sin(raddeg * S_latitude)

cz = sd * sl + cd * cl * ch

/* Watch out for the roundoff errors (Hold over from original C code.)       */
if abs(cz) > 1.0 then do
   if cz >= 0.0 then
      cz =  1.0
   else
      cz = -1.0
   end 

S_zenetr = acos(cz) * degrad

/* Limit the degrees below the horizon to 9 [+90 -> 99] */
/* Disabled by DLR Nov 30, 2000                         */
/* if ( S_zenetr > 99.0 ) then S_zenetr = 99.0          */

/* Sunset hour angle ref I1983 */
S_ssha  = 90.0
cdcl    = cd * cl

if abs(cdcl) >= 0.001 then do
   cssha = -sl * sd / cdcl
   /* This keeps the cosine from blowing on roundoff */
   select   
      when cssha < -1.0 then
         S_ssha = 180.0
      when cssha > 1.0 then 
         S_ssha = 0.0
      otherwise
         S_ssha = degrad * acos(cssha)
      end /* select */
   end /* if abs(cdcl) ... */


/* Shadowband correction factor ref D1956 */
p      = 0.1526 * pow(cd,3)
t1     = sl * sd * S_ssha * raddeg
t2     = cl * cd * sin(S_ssha * raddeg)
S_sbcf = 0.04 + 1.0 / ( 1.0 - p * (t1 + t2) )

/* TST -> True Solar Time = local standard time + TSTfix  ref I1983 */
S_tst    = (180.0 + S_hrang) * 4.0
S_tstfix = S_tst - S_hour * 60.0 - S_minute - S_second / 60.0
S_eqntim = S_tstfix + 60.0 * S_timezone - 4.0 * S_longitude

/* Sunrise and sunset times */
select
   when ( S_ssha <= 1.0 ) then do
      S_sunrise   =  2999.0
      S_sunset    = -2999.0
      end
   when ( S_ssha >= 179.0 ) then do
      S_sunrise   = -2999.0
      S_sunset    =  2999.0
      end
   otherwise do
      S_sunrise   = 720.0 - 4.0 * S_ssha - S_tstfix
      S_sunset    = 720.0 + 4.0 * S_ssha - S_tstfix
      end
   end /* select */

/* Solar azimuth angle  ref I1983*/
ce         = sin(raddeg * S_zenetr)
se         = cos(raddeg * S_zenetr)

S_azim     = 180.0
cecl       = ce * cl
if abs(cecl) >= 0.001 then do
   ca     = ( se * sl - sd ) / cecl
   if ca >  1.0 then 
      ca =  1.0
   if ca < -1.0 then 
      ca = -1.0

   S_azim = 180.0 - acos(ca) * degrad
   if S_hrang > 0 then
      S_azim  = 360.0 - S_azim
   end /* if abs(cecl) ... */

/* Refraction correction ref Z1981 */
elev = 90.0 - S_zenetr

select
   when elev > 85.0 then
      /* If the sun is near zenith, refraction near 0 */
      S_elevref = elev
   when elev < -9.0 then
      /* Sun below horizon at earth surface.  DLR */
      S_elevref = elev
   otherwise do
   /* Otherwise, we have refraction */
      tanelev = tan(raddeg * elev)
      select
         when elev >= 5.0 then 
            refcor  = 58.1/tanelev - 0.07/pow(tanelev,3) + 0.000086 /pow(tanelev,5)
         when elev >= -0.575 then
            refcor  = 1735.0 + elev * (-518.2 + elev * (103.4 + elev * (-12.79 + elev * 0.711)))
         otherwise 
            refcor  = -20.774 / tanelev
         end /* select */
      prestemp    = (S_press * 283.0) / (1013.0 * (273.0 + S_temp))
      refcor     = refcor * (prestemp /3600.0)
      /* Refracted solar elevation angle */
      S_elevref = elev + refcor*degrad
      end /* otherwise do ... */
   end /* select */

/* (limit the degrees below the horizon to 9)  */
/* Disabled by DLR Nov 30, 2000                */
/* if S_elevref < -9.0 then S_elevref = -9.0   */

/* Refracted solar zenith angle */
S_zenref  = 90. - S_elevref

/* Airmass  ref K1989 */
cz = cos(raddeg * S_zenref)
if S_zenref > 93.0 then
   S_amass = -1.0
else
   S_amass = 1.0 / (cz + 0.50572 * pow((96.07995 - S_zenref),-1.6364))

S_ampress   = S_amass * S_press / 1013.0

/* Prime and Unprime: Prime converts Kt to normalized Kt', etc. Unprime      */
/* deconverts Kt' to Kt, etc.  ref P1990                                     */
S_unprime = 1.031 * exp(-1.4 / ( 0.9 + 9.4 / S_amass ) ) + 0.1
S_prime   = 1.0 / S_unprime

/* Cosine of the angle between the sun and a tipped flat surface useful for  */
/* calculating solar energy on tilted surfaces                               */
ca        = cos(raddeg * S_azim)
cp        = cos(raddeg * S_aspect)
ct        = cos(raddeg * S_tilt)
sa        = sin(raddeg * S_azim)
sp        = sin(raddeg * S_aspect)
st        = sin(raddeg * S_tilt)
sz        = sin(raddeg * S_zenref)
S_cosinc  = cz * ct + sz * st * ( ca * cp + sa * sp )

/* Extraterrestrial (top-of-atmosphere) solar irradiance */
if cz > 0.0 then do 
   S_etrn = S_solcon * S_erv
   S_etr  = S_etrn * cz
   end
else do
   S_etrn = 0.0
   S_etr  = 0.0
   end
 
if S_cosinc > 0.0 then
   S_etrtilt = S_etrn * S_cosinc
else 
   S_etrtilt = 0.0


S_sunrise2 = trunc(S_sunrise/60)':'trunc(S_sunrise//60)
S_sunset2 = trunc(S_sunset/60)':'trunc(S_sunset//60)


say 'mm/dd/yyyy  Hour:minute:second:           ' S_month'/'S_day'/'S_year  S_hour':'S_minute':'S_second   
say 'Universal (Greenwich) time:               ' s_utime 
say 'Time zone:                                ' S_timezone
say 'Latitude Longitude:                       ' S_latitude S_longitude

if S_elevref > -9 then
   say 'Solar azimuth & elev. Actual (Refracted): ' format(S_azim,4,3) format(90-S_zenetr,3,3) '('strip(format(S_elevref,3,3))')'
else 
   say 'Solar azimuth & elev. Actual (Refracted): ' format(S_azim,4,3) format(90-S_zenetr,3,3) '('below horizon')'

say 'Local sunrise/sunset:                     ' S_sunrise2 S_sunset2
say
say 'Irradiance onto a panel, assuming no atmospheric effects. W/(sq m**2)'
say 'These are the theoretical maximum.'
say '   Cosine of solar incidence angle:       ' S_cosinc
say '   Global horizontal solar irradiance:    ' S_etr
say '   Direct normal solar irradiance:        ' S_etrn
say '   Global irradiance on a tilted surface: ' S_etrtilt

return 0




/*============================================================================
*    Integer function S_date, adapted from the VAX solar libraries
*
*    This function computes the month/day from the day number.
*
*    Requires:
*        Year and day number:
*            S_year           NOTE:  2-digit year "xx" is assumed "19xx"
*            S_daynum         RANGE: -1 to a large positive integer
*
*    Returns Year, month, day: S_year S_month S_day
*----------------------------------------------------------------------------*/
S_date:
/* We will allow -1 (in order to adjust to the previous year) but no less */
if ( S_daynum < -1 ) then do
   say  "Please fix the day number:" S_daynum  
   return -1
   end

/* If a 2-digit year "xx" is given, assume "19xx" */
if ( S_year < 100 ) then
   S_year = S_year + 1900

/* Set the leap year switch */
if S_year%4=0 & (S_year%100\=0) | (S_year%400=0) then 
   leap = 1
else
   leap = 0

/* If S_daynum is a large number, it is in a future year */
ndy = S_daynum
do while (ndy > lenyr.leap )
   S_year = S_year + 1
   ndy = ndy-lenyr.leap
   if ( ((S_year % 4) = 0) & ( ((S_year % 100) \= 0) | ((S_year % 400) == 0) ) ) then
      leap = 1
   else
      leap = 0
   end

/* A non-positive day number indicates the previous year */
if ( ndy <= 0 ) then do
   S_year = S_year - 1
   leap =  0
   ndy  = ndy+365
   end  

/* Now that the year is fixed, find the month */
imon = 12
do while ( ndy <= month_days.leap.imon )
   imon = right(imon-1,2,'0')
   /* Set the month and day of month */
   S_month = imon
   S_day   = ndy - month_days.leap.imon
   end

say S_month S_day
return 1


/* References                                                                */
/*                                                                           */
/* Drummond, A. J. 1956. A contribution to absolute pyrheliometry. Q. J. R.  */
/*    Meteorol. Soc. 82, pp. 481-493                                         */
/*                                                                           */
/* Iqbal, M. 1983. An Introduction to Solar Radiation. Academic Press, NY,p15*/
/*                                                                           */
/* Kasten, F. and Young, A.  1989.  Revised optical air mass tables and      */
/*    approximation formula.  Applied Optics 28 (22), pp. 4735-4738          */
/*                                                                           */
/* Michalsky, J.  1988.  The Astronomical Almanac's algorithm for approximate*/
/*    solar position (1950-2050). Solar Energy 40 (3),pp. 227-235.           */
/*                                                                           */
/* Perez, R., P. Ineichen, Seals, R., & Zelenka, A.  1990.  Making full use  */
/*    of the clearness index for parameterizing hourly insolation conditions.*/
/*    Solar Energy 45 (2), pp. 111-114                                       */
/*                                                                           */
/* Spencer, J. W.  1971.  Fourier series representation of the position of   */
/*    the sun.  Search 2 (5), page 172                                       */
/*                                                                           */
/* Zimmerman, John C.  1981.  Sun-pointing programs and their accuracy.      */
/*    SAND81-0761, Experimental Systems Operation Division 4721, Sandia      */
/*    National Laboratories, Albuquerque, NM.                                */

