#include <cstdio>
#include <cstdlib>
#include <cstring>
#include "common.h"
#include "dllibrary.h"
#include "logger.h"

/*=====================Windows======================================*/
#ifdef _WIN32 //.dll
#include <windows.h>

const char* DLLibrary::OSLIBEXT = ".dll";

DLLibrary::DLLibrary(const char* libname) : name(libname) {
    library = LoadLibrary(libname);
    if( !library ) {
        logger->warning("Unable to load library \"%s\"\n",
                        libname);
        throw 0;
    }
}

DLLibrary::~DLLibrary() {
    FreeLibrary(library);
}

void* DLLibrary::getFunction(const char* funcname) {
    void *retval = (void*)GetProcAddress(library, funcname);
    if( retval == NULL ) {
        logger->warning("Unable to extract function \"%s\" from library \"%s\"\n", funcname, name.c_str());
    }
    return retval;
}

/*=====================MacOS X======================================*/
#elif defined(__APPLE__)
const char* DLLibrary::OSLIBEXT = ".so";
DLLibrary::DLLibrary(const char* libname) : name(libname) {
    NSObjectFileImage image;

    if(NSCreateObjectFileImageFromFile(libname, &image) == NSObjectFileImageSuccess) {
        library = NSLinkModule(image, libname,
                               NSLINKMODULE_OPTION_RETURN_ON_ERROR|
                               NSLINKMODULE_OPTION_PRIVATE);
        if (!library) {
            NSLinkEditErrors ler;
            int lerno;
            const char* errstr;
            const char* file;
            NSLinkEditError(&ler,&lerno,&file,&errstr);
            logger->warning("Unable to load library \"%s\": %s\n", libname, errstr);
            throw 0;
        }
        NSDestroyObjectFileImage(image);
    } else {
        logger->warning("Unable to load library \"%s\"\n", libname);
        throw 0;
    }
}

DLLibrary::~DLLibrary() {
    NSUnLinkModule(library, 0);
}

void* DLLibrary::getFunction(const char* funcname) {
    char* fsname = new char[strlen(funcname)+2];
    sprintf(fsname, "_%s", funcname);
    NSSymbol sym = NSLookupSymbolInModule(library, fsname);
    delete[] fsname;
    if (sym) {
        return NSAddressOfSymbol(sym);
    }
    logger->warning("Unable to extract function \"%s\" from library \"%s\"\n", funcname, name.c_str());
    return NULL;
}

/*=====================Some form of standard unix======================================*/
#else // .so
#include <dlfcn.h>
#ifdef RTLD_NOW
#define LIBOPTION RTLD_NOW
#elif defined(RTLD_LAZY)
#define LIBOPTION RTLD_LAZY
#elif defined(DL_LAZY)
#define LIBOPTION DL_LAZY
#else
#define LIBOPTION 0
#endif
const char* DLLibrary::OSLIBEXT = ".so";

DLLibrary::DLLibrary(const char* libname) : name(libname) {
    library = dlopen(libname, LIBOPTION);
    if( !library ) {
        logger->warning("Unable to load library \"%s\": %s\n", libname, dlerror());
        throw 0;
    }
}

DLLibrary::~DLLibrary() {
    dlclose(library);
}

void* DLLibrary::getFunction(const char* funcname) {
    void *retval = (void*)dlsym(library, funcname);
    //FIXME: Some dynamic loaders require a prefix on the shared names.
    //       the only one I've encountered is "_" which is required on openBSD
    //       and possible some other platform. This should be detected at
    //       compiletime but libs are loaded so seldom this will have to do
    //       for now.
    if (retval == NULL) {
        char* fsname;
        fsname = new char[strlen(funcname)+2];
        sprintf(fsname, "_%s", funcname);
        retval = (void*)dlsym(library, fsname);
        delete[] fsname;
    }
    if (retval == NULL) {
        logger->warning("Unable to extract function \"%s\" from library \"%s\": %s\n", funcname, name.c_str(), dlerror());
    }
    return retval;
}

#endif

