//-------------------------------------------------------------------------
// The following is part of the VIOWIN user-interface library source code.
// The code is copyright (c) 1994-1995 by Larry Salomon, Jr. and may not
// be used in a for-profit application without the expressed, written
// consent of Larry Salomon, Jr.  All rights reserved.
//-------------------------------------------------------------------------
#define VARS
#define INCL_DOSMODULEMGR
#define INCL_DOSPROCESS
#define INCL_VIO
#include "vwdefs.h"
#include <stdlib.h>

BOOL EXPENTRY vwInitialize(VOID)
//-------------------------------------------------------------------------
// This function initializes the library.
//
// 1) Initializes the memory manager
// 2) Creates the anchor block and message queue
// 3) Creates the linked lists for housekeeping
// 4) Initializes the system values array
// 5) Starts the keyboard monitor
// 6) Calls ordinal 1 in the class library so that the public classes
//    may be registered.
//
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------
{
   HCMMEM hcmWork;
   struct _VIOCURSORINFO vciCursor;
   PFNDLLINIT pfnInit;

   if (habAnchor!=NULL) {
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // Initialize the memory manager
   //----------------------------------------------------------------------
   if (CmnMemInitialize(NULL,&hcmWork)!=MEM_ERR_NOERROR) {
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // Create the anchor block and message queue and initialize them
   //----------------------------------------------------------------------
   CmnMemAllocate(hcmWork,sizeof(VWAB),(PPVOID)&habAnchor);
   if (habAnchor==NULL) {
      CmnMemTerminate(&hcmWork);
      return FALSE;
   } /* endif */

   habAnchor->ulSzStruct=sizeof(VWAB);
   habAnchor->ulStatus=0;
   habAnchor->hcmWork=hcmWork;

   //----------------------------------------------------------------------
   // Create the housekeeping lists
   //----------------------------------------------------------------------
   if (!CmnLstCreateList(sizeof(VWCLASSINFO),&habAnchor->hclClasses)) {
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      return FALSE;
   } /* endif */

   if (!CmnLstCreateList(sizeof(VWWND),&habAnchor->hclWindows)) {
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      return FALSE;
   } /* endif */

   if (!CmnLstCreateList(sizeof(VWTIMERINFO),&habAnchor->hclTimers)) {
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      return FALSE;
   } /* endif */

   habAnchor->bIsSendMsg=FALSE;
   habAnchor->hwndFocus=NULL;

   CmnMemAllocate(habAnchor->hcmWork,
                  sizeof(VWCURSORINFO),
                  (PPVOID)&habAnchor->pciCursor);
   if (habAnchor->pciCursor==NULL) {
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnLstDestroyList(&habAnchor->hclTimers);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      return FALSE;
   } /* endif */

   CmnMemAllocate(habAnchor->hcmWork,sizeof(VWMQ),(PPVOID)&hmqQueue);
   if (hmqQueue==NULL) {
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnLstDestroyList(&habAnchor->hclTimers);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      return FALSE;
   } /* endif */

   hmqQueue->ulSzStruct=sizeof(VWMQ);
   hmqQueue->ulHead=0;
   hmqQueue->ulTail=0;

   habAnchor->pciCursor->hwndWnd=NULL;
   habAnchor->pciCursor->sX=0;
   habAnchor->pciCursor->sY=0;
   habAnchor->pciCursor->usFlags=0;

   habAnchor->usCursorState=0;

   //----------------------------------------------------------------------
   // Initialize the system values array
   //----------------------------------------------------------------------
   habAnchor->alSysValues[VWSV_ERRORFREQ]=200;
   habAnchor->alSysValues[VWSV_NOTEFREQ]=800;
   habAnchor->alSysValues[VWSV_WARNINGFREQ]=400;
   habAnchor->alSysValues[VWSV_ERRORDURATION]=75;
   habAnchor->alSysValues[VWSV_NOTEDURATION]=75;
   habAnchor->alSysValues[VWSV_WARNINGDURATION]=75;
   habAnchor->alSysValues[VWSV_CXSCREEN]=80;
   habAnchor->alSysValues[VWSV_CYSCREEN]=25;
   habAnchor->alSysValues[VWSV_INSERTMODE]=TRUE;
   habAnchor->alSysValues[VWSV_CTIMERS]=16;

   //----------------------------------------------------------------------
   // Start the keyboard monitor
   //----------------------------------------------------------------------
   if (_beginthread(keyMonitor,NULL,0x2000,NULL)==-1) {
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnLstDestroyList(&habAnchor->hclTimers);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      hmqQueue=NULL;
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // Loop until we get a status back
   //----------------------------------------------------------------------
   while ((habAnchor->ulStatus &
          (VW_HABST_MONINITGOOD | VW_HABST_MONINITBAD))==0) {
      DosSleep(0);
   } /* endwhile */

   //----------------------------------------------------------------------
   // If bad, cleanup and return an error
   //----------------------------------------------------------------------
   if ((habAnchor->ulStatus & VW_HABST_MONINITBAD)!=0) {
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnLstDestroyList(&habAnchor->hclTimers);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      hmqQueue=NULL;
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // Hide the real cursor
   //----------------------------------------------------------------------
   VioGetCurType(&vciCursor,NULLHANDLE);
   vciCursor.attr=-1;
   VioSetCurType(&vciCursor,NULLHANDLE);

   //----------------------------------------------------------------------
   // Load the classes DLL and call ordinal 1
   //----------------------------------------------------------------------
   if (DosLoadModule(NULL,0,DLL_CLASS,&habAnchor->hmClasses)!=0) {
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnLstDestroyList(&habAnchor->hclTimers);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      hmqQueue=NULL;
      return FALSE;
   } /* endif */

   if (DosQueryProcAddr(habAnchor->hmClasses,1,NULL,(PFN *)&pfnInit)!=0) {
      DosFreeModule(habAnchor->hmClasses);
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnLstDestroyList(&habAnchor->hclTimers);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      hmqQueue=NULL;
      return FALSE;
   } /* endif */

   if (!(*pfnInit)()) {
      DosFreeModule(habAnchor->hmClasses);
      CmnLstDestroyList(&habAnchor->hclClasses);
      CmnLstDestroyList(&habAnchor->hclWindows);
      CmnLstDestroyList(&habAnchor->hclTimers);
      CmnMemTerminate(&hcmWork);
      habAnchor=NULL;
      hmqQueue=NULL;
      return FALSE;
   } /* endif */

   return TRUE;
}

BOOL EXPENTRY vwTerminate(VOID)
//-------------------------------------------------------------------------
// This function terminates the library.
//
// Returns:  TRUE if successful, FALSE otherwise.
//-------------------------------------------------------------------------
{
   ULONG ulNumItems;
   ULONG ulIndex;
   HVWWND hwndList;
   PVWTIMERINFO ptiTimer;
   HCMMEM hcmWork;
   struct _VIOCURSORINFO vciCursor;
   BYTE abAttr[2];

   //----------------------------------------------------------------------
   // Make sure we were initialized
   //----------------------------------------------------------------------
   if (hmqQueue==NULL) {
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // Destroy the windows
   //----------------------------------------------------------------------
   ulNumItems=CmnLstQueryRecordCount(habAnchor->hclWindows);

   for (ulIndex=0; ulIndex<ulNumItems; ulIndex++) {
      if (ulIndex==0) {
         hwndList=(HVWWND)CmnLstQueryRecord(habAnchor->hclWindows,0);
      } else {
         hwndList=(HVWWND)CmnLstQueryRelative(hwndList,LQR_NEXT);
      } /* endif */

      vwDestroyWindow(hwndList);
   } /* endfor */

   //----------------------------------------------------------------------
   // Destroy the timers
   //----------------------------------------------------------------------
   ulNumItems=CmnLstQueryRecordCount(habAnchor->hclTimers);

   for (ulIndex=0; ulIndex<ulNumItems; ulIndex++) {
      if (ulIndex==0) {
         ptiTimer=(PVWTIMERINFO)CmnLstQueryRecord(habAnchor->hclTimers,0);
      } else {
         ptiTimer=(PVWTIMERINFO)CmnLstQueryRelative(ptiTimer,LQR_NEXT);
      } /* endif */

      vwStopTimer(ptiTimer->hwndWnd,ptiTimer->ulId);
   } /* endfor */

   //----------------------------------------------------------------------
   // Tell the keyboard monitor to disengage itself
   //----------------------------------------------------------------------
   habAnchor->ulStatus|=VW_HABST_MONSHOULDTERM;

   while ((habAnchor->ulStatus & VW_HABST_MONTERM)!=0) {
      DosSleep(0);
   } /* endwhile */

   //----------------------------------------------------------------------
   // Cleanup the rest
   //----------------------------------------------------------------------
   DosFreeModule(habAnchor->hmClasses);
   CmnLstDestroyList(&habAnchor->hclWindows);
   CmnLstDestroyList(&habAnchor->hclClasses);
   CmnLstDestroyList(&habAnchor->hclTimers);

   hcmWork=habAnchor->hcmWork;
   CmnMemTerminate(&hcmWork);

   habAnchor=NULL;
   hmqQueue=NULL;

   //----------------------------------------------------------------------
   // Restore the cursor
   //----------------------------------------------------------------------
   VioGetCurType(&vciCursor,NULLHANDLE);
   vciCursor.attr=0;
   VioSetCurType(&vciCursor,NULLHANDLE);

   //----------------------------------------------------------------------
   // Clear the screen
   //----------------------------------------------------------------------
   abAttr[0]=' ';
   abAttr[1]=((((BYTE)VWCLR_BLACK) & 0x07)<<4)|(((BYTE)VWCLR_WHITE) & 0x07);

   VioScrollDn(0,0,25,80,999,abAttr,NULLHANDLE);

   return TRUE;
}

LONG EXPENTRY vwQuerySysValue(ULONG ulIndex)
//-------------------------------------------------------------------------
// This function returns the appropriate system value.
//
// Input:  ulIndex - specifies the VWSV_* constant representing the
//                   system value to return
// Returns:  system value if successful, -1 otherwise
//-------------------------------------------------------------------------
{
   //----------------------------------------------------------------------
   // Make sure we were initialized
   //----------------------------------------------------------------------
   if (hmqQueue==NULL) {
      return -1;
   } /* endif */

   if (ulIndex>=VWSV_CSYSVALUES) {
      return -1;
   } /* endif */

   return habAnchor->alSysValues[ulIndex];
}

BOOL EXPENTRY vwSetSysValue(ULONG ulIndex,LONG lValue)
//-------------------------------------------------------------------------
// This function sets a system value, if allowed.
//
// Input:  ulIndex - specifies the VWSV_* constant representing the
//                   system value to set
//         lValue - specifies the new value
// Returns:  TRUE if successful, FALSE otherwise or if the system value
//           is not allowed to be changed
//-------------------------------------------------------------------------
{
   //----------------------------------------------------------------------
   // Make sure we were initialized
   //----------------------------------------------------------------------
   if (hmqQueue==NULL) {
      return FALSE;
   } /* endif */

   if (ulIndex>=VWSV_CSYSVALUES) {
      return FALSE;
   } else {
      switch (ulIndex) {
      case VWSV_INSERTMODE:
         habAnchor->alSysValues[ulIndex]=(lValue!=0);
         break;
      default:
         return FALSE;
      } /* endswitch */
   } /* endif */

   return TRUE;
}

BOOL EXPENTRY vwSetFocus(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function sets the focus to another window.
//
// Input:  hwndWnd - specifies the handle of the window to receive the
//                   focus
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   HVWWND hwndLose;

   //----------------------------------------------------------------------
   // Validate the window handle
   //----------------------------------------------------------------------
   if ((hwndWnd!=NULL) && !vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   hwndLose=habAnchor->hwndFocus;

   //----------------------------------------------------------------------
   // Send a WM_SETFOCUS message to the window losing the focus.  Also,
   // send a WM_PAINT message (this deviates from PM).
   //----------------------------------------------------------------------
   if (hwndLose!=NULL) {
      habAnchor->hwndFocus=NULL;
      vwSendMsg(hwndLose,WM_SETFOCUS,MPFROMHWND(hwndWnd),MPFROMLONG(FALSE));
      vwSendMsg(hwndLose,WM_PAINT,0,0);
   } /* endif */

   //----------------------------------------------------------------------
   // Send a WM_SETFOCUS message to the window gaining the focus.  Again,
   // also send a WM_PAINT message (deviation).
   //----------------------------------------------------------------------
   if (hwndWnd!=NULL) {
      vwSendMsg(hwndWnd,
                WM_SETFOCUS,
                MPFROMHWND(hwndLose),
                MPFROMLONG(TRUE));
      habAnchor->hwndFocus=hwndWnd;
      vwSendMsg(hwndWnd,WM_PAINT,0,0);
   } else {
      habAnchor->hwndFocus=NULL;
   } /* endif */

   return TRUE;
}

HVWWND EXPENTRY vwQueryFocus(VOID)
//-------------------------------------------------------------------------
// This function returns the handle of the window currently with the
// focus.
//
// Returns:  window handle or NULLHANDLE if no window has the focus
//-------------------------------------------------------------------------
{
   //----------------------------------------------------------------------
   // Make sure we were initialized
   //----------------------------------------------------------------------
   if (hmqQueue==NULL) {
      return NULL;
   } /* endif */

   return habAnchor->hwndFocus;
}

BOOL EXPENTRY vwAlarm(USHORT usType)
//-------------------------------------------------------------------------
// This function beeps according to the type of beep specified.
//
// Input:  usType - a WA_* constant specifying the beep type
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   ULONG ulFreq;
   ULONG ulDuration;

   //----------------------------------------------------------------------
   // Make sure we were initialized
   //----------------------------------------------------------------------
   if (hmqQueue==NULL) {
      return FALSE;
   } /* endif */

   switch (usType) {
   case WA_NOTE:
      ulFreq=VWSV_NOTEFREQ;
      ulDuration=VWSV_NOTEDURATION;
      break;
   case WA_WARNING:
      ulFreq=VWSV_WARNINGFREQ;
      ulDuration=VWSV_WARNINGDURATION;
      break;
   case WA_ERROR:
      ulFreq=VWSV_ERRORFREQ;
      ulDuration=VWSV_ERRORDURATION;
      break;
   default:
      return FALSE;
   } /* endswitch */

   ulFreq=vwQuerySysValue(ulFreq);
   ulDuration=vwQuerySysValue(ulDuration);
   DosBeep(ulFreq,ulDuration);
   return TRUE;
}
