rem 
rem $Header: dbmsprobe.sql 7020200.1 95/02/15 18:31:25 cli Generic<base> $ 
rem 
Rem  Copyright (c) 1992 by Oracle Corporation 
Rem    NAME
Rem      dbmsprobe.sql - The PL/SQL debugger (PROBE) packages
Rem    DESCRIPTION
Rem      This huge file is all the DE packages put together.
Rem      This was done for convenience and the original sources
Rem      can be found in the knlde directory of a V7PLS dve.
Rem
Rem      The files that were concatenated here were
Rem          pbutl.sql
Rem          pbreak.sql
Rem          pbsde.sql
Rem          pbrph.sql
Rem
Rem      Detailed description of each package can be found later in this file
Rem
Rem    RETURNS
Rem 
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem    MODIFIED   (MM/DD/YY)
Rem     jmallory   01/17/95 -  222085 - triggers from client-side
Rem     jmallory   08/15/94 -  Add PBRPH.shutdown
Rem     jmallory   07/24/94 -  Dont default pipe base in PBRPH.attach
Rem     rhari      07/18/94 -  changes to pbutl and pbrph
Rem     rhari      07/05/94 -  Creation

connect internal

Rem    NAME
Rem      pbutl.sql - Plsql deBugger UTiLity package.
Rem    DESCRIPTION
Rem      Defines constants and types used by the various debugger packages.
Rem    RETURNS
Rem 
Rem    NOTES
Rem      
create or replace package pbutl is

  -- Largest amount to read/write to pipe. (Should include a healthy fudge
  -- factor, since the message status must also go in the message )
  --
  -- $$$ This value could probably be larger still $$$  
  pipe_buffer_size  constant binary_integer := 1000;

  -- source_table is used to transfer large objects (like the entire source
  -- for a unit) over the pipe.
  -- Should be varchar2(pipe_buffer_size) (except that isn't allowed)
  type source_table_type is table of varchar2(1000) index by binary_integer;


  -- the string to concatenate to the end of pipe_base to obtain the 
  -- the input and output pipes.
  pls_to_de_extension  CONSTANT  varchar2(10) := '$PLS_TO_DE';
  de_to_pls_extension  CONSTANT  varchar2(10) := '$DE_TO_PLS';


  -- Some requests that can be read from the input pipe
  request_continue                  CONSTANT binary_integer := 0;
  --request_step                      CONSTANT binary_integer := 1;
  --request_step_into                 CONSTANT binary_integer := 2;
  --request_step_return               CONSTANT binary_integer := 3;
  request_quit                      CONSTANT binary_integer := 4;

  request_set_breakpoint_entry      CONSTANT binary_integer := 10;
  request_set_breakpoint_line       CONSTANT binary_integer := 11;
  request_set_breakpoint_except     CONSTANT binary_integer := 12;
  --request_set_debug_flags           constant binary_integer := 13;
  request_get_debug_flags           constant binary_integer := 14;

  request_delete_breakpoint         CONSTANT binary_integer := 20;
  request_disable_breakpoint        CONSTANT binary_integer := 21;
  request_enable_breakpoint         CONSTANT binary_integer := 22;

  request_display_frame             CONSTANT binary_integer := 30;
  request_print_backtrace           CONSTANT binary_integer := 31;
  request_print_source              CONSTANT binary_integer := 32;
  request_print_source_location     CONSTANT binary_integer := 33;
  request_print_pcode_line          CONSTANT binary_integer := 34;
  request_print_breakpoints         CONSTANT binary_integer := 35;

  request_abort                     CONSTANT binary_integer := 100;
  request_unknown                   CONSTANT binary_integer := 999;


  -- libunit types (used when setting breakpoints)
  -- (These correspond to KGLNPRCD and KGLNBODY, from kgl.h)
  libunit_spec                 CONSTANT binary_integer := 1;
  libunit_body                 CONSTANT binary_integer := 2;
  libunit_procedure            CONSTANT binary_integer := 1;


  -- Events (reasons) that can be read from the pipe: these MUST
  -- correspond exactly to those in pfrdef.h
  de_none                constant binary_integer := 0;
  de_bogus_init          constant binary_integer := 1;
  de_init                constant binary_integer := 2;
  de_breakpoint          constant binary_integer := 3;
  de_breakpoint_line     constant binary_integer := 4;
  de_breakpoint_entry    constant binary_integer := 5;
  de_enter               constant binary_integer := 6;
  de_return              constant binary_integer := 7;
  de_finish              constant binary_integer := 8;
  de_line                constant binary_integer := 9;
  de_interrupt           constant binary_integer := 10;
  de_exception           constant binary_integer := 11;
  de_icd_call            constant binary_integer := 12;
  de_icd_return          constant binary_integer := 13;
  de_watch               constant binary_integer := 14;
  de_exit                constant binary_integer := 15;
  de_exception_handler   constant binary_integer := 16;
  de_timeout             constant binary_integer := 17;
  de_rpc                 constant binary_integer := 18;
  de_unhandled_exception constant binary_integer := 19;
  de_instantiate         constant binary_integer := 20;
  de_abort               constant binary_integer := 21;


  -- de_client statuses: these are messages from the de_client that have to
  -- do with the success or failure of the requested operation.
  -- Make sure that these dont overlap the de statuses immediately above.
  --
  be_success               CONSTANT binary_integer := 1000;
  be_failure               CONSTANT binary_integer := 1001;
  be_warning               CONSTANT binary_integer := 1002;
  be_exception             CONSTANT binary_integer := 1003;
  -- be_indexed_write : this message is part of a multi-line message
  be_indexed_write         CONSTANT binary_integer := 1004;

  -- Additional messages
  tidl_coming           constant binary_integer := 2000;
  tidl_failure          constant binary_integer := 2001;
  send_next_tidl        constant binary_integer := 2002;
  source_coming         constant binary_integer := 2010;
  source_failure        constant binary_integer := 2011;
  send_next_source      constant binary_integer := 2012;

  --
  -- Context passed to pbrph entry-points. The fields are basically for 
  -- debugging the debugger.
  --
  type de_context is record (
    status     binary_integer,
    be_status  binary_integer,
    terminated boolean := false
  );

end pbutl;
/
show errors

drop public synonym pbutl;
create public synonym pbutl for sys.pbutl;
grant execute on pbutl to public;



-- Breakpoint package


Rem    NAME
Rem      pbreak.sql -  Plsql deBugger bREAKpoint package.
Rem    DESCRIPTION
Rem      This package is the pl/sql API for the breakpoint interface.
Rem      (The corresponding C interface is peibpt.h).
Rem      
Rem    RETURNS
Rem 
Rem    NOTES
Rem      Should be installed as SYS.
-- Note about the pragma interfaces: at some point in the (near) future, the
-- syntax will include a "target name" for ICDs: 
--
--    pragma interface(C, long-meaningful-plsql-name, short-OCCS-C-name);
--
-- and then P2ICD will generate a table of the short-OCCS-C-name's. 
-- So we dont need to worry about the naming of procedures in plsql packages 
-- merely because they are going to be implemented as ICDs.
--
-- (Note that the rdbmsfolk have already gone and done this: for example see
--  the dbmspipe ICDs. They dont use p2icd to generate their ICDs).
--


drop table pbreak_raw_table;
create table pbreak_raw_table (a long);


create or replace package pbreak is

  -- Public types:
  --   (mostly copied from pdbspc.pls)

  -- (peidef, plspgadf, kglhd are currently implemented via binary_integer
  --  due to problems with passing to/from ICD's).
  --
  type peidef   is record (dummy raw(12));   -- opaque
  type plspgadf is record (dummy raw(12));   -- opaque
  type kglhd    is record (dummy raw(12));   -- opaque

  type perst    is record (dummy raw(12));   -- opaque

  -- peibpa drives the "one-call-does-all" get/set interface: the mask 
  -- indicates which of the fields are of interest.
  type peibpa is record (
        mask            binary_integer,
        percflags       binary_integer,
        framecount      binary_integer,
        line            binary_integer,
        breakpoint      binary_integer,
        ith_peibpa      binary_integer,
        ith_frame       perst
  );

  -- mask values for peibpa
  mask_percflags        constant binary_integer := 1;
  mask_framecount       constant binary_integer := 2;
  mask_line             constant binary_integer := 4;
  mask_breakpoint       constant binary_integer := 8;
  mask_ith_peibpa       constant binary_integer := 16;
  mask_ith_frame        constant binary_integer := 32;

  ------------------  Probe Debug levels ---------------
  -- The possible values to pass to set_debug (see below).
  --
  -- debug_rpc_only   : the interpreter will run in debug-mode on the server
  --                    ONLY if it is started from a PLSQL RPC call.
  -- debug_everything : the interpreter will always run in debug-mode on the
  --                    server
  --
  -- (These values MUST match DE_DEBUG_ON, DE_DEBUG_PEND in pliu.h)
  debug_off             constant binary_integer := 0;
  debug_rpc_only        constant binary_integer := 1;  -- (DE_DEBUG_PEND)
  debug_everything      constant binary_integer := 2;  -- (DE_DEBUG_ON)

  --------------------- Public utility functions --------------

  -- set_debug: set the debug status in the UGA. 
  --   Subsequent calls to plsql (in this session) will be debugged.
  -- (Should later be modified to take a perc probably. For now we are leaving
  --  the perc out for ease of use from sqldba).
  --
  procedure set_debug(debug_level in binary_integer := debug_everything);

  -- clear_debug: clear the UGA debug status.
  -- Subsequent calls to plsql (in this session) will NOT be debugged.
  --
  procedure clear_debug;


  -- Print the pcode corresponding to current PC.
  procedure print_pcode_line(perc in peidef, lin out varchar2);

  -- Print the source line corresponding to current PC.
  procedure print_source(perc        in     peidef, 
                         low         in     binary_integer,
                         high        in     binary_integer,
                         window      in     binary_integer,
                         print_arrow in     binary_integer,
                         srctab      in out pbutl.source_table_type,
			 srctab_size in out binary_integer);

  -- Print out the current line number and the unit name.
  function print_source_location(perc in peidef, 
                                  buf  in out varchar2) return boolean;

  -- Return the plsp associated with perc.
  function get_plsp_from_perc(perc in peidef) return plspgadf;



  --------------------- Breakpoint interface -----------------------------
  -- This interface specifies how to set (and delete/disable) breakpoints, and
  -- how to get information about where breakpoints may be set for a given 
  -- libunit.
  --
  -- (These are mostly pl/sql wrappers for ICD's that call Frank's breakpoint
  --  API routines).

  -- set/get_multiple - access multiple attributes via the peibpa record.
  --
  function set_multiple(perc in peidef, bpap in out peibpa) return boolean;
  function get_multiple(perc in peidef, bpap in out peibpa) return boolean;


  -- Set_break_entry : set a breakpoint at the supplied entrypoint.
  -- Set_break_line  : set a breakpoint at the supplied line number.
  --       Both return the breakpoint number, or 0 for failure.
  --
  -- ub4 peibpentry(/*_ plspgadf *plsp, kglhd *lu, ub4 entry _*/);
  -- ub4 peibpline(/*_ plspgadf *plsp, kglhd *lu, ub4 line _*/);
  --
  function set_break_entry(plsp in plspgadf,
                           unit_name in varchar2,
                           unit_type in binary_integer,
                           entry in binary_integer) return binary_integer;

  function set_break_line(plsp in plspgadf,
                          unit_name in varchar2,
                          unit_type in binary_integer,
                          entry in binary_integer) return binary_integer;


  -- Delete_breakpoint : delete the specified breakpoint (which should be
  --                     the return-value from set_break_entry etc.)
  --
  -- void peibpdelete(/*_ plspgadf *plsp, ub4 bpt_index _*/);
  --
  procedure delete_breakpoint(breakpoint in binary_integer);

  -- enable_breakpoint  : enables a (previously-set) breakpoint
  -- disable_breakpoint : disables a (previously-set) breakpoint
  -- 
  -- void peibpenable(/*_ plspgadf *plsp, ub4 bpt_index _*/);
  -- void peibpdisable(/*_ plspgadf *plsp, ub4 bpt_index _*/);
  --
  procedure enable_breakpoint(breakpoint in binary_integer);
  procedure disable_breakpoint(breakpoint in binary_integer);

  -- /* Get line number and entry information for a lib unit */
  -- void peibplu_info(/*_ plspgadf *plsp, kglhd *lu, bitvec *bit_table, 
  --             ub4 bit_table_size, ub4 *max_entries_p, ub4 *max_lines_p
  --         _*/);
  -- 
  --procedure get_line_and_entry_info(plsp in plspgadf, lu in kglhd,
  --                                 bit_table out <some table>,
  --                                 bit_table_size in binary_integer,
  --                                 max_entries in binary_integer,
  --                                 max_lines in binary_integer);


  --------------------- Display interface ------------------------------
  -- Interface for getting information about the current state of a suspended
  -- program.
  -- ($$$ We need to address the problem of putting together multiple chunks
  --      of the display: eg. reconstructing a call-stack that includes one
  --      or more RPC calls).
  
  -- display_frame : display one or more frames from the current stack.
  procedure display_frame(
              perc        in           peidef,
              start_frame_depth in     binary_integer,
              frame_count in     binary_integer,       -- # of frames to fetch
              flags in binary_integer,                 -- ????
              max_string_length in     binary_integer, -- longest string
              max_index_values  in     binary_integer, -- max table values
              tidl_buf          in out raw,           -- buf to place result in
              -->        IN just for debugging <--
              more out boolean);                       -- more pieces to come
              
   -- display_frame_more : call to get more of the frame, if display_frame
   --                      indicated that it couldn't finish in one call.
   procedure display_frame_more(tidl_buf out raw,
                                more out boolean);



  --------------------- Interpreter Control interface ------------------
  -- Interface for setting the debug flags that are used by the interpreter for
  -- control. (eg. breaking on function-call, stepping one line, ...)

  -- 
  -- Set percdefl to indicate interesting events on which to break.
  --   "request" must be one of the pbutl.request_* constants.
  -- 
  -- void peibpfset(/*_ peidef *perc, ub4 flags _*/);
  -- ub4 peibpfget(/*_ peidef *perc _*/);

  function set_debug_flags(perc in peidef, request in binary_integer)
                          return binary_integer;

  function get_debug_flags(perc in peidef) return binary_integer;


  ------------------  Debugging routines: -----------------------------
  procedure print_all_breakpoints(plsp in plspgadf, listing in out varchar2);
  procedure print_backtrace(perc in peidef, listing out varchar2);

  -- For debugging the debugger: writes a message to the trace file.
  procedure debug_message(buf in varchar2);

end pbreak;
/
show errors


create or replace package body pbreak is

  ------------- pragma interfaces -----------
  --
  -- NOTE: returning record-types from ICD seems to mess up the argc, so
  --       for now we use raw (and have the plsql wrapper deal with it).
  --

  -- (most of these could go directly into the package header, if we
  --  only allowed that...)

  --procedure pokeme(perc in peidef, reason in binary_integer,
  --                 buf out varchar2);
  --pragma interface(c, pokeme);

  -- set_multiple : set multiple fields
  --> $$$ NYI.
  function set_multiple(perc in peidef, bpap in out peibpa) return boolean is
  begin
    return FALSE;
  end set_multiple;

  -- get_multiple : get multiple fields
  --> $$$ NYI.
  function get_multiple(perc in peidef, bpap in out peibpa) return boolean is
  begin
    return FALSE;
  end get_multiple;

  -- pbbsd : PBB Set Debug
  procedure pbbsd(perc in raw, debug_level in binary_integer);
  pragma interface(c, pbbsd);                               -- 1 (ICD #1)

  -- pbbcd : PBB Clear Debug
  procedure pbbcd(perc in raw);
  pragma interface(c, pbbcd);                               -- 2


  -- PBBHS - return Handle from String
  --     ($$$ Should return kglhd, but ICAL is broken)
  function pbbhs(unitname in varchar2,
                  unit_type in binary_integer) return raw;
  pragma interface(c, pbbhs);                               -- 3

  -- PBBBE set Break for Entry
  function pbbbe(plsp in raw, lu in raw,
                  entry in binary_integer) return binary_integer;
  pragma interface(c, pbbbe);                               -- 4

  -- PBB set Break for Line
  function pbbbl(plsp in raw, lu in raw,
                  entry in binary_integer) return binary_integer;
  pragma interface(c, pbbbl);                               -- 5

  -- PBB Remove (Delete) Breakpoint
  procedure pbbrb(breakpoint in binary_integer);
  pragma interface(c, pbbrb);                               -- 6

  -- PBB Enable Breakpoint
  procedure pbbeb(breakpoint in binary_integer);
  pragma interface(c, pbbeb);                               -- 7

  -- PBB Disable Breakpoint
  procedure pbbdb(breakpoint in binary_integer);
  pragma interface(c, pbbdb);                               -- 8

  -- PBB Plsp from Perc
  function pbbpfp(perc in raw) return raw;
  pragma interface(c, pbbpfp);                              -- 9

  -- PBB Print PCode Line
  procedure pbbppl(perc in raw, lin out varchar2);
  pragma interface(c, pbbppl);                              -- 10

  -- PBB Print BreakpointS
  procedure pbbpbs(plsp in raw, listing out varchar2);
  pragma interface(c, pbbpbs);                              -- 11

  -- PBB Set Perc FLAGS
  function pbbspf(perc in raw, request in binary_integer) 
                     return binary_integer;
  pragma interface(c, pbbspf);                              -- 12

  -- PBB Get Perc FLAGS
  function pbbgpf(perc in raw) return binary_integer;
  pragma interface(c, pbbgpf);                              -- 13

  -- PBB Get Line Number
  procedure pbbgln(perc  in raw, 
                   linnum        out binary_integer,
                   unit_name  in out varchar2,
                   namespace     out binary_integer,
                   unit_owner in out varchar2,
                   dblink     in out varchar2);
  pragma interface(c, pbbgln);                              -- 14
  
  -- PBB Print Source Line
  procedure pbbpsl(perc        in     raw,
                   low         in     binary_integer, 
                   high        in     binary_integer, 
                   window      in     binary_integer,
                   print_arrow in     binary_integer,
                   srclin      in out varchar2,
		   srclin_size in     binary_integer,
		   next_low       out binary_integer,
		   next_high      out binary_integer);
 
  pragma interface(C, pbbpsl);                              -- 15

  procedure get_tidl_frame(perc              in     raw,
                           start_frame_depth in     binary_integer,
                           frame_count       in     binary_integer,
                           flags             in     binary_integer,
                           max_string_length in     binary_integer,
                           max_index_values  in     binary_integer,
                           buf                  out raw
                           --,more out boolean
                           );
  pragma interface(c, get_tidl_frame);                      -- 16 (pbbgtf)
  
  
  procedure do_print_backtrace(perc in raw, listing out varchar2);
  pragma interface(c, do_print_backtrace);                  -- 17 (pbbpbt)

  procedure debug_message_aux(buf in varchar2);
  pragma interface(c, debug_message_aux);                   -- 18 (pbbwdm)

  ----- end pragma interfaces -------


----------------------------------------------------------------------------
-- Set the debug bit in UGA. 
-- Actually this is no longer a "bit" - it is a 3-state variable that
-- indicates one of: "no debugging," "debug only RPCs," "debug everything."
--
procedure set_debug(debug_level in binary_integer := debug_everything) is
  perc peidef;
begin
  perc.dummy := null;
  pbbsd(perc.dummy, debug_level);
end set_debug;


procedure clear_debug is
  perc peidef;
begin
  perc.dummy := null;
  pbbsd(perc.dummy, debug_off);
end clear_debug;

----------------------------------------------------------------------------
-- Print the current source line, or some subset of the current source.
--
-- Writes the lines into srctab (1-based).
-- Call pbbpsl repeatedly until next_low comes back as 0: then we are done.
--
procedure print_source(perc        in     peidef, 
                       low         in     binary_integer,
                       high        in     binary_integer,
                       window      in     binary_integer,
                       print_arrow in     binary_integer,
		       srctab      in out pbutl.source_table_type,
                       srctab_size in out binary_integer) is

  linnum     binary_integer := 0;
  unit_name  varchar2(100) := '';
  unit_owner varchar2(100) := '';
  dblink     varchar2(100) := '';
  namespace  binary_integer;
  current_low  binary_integer := low;
  current_high binary_integer := high;
  current_window binary_integer := window;
  next_low     binary_integer := 1;
  next_high    binary_integer := 0;
begin

    -- $$$ pbbgln is called so that we have the line-number for the
    --     exception-handler (in case anything goes wrong in pbbpsl)
    pbbgln(perc.dummy, linnum, unit_name, namespace, unit_owner, dblink);
    
    srctab_size := 0;   -- assume failure
    while (next_low <> 0) loop
      srctab(srctab_size+1) := '+=+=+=EMPTY+=+=+=+=';
      pbbpsl(perc.dummy, current_low, current_high, current_window, 
             print_arrow, 
             srctab(srctab_size+1), 
	     pbutl.pipe_buffer_size,
	     next_low, next_high);
      if (srctab(srctab_size+1) is not null) then
        -- successfully wrote a line into the table
	srctab_size := srctab_size + 1;
	current_low := next_low;
	current_high := next_high;
	current_window := 0;
      else
        return;   -- failure
      end if;
    end loop;
exception
  when others then
    srctab_size := 0;
    --lin := unit_name || ' [' || linnum || '] ** Source not available **';
end print_source;

----------------------------------------------------------------------------
-- Print the current pcode line.
--
procedure print_pcode_line(perc in peidef, lin out varchar2) is
begin
  pbbppl(perc.dummy, lin);    -- $$$
exception
  when others then
     lin := null;
end print_pcode_line;

----------------------------------------------------------------------
-- Get information about the current unit. This includes the line-number,
-- the name of the unit, and the type of the unit.
-- $$$ Anything else needed?
--
function print_source_location(perc in     peidef,
                               buf  in out varchar2) return boolean is

  linnum     binary_integer := -1;
  namespace  binary_integer := -1;
  unit_name  varchar2(100);
  unit_owner varchar2(100);
  dblink     varchar2(100);
  
begin
  buf := '';
  pbbgln(perc.dummy, linnum, unit_name, namespace, unit_owner, dblink );
  if (namespace is null) then
    buf := linnum;
    return false;
  else
    buf := linnum || ' ' ||
           unit_name || ' ' ||
           namespace || ' ' || 
           unit_owner || ' ' || 
           dblink;
     -- $$$DEBUGGING
     --insert into pbreak_raw_table values(
     --   linnum || ' ' || unit_name || ' ' || namespace || ' ' || unit_owner 
     --   || ' ' || dblink);
     --commit;
     --    end DEBUGGING
    return true;
  end if;
exception
  when others then buf := linnum || ' ' || -1 || ' ** Unit not available ** ';
  return false;
end print_source_location;

----------------------------------------------------------------------
function get_plsp_from_perc(perc in peidef) return plspgadf is
  plsp plspgadf;
begin
  plsp.dummy := pbbpfp(perc.dummy);  -- $$$
  return plsp;
end get_plsp_from_perc;


----------------------------------------------------------------------
function set_break_entry(plsp in plspgadf, 
                         unit_name in varchar2,
                         unit_type in binary_integer,
                         entry in binary_integer) return binary_integer is
  lu       kglhd;
  breaknum binary_integer := null;
begin
  lu.dummy := pbbhs(unit_name, unit_type);   -- $$$
  if (lu.dummy is not null) then
    breaknum := pbbbe(plsp.dummy, lu.dummy, entry);
  end if;
  return breaknum;
end set_break_entry;


----------------------------------------------------------------------
function set_break_line(plsp in plspgadf,
                         unit_name in varchar2,
                         unit_type in binary_integer,
                         entry in binary_integer) return binary_integer is
  lu       kglhd;
  breaknum binary_integer := null;
begin
  lu.dummy := pbbhs(unit_name, unit_type);  -- $$$
  if (lu.dummy is not null) then
    breaknum := pbbbl(plsp.dummy, lu.dummy, entry);
  end if;
  return breaknum;
end set_break_line;


--procedure pokeme1(perc in peidef,
--                  reason in binary_integer,    -- what to do 
--                  buf out varchar2      -- where to put the results
--                  ) is
--begin
--  pokeme(perc, reason, buf);
--end pokeme1;
--

----------------------------------------------------------------------
-- get_handle : return the kglhd associated with the given name.
--              (unitname must include the entire name).
--
function get_handle(unitname in varchar2, unit_type in binary_integer)
                    return kglhd is
  lu kglhd;
begin
  lu.dummy := pbbhs(unitname, unit_type);  -- $$$
  return lu;
end get_handle;


----------------------------------------------------------------------
procedure delete_breakpoint(breakpoint in binary_integer) is
begin
  pbbrb(breakpoint);
end delete_breakpoint;


----------------------------------------------------------------------
procedure enable_breakpoint(breakpoint in binary_integer) is
begin
  pbbeb(breakpoint);
end enable_breakpoint;


----------------------------------------------------------------------
procedure disable_breakpoint(breakpoint in binary_integer) is
begin
  pbbdb(breakpoint);
end disable_breakpoint;

----------------------------------------------------------------------
-- Create a listing of all the current breakpoints that the runtime is
-- aware of. 
--
procedure print_all_breakpoints(plsp in plspgadf, listing in out varchar2) is
begin
  pbbpbs(plsp.dummy, listing);
exception
  when others then listing := null;
end print_all_breakpoints;

----------------------------------------------------------------------
-- Set perc debug flags to indicate debugging events.
-- Return value: 
--  0 (false) --> fail
--  1 (true)  --> succeed
--
-- $$$ We used to handle the flags according to constants declared in 
--     pbutl, but now we're pushing this off onto the client. So just pass
--     the request through.
--
function set_debug_flags(perc in peidef, request in binary_integer)
           return binary_integer is
begin 
    return pbbspf(perc.dummy, request);
exception
  when others then return 0;
end set_debug_flags;

----------------------------------------------------------------------
-- Return current value of perc debug flags.
--
function get_debug_flags(perc in peidef) return binary_integer is
begin
  return pbbgpf(perc.dummy);
  exception
    when others then return -1;
end get_debug_flags;


--------------------------------------------------------------------------------
-- display_frame : display one or more frames from the current stack.
-- For now it is required that the whole listing fit into one varchar2: the
-- "more" argument may actually be used later.
--
procedure display_frame(
              perc        in          peidef,
              start_frame_depth in    binary_integer, -- frame to start from
              frame_count in          binary_integer,  -- # of frames to fetch
              flags       in          binary_integer,  -- ????
              max_string_length in    binary_integer, -- longest string
              max_index_values  in    binary_integer, -- max table values
              tidl_buf          in  out raw,      -- buf to place result in
              more out boolean                        -- couldn't get all?
              ) is
begin
  more := false;
  get_tidl_frame(perc.dummy, 
                 start_frame_depth, frame_count, flags,
                 max_string_length, max_index_values,
                 tidl_buf 
                 --, more
                 );

  -- DEBUGGING --
  --declare
  --  v varchar2(32767);
  --begin
  --  insert into pbreak_raw_table values (tidl_buf);
  --  v := tidl_buf;
  --  insert into de_debug_session_table(msg)  
  --           values('lengthb: ' || lengthb(v));
  --  commit;
  --end;
  -- DEBUGGING --

end display_frame;  
              
------------------------------------------------------------------------------
-- display_frame_more : call to get more of the frame, if display_frame
--                      indicated that it couldn't finish in one call.
--
procedure display_frame_more(tidl_buf out raw,
                             more out boolean) is
begin
  more := true;
  tidl_buf := '** ERROR ERROR ERROR ERROR **';
end display_frame_more;

----------------------------------------------------------------------
--
procedure print_backtrace(perc in peidef, listing out varchar2) is
begin
  do_print_backtrace(perc.dummy, listing);
end print_backtrace;

----------------------------------------------------------------------
-- Dump a debug message to the tracefile.
procedure debug_message(buf in varchar2) is
begin
  debug_message_aux(buf);      -- Call the ICD
end debug_message;

end pbreak;
/
show errors

drop public synonym pbreak;
create public synonym pbreak for sys.pbreak;
grant execute on pbreak to public;



Rem    NAME
Rem      pbsde.sql - Plsql deBugger Server-side DE package.
Rem    DESCRIPTION
Rem      This package acts as the server-side equivalent of DE. The user of the
Rem      package passes a perc to debug_loop, and the operations available on
Rem      the perc should be identical to those available to DE on the client-
Rem      side with a suspended perc.
Rem    RETURNS
Rem 
Rem    NOTES
Rem      This package is the default "back-end" for server-side debugging. 
Rem      It is passed a perc and then reads operations from a pipe and executes
Rem      them on the perc. It is called from the server-side DE callback, which
Rem      is called from the plsql interpreter whenever a DE event happens.
Rem      Should be installed as SYS.
Rem
Rem      Catastrophic errors (eg. pipe not open) are signalled by writing
Rem      to the trace file (see pbreak.debug_message).
Rem      Less drastic errors are handled by sending messages over the pipe
Rem      to the front-end. 
Rem
Rem  Messages: 
Rem      Messages consist of a message-number (see pbutl) and additional args.
Rem      The message is packed into one varchar2 and read/written as a unit
Rem      to the output/input pipe.
Rem      The maximum message size depends on the buffer-size that the 
Rem      dbms_pipes package was built with.
Rem
Rem      Messages should probably be compacted at some point in the future: 
Rem      for now a simple text format will suffice.
Rem   
Rem
 
-- Usage:
-- 1. init()   
--    Set up the pipe-names.
--    init *must* be called before any output is placed on the pipe.
--    If passed the arg 'true' and serveroutput is enabled, dumps the pipe-base
--    to stdout. (Useful when debugging via separate sqldba sessions).
--
-- 2. debug_loop() : sit in a loop writing messages to output_pipe and reading
--    messages from input_pipe and executing them.
--    Exit when the message is to continue.
--
-- 3. shutdown() cleans up. (Dumps a termination message to the output pipe).
--


-- connect internal

create or replace package pbsde is

  default_timeout  constant binary_integer := 3600;  -- 1 hour
  next_item_vc2    constant binary_integer := 9;  -- dbms_pipe code for varchar2
  next_item_number constant binary_integer := 6;  -- dbms_pipe code for number
  pipe_failure     constant binary_integer := -1;
  crlf             constant varchar2(2) := '
';


  -- debug_loop : the top-level procedure called from the interpreter upon
  --              encountering an event. Writes the reason to the pipe and
  --              then executes requests from the pipe until told to 
  --              continue.
  -- (should probably be renamed psdevn)
  --
  function debug_loop(perc in pbreak.peidef,
                      reason in binary_integer
                     ) return binary_integer;

  -- initialize (generate the pipe names)
  procedure init(base in varchar2 := null, notify in boolean := false);

  -- Shutdown (ie. close the pipes.) 
  -- $$$ We might want to send some kind of termination message to the pipes
  --     upon shutdown, just to make sure that anyone who is listening 
  --     understands that they wont be hearing any more.
  --
  procedure shutdown(notify in boolean := false);

  -- Use dbms_output to print the current pipe names. (Used when debugging
  -- via sqldba when the pipe-names are needed to feed to the attach
  -- command. See the front-end).
  --
  procedure stat;

  -- Get_pipe_base does the same thing as stat() but returns the pipe_base.
  -- It is provided because variable-references via RPC do not (currently?)
  -- work.
  -- It is used as part of the RPC implementation.
  function get_pipe_base return varchar2;

  -- write msg to output_pipe
  function write_pipe(status in binary_integer,
                      msg varchar2) return binary_integer;

  -- read a message from input_pipe
  function read_pipe(msg           in out varchar2, 
                     expected_type in     binary_integer,
                     timeout       in     binary_integer := default_timeout
                    ) return binary_integer;

  -- Handle one request (set a breakpoint, continue, etc.)
  -- Puts a reply to the output pipe indicating the status of the request.
  --
  -- "continue" is set to true if the request was to step, continue, or quit.
  -- 
  function handle_request(plsp in pbreak.plspgadf,
                           perc in pbreak.peidef,
                           request in binary_integer,
                           msg in varchar2,
                           continue out boolean
                           ) return binary_integer;



  ------  Global state variables: ------

  -- pipe_base is used as a base to construct input_pipe and output_pipe.
  -- pipes_open is true when the pipe-names have been constructed (by a call
  -- to init()).
  output_pipe  varchar2(50) := NULL;
  input_pipe   varchar2(50) := NULL;
  pipe_base    varchar2(50) := NULL;
  pipes_open   boolean := false;

  -- Debugging variables:
  -- Set "debugging" to true in order to get regular informative messages about
  --     what the backend is doing.
  -- Set "pipe_debugging" to true in order to get messages about every pipe
  --     operation. (Warning: this may be too verbose).
  debugging       boolean := false;
  pipe_debugging  boolean := false;

end pbsde;
/
show errors;

create or replace package body pbsde is 

----------------------------------------------------------------------------
procedure print_pipe_names(msg in varchar2) is
begin
  --dbms_output.put_line('***** PBSDE: ' || msg || '*****');
  --dbms_output.put_line('  pipe base is        : ' || pipe_base);
  --dbms_output.put_line('  (output pipe is named: ' || output_pipe || ')');
  --dbms_output.put_line('  (input pipe is named:  ' || input_pipe  || ')');
  --dbms_output.new_line;
  dbms_output.put_line('  execute pbrph.attach(''' || pipe_base || ''')');
end print_pipe_names;

----------------------------------------------------------------------------
-- Initialize: set up the input and output pipe names for use by this 
-- session. 
--
procedure init(base in varchar2 := null, notify in boolean := false) is
begin
  --output_pipe := 'DE_PROGRAM_PIPE';
  --input_pipe := 'DE_USER_PIPE';

  if (base is not null) then
    -- Mostly for ease of debugging.
    pipe_base := base;
  else
    pipe_base := dbms_pipe.unique_session_name;
  end if;

  if (debugging) then
    pbreak.debug_message('Probe:SDE:init: pipe base: "' || base || '"');
  end if;

  output_pipe := pipe_base || pbutl.pls_to_de_extension;
  input_pipe :=  pipe_base || pbutl.de_to_pls_extension;

  /* Create secure pipes */
  declare
    dummy integer;
  begin
    /* $$$ Amit says we can ignore the retval 'dummy' */
    dummy := dbms_pipe.create_pipe(input_pipe);
    dummy := dbms_pipe.create_pipe(output_pipe);
  end;

  pipes_open := true;

  if (notify) then
    print_pipe_names('INIT');
  end if;
end init;

----------------------------------------------------------------------------
-- Echo the pipe names via dbms_output. For use in sqldba.
-- 
procedure stat is
begin
  if (pipes_open) then
    print_pipe_names('STAT');
  else
    dbms_output.put_line('STAT: no pipes currently open');
  end if;
end stat;

----------------------------------------------------------------------------
function get_pipe_base return varchar2 is
begin
  return pipe_base;
end get_pipe_base;

----------------------------------------------------------------------------
-- To be called at end of debug session. After calling shutdown, a new init
-- will be required in order to do any more debugging.
--
procedure shutdown(notify in boolean := false) is
begin

  pipes_open := false;

  --
  -- $$$ Send a termination message to the output pipe, and make sure that
  --     no data is on the input pipe.
  --
  -- retval := write_pipe('ABORT');

  /* Remove secure pipes */
  declare
    dummy integer;
  begin
    /* $$$ Amit says we can ignore the retval 'dummy' */
    dummy := dbms_pipe.remove_pipe(input_pipe);
    dummy := dbms_pipe.remove_pipe(output_pipe);
  end;

  if (notify) then
    dbms_output.put_line('***** PBSDE: shut down. *****');
  end if;

end shutdown;

----------------------------------------------------------------------------
-- Read a message from the input_pipe.
-- Unpack into the buffer if the type of the next message is expected_type.
--
-- Returns: 0 for success, 
--          else the error-code from receive_message 
--          else the type read by receive_message (if different from 
--             the requested type)
--          
--
-- $$$ Should probably pass in multiple params (or a record of the different
--     types) and just set the appropriate one. For now the only type
--     that is seen is vc2
--
function read_pipe(msg           in out varchar2, 
                   expected_type in     binary_integer,
                   timeout       in binary_integer := default_timeout
                   ) return binary_integer is
  retval     binary_integer;
  item_type  binary_integer;
begin
  retval := dbms_pipe.receive_message(input_pipe, timeout);
  if (retval <> 0) then 
    pbreak.debug_message('Probe:SDE:rp: receive failed with status:' || retval);
    return retval; 
  end if;

    -- success: make sure that we got the expected type
  item_type := dbms_pipe.next_item_type;
  if (item_type = expected_type) then
    dbms_pipe.unpack_message(msg);
    if (pipe_debugging) then
      pbreak.debug_message('Probe:SDE:rp: received: "' || msg || '"');
    end if;
    return retval;
  else
    pbreak.debug_message('Probe:SDE:rp: requested type:' || expected_type ||
                         ' but received type:' || item_type);
    return item_type;
  end if;
exception
  when others then pbreak.debug_message('Probe:SDE:rp: *** exception ***');
  return pipe_failure;
end read_pipe;

----------------------------------------------------------------------------
-- Write a message to the output_pipe.
-- The dbms_pipe message buffer size is limited, so the message may need to be
-- be broken into pieces.
--
function write_pipe(status in binary_integer,
                    msg varchar2) return binary_integer is
  msg_size   binary_integer := length(msg);
  num_pieces binary_integer;
  retval     binary_integer;
begin
  if (pipes_open) then
    -- $$$
    -- Figure out the number of pieces and send them individually
    --
    if (pipe_debugging) then
      pbreak.debug_message('Probe:SDE:wp: sending "' || msg || '"');
    end if;
    dbms_pipe.pack_message(status || ' ' || msg);
    retval := dbms_pipe.send_message(output_pipe);
    if (retval <> 0) then
      pbreak.debug_message('Probe:SDE:wp: send failed with status:' || retval);
    end if;
    return retval;
  else
    pbreak.debug_message('Probe:SDE:wp: pipe not open. Request: ' || msg);
    return pipe_failure;
  end if;
exception
  when others then pbreak.debug_message('Probe:SDE:wp: *** exception ***');
  return pipe_failure;
end write_pipe;

----------------------------------------------------------------------------
-- Read one word from vchar, ie. read up until end of vchar or until the
-- first whitespace. Return the index of the whitespace in "end_offset".
-- "finished" is true if the word consumed the rest of vchar.
-- 
procedure read_word(vchar in varchar2,
                    word out varchar2,
                    end_offset out binary_integer,
                    finished   out boolean) is
  tempv varchar2(100);
  offset binary_integer;
  space  binary_integer;
begin
  tempv := ltrim(vchar);
  if (tempv is null) then
    finished := true;
    return;
  end if;

  offset := length(vchar) - length(tempv);
  space := instr(tempv, ' ');
  end_offset := offset + space;

  if (space = 0) then
    -- last word in the string
    word := tempv;
    finished := true;
  else
    finished := false;
    word := substr(tempv, 1, space);
  end if;
--exception
--  when others then raise VALUE_ERROR;
end read_word;

----------------------------------------------------------------------------
-- Notes: perc is an in-param so we dont need to worry about writing back
--        its value upon return.
--        "reason" specifies the reason for this suspension
--
function debug_loop(perc in pbreak.peidef,
                    reason in binary_integer
                    ) return binary_integer is
  plsp      pbreak.plspgadf;
  msg       varchar2(100);
  retval    binary_integer;
  request   binary_integer;
  continue  boolean := false;
begin
  if not pipes_open then
    -- Things are not correctly set up: a call to debug_loop must always be
    -- preceded by a call to init (to set the pipe names). We could call init
    -- here ourselves but we cant be sure that the front-end will use the
    -- new names that we generate. So just return with error-status.
    pbreak.debug_message('Probe:SDE:dl: pipes not open; reason:' || reason);
    return 1;
  end if;

  if (debugging) then 
    pbreak.debug_message('Probe:SDE: debug_loop: reason ' || reason);
  end if;

  -- Tell the pipe that we have stopped, and the reason for stopping.
  retval := write_pipe(reason, 'suspended');
  if (retval <> 0) then return retval; end if;

  plsp := pbreak.get_plsp_from_perc(perc);
  
  -- Now sit in a loop reading requests from the pipe and executing them.
  -- Terminate when told to continue.
  while (not continue) loop
    -- read an action from the pipe and execute it
    retval := read_pipe(msg, next_item_vc2);
    if (retval <> 0) then return retval; end if;
    request := to_number(substr(msg,1,3));
    retval := handle_request(plsp, perc, request, substr(msg, 4), continue);
   if (retval <> 0) then return retval; end if;
  end loop;

  -- presumably everything went fine
  return 0;
exception
  when others then
    retval := write_pipe(pbutl.be_warning, '** Exception in debug_loop **');
end debug_loop;


----------------------------------------------------------------------------
-- Enable, disable, or delete a previously-set breakpoint.
--
procedure change_breakpoint_status(plsp in pbreak.plspgadf,
                                   breakpoint_number in binary_integer,
                                   breakpoint_op in binary_integer) is
  retval binary_integer;
  response varchar2(30);
begin
  if (breakpoint_op = pbutl.request_disable_breakpoint) then
    pbreak.disable_breakpoint(breakpoint_number);
    retval := write_pipe(pbutl.be_success, 
                         'breakpoint# ' || breakpoint_number || ' disabled');
  elsif (breakpoint_op = pbutl.request_enable_breakpoint) then
    pbreak.enable_breakpoint(breakpoint_number);
    retval := write_pipe(pbutl.be_success,
                         'breakpoint# ' || breakpoint_number || ' enabled');
  elsif (breakpoint_op = pbutl.request_delete_breakpoint) then
    pbreak.delete_breakpoint(breakpoint_number);
    retval := write_pipe(pbutl.be_success,
                         'breakpoint# ' || breakpoint_number || ' deleted');
  else
    retval := write_pipe(pbutl.be_warning,
                         '** Error: didnt understand message **');
  end if;
exception
  when others then
    retval := write_pipe(pbutl.be_warning,
                         '** Exception in change_breakpoint_status **');
end change_breakpoint_status;

----------------------------------------------------------------------------
-- Get the unit name (from which a kglhd can be obtained) and use that to
-- set the line or entry breakpoint number.
--
-- $$$ Enhancement: should be able to pass the entry-name and from that figure
--                  out the entry-number to set.
--
procedure set_breakpoint(plsp in pbreak.plspgadf,
                         request in binary_integer,
                         msg in varchar2) is
  endname   binary_integer;
  i         binary_integer;
  j         binary_integer;
  unit_name varchar2(50);
  temp      varchar2(50);
  entry     binary_integer;
  breaknum  binary_integer;
  unit_type binary_integer;
  retval    binary_integer;
begin
  temp := ltrim(msg);
  endname := instr(temp, ' ');
  -- ASSERT(endname <> 0)
  unit_name := rtrim(substr(msg,1,endname));

  temp := ltrim(substr(msg, endname+1));
  endname := instr(temp, ' ');
  unit_type := substr(temp,1,endname);

  temp := ltrim(substr(temp, endname+1));
  endname := instr(temp, ' ');
  if (endname = 0) then
    entry := substr(temp,1);
  else
    entry := substr(temp,1,endname);
  end if;

  if (debugging) then
    pbreak.debug_message('Probe:SDE:sb: req:' || request || ' name:' || 
                          unit_name || ' type:' || unit_type || ' entry:' || 
			  entry);
  end if;

  if (request = pbutl.request_set_breakpoint_entry) then
    breaknum := pbreak.set_break_entry(plsp, unit_name, unit_type, entry);
  else 
    breaknum := pbreak.set_break_line(plsp, unit_name, unit_type, entry);
  end if;

  if (breaknum is NULL) or (breaknum = 0) then
    --retval := write_pipe(pbutl.be_failure,
    --                     '*** Failed to set breakpoint ***');
    retval := write_pipe(pbutl.be_failure, 0);
  else
    --retval := write_pipe(pbutl.be_success, 'breakpoint# ' || breaknum);
    retval := write_pipe(pbutl.be_success, breaknum);
  end if;
exception
  when others then
    retval := write_pipe(pbutl.be_failure, 
                         '** Exception in set_breakpoint **');
end set_breakpoint;

----------------------------------------------------------------------
-- Break buf into pieces and put into the provided table. 
-- Set tabsize to be the size of the table. (The table is 1-based).
--
procedure disassemble(buf        in out varchar2,
                      result_tab in out pbutl.source_table_type,
                      tabsize    in out binary_integer) is
 len       binary_integer;     -- current length of buf
 piecelen  binary_integer;     -- length of next piece to write
 last      boolean := false;   -- are we done?
begin
  tabsize := 1;
  len := length(buf);
  -- break up buf into pieces and insert into the table
  while (len > 0) loop
    if (len > pbutl.pipe_buffer_size) then
      piecelen := pbutl.pipe_buffer_size;
    else
      last := true;
      piecelen := len;
    end if;
    result_tab(tabsize) := substr(buf,1,piecelen);
    if last then
      len := 0;
    else
      tabsize := tabsize+1;
      buf := substr(buf,piecelen+1);
      len := length(buf);
    end if;
  end loop;
end disassemble;

----------------------------------------------------------------------------
-- Write the contents of the table to the pipe.
--
-- First write: <notifier> <tabsize> to tell the far side what is coming and
-- how many pieces it is broken up into.
-- Then write out each entry in the table, pausing after each to receive a
-- response (except if this is the last entry).
-- Error if the response is not <confirmation>
--
procedure write_table(tab          in out pbutl.source_table_type,
                      tabsize      in     binary_integer,
	 	      notifier     in     binary_integer,
		      confirmation in     binary_integer) is

  retval    binary_integer;
  buf       varchar2(10000);   -- $$$ SIZE???
  tidl_tab  pbutl.source_table_type; -- holds the varchar form of the split TIDL
  response  varchar2(20);
begin
  
  if (debugging) then
    pbreak.debug_message('Probe:SDE:st: sending ' || tabsize || ' pieces');
  end if;
  
  -- Write out how many pieces are coming, then sit in a loop writing out
  -- the pieces and waiting for the confirmation.
  -- (We cant just write all the pieces out because dbms_pipe doesn't handle
  --  the pipe flow)
  --
  if (write_pipe(pbutl.be_success, notifier || ' ' || tabsize) = 0) then
    for i in 1..tabsize loop
      if (debugging) then
        pbreak.debug_message(' Probe:SDE:st: (' || i || ') ' || tab(i));
      end if;
      dbms_pipe.pack_message(tab(i));
      retval := dbms_pipe.send_message(output_pipe);
      -- Dont wait for a response if this is the last message
      if (i = tabsize) then return; end if;
      retval := read_pipe(response, next_item_vc2);
      if (retval <> 0) or (response <> confirmation) then
        -- Error: dont bother sending any more pieces. 
	--        Front-end better recover from the mess, because 
	--        there's nothing we can do here except log the error...
	pbreak.debug_message('Probe:SDE:st: error: ' || retval || ' ' 
	                     || response);
	return;
      end if;
    end loop;
  end if;
end write_table;

----------------------------------------------------------------------------
-- Write the data in tidl_buf to the pipe, breaking into pieces as
-- necessary. (dbms_pipe utililizes a buffer into which messages are 
-- packed, and this places an upper limit on the size of each message).
--
procedure send_tidl(tidl_buf in out raw) is
  buf       varchar2(10000);   -- $$$ SIZE???
  tidl_tab  pbutl.source_table_type; -- holds the varchar form of the split TIDL
  tabsize   binary_integer;        -- size of tidl_tab
begin
  buf := tidl_buf;
  disassemble(buf, tidl_tab, tabsize);
  write_table(tidl_tab, tabsize, pbutl.tidl_coming, pbutl.send_next_tidl);
end send_tidl;


----------------------------------------------------------------------------
-- The display frame message should be: 
--  <request-display-frame> <start> <count> <flags> <max-strlen> <max-ivals>
-- where 
--   <max-strlen> is the value at which to truncate long strings
--   <max-ivals> is the most indexed-table values to display
--
procedure display_frame(perc    in pbreak.peidef,
                        request in binary_integer,
                        msg     in varchar2) is
  start_frame_depth  binary_integer;
  frame_count        binary_integer;
  flags              binary_integer;
  max_string_length  binary_integer;
  max_index_values   binary_integer;
  more               boolean := false;  -- more pieces of TIDL to fetch?
  end_offset         binary_integer;
  finished           boolean;        -- are we finished getting the fields?

  tempv              varchar2(100);
  tempv2             varchar2(20);
  tidl_buf           raw(32767);
  retval             binary_integer;
begin
  -- unpack the fields
  tempv := msg;
  read_word(tempv, start_frame_depth, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, frame_count, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, flags, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, max_string_length, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, max_index_values, end_offset, finished);
  
  -- Get the TIDL and ship it across the pipe
  pbreak.display_frame(perc, 
                       start_frame_depth,
                       frame_count,
                       flags,
                       max_string_length,
                       max_index_values,
                       tidl_buf,
                       more);
  send_tidl(tidl_buf);
exception
  when others then retval := write_pipe(pbutl.be_failure, pbutl.tidl_failure);
end display_frame;

----------------------------------------------------------------------------
-- Print a backtrace for the current execution stack.
--
procedure print_backtrace(perc  in pbreak.peidef,
                          msg   in varchar2) is
  retval binary_integer;
  buf    varchar2(500);    -- buffer for the backtrace
begin
  pbreak.print_backtrace(perc, buf);
  retval := write_pipe(pbutl.be_success, buf);
exception
  when others then retval := write_pipe(pbutl.be_failure, 
                                        '** exception in print_backtrace **');
end print_backtrace;

----------------------------------------------------------------------------
-- NOT Recommended unless client doesn't have access to KGL. 
-- (If possible, pin the source via KGL instead).
-- 
procedure print_source(perc in pbreak.peidef,
                       msg in varchar2) is

  low                 binary_integer;      -- bottom of line range
  high                binary_integer;      -- top of line range
  window              binary_integer;      -- window (number of lines) to print
  print_arrow         binary_integer;      -- whether to print an arrow 
  finished            boolean;
  end_offset          binary_integer;
  retval              binary_integer;
  tempv               varchar2(100);
  srctab              pbutl.source_table_type;
  srctab_size         binary_integer := 0;    -- number of elements in srctab

begin
  -- get low, high, window, print_arrow off the message
  tempv := msg;
  read_word(tempv, low, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, high, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, window, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, print_arrow, end_offset, finished);
  --if (finished) then raise VALUE_ERROR; end if;

  pbreak.print_source(perc, low, high, window, print_arrow, srctab, 
                      srctab_size);
  if ((srctab_size is null) or (srctab_size = 0)) then
    srctab(1) := '*** Source not available ***';
    srctab_size := 1;
  end if;

  write_table(srctab, srctab_size, pbutl.source_coming, pbutl.send_next_source);

end print_source;

----------------------------------------------------------------------------
-- Write the line-number and unit that execution is suspended at.
procedure print_source_location(perc in pbreak.peidef,
                                msg  in varchar2) is
  unit_and_location varchar2(100);
  retval            binary_integer;
begin
  if (pbreak.print_source_location(perc, unit_and_location)) then
    retval := write_pipe(pbutl.be_success, unit_and_location);
  else
    retval := write_pipe(pbutl.be_failure, unit_and_location);
  end if;
  if (debugging) then
    pbreak.debug_message('Probe:SDE:psl: ' || unit_and_location);
  end if;
exception
  when others then
    retval := write_pipe(pbutl.be_exception, 
                         '** Exception in print_source_location **');
end print_source_location;

----------------------------------------------------------------------------
-- msg contains the original request minus the first 3 chars (which were
-- "request").
--
-- Returns: 0   for regular success
--          non-zero for error from dbms_pipe.
--
function handle_request(plsp in pbreak.plspgadf,
                        perc in pbreak.peidef,
                        request in binary_integer,
                        msg in varchar2,
                        continue out boolean 
                        ) return binary_integer is
  retval              binary_integer;
  flag_status         binary_integer := null;
  breakpoint_listing  varchar2(1000);
  break_flags         binary_integer;
begin
  if (debugging) then
    pbreak.debug_message('Probe:SDE:hr: request: ' || request || ' args: ' ||
                          msg);
  end if;
  continue := false;

  if (request = pbutl.request_continue) then
    -- Continue: set the debug flags (error if there aren't any) and continue
    --
    if (msg is null) then 
      retval := write_pipe(pbutl.be_failure, '** Null step flags **');
      pbreak.debug_message('Probe:SDE:hr: null step flags');
      return 0;
    end if;

    flag_status := pbreak.set_debug_flags(perc, msg);
    if (flag_status = 0) then
      retval := write_pipe(pbutl.be_failure, '** Failed to set step flags **');
      pbreak.debug_message('Probe:SDE:hr: failed to set step flags');
    else
      retval := write_pipe(pbutl.be_success, 'continuing');
      continue := true;
    end if;    
    return 0;

--  elsif (request = pbutl.request_set_debug_flags) then
--    flag_status := pbreak.set_debug_flags(perc, msg);
--    if (flag_status = 0) then
--      retval := write_pipe(pbutl.be_failure,'** Failed to set step flags **');
--    else
--      retval := write_pipe(pbutl.be_success, 'success');
--    end if;
--    return 0;

  elsif (request = pbutl.request_get_debug_flags) then
    break_flags := pbreak.get_debug_flags(perc);
    if (break_flags = -1) then
      retval := write_pipe(pbutl.be_failure, 'failure');
    else
      retval := write_pipe(pbutl.be_success, break_flags);
    end if;
    return 0;

  elsif ( (request = pbutl.request_set_breakpoint_entry) or
          (request = pbutl.request_set_breakpoint_line) ) then
    set_breakpoint(plsp, request, msg);
    return 0;
    
  elsif ( (request = pbutl.request_delete_breakpoint) or
          (request = pbutl.request_disable_breakpoint) or 
          (request = pbutl.request_enable_breakpoint) ) then
    change_breakpoint_status(plsp, msg, request);
    return 0;

  elsif (request = pbutl.request_print_breakpoints) then
    pbreak.print_all_breakpoints(plsp, breakpoint_listing);
    retval := write_pipe(pbutl.be_success, 
                         crlf || breakpoint_listing);
    return 0;

  elsif (request = pbutl.request_print_source) then
    print_source(perc, msg);
    return 0;

  elsif (request = pbutl.request_print_source_location) then
    print_source_location(perc, msg);
    return 0;

  elsif (request = pbutl.request_display_frame) then
    display_frame(perc, request, msg);
    return 0;
  elsif (request = pbutl.request_print_backtrace) then
    print_backtrace(perc, msg);
    return 0;
  elsif (request = pbutl.request_print_pcode_line) then
    retval := write_pipe(pbutl.be_warning, request || ' ... NYI');
    return 0;
  elsif (request = pbutl.request_abort) then
    retval := write_pipe(pbutl.be_warning, request || ' ... NYI');
    return 0;
  else
    pbreak.debug_message('Probe:SDE:hr: ??? request:' || request);
    retval := write_pipe(pbutl.be_warning, request || ' ... NOT UNDERSTOOD');
    return 1;
  end if;

exception
  when others then 
    retval := write_pipe(pbutl.be_warning, 
                         '** Exception in handle_request **');
end handle_request;

end pbsde;
/
show errors;

drop public synonym pbsde;
create public synonym pbsde for sys.pbsde;
grant execute on pbsde to public;


-- remote procedure handler 

Rem    NAME
Rem      pbrph.sql - Plsql deBugger Remote Procedure Handler
Rem    DESCRIPTION
Rem      Package that takes requests and writes them to a pipe. Then reads
Rem      the results from the pipe (which presumably were put there by 
Rem      pbsde.sql, which was called from the debugged plsql interpeter) and
Rem      returns those as OUT parameters.
Rem
Rem      In the default server configuration, this package will be called from
Rem      the synchronous (ie. non-blocking) RPC connection and will communicate
Rem      over the pipe with the asynchronous (blocking) RPC connection that is
Rem      executing the actual plsql RPC.
Rem
Rem      Alternatively, this package can be called by a client tool to debug
Rem      a plsql execution that is taking place in another process. The pipe
Rem      base must be set up to be the same as that in the other process.
Rem
Rem    RETURNS
Rem 
Rem    NOTES
Rem      Should be installed as SYS.
Rem      Currently updates debugging information by inserting into a table and
Rem        doing a commit. This may not be what you want! See comments on 
Rem        procedure debug_message.
Rem
-------------------------------------------------------------------------------
-- $$$ To do: delete-break et.al. should return statuses to indicate whether
--            they succeeded.
--
-- $$$ Everything that does more than one pipe operation should return an 
--     error indicating which of the pipe operations failed.
--
--
-------------------------------------------------------------------------------
--
-- PBRPH - 
--         (nee fe.sql)
--         This package is called via synchronous RPC in order to debug an
--         (asynchronous) RPC. 
--
-- This interface is almost identical to package pbreak. 
-- Most entrypoints take the request and write it (in an encoded form) to the
-- pipe, and then wait for a response. The server-side process reads the 
-- request from the pipe, executes it, and puts a response back on the pipe.
--
-- None of the entries take a plspga or a perc: they all operate upon the
-- plspga or perc that the server-side process is executing with.
-- 
--
-- In addition, there are entrypoints to start and continue execution. This 
-- corresponds (on the client-side) to starting and suspending the interpreter.
--
--
-- Usage: 
--  (1) connect <user>
--  (2) set serveroutput on
--  (3) execute pbrph.attach(<pipe-base>)     --> initialize the pipes
--  (4) execute pbrph.command(<command>)      --> tell back-end to do <command>
--                                             and wait for response
--
--
-- Global variables "response", "operation", and "status" may be used to
-- determine if the interchange was effected successfully.
--
-- Initialization: 
--             attach(<pipe-base>) initializes the pipes by adding suffixes
--             for the two pipes to <pipe-base>.
--             (<pipe-base> is obtained by calling back-end init).
--
-- Pipe read/writes timeout after the given interval (default 120 seconds).
--

-- set echo on
-- connect internal

drop table de_debug_table;
create table de_debug_table(tic number, msg varchar2(100), pipe varchar2(30));
grant select on de_debug_table to public;

create or replace package pbrph is
  default_timeout binary_integer := 120;  -- 2 minute timeout
  -- $$$ All of the functions that take "timeout" as an arg (ie. most of them)
  --     should have it default to default_timeout. However I've removed this
  --     for now because it makes it difficult to add parameters without
  --     shooting yourself in the foot (ie. by forgetting to add them in
  --     calling routines).

  -- Some error statuses that can be returned to the caller. These are mostly
  -- for debugging purposes. (ie. to figure out what exactly went wrong in 
  -- a call to pbrph).
  min_op_status           constant binary_integer := 0;
  op_success              constant binary_integer := 0;
  op_timed_out            constant binary_integer := 1;
  op_exception            constant binary_integer := 2;
  op_unpack_error         constant binary_integer := 3;
  op_pack_message_error   constant binary_integer := 4;
  op_unpack_message_error constant binary_integer := 5;
  op_send_message_error   constant binary_integer := 6;
  op_be_failure           constant binary_integer := 7;
  op_generic_failure      constant binary_integer := 8;   -- unidentifed failure
  op_uninitialized        constant binary_integer := 9;  -- pipe not open
  op_request_queued       constant binary_integer := 10;
  op_synch_error          constant binary_integer := 11;  -- continue() out of
                                                          -- phase
  max_op_status           constant binary_integer := 11;

  request varchar2(200);   -- general-purpose buffer
  buf     varchar2(200);   -- general-purpose buffer

  type request_list is table of varchar2(200) index by binary_integer;
  type request_table is record(
    entries request_list,
    len     binary_integer := 0
  );

  -- $$$
  -- $$$ Should have a "buffer type" somewhere rather than all these various 
  -- varchar2s all over the place.
  --
  request_queue        request_table;   -- any pending requests
  


  -- debugging levels
  no_debugging     constant binary_integer := 0;
  full_debugging   constant binary_integer := 1;


  --
  -- Attach : set the input and output pipe names. 
  --
  function attach(new_pipe_base in varchar2) return boolean;
  
  -- Shutdown : delete the pipes.
  procedure shutdown;

  --
  -- set_debug_level : only off (0) and on (1) are currently supported.
  --
  function set_debug_level(debug_level in binary_integer) return boolean;

  -- get_pipe_names :  return the pipe names
  -- It is provided because variable-references via RPC do not (currently?) 
  -- work.
  procedure get_pipe_names(inpipe out varchar2, outpipe out varchar2);

  -- Print out some or all of the source into the output buffer "lin".
  -- (The preferred method is to use KGL on client-side to pin the source,
  --  since that doesn't require repeated passing of text via RPC).
  --
  procedure print_source(ctx in out pbutl.de_context,
                         low in binary_integer,         -- start line
                         high in binary_integer,        -- ending line
                         window in binary_integer,      -- number of lines
                         print_arrow in binary_integer, -- print arrow at PC?
                         buf in out varchar2,           -- output buffer
			 buflen  in binary_integer,     -- size of buf
			 pieces out binary_integer,    -- # of additional pieces
                         timeout binary_integer);
                         
  procedure get_more_source(ctx    in out pbutl.de_context,
                            buf    in out varchar2,
			    buflen in     binary_integer,
                            piece  in     binary_integer);


  -- Return the unit-name and line-number.
  procedure print_source_location(ctx in out pbutl.de_context,
                                  source_line   out binary_integer,
                                  unit_name  in out varchar2,
                                  namespace     out binary_integer,
                                  unit_owner in out varchar2,
                                  dblink     in out varchar2,
                                  timeout    in     binary_integer);

  -- set/get_multiple - access multiple attributes via the peibpa record.
  --  $$$ NYI
  function set_multiple(bpap in out pbreak.peibpa,
                        timeout binary_integer) return boolean;
  function get_multiple(bpap in out pbreak.peibpa,
                        timeout binary_integer) return boolean;

  -- Set_break_entry : set a breakpoint at the supplied entrypoint.
  -- Set_break_line  : set a breakpoint at the supplied line number.
  --       Both return the breakpoint number, or 0 for failure.
  --
  function set_break_entry(ctx in out pbutl.de_context,
                           unit_name in varchar2,
                           unit_type in binary_integer,
                           entry in binary_integer,
                           timeout binary_integer) 
        return binary_integer;

  function set_break_line(ctx in out pbutl.de_context,
                          unit_name in varchar2,
                          unit_type in binary_integer,
                          entry in binary_integer,
                          timeout binary_integer) 
        return binary_integer;


  -- Delete_breakpoint : delete the specified breakpoint (which should be
  --                     the return-value from set_break_entry etc.)
  --
  procedure delete_breakpoint(ctx in out pbutl.de_context,
                              breakpoint in binary_integer,
                              timeout binary_integer);

  -- enable_breakpoint  : enables a (previously-set) breakpoint
  -- disable_breakpoint : disables a (previously-set) breakpoint
  -- 
  procedure enable_breakpoint(ctx in out pbutl.de_context,
                              breakpoint in binary_integer,
                              timeout binary_integer);
  procedure disable_breakpoint(ctx in out pbutl.de_context,
                               breakpoint in binary_integer,
                               timeout binary_integer);

--
-- Note: the C interface includes a set_debug_flags call, which is used to
--       set the temporary perc flags before resuming the interpreter.
--       In the plsql interface we have the freedom to specify the flags
--       at the continue() call, and this results in one less round-trip
--       across the wire (and one less pipe message too).
--
--       A client-to-server RPC executed by Probe takes advantage of this
--       interface by only setting the local perc flags when requested by
--       the user, and then transmitting those flags with the continue.
--   
--       This call can certainly be replaced if there is any demand for
--       it.
--
--  function set_debug_flags(ctx in out pbutl.de_context,
--                           debug_flags in binary_integer,
--                           timeout binary_integer) return binary_integer;

  function get_debug_flags(ctx in out pbutl.de_context,
                           debug_flags in out binary_integer,
                           timeout binary_integer) return binary_integer;

  procedure print_all_breakpoints(ctx in out pbutl.de_context,
                                  listing in out varchar2,
                                  timeout binary_integer);


  ------------------------------------------------------------------------------
  --                        <-- Display Section -->

  procedure display_frame(ctx               in out pbutl.de_context, 
                           start_frame       in     binary_integer,
                           frame_count       in     binary_integer,
                           flags             in     binary_integer,   -- ???
                           max_string_length in     binary_integer,
                           max_index_values  in     binary_integer,
                           tidl_buf          in out raw,
			   --tidl_buflen       in     binary_integer,
			   --pieces               out binary_integer,
                           timeout           in     binary_integer);

  procedure print_backtrace(ctx     in out pbutl.de_context,
                            listing in out varchar2,
                            timeout binary_integer);

  --                        <-- End of Display Section -->
  ------------------------------------------------------------------------------


  -- get_pending_request : return the n'th pending request, if one exists.
  --  (This interface is ugly, but the alternatives are worse: 
  --    -1- return a giant string of all the pending requests, which will
  --        probably overflow
  --    OR
  --    -2- return an indexed table of the pending requests, and get an
  --        exception on user-side if you try and look at any of them
  --  
  --   Probably 1 is the better choice.)
  --
  procedure get_pending_request(ctx in out pbutl.de_context,
                                one_request in out varchar2,
                                nth in binary_integer);


  -- listen : listen for a value on input_pipe.
  --          When starting an RPC, listen is typically used to wait for the
  --          first information (DE_INIT) from the RPC.
  procedure listen(ctx in out pbutl.de_context,
                   reason      in out binary_integer,
                   source_line    out binary_integer,
		   unit_name in  out varchar2,
		   namespace     out binary_integer,
                   unit_owner  in out varchar2,
                   dblink      in out varchar2,
                   depth       in out binary_integer,
                   timeout     in     binary_integer);

  -- flush_pipes : flush the pipes by repeatedly reading from both of them.
  --
  procedure flush_pipes(ctx in out pbutl.de_context,
                        num_in_messages  in out binary_integer,
                        num_out_messages in out binary_integer);

  -- Delete_debug_table : empty the debug table.
  -- (This has to be done here because it has to be run by the owner of this
  --  package, rather than who the client is logged in as).
  --
  procedure delete_debug_table(ctx in out pbutl.de_context);

  -- continue : tell the server to continue execution. 
  --            Waits for the next communication if the current depth is
  --            greater than 0. Otherwise sets "terminated" to 1 and 
  --            returns.
  procedure continue(ctx         in out pbutl.de_context,
                     debug_flags in     binary_integer,
                     reason      in out binary_integer,
                     source_line    out binary_integer,
		     unit_name in   out varchar2,
		     namespace      out binary_integer,
                     unit_owner  in out varchar2,
                     dblink      in out varchar2,
                     depth       in out binary_integer,
                     timeout     in     binary_integer);
        
  -- 
  -- Query for the available Oracle processes. This is just a convenience
  -- for tool2, and it is run here instead of on client-side because this 
  -- package is assumed to be installed as internal (or at least as some schema
  -- that has access to v$process).
  --
  procedure get_oracle_processes(ctx in out pbutl.de_context,
                                 padsize in binary_integer,
                                 result in out varchar2);


----------------------- Private routines -------------------------------------

  -- auxiliary procedure called from continue. 
  procedure continue2(ctx          in out pbutl.de_context,
                      debug_flags  in     binary_integer,
                      reason       in out binary_integer,
                      source_line     out binary_integer,
		      unit_name in    out varchar2,
		      namespace       out binary_integer,
                      unit_owner   in out varchar2,
                      dblink       in out varchar2,
                      depth        in out binary_integer,
                      timeout      in     binary_integer,
		      ignore_depth in    boolean);

  -- Enqueue the request on the provided queue.
  -- "len" is the previous end of queue.
  procedure enqueue(ctx  in out pbutl.de_context,
                    request in varchar2, 
                    queue in out request_table);

  procedure send_pending_requests(ctx in out pbutl.de_context,
                                  request_queue in out request_table,
                                  timeout in     binary_integer);

  -- Dump the nesting depth to the debug stream
  procedure print_current_depth;

  -- Either send a request or enqueue it.
  function process_request(ctx        in out pbutl.de_context,
                           request    in out varchar2,
                           result_buf in out varchar2,
                           timeout    in     binary_integer
                           ) return binary_integer;

  -- Read a set of messages into the global indexed table
  -- (Used to get TIDL as well as source)
  function read_into_table(ctx             in out pbutl.de_context,
	                   initial_message in     varchar2,
	                   response        in     binary_integer,
	                   next_message    in     binary_integer,
                           timeout         in     binary_integer
			   ) return boolean;

  -- Auxiliary used by read_into_table
  function multi_read(ctx in out pbutl.de_context,
                      pieces in binary_integer,
                      next in varchar2,
                      timeout in  binary_integer) return boolean ;

  ------ Global variables and constants -----

  -- the input and output pipes
  pipe_base   varchar2(50) := NULL;   -- just for debugging purposes
  input_pipe  varchar2(50) := NULL;
  output_pipe varchar2(50) := NULL;
  pipes_open  boolean := false;
  
  -- The current depth (ie. how many pfrrun's are currently on the C stack
  -- on this host). The depth may be greater than one if a SQL statement 
  -- results in a trigger firing.
  -- Incremented upon DE_INIT and decremented upon DE_EXIT.
  -- All the control entrypoints are defined to return TRUE if current_depth
  -- is 0: this means that the RPC is over.
  -- 
  current_depth     binary_integer := 0;
  
  -- RPC instantiation of package spec and/or body is done via additional
  -- calls to the interpreter. Instantiate_depth, if non-zero, indicates
  -- that we are just instantiating a package: therefore DO NOT exit if
  -- depth becomes 0 : the requested entrypoint is still to be run.
  instantiate_depth binary_integer := 0;

  operation   binary_integer;   -- the current (or latest) pipe operation
                                -- (helpful in debugging pipe problems)
  reason      binary_integer;   -- reason for the current stoppage
--  status      binary_integer;   -- the status returned
  response    varchar2(1000);   -- the response received from the pipe


  -- timeouts: the number of accumulated timeouts since the last acknowledged 
  -- message. If the client ever comes alive and reads the messages that have
  -- been placed on the pipe, it will reply with at least one response for each
  -- of those messages.
  -- (The alternative is to never return with a timeout: just keep waiting until
  --  a response is heard. The difficulty here is that a hung program hangs
  --  the debugger as well. On the other hand it could be argued that returning
  --  with a timeout doesn't actually allow the debugger to do anything useful
  --  to the program being debugged, since that program is effectively 
  --  incommunicado).
  --
  -- The model I had in mind was to treat requests that result in timeouts as
  -- ignored messages, but this doesn't quite work because if the request was
  -- for something like setting a breakpoint then that request will eventually
  -- get executed, and so we need to know the response (since it will contain
  -- the breakpoint number, which the client needs to know).
  --
  -- I guess what we really need (if clients are going to set small timeouts)
  -- is more granularity: ie. remember what the request that caused the timeout
  -- was, and be able to service the reply later. (An indexed table of timed
  -- out requests).
  --
  timeouts    binary_integer;

  -- debugging variable. Set by calling "set_debug_level"
  debugging  binary_integer := no_debugging;     --> Default to debugging OFF
  --debugging  binary_integer := full_debugging;  --> Default to debugging ON
  counter    binary_integer := 0;
  
  source_tab     pbutl.source_table_type;
  source_tab_len binary_integer := 0;

end pbrph;
/
show errors;



create or replace package body pbrph is

----------------------------------------------------------------------------
-- Insert a debugging message into the global debugging table. 
--
-- Note that this does a commit! This is fine if you are using one process to
-- debug another one, but it probably wont work if both processes are in the
-- same session. In such a case (ie. the intended manner of server-side 
-- debugging) an alternate method is needed that doesn't require a commit.
--
procedure debug_message(msg in varchar2) is
begin
  insert into SYS.de_debug_table values(
                  counter, 
                  substr(msg, 1, 80),   -- dont overflow the de_debug_table
                  pipe_base);
  counter := counter + 1;
  commit;
end debug_message;

----------------------------------------------------------------------------
-- Dump the current depth to the debug stream. 
--
procedure print_current_depth is
begin
  if (current_depth < 0) then
    debug_message('*ERROR* pnl: current depth ' || current_depth);
  else
    debug_message('pnl: current depth ' || current_depth);
  end if;
end print_current_depth;


----------------------------------------------------------------------------
procedure get_pipe_names(inpipe out varchar2, outpipe out varchar2) is
begin
  inpipe  := input_pipe;
  outpipe := output_pipe;
end get_pipe_names;


----------------------------------------------------------------------------
-- Set input and output pipe names. 
--   "pipe_base" should (preferably) have been obtained by a call to 
--   pbsde.init
--   (in the process being debugged).
--
-- We reset "timeouts" since there should not be any messages from the 
-- debugger stacked on the pipe yet.
--
function attach(new_pipe_base in varchar2) return boolean is
begin

  --pipe_base   := 'DE_PIPE';        -- $$$ temporarily default to a constant
  pipe_base   := new_pipe_base;
  output_pipe := pipe_base || pbutl.de_to_pls_extension;
  input_pipe  := pipe_base ||  pbutl.pls_to_de_extension;

  if (debugging = full_debugging) then
    debug_message('pipe: ' || pipe_base);
  end if;

  timeouts := 0;

  return true;
exception
  when others then return false;
end attach;

----------------------------------------------------------------------------
-- Shut down the pipes. Another call to init will be required in order to 
-- run after this (to create new pipes).
-- 
procedure shutdown is
  dummy integer;
begin
  /* $$$ Amit says we can ignore the retval 'dummy' */
  dummy := dbms_pipe.remove_pipe(input_pipe);
  dummy := dbms_pipe.remove_pipe(output_pipe);

  pipes_open := false;     -- not actually used in pbrph, but it should be
  input_pipe := NULL;
  output_pipe := NULL;

end shutdown;

----------------------------------------------------------------------------
function set_debug_level(debug_level in binary_integer) return boolean is
begin
  -- DEBUGGING
  debug_message('Setting debug_level to: ' || debug_level);

  if (debug_level = full_debugging) then 
    debugging := full_debugging;
  elsif (debug_level = no_debugging) then
    debugging := no_debugging;
  else
    debug_message('Debug level defaulting to 1 (dont understand '||
                   debug_level || ' )');
    debugging := full_debugging;
    return false;
  end if;
  return true;
exception
  when others then return false;
end set_debug_level;

----------------------------------------------------------------------------
-- Return true if the pipe is not open. Set the ctx.status as a side
-- effect.
--
function pipe_not_open(ctx in out pbutl.de_context) return boolean is
begin
  if ((input_pipe is null) or (output_pipe is null)) then
    ctx.status := op_uninitialized;
    ctx.be_status := null;
    return true;
  end if;
  return false;
end pipe_not_open;

----------------------------------------------------------------------------
-- Read one word from vchar, ie. read up until end of vchar or until the
-- first whitespace. Return the index of the whitespace in "end_offset".
-- "finished" is true if the word consumed the rest of vchar.
-- 
procedure read_word(vchar in varchar2,
                    word out varchar2,
                    end_offset out binary_integer,
                    finished   out boolean) is
  tempv varchar2(100);
  offset binary_integer;
  space  binary_integer;
begin
  tempv := ltrim(vchar);
  if (tempv is null) then
    finished := true;
    return;
  end if;

  offset := length(vchar) - length(tempv);
  space := instr(tempv, ' ');
  end_offset := offset + space;

  if (space = 0) then
    -- last word in the string
    word := tempv;
    finished := true;
  else
    finished := false;
    word := substr(tempv, 1, space-1);
  end if;
--exception
--  when others then raise VALUE_ERROR;
end read_word;


----------------------------------------------------------------------------
-- write_request : write the request to the pipe, then read the reply.
--                 Return false if anything goes wrong.
--
-- $$$ What is the relationship between "status" and the return-value?
--
function write_request(ctx       in out pbutl.de_context,
                       request   in     varchar2, 
                       reply     in out varchar2,
                       timeout   in     binary_integer
                       ) return boolean is
  first_space binary_integer;
begin
  if pipe_not_open(ctx) then return false; end if;

  if (debugging = full_debugging) then
    debug_message('wr -sending : ' || request);
  end if;

  ctx.status := op_pack_message_error;
  ctx.be_status := pbutl.be_failure;
  dbms_pipe.pack_message(request);
  ctx.status := op_send_message_error;
  ctx.status := dbms_pipe.send_message(output_pipe);

  if (ctx.status = 0) then   -- wait for a reply
    ctx.status := op_timed_out;

    ctx.status := dbms_pipe.receive_message(input_pipe, timeout);
    if (ctx.status = 0) then
      ctx.status := op_unpack_message_error;
      dbms_pipe.unpack_message(reply);
      if (debugging = full_debugging) then
        debug_message('wr -response: ' || reply);
      end if;
      ctx.status := op_success;
      ctx.be_status := pbutl.be_success;

      -- trim any blanks off "reply", then unpack it into a return-code plus
      -- additional string
      reply := ltrim(reply);
      first_space := instr(reply, ' ');
      ctx.be_status := substr(reply, 1, first_space);
      reply := substr(reply, first_space, length(reply));
      return true;
    else  -- 
      if (debugging = full_debugging) then
        debug_message('wr - failed to receive message, status: ' || ctx.status);
      end if;
      return false;       -- timed out waiting for the reply
    end if;

  else
    if (debugging = full_debugging) then
      debug_message('wr - failed to send message, status: ' || ctx.status);
    end if;
    return false;        -- the send failed
  end if;
exception
  when others then 
    ctx.status := op_exception;
    if (debugging = full_debugging) then
      debug_message('wr - exception was raised');
    end if;
    return false;
end write_request;

----------------------------------------------------------------------------
-- Get the requested row from source_tab.
-- 
-- $$$ buflen is not currently used: use it later to figure out if we can
--     get more than one piece from source_table into buf.
--
procedure get_more_source(ctx    in out pbutl.de_context,
                          buf    in out varchar2,
			  buflen in     binary_integer,
			  piece  in     binary_integer) is
begin
  if (piece > source_tab_len) then
    ctx.status := op_exception;
    buf := null;
    return;
  end if;
  ctx.status := op_success;
  buf := source_tab(piece);
  -- source_tab(piece) := null;  --> $$$ Free the space ???
end get_more_source;


----------------------------------------------------------------------------
-- Print out the source that is currently being executed. 
-- low,high,range,print_arrow specify the range of lines to print out, and 
-- whether to print an arrow next to the current line.
--
-- Read the source into the global indexed table if it is too large for the
-- provided buffer. Additional calls can be made to get the rest of the
-- source.
--
-- "pieces" indicates how many pieces were constructed:
--    0 : error
--    1 : all of whole source was put in buf
--   >1 : additional pieces can be obtained via calls to get_more_source.
--
procedure print_source(ctx in out pbutl.de_context,
                       low         in     binary_integer,
                       high        in     binary_integer,
                       window      in     binary_integer,
                       print_arrow in     binary_integer,
                       buf         in out varchar2,
		       buflen      in     binary_integer,
		       pieces         out binary_integer,
                       timeout            binary_integer) is
  request varchar2(30);
begin
  request := pbutl.request_print_source || ' ' ||
             low || ' ' || high || ' ' || window || ' ' || print_arrow;

  if read_into_table(ctx, request, pbutl.source_coming, pbutl.send_next_source,
                    timeout) then
    if (source_tab_len > 0) then
      buf := source_tab(1);
    else
      buf := '';
    end if;
    pieces := source_tab_len;
  else
    -- error.
    buf := '<-- Failed to obtain the source -->';
    pieces := 0;
  end if;

  -- $$$ ??? $$$ ???
  -- Strip the extra space from the front of source
  --lin := substr(lin,2);
end print_source;

----------------------------------------------------------------------------
function set_multiple(bpap in out pbreak.peibpa,
                      timeout binary_integer) return boolean is
begin
  --NYI $$$
  return false;
end set_multiple;

----------------------------------------------------------------------------
function get_multiple(bpap in out pbreak.peibpa,
                      timeout binary_integer)
        return boolean is
begin
  --NYI $$$
  return false;
end get_multiple;

----------------------------------------------------------------------------
-- send_pending_requests : do these in the order in which they came in, since
--     the user is referring to them by number.
-- 
-- $$$ Make sure that we are disposing of the memory used by the pending 
--     request. (Does setting to null actually do that?)
-- $$$ Would be nice to have some feedback to the user.
-- $$$ Need to establish the correlation between user's local breakpoint 
--     numbers and the remote numbers.
--
procedure send_pending_requests(ctx in out pbutl.de_context,
                                request_queue in out request_table,
                                timeout in binary_integer) is
  temp    boolean;
  counter binary_integer := 1;
begin
  if (request_queue.len is null) then return; end if;

  while (counter <= request_queue.len) loop
    if (debugging = full_debugging) then
      debug_message('spr: ' || request_queue.entries(counter));
    end if;
    temp := write_request(ctx, request_queue.entries(counter), buf, timeout);
    --> $$$ Clear the pending request
    request_queue.entries(counter) := '';
    counter := counter + 1;
  end loop;
  request_queue.len := 0;         -- no more pending requests
  ctx.status := op_success;

exception
  when others then ctx.status := op_exception;
end send_pending_requests;

----------------------------------------------------------------------------
-- Enqueue the request on the provided queue.
-- "len" is the previous end of queue.
-- Note that this is a 1-based queue. (first entry is queue(1)).
--
procedure enqueue(ctx in out pbutl.de_context,
                  request in varchar2, 
                  queue in out request_table) is
begin
  queue.len := queue.len + 1;
  queue.entries(queue.len) := request;
  ctx.status := op_request_queued;
  if (debugging = full_debugging) then
    debug_message('queued: ' || queue.entries(queue.len));
  end if;
exception
  when others then ctx.status := op_exception;
end enqueue;

----------------------------------------------------------------------------
-- Process a request: ie. either send it and read the results or enqueue it
-- for later transmission.
--
-- Returns:  0 : normal completion
--           1 : request was enqueued
--           2 : error in write_request (see ctx for actual error)
--
-- NOTE: global variable "current_depth" is used to determine if the pipe is
--       available. This means that requests will not get sent directly if 
--       control is paused at a DE_INIT or a DE_EXIT for the top-level unit
--       (since depth will have gone back to 0). If such functionality is
--       desired then an additional status variable will be required (and in
--       fact it might not be a bad idea...)
--
function process_request(ctx        in out pbutl.de_context,
                         request    in out varchar2,
                         result_buf in out varchar2,
                         timeout    in     binary_integer
			 ) return binary_integer is
begin
  if (output_pipe is null) or (current_depth < 1) then
    -- The pipe is not currently available
    enqueue(ctx, request, request_queue);
    return 1;
  else
    if write_request(ctx, request, result_buf, timeout) then
      return 0;
    else
      return 2;   -- error in write_request
    end if;
  end if;
end process_request;

----------------------------------------------------------------------------
-- returns the breakpoint number
--
function set_break_entry(ctx in out pbutl.de_context,
                         unit_name in varchar2,
                         unit_type in binary_integer,
                         entry in binary_integer,
                         timeout binary_integer)
        return binary_integer is
begin
  ctx.status := 0;
  request := pbutl.request_set_breakpoint_entry || ' ' || unit_name || 
             ' ' || unit_type || ' ' || entry;

  if (process_request(ctx, request, buf, timeout) = 0) then
    -- the request completed successfully
    return buf;
  else
    -- either the request was queued or there was a failure
    return 0;
  end if;
exception 
  when others then return op_exception;
end set_break_entry;

----------------------------------------------------------------------------
-- returns the breakpoint number
--
function set_break_line(ctx in out pbutl.de_context,
                        unit_name in varchar2,
                        unit_type in binary_integer,
                        entry in binary_integer,
                        timeout binary_integer)
        return binary_integer is
begin
  ctx.status := 0;
  request := pbutl.request_set_breakpoint_line || ' ' || unit_name || 
             ' ' || unit_type || ' ' || entry;

  if (process_request(ctx, request, buf, timeout) = 0) then
    -- the request completed successfully
    return buf;
  else
    -- either the request was queued or there was a failure
    return 0;
  end if;
exception
  when others then return op_exception;
end set_break_line;

----------------------------------------------------------------------------
-- $$$ This should probably return a status.
--     It is most unpleasant that the response gets dropped on the floor.
--     However this interface is supposed to be as close as possible to 
--     the C interface.
--
procedure delete_breakpoint(ctx in out pbutl.de_context,
                            breakpoint in binary_integer,
                            timeout binary_integer) is
 
  temp binary_integer;
begin
  request := pbutl.request_delete_breakpoint || ' ' || breakpoint;
  temp := process_request(ctx, request, buf, timeout);
--  temp := write_request(ctx, request, buf, timeout);
end delete_breakpoint;

----------------------------------------------------------------------------
-- $$$ Should probably return a status
procedure enable_breakpoint(ctx in out pbutl.de_context,
                            breakpoint in binary_integer,
                            timeout binary_integer) is
temp binary_integer;
begin
  request := pbutl.request_enable_breakpoint || ' ' || breakpoint;
  temp := process_request(ctx, request, buf, timeout);
  --temp := write_request(ctx, request, buf, timeout);
end enable_breakpoint;

----------------------------------------------------------------------------
-- $$$ Should probably return a status
procedure disable_breakpoint(ctx in out pbutl.de_context,
                             breakpoint in binary_integer,
                             timeout binary_integer) is
  temp binary_integer;
begin
  request := pbutl.request_disable_breakpoint || ' ' || breakpoint;
  temp := process_request(ctx, request, buf, timeout);
  --temp := write_request(ctx, request, buf, timeout);
end disable_breakpoint;

----------------------------------------------------------------------------
-- $$$ No longer call set_debug_flags : instead pass them to continue.
--
--function set_debug_flags(ctx in out pbutl.de_context,
--                         debug_flags in binary_integer,
--                         timeout binary_integer) return binary_integer is
--  temp binary_integer;
--begin
--
--  if (debugging = full_debugging) then
--    debug_message('sdf : ' || debug_flags);
--  end if;
--
--  request := pbutl.request_set_debug_flags || ' ' || debug_flags;
--  if write_request(ctx, request, buf, timeout) then
--    return op_success;
--  else
--    return op_generic_failure;
--  end if;
--exception
--  when others then return op_exception;
--end set_debug_flags;

----------------------------------------------------------------------------
--
function get_debug_flags(ctx in out pbutl.de_context,
                         debug_flags in out binary_integer,
                         timeout binary_integer) return binary_integer is
  temp binary_integer;
begin
  request := pbutl.request_get_debug_flags;
  if write_request(ctx, request, buf, timeout) then
    if (debugging = full_debugging) then
      debug_message('gdf : ' || buf);
    end if;
    debug_flags := buf;
    return op_success;
  end if;
    return -1;
exception
  when others then ctx.status := op_exception; return -1;
end get_debug_flags;

----------------------------------------------------------------------------
-- Give a listing of all the breakpoints that are being remembered until 
-- a connection is established.
--
procedure get_pending_request(ctx in out pbutl.de_context,
                              one_request in out varchar2,
                              nth in binary_integer) is
begin

  if (nth > 0) and (nth <= request_queue.len) then
    one_request := request_queue.entries(nth);
    ctx.status := op_success;
  else
    ctx.status := op_generic_failure;
  end if;
--
-- The old implementation: this one returned a table of all the pending
-- requests. The problem with this is that I got an exception on user-side if
-- I tried to look at any of these entries.
--
--  out_table.len := request_queue.len;
--  while (out_table.len < request_queue.len) loop
--    out_table.entries(out_table.len) := request_queue.entries(out_table.len);
--    out_table.len := out_table.len + 1;
--  end loop;
--
  exception
    when others then ctx.status := op_generic_failure;
end get_pending_request;


----------------------------------------------------------------------------
procedure print_all_breakpoints(ctx in out pbutl.de_context,
                                listing in out varchar2,
                                timeout binary_integer) is
  temp binary_integer;
begin
  listing := '=+=+ No breakpoint listing available +=+=';
  request := pbutl.request_print_breakpoints;

  if write_request(ctx, request, listing, timeout) then
    -- the listing includes the back-end status
    listing := ltrim(listing);
    temp := instr(listing,' ');
    ctx.be_status := substr(listing, 1, temp);
    listing := substr(listing, temp, length(listing));
  end if;
exception
  when others then
    if (ctx.status = 0) then ctx.status := op_exception; end if;
    if (listing is null) then listing := '** exception in p_a_b **'; end if;
end print_all_breakpoints;

----------------------------------------------------------------------------
-- Return the current unit, line-number, etc.
--
procedure print_source_location(ctx in out pbutl.de_context,
                                source_line    out binary_integer,
                                unit_name   in out varchar2,
                                namespace      out binary_integer,
                                unit_owner  in out varchar2,
                                dblink      in out varchar2,
                                timeout     in     binary_integer
                               ) is

  -- $$$ Direct references dont work: so put into this bogus varchar2.
  request varchar2(20);
  temp    boolean;
  ndx     binary_integer;
  buf     varchar2(200);
  finished boolean;
begin
  request := pbutl.request_print_source_location;
  unit_name := '';
  namespace := -1;
  source_line := -1;
  if (write_request(ctx, request, buf, timeout)) then
    if (ctx.be_status <> pbutl.be_success) then
      -- the line-number will be the only thing in the message
      source_line := buf;
    else
      -- buf looks like: <source_line> <unit_name> <namespace> <user> <dblink>
      read_word(buf, source_line, ndx, finished);
      if finished then raise VALUE_ERROR; end if;
      
      buf := substr(buf, ndx);
      read_word(buf, unit_name, ndx, finished);
      if finished then raise VALUE_ERROR; end if;

      buf := substr(buf, ndx);
      read_word(buf, namespace, ndx, finished);
      if finished then raise VALUE_ERROR; end if;

      buf := substr(buf, ndx);
      read_word(buf, unit_owner, ndx, finished);
      if finished then raise VALUE_ERROR; end if;

      buf := substr(buf, ndx);
      read_word(buf, dblink, ndx, finished);
    end if;
  end if;
exception
  when others then
    if (ctx.status = op_success) then
      ctx.status := op_exception;
    end if;
end print_source_location;

----------------------------------------------------------------------------
-- continue : put a message on the pipe saying to continue.
--            Wait for the next communication if the current depth is 
--            greater than 0.
--
-- RPC instantiation of package spec and body is implemented by additional
-- calls to the interpreter. This means that if a package entrypoint is
-- called and the package has not been instantiated then the interpreter
-- will be called once for each instantiation that requires initialization
-- and then once for the requested entrypoint. Therefore we have an 
-- additional event DE_INSTANTIATION, and the idea is that, upon seeing this,
-- we do NOT return when we see the corresponding DE_EXIT. Instead we
-- continue (using the same flags) and listen for the next event.
--
procedure continue(ctx         in out pbutl.de_context,
                   debug_flags in     binary_integer,
                   reason      in out binary_integer,
                   source_line    out binary_integer,
		   unit_name in   out varchar2,
		   namespace      out binary_integer,
                   unit_owner  in out varchar2,
                   dblink      in out varchar2,
                   depth       in out binary_integer,
                   timeout     in     binary_integer) is
begin
  continue2(ctx, debug_flags, reason, source_line, unit_name, namespace,
            unit_owner, dblink, depth, timeout,
	    false);
end continue;


procedure continue2(ctx         in out pbutl.de_context,
                    debug_flags in     binary_integer,
                    reason      in out binary_integer,
                    source_line    out binary_integer,
		    unit_name in   out varchar2,
		    namespace      out binary_integer,
                    unit_owner  in out varchar2,
                    dblink      in out varchar2,
                    depth       in out binary_integer,
                    timeout     in     binary_integer,
		    ignore_depth in    boolean) is
  retval    binary_integer;
  bogval    boolean;
  terminate boolean := false;
begin
  if (debugging = full_debugging) then
    debug_message('Continue ' || debug_flags);
  end if;

  depth := current_depth;
  ctx.terminated := false;

  if ((current_depth <= 0) and not ignore_depth) then
    -- Synchronization error: no program is currently being debugged
    -- (call listen() to get the first communication from a program)
    ctx.status := op_synch_error;
    -- clear the OUT params, just in case caller looks at them
    reason := 0; source_line := ''; unit_name := ''; namespace := -1;
    unit_owner := ''; dblink := '';
    return;
  else
    -- $$$ HACK!!!  pbsde currently still reads the first 3 characters and 
    --              converts them to a number. Fix this!!.
    --request := pbutl.request_continue || ' ' || debug_flags;
    request := pbutl.request_continue || '   ' || debug_flags;

    if write_request(ctx, request, buf, timeout) then
    -- $$$ Check that the return (buf) was "OK" ?

    -- listen for the next break
      listen(ctx, reason, --frame_count, offset, 
             source_line, unit_name, namespace, unit_owner, dblink, 
	     depth, timeout);
      ---------------------------------------------------------------------
      -- No pausing allowed at DE_EXIT (since there's really nothing useful
      -- that can be done there). Instead, automatically continue (with
      -- the same execution flags), unless we are at level 0.
      --
      if (reason = pbutl.de_exit) then
        if (depth = 0) then
	  if (instantiate_depth > 0) then
	    instantiate_depth := instantiate_depth - 1;
	  else
	    terminate := true;
	  end if;
	end if;   -- (depth = 0)

        if (terminate) then
	  -- We are done. Release the debugged process by telling it to
	  -- continue (with flags 0).
          request := pbutl.request_continue || '   0';
	  bogval := write_request(ctx, request, buf, timeout);
	  ctx.terminated := true;
          if (debugging = full_debugging) then
            debug_message('cont: terminating at level ' || current_depth);
          end if;
	else
	  -- keep going, via a recursive call to continue2. 
	  -- (We pass in true for ignore_depth since we know that either 
	  --  depth is non-zero or that we can safely ignore it).
          continue2(ctx, debug_flags, reason, source_line, unit_name, 
	            namespace, unit_owner, dblink, depth, timeout, 
		    true);
	end if;  -- (terminate)
      end if;    -- (reason = pbutl.de_exit)
    end if;      -- write_request
  end if;        -- (current_depth <= 0)
exception
  when others then ctx.status := op_exception;
end continue2;


----------------------------------------------------------------------------
-- flush_pipes : flush both pipes by repeatedly reading from them.
--
procedure flush_pipes(ctx in out pbutl.de_context,
                      num_in_messages  in out binary_integer,
                      num_out_messages in out binary_integer) is
  flush_status binary_integer;
begin
  ctx.status := op_success;
  num_in_messages := 0;
  num_out_messages := 0;
  
  if pipe_not_open(ctx) then return; end if;

  -- flush the input pipe
  flush_status := 0;
  while (flush_status = 0) loop
    flush_status := dbms_pipe.receive_message(input_pipe,1);
    if (flush_status = 0) then
      num_in_messages := num_in_messages + 1;
    end if;
  end loop;

  -- flush the output pipe
  flush_status := 0;
  while (flush_status = 0) loop
    flush_status := dbms_pipe.receive_message(output_pipe,1);
    if (flush_status = 0) then
      num_out_messages := num_out_messages + 1;
    end if;
  end loop;
exception
  when others then ctx.status := op_exception;
end flush_pipes;

----------------------------------------------------------------------------
-- Delete_debug_table : empty the debug table.
procedure delete_debug_table(ctx in out pbutl.de_context) is
begin
  -- No point in doing any debug_info, since we're deleting that table.
  delete SYS.de_debug_table;
exception
  when others then ctx.status := op_exception;
end delete_debug_table;

----------------------------------------------------------------------------
-- listen : Wait for a message on input_pipe.
--
procedure listen(ctx         in out pbutl.de_context,
                 reason      in out binary_integer,
                 --frame_count  out binary_integer,
                 --offset       out binary_integer,
                 source_line    out binary_integer,
                 unit_name   in out varchar2,
                 namespace      out binary_integer,
                 unit_owner  in out varchar2,
                 dblink      in out varchar2,
                 depth       in out binary_integer,
                 timeout     in     binary_integer) is
  response varchar2(200);
  temp     binary_integer;
begin

  ctx.terminated := false;   -- should ALWAYS be false
  reason := pbutl.de_none;
  -- Set all the OUT variables (note that some are in/out just for convenience)
  unit_name := ''; unit_owner := ''; dblink := '';
  namespace := -1; source_line := -1;

  if pipe_not_open(ctx) then return; end if;

  ctx.status := dbms_pipe.receive_message(input_pipe, timeout);
  depth := current_depth;
  if (ctx.status = 0) then
    ctx.status := op_unpack_error;
    dbms_pipe.unpack_message(response);
    if (debugging = full_debugging) then
      debug_message('listen -received: ' || response);
    end if;
    ctx.status := op_success;

    -- The reason for the suspension is the only thing in the message
    response := rtrim(response);
    temp := instr(response,' ');
    if (temp = 0) then
      reason := response;
    else
      -- the message contained more than one field: things are hosed. Try to
      -- avoid an exception by taking the first field (and hope that it's a
      -- number).
      reason := substr(response,1,temp);
    end if;
    if (reason = pbutl.de_init) then
      current_depth := current_depth + 1;
    elsif (reason = pbutl.de_instantiate) then
      current_depth := current_depth + 1;
      instantiate_depth := instantiate_depth + 1;
    elsif (reason = pbutl.de_exit) then
      current_depth := current_depth - 1;
      -- NOTE:  instantiate_depth is decremented in continue(), which 
      --        we assume that listen was called from.
    end if;
    depth := current_depth;
    
    -- For debugging any problems with depth (either premature detachment or
    -- hanging on after the server-side program has actually exited).
    if (debugging = full_debugging) then
      print_current_depth;
    end if;

    -- Print source if it is meaningful at this point
    if ((reason <> pbutl.de_init) and (reason <> pbutl.de_instantiate) and
        (reason <> pbutl.de_exit)) then
      print_source_location(ctx, source_line, unit_name, namespace, unit_owner,
			    dblink, timeout);
    end if;
    -- Do any work that we have queued
    send_pending_requests(ctx, request_queue, timeout);
  else
    -- The pipe timed out. 
    ctx.status := op_timed_out;
  end if;
exception
  -- status should already indicate the error: if it doesn't then set it to
  -- indicate an exception occurred.
  when others then 
    if (ctx.status = op_success) then 
      ctx.status := op_exception; 
    end if;
end listen;

----------------------------------------------------------------------------
-- get_oracle_processes: return a listing of the processes.
-- Mainly for use by the process browser/debugger (tool2).
--
procedure get_oracle_processes(ctx     in out pbutl.de_context, 
                               padsize in binary_integer,
                               result  in out varchar2) is
  cursor get_proc_cursor is 
    select spid ORACLE, process SQLDBA, v$session.terminal
    from v$session,v$process
    where (v$process.program like '%V1%' 
           or v$process.program like '%Two-Task%')
           and v$session.terminal = v$process.terminal;

  oracle_pid varchar2(20);
  sqldba_pid varchar2(20);
  terminal   varchar2(20);
begin
  ctx.status := op_success;
  open get_proc_cursor;
  loop
    fetch get_proc_cursor into oracle_pid, sqldba_pid, terminal;
    if (get_proc_cursor%NOTFOUND) then exit; end if;
    result := result || lpad(terminal, padsize)
                     || lpad(oracle_pid, padsize)
                     || lpad(sqldba_pid, padsize)
                     || '
';  
  end loop;
exception
  when others then ctx.status := op_exception;
end get_oracle_processes;

--------------------------------------------------------------------------------
--                        DISPLAY SECTION

----------------------------------------------------------------------------
-- $$$ Add a backtrace argument so that you can get only part of the 
--     backtrace upon demand.
--
procedure print_backtrace(ctx     in out pbutl.de_context,
                          listing in out varchar2,
                          timeout        binary_integer) is
  retval boolean;
begin
  ctx.status := op_success;
  request := pbutl.request_print_backtrace;
  retval := write_request(ctx, request, listing, timeout);
exception
  when others then ctx.status := op_generic_failure;
end print_backtrace;


----------------------------------------------------------------------------
-- multi_read - read n messages, responding after each.
-- Store each message into source_tab.
--
-- (dbms_pipe imposes a limit on the message-size, so large messages must
--  be split into smaller ones).
--
-- Return: false if any error
--
-- $$$ THIS SHOULD CALL WRITE_REQUEST (OR A VERSION OF WRITE_REQUEST) INSTEAD
--     OF DOING PIPE READS DIRECTLY.
--
function multi_read(ctx in out pbutl.de_context,
                    pieces in binary_integer,  -- how many messages to read
                    next in varchar2,          -- send this message to request
		                               -- the next piece
		    timeout in  binary_integer
                    ) return boolean is
  temp_buf varchar2(2000);
begin    

  -- Clean out anything that's left in source_tab
  for i in 1..source_tab_len loop
    source_tab(i) := '';
  end loop;
  source_tab_len := 0;

  for i in 1..pieces loop
    ctx.status := op_timed_out;
    temp_buf := '';
    ctx.status := dbms_pipe.receive_message(input_pipe, timeout);
    if (ctx.status = 0) then
      ctx.status := op_unpack_message_error;
      dbms_pipe.unpack_message(temp_buf);
      if (debugging = full_debugging) then
        -- message: 'mr: [<number>,<length>] <message>'
        debug_message('mr: [' || i || ',' || length(temp_buf) ||
	              '] ' || temp_buf);
      end if;
      source_tab(i) := temp_buf;
      source_tab_len := source_tab_len + 1;
      -- send a request for the next piece, unless this is the last one
      if (i < pieces) then
	if (debugging = full_debugging) then
	  debug_message('mr - sending : ' || next);
	end if;
        dbms_pipe.pack_message(next);
	ctx.status := dbms_pipe.send_message(output_pipe);
	if (ctx.status <> 0) then
	  if (debugging = full_debugging) then
	    debug_message('mr - failed to send message');
	  end if;
	  return false;
	end if;  -- ctx.status <> 0
      end if;    -- i < pieces
    else  -- 
      if (debugging = full_debugging) then
        debug_message('mr - failed to receive message, status: ' || 
	               ctx.status);
      end if;
      return false;
    end if;   -- ctx.status = 0
  end loop;

  ctx.status := op_success;
  ctx.be_status := pbutl.be_success;
  return true;

exception
  when others then
    if (debugging = full_debugging) then
      debug_message('** Exception in multi_read **');
    end if;
    return false;
end multi_read;

-----------------------------------------------------------------------------
-- read_into_table : read a sequence of messages into source_tab.
-- Return false if there were any problems, else true, in which case
-- source_tab has the messages, and source_tab_len indicates how many there
-- are. (source_tab is 1-based).
--
function read_into_table(
              ctx             in out pbutl.de_context,
	      initial_message in     varchar2,       -- initial request to send
	      response        in     binary_integer, -- reply from pipe
	      next_message    in     binary_integer, -- synch message to pipe
              timeout         in     binary_integer) 
	      return boolean is
  buf       varchar2(20);
  reply     binary_integer;
  pieces    binary_integer;
  ndx       binary_integer;
  finished  boolean;
begin
  ctx.status := op_success;

  if write_request(ctx, initial_message, buf, timeout) then
    -- buf should be: <response> <number of messages>
    read_word(buf, reply, ndx, finished);
    if finished then raise VALUE_ERROR; end if;
    if (reply <> response) then
      -- Either a pipe logic error or else the requested source/TIDL isn't
      -- available.
      return false;
    end if;

    buf := substr(buf, ndx);
    read_word(buf, pieces, ndx, finished);

    if multi_read(ctx, pieces, next_message, timeout) then
      -- Results have been written into source_tab
      return true;
    else
      -- some error in reading the messages
      return false;
    end if;
  else
    -- back-end couldn't create the source/tidl for some reason
    return false;
  end if;

  -- should never be reached
  return false;

exception
  when others then 
    if (ctx.status = op_success) then ctx.status := op_exception; end if;
    if (debugging = full_debugging) then
      debug_message('rit - exception');
    end if;
    return false;
end read_into_table;



----------------------------------------------------------------------------
-- Print out the frame information for one or more frames on the current
-- call stack.
--
-- $$$ Modify to place only tidl_buflen characters into tidl_buf and then
--     allow user to get additional pieces via another procedure like
--     print_source does.
-- $$$
procedure display_frame(
              ctx in out pbutl.de_context,
              start_frame       in     binary_integer,  -- frame to start from
              frame_count       in     binary_integer,  -- # of frames to fetch
              flags             in     binary_integer,  -- ????
              max_string_length in     binary_integer,  -- longest string
              max_index_values  in     binary_integer,  -- max table values
              tidl_buf          in out raw,             -- buf to place result
	      --tidl_buflen       in     binary_integer,  -- size of tidl_buf
	      --pieces             out  binary_integer,
              timeout           in binary_integer) is
  temp_buf  varchar2(32760);
begin
  ctx.status := op_success;
  request := pbutl.request_display_frame || ' ' ||
             start_frame       || ' ' ||
             frame_count       || ' ' ||
             flags             || ' ' ||
             max_string_length || ' ' ||
             max_index_values;

  if read_into_table(ctx, request, pbutl.tidl_coming, pbutl.send_next_tidl, 
                    timeout) then
    -- assemble the pieces from source_tab
    tidl_buf := '';
    temp_buf := '';
    for i in 1..source_tab_len loop
      temp_buf := temp_buf || source_tab(i);
    end loop;
    tidl_buf := temp_buf;
  else
    -- no TIDL for us
    tidl_buf := '';
  end if;

exception
  when others then 
    if (ctx.status = op_success) then ctx.status := op_exception; end if;
    if (debugging = full_debugging) then
      debug_message('df - exception');
    end if;
end display_frame;

end pbrph;
/
show errors;

drop public synonym pbrph;
create public synonym pbrph for sys.pbrph;
grant execute on pbrph to public;
