rem 
rem $Header: prvtpb.sql 7020200.1 95/02/15 18:26:01 cli Generic<base> $ 
rem 
Rem  Copyright (c) 1995 by Oracle Corporation 
Rem    NAME
Rem      prvtpb.sql - ProBe (PL/SQL debugger) server-side packages.
Rem
Rem    DESCRIPTION
Rem      These packages implement server-side Probe support.
Rem 
Rem    NOTES
Rem      This package must be installed as SYS.
Rem    
Rem    MODIFIED   (MM/DD/YY)
Rem     jmallory   02/13/95 -  Branch_for_patch
Rem     jmallory   02/09/95 -  Creation

-------------------------------------------------------------------------------
-- Just used in commented-out debugging code.
--drop table pbreak_raw_table;
--create table pbreak_raw_table (a long);


create or replace package body pbreak is

-- Note about the pragma interfaces: at some point in the (near) future, the
-- syntax will include a "target name" for ICDs: 
--
--    pragma interface(C, long-meaningful-plsql-name, short-OCCS-C-name);
--
-- and then P2ICD will generate a table of the short-OCCS-C-name's. 
-- So we dont need to worry about the naming of procedures in plsql packages 
-- merely because they are going to be implemented as ICDs.
--
-- (Note that the rdbmsfolk have already gone and done this: for example see
--  the dbmspipe ICDs. They dont use p2icd to generate their ICDs).
--

  ------------- pragma interfaces -----------
  --
  -- NOTE: returning record-types from ICD seems to mess up the argc, so
  --       for now we use raw (and have the plsql wrapper deal with it).
  --

  -- set_multiple : set multiple fields
  --> $$$ NYI.
  function set_multiple(
       perc in peidef, 
       bpap in out pbutl.peibpa) return boolean is
  begin
    return FALSE;
  end set_multiple;

  -- get_multiple : get multiple fields
  --> $$$ NYI.
  function get_multiple(
       perc in peidef, 
       bpap in out pbutl.peibpa) return boolean is
  begin
    return FALSE;
  end get_multiple;

  -- pbbsd : PBB Set Debug
  procedure pbbsd(perc in raw, debug_level in binary_integer);
  pragma interface(c, pbbsd);                               -- 1 (ICD #1)

  -- pbbcd : PBB Clear Debug
  procedure pbbcd(perc in raw);
  pragma interface(c, pbbcd);                               -- 2


  -- PBBHS - return Handle from String
  --     ($$$ Should return kglhd, but ICAL is broken)
  function pbbhs(unitname in varchar2,
                  unit_type in binary_integer) return raw;
  pragma interface(c, pbbhs);                               -- 3

  -- PBBBE set Break for Entry
  function pbbbe(plsp in raw, lu in raw,
                  entry in binary_integer) return binary_integer;
  pragma interface(c, pbbbe);                               -- 4

  -- PBB set Break for Line
  function pbbbl(plsp in raw, lu in raw,
                  entry in binary_integer) return binary_integer;
  pragma interface(c, pbbbl);                               -- 5

  -- PBB Remove (Delete) Breakpoint
  procedure pbbrb(breakpoint in binary_integer);
  pragma interface(c, pbbrb);                               -- 6

  -- PBB Enable Breakpoint
  procedure pbbeb(breakpoint in binary_integer);
  pragma interface(c, pbbeb);                               -- 7

  -- PBB Disable Breakpoint
  procedure pbbdb(breakpoint in binary_integer);
  pragma interface(c, pbbdb);                               -- 8

  -- PBB Plsp from Perc
  function pbbpfp(perc in raw) return raw;
  pragma interface(c, pbbpfp);                              -- 9

  -- PBB Print PCode Line
  procedure pbbppl(perc in raw, lin out varchar2);
  pragma interface(c, pbbppl);                              -- 10

  -- PBB Print BreakpointS
  procedure pbbpbs(plsp in raw, listing out varchar2);
  pragma interface(c, pbbpbs);                              -- 11

  -- PBB Set Perc FLAGS
  function pbbspf(perc in raw, request in binary_integer) 
                     return binary_integer;
  pragma interface(c, pbbspf);                              -- 12

  -- PBB Get Perc FLAGS
  function pbbgpf(perc in raw) return binary_integer;
  pragma interface(c, pbbgpf);                              -- 13

  -- PBB Get Line Number
  procedure pbbgln(perc  in raw, 
                   linnum        out binary_integer,
                   unit_name  in out varchar2,
                   namespace     out binary_integer,
                   unit_owner in out varchar2,
                   dblink     in out varchar2);
  pragma interface(c, pbbgln);                              -- 14
  
  -- PBB Print Source Line
  procedure pbbpsl(perc        in     raw,
                   low         in     binary_integer, 
                   high        in     binary_integer, 
                   window      in     binary_integer,
                   print_arrow in     binary_integer,
                   srclin      in out varchar2,
		   srclin_size in     binary_integer,
		   next_low       out binary_integer,
		   next_high      out binary_integer);
 
  pragma interface(C, pbbpsl);                              -- 15

  procedure get_tidl_frame(perc              in     raw,
                           start_frame_depth in     binary_integer,
                           frame_count       in     binary_integer,
                           flags             in     binary_integer,
                           max_string_length in     binary_integer,
                           max_index_values  in     binary_integer,
                           buf                  out raw
                           --,more out boolean
                           );
  pragma interface(c, get_tidl_frame);                      -- 16 (pbbgtf)
  
  
  procedure do_print_backtrace(perc in raw, listing out varchar2);
  pragma interface(c, do_print_backtrace);                  -- 17 (pbbpbt)

  procedure debug_message_aux(buf in varchar2);
  pragma interface(c, debug_message_aux);                   -- 18 (pbbwdm)

  ----- end pragma interfaces -------


----------------------------------------------------------------------------
-- Set the debug bit in UGA. 
-- Actually this is no longer a "bit" - it is a 3-state variable that
-- indicates one of: "no debugging," "debug only RPCs," "debug everything."
--
procedure set_debug(debug_level in binary_integer := debug_everything) is
  perc peidef;
begin
  perc.dummy := null;
  pbbsd(perc.dummy, debug_level);
end set_debug;


procedure clear_debug is
  perc peidef;
begin
  perc.dummy := null;
  pbbsd(perc.dummy, debug_off);
end clear_debug;

----------------------------------------------------------------------------
-- Print the current source line, or some subset of the current source.
--
-- Writes the lines into srctab (1-based).
-- Call pbbpsl repeatedly until next_low comes back as 0: then we are done.
--
procedure print_source(
       perc        in     peidef, 
       low         in     binary_integer,
       high        in     binary_integer,
       window      in     binary_integer,
       print_arrow in     binary_integer,    -- print an arrow at current line?
       srctab      in out pbutl.source_table_type,
       srctab_size in out binary_integer) is

  linnum     binary_integer := 0;
  unit_name  varchar2(100) := '';
  unit_owner varchar2(100) := '';
  dblink     varchar2(100) := '';
  namespace  binary_integer;
  current_low  binary_integer := low;
  current_high binary_integer := high;
  current_window binary_integer := window;
  next_low     binary_integer := 1;
  next_high    binary_integer := 0;
begin

    -- $$$ pbbgln is called so that we have the line-number for the
    --     exception-handler (in case anything goes wrong in pbbpsl)
    pbbgln(perc.dummy, linnum, unit_name, namespace, unit_owner, dblink);
    
    srctab_size := 0;   -- assume failure
    while (next_low <> 0) loop
      srctab(srctab_size+1) := '+=+=+=EMPTY+=+=+=+=';
      pbbpsl(perc.dummy, current_low, current_high, current_window, 
             print_arrow, 
             srctab(srctab_size+1), 
	     pbutl.pipe_buffer_size,
	     next_low, next_high);
      if (srctab(srctab_size+1) is not null) then
        -- successfully wrote a line into the table
	srctab_size := srctab_size + 1;
	current_low := next_low;
	current_high := next_high;
	current_window := 0;
      else
        return;   -- failure
      end if;
    end loop;
exception
  when others then
    srctab_size := 0;
    --lin := unit_name || ' [' || linnum || '] ** Source not available **';
end print_source;

----------------------------------------------------------------------------
-- *** Not implemented, mainly for security reasons. May be put back later ***
-- Print the current pcode line.
-- 
--
--procedure print_pcode_line(perc in peidef, lin out varchar2) is
--begin
--  pbbppl(perc.dummy, lin);    -- $$$
--exception
--  when others then
--     lin := null;
--end print_pcode_line;

----------------------------------------------------------------------
-- Get information about the current unit. This includes the line-number,
-- the name of the unit, and the type of the unit.
-- $$$ Anything else needed?
--
function print_source_location(perc in     peidef,
                               buf  in out varchar2) return boolean is

  linnum     binary_integer := -1;
  namespace  binary_integer := -1;
  unit_name  varchar2(100);
  unit_owner varchar2(100);
  dblink     varchar2(100);
  
begin
  buf := '';
  pbbgln(perc.dummy, linnum, unit_name, namespace, unit_owner, dblink );
  if (namespace is null) then
    buf := linnum;
    return false;
  else
    buf := linnum || ' ' ||
           unit_name || ' ' ||
           namespace || ' ' || 
           unit_owner || ' ' || 
           dblink;
     -- $$$DEBUGGING
     --insert into pbreak_raw_table values(
     --   linnum || ' ' || unit_name || ' ' || namespace || ' ' || unit_owner 
     --   || ' ' || dblink);
     --commit;
     --    end DEBUGGING
    return true;
  end if;
exception
  when others then buf := linnum || ' ' || -1 || ' ** Unit not available ** ';
  return false;
end print_source_location;

----------------------------------------------------------------------
function get_plsp_from_perc(perc in peidef) return plspgadf is
  plsp plspgadf;
begin
  plsp.dummy := pbbpfp(perc.dummy);  -- $$$
  return plsp;
end get_plsp_from_perc;


----------------------------------------------------------------------
function set_break_entry(plsp in plspgadf, 
                         unit_name in varchar2,
                         unit_type in binary_integer,
                         entry in binary_integer) return binary_integer is
  lu       kglhd;
  breaknum binary_integer := null;
begin
  lu.dummy := pbbhs(unit_name, unit_type);   -- $$$
  if (lu.dummy is not null) then
    breaknum := pbbbe(plsp.dummy, lu.dummy, entry);
  end if;
  return breaknum;
end set_break_entry;


----------------------------------------------------------------------
function set_break_line(plsp in plspgadf,
                         unit_name in varchar2,
                         unit_type in binary_integer,
                         entry in binary_integer) return binary_integer is
  lu       kglhd;
  breaknum binary_integer := null;
begin
  lu.dummy := pbbhs(unit_name, unit_type);  -- $$$
  if (lu.dummy is not null) then
    breaknum := pbbbl(plsp.dummy, lu.dummy, entry);
  end if;
  return breaknum;
end set_break_line;


----------------------------------------------------------------------
-- get_handle : return the kglhd associated with the given name.
--              (unitname must include the entire name).
--
function get_handle(unitname in varchar2, unit_type in binary_integer)
                    return kglhd is
  lu kglhd;
begin
  lu.dummy := pbbhs(unitname, unit_type);  -- $$$
  return lu;
end get_handle;


----------------------------------------------------------------------
procedure delete_breakpoint(breakpoint in binary_integer) is
begin
  pbbrb(breakpoint);
end delete_breakpoint;


----------------------------------------------------------------------
procedure enable_breakpoint(breakpoint in binary_integer) is
begin
  pbbeb(breakpoint);
end enable_breakpoint;


----------------------------------------------------------------------
procedure disable_breakpoint(breakpoint in binary_integer) is
begin
  pbbdb(breakpoint);
end disable_breakpoint;

----------------------------------------------------------------------
-- Create a listing of all the current breakpoints that the runtime is
-- aware of. 
--
procedure print_all_breakpoints(plsp in plspgadf, listing in out varchar2) is
begin
  pbbpbs(plsp.dummy, listing);
exception
  when others then listing := null;
end print_all_breakpoints;

----------------------------------------------------------------------
-- Set perc debug flags to indicate debugging events.
-- Return value: 
--  0 (false) --> fail
--  1 (true)  --> succeed
--
-- $$$ We used to handle the flags according to constants declared in 
--     pbutl, but now we're pushing this off onto the client. So just pass
--     the request through.
--
function set_debug_flags(perc in peidef, request in binary_integer)
           return binary_integer is
begin 
    return pbbspf(perc.dummy, request);
exception
  when others then return 0;
end set_debug_flags;

----------------------------------------------------------------------
-- Return current value of perc debug flags.
--
function get_debug_flags(perc in peidef) return binary_integer is
begin
  return pbbgpf(perc.dummy);
  exception
    when others then return -1;
end get_debug_flags;


--------------------------------------------------------------------------------
-- display_frame : display one or more frames from the current stack.
-- For now it is required that the whole listing fit into one varchar2: the
-- "more" argument may actually be used later.
--
  -- ($$$ We need to address the problem of putting together multiple chunks
  --      of the display: eg. reconstructing a call-stack that includes one
  --      or more RPC calls).
  
procedure display_frame(
              perc        in          peidef,
              start_frame_depth in    binary_integer, -- frame to start from
              frame_count in          binary_integer,  -- # of frames to fetch
              flags       in          binary_integer,  -- ????
              max_string_length in    binary_integer, -- longest string
              max_index_values  in    binary_integer, -- max table values
              tidl_buf          in  out raw,      -- buf to place result in
              more out boolean                        -- couldn't get all?
              ) is
begin
  more := false;
  get_tidl_frame(perc.dummy, 
                 start_frame_depth, frame_count, flags,
                 max_string_length, max_index_values,
                 tidl_buf 
                 --, more
                 );

  -- DEBUGGING --
  --declare
  --  v varchar2(32767);
  --begin
  --  insert into pbreak_raw_table values (tidl_buf);
  --  v := tidl_buf;
  --  insert into de_debug_session_table(msg)  
  --           values('lengthb: ' || lengthb(v));
  --  commit;
  --end;
  -- DEBUGGING --

end display_frame;  
              
------------------------------------------------------------------------------
-- display_frame_more : call to get more of the frame, if display_frame
--                      indicated that it couldn't finish in one call.
--
procedure display_frame_more(tidl_buf out raw,
                             more out boolean) is
begin
  more := true;
  tidl_buf := '** ERROR ERROR ERROR ERROR **';
end display_frame_more;

----------------------------------------------------------------------
--
procedure print_backtrace(perc in peidef, listing out varchar2) is
begin
  do_print_backtrace(perc.dummy, listing);
end print_backtrace;

----------------------------------------------------------------------
-- Dump a debug message to the tracefile.
procedure debug_message(buf in varchar2) is
begin
  debug_message_aux(buf);      -- Call the ICD
end debug_message;

end pbreak;
/

-------------------------------------------------------------------------------
create or replace package body pbsde is 
  next_item_vc2    constant binary_integer := 9;  -- dbms_pipe code for varchar2
  next_item_number constant binary_integer := 6;  -- dbms_pipe code for number
  pipe_failure     constant binary_integer := -1;
  crlf             constant varchar2(2) := '
';

----------------------------------------------------------------------------
procedure print_pipe_names(msg in varchar2) is
begin
  --dbms_output.put_line('***** PBSDE: ' || msg || '*****');
  --dbms_output.put_line('  pipe base is        : ' || pipe_base);
  --dbms_output.put_line('  (output pipe is named: ' || output_pipe || ')');
  --dbms_output.put_line('  (input pipe is named:  ' || input_pipe  || ')');
  --dbms_output.new_line;
  dbms_output.put_line('  execute pbrph.attach(''' || pipe_base || ''')');
end print_pipe_names;

----------------------------------------------------------------------------
-- Initialize: set up the input and output pipe names for use by this 
-- session. 
--
procedure init(base in varchar2 := null, notify in boolean := false) is
begin
  --output_pipe := 'DE_PROGRAM_PIPE';
  --input_pipe := 'DE_USER_PIPE';

  if (base is not null) then
    -- Mostly for ease of debugging.
    pipe_base := base;
  else
    pipe_base := dbms_pipe.unique_session_name;
  end if;

  if (debugging) then
    pbreak.debug_message('Probe:SDE:init: pipe base: "' || base || '"');
  end if;

  output_pipe := pipe_base || pbutl.pls_to_de_extension;
  input_pipe :=  pipe_base || pbutl.de_to_pls_extension;

  /* Create secure pipes */
  declare
    dummy integer;
  begin
    /* $$$ Amit says we can ignore the retval 'dummy' */
    dummy := dbms_pipe.create_pipe(input_pipe);
    dummy := dbms_pipe.create_pipe(output_pipe);
  end;

  pipes_open := true;

  if (notify) then
    print_pipe_names('INIT');
  end if;
end init;

----------------------------------------------------------------------------
-- Echo the pipe names via dbms_output. For use in sqldba.
-- 
procedure stat is
begin
  if (pipes_open) then
    print_pipe_names('STAT');
  else
    dbms_output.put_line('STAT: no pipes currently open');
  end if;
end stat;

----------------------------------------------------------------------------
function get_pipe_base return varchar2 is
begin
  return pipe_base;
end get_pipe_base;

----------------------------------------------------------------------------
-- To be called at end of debug session. After calling shutdown, a new init
-- will be required in order to do any more debugging.
--
-- $$$ We might want to send some kind of termination message to the pipes
--     upon shutdown, just to make sure that anyone who is listening 
--     understands that they wont be hearing any more.
--
procedure shutdown(notify in boolean := false) is
begin

  pipes_open := false;

  --
  -- $$$ Send a termination message to the output pipe, and make sure that
  --     no data is on the input pipe.
  --
  -- retval := write_pipe('ABORT');

  /* Remove secure pipes */
  declare
    dummy integer;
  begin
    /* $$$ Amit says we can ignore the retval 'dummy' */
    dummy := dbms_pipe.remove_pipe(input_pipe);
    dummy := dbms_pipe.remove_pipe(output_pipe);
  end;

  if (notify) then
    dbms_output.put_line('***** PBSDE: shut down. *****');
  end if;

end shutdown;

----------------------------------------------------------------------------
-- Read a message from the input_pipe.
-- Unpack into the buffer if the type of the next message is expected_type.
--
-- Returns: 0 for success, 
--          else the error-code from receive_message 
--          else the type read by receive_message (if different from 
--             the requested type)
--          
--
-- $$$ Should probably pass in multiple params (or a record of the different
--     types) and just set the appropriate one. For now the only type
--     that is seen is vc2
--
function read_pipe(msg           in out varchar2, 
                   expected_type in     binary_integer,
                   timeout       in binary_integer := default_timeout
                   ) return binary_integer is
  retval     binary_integer;
  item_type  binary_integer;
begin
  retval := dbms_pipe.receive_message(input_pipe, timeout);
  if (retval <> 0) then 
    pbreak.debug_message('Probe:SDE:rp: receive failed with status:' || retval);
    return retval; 
  end if;

    -- success: make sure that we got the expected type
  item_type := dbms_pipe.next_item_type;
  if (item_type = expected_type) then
    dbms_pipe.unpack_message(msg);
    if (pipe_debugging) then
      pbreak.debug_message('Probe:SDE:rp: received: "' || msg || '"');
    end if;
    return retval;
  else
    pbreak.debug_message('Probe:SDE:rp: requested type:' || expected_type ||
                         ' but received type:' || item_type);
    return item_type;
  end if;
exception
  when others then pbreak.debug_message('Probe:SDE:rp: *** exception ***');
  return pipe_failure;
end read_pipe;

----------------------------------------------------------------------------
-- Write a message to the output_pipe.
-- The dbms_pipe message buffer size is limited, so the message may need to be
-- be broken into pieces.
--
function write_pipe(status in binary_integer,
                    msg varchar2) return binary_integer is
  msg_size   binary_integer := length(msg);
  num_pieces binary_integer;
  retval     binary_integer;
begin
  if (pipes_open) then
    -- $$$
    -- Figure out the number of pieces and send them individually
    --
    if (pipe_debugging) then
      pbreak.debug_message('Probe:SDE:wp: sending "' || msg || '"');
    end if;
    dbms_pipe.pack_message(status || ' ' || msg);
    retval := dbms_pipe.send_message(output_pipe);
    if (retval <> 0) then
      pbreak.debug_message('Probe:SDE:wp: send failed with status:' || retval);
    end if;
    return retval;
  else
    pbreak.debug_message('Probe:SDE:wp: pipe not open. Request: ' || msg);
    return pipe_failure;
  end if;
exception
  when others then pbreak.debug_message('Probe:SDE:wp: *** exception ***');
  return pipe_failure;
end write_pipe;

----------------------------------------------------------------------------
-- Read one word from vchar, ie. read up until end of vchar or until the
-- first whitespace. Return the index of the whitespace in "end_offset".
-- "finished" is true if the word consumed the rest of vchar.
-- 
procedure read_word(vchar in varchar2,
                    word out varchar2,
                    end_offset out binary_integer,
                    finished   out boolean) is
  tempv varchar2(100);
  offset binary_integer;
  space  binary_integer;
begin
  tempv := ltrim(vchar);
  if (tempv is null) then
    finished := true;
    return;
  end if;

  offset := length(vchar) - length(tempv);
  space := instr(tempv, ' ');
  end_offset := offset + space;

  if (space = 0) then
    -- last word in the string
    word := tempv;
    finished := true;
  else
    finished := false;
    word := substr(tempv, 1, space);
  end if;
--exception
--  when others then raise VALUE_ERROR;
end read_word;

----------------------------------------------------------------------------
-- Notes: perc is an in-param so we dont need to worry about writing back
--        its value upon return.
--        "reason" specifies the reason for this suspension
--
function debug_loop(perc in pbreak.peidef,
                    reason in binary_integer
                    ) return binary_integer is
  plsp      pbreak.plspgadf;
  msg       varchar2(100);
  retval    binary_integer;
  request   binary_integer;
  continue  boolean := false;
begin
  if not pipes_open then
    -- Things are not correctly set up: a call to debug_loop must always be
    -- preceded by a call to init (to set the pipe names). We could call init
    -- here ourselves but we cant be sure that the front-end will use the
    -- new names that we generate. So just return with error-status.
    pbreak.debug_message('Probe:SDE:dl: pipes not open; reason:' || reason);
    return 1;
  end if;

  if (debugging) then 
    pbreak.debug_message('Probe:SDE: debug_loop: reason ' || reason);
  end if;

  -- Tell the pipe that we have stopped, and the reason for stopping.
  retval := write_pipe(reason, 'suspended');
  if (retval <> 0) then return retval; end if;

  plsp := pbreak.get_plsp_from_perc(perc);
  
  -- Now sit in a loop reading requests from the pipe and executing them.
  -- Terminate when told to continue.
  while (not continue) loop
    -- read an action from the pipe and execute it
    retval := read_pipe(msg, next_item_vc2);
    if (retval <> 0) then return retval; end if;
    request := to_number(substr(msg,1,3));
    retval := handle_request(plsp, perc, request, substr(msg, 4), continue);
   if (retval <> 0) then return retval; end if;
  end loop;

  -- presumably everything went fine
  return 0;
exception
  when others then
    retval := write_pipe(pbutl.be_warning, '** Exception in debug_loop **');
end debug_loop;


----------------------------------------------------------------------------
-- Enable, disable, or delete a previously-set breakpoint.
--
procedure change_breakpoint_status(plsp in pbreak.plspgadf,
                                   breakpoint_number in binary_integer,
                                   breakpoint_op in binary_integer) is
  retval binary_integer;
  response varchar2(30);
begin
  if (breakpoint_op = pbutl.request_disable_breakpoint) then
    pbreak.disable_breakpoint(breakpoint_number);
    retval := write_pipe(pbutl.be_success, 
                         'breakpoint# ' || breakpoint_number || ' disabled');
  elsif (breakpoint_op = pbutl.request_enable_breakpoint) then
    pbreak.enable_breakpoint(breakpoint_number);
    retval := write_pipe(pbutl.be_success,
                         'breakpoint# ' || breakpoint_number || ' enabled');
  elsif (breakpoint_op = pbutl.request_delete_breakpoint) then
    pbreak.delete_breakpoint(breakpoint_number);
    retval := write_pipe(pbutl.be_success,
                         'breakpoint# ' || breakpoint_number || ' deleted');
  else
    retval := write_pipe(pbutl.be_warning,
                         '** Error: didnt understand message **');
  end if;
exception
  when others then
    retval := write_pipe(pbutl.be_warning,
                         '** Exception in change_breakpoint_status **');
end change_breakpoint_status;

----------------------------------------------------------------------------
-- Get the unit name (from which a kglhd can be obtained) and use that to
-- set the line or entry breakpoint number.
--
-- $$$ Enhancement: should be able to pass the entry-name and from that figure
--                  out the entry-number to set.
--
procedure set_breakpoint(plsp in pbreak.plspgadf,
                         request in binary_integer,
                         msg in varchar2) is
  endname   binary_integer;
  i         binary_integer;
  j         binary_integer;
  unit_name varchar2(50);
  temp      varchar2(50);
  entry     binary_integer;
  breaknum  binary_integer;
  unit_type binary_integer;
  retval    binary_integer;
begin
  temp := ltrim(msg);
  endname := instr(temp, ' ');
  -- ASSERT(endname <> 0)
  unit_name := rtrim(substr(msg,1,endname));

  temp := ltrim(substr(msg, endname+1));
  endname := instr(temp, ' ');
  unit_type := substr(temp,1,endname);

  temp := ltrim(substr(temp, endname+1));
  endname := instr(temp, ' ');
  if (endname = 0) then
    entry := substr(temp,1);
  else
    entry := substr(temp,1,endname);
  end if;

  if (debugging) then
    pbreak.debug_message('Probe:SDE:sb: req:' || request || ' name:' || 
                          unit_name || ' type:' || unit_type || ' entry:' || 
			  entry);
  end if;

  if (request = pbutl.request_set_breakpoint_entry) then
    breaknum := pbreak.set_break_entry(plsp, unit_name, unit_type, entry);
  else 
    breaknum := pbreak.set_break_line(plsp, unit_name, unit_type, entry);
  end if;

  if (breaknum is NULL) or (breaknum = 0) then
    --retval := write_pipe(pbutl.be_failure,
    --                     '*** Failed to set breakpoint ***');
    retval := write_pipe(pbutl.be_failure, 0);
  else
    --retval := write_pipe(pbutl.be_success, 'breakpoint# ' || breaknum);
    retval := write_pipe(pbutl.be_success, breaknum);
  end if;
exception
  when others then
    retval := write_pipe(pbutl.be_failure, 
                         '** Exception in set_breakpoint **');
end set_breakpoint;

----------------------------------------------------------------------
-- Break buf into pieces and put into the provided table. 
-- Set tabsize to be the size of the table. (The table is 1-based).
--
procedure disassemble(buf        in out varchar2,
                      result_tab in out pbutl.source_table_type,
                      tabsize    in out binary_integer) is
 len       binary_integer;     -- current length of buf
 piecelen  binary_integer;     -- length of next piece to write
 last      boolean := false;   -- are we done?
begin
  tabsize := 1;
  len := length(buf);
  -- break up buf into pieces and insert into the table
  while (len > 0) loop
    if (len > pbutl.pipe_buffer_size) then
      piecelen := pbutl.pipe_buffer_size;
    else
      last := true;
      piecelen := len;
    end if;
    result_tab(tabsize) := substr(buf,1,piecelen);
    if last then
      len := 0;
    else
      tabsize := tabsize+1;
      buf := substr(buf,piecelen+1);
      len := length(buf);
    end if;
  end loop;
end disassemble;

----------------------------------------------------------------------------
-- Write the contents of the table to the pipe.
--
-- First write: <notifier> <tabsize> to tell the far side what is coming and
-- how many pieces it is broken up into.
-- Then write out each entry in the table, pausing after each to receive a
-- response (except if this is the last entry).
-- Error if the response is not <confirmation>
--
procedure write_table(tab          in out pbutl.source_table_type,
                      tabsize      in     binary_integer,
	 	      notifier     in     binary_integer,
		      confirmation in     binary_integer) is

  retval    binary_integer;
  buf       varchar2(10000);   -- $$$ SIZE???
  tidl_tab  pbutl.source_table_type; -- holds the varchar form of the split TIDL
  response  varchar2(20);
begin
  
  if (debugging) then
    pbreak.debug_message('Probe:SDE:st: sending ' || tabsize || ' pieces');
  end if;
  
  -- Write out how many pieces are coming, then sit in a loop writing out
  -- the pieces and waiting for the confirmation.
  -- (We cant just write all the pieces out because dbms_pipe doesn't handle
  --  the pipe flow)
  --
  if (write_pipe(pbutl.be_success, notifier || ' ' || tabsize) = 0) then
    for i in 1..tabsize loop
      if (debugging) then
        pbreak.debug_message(' Probe:SDE:st: (' || i || ') ' || tab(i));
      end if;
      dbms_pipe.pack_message(tab(i));
      retval := dbms_pipe.send_message(output_pipe);
      -- Dont wait for a response if this is the last message
      if (i = tabsize) then return; end if;
      retval := read_pipe(response, next_item_vc2);
      if (retval <> 0) or (response <> confirmation) then
        -- Error: dont bother sending any more pieces. 
	--        Front-end better recover from the mess, because 
	--        there's nothing we can do here except log the error...
	pbreak.debug_message('Probe:SDE:st: error: ' || retval || ' ' 
	                     || response);
	return;
      end if;
    end loop;
  end if;
end write_table;

----------------------------------------------------------------------------
-- Write the data in tidl_buf to the pipe, breaking into pieces as
-- necessary. (dbms_pipe utililizes a buffer into which messages are 
-- packed, and this places an upper limit on the size of each message).
--
procedure send_tidl(tidl_buf in out raw) is
  buf       varchar2(10000);   -- $$$ SIZE???
  tidl_tab  pbutl.source_table_type; -- holds the varchar form of the split TIDL
  tabsize   binary_integer;        -- size of tidl_tab
begin
  buf := tidl_buf;
  disassemble(buf, tidl_tab, tabsize);
  write_table(tidl_tab, tabsize, pbutl.tidl_coming, pbutl.send_next_tidl);
end send_tidl;


----------------------------------------------------------------------------
-- The display frame message should be: 
--  <request-display-frame> <start> <count> <flags> <max-strlen> <max-ivals>
-- where 
--   <max-strlen> is the value at which to truncate long strings
--   <max-ivals> is the most indexed-table values to display
--
procedure display_frame(perc    in pbreak.peidef,
                        request in binary_integer,
                        msg     in varchar2) is
  start_frame_depth  binary_integer;
  frame_count        binary_integer;
  flags              binary_integer;
  max_string_length  binary_integer;
  max_index_values   binary_integer;
  more               boolean := false;  -- more pieces of TIDL to fetch?
  end_offset         binary_integer;
  finished           boolean;        -- are we finished getting the fields?

  tempv              varchar2(100);
  tempv2             varchar2(20);
  tidl_buf           raw(32767);
  retval             binary_integer;
begin
  -- unpack the fields
  tempv := msg;
  read_word(tempv, start_frame_depth, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, frame_count, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, flags, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, max_string_length, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, max_index_values, end_offset, finished);
  
  -- Get the TIDL and ship it across the pipe
  pbreak.display_frame(perc, 
                       start_frame_depth,
                       frame_count,
                       flags,
                       max_string_length,
                       max_index_values,
                       tidl_buf,
                       more);
  send_tidl(tidl_buf);
exception
  when others then retval := write_pipe(pbutl.be_failure, pbutl.tidl_failure);
end display_frame;

----------------------------------------------------------------------------
-- Print a backtrace for the current execution stack.
--
procedure print_backtrace(perc  in pbreak.peidef,
                          msg   in varchar2) is
  retval binary_integer;
  buf    varchar2(500);    -- buffer for the backtrace
begin
  pbreak.print_backtrace(perc, buf);
  retval := write_pipe(pbutl.be_success, buf);
exception
  when others then retval := write_pipe(pbutl.be_failure, 
                                        '** exception in print_backtrace **');
end print_backtrace;

----------------------------------------------------------------------------
-- NOT Recommended unless client doesn't have access to KGL. 
-- (If possible, pin the source via KGL instead).
-- 
procedure print_source(perc in pbreak.peidef,
                       msg in varchar2) is

  low                 binary_integer;      -- bottom of line range
  high                binary_integer;      -- top of line range
  window              binary_integer;      -- window (number of lines) to print
  print_arrow         binary_integer;      -- whether to print an arrow 
  finished            boolean;
  end_offset          binary_integer;
  retval              binary_integer;
  tempv               varchar2(100);
  srctab              pbutl.source_table_type;
  srctab_size         binary_integer := 0;    -- number of elements in srctab

begin
  -- get low, high, window, print_arrow off the message
  tempv := msg;
  read_word(tempv, low, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, high, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, window, end_offset, finished);
  if (finished) then raise VALUE_ERROR; end if;

  tempv := substr(tempv, end_offset);
  read_word(tempv, print_arrow, end_offset, finished);
  --if (finished) then raise VALUE_ERROR; end if;

  pbreak.print_source(perc, low, high, window, print_arrow, srctab, 
                      srctab_size);
  if ((srctab_size is null) or (srctab_size = 0)) then
    srctab(1) := '*** Source not available ***';
    srctab_size := 1;
  end if;

  write_table(srctab, srctab_size, pbutl.source_coming, pbutl.send_next_source);

end print_source;

----------------------------------------------------------------------------
-- Write the line-number and unit that execution is suspended at.
procedure print_source_location(perc in pbreak.peidef,
                                msg  in varchar2) is
  unit_and_location varchar2(100);
  retval            binary_integer;
begin
  if (pbreak.print_source_location(perc, unit_and_location)) then
    retval := write_pipe(pbutl.be_success, unit_and_location);
  else
    retval := write_pipe(pbutl.be_failure, unit_and_location);
  end if;
  if (debugging) then
    pbreak.debug_message('Probe:SDE:psl: ' || unit_and_location);
  end if;
exception
  when others then
    retval := write_pipe(pbutl.be_exception, 
                         '** Exception in print_source_location **');
end print_source_location;

----------------------------------------------------------------------------
-- msg contains the original request minus the first 3 chars (which were
-- "request").
--
-- Returns: 0   for regular success
--          non-zero for error from dbms_pipe.
--
function handle_request(plsp in pbreak.plspgadf,
                        perc in pbreak.peidef,
                        request in binary_integer,
                        msg in varchar2,
                        continue out boolean 
                        ) return binary_integer is
  retval              binary_integer;
  flag_status         binary_integer := null;
  breakpoint_listing  varchar2(1000);
  break_flags         binary_integer;
begin
  if (debugging) then
    pbreak.debug_message('Probe:SDE:hr: request: ' || request || ' args: ' ||
                          msg);
  end if;
  continue := false;

  if (request = pbutl.request_continue) then
    -- Continue: set the debug flags (error if there aren't any) and continue
    --
    if (msg is null) then 
      retval := write_pipe(pbutl.be_failure, '** Null step flags **');
      pbreak.debug_message('Probe:SDE:hr: null step flags');
      return 0;
    end if;

    flag_status := pbreak.set_debug_flags(perc, msg);
    if (flag_status = 0) then
      retval := write_pipe(pbutl.be_failure, '** Failed to set step flags **');
      pbreak.debug_message('Probe:SDE:hr: failed to set step flags');
    else
      retval := write_pipe(pbutl.be_success, 'continuing');
      continue := true;
    end if;    
    return 0;

--  elsif (request = pbutl.request_set_debug_flags) then
--    flag_status := pbreak.set_debug_flags(perc, msg);
--    if (flag_status = 0) then
--      retval := write_pipe(pbutl.be_failure,'** Failed to set step flags **');
--    else
--      retval := write_pipe(pbutl.be_success, 'success');
--    end if;
--    return 0;

  elsif (request = pbutl.request_get_debug_flags) then
    break_flags := pbreak.get_debug_flags(perc);
    if (break_flags = -1) then
      retval := write_pipe(pbutl.be_failure, 'failure');
    else
      retval := write_pipe(pbutl.be_success, break_flags);
    end if;
    return 0;

  elsif ( (request = pbutl.request_set_breakpoint_entry) or
          (request = pbutl.request_set_breakpoint_line) ) then
    set_breakpoint(plsp, request, msg);
    return 0;
    
  elsif ( (request = pbutl.request_delete_breakpoint) or
          (request = pbutl.request_disable_breakpoint) or 
          (request = pbutl.request_enable_breakpoint) ) then
    change_breakpoint_status(plsp, msg, request);
    return 0;

  elsif (request = pbutl.request_print_breakpoints) then
    pbreak.print_all_breakpoints(plsp, breakpoint_listing);
    retval := write_pipe(pbutl.be_success, 
                         crlf || breakpoint_listing);
    return 0;

  elsif (request = pbutl.request_print_source) then
    print_source(perc, msg);
    return 0;

  elsif (request = pbutl.request_print_source_location) then
    print_source_location(perc, msg);
    return 0;

  elsif (request = pbutl.request_display_frame) then
    display_frame(perc, request, msg);
    return 0;
  elsif (request = pbutl.request_print_backtrace) then
    print_backtrace(perc, msg);
    return 0;
  --elsif (request = pbutl.request_print_pcode_line) then
  --  retval := write_pipe(pbutl.be_warning, request || ' ... NYI');
  --  return 0;
  elsif (request = pbutl.request_abort) then
    retval := write_pipe(pbutl.be_warning, request || ' ... NYI');
    return 0;
  else
    pbreak.debug_message('Probe:SDE:hr: ??? request:' || request);
    retval := write_pipe(pbutl.be_warning, request || ' ... NOT UNDERSTOOD');
    return 1;
  end if;

exception
  when others then 
    retval := write_pipe(pbutl.be_warning, 
                         '** Exception in handle_request **');
end handle_request;

end pbsde;
/


-------------------------------------------------------------------------------
-- Table used by pbrph for debugging Probe.
--
drop table de_debug_table;
create table de_debug_table(tic number, msg varchar2(100), pipe varchar2(30));
grant select on de_debug_table to public;

-------------------------------------------------------------------------------
Rem    Currently updates debugging information (for debugging the debugger)
Rem      by inserting into a table and doing a commit. This may not be what 
Rem      you want! See comments on procedure debug_message.
Rem

create or replace package body pbrph is


----------------------------------------------------------------------------
-- Insert a debugging message into the global debugging table. 
--
-- Note that this does a commit! This is fine if you are using one process to
-- debug another one, but it probably wont work if both processes are in the
-- same session. In such a case (ie. the intended manner of server-side 
-- debugging) an alternate method is needed that doesn't require a commit.
--
procedure debug_message(msg in varchar2) is
begin
  insert into SYS.de_debug_table values(
                  counter, 
                  substr(msg, 1, 80),   -- dont overflow the de_debug_table
                  pipe_base);
  counter := counter + 1;
  commit;
end debug_message;

----------------------------------------------------------------------------
-- Dump the current depth to the debug stream. 
--
procedure print_current_depth is
begin
  if (current_depth < 0) then
    debug_message('*ERROR* pnl: current depth ' || current_depth);
  else
    debug_message('pnl: current depth ' || current_depth);
  end if;
end print_current_depth;


----------------------------------------------------------------------------
procedure get_pipe_names(inpipe out varchar2, outpipe out varchar2) is
begin
  inpipe  := input_pipe;
  outpipe := output_pipe;
end get_pipe_names;


----------------------------------------------------------------------------
-- Set input and output pipe names. 
--   "pipe_base" should (preferably) have been obtained by a call to 
--   pbsde.init
--   (in the process being debugged).
--
-- We reset "timeouts" since there should not be any messages from the 
-- debugger stacked on the pipe yet.
--
function attach(new_pipe_base in varchar2) return boolean is
begin

  --pipe_base   := 'DE_PIPE';        -- $$$ temporarily default to a constant
  pipe_base   := new_pipe_base;
  output_pipe := pipe_base || pbutl.de_to_pls_extension;
  input_pipe  := pipe_base ||  pbutl.pls_to_de_extension;

  if (debugging = full_debugging) then
    debug_message('pipe: ' || pipe_base);
  end if;

  --timeouts := 0;    --> Not currently used.

  return true;
exception
  when others then return false;
end attach;

----------------------------------------------------------------------------
-- Shut down the pipes. Another call to init will be required in order to 
-- run after this (to create new pipes).
-- 
procedure shutdown is
  dummy integer;
begin
  /* $$$ Amit says we can ignore the retval 'dummy' */
  dummy := dbms_pipe.remove_pipe(input_pipe);
  dummy := dbms_pipe.remove_pipe(output_pipe);

  pipes_open := false;     -- not actually used in pbrph, but it should be
  input_pipe := NULL;
  output_pipe := NULL;

end shutdown;

----------------------------------------------------------------------------
function set_debug_level(debug_level in binary_integer) return boolean is
begin
  -- DEBUGGING
  debug_message('Setting debug_level to: ' || debug_level);

  if (debug_level = full_debugging) then 
    debugging := full_debugging;
  elsif (debug_level = no_debugging) then
    debugging := no_debugging;
  else
    debug_message('Debug level defaulting to 1 (dont understand '||
                   debug_level || ' )');
    debugging := full_debugging;
    return false;
  end if;
  return true;
exception
  when others then return false;
end set_debug_level;

----------------------------------------------------------------------------
-- Return true if the pipe is not open. Set the ctx.status as a side
-- effect.
--
function pipe_not_open(ctx in out pbutl.de_context) return boolean is
begin
  if ((input_pipe is null) or (output_pipe is null)) then
    ctx.status := op_uninitialized;
    ctx.be_status := null;
    return true;
  end if;
  return false;
end pipe_not_open;

----------------------------------------------------------------------------
-- Read one word from vchar, ie. read up until end of vchar or until the
-- first whitespace. Return the index of the whitespace in "end_offset".
-- "finished" is true if the word consumed the rest of vchar.
-- 
procedure read_word(vchar in varchar2,
                    word out varchar2,
                    end_offset out binary_integer,
                    finished   out boolean) is
  tempv varchar2(100);
  offset binary_integer;
  space  binary_integer;
begin
  tempv := ltrim(vchar);
  if (tempv is null) then
    finished := true;
    return;
  end if;

  offset := length(vchar) - length(tempv);
  space := instr(tempv, ' ');
  end_offset := offset + space;

  if (space = 0) then
    -- last word in the string
    word := tempv;
    finished := true;
  else
    finished := false;
    word := substr(tempv, 1, space-1);
  end if;
--exception
--  when others then raise VALUE_ERROR;
end read_word;


----------------------------------------------------------------------------
-- write_request : write the request to the pipe, then read the reply.
--                 Return false if anything goes wrong.
--
-- $$$ What is the relationship between "status" and the return-value?
--
function write_request(ctx       in out pbutl.de_context,
                       request   in     varchar2, 
                       reply     in out varchar2,
                       timeout   in     binary_integer
                       ) return boolean is
  first_space binary_integer;
begin
  if pipe_not_open(ctx) then return false; end if;

  if (debugging = full_debugging) then
    debug_message('wr -sending : ' || request);
  end if;

  ctx.status := op_pack_message_error;
  ctx.be_status := pbutl.be_failure;
  dbms_pipe.pack_message(request);
  ctx.status := op_send_message_error;
  ctx.status := dbms_pipe.send_message(output_pipe);

  if (ctx.status = 0) then   -- wait for a reply
    ctx.status := op_timed_out;

    ctx.status := dbms_pipe.receive_message(input_pipe, timeout);
    if (ctx.status = 0) then
      ctx.status := op_unpack_message_error;
      dbms_pipe.unpack_message(reply);
      if (debugging = full_debugging) then
        debug_message('wr -response: ' || reply);
      end if;
      ctx.status := op_success;
      ctx.be_status := pbutl.be_success;

      -- trim any blanks off "reply", then unpack it into a return-code plus
      -- additional string
      reply := ltrim(reply);
      first_space := instr(reply, ' ');
      ctx.be_status := substr(reply, 1, first_space);
      reply := substr(reply, first_space, length(reply));
      return true;
    else  -- 
      if (debugging = full_debugging) then
        debug_message('wr - failed to receive message, status: ' || ctx.status);
      end if;
      return false;       -- timed out waiting for the reply
    end if;

  else
    if (debugging = full_debugging) then
      debug_message('wr - failed to send message, status: ' || ctx.status);
    end if;
    return false;        -- the send failed
  end if;
exception
  when others then 
    ctx.status := op_exception;
    if (debugging = full_debugging) then
      debug_message('wr - exception was raised');
    end if;
    return false;
end write_request;

----------------------------------------------------------------------------
-- Get the requested row from source_tab.
-- 
-- $$$ buflen is not currently used: use it later to figure out if we can
--     get more than one piece from source_table into buf.
--
procedure get_more_source(ctx    in out pbutl.de_context,
                          buf    in out varchar2,
			  buflen in     binary_integer,
			  piece  in     binary_integer) is
begin
  if (piece > source_tab_len) then
    ctx.status := op_exception;
    buf := null;
    return;
  end if;
  ctx.status := op_success;
  buf := source_tab(piece);
  -- source_tab(piece) := null;  --> $$$ Free the space ???
end get_more_source;


----------------------------------------------------------------------------
-- Print out the source that is currently being executed. 
-- low,high,range,print_arrow specify the range of lines to print out, and 
-- whether to print an arrow next to the current line.
--
-- Read the source into the global indexed table if it is too large for the
-- provided buffer. Additional calls can be made to get the rest of the
-- source.
--
-- "pieces" indicates how many pieces were constructed:
--    0 : error
--    1 : all of whole source was put in buf
--   >1 : additional pieces can be obtained via calls to get_more_source.
--
procedure print_source(ctx in out pbutl.de_context,
                       low         in     binary_integer,
                       high        in     binary_integer,
                       window      in     binary_integer,
                       print_arrow in     binary_integer,
                       buf         in out varchar2,
		       buflen      in     binary_integer,
		       pieces         out binary_integer,
                       timeout            binary_integer) is
  request varchar2(30);
begin
  request := pbutl.request_print_source || ' ' ||
             low || ' ' || high || ' ' || window || ' ' || print_arrow;

  if read_into_table(ctx, request, pbutl.source_coming, pbutl.send_next_source,
                    timeout) then
    if (source_tab_len > 0) then
      buf := source_tab(1);
    else
      buf := '';
    end if;
    pieces := source_tab_len;
  else
    -- error.
    buf := '<-- Failed to obtain the source -->';
    pieces := 0;
  end if;

  -- $$$ ??? $$$ ???
  -- Strip the extra space from the front of source
  --lin := substr(lin,2);
end print_source;

----------------------------------------------------------------------------
function set_multiple(bpap in out pbutl.peibpa,
                      timeout binary_integer) return boolean is
begin
  --NYI $$$
  return false;
end set_multiple;

----------------------------------------------------------------------------
function get_multiple(bpap in out pbutl.peibpa,
                      timeout binary_integer)
        return boolean is
begin
  --NYI $$$
  return false;
end get_multiple;

----------------------------------------------------------------------------
-- send_pending_requests : do these in the order in which they came in, since
--     the user is referring to them by number.
-- 
-- $$$ Make sure that we are disposing of the memory used by the pending 
--     request. (Does setting to null actually do that?)
-- $$$ Would be nice to have some feedback to the user.
-- $$$ Need to establish the correlation between user's local breakpoint 
--     numbers and the remote numbers.
--
procedure send_pending_requests(ctx in out pbutl.de_context,
                                request_queue in out request_table,
                                timeout in binary_integer) is
  temp    boolean;
  counter binary_integer := 1;
begin
  if (request_queue.len is null) then return; end if;

  while (counter <= request_queue.len) loop
    if (debugging = full_debugging) then
      debug_message('spr: ' || request_queue.entries(counter));
    end if;
    temp := write_request(ctx, request_queue.entries(counter), buf, timeout);
    --> $$$ Clear the pending request
    request_queue.entries(counter) := '';
    counter := counter + 1;
  end loop;
  request_queue.len := 0;         -- no more pending requests
  ctx.status := op_success;

exception
  when others then ctx.status := op_exception;
end send_pending_requests;

----------------------------------------------------------------------------
-- Enqueue the request on the provided queue.
-- Note that this is a 1-based queue. (first entry is queue(1)).
--
procedure enqueue(ctx in out pbutl.de_context,
                  request in varchar2, 
                  queue in out request_table) is
begin
  queue.len := queue.len + 1;
  queue.entries(queue.len) := request;
  ctx.status := op_request_queued;
  if (debugging = full_debugging) then
    debug_message('queued: ' || queue.entries(queue.len));
  end if;
exception
  when others then ctx.status := op_exception;
end enqueue;

----------------------------------------------------------------------------
-- Process a request: ie. either send it and read the results or enqueue it
-- for later transmission.
--
-- Returns:  0 : normal completion
--           1 : request was enqueued
--           2 : error in write_request (see ctx for actual error)
--
-- NOTE: global variable "current_depth" is used to determine if the pipe is
--       available. This means that requests will not get sent directly if 
--       control is paused at a DE_INIT or a DE_EXIT for the top-level unit
--       (since depth will have gone back to 0). If such functionality is
--       desired then an additional status variable will be required (and in
--       fact it might not be a bad idea...)
--
function process_request(ctx        in out pbutl.de_context,
                         request    in out varchar2,
                         result_buf in out varchar2,
                         timeout    in     binary_integer
			 ) return binary_integer is
begin
  if (output_pipe is null) or (current_depth < 1) then
    -- The pipe is not currently available
    enqueue(ctx, request, request_queue);
    return 1;
  else
    if write_request(ctx, request, result_buf, timeout) then
      return 0;
    else
      return 2;   -- error in write_request
    end if;
  end if;
end process_request;

----------------------------------------------------------------------------
-- returns the breakpoint number
--
function set_break_entry(ctx in out pbutl.de_context,
                         unit_name in varchar2,
                         unit_type in binary_integer,
                         entry in binary_integer,
                         timeout binary_integer)
        return binary_integer is
begin
  ctx.status := 0;
  request := pbutl.request_set_breakpoint_entry || ' ' || unit_name || 
             ' ' || unit_type || ' ' || entry;

  if (process_request(ctx, request, buf, timeout) = 0) then
    -- the request completed successfully
    return buf;
  else
    -- either the request was queued or there was a failure
    return 0;
  end if;
exception 
  when others then return op_exception;
end set_break_entry;

----------------------------------------------------------------------------
-- returns the breakpoint number
--
function set_break_line(ctx in out pbutl.de_context,
                        unit_name in varchar2,
                        unit_type in binary_integer,
                        entry in binary_integer,
                        timeout binary_integer)
        return binary_integer is
begin
  ctx.status := 0;
  request := pbutl.request_set_breakpoint_line || ' ' || unit_name || 
             ' ' || unit_type || ' ' || entry;

  if (process_request(ctx, request, buf, timeout) = 0) then
    -- the request completed successfully
    return buf;
  else
    -- either the request was queued or there was a failure
    return 0;
  end if;
exception
  when others then return op_exception;
end set_break_line;

----------------------------------------------------------------------------
-- $$$ This should probably return a status.
--     It is most unpleasant that the response gets dropped on the floor.
--     However this interface is supposed to be as close as possible to 
--     the C interface.
--
procedure delete_breakpoint(ctx in out pbutl.de_context,
                            breakpoint in binary_integer,
                            timeout binary_integer) is
 
  temp binary_integer;
begin
  request := pbutl.request_delete_breakpoint || ' ' || breakpoint;
  temp := process_request(ctx, request, buf, timeout);
--  temp := write_request(ctx, request, buf, timeout);
end delete_breakpoint;

----------------------------------------------------------------------------
-- $$$ Should probably return a status
procedure enable_breakpoint(ctx in out pbutl.de_context,
                            breakpoint in binary_integer,
                            timeout binary_integer) is
temp binary_integer;
begin
  request := pbutl.request_enable_breakpoint || ' ' || breakpoint;
  temp := process_request(ctx, request, buf, timeout);
  --temp := write_request(ctx, request, buf, timeout);
end enable_breakpoint;

----------------------------------------------------------------------------
-- $$$ Should probably return a status
procedure disable_breakpoint(ctx in out pbutl.de_context,
                             breakpoint in binary_integer,
                             timeout binary_integer) is
  temp binary_integer;
begin
  request := pbutl.request_disable_breakpoint || ' ' || breakpoint;
  temp := process_request(ctx, request, buf, timeout);
  --temp := write_request(ctx, request, buf, timeout);
end disable_breakpoint;

----------------------------------------------------------------------------
-- $$$ No longer call set_debug_flags : instead pass them to continue.
--
--function set_debug_flags(ctx in out pbutl.de_context,
--                         debug_flags in binary_integer,
--                         timeout binary_integer) return binary_integer is
--  temp binary_integer;
--begin
--
--  if (debugging = full_debugging) then
--    debug_message('sdf : ' || debug_flags);
--  end if;
--
--  request := pbutl.request_set_debug_flags || ' ' || debug_flags;
--  if write_request(ctx, request, buf, timeout) then
--    return op_success;
--  else
--    return op_generic_failure;
--  end if;
--exception
--  when others then return op_exception;
--end set_debug_flags;

----------------------------------------------------------------------------
--
function get_debug_flags(ctx in out pbutl.de_context,
                         debug_flags in out binary_integer,
                         timeout binary_integer) return binary_integer is
  temp binary_integer;
begin
  request := pbutl.request_get_debug_flags;
  if write_request(ctx, request, buf, timeout) then
    if (debugging = full_debugging) then
      debug_message('gdf : ' || buf);
    end if;
    debug_flags := buf;
    return op_success;
  end if;
    return -1;
exception
  when others then ctx.status := op_exception; return -1;
end get_debug_flags;

----------------------------------------------------------------------------
-- Give a listing of all the breakpoints that are being remembered until 
-- a connection is established.
--
procedure get_pending_request(ctx in out pbutl.de_context,
                              one_request in out varchar2,
                              nth in binary_integer) is
begin

  if (nth > 0) and (nth <= request_queue.len) then
    one_request := request_queue.entries(nth);
    ctx.status := op_success;
  else
    ctx.status := op_generic_failure;
  end if;
--
-- The old implementation: this one returned a table of all the pending
-- requests. The problem with this is that I got an exception on user-side if
-- I tried to look at any of these entries.
--
--  out_table.len := request_queue.len;
--  while (out_table.len < request_queue.len) loop
--    out_table.entries(out_table.len) := request_queue.entries(out_table.len);
--    out_table.len := out_table.len + 1;
--  end loop;
--
  exception
    when others then ctx.status := op_generic_failure;
end get_pending_request;


----------------------------------------------------------------------------
procedure print_all_breakpoints(ctx in out pbutl.de_context,
                                listing in out varchar2,
                                timeout binary_integer) is
  temp binary_integer;
begin
  listing := '=+=+ No breakpoint listing available +=+=';
  request := pbutl.request_print_breakpoints;

  if write_request(ctx, request, listing, timeout) then
    -- the listing includes the back-end status
    listing := ltrim(listing);
    temp := instr(listing,' ');
    ctx.be_status := substr(listing, 1, temp);
    listing := substr(listing, temp, length(listing));
  end if;
exception
  when others then
    if (ctx.status = 0) then ctx.status := op_exception; end if;
    if (listing is null) then listing := '** exception in p_a_b **'; end if;
end print_all_breakpoints;

----------------------------------------------------------------------------
-- Return the current unit, line-number, etc.
--
procedure print_source_location(ctx in out pbutl.de_context,
                                source_line    out binary_integer,
                                unit_name   in out varchar2,
                                namespace      out binary_integer,
                                unit_owner  in out varchar2,
                                dblink      in out varchar2,
                                timeout     in     binary_integer
                               ) is

  -- $$$ Direct references dont work: so put into this bogus varchar2.
  request varchar2(20);
  temp    boolean;
  ndx     binary_integer;
  buf     varchar2(200);
  finished boolean;
begin
  request := pbutl.request_print_source_location;
  unit_name := '';
  namespace := -1;
  source_line := -1;
  if (write_request(ctx, request, buf, timeout)) then
    if (ctx.be_status <> pbutl.be_success) then
      -- the line-number will be the only thing in the message
      source_line := buf;
    else
      -- buf looks like: <source_line> <unit_name> <namespace> <user> <dblink>
      read_word(buf, source_line, ndx, finished);
      if finished then raise VALUE_ERROR; end if;
      
      buf := substr(buf, ndx);
      read_word(buf, unit_name, ndx, finished);
      if finished then raise VALUE_ERROR; end if;

      buf := substr(buf, ndx);
      read_word(buf, namespace, ndx, finished);
      if finished then raise VALUE_ERROR; end if;

      buf := substr(buf, ndx);
      read_word(buf, unit_owner, ndx, finished);
      if finished then raise VALUE_ERROR; end if;

      buf := substr(buf, ndx);
      read_word(buf, dblink, ndx, finished);
    end if;
  end if;
exception
  when others then
    if (ctx.status = op_success) then
      ctx.status := op_exception;
    end if;
end print_source_location;

----------------------------------------------------------------------------
-- continue : put a message on the pipe saying to continue.
--            Wait for the next communication if the current depth is 
--            greater than 0.
--
-- RPC instantiation of package spec and body is implemented by additional
-- calls to the interpreter. This means that if a package entrypoint is
-- called and the package has not been instantiated then the interpreter
-- will be called once for each instantiation that requires initialization
-- and then once for the requested entrypoint. Therefore we have an 
-- additional event DE_INSTANTIATION, and the idea is that, upon seeing this,
-- we do NOT return when we see the corresponding DE_EXIT. Instead we
-- continue (using the same flags) and listen for the next event.
--
procedure continue(ctx         in out pbutl.de_context,
                   debug_flags in     binary_integer,
                   reason      in out binary_integer,
                   source_line    out binary_integer,
		   unit_name in   out varchar2,
		   namespace      out binary_integer,
                   unit_owner  in out varchar2,
                   dblink      in out varchar2,
                   depth       in out binary_integer,
                   timeout     in     binary_integer) is
begin
  continue2(ctx, debug_flags, reason, source_line, unit_name, namespace,
            unit_owner, dblink, depth, timeout,
	    false);
end continue;


procedure continue2(ctx         in out pbutl.de_context,
                    debug_flags in     binary_integer,
                    reason      in out binary_integer,
                    source_line    out binary_integer,
		    unit_name in   out varchar2,
		    namespace      out binary_integer,
                    unit_owner  in out varchar2,
                    dblink      in out varchar2,
                    depth       in out binary_integer,
                    timeout     in     binary_integer,
		    ignore_depth in    boolean) is
  retval    binary_integer;
  bogval    boolean;
  terminate boolean := false;
begin
  if (debugging = full_debugging) then
    debug_message('Continue ' || debug_flags);
  end if;

  depth := current_depth;
  ctx.terminated := false;

  if ((current_depth <= 0) and not ignore_depth) then
    -- Synchronization error: no program is currently being debugged
    -- (call listen() to get the first communication from a program)
    ctx.status := op_synch_error;
    -- clear the OUT params, just in case caller looks at them
    reason := 0; source_line := ''; unit_name := ''; namespace := -1;
    unit_owner := ''; dblink := '';
    return;
  else
    -- $$$ HACK!!!  pbsde currently still reads the first 3 characters and 
    --              converts them to a number. Fix this!!.
    --request := pbutl.request_continue || ' ' || debug_flags;
    request := pbutl.request_continue || '   ' || debug_flags;

    if write_request(ctx, request, buf, timeout) then
    -- $$$ Check that the return (buf) was "OK" ?

    -- listen for the next break
      listen(ctx, reason, --frame_count, offset, 
             source_line, unit_name, namespace, unit_owner, dblink, 
	     depth, timeout);
      ---------------------------------------------------------------------
      -- No pausing allowed at DE_EXIT (since there's really nothing useful
      -- that can be done there). Instead, automatically continue (with
      -- the same execution flags), unless we are at level 0.
      --
      if (reason = pbutl.de_exit) then
        if (depth = 0) then
	  if (instantiate_depth > 0) then
	    instantiate_depth := instantiate_depth - 1;
	  else
	    terminate := true;
	  end if;
	end if;   -- (depth = 0)

        if (terminate) then
	  -- We are done. Release the debugged process by telling it to
	  -- continue (with flags 0).
          request := pbutl.request_continue || '   0';
	  bogval := write_request(ctx, request, buf, timeout);
	  ctx.terminated := true;
          if (debugging = full_debugging) then
            debug_message('cont: terminating at level ' || current_depth);
          end if;
	else
	  -- keep going, via a recursive call to continue2. 
	  -- (We pass in true for ignore_depth since we know that either 
	  --  depth is non-zero or that we can safely ignore it).
          continue2(ctx, debug_flags, reason, source_line, unit_name, 
	            namespace, unit_owner, dblink, depth, timeout, 
		    true);
	end if;  -- (terminate)
      end if;    -- (reason = pbutl.de_exit)
    end if;      -- write_request
  end if;        -- (current_depth <= 0)
exception
  when others then ctx.status := op_exception;
end continue2;


----------------------------------------------------------------------------
-- flush_pipes : flush both pipes by repeatedly reading from them.
--
procedure flush_pipes(ctx in out pbutl.de_context,
                      num_in_messages  in out binary_integer,
                      num_out_messages in out binary_integer) is
  flush_status binary_integer;
begin
  ctx.status := op_success;
  num_in_messages := 0;
  num_out_messages := 0;
  
  if pipe_not_open(ctx) then return; end if;

  -- flush the input pipe
  flush_status := 0;
  while (flush_status = 0) loop
    flush_status := dbms_pipe.receive_message(input_pipe,1);
    if (flush_status = 0) then
      num_in_messages := num_in_messages + 1;
    end if;
  end loop;

  -- flush the output pipe
  flush_status := 0;
  while (flush_status = 0) loop
    flush_status := dbms_pipe.receive_message(output_pipe,1);
    if (flush_status = 0) then
      num_out_messages := num_out_messages + 1;
    end if;
  end loop;
exception
  when others then ctx.status := op_exception;
end flush_pipes;

----------------------------------------------------------------------------
-- Delete_debug_table : empty the debug table.
procedure delete_debug_table(ctx in out pbutl.de_context) is
begin
  -- No point in doing any debug_info, since we're deleting that table.
  delete SYS.de_debug_table;
exception
  when others then ctx.status := op_exception;
end delete_debug_table;

----------------------------------------------------------------------------
-- listen : Wait for a message on input_pipe.
--
procedure listen(ctx         in out pbutl.de_context,
                 reason      in out binary_integer,
                 --frame_count  out binary_integer,
                 --offset       out binary_integer,
                 source_line    out binary_integer,
                 unit_name   in out varchar2,
                 namespace      out binary_integer,
                 unit_owner  in out varchar2,
                 dblink      in out varchar2,
                 depth       in out binary_integer,
                 timeout     in     binary_integer) is
  response varchar2(200);
  temp     binary_integer;
begin

  ctx.terminated := false;   -- should ALWAYS be false
  reason := pbutl.de_none;
  -- Set all the OUT variables (note that some are in/out just for convenience)
  unit_name := ''; unit_owner := ''; dblink := '';
  namespace := -1; source_line := -1;

  if pipe_not_open(ctx) then return; end if;

  ctx.status := dbms_pipe.receive_message(input_pipe, timeout);
  depth := current_depth;
  if (ctx.status = 0) then
    ctx.status := op_unpack_error;
    dbms_pipe.unpack_message(response);
    if (debugging = full_debugging) then
      debug_message('listen -received: ' || response);
    end if;
    ctx.status := op_success;

    -- The reason for the suspension is the only thing in the message
    response := rtrim(response);
    temp := instr(response,' ');
    if (temp = 0) then
      reason := response;
    else
      -- the message contained more than one field: things are hosed. Try to
      -- avoid an exception by taking the first field (and hope that it's a
      -- number).
      reason := substr(response,1,temp);
    end if;
    if (reason = pbutl.de_init) then
      current_depth := current_depth + 1;
    elsif (reason = pbutl.de_instantiate) then
      current_depth := current_depth + 1;
      instantiate_depth := instantiate_depth + 1;
    elsif (reason = pbutl.de_exit) then
      current_depth := current_depth - 1;
      -- NOTE:  instantiate_depth is decremented in continue(), which 
      --        we assume that listen was called from.
    end if;
    depth := current_depth;
    
    -- For debugging any problems with depth (either premature detachment or
    -- hanging on after the server-side program has actually exited).
    if (debugging = full_debugging) then
      print_current_depth;
    end if;

    -- Print source if it is meaningful at this point
    if ((reason <> pbutl.de_init) and (reason <> pbutl.de_instantiate) and
        (reason <> pbutl.de_exit)) then
      print_source_location(ctx, source_line, unit_name, namespace, unit_owner,
			    dblink, timeout);
    end if;
    -- Do any work that we have queued
    send_pending_requests(ctx, request_queue, timeout);
  else
    -- The pipe timed out. 
    ctx.status := op_timed_out;
  end if;
exception
  -- status should already indicate the error: if it doesn't then set it to
  -- indicate an exception occurred.
  when others then 
    if (ctx.status = op_success) then 
      ctx.status := op_exception; 
    end if;
end listen;

----------------------------------------------------------------------------
-- get_oracle_processes: return a listing of the processes.
-- Mainly for use by the process browser/debugger (tool2).
--
procedure get_oracle_processes(ctx     in out pbutl.de_context, 
                               padsize in binary_integer,
                               result  in out varchar2) is
  cursor get_proc_cursor is 
    select spid ORACLE, process SQLDBA, v$session.terminal
    from v$session,v$process
    where (v$process.program like '%V1%' 
           or v$process.program like '%Two-Task%')
           and v$session.terminal = v$process.terminal;

  oracle_pid varchar2(20);
  sqldba_pid varchar2(20);
  terminal   varchar2(20);
begin
  ctx.status := op_success;
  open get_proc_cursor;
  loop
    fetch get_proc_cursor into oracle_pid, sqldba_pid, terminal;
    if (get_proc_cursor%NOTFOUND) then exit; end if;
    result := result || lpad(terminal, padsize)
                     || lpad(oracle_pid, padsize)
                     || lpad(sqldba_pid, padsize)
                     || '
';  
  end loop;
exception
  when others then ctx.status := op_exception;
end get_oracle_processes;

--------------------------------------------------------------------------------
--                        DISPLAY SECTION

----------------------------------------------------------------------------
-- $$$ Add a backtrace argument so that you can get only part of the 
--     backtrace upon demand.
--
procedure print_backtrace(ctx     in out pbutl.de_context,
                          listing in out varchar2,
                          timeout        binary_integer) is
  retval boolean;
begin
  ctx.status := op_success;
  request := pbutl.request_print_backtrace;
  retval := write_request(ctx, request, listing, timeout);
exception
  when others then ctx.status := op_generic_failure;
end print_backtrace;


----------------------------------------------------------------------------
-- multi_read - read n messages, responding after each.
-- Store each message into source_tab.
--
-- (dbms_pipe imposes a limit on the message-size, so large messages must
--  be split into smaller ones).
--
-- Return: false if any error
--
-- $$$ THIS SHOULD CALL WRITE_REQUEST (OR A VERSION OF WRITE_REQUEST) INSTEAD
--     OF DOING PIPE READS DIRECTLY.
--
function multi_read(ctx in out pbutl.de_context,
                    pieces in binary_integer,  -- how many messages to read
                    next in varchar2,          -- send this message to request
		                               -- the next piece
		    timeout in  binary_integer
                    ) return boolean is
  temp_buf varchar2(2000);
begin    

  -- Clean out anything that's left in source_tab
  for i in 1..source_tab_len loop
    source_tab(i) := '';
  end loop;
  source_tab_len := 0;

  for i in 1..pieces loop
    ctx.status := op_timed_out;
    temp_buf := '';
    ctx.status := dbms_pipe.receive_message(input_pipe, timeout);
    if (ctx.status = 0) then
      ctx.status := op_unpack_message_error;
      dbms_pipe.unpack_message(temp_buf);
      if (debugging = full_debugging) then
        -- message: 'mr: [<number>,<length>] <message>'
        debug_message('mr: [' || i || ',' || length(temp_buf) ||
	              '] ' || temp_buf);
      end if;
      source_tab(i) := temp_buf;
      source_tab_len := source_tab_len + 1;
      -- send a request for the next piece, unless this is the last one
      if (i < pieces) then
	if (debugging = full_debugging) then
	  debug_message('mr - sending : ' || next);
	end if;
        dbms_pipe.pack_message(next);
	ctx.status := dbms_pipe.send_message(output_pipe);
	if (ctx.status <> 0) then
	  if (debugging = full_debugging) then
	    debug_message('mr - failed to send message');
	  end if;
	  return false;
	end if;  -- ctx.status <> 0
      end if;    -- i < pieces
    else  -- 
      if (debugging = full_debugging) then
        debug_message('mr - failed to receive message, status: ' || 
	               ctx.status);
      end if;
      return false;
    end if;   -- ctx.status = 0
  end loop;

  ctx.status := op_success;
  ctx.be_status := pbutl.be_success;
  return true;

exception
  when others then
    if (debugging = full_debugging) then
      debug_message('** Exception in multi_read **');
    end if;
    return false;
end multi_read;

-----------------------------------------------------------------------------
-- read_into_table : read a sequence of messages into source_tab.
-- Return false if there were any problems, else true, in which case
-- source_tab has the messages, and source_tab_len indicates how many there
-- are. (source_tab is 1-based).
--
function read_into_table(
              ctx             in out pbutl.de_context,
	      initial_message in     varchar2,       -- initial request to send
	      response        in     binary_integer, -- reply from pipe
	      next_message    in     binary_integer, -- synch message to pipe
              timeout         in     binary_integer) 
	      return boolean is
  buf       varchar2(20);
  reply     binary_integer;
  pieces    binary_integer;
  ndx       binary_integer;
  finished  boolean;
begin
  ctx.status := op_success;

  if write_request(ctx, initial_message, buf, timeout) then
    -- buf should be: <response> <number of messages>
    read_word(buf, reply, ndx, finished);
    if finished then raise VALUE_ERROR; end if;
    if (reply <> response) then
      -- Either a pipe logic error or else the requested source/TIDL isn't
      -- available.
      return false;
    end if;

    buf := substr(buf, ndx);
    read_word(buf, pieces, ndx, finished);

    if multi_read(ctx, pieces, next_message, timeout) then
      -- Results have been written into source_tab
      return true;
    else
      -- some error in reading the messages
      return false;
    end if;
  else
    -- back-end couldn't create the source/tidl for some reason
    return false;
  end if;

  -- should never be reached
  return false;

exception
  when others then 
    if (ctx.status = op_success) then ctx.status := op_exception; end if;
    if (debugging = full_debugging) then
      debug_message('rit - exception');
    end if;
    return false;
end read_into_table;



----------------------------------------------------------------------------
-- Print out the frame information for one or more frames on the current
-- call stack.
--
-- $$$ Modify to place only tidl_buflen characters into tidl_buf and then
--     allow user to get additional pieces via another procedure like
--     print_source does.
-- $$$
procedure display_frame(
              ctx in out pbutl.de_context,
              start_frame       in     binary_integer,  -- frame to start from
              frame_count       in     binary_integer,  -- # of frames to fetch
              flags             in     binary_integer,  -- ????
              max_string_length in     binary_integer,  -- longest string
              max_index_values  in     binary_integer,  -- max table values
              tidl_buf          in out raw,             -- buf to place result
	      --tidl_buflen       in     binary_integer,  -- size of tidl_buf
	      --pieces             out  binary_integer,
              timeout           in binary_integer) is
  temp_buf  varchar2(32760);
begin
  ctx.status := op_success;
  request := pbutl.request_display_frame || ' ' ||
             start_frame       || ' ' ||
             frame_count       || ' ' ||
             flags             || ' ' ||
             max_string_length || ' ' ||
             max_index_values;

  if read_into_table(ctx, request, pbutl.tidl_coming, pbutl.send_next_tidl, 
                    timeout) then
    -- assemble the pieces from source_tab
    tidl_buf := '';
    temp_buf := '';
    for i in 1..source_tab_len loop
      temp_buf := temp_buf || source_tab(i);
    end loop;
    tidl_buf := temp_buf;
  else
    -- no TIDL for us
    tidl_buf := '';
  end if;

exception
  when others then 
    if (ctx.status = op_success) then ctx.status := op_exception; end if;
    if (debugging = full_debugging) then
      debug_message('df - exception');
    end if;
end display_frame;

end pbrph;
/
