/* routines that interface with the kernel's IPsec mechanism
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: kernel.c,v 1.2 2000/11/20 20:27:11 victor Exp $
 */
/*
 * changed by dev@fx.dk (port to OS/2) 24/07/1999
 * changed by dev@fx.dk 09/10/99
 * changed by dev@fx.dk 29/10/1999 (new snapshot)
 */
#include "os2port.h"


#include <stddef.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#ifdef KLIPS
/* A little experimentation indicates that the order of the next six
 * includes is critical.  I don't know why, or what the rules are.  Ugh!
 */
# include <sys/types.h>
# include <linux/types.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <netdb.h>

# include <unistd.h>
# include <fcntl.h>

# include <freeswan.h>

# include <radij.h>
# include <ipsec_encap.h>
# include <ipsec_netlink.h>
# include <ipsec_xform.h>
# include <ipsec_ipe4.h>
# include <ipsec_ah.h>
# include <ipsec_esp.h>

#else

/* this collection is included in the other case, but in an order
 * that makes it hard to make the #includes common.
 */
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>

# include <freeswan.h>

#endif

#include "constants.h"
#include "defs.h"
#include "rnd.h"
#include "connections.h"
#include "state.h"
#include "kernel.h"
#include "log.h"
#include"server.h"

int  IIDLL_install_sa (struct state *st);
int  IIDLL_remove_sa (struct state *st);

bool no_klips = FALSE;	/* don't actually use KLIPS */

/* Generate Unique SPI numbers.
 *
 * The specs say that the number must not be less than 0x100.
 * XXX This should be replaced by a call to the kernel when
 * XXX we get an API.
 * The returned SPI is in network byte order.
 * We use a random number as the initial SPI so that there is
 * a good chance that different Pluto instances will choose
 * different SPIs.  This is good for two reasons.
 * - the keying material for the initiator and responder only
 *   differs if the SPIs differ.
 * - if Pluto is restarted, it would otherwise recycle the SPI
 *   numbers and confuse everything.  When the kernel generates
 *   SPIs, this will no longer matter.
 */
ipsec_spi_t
get_ipsec_spi(void)
{
    static ipsec_spi_t spi = 0;	/* host order! */

    spi++;
    while (spi < 0x100)
	get_rnd_bytes((u_char *)&spi, sizeof(spi));

    DBG(DBG_CONTROL,
	{
	    ipsec_spi_t spi_net = htonl(spi);

	    DBG_dump("generate SPI:", (u_char *)&spi_net, sizeof(spi_net));
	});

    return htonl(spi);
}

/* do the "route" commands required:
 *
 * deleting an unknown but possible route (preliminary to adding a route):
 *  route del -net <peeruser> netmask <peerusermask>
 *
 * adding a route:
 *  route add -net <peeruser> netmask <peerusermask> dev <ipsecdev> gw <peernexthop>
 *  /sbin/ipfwadm -F -i accept -b -S <myusersubnet> -D <peerusersubnet>
 *
 * deleting a route that we had previously added:
 *  /sbin/ipfwadm -F -d accept -b -S <myusersubnet> -D <peerusersubnet>
 *  route del -net <peeruser> netmask <peerusermask> dev <ipsecdev> gw <peernexthop>
 *
 * Note that this is not general:
 * - the peer's client must be a network
 * - the peer must be the gateway
 */

#ifndef ROUTECMD
# define ROUTECMD "/sbin/route"
#endif

#ifndef IPFWADMCMD
# define IPFWADMCMD "/sbin/ipfwadm"
#endif

static bool
do_route(struct connection *c, bool add, bool route_unknown)
{
    char route_cmd[100];
    char ipfwadm_cmd[100];
    char
	myuser_str[SOCKADDR_STRING_SIZE],
	myusernetmask_str[SOCKADDR_STRING_SIZE],
	peer_str[SOCKADDR_STRING_SIZE],
	peer_nexthop_str[SOCKADDR_STRING_SIZE],
	peeruser_str[SOCKADDR_STRING_SIZE],
	peerusernetmask_str[SOCKADDR_STRING_SIZE];
    int r;
    bool firewall = c->this.firewall && !route_unknown
	&& (c->this.client_net.s_addr != c->this.host.s_addr
	    || c->this.client_net.s_addr != mask32.sin_addr.s_addr);

    strcpy(myuser_str, inet_ntoa(c->this.client_net));
    strcpy(myusernetmask_str, inet_ntoa(c->this.client_mask));
    strcpy(peer_str, inet_ntoa(c->that.host));
    strcpy(peer_nexthop_str, inet_ntoa(c->this.host_nexthop));
    strcpy(peeruser_str, inet_ntoa(c->that.client_net));
    strcpy(peerusernetmask_str, inet_ntoa(c->that.client_mask));

    r = snprintf(route_cmd, sizeof(route_cmd)
	, (add? "%s add -net %s netmask %s dev %s gw %s"
	    : route_unknown? "%s del -net %s netmask %s"
	    : "%s del -net %s netmask %s dev %s gw %s")
	, ROUTECMD, peeruser_str,  peerusernetmask_str
	, c->interface->vname, peer_nexthop_str);
    passert(r > 0);

    if (firewall)
    {
	r = snprintf(ipfwadm_cmd, sizeof(ipfwadm_cmd)
	    , IPFWADMCMD " -F -%c accept -b -S %s/%s -D %s/%s"
	    , add? 'i' : 'd'
	    , myuser_str, myusernetmask_str
	    , peeruser_str, peerusernetmask_str);
	passert(r > 0);
    }

    DBG(DBG_CONTROL,
	if (firewall && !add)
	    DBG_log("executing command: %s", ipfwadm_cmd);
	DBG_log("executing command: %s", route_cmd);
	if (firewall && add)
	    DBG_log("executing command: %s", ipfwadm_cmd);
	);

#ifdef KLIPS
    if (!no_klips)
    {
	if (firewall && !add)
	{
	    r = system(ipfwadm_cmd);
	    if (r != 0)
	    {
		log("firewall forwarding failed %d: %s", r, ipfwadm_cmd);
		/* don't know what to do -- so we'll ignore the problem */
	    }
	}

	r = system(route_cmd);
	if (r != 0)
	{
	    if (route_unknown)
	    {
		/* ??? only r == 1792 should be ignored (no route to delete)? */
		DBG(DBG_KLIPS
		, DBG_log("ignoring exit status %d of routing: %s", r, route_cmd));
	    }
	    else
	    {
		log("routing failed %d: %s", r, route_cmd);
		return FALSE;
	    }
	}
	else
	{
	    DBG(DBG_KLIPS, DBG_log("routing succeeded: %s", route_cmd));

	    if (firewall && add)
	    {
		r = system(ipfwadm_cmd);
		if (r != 0)
		{
		    log("firewall forwarding failed %d: %s", r, ipfwadm_cmd);
		    /* don't know what to do -- so we'll ignore the problem */
		}
	    }
	}
    }
#endif /* KLIPS */

    return TRUE;
}

bool
route_connection(struct connection *c)
{
    if (c->routed)
    {
	/* already done */
    }
    else if (c->this.host_port != IKE_UDP_PORT
    && inside_client(c->that.host, c->that))
    {
	log("cannot install route: peer is within its client");
    }
    else
    {
	struct connection *d = route_owner(c, FALSE);

	passert(c->eroute_owner == SOS_NOBODY);
	if (d == NULL)
	{
	    /* nobody's got it: we can try to install our route */
	    (void) do_route(c, FALSE, TRUE);	/* just in case; ignore failure */
	    if (do_route(c, TRUE, FALSE))
		c->routed = TRUE;
	}
	else if (d->this.host_nexthop.s_addr == c->this.host_nexthop.s_addr
	&& d->interface == c->interface)
	{
	    /* The other connection has the same nexthop and interface,
	     * so we're done.
	     */
	    c->routed = TRUE;
	}
	else if (d->eroute_owner == SOS_NOBODY)
	{
	    /* The other connection has the route, and it conflicts
	     * (the nexthop or interface differs), but that connection
	     * isn't using the route.  We'll steal it!  There might be
	     * other connections using the same route, but none of
	     * them is using it either (otherwise route_owner()
	     * would have returned one that did).
	     *
	     * A feature of LINUX allows us to install the new route
	     * before deleting the old if the nexthops differ.
	     * This reduces the "window of vulnerability" when packets
	     * might flow in the clear.
	     *
	     * c->routed must be set last so that route_owner()
	     * doesn't find it.
	     */

	    bool preadd = d->this.host_nexthop.s_addr != c->this.host_nexthop.s_addr;

	    if (!preadd || do_route(c, TRUE, FALSE))
	    {
		/* one unroute for all */
		if (do_route(d, FALSE, FALSE))
		{
		    do {
			passert(d->eroute_owner == SOS_NOBODY);
			d->routed = FALSE;
			d = route_owner(c, FALSE);
		    } while (d != NULL);

		    if (preadd || do_route(c, TRUE, FALSE))
			c->routed = TRUE;
		}
	    }
	}
	else
	{
	    log("cannot install route: connection \"%s\" already has it"
		, d->name);
	}
    }
    return c->routed;
}

void
unroute_connection(struct connection *c)
{
    /* only unroute if no other connection shares it */
    if (c->routed)
    {
	c->routed = FALSE;
	if (route_owner(c, FALSE) == NULL && !do_route(c, FALSE, FALSE))
	    c->routed = TRUE;	/* undo on failure */
    }
}

#ifdef KLIPS
/* Setup an IPsec route entry. Code taken from addrt.c.
 * op is the KLIPS operator: EMT_DELEROUTE, EMT_SETEROUTE, or EMT_RPLACEROUTE
 */
static bool
do_eroute(int fd, struct state *st, __u8 op, const char *opname)
{
    struct connection *c = st->st_connection;
    struct encap_msghdr emh;

    memset(&emh, '\0', sizeof(emh));	/* clear unused parts */

    emh.em_magic = EM_MAGIC;
    emh.em_msglen = sizeof(emh);
    emh.em_version = 0;
    emh.em_type = op;
    emh.em_eaddr.sen_len = emh.em_emask.sen_len
			 = sizeof(struct sockaddr_encap);
    emh.em_eaddr.sen_family = emh.em_emask.sen_family = 26;
    emh.em_eaddr.sen_type = SENT_IP4;
    emh.em_emask.sen_type = 255;

    emh.em_eaddr.sen_ip_src.s_addr = c->this.client_net.s_addr;
    emh.em_emask.sen_ip_src.s_addr = c->this.client_mask.s_addr;

    emh.em_eaddr.sen_ip_dst.s_addr = c->that.client_net.s_addr;
    emh.em_emask.sen_ip_dst.s_addr = c->that.client_mask.s_addr;

    /* Fill in peer's address (the other security gateway) and
     * figure out the SPI to reference.
     * these are not needed for deleting an eroute
     * At this time, we only need these for setting the eroute.
     */
    if (op != EMT_DELEROUTE)
    {
	emh.em_erdst.s_addr = c->that.host.s_addr;

	/* multiple SPIs will be grouped; we must refer to the
	 * last in the sequence: ah esp encap.
	 */
	if (st->st_ah.present)
	{
	    emh.em_erspi = st->st_ah.attrs.spi;
	    emh.em_erproto = IPPROTO_AH;
	}

	if (st->st_esp.present)
	{
	    emh.em_erspi = st->st_esp.attrs.spi;
	    emh.em_erproto = IPPROTO_ESP;
	}

	if (st->st_ah.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL
	|| st->st_esp.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL)
	{
	    emh.em_erspi = st->st_tunnel_spi;
	    emh.em_erproto = IPPROTO_IPIP;
	}
    }

    DBG(DBG_CONTROL,
	{
	    char mybuf[SOCKADDR_STRING_SIZE*2];
	    char peerbuf[SOCKADDR_STRING_SIZE*2];

	    subnettoa(c->this.client_net, c->this.client_mask, 0, mybuf, sizeof(mybuf));
	    subnettoa(c->that.client_net, c->that.client_mask, 0, peerbuf, sizeof(peerbuf));
	    DBG_log("%s eroute %s to %s", opname, mybuf, peerbuf);
	});

    DBG_cond_dump(DBG_KLIPS, "eroute => KLIPS:\n", &emh, sizeof(emh));

    if (!no_klips)
    {
	if (write(fd, &emh, sizeof(emh)) != sizeof(emh))
	{
	    log_errno((e, "write to %s eroute", opname));
	    return FALSE;
	}
    }
    return TRUE;
}

static bool
del_spi(int fd, ipsec_spi_t spi, int proto, struct in_addr dest)
{
    struct encap_msghdr delmsg;

    memset(&delmsg, '\0', sizeof(delmsg));
    delmsg.em_magic = EM_MAGIC;
    delmsg.em_version = 0;
    delmsg.em_msglen = EMT_SETSPI_FLEN;
    delmsg.em_type = EMT_DELSPI;
    delmsg.em_alg = XF_DEL;
    delmsg.em_dst = dest;
    delmsg.em_proto = proto;
    delmsg.em_spi = spi;
    DBG_cond_dump(DBG_KLIPS, "delete SPI => KLIPS:\n",
	(void *) &delmsg, EMT_SETSPI_FLEN);
    if (!no_klips)
    {
	if (write(fd, (caddr_t)&delmsg, EMT_SETSPI_FLEN) != EMT_SETSPI_FLEN)
	{
	    char buf[SATOA_BUF];

	    log_errno((e, "write() of delete SPI %s failed"
		, (satoa(delmsg.em_said, 0, buf, sizeof(buf)), buf)));
	    return FALSE;
	}
    }
    return TRUE;
}

/* Setup a pair of SAs. Code taken from setsa.c and spigrp.c, in
 * ipsec-0.5.
 */
static bool
setup_ipsec_sa(int fd, struct state *st, bool initiator)
{
    /* We need to build both a Sending SA and a Receiving SA.
     * Since the logic for each is similar, we build both
     * netlink packets at once.  The parallel variables are
     * prefixed by s_ and r_.
     * The Receiver uses st->st_keymat
     * The Sender uses st->st_peerkeymat
     */
    struct connection *c = st->st_connection;

    /* encap_msghdr is from "ipsec_netlink.h"
     * It preceded each netlink message.
     * We create "generic" s_ and r_ headers to save work.
     * User must fill in:
     *	?.h.em_alg
     *	?.h.em_msglen
     *	?.h.em_type
     *	?.h.em_spi
     *	?.h.em_proto
     */
    struct encap_msghdr r_h, s_h;

    /* sender SPIs, saved for spigrouping, if necessary*/
    struct sa_id
	s_said[EM_MAXRELSPIS],
	*s_said_next = s_said,
	r_said[EM_MAXRELSPIS],
	*r_said_next = r_said;

    /* build generic message headers */

    memset(&r_h, '\0', sizeof(r_h));
    r_h.em_magic = EM_MAGIC;
    r_h.em_version = 0;
    r_h.em_if = 0;	/* ??? what is this? */

    s_h = r_h;	/* replicate hdr */

    r_h.em_flags |= EMT_INBOUND;

    r_h.em_dst = c->this.host;
    s_h.em_dst = c->that.host;

    /* Set our receiving SA */

    if (st->st_ah.present)
    {
	/* PROTO_IPSEC_AH */
	/* ??? mysterious code stolen from setsa.c:f_hmac()
	 * ??? why is struct ahhmacmd5_edata OK for SHA too?
	 */

	/* ahhmacmd5_edata is from "ipsec_ah.h"
	 * ahhmacsha1_edata has identical fields except that the ame_key
	 * field has a different length.
	 */
	union {
	    struct encap_msghdr h;
	    struct {
		char dummy[offsetof(struct encap_msghdr, em_dat)];
		struct ahhmacmd5_edata d;	/* works for both */
	    } u;
	} r_ah, s_ah;
	int alg;
	size_t ah_alen;

	memset(&r_ah, '\0', sizeof(r_ah));
	memset(&s_ah, '\0', sizeof(s_ah));

	r_ah.h = r_h;
	s_ah.h = s_h;

	r_ah.h.em_type = s_ah.h.em_type = EMT_SETSPI;

	r_ah.h.em_spi = st->st_ah.our_spi;
	s_ah.h.em_spi = st->st_ah.attrs.spi;
	r_ah.h.em_proto = s_ah.h.em_proto = IPPROTO_AH;

	switch (st->st_ah.attrs.auth)
	{
	case AUTH_ALGORITHM_HMAC_MD5:
	    alg = XF_AHHMACMD5;
	    ah_alen = MD5_DIGEST_SIZE;
	    break;

	case AUTH_ALGORITHM_HMAC_SHA1:
	    alg = XF_AHHMACSHA1;
	    ah_alen = SHA1_DIGEST_SIZE;
	    break;

	case AUTH_ALGORITHM_KPDK:
	case AUTH_ALGORITHM_DES_MAC:
	default:
	    log("%s not implemented yet",
		enum_show(&auth_alg_names, st->st_ah.attrs.auth));
	    goto fail;
	}

	/* ??? this length calculation is repulsive */
	r_ah.h.em_msglen = s_ah.h.em_msglen =
	    EMT_SETSPI_FLEN + 3 * sizeof(u_short) + 2 * sizeof(u_char) + st->st_ah.keymat_len;
	r_ah.h.em_alg = s_ah.h.em_alg = alg;
	r_ah.u.d.ame_alen = s_ah.u.d.ame_alen = ah_alen;
	r_ah.u.d.ame_klen = s_ah.u.d.ame_klen = st->st_ah.keymat_len;
	r_ah.u.d.ame_replayp = s_ah.u.d.ame_replayp = TRUE;
	r_ah.u.d.ame_ooowin = s_ah.u.d.ame_ooowin = 32;    /* ??? pure guess */
	memcpy(r_ah.u.d.ame_key, st->st_ah.our_keymat, st->st_ah.keymat_len);
	memcpy(s_ah.u.d.ame_key, st->st_ah.peer_keymat, st->st_ah.keymat_len);

	DBG_cond_dump(DBG_KLIPS, "setup inbound AH transform => KLIPS:\n", &r_ah, r_ah.h.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &r_ah, r_ah.h.em_msglen) != r_ah.h.em_msglen)
	    {
		char buf[SATOA_BUF];

		log_errno((e, "write() of inbound AH SPI %s failed in setup_ipsec_sa()"
		    , (satoa(r_ah.h.em_said, 0, buf, sizeof(buf)), buf)));
		goto fail;
	    }
	}
	*r_said_next++ = r_ah.h.em_said;

	DBG_cond_dump(DBG_KLIPS, "setup outbound AH transform => KLIPS:\n", &s_ah, s_ah.h.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &s_ah, s_ah.h.em_msglen) != s_ah.h.em_msglen)
	    {
		char buf[SATOA_BUF];

		log_errno((e, "write() of outbound AH SPI %s failed in setup_ipsec_sa()"
		    , (satoa(s_ah.h.em_said, 0, buf, sizeof(buf)), buf)));
		goto fail;
	    }
	}
	*s_said_next++ = s_ah.h.em_said;
    }

    if (st->st_esp.present)
    {
	/* PROTO_IPSEC_ESP */

	/* espblkrply_edata is from "ipsec_esp.h"
	 */
	union {
	    struct encap_msghdr h;
	    struct {
		char dummy[offsetof(struct encap_msghdr, em_dat)];
		struct espblkrply_edata d;	/* works for all */
	    } u;
	} r_esp, s_esp;

	struct esp_info {
	    u_int8_t transid;	/* negotiated ESP transform */
	    u_int16_t auth;	/* negotiated AUTH */

	    int alg;	/* KLIPS encoding for ESP x AUTH */
	    size_t ivlen;	/* length of IV [no longer used] */
	    size_t enckeylen;	/* keylength for ESP transform */
	    size_t authkeylen;	/* keylength for AUTH */
	};

	static const struct esp_info esp_info[] = {
	    { ESP_NULL, AUTH_ALGORITHM_HMAC_MD5,
		XF_ESPNULLMD596, 0, 0, AHMD596_KLEN },
	    { ESP_NULL, AUTH_ALGORITHM_HMAC_SHA1,
		XF_ESPNULLSHA196, 0, 0, AHSHA196_KLEN },

	    { ESP_DES, AUTH_ALGORITHM_NONE,
		XF_ESPDES, EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, 0 },
	    { ESP_DES, AUTH_ALGORITHM_HMAC_MD5,
		XF_ESPDESMD596, EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, AHMD596_KLEN },
	    { ESP_DES, AUTH_ALGORITHM_HMAC_SHA1,
		XF_ESPDESSHA196, EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, AHSHA196_KLEN },

	    { ESP_3DES, AUTH_ALGORITHM_NONE,
		XF_ESP3DES, EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, 0 },
	    { ESP_3DES, AUTH_ALGORITHM_HMAC_MD5,
		XF_ESP3DESMD596, EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, AHMD596_KLEN },
	    { ESP_3DES, AUTH_ALGORITHM_HMAC_SHA1,
		XF_ESP3DESSHA196, EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, AHSHA196_KLEN },
	};

	const struct esp_info *ei;

	memset(&r_esp, '\0', sizeof(r_esp));
	memset(&s_esp, '\0', sizeof(s_esp));

	r_esp.h = r_h;
	s_esp.h = s_h;

	r_esp.h.em_type = s_esp.h.em_type = EMT_SETSPI;

	r_esp.h.em_spi = st->st_esp.our_spi;
	s_esp.h.em_spi = st->st_esp.attrs.spi;
	r_esp.h.em_proto = s_esp.h.em_proto = IPPROTO_ESP;

	for (ei = esp_info; ; ei++)
	{
	    if (ei == &esp_info[elemsof(esp_info)])
	    {
		/* note: enum_show may use a static buffer, so two
		 * calls in one printf would be a mistake.
		 * enum_name does the same job, without a static buffer,
		 * assuming the name will be found.
		 */
		log("ESP transform %s / auth %s not implemented yet",
		    enum_name(&esp_transformid_names, st->st_esp.attrs.transid),
		    enum_name(&auth_alg_names, st->st_esp.attrs.auth));
		goto fail;
	    }

	    if (st->st_esp.attrs.transid == ei->transid
	    && st->st_esp.attrs.auth == ei->auth)
		break;
	}

	r_esp.h.em_alg = s_esp.h.em_alg = ei->alg;

	if (initiator)
	{
	    r_esp.u.d.eme_flags |= EME_INITIATOR;
	    s_esp.u.d.eme_flags |= EME_INITIATOR;
	}

	/* divide up keying material */
	passert(st->st_esp.keymat_len == ei->enckeylen + ei->authkeylen);
	passert(ei->enckeylen <= sizeof(r_esp.u.d.eme_key));
	passert(ei->authkeylen <= sizeof(r_esp.u.d.ame_key));

	memcpy(r_esp.u.d.eme_key, st->st_esp.our_keymat, ei->enckeylen);
	memcpy(s_esp.u.d.eme_key, st->st_esp.peer_keymat, ei->enckeylen);
	r_esp.u.d.eme_klen = s_esp.u.d.eme_klen = ei->enckeylen;

	memcpy(r_esp.u.d.ame_key, st->st_esp.our_keymat+ei->enckeylen, ei->authkeylen);
	memcpy(s_esp.u.d.ame_key, st->st_esp.peer_keymat+ei->enckeylen, ei->authkeylen);
	r_esp.u.d.ame_klen = s_esp.u.d.ame_klen = ei->authkeylen;

	/* window: always full size */
	r_esp.u.d.eme_ooowin = s_esp.u.d.eme_ooowin = 32;

	r_esp.h.em_msglen = s_esp.h.em_msglen =
	    EMT_SETSPI_FLEN + sizeof(struct espblkrply_edata);

	DBG_cond_dump(DBG_KLIPS, "setup inbound ESP transform => KLIPS:\n", &r_esp, r_esp.h.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &r_esp, r_esp.h.em_msglen) != r_esp.h.em_msglen)
	    {
		char buf[SATOA_BUF];

		log_errno((e, "write() of inbound ESP SPI %s failed in setup_ipsec_sa()"
		    , (satoa(r_esp.h.em_said, 0, buf, sizeof(buf)), buf)));
		goto fail;
	    }
	}
	*r_said_next++ = r_esp.h.em_said;

	DBG_cond_dump(DBG_KLIPS, "setup outbound ESP transform => KLIPS:\n", &s_esp, s_esp.h.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &s_esp, s_esp.h.em_msglen) != s_esp.h.em_msglen)
	    {
		char buf[SATOA_BUF];

		log_errno((e, "write() of outbound ESP SPI %s failed in setup_ipsec_sa()"
		    , (satoa(s_esp.h.em_said, 0, buf, sizeof(buf)), buf)));
		goto fail;
	    }
	}
	*s_said_next++ = s_esp.h.em_said;
    }

    if (st->st_ah.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL
    || st->st_esp.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL)
    {
	/*
	 * XXX hack alert -- we SHOULD NOT HAVE TO HAVE A DIFFERENT SPI
	 * XXX FOR IP-in-IP ENCAPSULATION!
	 */

	/* ipe4_xdata is from "ipsec_ipe4.h" */
	union {
	    struct encap_msghdr h;
	    struct {
		char dummy[offsetof(struct encap_msghdr, em_dat)];
		struct ipe4_xdata d;
	    } u;
	} m;	/* encapsulating message */

	/* Allocate an SPI for the outgoing tunnel.
	 * Since our peer will never see this,
	 * and it comes from its own number space,
	 * it is purely a local implementation wart.
	 */
	{
	    static ipsec_spi_t last_tunnel_spi = 0x100;

	    st->st_tunnel_spi = htonl(++last_tunnel_spi);
	}

	memset(&m, '\0', sizeof(m));
	m.h = s_h;

	m.h.em_type = EMT_SETSPI;
	m.h.em_spi = st->st_tunnel_spi;
	m.h.em_proto = IPPROTO_IPIP;

	m.h.em_msglen = EMT_SETSPI_FLEN + EMT_IPE4_ULEN;
	m.h.em_alg = XF_IP4;

	m.u.d.i4_src = c->this.host;
	m.u.d.i4_dst = c->that.host;

	DBG_cond_dump(DBG_KLIPS, "setup encapsulation => KLIPS:\n", &m, m.h.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &m, m.h.em_msglen) != m.h.em_msglen)
	    {
		char buf[SATOA_BUF];

		log_errno((e, "write() of encapsulation %s failed in setup_ipsec_sa()"
		    , (satoa(m.h.em_said, 0, buf, sizeof(buf)), buf)));
		goto fail;
	    }
	}
	*s_said_next++ = m.h.em_said;
    }

    if (s_said_next > &s_said[1])
    {
	/* Group the multiple sending SPIs
	 * All the info lives in the header!
	 * EM_MAXRELSPIS is max number of groupings.
	 */
	struct encap_msghdr m;
	int i;

	memset(&m, '\0', sizeof(m));

	m.em_magic = EM_MAGIC;
	m.em_version = 0;
	m.em_type = EMT_GRPSPIS;
	m.em_msglen = EMT_GRPSPIS_FLEN;

	/* reverse order of creation in above code:
	 * encap, esp, ah.
	 */
	for (i = 0; i != s_said_next - s_said; i++)
	{
	    m.em_rel[i].emr_said = s_said_next[ - 1 - i];
	    m.em_msglen += sizeof(m.em_rel[0]);

	    DBG(DBG_KLIPS,
		char buf[SATOA_BUF];

		DBG_log("grouping %s"
		    , (satoa(m.em_rel[i].emr_said, 0, buf, sizeof(buf)), buf)));
	}

	DBG_cond_dump(DBG_KLIPS, "setup grouping => KLIPS:\n", &m, m.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &m, m.em_msglen) != m.em_msglen)
	    {
		log_errno((e, "write() of grouping failed in setup_ipsec_sa()"));
		goto fail;
	    }
	}
	/* could update s_said, but it will not be used */
    }

    return TRUE;

fail:
    {
	/* undo the done SPIs */
	while (r_said_next-- != r_said)
	    (void) del_spi(fd, r_said_next->spi, r_said_next->proto
		, r_said_next->dst);

	while (s_said_next-- != s_said)
	    (void) del_spi(fd, s_said_next->spi, s_said_next->proto
		, s_said_next->dst);
	return FALSE;
    }
}

/* teardown_ipsec_sa is a canibalized version of setup_ipsec_sa */

static bool
teardown_ipsec_sa(int fd, struct state *st)
{
    /* We need to tear down both a Sending SA and a Receiving SA.
     * On the receiving side, only AH and ESP are possible (no encapsulation).
     * On the sending side, first of AH and ESP is fine:
     * since only one of a group needs to be deleted, either there
     * was no group, or the AH or ESP will be in it.
     */
    struct connection *c = st->st_connection;
    ipsec_spi_t *an_out_spi = NULL;
    int an_out_proto;

    /* delete receiving SPIs */

    if (st->st_ah.present)
    {
	(void) del_spi(fd, st->st_ah.our_spi, IPPROTO_AH, c->this.host);
	an_out_spi = &st->st_ah.attrs.spi;
	an_out_proto = IPPROTO_AH;
    }

    if (st->st_esp.present)
    {
	(void) del_spi(fd, st->st_esp.our_spi, IPPROTO_ESP, c->this.host);
	an_out_spi = &st->st_esp.attrs.spi;
	an_out_proto = IPPROTO_ESP;
    }

    /* delete some outgoing SPI (grouping gets rest) */

    (void) del_spi(fd, *an_out_spi, an_out_proto, c->that.host);
    return TRUE;
}

#endif /* KLIPS */

bool
install_ipsec_sa(struct state *st, bool initiator)
{
#ifdef KLIPS
    int fd = NULL_FD;
    bool res;
    struct connection
	*c = st->st_connection,
	*ero = route_owner(c, TRUE);	/* who, if anyone, owns our eroute? */

    if (ero != NULL && ero != c && ero->eroute_owner != SOS_NOBODY)
    {
	log("cannot install eroute -- it is in use for \"%s\"", ero->name);
	return FALSE;	/* another connection already using the eroute */
    }

    if (!route_connection(c))
	return FALSE;

    if (!no_klips)
    {
	fd = open("/dev/ipsec", O_WRONLY);
	if (fd < 0)
	{
	    log_errno((e, "open() of /dev/ipsec failed in install_ipsec_sa()"));
	    return FALSE;
	}
    }

    /* no other connection has this eroute, but
     * this connection might have it for another SA.
     * This is expected when rekeying.
     * If another SA has it, we replace the eroute.
     */
    res = setup_ipsec_sa(fd, st, initiator)
	&& (c->eroute_owner == SOS_NOBODY
	    ? do_eroute(fd, st, EMT_SETEROUTE, "set")
	    : do_eroute(fd, st, EMT_RPLACEROUTE, "replace"));

    if (res)
	c->eroute_owner = st->st_serialno;

    if (fd != NULL_FD)
	close(fd);

    return res;
#else
    DBG(DBG_CONTROL, DBG_log("install_ipsec_sa()"));
/* dev@fx.dk */
    return IIDLL_install_sa (st);
/*
    return route_connection(st->st_connection);
*/
#endif
}

bool
delete_ipsec_sa(struct state *st)
{
#ifdef KLIPS
    struct connection *c = st->st_connection;
    bool own_eroute = c->eroute_owner == st->st_serialno;
    int fd = NULL_FD;
    bool res = TRUE;

    if (!no_klips)
    {
	fd = open("/dev/ipsec", O_WRONLY);
	if (fd < 0)
	{
	    log_errno((e, "open() of /dev/ipsec failed in delete_ipsec_sa()"));
	    return FALSE;
	}
    }

    if (own_eroute)
    {
	res = do_eroute(fd, st, EMT_DELEROUTE, "delete");
	if (res)
	    c->eroute_owner = SOS_NOBODY;
    }
    if (res)
	res = teardown_ipsec_sa(fd, st);

    if (fd != NULL_FD)
	close(fd);	/* rain or shine */

    return res;
#else /* !KLIPS */
/*
    DBG(DBG_CONTROL, DBG_log("if I knew how, I'd do_eroute() and teardown_ipsec_sa()"));
*/
/* dev@fx.dk */
    if ((st == NULL) || (st->st_connection == NULL)) {
      DBG(DBG_CONTROL, DBG_log("can't find connection to delete"));
      return FALSE;
    }
    IIDLL_remove_sa (st);
    return TRUE;
#endif /* !KLIPS */
}
