/* whack communicating routines
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: kernel_comm.c,v 1.1 2000/11/18 00:30:21 bj Exp $
 */
/*
 * changed by dev@fx.dk (port to OS/2) 24/07/1999
 * changed by dev@fx.dk 09/10/99
 * changed by dev@fx.dk 29/10/1999 (new snapshot)
 */
#include "os2port.h"


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifndef __WIN32__
  #include <sys/un.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "connections.h"
#include "whack.h"	/* needs connections.h */
#include "packet.h"
#include "demux.h"	/* needs packet.h */
#include "state.h"
#include "ipsec_doi.h"	/* needs demux.h and state.h */
#include "kernel.h"
#include "kernel_comm.h"
#include "log.h"
#include "preshared.h"
#include "server.h"



int readbuf(int sock, int len2read, char *buf)
{
   int len_read = 0, bytes_read, max_read;
   char readdatabuf[600];

   /* Read until we have the length we need.. Returns 0 in case of failure */
   while (len_read < len2read)
   {
      max_read = MIN(len2read - len_read, 512);

      bytes_read = recv(sock, readdatabuf, max_read, 0);
      if (bytes_read > 0 && bytes_read <= 512)
      {
         memcpy(buf + len_read, readdatabuf, bytes_read);
         len_read += bytes_read;
      }
      else
         return -1;
   }

   return len_read;
}

/*
 * XXX Eventually it should use PF_KEY or what kernel API is available
 * XXX but for now this will have to do.
 */

/* Handle a kernel request. Supposedly, there's a message in
 * the kernelsock socket.
 */
void
whack_handle(int whackctlfd)
{
    struct whack_message msg;
#ifdef __WIN32__
    struct sockaddr whackaddr;
    int whackaddrlen = sizeof(whackaddr);
#else
    struct sockaddr_un whackaddr;
    int whackaddrlen = sizeof(whackaddr);
#endif
    int whackfd = accept(whackctlfd, (struct sockaddr *)&whackaddr, &whackaddrlen);
    ssize_t n;

    if (whackfd < 0)
    {
	log_errno((e, "accept() failed in whack_handle()"));
	return;
    }

// dev@fx.dk strange things do the linux guys ;)
//    n = read(whackfd, &msg, sizeof(msg));
    n = readbuf(whackfd, sizeof (msg), &msg);
//

    if (n == -1)
    {
	log_errno((e, "read() failed in whack_handle()"));
	close(whackfd);
	return;
    }

    whack_log_fd = whackfd;

    /* sanity check message */
    {
	int ughno = RC_BADWHACKMESSAGE;
	char ugh[200];

	if (n != sizeof(msg))
	{
	    snprintf(ugh, sizeof(ugh)
		, "truncated message from whack: got %d bytes, expected %d.  Message ignored."
		, n, (int) sizeof(msg));
	}
	else if (msg.magic != WHACK_MAGIC)
	{
//	    snprintf(ugh, sizeof(ugh)
//		, "message from whack has bad magic %d; should be %d; probably wrong version.  Message ignored"
//		, msg.magic, WHACK_MAGIC);
/* dev@fx.dk */
	    snprintf(ugh, sizeof(ugh)
		, "message from ipsec module has bad magic %d; should be %d; probably wrong version. Please consider software update"
		, msg.magic, WHACK_MAGIC);
	    log("%s", ugh);
	    ughno = 0;
//
	}
	else if (msg.name[CONNECTION_NAME_LIMIT-1] != '\0')
	{
	    snprintf(ugh, sizeof(ugh)
		, "message from whack has malformed name field (whack bug?)");
	}
	else
	{
	    ughno = 0;	/* ran through the gauntlet -- success */
	}

	if (ughno != 0)
	{
	    whack_log_fd = NULL_FD;	/* early so next log message is suppressed */
	    log("%s", ugh);
	    whack_log(whackfd, ughno, "%s", ugh);
	    close(whackfd);
	    return;
	}
    }
    if (msg.whack_options)
    {
#ifdef DEBUG
	debugging = msg.debugging;
	DBG(DBG_CONTROL,
	    DBG_log("debugging = %s", bitnamesof(debug_bit_names, debugging)));
#endif
    }

    /* Deleting combined with adding a connection works as replace.
     * To make this more useful, in only this combination,
     * delete will silently ignore the lack of the connection.
     */
    if (msg.whack_delete)
    {
	struct connection *c = con_by_name(msg.name
	    , msg.whack_connection? NULL_FD : whackfd);

	/* note: this is a "while" because road warrior
	 * leads to multiple connections with the same name.
	 */
	for (; c != NULL; c = con_by_name(msg.name, NULL_FD))
	    delete_connection(c);
    }

    if (msg.whack_connection)
	add_connection(&msg, whackfd);

    /* process "listen" before any operation that could require it */
    if (msg.whack_listen)
    {
	log("listening for IKE messages");
	listening = TRUE;
	find_ifaces(msg.left.end.host.s_addr);
	load_preshared_secrets();
/* aggr-patch */


	rw_replace_address (msg.left.end.host.s_addr); /* dev@fx.dk 06/08/99 */
//	rw_replace_address (msg.left.host.s_addr); /* dev@fx.dk 06/08/99 */

    }

    if (msg.whack_password)
      replace_preshared_secret (msg.left.end.host.s_addr, msg.right.end.host.s_addr, msg.name, strlen (msg.name));

    if (msg.whack_route)
    {
	if (!listening)
	    whack_log(whackfd, RC_DEAF, "need --listen before --route");
	else
	{
	    struct connection *c = con_by_name(msg.name, whackfd);

	    cur_connection = c;
	    if (c == NULL)
		;	/* already whack_logged */
	    else if (!orient(c, TRUE))
		whack_log(whackfd, RC_ORIENT, "could not orient connection");
	    else if (!route_connection(c))
		whack_log(whackfd, RC_ROUTE, "could not route");
	    cur_connection = NULL;
	}
    }

    if (msg.whack_unroute)
    {
	struct connection *c = con_by_name(msg.name, whackfd);

	if (c != NULL)
	{
	    cur_connection = c;
	    if (c->eroute_owner != SOS_NOBODY)
		whack_log(whackfd, RC_RTBUSY, "cannot unroute: route busy");
	    else
		unroute_connection(c);
	    cur_connection = NULL;
	}
    }

    if (msg.whack_initiate)
    {
	if (!listening)
	    whack_log(whackfd, RC_DEAF, "need --listen before --initiate");
	else
	    initiate_connection(msg.name, msg.whack_async? NULL_FD : whackfd);
    }

    if (msg.whack_terminate)
	terminate_connection(msg.name, whackfd);

    if (msg.whack_status)
    {
	show_connections_status(whackfd);
	whack_log(whackfd, RC_COMMENT, BLANK_FORMAT);	/* spacer */
	show_states_status(whackfd);
    }

    if (msg.whack_shutdown)
    {
	log("shutting down");
	exit_pluto(0);	/* delete lock and leave, with 0 status */
    }

    whack_log_fd = NULL_FD;
    if (whackfd != NULL_FD)
	close(whackfd);
}
