/*
 * $Id: ipsHookInterface.h,v 1.4 2003/12/08 23:31:37 hei Exp $
 * This unit defines standard type and function definitions used by the IPS
 * plug-in interface.
 *
 * Translated from pascal by Harald Eilertsen
 *
 * $Log: ipsHookInterface.h,v $
 * Revision 1.4  2003/12/08 23:31:37  hei
 * Redid structs and constants for ips v1.0.0 beta
 *
 * Revision 1.3  2003/10/26 21:57:12  hei
 * Added doxygen doc-comments.
 * Added direct-call callback api (implemented in callbacks.c)
 * Added ILC_MAIN_LOG and ILC_CONSOLE
 *
 * Revision 1.2  2003/10/06 22:00:39  hei
 * Fixed calling convention issues for Windows (Watcom and Microsoft compilers).
 * Changed IPSHOOKPROC struct so that member functions are callable directly.
 * Changed calling convention for callbacks to IPSCALL, use IPSEXPORT only for
 * functions to be exported.
 * Added documentational comments.
 *
 * Revision 1.1.1.1  2003/09/22 23:06:39  hei
 * Initial revision
 *
 */

#ifndef IPSHOOKINTERFACE_H
#define IPSHOOKINTERFACE_H

/**
 * @file ipsHookInterface.h
 *
 * This header describes the standard type and function definitions used by
 * the IPS plugin interface.
 *
 * With this header you will be able to write IPS hooks as a Windows or OS/2 
 * DLL in C. Theese will be loaded by IPS on the conditions specified in the IPS
 * documentation.
 *
 * $Revision: 1.4 $
 * @author  Harald Eilertsen (haraldei@anduin.net)
 */

#ifdef __cplusplus
extern "C" {
#endif

#if defined(__WATCOMC__)
#pragma pack(1)
#endif

/**
 * Define the chook interface version.
 */
#define IPS_CHOOK_VERSION          1

/**
 * Define the return type used by all IPS hook functions.
 * Valid values are TRUE and FALSE.
 */
typedef unsigned int IPSBOOL;             // 32 bit boolean type

/*
 * Define TRUE and FALSE if not already defined.
 */
#ifndef FALSE
#define FALSE  0              //!< Symbolic constant for a false value.
#endif
#ifndef TRUE
#define TRUE   !FALSE         //!< Symbolic value for a TRUE value.
#endif

/**
 * @def IPSCALL
 * Define IPS calling convention.
 * On OS/2 this is equivalent to __syscall, on Windows it is  equivalent to __stdcall.
 */
 
/**
 * @def IPSEXPORT
 * This is equivalent to IPSCALL but with the addition of the __export 
 * directive (Watcom) or __declspec(dllexport) (Microsoft). 
 */

#if defined (__WATCOMC__)                 // Windows and OS/2
   #if defined(__OS2__)
      #define IPSCALL __syscall 
   #elif defined(__NT__)
      #define IPSCALL __stdcall
   #endif
   #define IPSEXPORT IPSCALL __export
#elif defined(_MSC_VER)                   // Windows
   #define IPSCALL __stdcall
   #define IPSEXPORT __declspec(dllexport)
#else                                     // Just to make doxygen happy
   #define IPSCALL system calling convention
   #define IPSEXPORT exported entry
#endif

/*
 * Predeclare some types.
 */
typedef struct _IPSHOOKPROC	IPSHOOKPROC;     //!< Type declaration for plugin callback functions
typedef struct _IPSHOOKPROC	*PIPSHOOKPROC;   //!< Pointer to plugin callback functions

/**
 * Structure for data passed directly to the plug-in routine.
 */
struct _IPSHOOKDATA
{
	char	*arg;							//!< Address of 0-terminated textural string arguments
	char	*result;						//!< Address of 0-terminated textural string result
};

typedef struct _IPSHOOKDATA IPSHOOKDATA;     //!< Hook data type
typedef struct _IPSHOOKDATA *PIPSHOOKDATA;   //!> Pointer to hook data

/**
 * Structure for the standard record passed as the only argument to
 * most plug-in routines. A Pointer to this record is passed.
 */
struct _IPSHOOKINFO
{
   char                 *ipsVersionString;   //!< Full version string of running IPS.
   union {
      unsigned int      ipsVersionID;        //!< Version of IPS.
      struct part {
         unsigned char  subRevision;         //!< Least significant part of version number.
         unsigned char  revision;            //!< Revision part of version number.
         unsigned char  minorVersion;        //!< Minor version number.
         unsigned char  majorVersion;        //!< Major version number.
      };
   } ver;
	unsigned int	      hookVersion;	      //!< ipsCHookVersion, version id of plugin interface
	unsigned int	      sessionHandle;	      //!< Handle of current session which is to be used for callbacks
	PIPSHOOKPROC	      pproc;			      //!< Pointer to ipsTHookProc structure
	PIPSHOOKDATA	      pdata;			      //!< Pointer to ipsTHookData structure
};

typedef struct _IPSHOOKINFO	IPSHOOKINFO;     //!< IPS hook info type
typedef struct _IPSHOOKINFO	*PIPSHOOKINFO;   //!< Pointer to IPS hook info type

typedef IPSBOOL (IPSCALL *PFIPSGETVALUE)(PIPSHOOKINFO, int, char*, char*, int);
typedef IPSBOOL (IPSCALL *PFIPSSETVALUE)(PIPSHOOKINFO, int, char*, char*); 
typedef IPSBOOL (IPSCALL *PFIPSSEND)(PIPSHOOKINFO, void*, int);
typedef IPSBOOL (IPSCALL *PFIPSRECV)(PIPSHOOKINFO, void*, int);	// not implemented
typedef IPSBOOL (IPSCALL *PFIPSLOG)(PIPSHOOKINFO, unsigned, int, char*);


/**
 * Structure defining the available callback methods provided by IPS.
 * This is methods your plug-in routine may use during execution to interface
 * with the IPS server. All callback methods share a common first argument
 * which is the same HookInfo pointer passed to your hook routine.
 */
struct _IPSHOOKPROC
{
	PFIPSGETVALUE GetValue;      //!< Function to get value from IPS.
	PFIPSSETVALUE SetValue;      //!< Function to set value in IPS.
	PFIPSSEND	  Send;          //!< Function to send data to client.		
	PFIPSRECV     Recv;		     //!< Function to receive data from client (Not implemented).
	PFIPSLOG      Log;		     //!< Function to add messages to the log files.
};

//
// IPS callback API
//
IPSBOOL IPSCALL IpsGetValue(PIPSHOOKINFO pHookInfo, int type, char* ident, char* buf, int size);
IPSBOOL IPSCALL IpsSetValue(PIPSHOOKINFO pHookInfo, int type, char* ident, char* text);

/**
 * @addtogroup valuetypes Types of values
 *
 * These constants are used by the type parameter of the \c GetValue and \c SetValue
 * functions.
 */
/*@{*/
#define IVT_RESULTVALUE     0x00000000    //!< Used to set a simple Result value, no Ident required
#define IVT_VARIABLEVALUE   0x00000001    //!< Any internal variable, note that most are read only
#define IVT_CONFIGVALUE     0x00000002    //!< Not implemented
#define IVT_RESOURCEVALUE   0x00000003    //!< Not implemented
/*@}*/

IPSBOOL IPSCALL IpsSend(PIPSHOOKINFO pHookInfo, void* buf, int size);
IPSBOOL IPSCALL IpsRecv(PIPSHOOKINFO pHookInfo, void* buf, int size);
IPSBOOL IPSCALL IpsLog(PIPSHOOKINFO pHookInfo, unsigned code, int level, char* text);

/**
 * @addtogroup loggingcodes Logging codes
 *
 * These constants are used by the code parameter of the Log function.
 * Unless specifically noted, messages can only be logged from a hook with
 * a valid session. At the time when IPS calls \c ipsInitialize or 
 * \c ipsFinalize no session exists. In this situatin, only \c ILC_MAIN_LOG 
 * and \c ILC_CONSOLE are valid codes.
 */
/*@{*/
// Constants for aCode in Log callback
#define ILC_DEFAULT         0xf0000000    //!< Log to default log for the current session (Requires an active session)
#define ILC_SYSTEM_LOG      0xf1000000    //!< Log to main log file (Also works from ipsInitialize and ipsFinalize without a session)
#define ILC_SYSTEM_CONSOLE  0xf2000000    //!< Log to system console (Also works from ipsInitialize and ipsFinalize without a session)
#define ILC_SYSTEM_LOG_CONSOLE   \
      ILC_SYSTEM_LOG|ILC_SYSTEM_CONSOLE   //!< Log to main log file and system console (Also works from ipsInitialize and ipsFinalize without a session)
/*@}*/

#if defined(__WATCOMC__)
#pragma pack(pop)
#endif

#ifdef __cplusplus
}		// extern "C"
#endif


#endif  // IPSHOOKINTERFACE_H
