/******************/
/* Procedure to read a multipart/form-data type of submission.
   This should be included in the REXX program that calls it --
   classic REXX can NOT use it as an external program 

This will read the content uplodaed by a web browser that is responding 
to a form with an action of POST and an enctype of "multipart/form-data"

syntax:
   astatus=parse_multipart(post_body)

where post_body is the body of the POST response (from the browser)

Data is returned in the OPTNAME. stem variable (which is exposed)

  optname.0 =space delimited  list of variable names
  optname.!varname = contents of the "varname" variable (varname is one of the words
                     stored in varname.0)
  optname.!varname.!filename = if the contents are due to submission of a file,
                               this contains the name of the file (as supplied by
                               the browser). Otherwise, it is blank
  optname.!varname.!CONTENT_TYPE = if the contents are due to submission of a file,
                                   this contains the content-type. If no content-type
                                   was specified, this will be text/plain.
                                   If NOT due to a file submission, this is blank.

Astatus contains status info:
  0 = Success
  1 = Not a multipart/form-data 
  2 = No multipart/form-data in content-type 
  3 = No boundary 
  4 = No data recieved 


*/
parse_multipart:procedure expose  optname.
parse arg abody

crlf='0d0a'x
rept=''

optname.=''

/*get content type request header */
atype=sre_reqfield('content-type')

/*typical value of atype
      multipart/form-data; boundary=---------------------------309151678928465
*/

/* is there a content-type request header ? */
if atype="" then do
    return 1
end

parse var atype thetype ";" boog 'boundary=' abound0    /* get the type */

abound="--"||strip(abound0)  /* since boundaries always start with -- */

if strip(translate(thetype))<>"MULTIPART/FORM-DATA" then do
   return 2
end

if abound="" then do
   return 3
end

/* basic status okay. So pull out stuff */

abd1=crlf||abound
abd2=abound

/* A multi-part submission typically has the following structure:

   Content-Type: multipart/form-data; boundary=AaB03x

   --AaB03x
   Content-Disposition: form-data; name="personal_name"

   Larry
   --AaB03x
   Content-Disposition: form-data; name="myfile"; filename="file1.txt"
   Content-Type: text/plain

   ... contents of file1.txt ...
   --AaB03x--

  Thus, processing works by:
   a) pull out a block, where blocks are defined by "boundary strings"
   b) for each block, pull out the headers, where the headers end at the first
      blank line
   c) the contents is everything remaining in the block, after this empty line
   
  There are two important headers, the content-disposition and content-type.
  If the content-disposition contains a  filename="xx" field, then this block
  contains a FILE upload. Otherwise, this block contains the value of a regular form
  field (such as an <INPUT type="TEXT" var="personal_name"> element)

  The content-type is sometimes also sent by the browser.
 
*/

parse var abody foo1 (abd2) abody    /* move beyond first boundary and it's crlf */

/* check for netscape 2.0 incorrect format */
if pos(abound,abody)=0 then do   /* no ending boundary, so add one */
   abody=abody||crlf||abound||" -- "
end
mm=0

do until abody=""
  if mm>0 then do
    if abbrev(abody,'--')=1 then leave        /* -- signals no more */
  end

  parse var abody . (crlf) thestuff (abd1) abody        /* get a  boundary defined block */
  mm=mm+1

  varname=''
  filename=''
  ctype=''

/* we have a "block", as delimited by the boundary string. */

  do forever            /* get block headers.  Stop when hit a blank line */
     parse var thestuff anarg (crlf) thestuff

     if anarg="" then do                /* empty line means "end of headers */
           leave
     end
     else do                    /* extract the arguments on this line */
         parse var anarg ahd ':' anarg
         ahd=strip(translate(ahd))
         select
            when ahd='CONTENT-TYPE' then do
                ctype=strip(anarg)
            end 
            otherwise do                /* strip out arguments */
               do until anarg=''
                  parse var anarg anarg1 ';' anarg
                  if pos('=',anarg1)='' then iterate
                  parse var anarg1 cvar '=' cval
                  cval2=strip(strip(cval),,'"')
                  cvar=strip(translate(cvar))
                  select               
                    when cvar="NAME" then varname=strip(translate(cval2))          /* name of this block */
                    when cvar="FILENAME" then filename=strip(translate(cval2))   /* original file name */
                    when cvar="CONTENT-TYPE" then strip(cval2)        /* some browsers might send this here */
                    otherwise nop               /* ignore other arguments */
                  end           /* select */
               end     /* extract arguments */
            end        /* otherwise */
         end            /* select */
     end                /* else do*/
   end                    /* get a line */

/* when here, got a block's headers. The rest is the content of the block (i.e.;
   the value of a variable */
  if varname='' then iterate            /* no name, so ignore */
  vv=strip(translate(varname))
  optname.0=optname.0||' '||vv
  vv1='!'||vv
  optname.vv1=thestuff
  optname.vv1.!filename=filename
  if filename<>'' & ctype='' then ctype='text/plain'
  if filename<>'' then optname.vv1.!content_type=ctype

end             /* back to top to get the next block */


if mm=0 then do
     return 4
end


return 0

