//by Jose Da Silva, Vancouver, BC, Canada, (C) Nov 19 2001
//
//Hopefully some of the concepts are clearer here for programmers wishing to
//use joysticks in their OS/2 programs. Use this program if you like.
//
//This code should provide the most compatibility for programmers wanting to
//insert joystick functionality but not knowing what standard(s) to follow.
//This should run on IBM's JOYSTICK.ZIP(1995), or Vesa Jskelinen's
//GPP_DRIV.SYS(1999) or Jose Da Silva's GAMEDD2.ZIP(2001).
//
//The advantages of using this joystick driver is that the programmer does
//not need to figure out where a joystick is located because now adays the
//joysticks might be located at a PCI address or some other address<>0201h.
//Joysticks may be digital or other in nature, but to the programmer, this
//driver should look like the basic analog joystick port which contains
//2 analog joysticks with 2 buttons each, or one joystick with 2 or 4 buttons.
//

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSMEMMGR
#include <os2.h>
#include "joyos2.h"

BOOL FindJoystickRange(HFILE hGame, GAME_CALIB_STRUCT *stRange, GAME_PARM_STRUCT *stParms) {
//
//Initalize joystick ranges for program use by asking the user to roll the
//joystick around to find joystick min and max values available.
//Then calculate all setpoints needed for your program.
//Return TRUE if everything went okay, else return FALSE on error.
//
	APIRET			rc;
	BOOL			done;		//everything done okay=TRUE
	ULONG			ulDataLen;
	GAME_STATUS_STRUCT	stStatus;	//used for getting joysticks

	done = TRUE;
	printf("     Please roll joystick(s) around to catch min-max values.\n");
	printf("     Then press a joystick button when done.\n");
	//
	//The joysticks have an unknown range of values when a program starts.
	//We need to figure out this range. We will do this test on-the-fly for
	//this example program. As far as we know, joysticks will go from 0 to
	//a positive value. Note the variables are defined as SHORT and not as
	//USHORT, so maybe one day in the future there will be joysticks that
	//also go in the negative direction. This driver only goes positive
	//and it is best if programmers assume that joysticks have a positve
	//range since that is what DOS programs used in the past.
	//
	printf("     Start at zero.");
	stRange->Ax.lower = 32767;	//this is max value for a SHORT
	stRange->Ax.centre = 0;		//don't go below 0
	stRange->Ax.upper = 0;		//don't go below 0
	stRange->Ay.lower = 32767;	//defined as SHORT
	stRange->Ay.centre = 0;		//don't go below 0
	stRange->Ay.upper = 0;		//don't go below 0
	stRange->Bx.lower = 32767;	//defined as SHORT
	stRange->Bx.centre = 0;		//don't go below 0
	stRange->Bx.upper = 0;		//don't go below 0
	stRange->By.lower = 32767;	//defined as SHORT
	stRange->By.centre = 0;		//don't go below 0
	stRange->By.upper = 0;		//don't go below 0

	//
	//DOS and this OS/2 driver recognize 0xF0 as no buttons pressed yet
	//
	stStatus.curdata.butMask = 0xF0; //initialize loop
	printf(" Reading joystick(s) until button pressed.\n");
	while (done && ((stStatus.curdata.butMask & 0xF0) == 0x0F0)) {
		//
		//get current joystick values
		//
		ulDataLen = sizeof(stStatus);
		rc = DosDevIOCtl(hGame, IOCTL_CAT_USER, GAME_GET_STATUS,
			NULL, 0, NULL, &stStatus, ulDataLen, &ulDataLen);
		if (rc != 0) {
			printf("ERROR %d Could not get GAME$ joystick positions.\n", rc);
			done = FALSE;
		}
		printf("\x0D    ");
		//
		//show position info for joystick A
		//
		if (stParms->useA) {
			printf(" Axy: %x %x", stStatus.curdata.A.x, stStatus.curdata.A.y);

			//
			//calculate joystickA range while in this loop
			//
			if (stStatus.curdata.A.x < stRange->Ax.lower)
				stRange->Ax.lower = stStatus.curdata.A.x;
			if (stStatus.curdata.A.x > stRange->Ax.upper)
				stRange->Ax.upper = stStatus.curdata.A.x;
			if (stStatus.curdata.A.y < stRange->Ay.lower)
				stRange->Ay.lower = stStatus.curdata.A.y;
			if (stStatus.curdata.A.y > stRange->Ay.upper)
				stRange->Ay.upper = stStatus.curdata.A.y;
			//
			//note possible error if (lower+upper>32767)
			//
			stRange->Ax.centre = (stRange->Ax.lower + stRange->Ax.upper) >> 1;
			stRange->Ay.centre = (stRange->Ay.lower + stRange->Ay.upper) >> 1;

			//
			//demonstrate linear location from 0..100%
			//
			if (stRange->Ax.upper > stRange->Ax.lower)
				printf(" %d%%",((stStatus.curdata.A.x - stRange->Ax.lower) * 100)/(stRange->Ax.upper - stRange->Ax.lower));
			if (stRange->Ay.upper > stRange->Ay.lower)
				printf(" %d%%",((stStatus.curdata.A.y - stRange->Ay.lower) * 100)/(stRange->Ay.upper - stRange->Ay.lower));
		}
		//
		//show position info for joystick B
		//
		if (stParms->useB) {
			printf(" Bxy: %x %x", stStatus.curdata.B.x, stStatus.curdata.B.y);

			//
			//calculate joystickB range while in this loop
			//
			if (stStatus.curdata.B.x < stRange->Bx.lower)
				stRange->Bx.lower = stStatus.curdata.B.x;
			if (stStatus.curdata.B.x > stRange->Bx.upper)
				stRange->Bx.upper = stStatus.curdata.B.x;
			if (stStatus.curdata.B.y < stRange->By.lower)
				stRange->By.lower = stStatus.curdata.B.y;
			if (stStatus.curdata.B.y > stRange->By.upper)
				stRange->By.upper = stStatus.curdata.B.y;
			//
			//note possible error if (lower+upper>32767)
			//
			stRange->Bx.centre = (stRange->Bx.lower + stRange->Bx.upper) >> 1;
			stRange->By.centre = (stRange->By.lower + stRange->By.upper) >> 1;

			//
			//demonstrate linear location from 0..100%
			//
			if (stRange->Bx.upper > stRange->Bx.lower)
				printf(" %d%%",((stStatus.curdata.B.x - stRange->Bx.lower) * 100)/(stRange->Bx.upper - stRange->Bx.lower));
			if (stRange->By.upper > stRange->By.lower)
				printf(" %d%%",((stStatus.curdata.B.y - stRange->By.lower) * 100)/(stRange->By.upper - stRange->By.lower));
		}
		printf("   ");	//clean-up rest of line
	}
	printf("\n");

	return done;	//TRUE=okay, FALSE=error
}

int main (int argc, char *argv[]) {
	APIRET			rc;
	HFILE			hGame;
	ULONG			ulAction, ulVersion, ulDataLen;
	GAME_PARM_STRUCT	stGameParms;
	GAME_CALIB_STRUCT	stGameCalib, stGameRange;
	GAME_STATUS_STRUCT	stGameStatus;

	printf("GAMEDD.SYS joystick programming example.\n");

	printf("  1) Open a handle to GAME$ device.\n");
	rc = DosOpen((PSZ)GAMEPDDNAME, &hGame, &ulAction, 0, FILE_READONLY,
		FILE_OPEN, OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE, NULL);
	if (rc != 0) {
		printf("ERROR %d Could not open GAME$ device.\n", rc);
		exit(-1);
	}

	//
	//To be usable for as many users as possible, the lowest version
	//you should look for is 2.0 (as far as I know). If a program looks
	//for 2.0 or higher, then your program should be able to run on the
	//largest available group of users because your program can run on
	//IBM's joystick driver, Vesa's joystick driver and this driver too.
	//
	printf("  2) Get GAME$ version, must be 2.0 or higher.");
	ulVersion = 0;
	ulDataLen = sizeof(ulVersion);
	rc = DosDevIOCtl( hGame, IOCTL_CAT_USER, GAME_GET_VERSION,
		NULL, 0, NULL, &ulVersion, ulDataLen, &ulDataLen);
	if (rc != 0) {
		printf("\nERROR %d Could not get GAME$ version\n", rc);
		DosClose(hGame);
		exit(-1);
	}
	printf(" Found version %x.\n", ulVersion);
	if (ulVersion < GAME_VERSION) {
		printf("ERROR, GAME$ driver needs updating to %x or higher.\n", GAME_VERSION);
		DosClose(hGame);
		exit(-1);
	}

	printf("  3) Find out if we have joystickA and/or joystickB.\n");
	ulDataLen = sizeof(stGameParms);
	rc = DosDevIOCtl(hGame, IOCTL_CAT_USER, GAME_GET_PARMS,
		NULL, 0, NULL, &stGameParms, ulDataLen, &ulDataLen);
	if (rc != 0) {
		printf("ERROR %d Could not get GAME$ parameters.\n", rc);
		DosClose(hGame);
		exit(-1);
	}
	if ((stGameParms.useA == 0) && (stGameParms.useB == 0)) {
		printf("No joysticks detected.\n");
		DosClose(hGame);
		exit(0);
	}

	//
	//This step allows you to find out what the joystick ranges are for
	//your program since you need to find them out. Analog joysticks can
	//appear to drift a bit due to your computer warming up from initial
	//power up. It would be recommended that you play music or sounds at
	//this time since some older computer soundcards will affect the
	//results of analog joysticks if the souncard is running or not.
	//Digital joysticks tend to be self-calibrating and immune to
	//soundcard power fluctuations but not everybody has a digital
	//joystick or high quality soundcard.
	//Another detail that should be mentioned is that this may be an
	//excellent way to get linear values for analog joysticks, but it
	//may give poor results for gamepad style joysticks because they may
	//read maximum most of the time but not all the time therefore giving
	//somewhat "jumpy" movement. To make your program useful for people
	//with only gamepads, it is recommended that you allow movement if it
	//reads left<-0.70*centerX, centerX*1.30->right, up<-0.70*centerY,
	//centerY*1.30->down.
	//Best for the programmer to be aware of this, that is why these
	//details are mentioned here.
	//
	printf("  4) Initialize Min-Max joystick ranges.\n");
	if (!FindJoystickRange(hGame, &stGameRange, &stGameParms)) {
		printf("  Error on initializing ranges.\n");
		DosClose(hGame);
		exit(-1);
	}

	//
	//This step is not needed, but is shown so you know what it is.
	//Maybe you might want to make use of it. It is up to you.
	//The advantage is that you do not need to ask the user to roll
	//their joystick around for program calibration. It is only useful
	//for programs that make use of left|center|right and up/stop/back
	//movements. If the user has a gamepad style joystick, this is
	//probably the best selection for program movement.
	//These values are collected when GAME$ first probe the joysticks
	//on power-up. It is created by taking the centre value * 0.70 for
	//the lower trip-point and * 1.30 for the upper trip-point.
	//If you create a program that simply has go/stop action, this may
	//be all the range information you need because the lower and upper
	//trip points are already figured out, therefore you can skip step 4.
	//The joysticks are expected to be in a "centered" position for this
	//data to be useful. This data is used by GAMEVDD.SYS for DOS games
	//if you like to have a gamepad style response. We could probably
	//use it here if we assume that the joysticks were in the centre
	//position on computer power up.
	//There is a program called JOYTUNE which you can use to tune these
	//numbers for the IBM joystick driver. GAMEDD2.ZIP can be tuned with
	//JOYTUNE also, or it can be setup on powerup with the /C option.
	//
	printf("  5) Collect calibration values from GAME$.\n");
	ulDataLen = sizeof(stGameCalib);
	rc = DosDevIOCtl(hGame, IOCTL_CAT_USER, GAME_GET_CALIB,
		NULL, 0, NULL, &stGameCalib, ulDataLen, &ulDataLen);
	if (rc != 0) {
		printf("ERROR %d Could not get GAME$ calibrations.\n", rc);
		DosClose(hGame);
		exit(-1);
	}
	if (stGameParms.useA != 0) {
		printf("     Joystick Ax approximate center: %d\n", stGameCalib.Ax.centre);
		printf("     Joystick Ax left trip point   : %d\n", stGameCalib.Ax.lower);
		printf("     Joystick Ax right trip point  : %d\n", stGameCalib.Ax.upper);
		printf("     Joystick Ay approximate center: %d\n", stGameCalib.Ay.centre);
		printf("     Joystick Ay top trip point    : %d\n", stGameCalib.Ay.lower);
		printf("     Joystick Ay bottom trip point : %d\n", stGameCalib.Ay.upper);
	}
	if (stGameParms.useB != 0) {
		printf("     Joystick Bx approximate center: %d\n", stGameCalib.Bx.centre);
		printf("     Joystick Bx left trip point   : %d\n", stGameCalib.Bx.lower);
		printf("     Joystick Bx right trip point  : %d\n", stGameCalib.Bx.upper);
		printf("     Joystick By approximate center: %d\n", stGameCalib.By.centre);
		printf("     Joystick By top trip point    : %d\n", stGameCalib.By.lower);
		printf("     Joystick By bottom trip point : %d\n", stGameCalib.By.upper);
	}

	printf("  6) Current Joystick Response Settings, press a key to quit\n");
	while(!kbhit()) {
		ulDataLen = sizeof(stGameStatus);
		rc = DosDevIOCtl(hGame, IOCTL_CAT_USER, GAME_GET_STATUS,
			NULL, 0, NULL, &stGameStatus, ulDataLen, &ulDataLen);
		if (rc != 0) {
			printf("ERROR %d Could not get GAME$ joystick positions.\n", rc);
			DosClose(hGame);
			exit(-1);
		}

		printf("\x0D");
		//
		//show position info for joystick A
		//
		if (stGameParms.useA) {
			printf("Axy: %x %x", stGameStatus.curdata.A.x, stGameStatus.curdata.A.y);

			//
			//demonstrate 100% range for analog control (collected from step #4)
			//
			if (stGameRange.Ax.upper > stGameRange.Ax.lower)
				printf(" %d%%",((stGameStatus.curdata.A.x - stGameRange.Ax.lower) * 100)/(stGameRange.Ax.upper - stGameRange.Ax.lower));
			if (stGameRange.Ay.upper > stGameRange.Ay.lower)
				printf(" %d%% ",((stGameStatus.curdata.A.y - stGameRange.Ay.lower) * 100)/(stGameRange.Ay.upper - stGameRange.Ay.lower));

			//
			//demonstrate digital gamepad type control (collected from step #5)
			//
			if (stGameStatus.curdata.A.x < stGameCalib.Ax.lower)
				printf("<");
			else {
				if (stGameStatus.curdata.A.x > stGameCalib.Ax.upper)
					printf(">");
				else
					printf("c");
			}
			if (stGameStatus.curdata.A.y < stGameCalib.Ay.lower)
				printf("^");
			else {
				if (stGameStatus.curdata.A.y > stGameCalib.Ay.upper)
					printf("v");
				else
					printf("c");
			}
		}

		//
		//show position info for joystick B
		//
		if (stGameParms.useB) {
			printf(" Bxy: %x %x", stGameStatus.curdata.B.x, stGameStatus.curdata.B.y);

			//
			//demonstrate 100% range for analog control (collected from step #4)
			//
			if (stGameRange.Bx.upper > stGameRange.Bx.lower)
				printf(" %d%%",((stGameStatus.curdata.B.x - stGameRange.Bx.lower) * 100)/(stGameRange.Bx.upper - stGameRange.Bx.lower));
			if (stGameRange.By.upper > stGameRange.By.lower)
				printf(" %d%% ",((stGameStatus.curdata.B.y - stGameRange.By.lower) * 100)/(stGameRange.By.upper - stGameRange.By.lower));

			//
			//demonstrate digital gamepad type control (collected from step #5)
			//
			if (stGameStatus.curdata.B.x < stGameCalib.Bx.lower)
				printf("<");
			else {
				if (stGameStatus.curdata.B.x > stGameCalib.Bx.upper)
					printf(">");
				else
					printf("c");
			}
			if (stGameStatus.curdata.B.y < stGameCalib.By.lower)
				printf("^");
			else {
				if (stGameStatus.curdata.B.y > stGameCalib.By.upper)
					printf("v");
				else
					printf("c");
			}
		}

		printf(" buttons: %x    ", stGameStatus.curdata.butMask);
	}
	printf("\n");

	printf("  7) Close handle to GAME$. Done.\n");
	DosClose(hGame);
}
