/*****************************************************************************
 * $Id: ak_zip.c,v 2.2 1996/06/06 20:02:59 ak Exp $
 *****************************************************************************
 * $Log: ak_zip.c,v $
 * Revision 2.2  1996/06/06 20:02:59  ak
 * Separate enablers for compression engines.
 * Better compression error checks and messages.
 * ZLIB disabled by default - unstable.
 *
 * Revision 2.1  1996/06/06 19:17:54  ak
 * Compression wrapper.
 *
 *****************************************************************************/

static char *rcsid = "$Id: ak_zip.c,v 2.2 1996/06/06 20:02:59 ak Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <io.h>

#ifdef FZIP_AK
#include <libx.h>
#endif

extern int		f_compress_files;

/*****************************************************************************/
#ifdef FZIP_ZLIB

#include <zlib.h>

#define Chunk		(16 * 1024L)
#define Magic           "ZLIBa"

static z_stream		zs;
static Byte *		zbuf;

static voidpf
zalloc(voidpf opaque, uInt items, uInt size)
{
    voidpf p = malloc(items * size);
    return p ? p : Z_NULL;
}

static void
zfree(voidpf opaque, voidpf p)
{
    free(p);
}

static int
zinit(void)
{
    if (zbuf == NULL) {
	memset(&zs, 0, sizeof zs);
	zs.zalloc = zalloc;
	zs.zfree  = zfree;
	zs.opaque = NULL;

	zbuf = malloc(Chunk);
	if (zbuf == NULL) {
	    errno = ENOMEM;
	    return -1;
	}
    }
    return 0;
}

static int
z_zip_from_file (int fd, char *dst, int dlen)
{
    int nr, n, rc;

    if (zinit())
	return -1;

    rc = deflateInit(&zs, f_compress_files - 1);
    switch (rc) {
    case Z_OK:
	break;
    case Z_MEM_ERROR:
	errno = ENOMEM;
	return -1;
    default:
	errno = EINVAL;
	return -1;
    }

    zs.total_in  = 0;

    n = strlen(Magic) + 1;
    memcpy(dst, Magic, n);
    zs.next_out  = dst + n;
    zs.avail_out = dlen - n;
    zs.total_out = n;

    for (;;) {
	for (nr = 0; (n = read(fd, zbuf+nr, Chunk-nr)) > 0; nr += n)
	    ;
	if (n < 0)
	    break;
	if (nr == 0)
	    break;

	if (zs.avail_out == 0) {
	    deflateEnd(&zs);
	    return 0;
	}

	zs.next_in  = zbuf;
	zs.avail_in = nr;
	zs.total_in += nr;

	rc = deflate(&zs, Z_NO_FLUSH);
	switch (rc) {
	case Z_OK:
	case Z_STREAM_END:
	    break;
	default:
	    deflateEnd(&zs);
	    errno = EINVAL;
	    return -1;
	}
    }

    if (n < 0) {
	deflateEnd(&zs);
	return -1;
    }

    do {
	n = deflate(&zs, Z_FINISH);
    } while (n == Z_OK && zs.avail_out > 0);
    switch (n) {
    case Z_STREAM_END:
	break;
    case Z_OK:
	deflateEnd(&zs);
	return 0;
    default:
	deflateEnd(&zs);
	errno = EINVAL;
	return -1;
    }

    deflateEnd(&zs);
    return zs.total_out;
}

static int
z_unzip_to_file (int fd, void *src, int slen)
{
    int nw, na, n, rc;

    n = strlen(Magic) + 1;
    if (memcmp(src, Magic, n) != 0) {
	inflateEnd(&zs);
	errno = EINVAL;
	return -2;
    }

    if (zinit())
	return -1;

    rc = inflateInit(&zs);
    switch (rc) {
    case Z_OK:
	break;
    case Z_MEM_ERROR:
	errno = ENOMEM;
	return -1;
    default:
	errno = EINVAL;
	return -1;
    }

    zs.next_in  = src + n;
    zs.avail_in = slen - n;
    zs.total_in = slen;

    zs.total_out = 0;

    for (;;) {
	zs.next_out  = zbuf;
	zs.avail_out = Chunk;

	rc = inflate(&zs, Z_PARTIAL_FLUSH);

	na = Chunk - zs.avail_out;
	if (na) {
	    for (nw = 0; nw < na && (n = write(fd, zbuf+nw, na-nw)) > 0; nw += n)
		;
	    if (nw != na) {
		inflateEnd(&zs);
		return -1;
	    }
	}

	switch (rc) {
	case Z_OK:
	    break;
	case Z_STREAM_END:
	    inflateEnd(&zs);
	    return zs.total_out;
	case Z_MEM_ERROR:
	    inflateEnd(&zs);
	    errno = ENOMEM;
	    return -1;
	default:
	    inflateEnd(&zs);
	    errno = EINVAL;
	    return -1;
	}
    }
}

int
z_unzip_compare (int fd, void *src, int slen)
{
    int nr, na, n, rc;
    char c;

    n = strlen(Magic) + 1;
    if (memcmp(src, Magic, n) != 0) {
	inflateEnd(&zs);
	errno = EINVAL;
	return -2;
    }

    if (zinit())
	return -1;

    rc = inflateInit(&zs);
    switch (rc) {
    case Z_OK:
	break;
    case Z_MEM_ERROR:
	errno = ENOMEM;
	return -1;
    default:
	errno = EINVAL;
	return -1;
    }

    zs.next_in  = src + n;
    zs.avail_in = slen - n;
    zs.total_in = slen;

    zs.total_out = 0;

    for (;;) {
	zs.next_out  = zbuf;
	zs.avail_out = Chunk/2;

	rc = inflate(&zs, Z_PARTIAL_FLUSH);

	na = Chunk - zs.avail_out;
	if (na) {
	    for (nr = 0; nr < na && (n = read(fd, zbuf+Chunk/2+nr, na-nr)) > 0; nr += n)
		;
	    if (nr != na) {
		inflateEnd(&zs);
		return 0;
	    }
	    if (memcmp(zbuf, zbuf+Chunk/2, na) != 0) {
		inflateEnd(&zs);
		return 0;
	    }
	}

	switch (rc) {
	case Z_OK:
	    break;
	case Z_STREAM_END:
	    inflateEnd(&zs);
	    return (read(fd, &c, 1) == 1) ? 0 : zs.total_out;
	case Z_MEM_ERROR:
	    inflateEnd(&zs);
	    errno = ENOMEM;
	    return -1;
	default:
	    inflateEnd(&zs);
	    errno = EINVAL;
	    return -1;
	}
    }
}

#endif
/*****************************************************************************/

int
zip_from_file (int fd, char *dst, int dlen)
{
#ifdef FZIP_AK
    if (f_compress_files == 1)
	return compress_file(fd, dst, dlen);
#else
    if (f_compress_files == 1)
	f_compress_files = 2;
#endif

#ifdef FZIP_ZLIB
    if (f_compress_files >= 2)
	return z_zip_from_file(fd, dst, dlen);
#endif

    errno = ENXIO;
    return -2;
}

int
unzip_to_file (int fd, void *src, int slen)
{
    int rc;

#ifdef FZIP_AK
    rc = decompress_file(fd, src, slen);
    if (rc >= -1)
	return rc;
#endif

#ifdef FZIP_ZLIB
    rc = z_unzip_to_file(fd, src, slen);
    if (rc >= -1)
	return rc;
#endif

    errno = ENXIO;
    return -2;
}

int
unzip_compare (int fd, void *src, int slen)
{
    int rc;

#ifdef FZIP_AK
    rc = decompress_compare(fd, src, slen);
    if (rc >= -1)
	return rc;
#endif

#ifdef FZIP_ZLIB
    rc = z_unzip_compare(fd, src, slen);
    if (rc >= -1)
	return rc;
#endif

    errno = ENXIO;
    return -2;
}
