/* $Id: renamepics.c,v 2.4 1999-09-05 17:00:05-04 rl Exp $ */


/*************************************************************************
 *                                                                       *
 * renamepics.c                                                          *
 * Rename Olympus photo files and store exposure infos as EAs            *
 *                                                                       *
 * Compile with emx!                                                     *
 *                                                                       *
 *************************************************************************/


#include <errno.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/ea.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "denise.h"
#include "exif.h"
#include "renamepics.h"


/*************************************************************************
 *                                                                       *
 * main()                                                                *
 * Main procedure                                                        *
 *                                                                       *
 *************************************************************************/
int main ( int argc, char* argv[] )
  {

  int i, k;
  int rc;
  int error;
  FILE *file;
  struct stat filebuf;
  int filesize;
  char* start;
  char* end;
  char* p;
  char filename [FILENAME_MAX+1];
  TfileType type = autodetect;
  int firstfile = 1;
  int dump = NO;

  error = NO;

  if( 1 == argc )
    {
    help();
    return RET_HELP;
    }
  for( i = 1; i < argc; i += 1 )
    if( (EQUAL == stricmp("/d",argv[i])) || (EQUAL == stricmp("-d",argv[i])) )
      {
      dump = YES;
      firstfile = i + 1;
      }
    else if( (EQUAL == stricmp("/h",argv[i])) || (EQUAL == stricmp("-h",argv[i])) )
      {
      help();
      return RET_HELP;
      }

  /* Expand wildcards in command line arguments */
  _wildcard( &argc, &argv );

  /* Iterate through all command line arguments (assumed to be file names) */
  for( i = firstfile; i < argc; i += 1 )
    {

    /* Rename the variable that holds the file name */
    strcpy( filename, argv[i] );

    /* Open file for reading */
    file = fopen( filename, "rb" );
    if( NULL == file ) 
      {
      printf( "*Error* Unable to open file %s\n", filename );
      error = YES;
      continue;
      }

    /* Get file size */
    fstat( fileno(file), &filebuf );

    /* Rename the variable that holds the file size */
    filesize = filebuf.st_size;

    /* Allocate memory for file */
    start = (char*) malloc( filesize );

    /* Load header of file */
    if( filesize < HEADER_LEN )
      {
      printf( "*Error* File %s too short to determine format\n", filename );
      error = YES;
      fclose( file );
      free( start );
      continue;
      }
    p = start;
    for( k = 0; k < HEADER_LEN; k += 1 ) 
      {
      *p = fgetc( file );
      p += 1;
      }

    /* Check format and call appropriate function to process file */
    if( isExif(start) )
      {

      /* Load remaining data */
      for( k = HEADER_LEN; k < filesize; k += 1 ) 
        {
        *p = fgetc( file );
        p += 1;
        }
      end = p - 1;

      fclose( file );

      if( YES == dump )
        rc = dumpExif( filename, start, end );
      else
        rc = exifPhoto( file, filename, start, end );
  
      if( rc != RET_OK )
        error = YES;

      free( start );

      }
    else
      {

      /* Free memory and close file, since denise() will load whole file again */
      free( start );
      fclose( file );

      /* Try to run old code */
      printf( "Note: File %s not in EXIF format, calling code for old Olympus format\n", filename );
      denise( filename );

      }

    }   /* end for */

  if( YES == error )
    return RET_ERROR;
  else
    return RET_OK;

  }


/*************************************************************************
 *                                                                       *
 * help()                                                                *
 * Print help screen                                                     *
 *                                                                       *
 *************************************************************************/
void help ( void )
  {

  puts(
    "\n"
    "RenamePics 2.1\n"
    "  Original code by Denise Hallmark <nesy@flash.net>. Modified for EXIF by Rolf\n"
    "  Lochbhler <rolf@together.net> and Brian Morrison <bdm@fenrir.demon.co.uk>.\n"
    "Purpose:\n"
    "  Rename photo file(s) in either Olympus format or EXIF format. Convert\n"
    "  exposure info contained in the file(s) into OS/2 extended attributes.\n"
    "Usage:\n"
    "  renamepics [/d] [/h] [File]\n"
    "Parameters:\n"
    "  /d        Only print complete file information to standard output\n"
    "            [default: rename file(s) and generate EAs quietly]\n"
    "  /h        Print this help screen, then abort [default]\n"
    "  File      File name, or template for file names (wildcards: * or ?)\n"
    "Examples:\n"
    "  renamepics /d 00000001.jpg > afile\n"
    "  renamepics *.jpg\n"
    "Note:\n"
    "  Run RenamePics before editing your photo files. Non EXIF aware graphics\n"
    "  software usually destroys the exposure info in EXIF files."
    );

  return;

  }


/*************************************************************************
 *                                                                       *
 * writeLongnameEa()                                                     *
 * Write a .LONGNAME extended attribute                                  *
 *                                                                       *
 *************************************************************************/
void writeLongnameEa ( char *fileName, char *longName )
  {

  EAOP2 eaop2;
  FEA2* fea2;
  char* p;
  int rc;

  eaop2.fpGEA2List = 0;
  eaop2.fpFEA2List = malloc( sizeof(FEA2LIST) );
  fea2 = &eaop2.fpFEA2List->list[0];

  fea2->fEA = 0;
  fea2->cbName = strlen( ".LONGNAME" );
  fea2->cbValue = strlen(longName) + 5 * sizeof(unsigned short);
 
  p = fea2->szName;
  strcpy( p, ".LONGNAME" );

  p += strlen(".LONGNAME") + 1;
  *(unsigned short *)p = EAT_ASCII;

  p += sizeof( unsigned short );
  *(unsigned short *)p = strlen(longName);

  p += sizeof( unsigned short );
  strncpy( p, longName, strlen(longName) );   /* No terminating '\0' */
  
  fea2->oNextEntryOffset = 0;
  eaop2.fpFEA2List->cbList = sizeof(FEA2LIST) - 1 + fea2->cbName + fea2->cbValue;
 
  rc = DosSetPathInfo( fileName, 2, &eaop2, sizeof(eaop2), 0 );

  if( 0 != rc )
    printf( "*Error* Cannot store .LONGNAME EA for file %s\n", fileName );

  free( eaop2.fpFEA2List );

  return;

  }



