/* fatcopy.cmd,v 4.13 1999-02-19 21:18:11-05 rl Exp */

/************************************************************************* 
 *                                                                       * 
 * fatcopy.cmd                                                           * 
 * A REXX script to copy files from HPFS to FAT                          * 
 * 1997-09-27, Rolf Lochbuehler <rolf@together.net>                      * 
 *                                                                       * 
 * Note: Requires that RexxUtil library functions are available          * 
 *                                                                       * 
 *************************************************************************/

AUTHOR = 'Rolf Lochbuehler'
REVISION = '4.13'
PROGRAM = 'FatCopy'
PROGRAM_CALL = translate( PROGRAM, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ' )
PROGRAM_UPPER = translate( PROGRAM )
PROGRAM_LEN = length( PROGRAM )
EMAIL = '<rolf@together.net>'

SRC_TYPE_DIRECTORY = 1
SRC_TYPE_FILE = 2

/* You should load the RexxUtil library already at system startup time! See ReadMe file. */
/*
if 0 <> rxfuncquery('sysdriveinfo')    then call rxfuncadd 'sysdriveinfo',    'rexxutil', 'SysDriveInfo'
if 0 <> rxfuncquery('sysfiledelete')   then call rxfuncadd 'sysfiledelete',   'rexxutil', 'SysFileDelete'
if 0 <> rxfuncquery('sysfiletree')     then call rxfuncadd 'sysfiletree',     'rexxutil', 'SysFileTree'
if 0 <> rxfuncquery('sysgetea')        then call rxfuncadd 'sysgetea',        'rexxutil', 'SysGetEA'
if 0 <> rxfuncquery('sysputea')        then call rxfuncadd 'sysputea',        'rexxutil', 'SysPutEA'
if 0 <> rxfuncquery('sysmkdir')        then call rxfuncadd 'sysmkdir',        'rexxutil', 'SysMkDir'
if 0 <> rxfuncquery('systempfilename') then call rxfuncadd 'systempfilename', 'rexxutil', 'SysTempFileName'
*/

/* Parse command line arguments */
parse arg opt
opt1 = translate( opt )
nargs.0 = words( opt )
do i = 1 to nargs.0
  nargs.i = 0
end
lastSwitch = 0

/* Set name of temporary directory */
tempdir = value( 'TEMP', , 'OS2ENVIRONMENT' )
if tempdir = '' then
  tempdir = value( 'TMP', , 'OS2ENVIRONMENT' )
if tempdir = '' then
  tempdir = '.'
if lastpos('\',tempdir) = length(tempdir) then
  tempdir = substr( tempdir, 1, length(tempdir)-1 )
TEMP_FILE = systempfilename( tempdir'\ea.???' )

/* Need help? */
if (length(opt) = 0) | (wordpos('/H',opt1) > 0) | (wordpos('-H',opt1) > 0) then
  do
  call help
  call exitprog 1
  end

/* Sufficient number of arguments? */
if (nargs.0 < 2) then
  do
  say '*'PROGRAM' Error* Insufficient number of arguments'
  say 'See 'PROGRAM_CALL' /h for help'
  call exitprog 1
  end

/* No EAs? */
keepea = 1
i = wordpos( '/NOEA', opt1 )
if 0 = i then
  i = wordpos( '-NOEA', opt1 )
if i > 0 then
  do
  nargs.i = 1
  if i > lastSwitch then
    lastSwitch = i
  keepea = 0
  end

/* Log file? */
LOG = 0
i = wordpos( '/LOG', opt1 )
if 0 = i then
  i = wordpos( '-LOG', opt1 )
if i > 0 then
  do
  nargs.i = 1
  if i > lastSwitch then
    lastSwitch = i
  LOG = 1
  end

/* Replace objects? */
replace = 0
i = wordpos( '/REPLACE', opt1 )
if 0 = i then
  i = wordpos( '-REPLACE', opt1 )
if i > 0 then
  do
  nargs.i = 1
  if i > lastSwitch then
    lastSwitch = i
  replace = 1
  end

/* Copy recursively? */
subdirs = 0
i = wordpos( '/S', opt1 )
if 0 = i then
  i = wordpos( '-S', opt1 )
if i > 0 then
  do
  nargs.i = 1
  if i > lastSwitch then
    lastSwitch = i
  subdirs = 1
  end

/* Source object? */
if 0 = pos('"',opt) then
  do
  /* Skip any switches */
  do i = lastSwitch + 1 while (pos('/',word(opt,i)) = 1) | (pos('-',word(opt,i)) = 1)
    i = i + 1
  end
  nargs.i = 1
  src = word( opt, i )
  end
else
  do
  i = pos( '"', opt )
  j = lastpos( '"', opt )
  parse var opt dummy =(i) . +1 src =(j) .
  m = words( dummy ) + 1
  n = m + words( src )
  do k = m to n
    nargs.k = 1
  end
  end
src = complete( src )
srcname = filespec( 'name', src )
srcdir = filespec('drive',src) || filespec('path',src)
srcdir = substr( srcdir, 1, length(srcdir)-1 )

/* Check for invalid wildcards in src */
if verify(srcdir,'*?','match') > 0 then
  do
  say '*'PROGRAM' Error* Invalid wildcards'
  say 'See 'PROGRAM_CALL' /h for help'
  call exitprog 1
  end

/* Check for valid wildcrads in src */
wildcards = 0
if verify(srcname,'*?','match') > 0 then
  wildcards = 1

/* Wildcards only for file names */
if (wildcards = 1) & (subdirs = 1) then
  do
  say '*'PROGRAM" Error* Don't use wildcards in directory names"
  say 'See 'PROGRAM_CALL' /h for help'
  call exitprog 1
  end

/* Destination object? */
if 0 = pos('"',opt) then
  do
  i = words( opt )
  nargs.i = 1
  destdir = word( opt, i )
  end
else
  do
  i = lastpos( '"', opt )
  parse var opt . =(i) . +1 destdir
  nargs.i = 1
  end
destdir = complete( destdir )

/* Check number of arguments */
do i = 1 to nargs.0
  if nargs.i <> 1 then
    do
    say '*'PROGRAM' Error* Invalid arguments'
    say 'See 'PROGRAM_CALL' /h for help'
    call exitprog 1
    end
end

/* From FAT to HPFS? */
tohpfs = 0
dummy = systempfilename( srcdir'\verylongfilename.????' )
if dummy = '' then
  /* FAT -> HPFS */
  tohpfs = 1
else
  /* HPFS -> FAT */
  tohpfs = 0
dummy = systempfilename( destdir'\verylongfilename.????' )
if dummy = '' then
  do
  if tohpfs = 1 then
    do
    /* FAT -> FAT */
    say '*'PROGRAM' Error* Both, source and destination file systems are FAT. Use Copy, Xcopy, or the WPS instead'
    call exitprog 1
    end
  else
    /* HPFS -> FAT */
    nop
  end
else
  do
  if tohpfs = 0 then
    do
    /* HPFS -> HPFS */
    say '*'PROGRAM' Error* Both, source and destination file systems are HPFS. Use Copy, Xcopy, or the WPS instead'
    call exitprog 1
    end
  else
    /* FAT -> HPFS */
    nop
  end

/* Check type of source object and if object exists */
if 1 = wildcards then
  do
  call sysfiletree src, 'allsrc', 'f'
  dummy.0 = allsrc.0
  end
else
  call sysfiletree src, 'dummy', 'b'
if dummy.0 > 0 then
  do
  parse var dummy.1 . . . attributes .
  if substr(attributes,2,1) = 'D' then
    srcType = SRC_TYPE_DIRECTORY
  else
    srcType = SRC_TYPE_FILE
  end
else
  do
  say '*'PROGRAM' Error* Cannot find' src
  call exitprog 1
  end

/* Check if destination object exists */
if length(destdir) = 3 then
  /* destdir is root directory of some drive */
  do
  info = sysdriveinfo( substr(destdir,1,2) )
  if info <> '' then
    /* destdir exists */
    dummy.0 = 1
  else
    /* destdir does not exist */
    dummy.0 = 0
  end
else
  /* destdir is not root directory of any drive */
  call sysfiletree destdir, 'dummy', 'do'
if dummy.0 = 0 then
  do
  say '*'PROGRAM' Error* Cannot find' destdir
  call exitprog 1
  end

calldir = directory()

parse value date('sorted') with year =5 month =7 day
datetime = year'-'month'-'day time('normal')
if 1 = LOG then
  do
  if 1 = tohpfs then
    do
    hpfslogfilename = PROGRAM'.log'
    parse value hpfswpsuniqname(destdir,hpfslogfilename,hpfslogfilename) with hpfslogfilename '\' wpslogfilename
    if length(hpfslogfilename) > 0 then
      do
      LOG_FILE = path( destdir, hpfslogfilename )
      call stream LOG_FILE, 'command', 'open write'
      call lineout LOG_FILE, 'Directories (d) and files (f) copied and renamed 'datetime' by 'PROGRAM' 'REVISION':'
      call lineout LOG_FILE, ''
      say 'Logfile is' LOG_FILE
      call addlongname LOG_FILE, wpslogfilename
      end
    else
      do
      say '*'PROGRAM' Error* Cannot create unique name for logfile in' destdir
      /* Continue, but without logging */
      LOG = 0
      end
    end
  else
    do
    fatlogfilename = fatuniqname( destdir, PROGRAM_UPPER'.LOG' )
    if length(fatlogfilename) > 0 then
      do
      LOG_FILE = path( destdir, fatlogfilename )
      call stream LOG_FILE, 'command', 'open write'
      call lineout LOG_FILE, 'Directories (d) and Files (f) copied and renamed 'datetime' by 'PROGRAM' 'REVISION':'
      call lineout LOG_FILE, ''
      say 'Logfile is' LOG_FILE
      end
    else
      do
      say '*'PROGRAM' Error* Cannot create unique name for logfile in' destdir
      /* Continue, but without loging */
      LOG = 0
      end
    end
  end

/* Go to start directory */
if srcType = SRC_TYPE_DIRECTORY then
  call directory src
else
  call directory srcdir

if tohpfs = 1 then 
  do 
  call charout , 'Copying from FAT to HPFS...'
  if srcType = SRC_TYPE_FILE then 
    do

    /* File(s) from FAT to HPFS */

    if 1 = wildcards then
      do j = 1 to allsrc.0
        srcname = filespec( 'name', allsrc.j )
        call filetohpfs srcname, destdir, replace, keepea
      end
    else
      call filetohpfs srcname, destdir, replace, keepea

    end
  else 

    /* Directory from FAT to HPFS */

    call dirtohpfs destdir, subdirs, replace, keepea

  end
else 
  do 
  call charout , 'Copying from HPFS to FAT...'
  if srcType = SRC_TYPE_FILE then 
    do

    /* File(s) from HPFS to FAT */

    if 1 = wildcards then
      do j = 1 to allsrc.0
        srcname = filespec( 'name', allsrc.j )
        call filetofat srcname, destdir, replace, keepea         
      end
    else
      call filetofat srcname, destdir, replace, keepea         

    end
  else 

    /* Directory from HPFS to FAT */

    call dirtofat destdir, subdirs, replace, keepea 

  end

say 'done'

if 1 = LOG then
  call stream LOG_FILE, 'command', 'close'

call directory calldir

call exitprog 0


/************************************************************************* 
 *                                                                       * 
 * help()                                                                * 
 * Display help screen for the user                                      * 
 *                                                                       * 
 *************************************************************************/
help: procedure expose AUTHOR EMAIL PROGRAM PROGRAM_CALL REVISION

  say PROGRAM' 'REVISION', 'AUTHOR' 'EMAIL
  say 'Purpose:'
  say '  Copy files and directories between HPFS and FAT, converting their names'
  say '  as required (HPFS -> FAT, 8.3 format) or possible (FAT -> HPFS, long names)'
  say 'Syntax:'
  say '  'PROGRAM_CALL' [/noea] [/h] [/log] [/replace] [/s] SrcDir DestDir'
  say '  'PROGRAM_CALL' [/noea] [/h] [/log] [/replace]      SrcFile DestDir'
  say 'Arguments:'
  say "  /h         Print this help info, then exit"
  say "  /noea      Don't copy EAs"
  say "  /log       Generate logfile in DestDir (default: no logfile)"
  say "  /replace   Replace files or directories rather than renaming the copies"
  say '  /s         Copy files in all subdirectories too'
  say '  SrcDir     Copy all files from SrcDir (physical name!)'
  say '  SrcFile    Copy only SrcFile (physical name!)'
  say '  DestDir    Copy to DestDir (physical name!)'
  say 'Notes:'
  say '  (1) SrcDir, SrcFile, and DestDir are assumed to be physical, i.e. FAT/HPFS'
  say '  names as displayed by DIR, rather than WPS names, for example "abc!1" instead'
  say '  of "abc:1". (2) Names with wildcards ("*" or "?") match file names only.'
  say 'Example:'
  say '  'PROGRAM_CALL' /s t:\hpfs\directory a:\'

  return


/*************************************************************************
 *                                                                       *
 * dirtofat()                                                            *
 * Copy directory, HPFS -> FAT                                           *
 *                                                                       *
 *************************************************************************/
dirtofat: procedure expose LOG LOG_FILE PROGRAM PROGRAM_UPPER PROGRAM_LEN TEMP_FILE

  parse arg fatparentdir, subdirs, replace, keepea

  /* Here: fatparentdir exists */
  /* Here: Present position is in HPFS directory to be copied */

  if subdirs = 1 then
    do
    call sysfiletree '*', 'd', 'do'
    call sysfiletree '*', 'f', 'fo'
    end
  else
    do
    d.0 = 0
    call sysfiletree '*', 'f', 'fo'
    end

  if f.0 > 0 then 
    do i = 1 to f.0

      call charout , '.'

      hpfsfilename = filespec( 'name', f.i )

      /* Ignore the following file */
      uppername = translate( hpfsfilename )
      lenname = length( hpfsfilename )
      if (substr(uppername,1,PROGRAM_LEN) = PROGRAM_UPPER) & (substr(uppername,lenname-2) = 'LOG') then
        do
        if LOG = 1 then
          call lineout LOG_FILE, 'f: 'f.i' -> (not copied)'
        iterate i
        end

      call filetofat hpfsfilename, fatparentdir, replace, keepea

    end

  if d.0 > 0 then
    do i = 1 to d.0

      call charout , '.'

      hpfsdirname = filespec( 'name', d.i )

      fatdirname = fatname( hpfsdirname )
      if replace = 0 then
        fatdirname = fatuniqname( fatparentdir, fatdirname )

      fatdir = path( fatparentdir, fatdirname )

      call sysfiletree fatdir, 'exists', 'do'
      if (exists.0 = 0) | (replace = 0) then
        call sysmkdir fatdir

      if exists.0 = 0 then 
        do
        if keepea = 1 then
          do
          call copyea hpfsdirname, fatdir
          call addlongname fatdir, wpsname(hpfsdirname)
          end
        else
          nop
        end
      else
        do
        if keepea = 1 then
          do
          call mergeea hpfsdirname, fatdir
          call addlongname fatdir, wpsname(hpfsdirname)
          end
        else
          call scrapea fatdir
        end
  
      if 1 = LOG then
        call lineout LOG_FILE, 'd: 'd.i' -> 'fatdir

      if subdirs = 1 then
        do
        call directory hpfsdirname
        call dirtofat fatdir, subdirs, replace, keepea
        call directory '..'
        end

    end

  return


/************************************************************************* 
 *                                                                       * 
 * dirtohpfs()                                                           * 
 * Copy directory, FAT -> HPFS                                           * 
 *                                                                       * 
 *************************************************************************/
dirtohpfs: procedure expose LOG LOG_FILE PROGRAM PROGRAM_UPPER PROGRAM_LEN TEMP_FILE

  parse arg hpfsparentdir, subdirs, replace, keepea

  /* Here: hpfsparentdir exists */
  /* Here: Present position is in FAT directory to be copied */

  if subdirs = 1 then
    do
    call sysfiletree '*', 'd', 'do'
    call sysfiletree '*', 'f', 'fo'
    end
  else
    do
    d.0 = 0
    call sysfiletree '*', 'f', 'fo'
    end

  if f.0 > 0 then 
    do i = 1 to f.0

      call charout , '.'
  
      fatfilename = filespec( 'name', f.i )
   
      /* Ignore the following files */
      uppername = translate( fatfilename )
      lenname = length( fatfilename )
      if uppername = 'EA DATA. SF' then
        iterate i
      if (substr(uppername,1,PROGRAM_LEN) = PROGRAM_UPPER) & (substr(uppername,lenname-2) = 'LOG') then
        do
        if LOG = 1 then
          call lineout LOG_FILE, 'f: 'f.i' -> (not copied)'
        iterate i
        end
  
      call filetohpfs fatfilename, hpfsparentdir, replace, keepea

    end

  if d.0 > 0 then 
    do i = 1 to d.0

      call charout , '.'

      fatdirname = filespec( 'name', d.i )

      /* Ignore the following directory */
      if translate(fatdirname) = 'WP ROOT. SF' then
        iterate i

      if keepea = 1 then
        wpsdirname = wpsname( fatdirname )
      else
        wpsdirname = fatdirname
      hpfsdirname = hpfsname( wpsdirname )
      if replace = 0 then
        parse value hpfswpsuniqname(hpfsparentdir,hpfsdirname,wpsdirname) with hpfsdirname '\' wpsdirname

      hpfsdir = path( hpfsparentdir, hpfsdirname )

      call sysfiletree hpfsdir, 'exists', 'do'
      if (exists.0 = 0) | (replace = 0) then
        call sysmkdir hpfsdir

      if keepea = 1 then
        do
        call copyea fatdirname, hpfsdir
        call addlongname hpfsdir, wpsdirname
        end

      if 1 = LOG then
        call lineout LOG_FILE, 'd: 'd.i' -> 'hpfsdir

      if subdirs = 1 then
        do
        call directory fatdirname
        call dirtohpfs hpfsdir, subdirs, replace, keepea
        call directory '..'
        end

    end

  return


/************************************************************************* 
 *                                                                       * 
 * filetofat()                                                           * 
 * Copy file, HPFS -> FAT                                                * 
 *                                                                       * 
 *************************************************************************/
filetofat: procedure expose LOG LOG_FILE PROGRAM TEMP_FILE

  parse arg hpfsfilename, fatdir, replace, keepea

  /* Here: hpfsfilename is in current directory */
  /* Here: fatdir exists */

  fatfilename = fatname( hpfsfilename ) 
  if replace = 0 then
    fatfilename = fatuniqname( fatdir, fatfilename )

  fatfile = path( fatdir, fatfilename )

  call checkspace hpfsfilename, fatdir

  call copyfile hpfsfilename, fatfile

  if keepea = 0 then
    call scrapea fatfile
  else
    call addlongname fatfile, wpsname(hpfsfilename)

  if 1 = LOG then
    call lineout LOG_FILE, 'f:' current(hpfsfilename) '->' fatfile

  return


/************************************************************************* 
 *                                                                       * 
 * filetohpfs()                                                          * 
 * Copy file, FAT -> HPFS                                                * 
 *                                                                       * 
 *************************************************************************/
filetohpfs: procedure expose LOG LOG_FILE PROGRAM TEMP_FILE

  parse arg fatfilename, hpfsdir, replace, keepea

  /* Here: fatfilename is in current directory */
  /* Here: hpfsdir exists */

  if keepea = 1 then
    wpsfilename = wpsname( fatfilename )
  else
    wpsfilename = fatfilename

  hpfsfilename = hpfsname( wpsfilename ) 

  if replace = 0 then
    parse value hpfswpsuniqname(hpfsdir,hpfsfilename,wpsfilename) with hpfsfilename '\' wpsfilename

  call checkspace fatfilename, hpfsdir

  hpfsfile = path( hpfsdir, hpfsfilename )

  call copyfile fatfilename, hpfsfile

  if keepea = 0 then
    call scrapea hpfsfile
  else
    call addlongname hpfsfile, wpsfilename

  if 1 = LOG then
    call lineout LOG_FILE, 'f:' current(fatfilename) '->' hpfsfile

  return


/************************************************************************* 
 *                                                                       * 
 * complete()                                                            * 
 * Complete directory/file name, substitute ellipses, etc.               * 
 *                                                                       * 
 *************************************************************************/
complete: procedure

  parse arg d

  d = strip( d, 'both' )
  d = strip( d, 'both', '"' )

  if d = '' then
    dn = directory()
  
  else if d = '.' then
    dn = directory()
  
  else if d = '..' then
    do
    curdir = directory()
    call directory '..'
    dn = directory()
    call directory curdir
    end
  
  else if substr(d,1,1) = '\' then
    do

    /* Here: '\any\dir\name' (absolute path, but missing 'drive:') */

    drive = filespec( 'drive', directory() )
    dn = drive || d

    end
  
  else if 0 = pos(':',d) then

    /* Here: 'any\dir\name' (relative path) */

    dn = current(d)

  else if (length(d) = 2) & (2 = pos(':',d)) then

    /* Here: '?:' (drive) */

    dn = d'\'

  else
    dn = d

  return dn


/************************************************************************* 
 *                                                                       * 
 * fatname()                                                             * 
 * Generate name for FAT object                                          * 
 *                                                                       * 
 *************************************************************************/
fatname: procedure

  parse arg oldname

  newname = translate( oldname )

  newname = translate( newname, copies('!',33), xrange(d2c(0),d2c(32)) )
  newname = translate( newname, '_', '\' )
  newname = translate( newname, '_', '/' )
  newname = translate( newname, '_', ':' )
  newname = translate( newname, '_', '*' )
  newname = translate( newname, '_', '?' )
  newname = translate( newname, '_', '"' )
  newname = translate( newname, '_', '<' )
  newname = translate( newname, '_', '>' )
  newname = translate( newname, '_', '|' )
  newname = translate( newname, '_', ',' )
  newname = translate( newname, '_', '+' )
  newname = translate( newname, '_', '=' )
  newname = translate( newname, '_', '[' )
  newname = translate( newname, '_', ']' )
  newname = translate( newname, '_', ';' )

  /* Replace special file endings */
  m = length( newname )
  n = m - 5
  if n = pos('TAR.GZ',newname) then
    newname = overlay( 'TGZ', newname, n, n + 2 )
  if n = pos('TAR.Z',newname) then
    newname = overlay( 'TZ', newname, n, n + 2 )

  if pos('.',newname) > 0 then
    do

    /* Here: Has at least one period */

    i = lastpos( '.', newname )
    parse var newname head =(i) . +1 tail

    /* Replace all periods but last one by underscores */
    head = translate( head, '_', '.' )

    /* Keep only first 8 characters of part before last period */
    if length(head) > 8 then
      head = substr( head, 1, 8 )

    /* Keep only first 3 characters of part after last period */
    if length(tail) > 3 then
      tail = substr( tail, 1, 3)

    newname = head'.'tail

    end
  else
  
    /* Here: Has no periods */

    /* Keep only first 8 characters */
    if length(newname) > 8 then
      newname = substr( newname, 1, 8 )

  return newname


/************************************************************************* 
 *                                                                       * 
 * wpsname()                                                             * 
 * Return .LONGNAME if it exists, or physical object name otherwise      * 
 *                                                                       * 
 *************************************************************************/
wpsname: procedure

  parse arg objectname

  /* Here: objectname is in current directory */

  call sysgetea objectname, '.LONGNAME', 'longname'

  if (longname = 'LONGNAME') | (strip(longname) = '') then
    return objectname
  else
    do
    null = '00'x
    parse var longname . (null) longname (null)
    return longname
    end


/************************************************************************* 
 *                                                                       * 
 * fatuniqname()                                                         * 
 * Create unique name for FAT object                                     * 
 *                                                                       * 
 *************************************************************************/
fatuniqname: procedure expose LOG LOG_FILE PROGRAM TEMP_FILE

  parse arg fatdir, fatobjectname

  oldname = path( fatdir, fatobjectname )

  call sysfiletree oldname, 'exists', 'bo'
  if exists.0 > 0 then
    do

    /* Here: object of same name exists already */

    parse var fatobjectname head '.' tail

    /* Here: length(head) = 1..8, length(tail) = 3 */

    len = length( head )
    if len < 8 then
      head = head || copies( '0', 8 - len )
  
    if length(tail) > 0 then
      hasTail = 1
    else
      hasTail = 0

    do i = 1 while exists.0 > 0

      if length(i) = 9 then 
        do
        /* Here: Tried all numbers from 1 to 99999999, but no unique name found */
        ln = '*'PROGRAM' Error* Cannot create unique name for' fatobjectname 'in' fatdir
        say ln
        if 1 = LOG then
          call lineout LOG_FILE, ln
        call exitprog 1
        end

      /* Make name unique by overlaying a number */
      newname = overlay( i, head, 9-length(i) ) 

      /* Append old tail, if there is one */
      if hasTail = 1 then
        newname = newname'.'tail

      call sysfiletree path(fatdir,newname), 'exists', 'bo'

    end   /* end do */

    end
  else

    /* Here: object of same name does not exists in fatdir */

    newname = fatobjectname

  return newname


/************************************************************************* 
 *                                                                       * 
 * copyea()                                                              * 
 * Copy EAs to copy of object                                            * 
 *                                                                       * 
 *************************************************************************/
copyea: procedure expose TEMP_FILE

  parse arg old, new, longname

  old = cmdarg( old )
  '@eautil "'old'"' TEMP_FILE '/s /p /r 2>nul 1>nul'

  if '' <> stream(TEMP_FILE,'command','query exists') then
    do
    new = cmdarg( new )
    '@eautil "'new'"' TEMP_FILE '/j /o 2>nul 1>nul'
    end
  
  return


/************************************************************************* 
 *                                                                       * 
 * cmdarg()                                                              * 
 * Prepare string argument for cmd call                                  *
 *                                                                       * 
 * This is required for strings in double quotes passed to cmd           * 
 *                                                                       * 
 *************************************************************************/
cmdarg: procedure

  parse arg s

  i = pos( '%', s )
  t = ''
  do while i > 0
    parse var s t '%' s
    t = t'%%'
    i = pos( '%', s )
  end
  s = t || s

  return s


/************************************************************************* 
 *                                                                       * 
 * mergeea()                                                             * 
 * Merge EAs of object with EAs of existing copy of this object          *
 *                                                                       * 
 *************************************************************************/
mergeea: procedure expose TEMP_FILE

  parse arg old, new, longname

  old = cmdarg( old )
  '@eautil "'old'"' TEMP_FILE '/s /p /r 2>nul 1>nul'

  if '' <> stream(TEMP_FILE,'command','query exists') then
    do
    new = cmdarg( old )
    '@eautil "'new'"' TEMP_FILE '/j /m 2>nul 1>nul'
    end
  
  return


/************************************************************************* 
 *                                                                       * 
 * scrapea()                                                             * 
 * Remove EAs from object                                                * 
 *                                                                       * 
 *************************************************************************/
scrapea: procedure expose TEMP_FILE

  parse arg object

  object = cmdarg( object )
  '@eautil "'object'"' TEMP_FILE '/s /r 2>nul 1>nul'

  return


/************************************************************************* 
 *                                                                       * 
 * addlongname()                                                         * 
 * Add long name to EAs of object                                        * 
 *                                                                       * 
 *************************************************************************/
addlongname: procedure

  parse arg object, longname

  call sysputea object, '.LONGNAME', 'FDFF'x || d2c(length(longname)) || '00'x || longname

  return


/************************************************************************* 
 *                                                                       * 
 * hpfswpsuniqname()                                                     * 
 * Generate uniq HPFS and WPS names for FAT object                       * 
 *                                                                       * 
 * Old:          WPS:            HPFS:                                   * 
 * abcdefg       abcdefg:1       abcdefg!1                               * 
 * abcdefg.xxx   abcdefg:1.xxx   abcdefg!1.xxx                           * 
 *                                                                       * 
 *************************************************************************/
hpfswpsuniqname: procedure

  parse arg hpfsdir, hpfsobject, wpsobject

  head = ''

  call sysfiletree path(hpfsdir,hpfsobject), 'exists', 'bo'
  if exists.0 > 0 then 
    do

    n = lastpos( '.', hpfsobject )
    if n > 0 then
      do
      n1 = n + 1
      parse var hpfsobject head =(n) . =(n1) tail
      parse var wpsobject wpshead =(n) . =(n1) wpstail
      end
    else
      do
      head = hpfsobject
      tail = ''
      wpshead = wpsobject
      wpstail = ''
      end

    do i = 1 while exists.0 > 0
      hpfsobject = head'!'i
      if length(tail) > 0 then
        hpfsobject = hpfsobject'.'tail
      call sysfiletree path(hpfsdir,hpfsobject), 'exists', 'bo'
    end

    num = i - 1
    if wpstail <> '' then
      wpsobject = wpshead':'num'.'wpstail
    else
      wpsobject = wpshead':'num

    end

  return hpfsobject'\'wpsobject


/************************************************************************* 
 *                                                                       * 
 * hpfsname()                                                            * 
 * Substitute invalid characters in WPS name                             * 
 *                                                                       * 
 *************************************************************************/
hpfsname: procedure

  parse arg name

  name = translate( name, copies('!',33), xrange(d2c(0),d2c(32)) )
  name = translate( name, '!', '\' )
  name = translate( name, '!', '/' )
  name = translate( name, '!', ':' )
  name = translate( name, '!', '*' )
  name = translate( name, '!', '?' )
  name = translate( name, '!', '"' )
  name = translate( name, '!', '<' )
  name = translate( name, '!', '>' )
  name = translate( name, '!', '|' )

  return name


/************************************************************************* 
 *                                                                       * 
 * checkspace()                                                          * 
 * Check if enough space on target drive                                 * 
 *                                                                       * 
 *************************************************************************/
checkspace: procedure expose LOG LOG_FILE PROGRAM TEMP_FILE

  parse arg file, dir

  /* Here: file is in current directory */

  drive = filespec( 'drive', dir )

  parse value sysdriveinfo(drive) with . free . .

  call sysfiletree file, 'dummy', 'f'
  if dummy.0 > 0 then
    parse var dummy.1 . . size . .
  else
    do
    ln = '*'PROGRAM' Error* SysFileTree() returned no data for' current(file)
    say ln
    if 1 = LOG then
      call lineout LOG_FILE, ln
    call exitprog 1
    end

  if size <= free then
    return
  else
    do
    ln = '*'PROGRAM' Error* Not enough space on' drive 'to copy' current(file)
    say ln
    if 1 = LOG then
      call lineout LOG_FILE, ln
    call exitprog 1
    end


/************************************************************************* 
 *                                                                       * 
 * exitprog()                                                            * 
 * Exit program                                                          * 
 *                                                                       * 
 *************************************************************************/
exitprog: procedure expose LOG LOG_FILE PROGRAM TEMP_FILE

  parse arg n

  call sysfiledelete TEMP_FILE

  if LOG = 1 then
    call stream LOG_FILE, 'command', 'close'

  exit n


/************************************************************************* 
 *                                                                       * 
 * copyfile()                                                            * 
 * Copy file                                                             * 
 *                                                                       * 
 *************************************************************************/
copyfile: procedure

  parse arg f1, f2

/*
  '@copy' '"' || cmdarg(f1) || '" "' || cmdarg(f2) || '" 2>nul 1>nul'
*/

  call stream f1, 'command', 'open read'
  call stream f2, 'command', 'open write'

  buffer = charin( f1, 1, chars(f1) ) 
  call charout f2, buffer
  drop buffer

  call stream f1, 'command', 'close'
  call stream f2, 'command', 'close'

  return


/************************************************************************* 
 *                                                                       * 
 * current()                                                             * 
 * Append file name to name of current directory                         * 
 *                                                                       * 
 *************************************************************************/
current: procedure

  parse arg f

  return path( directory(), f )


/************************************************************************* 
 *                                                                       * 
 * path()                                                                * 
 * Make path                                                             * 
 *                                                                       * 
 *************************************************************************/
path: procedure

  parse arg d, f

  if length(d) <> lastpos('\',d) then
    return d'\'f
  else
    return d || f

