/* DirPath - List all files in PATH like variable
   Default is PATH, support LIBPATH
   Set exit code to number of matches - 1
   Fatal errors terminate with exit code 255

   Copyright (c) 2015-2022 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License.  The GPL Software License can be found in
   gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2015-05-15 SHL Baseline
   2017-05-17 SHL Sync with templates
   2017-05-17 SHL Set exit code to number of matches - 1
   2017-08-02 SHL Correct typo
   2017-10-30 SHL Expand wildcards
   2017-11-13 SHL Support short name output
   2019-06-29 SHL Sync with templates
   2019-10-21 SHL Supply extensions for HELP
   2020-01-19 SHL Search current directory first if PATH
   2020-03-28 SHL Support %N in NLSPATH etc.
   2022-03-23 SHL Sync with templates - drop FileSpecEx
   2022-03-23 SHL Tweak message
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.1'

gEnvSym = xrange('A','Z') || xrange('a','z') || xrange('0','9') || '_'

Globals = 'gAddHistory gArgList. gCmdName gDbgLvl gEnv gEnvSym gEnvVar',
	  'gQuiet gShort gVerbose gVersion'

call Initialize

Main:
  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  u = translate(gEnvVar)
  select
  when u == 'LIBPATH' then
    dirs = GetLibpath()
  otherwise
    dirs = value(gEnvVar,, gEnv)
    if gEnvVar == 'PATH' & left(dirs, 2) \== '.;' then
      dirs = '.;' || dirs	/* Must search current directory first */

  end

  cnt = 0

  do while dirs \== ''
    parse var dirs dir ';' dirs
    dirs = strip(dirs)

    if right(dir, 3) == '\%N' then
      dir = substr(dir, 1, length(dir) - 3)	/* Strip %N */

    if \ IsDir(dir) then
      call WarnMsg '* Warning: cannot access' dir 'directory'
    else do
      do fileNum = 1 to gArgList.0
	files = gArgList.fileNum
	ext = GetExt(files)
	/* If no extension, guess from gEnvVar */
	if ext == '' then do
	  if gEnvVar == 'LIBPATH' then
	    files = files || '.dll'
	  else if gEnvVar == 'PATH' then
	    files = files || '.cmd' files || '.pl' files || '.btm' files || '.exe'
	  else if gEnvVar == 'HELP' then
	    files = files || '.hlp ' files || '.inf'
	end

	do while files \== ''
	  parse var files file files
	  files = strip(files)

	  file = MakePath(dir, file)
	  call DbgMsg 2, 'Checking' file
	  select
	  when gQuiet then
	    opts = 'O'
	  when gVerbose then
	    opts = 'L'
	  otherwise
	    opts = 'O'
	  end /* select */
	  call SysFileTree file, 'a', opts
	  cnt = cnt + a.0
	  if \ gQuiet then do
	    do i = 1 to a.0
	      if gShort then
		file = GetBaseName(a.i)
	      else
		file = a.i
	      say file
	      if gAddHistory then
		'@history /a' file
	    end /* do i */
	  end /* not quiet */
	end /* while files */

      end /* do fileNum */

    end /* if dir exists */
  end /* while dirs */

  exit cnt

/* end main */

/*=== GetLibPath() Return LIBPATH ===*/

GetLibPath: procedure expose (Globals)

  /* Get base libpath from config.sys */
  libpath = GetLibpathFromConfigSys()

  if right(libpath, 1) \== ';' then
    libpath = libpath || ';'

  bpath = SysQueryExtLibPath('B')
  if bpath \== '' & right(bpath, 1) \== ';' then
    bpath = bpath || ';'
  epath = SysQueryExtLibPath('E')

  libpath = bpath || libpath || epath

  return libpath

/* end GetLibPath */

/*=== GetLibpathFromConfigSys() Read LIBPATH from config.sys and return ===*/

GetLibpathFromConfigSys: procedure expose (Globals)

  inFile = SysBootDrive() || '\config.sys'

  if stream(inFile, 'C', 'QUERY EXISTS') == '' then
    call Die inFile 'does not exist.'

  /* Scan and parse */

  call stream inFile, 'C', 'OPEN READ'
  drop ErrCondition

  s = ''

  do while lines(inFile) \= 0
    call on NOTREADY name CatchError	/* Avoid death on missing NL */
    line = linein(inFile)
    signal on NOTREADY name Error
    parse var line var'='value
    if strip(translate(var)) \= 'LIBPATH' then
      iterate
    s = strip(value)
  end /* while lines */
  call stream inFile, 'C', 'CLOSE'

  if s == '' then
    call Die 'SET LIBPATH not found in' inFile

  return s

/* end GetLibpathFromConfigSys */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted
  /* Preset defaults */
  gAddHistory = 0			/* Add to history */
  gDbgLvl = 0				/* Debug messages */
  gVerbose = 0				/* Verbose messages */
  gQuiet = 0				/* Minimize output */
  gShort = 0				/* Minimize output */
  gEnvVar = ''				/* Env var to use */
  gArgList.0 = 0			/* Files to check */
  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'a' then
    gAddHistory = 1
  when curSw == 'd' then
    gDbgLvl = gDbgLvl + 1
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'q' then
    gQuiet = 1
  when curSw == 's' then
    gShort = 1
  when curSw == 'v' then
    gVerbose = 1
  when curSw == 'V' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'switch '''curSw''' unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg

  /* Arg is environment variable name or relative file name */
  do 1
    if gEnvVar == '' then do
      if verify(curArg, gEnvSym, 'N') = 0 then do
	/* Looks like an environment variable */
	s = value(curArg,, gEnv)
	if s \= '' | translate(curarg) == 'LIBPATH' then do
	  if gEnvVar \== '' then
	    call ScanArgsUsage 'only one environment variable allowed'
	  else do
	    gEnvVar = translate(curArg)
	    call DbgMsg 'Checking' gEnvVar
	    leave			/* Got env var */
	  end
	end
      end
    end /* if envvar maybe */

    /* Otherwise assume filename
       If absolute path, strip path to allow PATH/LIBPATH search
    */
    if left(curarg, 1) == '\' | substr(curarg, 2, 1) == '\' then
      curarg = filespec('N', curarg)

    call DbgMsg 'FileName is' curarg
    fileNum = gArgList.0 + 1
    gArgList.fileNum = curArg
    gArgList.0 = fileNum

  end /* do 1 */

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)

  if gShort & ( gQuiet | gVerbose ) then
    call ScanArgsUsage '-s not allowed with -v or -q'

  if gAddHistory & gQuiet then
    call ScanArgsUsage '-a not allowed with -q'

  if gArgList.0 = 0 then
    call ScanArgsUsage 'filename required'

  if gEnvVar = '' then do
    /* Default env var - assume all files in either PATH or LIBPATH */
    ext = GetExt(gArgList.1)
    if translate(ext) == '.DLL' then
      gEnvVar = 'LIBPATH'
    else
      gEnvVar = 'PATH'
  end
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'List files found via path-like variable'
  say 'Set exit code to number of matches'
  say 'Fatal errors set exit code to 255'
  say
  say 'Usage:' gCmdName '[-a] [-d] [-h] [-q] [-s] [-v] [-V] [env-var] filename...'
  say
  say '  -a         Add matches to history'
  say '  -d         Enable debug logic, repeat for more verbosity'
  say '  -h         Display this message'
  say '  -q         Set exit code only; do not output names'
  say '  -s         Output short names'
  say '  -v         Enable verbose output'
  say '  -V         Display version'
  say
  say '  env-var    Environment variable to use (default is PATH)'
  say '  filename   File to check, wildcards ok'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-a] [-d] [-h] [-q] [-s] [-v] [-V] [env-var] filename...'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== DbgMsg([minLvl, ]message,...) Optionally write multi-line message to STDERR ===*/

/**
 * Write message if gDbgLvl >= minLvl
 * @param minLvl defaults to 1 if omitted
 * @returns true if message written
 */

DbgMsg: procedure expose (Globals)
  minLvl = arg(1)
  if datatype(minLvl, 'W') then
    start = 2
  else do
    minLvl = 1
    start = 1
  end
  if gDbgLvl >= minLvl then do
    do i = start to arg()
      msg = arg(i)
      if msg \== '' then
	msg = ' *' msg
      call lineout 'STDERR', msg
    end
  end
  return gDbgLvl >= minLvl

/* end DbgMsg */

/*=== GetBaseName(pathName) Return file base name stripping drive path and extension ===*/

GetBaseName: procedure
  parse arg s
  /* return path name with drive, directory and last extension stripped */
  s = filespec('N', s)
  i = lastpos('.', s)
  if i > 1 then
    s = left(s, i - 1)			/* Chop extension */
  return s
/* end GetBaseName */

/*=== GetExt(pathName) Return last file name extension including dot or empty string ===*/

GetExt: procedure
  parse arg s
  s = filespec('N', s)
  i = lastpos('.', s)
  if i > 1 then
    s = substr(s, i)			/* Include dot */
  else
    s = ''				/* No extension */
  return s
/* end GetExt */

/*=== IsDir(dirName[, full]) return true if directory is valid, accessible directory ===*/

IsDir: procedure
  /* If arg(2) not omitted, return full directory name or empty string */
  parse arg dir, full
  newdir = ''

  do 1
    if dir == '' then do
      cwd = ''				/* No restore needed */
      leave
    end
    dir = translate(dir, '\', '/')	/* Convert to OS/2 slashes */
    s = strip(dir, 'T', '\')		/* Chop trailing slashes unless root */
    if s \== '' & right(s, 1) \== ":" then
      dir = s				/* Chop */
    drv = filespec('D', dir)
    cwd = directory()			/* Remember */
    /* If have drive letter and requested directory on some other drive */
    if drv \== '' & translate(drv) \== translate(left(cwd, 2)) then do
      /* Avoid slow failures and unwanted directory changes */
      drvs = SysDriveMap('A:')
      if pos(translate(drv), drvs) = 0 then
	leave				/* Unknown drive */
      if SysDriveInfo(drv) == '' then
	leave				/* Drive not ready */
      cwd2 = directory(drv)		/* Remember current directory on other drive */
      newdir = directory(dir)		/* Try to change and get full path name */
      call directory cwd2		/* Restore current directory on other drive */
      leave
    end

    /* If no drive letter or same drive and not UNC name */
    if left(dir, 2) \== '\\' then do
      newdir = directory(dir)		/* Try to change and get full path name */
      leave
    end

    /* UNC name - hopefully server is accessible or this will be slow
       Accept
	 \\server
	 \\server\
	 \\server\dir\
	 \\server\dir
     */
    cwd = ''				/* No restore needed */
    wc = dir
    if right(wc, 1) \== '\' then
      wc = wc || '\'
    i = lastpos('\', wc)
    if substr(wc, 3, 1) == '\' then
      leave				/* Malformed UNC - no server name */
    if pos('*', wc) > 0 | pos('?', wc) > 0 then
      leave				/* No wildcards allowed */
    call SysFileTree wc, 'files', 'O'
    if files.0 > 0 then do
      s = files.1			/* Exists and is not empty */
      i = lastpos('\', s)
      newdir = left(s, i - 1)		/* Extract directory name from full path name */
      leave
    end
    /* Try wildcarded directory name */
    wc = strip(wc, 'T', '\')
    i = lastpos('\', wc)
    base = substr(wc, i + 1)
    if base == '' then
      leave				/* Should have matched above */
    wc = substr(wc, 1, i) || '*' || base || '*'
    call SysFileTree wc, 'files', 'DO'
    do fileNum = 1 to files.0
      /* Find directory name is list */
      s = files.fileNum
      i = lastpos('\', s)
      s2 = substr(s, i + 1)
      if translate(base) == translate(s2) then do
	newdir = left(s, i - 1)
	leave
      end
    end /* i */
  end /* 1 */

  if cwd \== '' then
    call directory cwd			/* Restore original directory and drive */

  if full \== '' then
    ret = newdir			/* Return full directory name or empty string */
  else
    ret = newdir \== ''			/* Return true if valid and accessible */
  return ret

/* end IsDir */

/*=== MakePath(pathparts,...) Make path name from parts ===*/

MakePath: procedure

  /* All parts optional - code guesses what caller means.
     If last arg begins with a dot and is not .. and does not
     contain a slash, it is assumed to be a file extension.
     To avoid this behavior, pass empty arg as last arg.
     Empty args are ignored.
     Automatically converts unix slashes to dos slashes.
     If 1st arg is drive letter, it must have trailing colon.
   */

  argCnt = arg()

  path = ''

  do argNum = 1 to argCnt
    s = arg(argNum)
    s = translate(s, '\', '/')		/* Ensure DOS */
    if s == '' & argNum = argCnt then
      iterate				/* Ignore nul last arg */
    if argNum = 1 then
      path = s
    else do
      lead = left(s, 1)
      tail = right(path, 1)
      if tail == ':' & argNum = 2 then
	path = path || s		/* Append path part to drive spec */
      else if lead == '.' & argNum = argCnt & s \== '..' & pos('\', s) = 0  then
	path = path || s		/* Assume extension unless .. or contains \ */
      else if tail == '\' & lead == '\' then
	path = path || substr(s, 2)	/* Drop extra backslash */
      else if path \== '' & tail \== '\' & lead \== '\' then
	path = path || '\' || s		/* Ensure have backslash */
      else
	path = path || s
    end
  end /* for */

  return path

/* end MakePath */

/*=== WarnMsg(message,...) Write multi-line warning message to STDERR ===*/

WarnMsg: procedure
  do i = 1 to arg()
    msg = arg(i)
    call lineout 'STDERR', msg
  end
  return

/* end WarnMsg */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* The end */
