
// ==============================================================
//  Rob Hamerling's file date/time program (mainly for HPFS)
//  Shows Creation- LastAccess- and LastWrite- date/time
//  and optionally change CreationDate.
// ==============================================================
// Compilable with IBM VAC 3.08 + OS/2 Toolkit 4.5
// See also makefile: HPFSdate.mak
// ==============================================================
// Summary of changes:
//    date    version
// 2003-11-15  1.0.0 - Completely revised release (HPFSdate was
//                     previously released as part of Downsort).
// ==============================================================

#define INCL_BASE
#define INCL_DOSFILEMGR
#define INCL_NOPMAPI
#include <os2.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define  PGM_MAJOR     1                        // major version number
#define  PGM_MINOR     0                        // minor version number
#define  PGM_SUB       0                        // subversion number
#define  PGM_NAME      "HPFSdate"               // name
#define  PGM_AUTHOR    "Rob Hamerling"          // author

typedef struct _timestamp {FDATE fd; FTIME ft;} TIMESTAMP;

static  UCHAR szSearchArgument[_MAX_PATH] = "*.*";      // file specification
static  BOOL  bDebug                      = FALSE;
static  BOOL  bQueryPathInfo              = FALSE;
static  BOOL  bResetCreationDate          = FALSE;
static  BOOL  bShowDirectories            = FALSE;
static  BOOL  bSubdirTree                 = FALSE;

static  ULONG ulDirCount,                       // total directories
              ulFileCount;                      // total files

static PUCHAR pszTitle = "\n%s %u.%u.%u - "
                         "File info display utility by %s\n\n";

/* prototypes */

APIRET  cmdline(int, char **);
PUCHAR  formatLine(TIMESTAMP []);
APIRET  help(void);
APIRET  showtree(PUCHAR);
APIRET  showfiles(PUCHAR);
ULONG   splitfilename(PUCHAR);


// ----------------------------------------
// Mainline
// ----------------------------------------
int  main(int argc, char **argv) {

  APIRET       rc;                              // returncode of DOS-calls
  FILESTATUS3  qp;                              // pathinfo

  printf(pszTitle, PGM_NAME, PGM_MAJOR, PGM_MINOR, PGM_SUB, PGM_AUTHOR);

  rc = cmdline(argc, argv);                     // process user parameters
  if (rc)
    return rc;

  printf("File dates with DosFindFirst/Next");
  if (bQueryPathInfo)
    printf(" (on first line)\n"
           "            and DosQueryPathInfo (on second line)");
  printf("\n\nSearch argument: %s\n\n", szSearchArgument);

  if (szSearchArgument[strlen(szSearchArgument) - 1] == '\\') // trailing backslash
    szSearchArgument[strlen(szSearchArgument) - 1] = '\0';  // strip it

  if (strchr(szSearchArgument, '*') == NULL  &&       // )
      strchr(szSearchArgument, '?') == NULL) {        // ) no wildcards
    if (bDebug)
      printf("DEBUG: No wildcards, check if directory specified\n");
    rc = DosQueryPathInfo(szSearchArgument, FIL_STANDARD, &qp, sizeof(qp));
    if (rc == 0) {                                // found!
      if ((qp.attrFile & FILE_DIRECTORY) == FILE_DIRECTORY) {
        if (bDebug)
          printf("DEBUG: '%s' is a directory\n", szSearchArgument);
        strcat(szSearchArgument, "\\*.*");
        if (bDebug)
          printf("DEBUG: Search argument modified to '%s'\n", szSearchArgument);
        }
      }
    }

  ulDirCount = ulFileCount = 0;

  if (bSubdirTree)                              // subdirectories req'd
    rc = showtree(szSearchArgument);            // whole tree
  else                                          // no subdirectories
    rc = showfiles(szSearchArgument);           // only current dir

  printf("     Total %lu files", ulFileCount);
  if (ulDirCount > 0)
    printf(" in %lu directories", ulDirCount);
  printf("\n");

  return(rc);
  }


// ---------------------------------------------------
// Process directory tree
//
// Input:  - pointer to pathspec
//         - pointer to filecounter
//
// Output: - file count
//
// Returns - result of last DosFindFirst/Next
//           (NO_MORE_FILES replaced by 0)
//
// Note: This function calls itself recursively for each subdirectory
// level. To prevent stack shortage with deep directory trees most
// data is allocated dynamically.
// ---------------------------------------------------
APIRET showtree(PUCHAR  pszFileSpec) {

typedef struct _data {
  FILEFINDBUF3 ff;                              // findfirst/next
  FILESTATUS3  qp;                              // pathinfo
  ULONG        Fentries;                        // # entries to be retrieved
  HDIR         FileHandle;                      // FindFirst/Last handle
  UCHAR        BaseDrive[_MAX_DRIVE],           // base drive
               BasePath[_MAX_PATH],             // base path
               File[_MAX_FNAME],                // filename
               Ext[_MAX_EXT],                   // extension
               PathSpec[_MAX_PATH],             // path spec parm
               SubDir[_MAX_PATH];               // subdirectory
  } TREEDATA, *PTREEDATA;

  APIRET       rc;                              // returncode of DOS-calls
  PTREEDATA    d;                               // pointer to data

  d = malloc(sizeof(TREEDATA));                 // not from stack!

  _splitpath(pszFileSpec, d->BaseDrive, d->BasePath, d->File, d->Ext);

  if (bDebug) {
    printf("DEBUG: showtree(%s)\n", pszFileSpec);
    printf("       Drive: '%s', Path: '%s', File: '%s', Ext: '%s'\n",
                   d->BaseDrive, d->BasePath, d->File, d->Ext);
    }

  _makepath(d->SubDir, d->BaseDrive, d->BasePath, "*", ".*");  // all subdirs!
  d->Fentries = 1;                              // one by one
  d->FileHandle = HDIR_CREATE;                  // new file handle
  rc = DosFindFirst(d->SubDir, &d->FileHandle,
                       MUST_HAVE_DIRECTORY,
                       &d->ff, sizeof(FILEFINDBUF3), &d->Fentries,
                       FIL_STANDARD);           // no EA's

  while (rc == 0) {                             // as long as no error

    if (bDebug) {
      printf("DEBUG: attrFile=%08lX  '%s'\n",
             d->ff.attrFile, d->ff.achName);
      fflush(stdout);
      }

    if (strcmp(d->ff.achName, ".") &&           // not current dir
        strcmp(d->ff.achName, "..") ) {         // not parent dir

      _makepath(d->SubDir, d->BaseDrive, d->BasePath, d->ff.achName, NULL);
      _fullpath(d->PathSpec, d->SubDir, _MAX_PATH);

      if (bDebug && !bShowDirectories)
        printf("DEBUG: %s (directory)\n", d->PathSpec);

      if (bShowDirectories) {                   // subdir info wanted
        printf("%s (directory) %c%c%c%c\n",
               d->PathSpec,
               (d->ff.attrFile & FILE_ARCHIVED) ? 'a' : '-',
               (d->ff.attrFile & FILE_SYSTEM)   ? 's' : '-',
               (d->ff.attrFile & FILE_HIDDEN)   ? 'h' : '-',
               (d->ff.attrFile & FILE_READONLY) ? 'r' : '-');
        printf("   %s\n", formatLine((TIMESTAMP *)&d->ff.fdateCreation));

        if (bQueryPathInfo) {                     // also DosQueryPathInfo?
          if (DosQueryPathInfo(d->PathSpec, FIL_STANDARD,
                               &d->qp, sizeof(FILESTATUS3)) == 0)
            printf("   %s\n", formatLine((TIMESTAMP *)&d->qp.fdateCreation));
          }

        }

      strcpy(d->SubDir, d->BasePath);           // original path
      strcat(d->SubDir, d->ff.achName);         // append new subdir
      _makepath(d->PathSpec, d->BaseDrive, d->SubDir, d->File, d->Ext);
      showtree(d->PathSpec);                    // next level

      }

    d->Fentries = 1;                            // one-by-one
    rc = DosFindNext(d->FileHandle,
                     &d->ff, sizeof(FILEFINDBUF3), &d->Fentries);
    }

  DosFindClose(d->FileHandle);                  // close directory association

  if (rc) {
    if (rc == ERROR_NO_MORE_FILES  ||
        rc == 65535)                            // NTFS bug
      rc = 0;                                   // normal end of search
    else
      printf("Directory search ended with rc=%lu\n", rc);
    }

  free(d);                                      // release memory

  rc = showfiles(pszFileSpec);                  // show files in this dir

  ++ulDirCount;                                 // one more directory!

  return rc;
  }


// ---------------------------------------------------------------------
// Display files of a single directory (not contents of subdirectories)
//
// Input:  - pointer to pathspec
//         - pointer to filecounter
//
// Output: - file count
//
// Returns - result of last DosFindFirst/Next
//           (NO_MORE_FILES replaced by 0)
//
// (not called recursively)
// ---------------------------------------------------------------------
APIRET showfiles(PUCHAR  pszFileSpec) {

  FILEFINDBUF3 ff;                              // findfirst/next
  FILESTATUS3  qp;                              // pathinfo
  ULONG        ulFentries;                      // FindFirst/Next parm
  ULONG        ulFileCountEntry;                // # at entry of function
  APIRET       rc;                              // returncode of DOS-calls
  HDIR         FileHandle;                      // FindFirst/Last handle
  UCHAR        pszBaseDrive[_MAX_DRIVE];        // base drive
  UCHAR        pszBasePath[_MAX_DIR];           // base path
  UCHAR        pszPathSpec[_MAX_PATH];          // local path spec

  _splitpath(pszFileSpec, pszBaseDrive, pszBasePath, NULL, NULL);

  if (bDebug)
    printf("DEBUG: showfiles(%s)\n", pszFileSpec);

  ulFileCountEntry = ulFileCount;               // at entry

  ulFentries = 1;                               // one by one
  FileHandle = HDIR_CREATE;                     // new file handle

  rc = DosFindFirst(pszFileSpec, &FileHandle,
                    FILE_ARCHIVED | FILE_SYSTEM | FILE_HIDDEN | FILE_READONLY,
                    &ff, sizeof(ff), &ulFentries,
                    FIL_STANDARD);              // no EA's
  while (rc == 0) {                             // as long as no error

    if (bDebug) {
      printf("DEBUG: %08lX %s\n", ff.attrFile, ff.achName);
      fflush(stdout);
      }

    ++ulFileCount;                              // one more matching file!

    _makepath(pszPathSpec, pszBaseDrive, pszBasePath, ff.achName, NULL);
    _fullpath(pszPathSpec, pszPathSpec, _MAX_PATH);
    printf("%s  %lu  %c%c%c%c\n",
           pszPathSpec, ff.cbFile,
           (ff.attrFile & FILE_ARCHIVED) ? 'a' : '-',
           (ff.attrFile & FILE_SYSTEM)   ? 's' : '-',
           (ff.attrFile & FILE_HIDDEN)   ? 'h' : '-',
           (ff.attrFile & FILE_READONLY) ? 'r' : '-');
    printf("   %s\n", formatLine((TIMESTAMP *)&ff.fdateCreation));

    if (bResetCreationDate || bQueryPathInfo) {     // PathInfo required
      rc = DosQueryPathInfo(pszPathSpec, FIL_STANDARD, &qp, sizeof(qp));
      if (rc != 0)
        printf("DosQueryPathInfo for %s failed: rc=%u\n", pszPathSpec, rc);
      else {
        if (bQueryPathInfo)                           // PathInfo wanted
          printf("(P)%s\n", formatLine((TIMESTAMP *)&qp.fdateCreation));

        if (bResetCreationDate) {                     // reset wanted
          qp.fdateCreation = qp.fdateLastWrite;       // date
          qp.ftimeCreation = qp.ftimeLastWrite;       // time
          rc = DosSetPathInfo(pszPathSpec, FIL_STANDARD,
                              &qp, sizeof(qp), 0L);
          if (rc != 0)
            printf("DosSetPathInfo for %s failed: rc=%u\n", pszPathSpec, rc);
          else
            printf("(R)%s\n", formatLine((TIMESTAMP *)&qp.fdateCreation));
          }
        }
      }

    ulFentries = 1;                             // one-by-one
    rc = DosFindNext(FileHandle, &ff, sizeof(ff), &ulFentries);
    }

  DosFindClose(FileHandle);                     // close directory association
  if (rc) {
    if (rc == ERROR_NO_MORE_FILES  ||           // OK
        rc == 65535)                            // NTFS bug?
      rc = 0;                                   // normal end of search
    else
      printf("File search ended with rc=%lu\n", rc);
    }

  if (ulFileCount - ulFileCountEntry > 0)       // any files
    printf("\n");                               // directory separator

  return rc;
  }


// -----------------------------------------------
// Format a line with dates and times
// COUNTRY format mm-dd-yy (USA) or dd-mm-jj (EUR)
// -----------------------------------------------
PUCHAR  formatLine(TIMESTAMP ts[]) {

  static UCHAR pszLine[80];
  int    ll, i;

  for (i=ll=0; i<3; i++)                        // all 3 timestamps
    ll += sprintf(pszLine + ll, "  %c: %4u-%02u-%02u %2u:%02u:%02u",
                  "CAW"[i],                     // type of date/time
                  ts[i].fd.year+1980, ts[i].fd.month,   ts[i].fd.day,
                  ts[i].ft.hours,     ts[i].ft.minutes, ts[i].ft.twosecs*2);
  return pszLine;
  }


// --------------------------------------------------------------
// Process commandline parameters (modifying default settings)
// --------------------------------------------------------------
APIRET  cmdline(int argc, char *argv[]) {

  int          i;                               // counter(s)

  for (i=1; i<argc; ++i) {
    if (argv[i][0] == '/' || argv[i][0] == '-') {  // option flag
      switch (argv[i][1]) {
        case 'd':
        case 'D': bShowDirectories = TRUE;
                  break;
        case 'p':
        case 'P': bQueryPathInfo = TRUE;
                  break;
        case 'r':
        case 'R': bResetCreationDate = TRUE;
                  break;
        case 's':
        case 'S': bSubdirTree = TRUE;
                  break;
        case '$': bDebug = TRUE;
                  break;
        default : return help();
        }
      }
    else                                        // not an 'option'
      strcpy(szSearchArgument, argv[i]);        // replace filespec
    }

  return 0;
  }


// --------------------------------------------------------------
// Display Command help information
// --------------------------------------------------------------
APIRET help() {

  printf("Syntax:  PATHDATE  [<pathspec>] [/d] [/p] [/r] [/s] [/$]\n\n"
         "  <pathspec>  Path specification; wildcards allowed, default: *.*\n"
         "              Date/time obtained by DosFindFirst/Next will be shown\n"
         "     /d       Show also time/date of all subdirectories"
                        " (only with /s option)\n"
         "     /p       Show also dates obtained by DosQueryPathInfo\n"
         "     /r       Reset CreationDate to LastWriteDate\n"
         "     /s       Search subdirectory tree\n"
         "     /$       Debug: show some intermediate information\n\n");
  return  99;
  }


