/*     REXX                               */

/*
#!/usr/local/bin/rexx
        --  For Unix-ish systems, using the Regina interpreter,
            put something like this 'shebang' comment
            as the first line.
For most other systems, make sure the comment with the word
'REXX' in it is on the first line, with the '/' and '*' in the first two
columns.
*/


/*   'PARSE'
      (c) Oct. 2001  by Dallas E. Legan   legan@acm.org
      I'm releasing this under a BSD License -
      copy, modify or use for any purpose whatsoever,
      but if any of it is significantly recognizable
      as derivative from this program, please
      credit me as the original author.
      I definitly am not trying to claim credit
      as the originator of the idea of writing
      filters in REXX, only this example of
      general purpose filter using a DO loop and I/O steps
      inside an INTERPRET REXX command,
      with the guts of the filtering action passed from
      the command line via a variable into this boilerplate program.
      If you come up with any interesting or useful derivatives or
      improvements, feel free to let me know about them,
      I can then give you some credit.
*/

/*  This version modified to provide initialized variables
    to talley or concatinate sums, products or found strings.
    Also, some variable names have been changed to match
    those of AWK, to allow easier learning and present
    an enviroment more consistent with other tools
    the user may have encountered.
*/

TRACE 'OFF'           ;
PARSE ARG   templateRoutine  ;

/* SYNTAX seems to be the only trap needed to trigger 'Help'.    */
SIGNAL   ON SYNTAX      NAME Help  ;
/* -- This must be set in the Main Routine to be effective in Main */
/*    See Daney, 'Programming in REXX', p. 204          */



CALL StartUp    ;


/*   Main Loop                           */


INTERPRET ,
'PARSE VAR QQQQ 'tempTest' ; ',
'DO  NR = 1  WHILE( BLines() ) ;',
' PARSE VAR DZ 'templateRoutine' ;',
' rc = LineOut( OP, US )  ;',
'END  ;'   ;




/*
   The first PARSE in the template is to check for invalid
   templates, and trigger a call to help for things like '-?'
   '--help' etc.
   Seemed necessary in OS/2 for triggering help.
   Variable 'QQQQ', with a default value of 'QQQQ' (it's own name),
   is parsed with the template.
   The only reason this is included here is so that there
   will only be one INTERPRET statement in the program,
   for efficiency reasons.  Otherwise it would of been
   performed earlier in the program, before this step,
   probably in the StartUp subroutine.

   'NR'  isn't really needed for the Do loop control,
   but is thrown in to allow access to the line number,
   which it counts in this context.
   Named after AWK variable with the same purpose.

   Previously had 'END NR ; ', but dropped this because it
   prevented early termination of the DO loop, with a
   grouping DO/END, similar to a Perl/Awk END blocks,
   at the end.
   Example:
   '.....END;DO <somestuff> END  ;'

   Placed entire DO loop in an Interpret statement to avoid
   having repeated calls to it inside a loop.
   This also allows ITERATE, LEAVE and EXIT statements
   inside the INTERPRET statement.

   'OP' was added to allow input triggered redirection of output.
   An example might be the classic mail message seperation task.

   Note how REXX allows literal strings to be joined by
   abuttal if there is something between them,
   in this case the REXX line continuation character ',',
   and their actually being on seperate lines.
*/

EXIT  ;    /*  Main Routine  */

StartUp:
/*  Initialize some variables, etc.  */

templateRoutine  =  Strip( templateroutine, 'B', ' ' )  ;

/*
I'm leaving this section for adjustment by hand.
This program might be ported to many other OS/Interpreter pairs,
in which case properly setting these up to select the right values
automaticly might get very complex compared to simply editing this source
by hand.
*/

/*  Classic REXX (OS/2 Warp) :  */
input         =  'STDIN:'  ;    /*  Adjust for appropriate OS STDIN       */
OP            =  'STDOUT:' ;    /*    "                       STDOUT      */
n             =  '0D0A'x   ;    /*    "                       EOL         */
/* Some shells will pass inclosing quotes into the template in OS/2, so:  */
quoter  =  Right( templateRoutine, 1 )  ;
IF  ( '"' = quoter | "'" = quoter ) & Abbrev( templateRoutine, quoter )  THEN
    templateRoutine  =  Strip( templateRoutine, 'B', quoter )  ;


/*
/*  Regina (Unix):       */
input         =  ''        ;    /*  Adjust for appropriate OS STDIN   */
OP            =  ''        ;    /*    "                       STDOUT  */
n             =  '0A'x     ;    /*    "                       EOL     */
*/



RS            =  n         ;    /*  For AWK compatibility             */
tab           =  '09'x     ;    /*  tab made available if needed      */
t.            =  0         ;    /*  Talleys                           */
p.            =  1         ;    /*  products                          */
c.            =  ''        ;    /*  concatenations                    */


SELECT
/*   set the default to set US to input on the main parse   */
/*   defaulting to output = input if no script is provided. */
WHEN  '' = templateRoutine  THEN
  DO
  templateRoutine  =  'US'  ;
  END
WHEN  1  = Pos( ';', templateRoutine )  THEN
  DO
  templateRoutine  =  'US' templateRoutine  ;
  END
OTHERWISE
  NOP  ;
END    /*    SELECT   */

PARSE VAR templateRoutine  tempTest ';' .   ;
/*  -- seperating out the template from any other commands
       specificly to test it for validity, and possibly
       trigger the Help subroutine.
*/


Return  1 ;   /*  StartUp  */

Blines:
/*
    Boolean Lines function for portability across REXX versions.
     See 'The REXX Cookbook',  Merrill Callaway, p. 2-4.
     Generally on most REXX's, Lines is not a Boolean function,
     and returns the number of lines remaining to be read on the
     file/stream.
     It is Boolean on OS/2, and Regina.

      Lines()
    only seems to indicate 0 after a read fails for Regina interpreter,
    so some mods were called for.

    Added some basic transformations/information for the line of input data
    that can easily be done via REXX that require no input
    to decide how to make them.

*/


DZ         =  LineIn( input )     ;

/* Some simple transformations that might be useful to have
   automaticly available for each input record:
*/
NF         =  Words( DZ )         ;   /* # of non-blank fields in record  */
RL         =  Length( DZ )        ;   /* Length of current record available
                                          for manipulations   */
ZD         =  Reverse( DZ )       ;   /* reversed current record  */

somelines  =  0 >< Lines( input )  ;
/*  -- Testing after the possible failure of the above LineIn(),
       For Regina.
*/

IF  \ somelines  THEN      /* Ran off the end of the input, need to reset  */
   NR  =  NR - 1  ;        /*  line counter back to last line on the input */


RETURN  somelines  ;     /*   BLines   */


Help:

SAY ,
n,
'USAGE: parse ''[ -h | <template>; [ rexx; commands; ]]''  < infile  > outfile'n,
n,
'       -h  help'n,
n,
'       http://www.rexswain.com/rexx.html  Rex Swain''s REXX Summary'n,
n,
'       US  =  default variable written to output (UnderScore (Perl) :-) )'n,
'       NR  =  Number of current Record/line (AWK)'n,
'       DZ  =  "$0" (Dollar Zero (AWK)), entire current input record'n,
'       ZD  =  entire current input record (DZ) reversed'n,
'       NF  =  Number of Fields in current record (AWK)'n,
'       OP  =  OutPut file (STDOUT by default)'n,
'       RS  =  Record Seperator / EOL (newline) (AWK)'n   ;


EXIT  ;     /*  HELP   */

/*  END of PARSE program  */
