/*
 * flist.c - Operations for Font List
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <sys/fcntl.h>

#include "xfont.h"

/*
 * initFontList, freeFontList
 */

LISTPTR initFontList(void)
{
    LISTPTR top ;

#ifdef  DEBUG
    printf("initFontList\n") ; fflush(stdout) ;
#endif
    
    if ((top = (LISTPTR) malloc(sizeof(LISTREC))) == NULL) {
        return NULL ;
    }
    memset(top, 0, sizeof(LISTREC)) ;
    
    return top ;
}

void    freeFontList(LISTPTR top)
{
    LISTPTR ptr, nxt ;

#ifdef  DEBUG
    printf("freeFontList\n") ; fflush(stdout) ;
#endif
    
    if (top == NULL) {
        return ;
    }
    for (ptr = top->next ; ptr != NULL ; ptr = nxt) {
        nxt = ptr->next ;
	if (ptr->full)  free(ptr->full) ;
	if (ptr->finfo) FreeFontInfo(ptr->finfo) ;
	free(ptr) ;
    }
    free(top) ;
}

/*
 * File Utilities
 */

static  char    *fullname(char *fname)
{
    char    buff[512] ;
    
    if (_abspath(buff, fname, 512) != 0) {
        return NULL ;
    }
    if (access(buff, R_OK) != 0) {
        return NULL ;
    }
    return strdup(buff) ;
}

static  char    *basename(char *fullname)
{
    char    *p ;
    
    if (fullname == NULL || *fullname == '\0') {
        return NULL ;
    }
    if ((p = strrchr(fullname, '/')) == NULL) {
        return NULL ;
    }
    p++ ;
    return p ;
}

static  int     fonttype(char *fullname)
{
    char    *p ;
    
    if (fullname == NULL || *fullname == '\0') {
        return FTYPE_NON ;
    }
    if ((p = strrchr(fullname, '.')) == NULL) {
        return FTYPE_NON ;
    }
    p++ ;

    if (stricmp(p, "PCF") == 0) {
        return FTYPE_PCF ;
    } else if (stricmp(p, "BDF") == 0) {
        return FTYPE_BDF ;
    } else {
        return FTYPE_NON ;
    }
}

/*
 * putFontFile - append font file into Font List
 */
 
LISTPTR putFontFile(LISTPTR top, char *fname)
{
    LISTPTR new, ptr, prv ;
    int     comp ;

    if (top == NULL) {
        return NULL ;
    }
    
#ifdef  DEBUG
    printf("putFontFile <%s>\n", fname) ; fflush(stdout) ;
#endif
    
    /*
     * Create List Entry
     */
     
    if ((new = (LISTPTR) malloc(sizeof(LISTREC))) == NULL) {
        return NULL ;
    }
    memset(new, 0, sizeof(LISTREC)) ;

    new->full   = fullname(fname)     ;
    new->base   = basename(new->full) ;
    new->ftype  = fonttype(new->full) ;
    new->status = FSTAT_NONE ;
    new->finfo  = NULL ;

    if (new->full == NULL || new->base == NULL || new->ftype == FTYPE_NON) {
        if (new->full) free(new->full) ;
	free(new) ;
	return NULL ;
    }

    /*
     * Link to FontList, with full name's order
     */
     
    for (prv = NULL, ptr = top->next ; ptr != NULL ; ptr = ptr->next) {
        comp = stricmp(ptr->full, new->full) ;
        if (comp < 0) {
	    prv = ptr ;
	    continue  ;
        } else if (comp == 0) {
	    free(new->full) ;
	    free(new) ;
	    return ptr ;
	} else {
	    break ;
	}
    }
    
    if (prv != NULL) {
        new->next = prv->next ;
	prv->next = new ;
    } else {
        new->next = top->next ;
	top->next = new ;
    }
    return new ;       
}

/*
 * getFontList, getFontNext - traverse on Font List
 */
 
LISTPTR getFontList(LISTPTR top)
{
    if (top == NULL) {
        return NULL ;
    }
    return top->next ;
}

LISTPTR getFontNext(LISTPTR ptr)
{
    if (ptr == NULL) {
        return NULL ;
    }
    return ptr->next ;
}

/*
 * loadFontInfo - Load Font Info. specified with LISTPTR
 */

static  void    loadThread(void *p)
{
    FILE    *fp ;
    LISTPTR ptr = (LISTPTR) p ;
    
    if (ptr->ftype == FTYPE_PCF) {
        fp = fopen(ptr->full, "rb") ;
    } else if (ptr->ftype == FTYPE_BDF) {
        fp = fopen(ptr->full, "r") ;
    } else {
        fp = NULL ;
    }
    if (fp == NULL) {
        /* printf("cannot open %s\n", ptr->base) ; fflush(stdout) ; */
        ptr->status = FSTAT_NONE ;
	(*ptr->fnDone) (ptr) ;
	return ;
    }
    
    if (ptr->ftype == FTYPE_PCF) {
        ptr->finfo = ReadPcfFont(fp) ;
    } else {
        ptr->finfo = ReadBdfFont(fp) ;
    }

    fclose(fp) ;
    
    if (ptr->finfo != NULL) {
        ptr->status = FSTAT_DONE ;
    } else {
        /* printf("cannot load %s\n", ptr->base) ; fflush(stdout) ; */
        ptr->status = FSTAT_NONE ;
    }    
    (*ptr->fnDone) (ptr) ;
}

LISTPTR loadFontInfo(LISTPTR ptr, void (*fnDone) (LISTPTR ptr))
{
    if (ptr == NULL) {
        return NULL ;
    }
    if (ptr->status != FSTAT_NONE) {
        return ptr ;
    }

    ptr->status = FSTAT_LOAD ;
    ptr->fnDone = fnDone     ;
    
#ifdef  DEBUG
    printf("loadFontInfo on %s\n", ptr->base) ;
#endif
    
    if (_beginthread(loadThread, NULL, 0x8000, ptr) < 0) {
        ptr->status = FSTAT_NONE ;
    }
    return ptr ;
}

#ifdef  TEST

static  char    *types[] = {
    "NON", "PCF", "BDF", "UNK", "UNK", "UNK"
} ;

void    done(LISTPTR ptr)
{
    char    *p ;
    
    if (ptr->status == FSTAT_DONE) {
        p = "done" ;
    } else {
        p = "fail" ;
    }
    
    printf("%s %s\n", p, ptr->base) ;
}

int main(int ac, char *av[])
{
    int     i ;
    LISTPTR list, p ;

    _wildcard(&ac, &av) ;
    
    list = initFontList() ;
    
    for (i = 1 ; i < ac ; i++) {
        putFontFile(list, av[i]) ;
    }

    printf("start loading...\n") ; fflush(stdout) ;
    
    for (p = getFontList(list) ; p != NULL ; p = getFontNext(p)) {
        loadFontInfo(p, done) ;
    }

waiting:
    for (p = getFontList(list) ; p != NULL ; p = getFontNext(p)) {
        if (p->status == FSTAT_LOAD) {
	    sleep(1) ;
	    goto waiting ;
	}
    }

    printf("loading finished\n") ; fflush(stdout) ;

    for (p = getFontList(list) ; p != NULL ; p = getFontNext(p)) {
        if (p->finfo) {
	    DumpFontInfo(p->finfo) ;
	}
    }
    freeFontList(list) ;
}   

#endif 
