/*
 * display.c - display tablet in client window
 */

#define INCL_PM
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tablet.h"
#include "resource.h"

/*
 * windows to use
 */
 
static  HWND    hwndFrame   = NULLHANDLE ;
static  HWND    hwndClient  = NULLHANDLE ;
static  HWND    hwndVScroll = NULLHANDLE ;
static  HWND    hwndHScroll = NULLHANDLE ;
static  HWND    hwndCorner  = NULLHANDLE ;

/*
 * system values
 */

static  LONG    svCxVScroll = 0 ;
static  LONG    svCyHScroll = 0 ;

/*
 * dispInit - start display management
 */
 
BOOL    dispInit(HWND hwnd)
{
    ULONG   flVScrollStyle = SBS_VERT | SBS_AUTOTRACK ;
    ULONG   flHScrollStyle = SBS_HORZ | SBS_AUTOTRACK ;

    hwndClient = hwnd ;
    hwndFrame  = WinQueryWindow(hwnd, QW_PARENT) ;
    
    /*
     * create scroll bars to control tablet position
     */
    
    hwndVScroll = WinCreateWindow(
                hwndClient,             /* parent window handle     */
		WC_SCROLLBAR,           /* window class             */
		"",                     /* window text              */
		flVScrollStyle,         /* window style             */
		0, 0, 0, 0,             /* position and size        */
		hwndClient,             /* owner window handle      */
		HWND_TOP,               /* Z-order                  */
		IDW_VSCR,               /* window ID                */
		NULL,                   /* control data             */
		NULL) ;                 /* presentation parameter   */

    hwndHScroll = WinCreateWindow(
                hwndClient,             /* parent window handle     */
		WC_SCROLLBAR,           /* window class             */
		"",                     /* window text              */
		flHScrollStyle,         /* window style             */
		0, 0, 0, 0,             /* position and size        */
		hwndClient,             /* owner window handle      */
		HWND_TOP,               /* Z-order                  */
		IDW_HSCR,               /* window ID                */
		NULL,                   /* control data             */
		NULL) ;                 /* presentation parameter   */

    hwndCorner = WinCreateWindow(
                hwndClient,             /* parent window handle     */
		WC_STATIC,              /* window class             */
		"",                     /* window text              */
		SS_BKGNDRECT,           /* window style             */
		0, 0, 0, 0,             /* position and size        */
		hwndClient,             /* owner window handle      */
		HWND_TOP,               /* Z-order                  */
		IDW_CORNER,             /* window ID                */
		NULL,                   /* control data             */
		NULL) ;                 /* presentation parameter   */

    if (hwndVScroll == NULLHANDLE || hwndHScroll == NULLHANDLE || hwndCorner == NULLHANDLE) {
        TRACE("failed to create scroll-bars\n") ;
        if (hwndVScroll != NULLHANDLE) WinDestroyWindow(hwndVScroll) ;
	if (hwndHScroll != NULLHANDLE) WinDestroyWindow(hwndHScroll) ;
	if (hwndCorner  != NULLHANDLE) WinDestroyWindow(hwndCorner)  ;
	hwndVScroll = hwndHScroll = hwndCorner = NULLHANDLE ;
        return FALSE ;
    }

    svCxVScroll = WinQuerySysValue(HWND_DESKTOP, SV_CXVSCROLL) ;
    svCyHScroll = WinQuerySysValue(HWND_DESKTOP, SV_CYHSCROLL) ;

    return TRUE ;
}

/* 
 * dispDone - finish display management
 */

void    dispDone(HWND hwnd)
{
    if (hwndVScroll != NULLHANDLE) {
        WinDestroyWindow(hwndVScroll) ;
	hwndVScroll = NULLHANDLE ;
    }
    if (hwndHScroll != NULLHANDLE) {
        WinDestroyWindow(hwndHScroll) ;
	hwndHScroll = NULLHANDLE ;
    }
}

/*
 * marks size of display objects
 */

#define OVER_CX     4       /* overhead for table area  */
#define OVER_CY     4       /* overhead for table area  */
#define SLACK_CX    2       /* slack on cell size       */
#define SLACK_CY    2       /* slack on cell size       */
#define OFFSET_X    2       /* character offset in cell */
#define OFFSET_Y    2       /* character offset in cell */

static  SIZEL   szCell   = { 0 } ;
static  SIZEL   szClient = { 0 } ;
static  SIZEL   szTablet = { 0 } ;

static  void    getCellSize(void)
{
    winQueryCharSize(&szCell) ;
    szCell.cx += SLACK_CX ;
    szCell.cy += SLACK_CY ;
}

static  void    setClientSize(SHORT cx, SHORT cy)
{
    szClient.cx = (LONG) cx ;
    szClient.cy = (LONG) cy ;
}

static  void    setTabletSize(TABLPTR tab)
{
    LONG    ns, nl, nc ;
    LINEPTR lp ;

    if (tab == NULL) {
        return ;
    }

    nl = nc = 0 ;
    
    for (lp = tab->line ; lp != NULL ; lp = lp->next) {
        ns = strlen(lp->data) / 2 ;     /* assume DBCS */
        nl += 1 ;
	nc = (ns > nc ? ns : nc) ;
    }
    szTablet.cx = szCell.cx * nc + OVER_CX * 2 ;
    szTablet.cy = szCell.cy * nl + OVER_CY * 2 ;
}

/*
 * position of tablet
 */

static  POINTL  ptTablet = { 0, 0 } ;   /* upper left corner of tablet  */
static  LONG    maxHorz = 0 ;           /* Horz. scroll range       */
static  LONG    maxVert = 0 ;           /* Vert. scroll range       */
static  LONG    adjHorz = 0 ;           /* Horz. current position   */
static  LONG    adjVert = 0 ;           /* Vert. current position   */

static  void    adjustPosition(void)
{
    BOOL    needHScr, needVScr ;
    LONG    mHorz, mVert ;

    needHScr = (szTablet.cx <= szClient.cx) ? FALSE : TRUE ;
    needVScr = (szTablet.cy <= szClient.cy) ? FALSE : TRUE ;
    mVert = mHorz = 0 ;
    
    if (needHScr) {
        mVert = svCyHScroll ;
	needVScr = ((szTablet.cy + mVert) <= szClient.cy) ? FALSE : TRUE ;
    }
    if (needVScr) {
        mHorz = svCxVScroll ;
	needHScr = ((szTablet.cx + mHorz) <= szClient.cx) ? FALSE : TRUE ;
    }

    if ((szTablet.cx + mHorz) <= szClient.cx) {
        maxHorz = adjHorz = 0 ;
        ptTablet.x = (szClient.cx - szTablet.cx - mHorz) / 2 ;
    } else {
        maxHorz = szTablet.cx - (szClient.cx - mHorz) ;
	adjHorz = (adjHorz > maxHorz) ? maxHorz : adjHorz ; 
	ptTablet.x = -adjHorz ;
    }

    if ((szTablet.cy + mVert) <= szClient.cy) {
	maxVert = adjVert = 0 ;
        ptTablet.y = szClient.cy - (szClient.cy - szTablet.cy - mVert) / 2 ;
    } else {
        maxVert = szTablet.cy - (szClient.cy - mVert) ;
	adjVert = (adjVert > maxVert) ? maxVert : adjVert ;
	ptTablet.y = szClient.cy + adjVert ;
    }
    
    if (needHScr == FALSE || needVScr == FALSE) {
        WinShowWindow(hwndCorner, FALSE) ;
    } else {
        WinSetWindowPos(hwndCorner, HWND_TOP, 
	    (szClient.cx - mHorz), 0, mHorz, mVert,
	    (SWP_MOVE | SWP_SIZE | SWP_SHOW | SWP_ZORDER)) ;
    }

    if (! needHScr) {
        WinShowWindow(hwndHScroll, FALSE) ;
    } else {
	WinSendMsg(hwndHScroll, SBM_SETTHUMBSIZE,
	    MPFROM2SHORT(szClient.cx, szTablet.cx), NULL) ;
        WinSendMsg(hwndHScroll, SBM_SETSCROLLBAR,
	    MPFROM2SHORT(adjHorz, 0), MPFROM2SHORT(0, maxHorz)) ;
        WinSetWindowPos(hwndHScroll, HWND_TOP, 
	    0, 0, (szClient.cx - mHorz), mVert,
	    (SWP_MOVE | SWP_SIZE | SWP_SHOW | SWP_ACTIVATE | SWP_ZORDER)) ;
    }

    if (! needVScr) {
        WinShowWindow(hwndVScroll, FALSE) ;
    } else {
	WinSendMsg(hwndVScroll, SBM_SETTHUMBSIZE,
	    MPFROM2SHORT(szClient.cy, szTablet.cy), NULL) ;
	WinSendMsg(hwndVScroll, SBM_SETSCROLLBAR,
	    MPFROM2SHORT(adjVert, 0), MPFROM2SHORT(0, maxVert)) ;
        WinSetWindowPos(hwndVScroll, HWND_TOP, 
	    (szClient.cx - mHorz), mVert, mHorz, (szClient.cy - mVert),
            (SWP_MOVE | SWP_SIZE | SWP_SHOW | SWP_ACTIVATE | SWP_ZORDER)) ;
    }
}

void    dispChangeSize(SHORT cx, SHORT cy, SHORT height)
{
    setClientSize(cx, height) ;
    adjustPosition() ;
}

void    dispCalcSize(TABLPTR tab)
{
    getCellSize()      ;
    setTabletSize(tab) ;
    adjustPosition()   ;
}

BOOL    dispScrollHorz(USHORT cmd, SHORT pos)
{
    LONG    new        ;
    BOOL    set = TRUE ;

    switch (cmd) {
    case SB_SLIDERTRACK    :
    case SB_SLIDERPOSITION :
        new = pos   ;
	set = FALSE ;
	break ;
    case SB_LINELEFT :
        new = adjHorz - (szCell.cx / 4) ;
	break ;
    case SB_LINERIGHT :
        new = adjHorz + (szCell.cx / 4) ;
	break ;
    case SB_PAGELEFT :
        new = adjHorz - szCell.cx ;
	break ;
    case SB_PAGERIGHT :
        new = adjHorz + szCell.cx ;
	break ;
    default :
        return ;
    }

    if (new < 0)       new = 0       ;
    if (new > maxHorz) new = maxHorz ;

    if (set) {
        WinSendMsg(hwndHScroll, SBM_SETSCROLLBAR,
	    MPFROM2SHORT(new, 0), MPFROM2SHORT(0, maxHorz)) ;
    }
    if (new == adjHorz) {
        return FALSE ;
    }
    ptTablet.x -= (new - adjHorz) ;
    adjHorz = new ;
    return TRUE   ;
}

BOOL    dispScrollVert(USHORT cmd, SHORT pos)
{
    LONG    new        ;
    BOOL    set = TRUE ;
    
    switch (cmd) {
    case SB_SLIDERTRACK    :
    case SB_SLIDERPOSITION :
        new = pos   ;
	set = FALSE ;
	break ;
    case SB_LINEUP :
        new = adjVert - (szCell.cy / 4) ;
	break ;
    case SB_LINEDOWN :
        new = adjVert + (szCell.cy / 4) ;
	break ;
    case SB_PAGEUP :
        new = adjVert - szCell.cy ;
	break ;
    case SB_PAGEDOWN :
        new = adjVert + szCell.cy ;
	break ;
    default :
        return ;
    }

    if (new < 0)       new = 0       ;
    if (new > maxVert) new = maxVert ;

    if (set) {
        WinSendMsg(hwndVScroll, SBM_SETSCROLLBAR,
	    MPFROM2SHORT(new, 0), MPFROM2SHORT(0, maxVert)) ;
    }
    if (new == adjVert) {
        return FALSE ;
    }
    ptTablet.y += (new - adjVert) ;
    adjVert = new ;
    return TRUE   ;
}

/*
 * paint now
 */

void    dispPaintTablet(HPS hps, TABLPTR tab, PRECTL rct)
{
    int     row, col, nRow, nCol ;
    POINTL  pt1, pt2 ;
    LINEPTR lp ;
    PUCHAR  cp ;
    LONG    color ;

    WinFillRect(hps, rct, GpiQueryBackColor(hps)) ;
    
    /*
     * paint characters, with counting max Line/Col
     */
     
    pt1.y = ptTablet.y - OVER_CY ;
    nRow = nCol = 0 ;

    for (lp = tab->line ; lp != NULL ; lp = lp->next) {

        pt1.y -= szCell.cy ;
	pt1.x = ptTablet.x + OVER_CX ;
	nRow += 1 ;

	cp  = lp->data ;
	col = 0        ;

	while (cp != NULL && cp[0] != '\0' && cp[1] != '\0') {
	    pt2.x = pt1.x + OFFSET_X ;
	    pt2.y = pt1.y + OFFSET_Y ;
	    GpiCharStringAt(hps, &pt2, 2, cp) ;
	    pt1.x += szCell.cx ;
	    cp  += 2 ;
	    col += 1 ;
	}
        nCol = (col > nCol) ? col : nCol ;
    }

    /*
     * paint grids
     */

    color = GpiQueryColor(hps)   ;
    GpiSetColor(hps, 0x00c0c0c0) ;
    
    /*
     * draw horizontal lines
     */
     
    pt1.y = pt2.y = ptTablet.y - OVER_CY ;
    pt1.x = ptTablet.x + OVER_CX ;
    pt2.x = pt1.x + szTablet.cx - OVER_CX * 2 ;
    
    for (row = 0 ; row <= nRow ; row++) {
        GpiMove(hps, &pt1) ;
	GpiLine(hps, &pt2) ;
	pt1.y -= szCell.cy ;
	pt2.y -= szCell.cy ;
    }
     
    /*
     * draw vertical lines
     */
     
    pt1.x = pt2.x = ptTablet.x + OVER_CX ;
    pt1.y = ptTablet.y - OVER_CY ;
    pt2.y = pt1.y - szTablet.cy + OVER_CY * 2 ;
    
    for (col = 0 ; col <= nCol ; col++) {
        GpiMove(hps, &pt1) ;
        GpiLine(hps, &pt2) ;
	pt1.x += szCell.cx ;
	pt2.x += szCell.cx ;
    }
    
    GpiSetColor(hps, color) ;
}

/*
 * dispHitChar - check if position matches any char
 */
 
BOOL    dispHitChar(HPS hps, TABLPTR tp, SHORT x, SHORT y, PUCHAR buff)
{
    LONG    x1, x2, y1, y2 ;
    LINEPTR lp ;
    PUCHAR  cp ;
    POINTL  pt1, pt2 ;
    LONG    fg, bg   ;

    if (buff != NULL) {
        buff[0] = '\0' ;
    }
    if (x <= ptTablet.x || x >= (ptTablet.x + szTablet.cx)) {
        return FALSE ;
    }
    if (y >= ptTablet.y || y <= (ptTablet.y - szTablet.cy)) {
        return FALSE ;
    }

    /*
     * look for matching line 
     */
    
    y1 = ptTablet.y - OVER_CY ;
    for (lp = tp->line ; lp != NULL ; lp = lp->next, y1 = y2) {
        if (y < y1 && y > (y2 = y1 - szCell.cy)) {
	    break ;
	}
    }
    if (lp == NULL) {
        return FALSE ;
    }

    /*
     * look for matching column
     */

    x1 = ptTablet.x + OVER_CX ;
    for (cp = lp->data ; cp[0] != '\0' && cp[1] != '\0' ; cp += 2, x1 = x2) {
        if (x > x1 && x < (x2 = x1 + szCell.cx)) {
	    break ;
	}
    }
    if (cp[0] == '\0' || cp[1] == '\0') {
        return FALSE ;
    }

    /*
     * get matched char
     */

    if (buff != NULL) {
        buff[0] = cp[0] ;
        buff[1] = cp[1] ;
	buff[2] = '\0'  ;
    }

    /*
     * flush matched char
     */
    
    if (hps != NULLHANDLE) {

        pt1.x = x1 + 1, pt1.y = y2 + 1 ;
        pt2.x = x2 - 1, pt2.y = y1 - 1 ;

        fg = GpiQueryColor(hps)     ;
        bg = GpiQueryBackColor(hps) ;

        GpiSetMix(hps, FM_XOR) ;
        GpiSetColor(hps, CLR_PALEGRAY) ;

        GpiMove(hps, &pt1) ;
        GpiBox(hps, DRO_FILL, &pt2, 0, 0) ;
        DosSleep(20) ;
        GpiMove(hps, &pt1) ;
        GpiBox(hps, DRO_FILL, &pt2, 0, 0) ;

	GpiSetMix(hps, FM_DEFAULT) ;
        GpiSetColor(hps, fg) ;
    }
    
    return TRUE ;
}
