/*
 * magnify.c - show magnified character
 */

#define INCL_PM
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tablet.h"
#include "resource.h"

/*
 * create bitmap for character
 */

static  HBITMAP bitmapFromChar(HWND hwnd, PUCHAR ch, 
                    SHORT cx, SHORT cy, PUCHAR fnt, ULONG fore, ULONG back)
{
    HAB     hab ;
    HDC     hdc ;
    HPS     hps ;
    SIZEL   siz ;
    POINTL  pt1 ;
    POINTL  pt2 ;
    HBITMAP hbm ;
    BITMAPINFOHEADER2   bmih2  ;
    PBITMAPINFOHEADER2  pbmih2 ;

    siz.cx = cx ;
    siz.cy = cy ;    

    memset((pbmih2 = &bmih2), 0, sizeof(bmih2)) ;
    pbmih2->cbFix = sizeof(BITMAPINFOHEADER2) ;
    pbmih2->cx = siz.cx ;
    pbmih2->cy = siz.cy ;
    pbmih2->cPlanes   = 1  ;
    pbmih2->cBitCount = 24 ;

    hab = WinQueryAnchorBlock(hwnd) ;
    hdc = DevOpenDC(hab, OD_MEMORY, "*", 0, NULL, NULLHANDLE) ;
    hps = GpiCreatePS(hab, hdc, &siz,
            PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC) ;
    hbm = GpiCreateBitmap(hps, (PBITMAPINFOHEADER2) pbmih2, 0, NULL, NULL) ;
    GpiSetBitmap(hps, hbm) ;

    GpiCreateLogColorTable(hps, 0, LCOLF_RGB, 0, 0, NULL) ;

    GpiSetColor(hps, back) ;
    pt1.x = 0, pt1.y = 0 ;
    pt2.x = siz.cx, pt2.y = siz.cy ;
    GpiMove(hps, &pt1) ;
    GpiBox(hps, DRO_OUTLINEFILL, &pt2, 0, 0) ;

    fontSetFont(hps, fnt) ;

    GpiSetColor(hps, fore) ;
    pt1.x = 0, pt1.y = 1 ;
    GpiCharStringAt(hps, &pt1, 2, ch) ;

    GpiDestroyPS(hps) ;
    DevCloseDC(hdc)   ;
    
    return hbm ;
}

/*
 * control block for magnified character
 */

typedef struct _MAGREC {
    HWND    frame  ;
    HWND    client ;
    HWND    popup  ;
    HBITMAP hbm    ;
    SHORT   cx     ;
    SHORT   cy     ;
    RECTL   cbox   ;
} MAGREC, *MAGPTR ;

/*
 * magSize - determine char box size & position
 */

static  void    magSize(HWND hwnd, MAGPTR mag, SHORT x, SHORT y)
{
    double  scale, xscale, yscale ;
    int     xdiff, ydiff ;

    /*
     * Calc. Scaling Factor
     */

    xscale = (double) x / (double) mag->cx ;
    yscale = (double) y / (double) mag->cy ;
    scale = (xscale < yscale) ? xscale : yscale ;

    /*
     * Scale Current Glyph
     */

    mag->cbox.xLeft   = 0 ;
    mag->cbox.yBottom = 0 ;
    mag->cbox.xRight  = (LONG) ((double) mag->cx * scale - 0.5) ;
    mag->cbox.yTop    = (LONG) ((double) mag->cy * scale - 0.5) ;

    /*
     * place char box to center
     */

    xdiff = x - (mag->cbox.xRight - mag->cbox.xLeft) ;
    ydiff = y - (mag->cbox.yTop - mag->cbox.yBottom) ;
    mag->cbox.xLeft   += (xdiff / 2) ;
    mag->cbox.yBottom += (ydiff / 2) ;
    mag->cbox.xRight  += (xdiff / 2) ;
    mag->cbox.yTop    += (ydiff / 2) ;

#ifdef  DEBUG
    printf("Magnify to %dx%d at (%d,%d)\n",
            (mag->cbox.xRight - mag->cbox.xLeft),
	    (mag->cbox.yTop - mag->cbox.yBottom),
	    mag->cbox.xLeft, mag->cbox.yBottom) ;
#endif
}

/*
 * magDraw - draw character
 */

static  void    magDraw(HWND hwnd, MAGPTR mag, HPS hps)
{
    RECTL   rct ;
    POINTL  apt[4] ;
    LONG    x, y, cx, cy ;
    double  sizex, sizey, stepx, stepy ;
    POINTL  pts, pte ;

    /*
     * clear background
     */
     
    WinQueryWindowRect(hwnd, &rct) ;
    WinFillRect(hps, &rct, CLR_DARKGRAY) ;

    /*
     * Draw Bitmap with Scaling
     */

    apt[0].x = mag->cbox.xLeft   ;  /* Target   */
    apt[0].y = mag->cbox.yBottom ;
    apt[1].x = mag->cbox.xRight  ;
    apt[1].y = mag->cbox.yTop    ;
    apt[2].x = 0                 ;  /* Source   */
    apt[2].y = 0                 ;
    apt[3].x = mag->cx           ;
    apt[3].y = mag->cy           ;
    GpiWCBitBlt(hps, mag->hbm, 4, apt, ROP_SRCCOPY, 0) ;

    /*
     * Draw Grids
     */

    GpiSetColor(hps, CLR_PALEGRAY) ;

    cx = (mag->cbox.xRight - mag->cbox.xLeft + 1) ;
    cy = (mag->cbox.yTop - mag->cbox.yBottom + 1) ;
    
    pts.x = mag->cbox.xLeft  ; 
    pte.x = mag->cbox.xRight ;
    for (y = 0 ; y <= mag->cy ; y++) {
        pts.y = pte.y = mag->cbox.yBottom + (cy * y / mag->cy) ;
	GpiMove(hps, &pts) ;
	GpiLine(hps, &pte) ;
    }

    pts.y = mag->cbox.yTop    ;
    pte.y = mag->cbox.yBottom ;
    for (x = 0 ; x <= mag->cx ; x++) {
        pts.x = pte.x = mag->cbox.xLeft + (cx * x / mag->cx) ;
	GpiMove(hps, &pts) ;
	GpiLine(hps, &pte) ;
    }
}

/*
 * popup menu
 */

static  void    magPopup(HWND hwnd, MAGPTR mag)
{
    LONG    sel ;
    ULONG   opt ;
    POINTL  pt  ;
    
    if (mag == NULL || mag->popup == NULLHANDLE) {
        return ;
    }

    WinQueryPointerPos(HWND_DESKTOP, &pt) ;
    WinMapWindowPoints(HWND_DESKTOP, hwnd, &pt, 1) ;
    
    sel = IDM_MAG_CODE ;
    opt = PU_POSITIONONITEM| PU_HCONSTRAIN | PU_VCONSTRAIN |
            PU_MOUSEBUTTON2DOWN |
            PU_KEYBOARD | PU_MOUSEBUTTON1 | PU_MOUSEBUTTON2 ;
    
    WinPopupMenu(mag->frame, mag->frame, mag->popup, pt.x, pt.y, sel, opt) ;
}

/*
 * window procedure
 */

#define WM_MAG_INIT     (WM_USER + 1)

MRESULT EXPENTRY magWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MAGPTR  mag ;
    HPS     hps ;
    
    mag = (MAGPTR) WinQueryWindowPtr(hwnd, 0) ;
    
    switch (msg) {
    
    case WM_MAG_INIT :
        mag = (MAGPTR) PVOIDFROMMP(mp1) ;
        WinSetWindowPtr(hwnd, 0, (PVOID) mag) ;
	return (MRESULT) 0 ;
	
    case WM_DESTROY  :
        if (mag != NULL) {
            if (mag->hbm != NULLHANDLE) GpiDeleteBitmap(mag->hbm) ;
	    free(mag) ;
	}
	return (MRESULT) 0 ;
	
    case WM_SIZE :
        magSize(hwnd, mag, SHORT1FROMMP(mp2), SHORT2FROMMP(mp2)) ;
	return (MRESULT) 0 ;

    case WM_PAINT    :
        hps = WinBeginPaint(hwnd, NULLHANDLE, NULL) ;
	magDraw(hwnd, mag, hps) ;
	WinEndPaint(hps) ;
	return (MRESULT) 0 ;
	
    case WM_BUTTON1DOWN :
        WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(IDM_MAG_MOVE), NULL) ;
        return WinDefWindowProc(hwnd, msg, mp1, mp2) ;
        
    case WM_BUTTON2DOWN :
        if (mag != NULL) {
            magPopup(hwnd, mag) ;
        }
	return (MRESULT) 0 ;

    case WM_BUTTON1DBLCLK :
    case WM_BUTTON2DBLCLK :
        WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(IDM_MAG_EXIT), NULL) ;
	return WinDefWindowProc(hwnd, msg, mp1, mp2) ;

    case WM_COMMAND :
        if (mag == NULL) {
	    return (MRESULT) 0 ;
	}
        switch (SHORT1FROMMP(mp1)) {
        case IDM_MAG_MOVE :
	    WinSendMsg(mag->frame, WM_TRACKFRAME, MPFROMLONG(TF_MOVE), NULL) ;
	    return (MRESULT) 0 ;
        case IDM_MAG_EXIT :
	    WinDestroyWindow(mag->frame) ;
	    return (MRESULT) 0 ;
        }
	return (MRESULT) 0 ;
    }
    return WinDefWindowProc(hwnd, msg, mp1, mp2) ;
}

/*
 * entry of Magnify
 */

static  BOOL    registed = FALSE ;

static  USHORT  sjis2jis(USHORT c1, USHORT c2)
{
    c1 -= (c1 >= 0xa0) ? 0xc1 : 0x81 ;
    if (c2 >= 0x9f) {
        c1 = c1 + c1 + 0x22 ;
        c2 -= 0x7e ;
    } else {
        c1 = c1 + c1 + 0x21 ;
        c2 -= (c2 <= 0x7e) ? 0x1f : 0x20 ;
    }
    return ((c1 << 8) & 0xff00) + (c2 & 0x00ff) ;
}

#define CX_MIN  128
#define CY_MIN  128

#define CX_MIN  128
#define CY_MIN  128

static  SHORT   nMagnify(SHORT cx, SHORT cy)
{
    SHORT   m ;

    for (m = 1 ; m < CX_MIN && m < CY_MIN ; m++) {
        if ((cx * m) >= CX_MIN && (cy * m) >= CY_MIN) {
            break ;
        }
    }
    return m ;
}

void    magnifyChar(HWND hwnd, PUCHAR ch, PUCHAR fnt, PULONG clr)
{
    MAGPTR  mag ;
    SIZEL   siz ;
    HWND    frame  ;
    HWND    client ;
    ULONG   flFrameStyle ;
    ULONG   flFrameFlags ;
    USHORT  c1, c2, code ;
    UCHAR   buff[32] ;
    LONG    cx, cy, m ;
    POINTL  pt ;
    
    /*
     * register window class for magnify
     */
     
    if (registed == FALSE) {
        WinRegisterClass(WinQueryAnchorBlock(hwnd),
	    "magnify", magWndProc, CS_SIZEREDRAW, sizeof(MAGPTR)) ;
        registed = TRUE ;
    }

    /*
     * create magnify control block
     */
    
    if ((mag = (MAGPTR) malloc(sizeof(MAGREC))) == NULL) {
        return ;
    }
    memset(mag, 0, sizeof(MAGREC)) ;
    
    mag->frame  = NULLHANDLE ;
    mag->client = NULLHANDLE ;
    mag->popup  = NULLHANDLE ;

    winQueryCharSize(&siz) ;
    mag->cx = (SHORT) siz.cx ;
    mag->cy = (SHORT) siz.cy ;
    mag->hbm = bitmapFromChar(hwnd, ch, siz.cx, siz.cy, fnt, clr[0], clr[1]) ;
    
    if (mag->hbm == NULLHANDLE) {
        GpiDeleteBitmap(mag->hbm) ;
	free(mag) ;
	return ;
    }

    /*
     * create magnify window
     */

    flFrameStyle = 0 ;
    flFrameFlags = FCF_SIZEBORDER ;

    mag->frame = WinCreateStdWindow(
            HWND_DESKTOP,               /* parent window handle     */
	    flFrameStyle,               /* frame window style       */
	    &flFrameFlags,              /* frame create flags       */
            "magnify",                  /* client window class      */
	    "",                         /* as frame title           */
            0,                          /* client window style      */
            NULLHANDLE,                 /* use internal resource    */
	    ID_MAGNIFY,                 /* window ID                */
	    &mag->client) ;             /* client window handle     */

    if (mag->frame == NULLHANDLE || mag->client == NULLHANDLE) {
        if (mag->client != NULLHANDLE) WinDestroyWindow(mag->client) ;
        if (mag->frame  != NULLHANDLE) WinDestroyWindow(mag->frame)  ;
	GpiDeleteBitmap(mag->hbm) ;
	free(mag) ;
	return ;
    }

    mag->popup = WinLoadMenu(mag->client, NULLHANDLE, ID_MAGNIFY) ;

    if (mag->popup == NULLHANDLE) {    
        WinDestroyWindow(mag->client) ;
        WinDestroyWindow(mag->frame)  ;
	GpiDeleteBitmap(mag->hbm) ;
	free(mag) ;
	return ;
    }
    
    c1 = (USHORT) ch[0] ;
    c2 = (USHORT) ch[1] ;    

    code = ((c1 << 8) & 0xff00) + (c2 & 0x00ff) ;
    sprintf(buff, "SJIS %04x", code) ;
    WinSendMsg(mag->popup, MM_SETITEMTEXT,
	            MPFROMSHORT(IDM_MAG_SJIS), MPFROMP(buff)) ;

    code = sjis2jis(c1, c2) ;
    sprintf(buff,  "JIS  %04x", code) ;
    WinSendMsg(mag->popup, MM_SETITEMTEXT,
	            MPFROMSHORT(IDM_MAG_JIS), MPFROMP(buff)) ;

    code |= 0x8080 ;
    sprintf(buff,  "EUC  %04x", code) ;
    WinSendMsg(mag->popup, MM_SETITEMTEXT,
	            MPFROMSHORT(IDM_MAG_EUC), MPFROMP(buff)) ;

    /*
     * calc. magnifier
     */
    
    m = nMagnify(mag->cx, mag->cy) ;
    cx = mag->cx * m ;
    cy = mag->cy * m ;

    WinQueryPointerPos(HWND_DESKTOP, &pt) ;
    pt.x -= (cx / 2) ;
    pt.y -= (cy - 2) ;
     
    /*
     * related MAGREC to window and show
     */
    
    WinSendMsg(mag->client, WM_MAG_INIT, MPFROMP(mag), NULL) ;
    WinSetWindowPos(mag->frame, HWND_TOP,
            pt.x, pt.y, cx, cy,
	    SWP_ACTIVATE | SWP_SHOW | SWP_ZORDER | SWP_MOVE | SWP_SIZE) ;
}
