/* Filename: copy-log.cmd                                                  */
/* Version:  0.2                                                           */
/*                                                                         */
/* Description : This script archives all files with a given suffix,       */
/*               *.log for example,  to a specified  directory by using    */
/*               the zip-utility. The filename of the archive is beeing    */
/*               composed of the current time and date automaticly.        */
/*               Each line in the optional command files will be executed  */
/*               before (before.clc) Copy-Log starts its job and/or        */
/*               afterwards (after.clc).                                   */
/*                                                                         */
/* Installation: none ( but the program 'zip.exe' is expected to be in the */
/*                      in the path (see procedure init) ).                */               
/*                                                                         */
/* Usage:        use 'copy-log.cmd -h' for help                            */
/*                                                                         */
/*                                        (w) by Alexander Hoff            */
/*                                                N.E.W.S. GbR             */
/*                                                Herrenhof 13             */
/*                                                41515 Grevenbroich       */
/*                                                Germany                  */
/*                                                Email: AHoff@news-gbr.de */
/*                                                Tel.: +49-2181-2426-30   */
/*                                                                         */

      /* load  REXX functions */
if RxFuncQuery('SysLoadFuncs') then do
 call RxFuncAdd 'SysLoadFuncs','REXXUTIL','SysLoadFuncs'
 call SysLoadFuncs 
end

     /* we don't need echo */
'@echo off'

    /* get commandline arguments */
parse upper arg arguments

call Init /* set some environment variables like the exe-name of the archive- */
          /* utility, script version, etc. */

    /* help requested? */
if (WORD(arguments,1)="-H") | (WORD(arguments,1)="-?") | (WORD(arguments,1)="/H") then signal usage

   /* check, if TMP environment is set. We will need it for our tmp directory */
call CheckTMP

   /* check given parameters */
call CheckParameters

   /* okay, all checks were done. Therefor we should store now the given */
   /* parameters in variables for a better handling.  */
call ParameterToVar

   /* Execute each line in the file 'cbfilename' */
   /* as a separate command. */
rc=ExecuteCommands(cbfilename)

    /* copy all files matching the given extension to a tmp directory */
tmpdir=CopyFiles()
  
    /* archive all files in the tmp directory which were created by */
    /* the procedure copyfiles. */
call ArchiveFiles

    /* delete the tmp-directory. We don't need it anymore */
call Deletetmpdir

   /* Execute each line in the file 'cafilename' */
   /* as a separate command. */
rc=ExecuteCommands(cafilename)
 
   /* finished */
exit


/* ---------------------------------------------------------------- */
/*              ======================================              */
/*              ===== Procedures and Subroutines =====              */
/*              ======================================              */
/* ---------------------------------------------------------------- */

Init:
     /* first, set some environment variables */
scriptname='Copy-Log.cmd'
scriptversion='0.2'

archiveutility='zip.exe'
recursive='-r'       /* parameter of the zip-utility for recursive operation */

maxcompression='-9'  /* parameter of the zip-utility */

allfiles='*.*'       /* wildcard */

cbfilename='before_run.clc'    /* since some files, like the log file of the */
                               /* apache web server, are locked when the     */
                               /* program is beeing executed, it may be      */
                               /* necessary to execute some commands or      */
                               /* programm-calls before 'Copy-Log' can copy  */
                               /* and archive the specified files. Therefor  */ 
                               /* 'Copy-Log' will execute each line in the   */
                               /* file 'cbfilename' as a separate command.   */

cafilename='after_run.clc'    /* The user may want to restart the programs    */
                              /* stopped via 'cbfilename'or do something else */
                              /* after 'Copy-Log' has finished its job. */


      /* define the command-strings which can be used in the both */
      /* command files (cbfilname and cafilename) */
pausefortoken='pausefor'    
remtoken=';'

tmpfilename='COPY-LOG.???' /* this is used as a template when generating */
                           /* the name for the temporary directory */

currentdir=GetCurrentDir()

currenttime=GetTime()
currentdate=Date('S')

return 0

/* -------------------------------------- */
GetCurrentDir: procedure
   /* get current drive and directory */

parse upper source . . sourcefile
prgdrive = filespec('DRIVE', sourcefile)
prghelp = filespec('PATH', sourcefile)
prgpath = prgdrive''prghelp

return prgpath

/* -------------------------------------- */
GetTime: procedure
    /* get the current time and change the time format */

currenttime=TIME()

   /* change time format from hh:mm:ss to hhmmss */
currenttime=DELSTR(currenttime,3,1)  
currenttime=DELSTR(currenttime,5,1)  

return currenttime

/* ---------------------------------------------------------------- */
usage:
      /* our help screen */

call SysCls
say'                          'scriptname' v'scriptversion' '    
say'                         -------------------             '
say'                by Alexander Hoff (AHoff@news-gbr.de)    '
say''
say'Description (see readme.txt for details!): '
say'   This script archives all files with a given suffix to a specified    '
say'   directory by using the zip-utility. The filename of the archive is   '
say'   beeing composed of the current time and date automaticly.  '
say''
say 'Usage:                                                                '
say '   'scriptname' <file-ext> <source directory> <target directory>            '
say ''
say 'where              '
say '   file-ext          is the file extension of the files you want to archive.'
say '   source directory  is the directory where the files which should be '
say '                     archived are located.'
say '   target directory  is the directory where the archive will be created. '
say ''
say 'Example:  '
say '   'scriptname' log C:\ETC E:\LOG'
say '   Archives all files in the directory C:\ETC matching *.log to'
say '   a zip-compressed file in the directory E:\LOG.  '

exit

/* ---------------------------------------------------------------- */
CheckTMP:
   /* check, if the tmp environment variable is set */

tmppath=GetValue('TMP')
  /* maybe someone uses the unix-convention for the TMP environment. To be    */
  /* on the save side we convert all front slashes ('/') to back slashes ('\')*/
tmppath=TRANSLATE(tmppath,'\','/')

if tmppath='' then do
  say ''
  say ' Error:'
  say '   TMP environment variable not set!'
  say ''
  say ' Please set the TMP-environment in your config.sys. It must point to an'
  say ' existing directory.'
  say '    For Example: SET TMP=D:\TMP\BAK'
  say ''
  exit
end /* if */

return 0

/* ---------------------------------------------------------------- */
CheckParameters: 
   /* check, whether the given parameters or correct or not */

     /* three parameters passed? */
select
   when WORD(arguments,1)="" then do
       call PrintErrHeader
       say ' Error:'
       say '   No arguments passed. '
       say ''
       say ' Type -h or -? for help.'
       say ''
       exit
   end
   when WORD(arguments,2)="" then do
       call PrintErrHeader
       say ' Error:'
       say '   No source directory specified. '
       say ''
       say ' Type -h or -? for help.'
       say ''
       exit
   end
   when WORD(arguments,3)="" then do
       call PrintErrHeader
       say ' Error:'
       say '   No target directory specified. '
       say ''
       say ' Type -h or -? for help.'
       say ''
       exit
   end
   otherwise
     /* nothing to do */
end /* select */

   /* no need to check the first parameter. Most file extensions are valid. */

   /* is the second argument a existing directory as required? */
help=WORD(arguments,2)
if (DirExists(help)<>0) then do
       call PrintErrHeader
       say ' Error:'
       say '   The source directory you specified does not exist. '
       say ''
       say ' Type -h or -? for help.'
       say ''
       exit
end /* if */

   /* is the third argument a existing directory as required? */
help=WORD(arguments,3)
if (DirExists(help)<>0) then do
       call PrintErrHeader
       say ' Error:'
       say '   The target directory you specified does not exist. '
       say ''
       say ' Type -h or -? for help.'
       say ''
       exit
end /* if */

return 0
/* ---------------------------------------------------------------- */
PrintErrHeader:
    /* Print the header of an error message on the screen */

say ''
say'                    ---------------------            '
say'                    - 'scriptname' v'scriptversion' - '    
say'                    ---------------------            '
say ''

return 0
/* ---------------------------------------------------------------- */
ParameterToVar:
   /* store all given commandline arguments in variables */
   /* (filter, sourcedir, targetdir)  */

   /* convert the given file extension to a regular wildcard expression */
filextension = WORD(arguments,1)
wildcard = '*.'
filter = wildcard''filextension

sourcedir=WORD(arguments,2)
  /* erase trailing back-slash */
sourcedir=deletebackslash(sourcedir)

targetdir=WORD(arguments,3)
  /* erase trailing back-slash */
targetdir=deletebackslash(targetdir)

return 0

/* ---------------------------------------------------------------- */
ExecuteCommands: procedure expose currentdir remtoken pausefortoken

parse arg commandfile .

   /* check, if the configuration file exists */
filename = INSERT(currentdir,commandfile)
result = STREAM(filename,'C','QUERY EXISTS') 

if result ='' then return 0 /* Since the file doesn't exist, there is */
                            /* nothing to do */

   /* execute each line in the file 'commandfile' as a separate command */
  
   /* to be on the save side, we close the file before reading */
rc=stream(filename,'c','close')

do while lines(filename)
  currentcommand=linein(filename)
    /* only execute the given line, if it isn't a comment line */
  if (POS(remtoken,currentcommand)<>1) then do  
          /* check, if the current command is an internal copy-log command. */
          /* if so, execute it otherwise run the command as an os/2 */
          /* command. */   
      select
        when (POS(pausefortoken,currentcommand)<>0) then do
            /* pausefor command */
            secondsparsed=WORD(currentcommand,2)
               /* only call SysSleep if the user parsed a valid whole number */
            if (DATATYPE(secondsparsed,'W')=1) then call SysSleep(secondsparsed)                
        end /* when */
        otherwise currentcommand
      end /* select */       
  end /* if */
end /* do */

rc=stream(file,'c','close')
   
return 0

/* ---------------------------------------------------------------- */
CopyFiles: procedure expose filter sourcedir tmppath tmpfilename
   /* copy all files matching the given suffix to a temporary directory */

   /*first, we have to find a unique name for our temporary directory */
dirfound=0
do until dirfound
    /* this is not a nice way to find a temporary directory name, but */
    /* it will work. */
   help=tmppath
   help=INSERT(help,'\')
   help=INSERT(help,tmpfilename)
   help=SysTempFilename(help)
   help=TRANSLATE(help,' ','\')
   tmpdirname=WORD(help,WORDS(help))
     /* ... now we got something like copy-log.399 */
      
     /* use the above created unique filename as a directory name and */
     /* check, whether it exists or not */
   help=tmppath
   help=INSERT(help,'\')
   tmpdir=INSERT(help,tmpdirname)

     /* does the directory already exist? If not, we have found a name for */
     /* our tmp-directory and we will created this directory */
   if DirExists(tmpdir)<>0 then do
      dirfound=1
      rc=SysMkDir(tmpdir)
   end /* if */
   else dirfound=0
end /* do */
  
   /* now copy all files to our new temporary directory */
command='xcopy 'sourcedir'\'filter' 'tmpdir' '

command

return tmpdir

/* ---------------------------------------------------------------- */
ArchiveFiles:
   /* archive all files in the tmp directory which were created by */
   /* the procedure copyfiles. */

   /* we need a filename for our zip-archive. Current date and time */
   /* is a good approach... */
   /* Note: 'currenttime' and 'currentdate' were set in the */
   /*        subroutine init.  */
archivefilename=''targetdir'\'currentdate'-'currenttime''

    /* if the last character is a backslash ('\'), we have to delete it */
    /* The zip-utility doesn't like a trailing back slash.  */
tmpdir=deletebackslash(tmpdir)

command=''archiveutility' 'recursive' 'maxcompression' 'archivefilename' 'tmpdir' '

  /* execute 'command' now */
command

return 0
/* ---------------------------------------------------------------- */
Deletetmpdir:
   /* deletes the previous generated temporary directory */

say tmpdir

rc=SysFileTree(tmpdir'\*.*',dirs,'BS', '*****', '-----')

do i=dirs.0 to 1 by -1
  parse var dirs.i day time size att file
  file=strip(file,'L');
  rc=syssetobjectdata(file,'NODELETE=NO;')
   if pos('D', att) = 2 then do;
        rc1=sysrmdir(file);
        end;
  else rc=sysfiledelete(file);
 end
 rc=sysrmdir(tmpdir)

return 0

/*---------------------------------------------------------------------*/
/*              ==================================                     */
/*              ===== common used procedures =====                     */
/*              ==================================                     */
/*---------------------------------------------------------------------*/

GetValue: Procedure             
   /* Obtain an environment variable      */

Arg EnvVal .
res  = value(EnvVal,,'OS2ENVIRONMENT')

Return res

/*---------------------------------------------------------------------*/
DirExists: Procedure 
   /* check, whether a given directory does exist or not */

  parse arg testdir .

      /* if the last character is a backslash ('\'), we have to delete it */
    testdir=deletebackslash(testdir)

                         /* init the return code                       */
   thisRC = ""
 
                         /* install a temporary error handler to check */
                         /* if the drive with the directory to test is */
                         /* ready                                      */
   SIGNAL ON NOTREADY NAME DirDoesNotExist
 
                         /* check if the drive is ready                */
   call stream testDir || "\*", "D"
 
 
                         /* save the current directory of the current  */
                         /* drive                                      */
   curDir = directory()
 
                         /* save the current directory of the drive    */
                         /* with the directory to test                 */
   curDir1 = directory( fileSpec( "drive", testDir ) )
 
 
                         /* test if the directory exist                */
   thisRC = directory( testDir )
 
                         /* restore the current directory of the drive */
                         /* with the directory to test                 */
   call directory curDir1
 
                         /* restore the current directory of the       */
                         /* current drive                              */
   call directory curDir

   if (thisRC=testDir) then thisRC=0
   else thisRC=1

   Return thisRC

DirDoesNotExist:
 
Return 2
 
/*---------------------------------------------------------------------*/   
deletebackslash: procedure
   /* delete the trailing backshlash ('\') in given path, if existing */

parse arg givendir .

if (LASTPOS('\',givendir)=LENGTH(givendir) ) then do
    lastbackslashpos=LASTPOS('\',givendir)
    givendir=DELSTR(givendir,lastbackslashpos,1)  
end /* if */

return givendir
/*---------------------------------------------------------------------*/   
