//*****************************************************************************
//***  NAME              : ipformatx.c                                      ***
//***  SHORT DESCRIPTION : formats an iptrace file                          ***
//*****************************************************************************

#define INCL_BASE
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <stddef.h>
#include <time.h>
#include <ctype.h>
#include <string.h>

#define  OS2
#include <types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/ioctl.h>

#include <arpa\nameser.h>

#include <net\if.h>
#include <net\route.h>
#include <net\if_arp.h>
#include <netinet\in.h>
#include <netinet\in_systm.h>
#include <netinet\ip.h>
#include <netinet\ip_icmp.h>
#include <netinet\if_ether.h>
#include <netinet\tcp.h>
#include <netinet\udp.h>

#include <rpc\rpc.h>
#include <rpc\pmap_pro.h>

// packet types   (from IPSPY.H)
#define  IP_PACKET_TYPE                  1        // other
#define  REGULAR_1822_PACKET_TYPE        2
#define  HDH_1822_PACKET_TYPE            3
#define  DDN_X25_PACKET_TYPE             4
#define  RFC877_X25_PACKET_TYPE          5
#define  ETHERNET_PACKET_TYPE            6        // Ethernet-Csmacd
#define  IEEE_802_3_PACKET_TYPE          7        // Iso88023-Csmacd
#define  IEEE_802_4_PACKET_TYPE          8        // Iso88024-Tokenbus
#define  IEEE_802_5_PACKET_TYPE          9        // Iso88025-Tokenring
#define  IEEE_802_6_PACKET_TYPE         10        // Iso88026-Man
#define  STARLAN_PACKET_TYPE            11
#define  PROTEON_10MBIT_PACKET_TYPE     12
#define  PROTEON_80MBIT_PACKET_TYPE     13
#define  HYPERCHANNEL_PACKET_TYPE       14
#define  FDDI_PACKET_TYPE               15
#define  LAPD_PACKET_TYPE               16
#define  SDLC_PACKET_TYPE               17
#define  DS1_PACKET_TYPE                18        // T-1
#define  E1_PACKET_TYPE                 19        // european equiv. of T-1
#define  BASIC_ISDN_PACKET_TYPE         20
#define  PRIMARY_ISDN_PACKET_TYPE       21        // proprietary serial
#define  PPP_SERIAL_PACKET_TYPE         22        // propPointToPointSerial
#define  PPP_PACKET_TYPE                23
#define  SOFTWARE_LB_PACKET_TYPE        24        // softwareLoopback
#define  EON_PACKET_TYPE                25        // CLNP over IP
#define  ETHERNET_3MBIT_PACKET_TYPE     26
#define  NSIP_PACKET_TYPE               27        // XNS over IP
#define  SLIP_PACKET_TYPE               28        // generic SLIP
#define  ULTRA_PACKET_TYPE              29        // ULTRA technologies
#define  DS3_PACKET_TYPE                30        // T-3
#define  SIP_PACKET_TYPE                31        // SMDS
#define  FRAME_RELAY_PACKET_TYPE        32

typedef  struct   _STFRAMEHDR {
   USHORT usType;
   USHORT usLength;
   ULONG  ulTimeStamp;
   }  STFRAMEHDR;

typedef struct _STFRAME {
   USHORT      us;
   STFRAMEHDR  hdr;
   UCHAR       auchData[IP_MAXPACKET];
   }  STFRAME;

static   STFRAME    stFrameData;

static   CHAR           szFilename[256] = "iptrace.dmp";

static   CHAR           *szICMP_type[] =  {  "Echo Reply",
                                             "",
                                             "",
                                             "Dest unreach",
                                             "Packet lost",
                                             "Redirect",
                                             "",
                                             "",
                                             "Echo",
                                             "",
                                             "",
                                             "Time exceeded",
                                             "IP hdr Bad",
                                             "Timestamp Request",
                                             "Timestamp Reply",
                                             "Info Request",
                                             "Info Reply",
                                             "Addr Mask request",
                                             "Addr Mask Reply",
                                          };

static   CHAR           *szICMP_code[] =  {  "Net Unreachable",
                                             "Host Unreachable",
                                             "Bad Protocol",
                                             "Port Unreachable",
                                             "Unable To Fragment",
                                             "No Route",
                                          };

VOID     Process_ARP_packet(struct arphdr * parp, ULONG ulLength);
VOID     Process_IP_packet(struct ip * pip, ULONG ulLength);
VOID     Process_ICMP_packet(struct icmp * picmp, ULONG ulLength);
VOID     Process_UDP_packet(struct udphdr * pudp);
VOID     Process_TCP_packet(struct tcphdr * ptcp, ULONG ulLength);

VOID     Process_DNS_packet(BYTE *pb, USHORT usLength);
VOID     Process_PMAP_packet(BYTE *pb, USHORT usLength);

VOID     Format_NameComponent(char *pMsg, char *pszName, char **px);
VOID     Format_DomainName(char *pMsg, char **px, char *pszName);
VOID     Decode_RDATA(char *pMsg, USHORT rdtype, USHORT rdclass, USHORT rdlen, char **px);
VOID     Dump_data(BYTE *pb, ULONG ulLength);

//**************************************************************************
//  Name       :  IpExitProc
//  Description:  close interface and reset adapter to normal mode
//  Parameters :  usTermCode - termination info
//  Return     :  none
//**************************************************************************
VOID APIENTRY  IpExitProc(ULONG usTermCode)
{
   fflush(NULL);

   printf("\nFinished\n");
}

USHORT         CountPackets(FILE *fp)
{
size_t   x;
USHORT   cnPkt = 0;

  // loop forever
   while (TRUE) {
      x = fread(&stFrameData.hdr, sizeof(STFRAMEHDR), 1, fp);
      if (x != 1) {
         if (feof(fp)) {
            break;
            }
         return 0;
         }

      x = fread(stFrameData.auchData, 1, stFrameData.hdr.usLength, fp);
      if (x != stFrameData.hdr.usLength) {
         return 0;
         }

      cnPkt++;
      }

   rewind(fp);

   return cnPkt;
}

int            comparison(const STFRAME **p1, const STFRAME **p2)
{
   if ((*p1)->hdr.ulTimeStamp > (*p2)->hdr.ulTimeStamp)
      return 1;
   if ((*p1)->hdr.ulTimeStamp < (*p2)->hdr.ulTimeStamp)
      return -1;

   return 0;
}

char           *str_port(USHORT port, char *protocol)
{
static   char  *szPort[1024] = { NULL };

struct servent    *p;
USHORT            hport = ntohs(port);

   if (hport > 1024)
      return "Unassigned port";

   if (szPort[hport - 1] == NULL ) {
      p = getservbyport(port, protocol);
      szPort[hport - 1] = p ? strdup(p->s_name) : "Reserved";
      }

   return szPort[hport - 1];
}

//**************************************************************************
//  Name       :  main
//  Description:
//  Parameters :  argc argv env
//  Return     :  none
//**************************************************************************
VOID           main(ULONG argc, UCHAR **argv, UCHAR **env)
{
FILE           *fp;
size_t         x;
USHORT         totPkt;
USHORT         cnPkt = 0;
STFRAME        **pPkts = NULL;

   if (argc == 2) {
      strcpy(szFilename, argv[1]);
      printf(" Input File: %s\n", szFilename);
      }

  // opens the iptrace file
   printf(" Opening %s ...  ", szFilename);

   if ((fp = fopen(szFilename, "r+b")) == NULL) {
      printf("could not open %s\n", szFilename);
      return;
      }

   printf("Successful\n");

  // good housekeeping
   DosExitList(EXLST_ADD, IpExitProc);

   printf(" Reading packet headers ...  ");

   totPkt = CountPackets(fp);

   printf("%hu headers read\n", totPkt);
   if (totPkt == 0) {
      printf("Nothing to format!\n");
      return;
      }

   pPkts = calloc(totPkt, sizeof(STFRAME *));

   for (cnPkt = 0; cnPkt < totPkt; cnPkt++) {
      x = fread(&stFrameData.hdr, sizeof(STFRAMEHDR), 1, fp);
      if (x != 1) {
         printf("Break: hdr\n");
         break;
         }

      pPkts[cnPkt] = malloc(sizeof(STFRAMEHDR) + sizeof(USHORT) + stFrameData.hdr.usLength);
      pPkts[cnPkt]->hdr = stFrameData.hdr;
      pPkts[cnPkt]->us = cnPkt + 1;
      x = fread(&pPkts[cnPkt]->auchData, 1, stFrameData.hdr.usLength, fp);
      if (x != pPkts[cnPkt]->hdr.usLength) {
         printf("Break: data\n");
         break;
         }
      }

   fclose(fp);

   printf(" PreProcess packet info\n");

//   qsort(pPkts, totPkt, sizeof(STFRAME *), comparison);

   for (cnPkt = 0; cnPkt < totPkt; cnPkt++) {
      printf("\n-------------------------- #:%hu --------------------------\n", pPkts[cnPkt]->us);
      printf(" Timestamp: %7lumsec   Packet Length: %hu\n", pPkts[cnPkt]->hdr.ulTimeStamp, pPkts[cnPkt]->hdr.usLength);

      switch (pPkts[cnPkt]->hdr.usType) {
         case IP_PACKET_TYPE: {
            struct ip      *pip = (struct ip *)pPkts[cnPkt]->auchData;

            printf(" IP:    Dest: %s", inet_ntoa(pip->ip_dst));
            printf(" Source: %s\n", inet_ntoa(pip->ip_src));
            Process_IP_packet(pip, pPkts[cnPkt]->hdr.usLength);
            break;
            }

         case DS1_PACKET_TYPE: {
            struct ip      *pip = (struct ip *)pPkts[cnPkt]->auchData;

            printf(" PPP:   Dest: %s", inet_ntoa(pip->ip_dst));
            printf(" Source: %s\n", inet_ntoa(pip->ip_src));
            Process_IP_packet(pip, pPkts[cnPkt]->hdr.usLength);
            break;
            }

         case ETHERNET_PACKET_TYPE: {
            struct ether_header  *p = (struct ether_header *)pPkts[cnPkt]->auchData;

            switch (p->ether_type) {
               case 0x0008: {
                  struct ip      *pip = (struct ip *)((char *)p + sizeof(struct ether_header));

                  printf(" DIX:   Dest: %02X:%02X:%02X:%02X:%02X:%02X   Source :%02X:%02X:%02X:%02X:%02X:%02X\n", p->ether_dhost[0], p->ether_dhost[1], p->ether_dhost[2], p->ether_dhost[3], p->ether_dhost[4], p->ether_dhost[5], p->ether_shost[0], p->ether_shost[1], p->ether_shost[2], p->ether_shost[3], p->ether_shost[4], p->ether_shost[5]);
                  printf(" DIX:   Dest: %s", inet_ntoa(pip->ip_dst));
                  printf(" Source: %s\n", inet_ntoa(pip->ip_src));
                  Process_IP_packet(pip, pPkts[cnPkt]->hdr.usLength - sizeof(struct ether_header));
                  break;
                  }

               case ETHERTYPE_ARP: {
                  struct arphdr  *parp = (struct arphdr *)((char *)p + sizeof(struct ether_header));

                  printf(" DIX:   Dest: %02X:%02X:%02X:%02X:%02X:%02X   Source :%02X:%02X:%02X:%02X:%02X:%02X\n", p->ether_dhost[0], p->ether_dhost[1], p->ether_dhost[2], p->ether_dhost[3], p->ether_dhost[4], p->ether_dhost[5], p->ether_shost[0], p->ether_shost[1], p->ether_shost[2], p->ether_shost[3], p->ether_shost[4], p->ether_shost[5]);
                  Process_ARP_packet(parp, pPkts[cnPkt]->hdr.usLength - sizeof(struct ether_header));
                  break;
                  }

               default:
                  printf("\tETHER\t:%04X src:%02x%02x%02x%02x%02x%02x dst:%02x%02x%02x%02x%02x%02x\n", p->ether_type, p->ether_shost[0], p->ether_shost[1], p->ether_shost[2], p->ether_shost[3], p->ether_shost[4], p->ether_shost[5], p->ether_dhost[0], p->ether_dhost[1], p->ether_dhost[2], p->ether_dhost[3], p->ether_dhost[4], p->ether_dhost[5]);
               }

            break;
            }

         default: {
            printf("----------------------------- RECORD HEADER ------------------------------\n");
            Dump_data((BYTE *)&pPkts[cnPkt]->hdr, sizeof(STFRAMEHDR));
            printf("--------------------------------- DATA -----------------------------------\n");
            Dump_data(pPkts[cnPkt]->auchData, pPkts[cnPkt]->hdr.usLength);
            break;
            }

         }

      }

}

VOID     Process_ARP_packet(struct arphdr * parp, ULONG ulLength)
{
char     *paddr;

   printf("-------------------------- ARP --------------------------\n");
   printf(" ARP:  Hardware Type:%hu     (Ethernet 10Mb)\n", ntohs(parp->ar_hrd));
   printf(" ARP:  Protocol Type:%04X (IP Address)\n", ntohs(parp->ar_pro));
   printf(" ARP:  Hardware Len:%hu\n", parp->ar_hln);
   printf(" ARP:  Protocol Len:%hu\n", parp->ar_pln);
   if (ntohs(parp->ar_op) == ARPOP_REQUEST) {
      printf(" ARP:  Operation:%hu  (ARP Request)\n", ntohs(parp->ar_op));
      paddr = (char *)parp + sizeof(struct arphdr);
      printf(" ARP:  Sender HW address: %02X%02X%02X%02X%02X%02X\n", paddr[0], paddr[1], paddr[2], paddr[3], paddr[4], paddr[5]);
      paddr += parp->ar_hln;
      printf(" ARP:  Sender PA: %s\n", inet_ntoa(*(struct in_addr *)paddr));
      paddr += parp->ar_pln;
      printf(" ARP:  Target HW address: %02X%02X%02X%02X%02X%02X\n", paddr[0], paddr[1], paddr[2], paddr[3], paddr[4], paddr[5]);
      paddr += parp->ar_hln;
      printf(" ARP:  Target PA: %s\n", inet_ntoa(*(struct in_addr *)paddr));
      paddr += parp->ar_pln;
      }
   else if (ntohs(parp->ar_op) == ARPOP_REPLY) {
      printf(" ARP:  Operation:%hu  (ARP Reply)\n", ntohs(parp->ar_op));
      paddr = (char *)parp + sizeof(struct arphdr);
      printf(" ARP:  Sender HW address: %02X%02X%02X%02X%02X%02X\n", paddr[0], paddr[1], paddr[2], paddr[3], paddr[4], paddr[5]);
      paddr += parp->ar_hln;
      printf(" ARP:  Sender PA: %s\n", inet_ntoa(*(struct in_addr *)paddr));
      paddr += parp->ar_pln;
      printf(" ARP:  Target HW address: %02X%02X%02X%02X%02X%02X\n", paddr[0], paddr[1], paddr[2], paddr[3], paddr[4], paddr[5]);
      paddr += parp->ar_hln;
      printf(" ARP:  Target PA: %s\n", inet_ntoa(*(struct in_addr *)paddr));
      paddr += parp->ar_pln;
      }
   else {
      printf(" ARP:  Operation:%hu  (Unknown)\n", ntohs(parp->ar_op));
      Dump_data((BYTE *)parp, ulLength);
      }
}

VOID     Process_IP_packet(struct ip * pip, ULONG ulLength)
{
char     ch;

   printf("----------------------- IP HEADER -----------------------\n");
   printf(" IP:  Version: %1hu Correct    Header Length: %2hu bytes\n", pip->ip_v, pip->ip_hl * 4);
   printf(" IP:  Type Of Service: %02X\n", pip->ip_tos);
   ch = pip->ip_tos & 0xe0;
   printf(" IP:     %1d%1d%1d. ....  %s\n", (pip->ip_tos & 0x80) >> 7, (pip->ip_tos & 0x40) >> 6, (pip->ip_tos & 0x20) >> 5, (pip->ip_tos & 0xe0) ? "Priority" : "Routine");
   printf(" IP:     ...%1d ....  Normal Delay\n", (pip->ip_tos & 0x10) >> 4);
   printf(" IP:     .... %1d...  Normal Throughput\n", (pip->ip_tos & 0x08) >> 3);
   printf(" IP:     .... .%1d..  Normal Reliability\n", (pip->ip_tos & 0x04) >> 2);
   printf(" IP:  Total Len: %hu (x%X) bytes\tId: %04X\n", ntohs(pip->ip_len), ntohs(pip->ip_len), ntohs(pip->ip_id));
   ch = (ntohs(pip->ip_off) & 0xf000) >> 12;
   printf(" IP:  Flags: %1X\n", ch);
   printf(" IP:     .%1d..       May Fragment\n", (ch & 0x04) >> 2);
   printf(" IP:     ..%1d.       Last Fragment\n", (ch & 0x02) >> 1);
   printf(" IP:  Fragment Offset: %03X\n", ntohs(pip->ip_off) &0x0fff);
   printf(" IP:  Time To Live: %hu sec     Protocol: %hu\n", pip->ip_ttl, pip->ip_p);
   printf(" IP:  Header Checksum: %04X\n", ntohs(pip->ip_sum));
   printf(" IP:  No Options\n");

   switch (pip->ip_p) {
      case IPPROTO_ICMP: {
         struct icmp     *picmp = (struct icmp *)((char *)pip + sizeof(struct ip));

         Process_ICMP_packet(picmp, ulLength - sizeof(struct ip));
         break;
         }

      case IPPROTO_UDP: {
         struct udphdr     *pudp = (struct udphdr *)((char *)pip + sizeof(struct ip));

         Process_UDP_packet(pudp);
         break;
         }

      case IPPROTO_TCP: {
         struct tcphdr     *ptcp = (struct tcphdr *)((char *)pip + sizeof(struct ip));

         Process_TCP_packet(ptcp, ulLength - sizeof(struct ip));
         break;
         }

      default:
         printf("Unknown protocol type %hu. Printing packet\n", pip->ip_p);
         printf("--------------------------------- PACKET -----------------------------------\n");
         Dump_data((PBYTE)pip, ulLength);
      }
}

VOID     Process_ICMP_packet(struct icmp * picmp, ULONG ulLength)
{
   printf("---------------------- ICMP HEADER ----------------------\n");
   printf(" ICMP:  Type: %02X  %s\n", picmp->icmp_type, szICMP_type[picmp->icmp_type]);
   printf(" ICMP:  Code: %02X  %s\n", picmp->icmp_code, szICMP_code[picmp->icmp_code]);
   printf(" ICMP:  Checksum: %04X\n", ntohs(picmp->icmp_cksum));

   if (picmp->icmp_type == ICMP_ECHO || picmp->icmp_type == ICMP_ECHOREPLY) {
      printf("--------------------------------- DATA -----------------------------------\n");
      Dump_data(picmp->icmp_data, ulLength - offsetof(struct icmp, icmp_data));
      }
   else if (picmp->icmp_type == ICMP_UNREACH) {
      printf("--------------------------------- DATA -----------------------------------\n");
      Dump_data(picmp->icmp_data, ulLength - offsetof(struct icmp, icmp_data));
      }
}

VOID     Process_UDP_packet(struct udphdr * pudp)
{
   printf("\n---------------------- UDP HEADER ----------------------\n");
   printf(" UDP:  Source Port: %hu (%s)    ", ntohs(pudp->uh_sport), str_port(pudp->uh_sport, "udp"));
   printf("Dest Port: %hu (%s)\n", ntohs(pudp->uh_dport), str_port(pudp->uh_dport, "udp"));
   printf(" UDP:  Length: %hu (x%X)\n", ntohs(pudp->uh_ulen), ntohs(pudp->uh_ulen));
   printf(" UDP:  Checksum: %04X\n", ntohs(pudp->uh_sum));

   if (ntohs(pudp->uh_sport) == NAMESERVER_PORT || ntohs(pudp->uh_dport) == NAMESERVER_PORT) {
      Process_DNS_packet((BYTE *)((char *)pudp + sizeof(struct udphdr)), ntohs(pudp->uh_ulen) - sizeof(struct udphdr));
      }
   else if (ntohs(pudp->uh_sport) == PMAPPORT || ntohs(pudp->uh_dport) == PMAPPORT) {
      Process_PMAP_packet((BYTE *)((char *)pudp + sizeof(struct udphdr)), ntohs(pudp->uh_ulen) - sizeof(struct udphdr));
      }
   else {
      printf("--------------------------------- DATA -----------------------------------\n");
      Dump_data((BYTE *)((char *)pudp + sizeof(struct udphdr)), ntohs(pudp->uh_ulen) - sizeof(struct udphdr));
      }
}

VOID     Process_TCP_packet(struct tcphdr * ptcp, ULONG ulLength)
{
char     *pOpt;
char     *pData;
char     *pEOP;
USHORT   cbOpt;
USHORT   cbData;
char     ch;

   printf("---------------------- TCP HEADER ----------------------\n");
   printf(" TCP:  Source Port: %hu  (%s)       ", ntohs(ptcp->th_sport), str_port(ptcp->th_sport, "tcp"));
   printf("Dest Port: %hu  (%s)\n", ntohs(ptcp->th_dport), str_port(ptcp->th_dport, "tcp"));
   printf(" TCP:  Sequence #: %lu\n", ntohl(ptcp->th_seq));
   printf(" TCP:  Ack #: %lu\n", ntohl(ptcp->th_ack));
   printf(" TCP:  Offset: %lu bytes\n", ptcp->th_off * 4);
   ch = ptcp->th_flags;
   printf(" TCP:  Flags: %02x\n", ptcp->th_flags);
   if ((ch & TH_URG) == TH_URG) {
      printf(" TCP:     ..1. ....  <URG> Urgent bit On\n");
      }
   else {
      printf(" TCP:     ..0. ....        Urgent bit Off\n");
      }
   if ((ch & TH_ACK) == TH_ACK) {
      printf(" TCP:     ...1 ....  <ACK> Ack bit On\n");
      }
   else {
      printf(" TCP:     ...0 ....        Ack bit Off\n");
      }
   if ((ch & TH_PUSH) == TH_PUSH) {
      printf(" TCP:     .... 1...  <PSH> Push bit On\n");
      }
   else {
      printf(" TCP:     .... 0...        Push bit Off\n");
      }
   if ((ch & TH_RST) == TH_RST) {
      printf(" TCP:     .... .1..  <RST> Reset bit On\n");
      }
   else {
      printf(" TCP:     .... .0..        Reset bit Off\n");
      }
   if ((ch & TH_SYN) == TH_SYN) {
      printf(" TCP:     .... ..1.  <SYN> Synchronize bit On\n");
      }
   else {
      printf(" TCP:     .... ..0.        Synchronize bit Off\n");
      }
   if ((ch & TH_FIN) == TH_FIN) {
      printf(" TCP:     .... ...1  <FIN> Finish bit On\n");
      }
   else {
      printf(" TCP:     .... ...0        Finish bit Off\n");
      }

   printf(" TCP:  Window: %hu      Checksum: %04X   %s\n", ntohs(ptcp->th_win), ntohs(ptcp->th_sum), NULL);

   pOpt  = ((char *)ptcp) + sizeof(struct tcphdr);
   pData = ((char *)ptcp) + ptcp->th_off * 4;
   pEOP  = ((char *)ptcp) + ulLength;
   cbOpt = pData - pOpt;
   cbData = pEOP - pData;

   if (cbOpt) {
      while (cbOpt) {
         switch (*pOpt) {
            case TCPOPT_NOP:
               printf(" TCP:  Option Code: 01     Length: 1 bytes	  [NOP]\n");
               printf(" TCP:    No Operation\n");
               pOpt++;
               cbOpt--;
               break;
            case TCPOPT_MAXSEG:
               printf(" TCP:  Option Code: 02     Length: %hu bytes	  [MSS]\n", *(pOpt + 1));
               printf(" TCP:    Max Segment Size %hu (%hx)\n", ntohs(*(PUSHORT)(pOpt + 2)), ntohs(*(PUSHORT)(pOpt + 2)));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            case TCPOPT_WINDOW:
               printf(" TCP:  Option Code: 03     Length: %hu bytes	  [WIN_SCALE]\n", *(pOpt + 1));
               printf(" TCP:    Window scale factor %hu (%hx)\n", *(pOpt + 2), *(pOpt + 2));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            case TCPOPT_SACK_PERMITTED:
               printf(" TCP:  Option Code: 04     Length: %hu bytes	  \n", *(pOpt + 1));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            case TCPOPT_SACK:
               printf(" TCP:  Option Code: 05     Length: %hu bytes	  \n", *(pOpt + 1));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            case TCPOPT_TIMESTAMP:
               printf(" TCP:  Option Code: 08     Length: %hu bytes	  [TIMESTAMP]\n", *(pOpt + 1));
               printf(" TCP:    TimeStamp Value %lu (%lx)\n", ntohl(*(PULONG)(pOpt + 2)), ntohl(*(PULONG)(pOpt + 2)));
               printf(" TCP:    TimeStamp Echo Reply %lu (%lx)\n", ntohl(*(PULONG)(pOpt + 6)), ntohl(*(PULONG)(pOpt + 6)));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            case TCPOPT_CC:
               printf(" TCP:  Option Code: 11     Length: %hu bytes	  [CC]\n", *(pOpt + 1));
               printf(" TCP:    Connection Count %lu (%lx)\n", ntohl(*(PULONG)(pOpt + 2)), ntohl(*(PULONG)(pOpt + 2)));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            case TCPOPT_CCNEW:
               printf(" TCP:  Option Code: 12     Length: %hu bytes	  [CCNew]\n", *(pOpt + 1));
               printf(" TCP:    New Connection Count %lu (%lx)\n", ntohl(*(PULONG)(pOpt + 2)), ntohl(*(PULONG)(pOpt + 2)));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            case TCPOPT_CCECHO:
               printf(" TCP:  Option Code: 13     Length: %hu bytes	  [CCEcho]\n", *(pOpt + 1));
               printf(" TCP:    Connection Count %lu (%lx)\n", ntohl(*(PULONG)(pOpt + 2)), ntohl(*(PULONG)(pOpt + 2)));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            default:
               printf(" TCP:  Option Code: %2hu     Length: %hu bytes\n", *pOpt, *(pOpt + 1));
               cbOpt -= *(pOpt + 1);
               pOpt += *(pOpt + 1);
               break;
            }
         }
      }
   else {
      printf(" TCP:  No Options\n");
      }

   if (cbData) {
      if (ntohs(ptcp->th_sport) == NAMESERVER_PORT || ntohs(ptcp->th_dport) == NAMESERVER_PORT) {
         Process_DNS_packet(pData, cbData);
         }
      else if (ntohs(ptcp->th_sport) == PMAPPORT || ntohs(ptcp->th_dport) == PMAPPORT) {
         Process_PMAP_packet(pData, cbData);
         }
      else {
         printf("--------------------------------- DATA -----------------------------------\n");
         Dump_data(pData, cbData);
         }
      }
   else {
      printf(" TCP:  No data or not output.\n");
      }
}

VOID     Process_DNS_packet(BYTE *pb, USHORT usLength)
{
HEADER   *pdns = (HEADER *)pb;
char     szName[256];
char     *px;

   printf("------------------------ DNS ----------------------------\n");
   printf(" DNS: ID: %hu\n", ntohs(pdns->id));
   printf(" DNS: %s  %s\n", (pdns->qr) ? "Query Response" : "Query", (pdns->opcode == QUERY) ? "Standard Query" : "?????????????????????");
   printf(" DNS: FLAGS: %02X\n", (((pdns->aa << 2) |
                                   (pdns->tc << 1) |
                                   (pdns->rd)) << 4) |
                                  ((pdns->ra << 3) |
                                   (pdns->pr << 2)) );
   if (pdns->qr) {
      if (pdns->aa) {
         printf(" DNS:    1.. .... = Authoritive Answer\n");
         }
      }
   printf(" DNS:    .%1d. .... = %s\n", pdns->tc, (pdns->tc) ? "Truncation" : "No Truncation");
   printf(" DNS:    ..%1d .... = %s\n", pdns->rd, (pdns->rd) ? "Recursion Desired" : "No Recursion Desired");
   if (pdns->qr) {
      printf(" DNS:    ... %1d... = %s\n", pdns->ra, (pdns->ra) ? "Recursion Available" : "No Recursion Available");
      }
   printf(" DNS: Return Code: %hu\n", pdns->rcode);
   printf(" DNS: Question Cnt: %2hu,    Answer Cnt: %2hu\n", ntohs(pdns->qdcount), ntohs(pdns->ancount));
   printf(" DNS: Authority Cnt: %2hu,   Additional Cnt: %2hu\n", ntohs(pdns->nscount), ntohs(pdns->arcount));

   px = (char *)pdns + sizeof(HEADER);

   if (pdns->qdcount) {
      printf(" DNS:  Question Section\n");
      szName[0] = '\0';
      Format_DomainName(pb, &px, szName);
      szName[strlen(szName) - 1] = '\0';
      printf(" DNS:    Name: %s\n", szName);
      printf(" DNS:    Type: %02X    Class: %02X\n", ntohs(*(PUSHORT)px), ntohs(*(PUSHORT)(px + 2)));
      px += 4;
      printf("\n");
      }

   if (pdns->ancount) {
      USHORT      rdtype, rdclass, rdlen, c;
      ULONG       rdttl;

      printf(" DNS:  Answer Section\n");
      c = ntohs(pdns->ancount);
      while (c) {
         szName[0] = '\0';
         Format_DomainName(pb, &px, szName);
         szName[strlen(szName) - 1] = '\0';
         printf(" DNS:    Name: %s\n", szName);
         rdtype = ntohs(*(PUSHORT)px);
         rdclass = ntohs(*(PUSHORT)(px + 2));
         px += 4;
         printf(" DNS:    Type: %02X    Class: %02X\n", rdtype, rdclass);
         rdttl = ntohl(*(PULONG)px);
         rdlen = ntohs(*(PUSHORT)(px + 4));
         px += 6;
         printf(" DNS:    TTL: %lu    Length: %hu\n", rdttl, rdlen);

         Decode_RDATA(pb, rdtype, rdclass, rdlen, &px);

         c--;
         printf("\n");
         }
      }

   if (pdns->nscount) {
      USHORT      rdtype, rdclass, rdlen, c;
      ULONG       rdttl;

      printf(" DNS:  Authority Section\n");
      c = ntohs(pdns->nscount);
      while (c) {
         szName[0] = '\0';
         Format_DomainName(pb, &px, szName);
         szName[strlen(szName) - 1] = '\0';
         printf(" DNS:    Domain Name: %s\n", szName);
         rdtype = ntohs(*(PUSHORT)px);
         rdclass = ntohs(*(PUSHORT)(px + 2));
         px += 4;
         printf(" DNS:    Type: %02X    Class: %02X\n", rdtype, rdclass);
         rdttl = ntohl(*(PULONG)px);
         rdlen = ntohs(*(PUSHORT)(px + 4));
         px += 6;
         printf(" DNS:    TTL: %lu    Length: %hu\n", rdttl, rdlen);

         Decode_RDATA(pb, rdtype, rdclass, rdlen, &px);

         c--;
         printf("\n");
         }
      }

   if (pdns->arcount) {
      USHORT      rdtype, rdclass, rdlen, c;
      ULONG       rdttl;

      printf(" DNS:  Additional Section\n");
      c = ntohs(pdns->arcount);
      while (c) {
         szName[0] = '\0';
         Format_DomainName(pb, &px, szName);
         szName[strlen(szName) - 1] = '\0';
         printf(" DNS:    Domain Name: %s\n", szName);
         rdtype = ntohs(*(PUSHORT)px);
         rdclass = ntohs(*(PUSHORT)(px + 2));
         px += 4;
         printf(" DNS:    Type: %02X    Class: %02X\n", rdtype, rdclass);
         rdttl = ntohl(*(PULONG)px);
         rdlen = ntohs(*(PUSHORT)(px + 4));
         px += 6;
         printf(" DNS:    TTL: %lu    Length: %hu\n", rdttl, rdlen);

         Decode_RDATA(pb, rdtype, rdclass, rdlen, &px);

         c--;
         printf("\n");
         }
      }
}

VOID     Process_PMAP_packet(BYTE *pb, USHORT usLength)
{
struct rpc_msg    *prpc = (struct rpc_msg *)pb;
BYTE              *pd;
USHORT            us;

   printf("------------------------------- PORTMAP ----------------------------------\n");
   printf(" PMAP:  Trans ID: %lu\n", htonl(prpc->rm_xid));
   pd = (BYTE *)prpc + sizeof(prpc->rm_xid) + sizeof(prpc->rm_direction);
   us = usLength - sizeof(prpc->rm_xid) - sizeof(prpc->rm_direction);
   if (htonl(prpc->rm_direction) == CALL) {
      printf(" PMAP:  Type: 0    Request %lu\n", sizeof(prpc->rm_call));
      printf(" PMAP:  RPC Ver: %lu\n", htonl(prpc->rm_call.cb_rpcvers));
      printf(" PMAP:  RPC Program: %lu     Ver: %lu\n", htonl(prpc->rm_call.cb_prog), htonl(prpc->rm_call.cb_vers));
      printf(" PMAP:  Procedure: %lu  Map Lookup\n", htonl(prpc->rm_call.cb_proc));
      printf(" PMAP:  Program: 0       Ver: 0\n");
      printf(" PMAP:  Protocol: 0\n");
      pd += sizeof(prpc->rm_call);
      us -= sizeof(prpc->rm_call);
      }
   else {
      printf(" PMAP:  Type: 1    Reply %lu\n", sizeof(prpc->acpted_rply));
      pd += sizeof(prpc->acpted_rply);
      us -= sizeof(prpc->acpted_rply);
      }
   Dump_data(pd, us);
}

VOID     Format_DomainName(char *pMsg, char **px, char *pszName)
{
   if ((**px & 0xc0) == 0xc0) {
      char        *py;
      USHORT      off;

      off = (ntohs(*(PUSHORT)*px) & 0x3fff);
      py = pMsg + off;
      Format_DomainName(pMsg, &py, pszName);
      *px += 2;
      }
   else {
      while (**px) {
         if ((**px & 0xc0) == 0xc0) {
            char        *py;
            USHORT      off;

            off = (ntohs(*(PUSHORT)*px) & 0x3fff);
            py = pMsg + off;
            Format_DomainName(pMsg, &py, pszName);
            *px += 2;
            return;
            }
         strncat(pszName, *px + 1, **px);
         strcat(pszName, ".");
         *px += **px + 1;
         }
      (*px)++;
      }
}

VOID     Decode_RDATA(char *pMsg, USHORT rdtype, USHORT rdclass, USHORT rdlen, char **px)
{
char     szName[256];

   while (rdlen) {
      if (rdtype == T_CNAME) {
         if ((**px & 0xc0) == 0xc0) {
            char        *py;
            USHORT      off;

            off = (ntohs(*(PUSHORT)*px) & 0x3fff);
            py = pMsg + off;
            szName[0] = '\0';
            while (*py) {
               strncat(szName, py + 1, *py);
               strcat(szName, ".");
               py += *py + 1;
               }
            szName[strlen(szName) - 1] = '\0';
            *px += 2;
            rdlen -= 2;
            }
         else {
            szName[0] = '\0';
            while (**px) {
               strncat(szName, *px + 1, **px);
               strcat(szName, ".");
               rdlen -= **px + 1;
               *px += **px + 1;
               }
            szName[strlen(szName) - 1] = '\0';
            (*px)++;
            rdlen--;
            }
         printf(" DNS:    Canonical Name: %s\n", szName);
         }
      else if (rdtype == T_A) {
         printf(" DNS:    Address: %s\n", inet_ntoa(*(struct in_addr *)*px));
         *px += 4;
         rdlen = 0;
         }
      else if (rdtype == T_NS) {
         szName[0] = '\0';
         Format_DomainName(pMsg, px, szName);
         szName[strlen(szName) - 1] = '\0';
         printf(" DNS:    Name Server: %s\n", szName);
         rdlen = 0;
         }
      else if (rdtype == T_PTR) {
         szName[0] = '\0';
         Format_DomainName(pMsg, px, szName);
         szName[strlen(szName) - 1] = '\0';
         printf(" DNS:    Domain Name: %s\n", szName);
         rdlen = 0;
         }
      else if (rdtype == T_SOA) {
         szName[0] = '\0';
         Format_DomainName(pMsg, px, szName);
         szName[strlen(szName) - 1] = '\0';
         printf(" DNS:    Domain Name: %s\n", szName);
         szName[0] = '\0';
         Format_DomainName(pMsg, px, szName);
         szName[strlen(szName) - 1] = '\0';
         printf(" DNS:    Mailbox: %s\n", szName);
         printf(" DNS:    Serial #: %lu\n", ntohl(*(PULONG)*px));
         printf(" DNS:    Refresh: %lu\n",  ntohl(*((PULONG)*px + 1)));
         printf(" DNS:    Retry: %lu\n",    ntohl(*((PULONG)*px + 2)));
         printf(" DNS:    Expire: %lu\n",   ntohl(*((PULONG)*px + 3)));
         printf(" DNS:    Minimum: %lu\n",  ntohl(*((PULONG)*px + 4)));
         rdlen = 0;
         }
      else {
         printf(" DNS:    ??: type: %hu  class: %hu  length: %hu\n", rdtype, rdclass, rdlen);

         Dump_data(*px, rdlen);

         *px += rdlen;
         rdlen = 0;
         }
   }
}

VOID     Dump_data(BYTE *pb, ULONG ulLength)
{
char     sz[17];
short    i;

   memset(sz, '\0', sizeof(sz));
   for (i = 0; i < ulLength; i++) {
      if ((i % 16) == 0) {
         printf("%04X", i);
         }

      printf(" %02X", pb[i]);
      if (isprint(pb[i])) {
         sz[i % 16] = pb[i];
         }
      else {
         sz[i % 16] = '.';
         }

      if (((i + 1) % 16) == 8) {
         printf("   ");
         }

      if (((i + 1) % 16) == 0) {
         printf("   %s\n", sz);
         memset(sz, '\0', sizeof(sz));
         }
      }

   if (strlen(sz)) {
      for (i = strlen(sz); i < 16; i++) {
         printf("   ");
         if (((i + 1) % 16) == 8) {
            printf("   ");
            }
         }
      printf("   %s\n\n", sz);
      }
   else {
      printf("\n\n");
      }
}

