/*
    Animated Mouse Pointer
    Copyright (C) 1997 Christian Langanke

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define  INCL_ERRORS
#define  INCL_PM
#define  INCL_DOS
#include <os2.h>

#define NEWLINE "\n"

#include "title.h"
#include "nls/amptreng.rch"
#include "dll.h"
#include "lx.h"
#include "mptrptr.h"
#include "mptrutil.h"
#include "macros.h"
#include "debug.h"
#include "bindata.h"

// define ordinals
#define ORDINAL_DESCRIPTION 0
#define ORDINAL_INFONAME    1
#define ORDINAL_INFOARTIST  2

#define ORDINAL_FIRST       ORDINAL_INFONAME
#define ORDINAL_LAST        ORDINAL_INFOARTIST

// length of a string in a stingtable is calulated for
// maximum string length of four digits (1 length byte, 4 digits, 1 zero byte)
#define STRINGRES_MAXLEN 6

// die AND Reihenfolge wird durch aulStringOrder sichergestellt
static ULONG aulResourceIdBase[]     = { 100, 200, 300, 400, 600, 700, 800, 900, 1000 };

// ID 500 wird im Code bercksichtigt ! { 0    1    2    3    4    5    6    7    8 };
static ULONG aulStringOrder[]         = { 0,   1,   2,   5,   3,   6,   4,   7,   8 };

// zero based !                         { 0    1    2    3    0,   4    5    6    7    8 };
static ULONG aulAnd2Ptr[]             = { 0,   1,   2,   5,   0,   3,   6,   4,   7,   8 };

// zero based !                         {   0,   1,   2,   3,   4,   5,   6,   7,    8 }
static ULONG aulPtr2And[]          =    { 100, 200, 300, 600, 800, 400, 700, 900, 1000 };

static PSZ pszDescription = "AniMouse Resource DLL file generated by "
                            "PTRCONV, the mouse pointer animation resource "
                            "converter of " __TITLE__ " V" __VERSION__ " "
                            "by " __AUTHOR__ " " __YEAR__;

// sum up o next page boundary
PBYTE
NEXTADRESS( PBYTE b, PBYTE t, ULONG l )
{
  ULONG newdatalen = (((PBYTE)t - (PBYTE)b ) % EXE_PAGELEN ) + l;

  if( newdatalen > EXE_PAGELEN ) {
    t = b + (((((PBYTE)t  - (PBYTE)b ) / EXE_PAGELEN ) + 1 ) * EXE_PAGELEN );
  }

  return t;
}

#define ADDRESOURCE( ptr, type, id, len, base, data )   \
  ptr->usResType = type;                                \
  ptr->usResName = id;                                  \
  ptr->ulResSize = len;                                 \
  ptr->usResObj  = plxheaderTemplate->ulObjCnt + 1;     \
  ptr->ulResOffs = (PBYTE)data - (PBYTE)base;           \
  ptr++

/*Ŀ
 * Name      : GetAnimouseBasePointerId                                   
 * Kommentar : ermittelt die Pointer-ID einer Animouse-Resource-DLL       
 *             zum System-Pointer-Index                                   
 *             Animouse-Resource-DLL                                      
 * Autor     : C.Langanke                                                 
 * Datum     : 23.07.1995                                                 
 * nderung  : 23.07.1995                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : ULONG   - Index des Systemzeigers                          
 * Aufgaben  : - Ressource-ID fr Animouse Resource DLL ermitteln         
 * Rckgabe  : Base Resource Id                                           
 *
 */

ULONG
GetAnimouseBasePointerId( ULONG ulPointerIndex ) {
  return aulPtr2And[ulPointerIndex];
}

/*Ŀ
 * Name      : GetSystemPointerIndexFromAnimouseId                        
 * Kommentar : ermittelt den System-Pointer-Index zur Pointer-ID          
 *             einer Animouse-Resource-DLL                                
 * Autor     : C.Langanke                                                 
 * Datum     : 23.07.1995                                                 
 * nderung  : 23.07.1995                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : ULONG   - Resource-ID eines Animouse Pointers              
 *                       1xx, 2xx, 3xx, 4xx, 6xx, 7xx, 8xx, 9xx, 10xx oder
 *                       1, 2, 3, 4, 6, 7, 8, 9, 10                       
 * Aufgaben  : - System-Pointer-Index ermitteln                           
 * Rckgabe  : System-Pointer-Index                                       
 *
 */

ULONG
GetSystemPointerIndexFromAnimouseId( ULONG ulPointerId )
{
  if( ulPointerId > 10 ) {
    return aulAnd2Ptr[ ( ulPointerId / 100 ) - 1];
  } else {
    return aulAnd2Ptr[ ulPointerId - 1];
  }
}

/*Ŀ
 * Name      : GetAnimouseStringId                                        
 * Kommentar : ermittelt die String-Id zu einer Zeiger-ID in einer        
 *             Animouse-Resource-DLL                                      
 * Autor     : C.Langanke                                                 
 * Datum     : 23.07.1995                                                 
 * nderung  : 23.07.1995                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : HMODULE - handle der Resource DLL                          
 *             ULONG   - ID des Zeigers                                   
 * Aufgaben  : - Pointer direkt laden                                     
 * Rckgabe  : Id oder -1                                                 
 *
 */

ULONG
GetAnimouseStringId( HMODULE hmodResource, ULONG ulPtrResourceId )
{
  APIRET rc  = NO_ERROR;
  ULONG  i;
  HAB    hab = WinQueryAnchorBlock( HWND_DESKTOP );

  BOOL   fReadError = FALSE;

  ULONG  ulStringId = -1;
  ULONG  ulCurrentId;

  ULONG  ulTimeoutSkipValue;
  CHAR   szTimeoutValue[ 20];

  do
  {
    if( hmodResource     == NULLHANDLE ) {
      break;
    }

    // resid des ersten timeout strings ermitteln
    ulTimeoutSkipValue = ulPtrResourceId / 100;
    ulCurrentId = 0;
    for( i = 0; i < ulTimeoutSkipValue - 1; i++ )
    {
      // anzahl lesen
      fReadError |= ( !LOADSTRING( ulCurrentId++, szTimeoutValue ));
      if( fReadError ) {
        rc = ERRORIDERROR( WinGetLastError( hab ));
        break;
      }
      // nchste Id lesen
      ulCurrentId += atol( szTimeoutValue );
    }

    if( rc != NO_ERROR ) {
      break;
    }

    // hand over found id
    ulCurrentId += ulPtrResourceId % 100;
    ulStringId = ulCurrentId;
  } while( FALSE );

  return ulStringId;
}

/*Ŀ
 * Name      : GetAnimousePointerResource                                 
 * Kommentar : ldt Pointer-Resource direkt aus Animouse DLL              
 * Autor     : C.Langanke                                                 
 * Datum     : 23.07.1995                                                 
 * nderung  : 23.07.1995                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : HMODULE      - handle der Resource DLL                     
 *             PICONINFO    - Zielvariable fr Pointerdaten               
 * Aufgaben  : - Pointer direkt laden                                     
 * Rckgabe  : BOOL - Erfolgsflag                                         
 *
 */

BOOL
GetAnimousePointerResource( HMODULE hmodResource, ULONG ulResourceId, PICONINFO piconinfo )
{
  BOOL   fSuccess = FALSE;
  APIRET rc       = NO_ERROR;
  ULONG  ulResourceSize;
  PVOID  pvResource;

  do
  {
    if(( hmodResource == NULLHANDLE ) ||
       ( piconinfo    == NULL ))
    {
      break;
    }

    // ICONINFO aufsetzen
    piconinfo->cb           = sizeof( ICONINFO );
    piconinfo->fFormat      = ICON_RESOURCE;
    piconinfo->hmod         = hmodResource;
    piconinfo->resid        = ulResourceId;

    // Daten direkt aus DLL lesen, damit das System nicht die DLL festhlt
    rc = DosQueryResourceSize( hmodResource, RT_POINTER, ulResourceId, &ulResourceSize );

    if( rc == NO_ERROR ) {
      // Speicher holen
      piconinfo->pIconData = malloc( ulResourceSize );
      if( piconinfo->pIconData ) {
        // Ressource lesen
        rc = DosGetResource( hmodResource, RT_POINTER, ulResourceId, &pvResource );

        if( rc == NO_ERROR ) {
          // copy data and change info
          memcpy( piconinfo->pIconData, pvResource, ulResourceSize );
          DosFreeResource( pvResource );
          piconinfo->fFormat      = ICON_DATA;
          piconinfo->cbIconData   = ulResourceSize;
          piconinfo->hmod         = NULLHANDLE;
          piconinfo->resid        = 0;
          fSuccess = TRUE;
        } else {
          // Daten konnten nicht gelesen werden:
          // Speicher wieder freigeben
          free( piconinfo->pIconData );
          piconinfo->pIconData = NULL;
        }
      }
    }
  } while( FALSE );

  return fSuccess;
}

/*Ŀ
 * Name      : CreateStringtable                                          
 * Comment   :                                                            
 * Author    : C.Langanke                                                 
 * Date      : 25.03.1998                                                 
 * Update    : 25.03.1998                                                 
 * called by : app                                                        
 * calls     : Dos*                                                       
 * Input     : PBTYE  - pointer to target buffer                          
 *             ULONG  - buffer len                                        
 *             PSZ[]  - pointer to 16 PSZ (epmty strings are NULL)        
 *             PULONG - target variable for resource name (base id)       
 *             PULONG - target variable for table size                    
 * Tasks     : - create a stringtable from 16 strings                     
 * returns   : APIRET - OS/2 error code                                   
 *
 */

APIRET
CreateStringtable( PBYTE pbData, ULONG ulBuflen, ULONG ulStartId,
                   PPSZ ppszStringTable, PULONG pulResName, PULONG pulTableLen )
{
  APIRET          rc = NO_ERROR;
  ULONG           i;
  ULONG           ulTableLen = 0;

  PRESSTRINGTABLE presstringtable;
  PRESSTRING      presstring;
  PSZ pszValue;

  do
  {
    if(( pbData          == NULL ) ||
       ( ppszStringTable == NULL ) ||
       ( pulResName      == NULL ) ||
       ( pulTableLen     == NULL ))
    {
      rc = ERROR_INVALID_PARAMETER;
      break;
    }

    // calculate buffer len
    ulTableLen = sizeof( RESSTRINGTABLE );
    for( i = 0; i < 16; i++ )
    {
      pszValue = *( ppszStringTable + i );
      ulTableLen += sizeof( RESSTRING ) + (( pszValue ) ?  strlen( pszValue ) : 0 );
    }

    // hand over table len
    *pulTableLen = ulTableLen;
    *pulResName  = RESSTRING_NAME( ulStartId );
    if( ulBuflen < ulTableLen ) {
      rc = ERROR_MORE_DATA;
      break;
    }

    // create table
    memset( pbData, 0, ulTableLen );
    presstringtable            = (PRESSTRINGTABLE)pbData;
    presstringtable->usSig     = SIG_RESSTRINGTABLE;
    presstring                 = (PRESSTRING)( pbData +  sizeof( RESSTRINGTABLE ));

    for( i = 0; i < 16; i++ )
    {
      pszValue = *( ppszStringTable + i );
      if( !pszValue ) {
        pszValue = "";
      }

      presstring->bLen = strlen( pszValue ) + 1;
      strcpy( presstring->szString, pszValue );
      presstring = RESSTRING_NEXT( presstring );
    }
  } while( FALSE );

  return rc;
}


/*Ŀ
 * Name      : LoadPointerFromAnimouseFile                                
 * Kommentar : ldt Pointer aus Animouse DLL                              
 * Autor     : C.Langanke                                                 
 * Datum     : 23.07.1995                                                 
 * nderung  : 23.07.1995                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : PSZ          - Name der Pointerdatei                       
 *             ULONG        - Index des Pointers                          
 *             ULONG        - Index der Pointerliste                      
 *           : PHPOINTER    - Zielvariable fr PointerHandle              
 *             PICONINFO    - Zielvariable fr Pointerdaten               
 * Aufgaben  : - Pointer laden                                            
 * Rckgabe  : BOOL - Flag: Pointer geladen/nicht geladen                 
 *
 */

BOOL
LoadPointerFromAnimouseFile( HMODULE hmodResource, ULONG ulPointerIndex,
                             PHPOINTER pahptr, PICONINFO paiconinfo, PHPOINTER phptrStatic,
                             PICONINFO piconinfoStatic, PULONG paulTimeout, PULONG pulEntries )
{
  HAB       hab = WinQueryAnchorBlock( HWND_DESKTOP );

  BOOL      fSuccess = FALSE;
  ULONG     i;
  HPOINTER  hptr = NULLHANDLE;

  ICONINFO  iconinfoTmp;
  ULONG     ulTimeoutTmp;

  PICONINFO piconinfo;
  PHPOINTER phptr;
  PULONG    pulTimeout;

  CHAR      szTimeoutValue[20];
  ULONG     ulCurrentId;
  ULONG     fReadError = FALSE;
  ULONG     ulPointerCount;

  ULONG     ulResourceId;

  do {
    // initialisieren
    memset( &iconinfoTmp, 0, sizeof( ICONINFO ));

    // Parameter prfen
    if(( hmodResource   == NULL )               ||
       ( ulPointerIndex >=  NUM_OF_SYSCURSORS ) ||
       ( pulEntries     == NULL ))
    {
      break;
    }

    // Static Pointer lesen, kein Fehler, wenn nicht vorhanden !
    piconinfo    = ( piconinfoStatic ) ? piconinfoStatic :  &iconinfoTmp;
    phptr        = ( phptrStatic )     ? phptrStatic     :  NULL;

    // Resource-ID ermitteln: 1 based !
    ulResourceId = GetAnimouseBasePointerId( ulPointerIndex );

    // ptr direkt lesen, wenn mglich
    GetAnimousePointerResource( hmodResource, ulResourceId, piconinfo );

    if( phptr ) {
      // Pointer erstellen
      hptr = CreatePtrFromIconInfo( piconinfo );

      // Ergebnis zurckgeben
      *phptr = hptr;
    }

    // Timeout lesen
    ulCurrentId = GetAnimouseStringId( hmodResource, ulResourceId );
    fReadError     |= ( !LOADSTRING( ulCurrentId++, szTimeoutValue ));
    ulPointerCount  = atol( szTimeoutValue );

    if( fReadError ) {
      break;
    }

    // Anzahl prfen
    if( ulPointerCount > *pulEntries ) {
      break;
    }
    *pulEntries = ulPointerCount;
    if( ulPointerCount == 0 ) {
      fSuccess = ( phptrStatic != NULLHANDLE );
      break;
    }

    // pointer lesen
    fSuccess = TRUE;
    for( i = 0; i < *pulEntries; i++ )
    {
      // temporren Speicher freigeben
      if( iconinfoTmp.pIconData ) {
        free( iconinfoTmp.pIconData );
      }

      piconinfo  = ( paiconinfo )  ? paiconinfo + i  : &iconinfoTmp;
      pulTimeout = ( paulTimeout ) ? paulTimeout + i : &ulTimeoutTmp;
      phptr      = ( pahptr )      ? pahptr + i      : NULL;

      // Resource-ID ermitteln: 1 based !
      ulResourceId = GetAnimouseBasePointerId( ulPointerIndex ) + i + 1;

      // ptr direkt lesen, wenn mglich
      GetAnimousePointerResource( hmodResource, ulResourceId, piconinfo );

      if( phptr ) {
        // Pointer erstellen
        hptr = CreatePtrFromIconInfo( piconinfo );

        // Ergebnis zurckgeben
        fSuccess |= ( hptr != NULLHANDLE );
        *phptr = hptr;
      }

      // timeout value lesen
      if( paulTimeout ) {
        if( LOADSTRING( ulCurrentId++, szTimeoutValue )) {
          *pulTimeout = atol( szTimeoutValue );
        }
      }
    }
  } while( FALSE );

  // cleanup
  if( iconinfoTmp.pIconData ) {
    free( iconinfoTmp.pIconData );
  }

  return fSuccess;
}

/*Ŀ
 * Name      : LoadFirstPointerFromAnimouseFile                           
 * Kommentar : ldt ersten Pointer aus Animouse DLL                       
 * Autor     : C.Langanke                                                 
 * Datum     : 23.07.1995                                                 
 * nderung  : 23.07.1995                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : PSZ          - Name der Datei                              
 *             PHPOINTER    - Zeiger auf handle-Variable                  
 *             PICONINFO    - Zeiger auf ICONINFO                         
 * Aufgaben  : - Pointer laden                                            
 * Rckgabe  : BOOL - Flag: Pointer geladen/nicht geladen                 
 *
 */

BOOL
LoadFirstPointerFromAnimouseFile( PSZ pszAnimationFile, PHPOINTER phpointer, PICONINFO piconinfo )
{
  BOOL    fSuccess = FALSE;
  APIRET  rc;
  ULONG   i;
  ULONG   ulResourceId;
  HMODULE hmodResource = NULLHANDLE;
  CHAR    szError[ 20];

  do {
    // Parameter prfen
    if(( pszAnimationFile == NULL ) ||
       ( phpointer        == NULL ) ||
       ( piconinfo        == NULL ))
    {
      break;
    }

    // DLL laden
    rc = DosLoadModule( szError,
                        sizeof( szError ),
                        pszAnimationFile,
                        &hmodResource );
    if( rc != NO_ERROR ) {
      break;
    }

    for( i = 0;
         i < ( sizeof( aulPtr2And ) / sizeof( ULONG ));
         i++ )
    {
      ulResourceId = aulPtr2And[ i];

      // try static pointer first
      fSuccess = GetAnimousePointerResource( hmodResource, ulResourceId, piconinfo );
      if( fSuccess ) {
        break;
      }

      // try first pointer of animation then
      fSuccess = GetAnimousePointerResource( hmodResource, ulResourceId + 1, piconinfo );
      if( fSuccess ) {
        break;
      }
    }

    if( fSuccess ) {
      // Pointer erstellen
      *phpointer = CreatePtrFromIconInfo( piconinfo );

      // ggfs. wieder lschen
      if( *phpointer == NULL ) {
        free( piconinfo->pIconData );
        memset( piconinfo, 0, sizeof( ICONINFO ));
        fSuccess = FALSE;
      }
    }
  } while( FALSE );

  if( hmodResource ) {
    DosFreeModule( hmodResource );
  }
  return fSuccess;
}

/*Ŀ
 * Name      : LoadFirstAnimationFromAnimouseFile                         
 * Kommentar : ldt ersten Pointer aus Animouse DLL                       
 * Autor     : C.Langanke                                                 
 * Datum     : 23.07.1995                                                 
 * nderung  : 23.07.1995                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : PSZ          - Name der Datei                              
 *             PHPOINTER    - Zeiger auf handle-Variable                  
 *             PICONINFO    - Zeiger auf ICONINFO                         
 * Aufgaben  : - Pointer laden                                            
 * Rckgabe  : BOOL - Flag: Pointer geladen/nicht geladen                 
 *
 */

BOOL
LoadFirstAnimationFromAnimouseFile( PSZ pszAnimationFile, PHPOINTER pahpointer, PULONG pulTimeout, PULONG pulEntries )
{
  BOOL     fSuccess = FALSE;
  APIRET   rc;
  ULONG    i;
  ULONG    ulResourceId;
  HMODULE  hmodResource = NULLHANDLE;
  CHAR     szError[ 20];
  ICONINFO iconinfo;
  HPOINTER hptrStatic = NULLHANDLE;

  do {
    // Parameter prfen
    if(( pszAnimationFile  == NULL ) ||
       ( pahpointer        == NULL ) ||
       ( pulEntries        == NULL )) {
      break;
    }

    // DLL laden
    rc = DosLoadModule( szError,
                        sizeof( szError ),
                        pszAnimationFile,
                        &hmodResource );
    if( rc != NO_ERROR ) {
      break;
    }

    for( i = 0;
         i < ( sizeof( aulPtr2And ) / sizeof( ULONG ));
         i++ )
    {
      ulResourceId = aulPtr2And[ i];

      // try static pointer first
      fSuccess = GetAnimousePointerResource( hmodResource, ulResourceId, &iconinfo );
      if( fSuccess ) {
        break;
      }

      // try first pointer of animation then
      fSuccess = GetAnimousePointerResource( hmodResource, ulResourceId + 1, &iconinfo );
      if( fSuccess ) {
        break;
      }
    }

    if( fSuccess ) {
      // free resources
      free( iconinfo.pIconData );

      // load this animation
      fSuccess = LoadPointerFromAnimouseFile( hmodResource,
                                              i,
                                              pahpointer,
                                              NULL,
                                              NULL,
                                              NULL,
                                              pulTimeout,
                                              pulEntries );
    }
  } while( FALSE );

  if( hmodResource ) {
    DosFreeModule( hmodResource );
  }
  if( hptrStatic ) {
    WinDestroyPointer( hptrStatic );
  }
  return fSuccess;
}

/*Ŀ
 * Name      : WriteAnimouseDllFile                                       
 * Comment   :                                                            
 * Author    : C.Langanke                                                 
 * Date      : 25.03.1998                                                 
 * Update    : 25.03.1998                                                 
 * called by : app                                                        
 * calls     : Dos*                                                       
 * Input     : ###                                                        
 * Tasks     : - create resource dll from animation                       
 *             - return values and file offsets                           
 * returns   : APIRET - OS/2 error code                                   
 *
 */

APIRET
WriteAnimouseDllFile( PSZ pszAnimationFileName, BOOL fSaveAll, PPOINTERLIST papl, PSZ pszInfoName, PSZ pszInfoArtist )
{
  APIRET         rc = NO_ERROR;
  ULONG          i, j;
  PPOINTERLIST   ppl;

  ULONG          ulAction, ulBytesWritten;

  HFILE          hfileAnimation = NULLHANDLE;
  ULONG          ulFilePtr;

  CHAR           szValue[ 20];

  CHAR           szModuleName[_MAX_PATH];
  PSZ            pszFileSpec;
  BYTE           abNameTable[ 20];

  ULONG          ulPointerIndex;
  ULONG          ulPtrFirst;
  ULONG          ulPtrLast;

  PDOSEXEHEADER  pdosexeheaderTemplate;
  PLXHEADER      plxheaderTemplate;

  LXHEADER       lxheader;
  POBJECT        pobject;
  POBJECTMAP     pobjectmap;
  PRESOURCE      presource;

  BOOL           fIsDuplicate;
  PRESOURCE      presourceDuplicate;

  ULONG          ulOffsetAdditive     = 0;

  ULONG          ulFilePageSize       = 0;
  ULONG          ulNewObjects         = 0;
  ULONG          ulNewPages           = 0;
  ULONG          ulNewPagesLen        = 0;

  ULONG          ulPtrCount           = 0;
  ULONG          ulStringCount        = 0;
  ULONG          ulStringTableCount   = 0;
  ULONG          ulResourceCount      = 0;
  ULONG          ulResourceLen        = 0;
  ULONG          ulStringResourceLen  = 0;
  ULONG          ulCurrentPage        = 0;
  ULONG          ulStringId           = 0;
  ULONG          ulNonResNameTableLen = 0;

  //                                    { 0    1    2    3    4    5    6    7    8 };
  static ULONG   aulStringBefore[]    = { 0,   1,   2,   5,   7,   3,   6,   8,   9 };
  ULONG          ulStringsBefore      = 0;
  ULONG          ulStringsAfter       = 0;

  PBYTE          pbResourceTable    = NULL;
  PPSZ           ppszStringTable    = NULL;
  PBYTE          pbStringTable      = NULL;
  PBYTE          pbResourceData     = NULL;
  PBYTE          pbNonResNameTable  = NULL;
  OBJECT         object;
  OBJECTMAP      objectmap;
  BYTE           bEntryTable = 0;

  PBYTE          pbData;

  ULONG          ulTemplateNameTableLen = 0;
  ULONG          ulNameTableLen      = 0;
  ULONG          ulResourceTableLen  = 0;
  ULONG          ulStringTableLen;

  extern BINDATA bindataDllTemplate;
  PBYTE          pbLXHeader;
  ULONG          ulDataLen;
  ULONG          ulData;
  ULONG          ulLastPageDataOffset;
  ULONG          ulSizeLeft;

  FUNCENTER();

  do
  {
    // check parms
    if(( pszAnimationFileName == NULL ) ||
       ( papl                 == NULL )) {
      rc = ERROR_INVALID_PARAMETER;
      break;
    }

    // if (!pszInfoName)   pszInfoName   = pszDummy;
    // if (!pszInfoArtist) pszInfoArtist = pszDummy;

    // determine what to save
    ulPointerIndex = papl->ulPtrIndex;
    ulPtrFirst       = ( fSaveAll ) ? 0                 : ulPointerIndex;
    ulPtrLast        = ( fSaveAll ) ? NUM_OF_SYSCURSORS : ulPointerIndex + 1;
    ulStringsBefore  = ( fSaveAll ) ? 0                 : aulStringBefore[ ulPointerIndex];
    ulStringsAfter   = ( fSaveAll ) ? 0                 : NUM_OF_SYSCURSORS - ulStringsBefore;

    // *** Open target file file
    rc = DosOpen( pszAnimationFileName,
                  &hfileAnimation,
                  &ulAction,
                  0,
                  0,
                  OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS,
                  OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_WRITEONLY,
                  NULL );
    if( rc != NO_ERROR ) {
      break;
    }

    // *** write DOS header and DOS stub to target file

    pdosexeheaderTemplate = (PDOSEXEHEADER)&bindataDllTemplate.bData[0];
    WRITEMEMORY( hfileAnimation, pdosexeheaderTemplate, pdosexeheaderTemplate->ulOfsExtHeader );

    plxheaderTemplate = (PLXHEADER)((PBYTE)pdosexeheaderTemplate + pdosexeheaderTemplate->ulOfsExtHeader );
    pbLXHeader = (PBYTE)plxheaderTemplate;

    // *** build resident name table
    // module name is the only entry here

    // cut off path and drive
    strcpy( szModuleName, pszAnimationFileName );
    pszFileSpec = Filespec( szModuleName, FILESPEC_NAME );
    strcpy( szModuleName, pszFileSpec );

    // cut off extension
    pszFileSpec = Filespec( szModuleName, FILESPEC_EXTENSION );
    if( pszFileSpec ) {
      *( pszFileSpec - 1 ) = 0;
    }
    ulNameTableLen  =  strlen( szModuleName ) + 4;
    ulNameTableLen +=  3; // 3 extra bytes ?
    memset( abNameTable, 0, sizeof( abNameTable ));
    abNameTable[ 0] = strlen( szModuleName );
    strcpy( &abNameTable[ 1], szModuleName );

    // *** build resources

    // count the pointers and strings needed
    // resource len is calulated for maximum string length (1 length byte, 4 digits, 1 zero byte)

    #define STRINGRES_MAXLEN 6

    ulCurrentPage = 0;
    ulStringCount += ulStringsBefore + ulStringsAfter;
    for( i = ulPtrFirst, ppl = papl;
         i < ulPtrLast;
         i++, ppl++ )
    {
      // at least one string needed here for the pointer list
      ulStringCount++;
      ulStringResourceLen += STRINGRES_MAXLEN;  // for one string

      // count up pointers
      if( ppl->ulPtrCount ) {
        ulPtrCount          += ppl->ulPtrCount;
        ulStringCount       += ppl->ulPtrCount;
        ulStringResourceLen += STRINGRES_MAXLEN * ppl->ulPtrCount;


        // sum up pointer data length
        // make sure that length is calulated
        // so that pointer data does not cross page boundary
        for( j = 0; j < ppl->ulPtrCount; j++ )
        {
          ulCurrentPage += ppl->iconinfo[ j].cbIconData;
          if( ulCurrentPage > EXE_PAGELEN ) {
            ulResourceLen += EXE_PAGELEN;
            ulCurrentPage  = ppl->iconinfo[ j].cbIconData;
          }
        }
      } else if( ppl->hptrStatic ) {
        // one static pointer, no string needed here
        ulPtrCount++;
      }
    }

    // how many string segments of 16 strings do we need ?
    // add some space for stringtable headers
    ulStringTableCount = ( ulStringCount / 16 ) + 1;
    ulStringResourceLen += ulStringTableCount * ( sizeof( RESSTRINGTABLE ) + sizeof( ULONG ));

    // adjust total resource size
    // - add one page for the last pointers
    ulResourceLen += EXE_PAGELEN;
    // - add the size for the string table
    ulResourceLen += ulStringResourceLen;

    // now we now how many resource entries we need
    ulResourceCount = ulStringTableCount + ulPtrCount;

    // get memory for several tables
    ulResourceTableLen =  ulResourceCount * sizeof( RESOURCE );
    pbStringTable   = malloc( ulStringResourceLen );
    pbResourceTable = malloc( ulResourceTableLen );
    pbResourceData  = malloc( ulResourceLen );
    memset( pbResourceData, 0, ulResourceLen );

    if(( !pbStringTable ) || ( !pbResourceTable ) || ( !pbResourceData )) {
      rc = ERROR_NOT_ENOUGH_MEMORY;
      break;
    }

    // initialize pointer resource variables
    pbData          = pbResourceData;
    presource       = (PRESOURCE)pbResourceTable;

    // collect pointer data
    for( i = ulPtrFirst; i < ulPtrLast; i++ )
    {
      // check buffer overflow
      if( pbData >  ( pbResourceData + ulResourceLen )) {
        rc = ERROR_BUFFER_OVERFLOW;
        break;
      }

      // adress current pointerlist, by AND order !
      ppl = ( fSaveAll ) ? (PPOINTERLIST)papl + aulStringOrder[ i] : papl;

      if( ppl->ulPtrCount ) {
        // ad resource data for this pointerlist
        for( j = 0; j < ppl->ulPtrCount; j++ )
        {
          fIsDuplicate = FALSE;
          // check for duplicate entries
          for( presourceDuplicate = (PRESOURCE)pbResourceTable;
               presourceDuplicate < presource;
               presourceDuplicate++ )
          {
            // check len
            if( presourceDuplicate->ulResSize != ppl->iconinfo[ j].cbIconData ) {
              continue;
            }

            // check data
            if( memcmp( pbResourceData + presourceDuplicate->ulResOffs,
                        ppl->iconinfo[ j].pIconData,
                        ppl->iconinfo[ j].cbIconData ) != NULL ) {
              continue;
            }

            // duplicate found
            fIsDuplicate = TRUE;
            break;
          }

          // now add resource
          if( fIsDuplicate ) {
            ADDRESOURCE( presource,
                         RT_POINTER,
                         aulResourceIdBase[ i] + 1 + j,
                         ppl->iconinfo[ j].cbIconData,
                         pbResourceData,
                         pbResourceData + presourceDuplicate->ulResOffs );
          } else {
            // add new icon data
            ADDRESOURCE( presource,
                         RT_POINTER,
                         aulResourceIdBase[ i] + 1 + j,
                         ppl->iconinfo[ j].cbIconData,
                         pbResourceData,
                         pbData );
            MEMCOPY( pbData, ppl->iconinfo[ j].pIconData, ppl->iconinfo[ j].cbIconData );
            *((PUSHORT)pbData ) = 0;
            pbData += sizeof( USHORT );
          }
        }
      } else if( ppl->hptrStatic ) {
        // one static pointer, no string needed here
        // write pointer to current adress
        ADDRESOURCE( presource,
                     RT_POINTER,
                     aulResourceIdBase[ i],
                     ppl->iconinfoStatic.cbIconData,
                     pbResourceData,
                     pbData );
        MEMCOPY( pbData, ppl->iconinfoStatic.pIconData, ppl->iconinfoStatic.cbIconData );
        *((PUSHORT)pbData ) = 0;
        pbData += sizeof( USHORT );
      }
    }
    if( rc != NO_ERROR ) {
      break;
    }

    // initialize string  resource variables
    ulStringTableLen = ulStringTableCount * 16 * sizeof( PSZ );
    ppszStringTable = malloc( ulStringTableLen );
    memset( ppszStringTable, 0, ulStringTableLen );

    // add strings before single pointer
    for( i = 0; i < ulStringsBefore; i++ )
    {
      *( ppszStringTable + ulStringId ) = strdup( "0" );
      ulStringId++;
    }

    // collect string data data
    for( i = ulPtrFirst; i < ulPtrLast; i++ )
    {
      // add one dummy count for unsused resource id 500
      if( aulStringOrder[ i] == 3 ) {
        *( ppszStringTable + ulStringId ) = "0";
        ulStringId++;
      }

      // adress current pointerlist
      ppl = ( fSaveAll ) ? (PPOINTERLIST)papl + aulStringOrder[ i] : papl;

      // add one string
      _ltoa( ppl->ulPtrCount, szValue, 10 );
      *( ppszStringTable + ulStringId ) = strdup( szValue );
      ulStringId++;

      if( ppl->ulPtrCount ) {
        // ad resource data for this pointerlist
        for( j = 0; j < ppl->ulPtrCount; j++ )
        {
          // add string for timeframe value
          _ltoa( ppl->aulTimer[ j], szValue, 10 );
          *( ppszStringTable + ulStringId ) = strdup( szValue );
          ulStringId++;
        }
      }
    }

    // add strings after single pointer
    for( i = 0; i < ulStringsAfter; i++ )
    {
      *( ppszStringTable + ulStringId ) = strdup( "0" );
      ulStringId++;
    }

    // make sure, that all string resources are written to one page
    pbData = NEXTADRESS( pbResourceData, pbData, ulStringResourceLen );


    // create stringtables
    for( i = 0; i < (( ulStringCount / 16 ) + 1 ); i++ )
    {
      ULONG ulResName;
      ULONG ulTableLen;

      // create the table
      rc = CreateStringtable( pbData,
                              ulResourceLen,
                              i * 16,
                              ppszStringTable + ( i * 16 ),
                              &ulResName,
                              &ulTableLen );
      if( rc != NO_ERROR ) {
        break;
      }

      // add the resource
      ADDRESOURCE( presource,
                   RT_STRING,
                   ulResName,
                   ulTableLen,
                   pbResourceData,
                   pbData );

      // adjust data pointer
      pbData += ulTableLen;
      *((PUSHORT)pbData ) = 0;
      pbData += sizeof( USHORT );
    }
    if( rc != NO_ERROR ) {
      break;
    }

    // now set the resource len to the actual size
    ulResourceLen = pbData - pbResourceData;

    // substract last extra WORD from size
    ulResourceLen -= sizeof( USHORT );

    // build non resident name table

    ulNonResNameTableLen =                   strlen( pszDescription ) + 3       +
                           ( pszInfoName   ? strlen( pszInfoName )    + 3 : 0 ) +
                           ( pszInfoArtist ? strlen( pszInfoArtist )  + 3 : 0 ) +
                           1;                       // end of table byte

    pbNonResNameTable = malloc( ulNonResNameTableLen );
    pbData = pbNonResNameTable;

    // write description
    *pbData = strlen( pszDescription );
    strcpy( pbData + 1, pszDescription );
    pbData += strlen( pszDescription ) + 1;
    *((PUSHORT)pbData ) = ORDINAL_DESCRIPTION;
    pbData += sizeof( USHORT );

    // write name
    if( pszInfoName ) {
      *pbData = strlen( pszInfoName );
      strcpy( pbData + 1, pszInfoName );
      pbData += strlen( pszInfoName ) + 1;
      *((PUSHORT)pbData ) = ORDINAL_INFONAME;
      pbData += sizeof( USHORT );
    }

    // write artist
    if( pszInfoArtist ) {
      *pbData = strlen( pszInfoArtist );
      strcpy( pbData + 1, pszInfoArtist );
      pbData += strlen( pszInfoArtist ) + 1;
      *((PUSHORT)pbData ) = ORDINAL_INFOARTIST;
      pbData += sizeof( USHORT );
    }

    *pbData = 0;

    // *** build entry table

    ulFilePageSize = 1 << plxheaderTemplate->ulPageShift;
    bEntryTable = 0;

    // #############################################################-
    // #############################################################-
    // #############################################################-

    // calculate new count values
    ulNewObjects   = 1;

    // Memory page size is normally 0x1000
    ulNewPages     = ( ulResourceLen / EXE_PAGELEN ) + (( ulResourceLen % EXE_PAGELEN ) ? 1 : 0 );

    // phisical size is bound to file page size (normally 0x200)
    ulNewPagesLen   = ( ulResourceLen / ulFilePageSize )  + (( ulResourceLen % ulFilePageSize )  ? 1 : 0 );
    ulNewPagesLen  *= ulFilePageSize;


    // copy old lxheader to new one and calculate values to be modified
    memcpy( &lxheader, plxheaderTemplate, sizeof( LXHEADER ));

    lxheader.ulMpages     += ulNewPages;                   // add new memory pages
    lxheader.ulFixupSize  += FIXUP_DUMMY_LEN * ulNewPages; // add empty fixup for each new page
    // lxheader.ulLdrSize    = 0;                          // calculated below

    // object table
    lxheader.ulObjCnt     += ulNewObjects;                 // add new memory objects
    ulOffsetAdditive      += ulNewObjects * sizeof( OBJECT );

    // object map table
    lxheader.ulObjMapOfs  += ulOffsetAdditive;
    ulOffsetAdditive      += ulNewPages * sizeof( OBJECTMAP );

    // new: resource table
    lxheader.ulRsrcTabOfs  = lxheader.ulObjMapOfs + ( lxheader.ulMpages * sizeof( OBJECTMAP ));
    lxheader.ulRsrcCnt     = ulResourceCount;
    ulOffsetAdditive      += ulResourceTableLen;

    // resident name table
    ulTemplateNameTableLen = lxheader.ulEntTabOfs - lxheader.ulResTabOfs;
    lxheader.ulResTabOfs  += ulOffsetAdditive;
    ulOffsetAdditive      += ulNameTableLen - ulTemplateNameTableLen;

    // entry table offset
    lxheader.ulEntTabOfs   += ulOffsetAdditive;

    // Fixup Table
    lxheader.ulFPageTabOfs += ulOffsetAdditive;
    ulOffsetAdditive       += ulNewPages * sizeof( ULONG );

    // Fixup Record Table
    lxheader.ulFRecTabOfs  += ulOffsetAdditive;

    // weitere Offsets
    lxheader.ulImpModOfs   += ulOffsetAdditive;
    lxheader.ulImpProcOfs  += ulOffsetAdditive;

    // Start der ersten page berechnen
    lxheader.ulDataPageOfs =
      ((( lxheader.ulImpProcOfs + pdosexeheaderTemplate->ulOfsExtHeader ) / ulFilePageSize ) + 1 )
      * ulFilePageSize;

    // Gre der Pages des Original berechnen
    ulDataLen = plxheaderTemplate->ulNResTabOfs   -
                plxheaderTemplate->ulDataPageOfs;

    // lxheader.ulNResTabOfs  = ulDataLen;

    lxheader.ulNResTabOfs  = lxheader.ulDataPageOfs + ulDataLen + ulNewPagesLen;
    lxheader.ulCbNResTab   = ulNonResNameTableLen;

    // loader size berechnen
    lxheader.ulLdrSize     = lxheader.ulFPageTabOfs -
                             lxheader.ulObjTabOfs;

    // *** write lx header
    WRITESTRUCT( hfileAnimation, lxheader );
    // *** write object table
    MOVEFILEPTR( hfileAnimation, 0 ); // ###

    // write old entries
    pobject   = (POBJECT)((PBYTE)pbLXHeader + plxheaderTemplate->ulObjTabOfs );
    ulDataLen = plxheaderTemplate->ulObjCnt * sizeof( OBJECT );
    WRITEMEMORY( hfileAnimation, pobject, ulDataLen );

    // new entry
    pobject = pobject + plxheaderTemplate->ulObjCnt - 1;
    memset( &object, 0, sizeof( OBJECT ));
    object.ulSize     = ulResourceLen;
    object.ulBase     = 0;
    object.ulFlags    = OBJECT_BIGORDEFAULT |
                        OBJECT_DISCARDABLE  |
                        OBJECT_SHARED       |
                        OBJECT_RESOURCE     |
                        OBJECT_READABLE;
    object.ulPageMap  = pobject->ulPageMap + pobject->ulMapSize;
    object.ulMapSize  = ulNewPages;

    // write new entry
    WRITEMEMORY( hfileAnimation,
                 &object,
                 sizeof( OBJECT ));

    // *** write page table
    // write old entries
    pobjectmap = (POBJECTMAP)((PBYTE)pbLXHeader + plxheaderTemplate->ulObjMapOfs );
    ulDataLen = plxheaderTemplate->ulMpages * sizeof( OBJECTMAP );
    WRITEMEMORY( hfileAnimation, pobjectmap, ulDataLen );

    // determine last page offset
    pobjectmap += plxheaderTemplate->ulMpages - 1;
    ulLastPageDataOffset  = pobjectmap->ulPageDataOffset;
    ulLastPageDataOffset += pobjectmap->usPageSize / ulFilePageSize;

    // write new entries
    ulSizeLeft = ulNewPagesLen;
    for( i = 0; i < ulNewPages; i++ )
    {
      objectmap.ulPageDataOffset = ulLastPageDataOffset;
      objectmap.usPageSize       = MIN( ulSizeLeft, EXE_PAGELEN );
      objectmap.usPageFlags      = 0;
      WRITEMEMORY( hfileAnimation, &objectmap, sizeof( OBJECTMAP ));
      ulLastPageDataOffset      += objectmap.usPageSize / ulFilePageSize;
      ulSizeLeft                -= objectmap.usPageSize;
    }

    // *** write resource table

    WRITEMEMORY( hfileAnimation, pbResourceTable, ulResourceTableLen );
    QUERYFILEPTR( hfileAnimation );

    // *** write
    // - resident name table
    WRITEMEMORY( hfileAnimation, &abNameTable[ 0], ulNameTableLen );  // - resident name table

    // - entry table
    WRITEMEMORY( hfileAnimation, &bEntryTable, sizeof( bEntryTable )); // - empty entry table (1 zero byte)

    // *** write fixup table
    // write old entries
    pbData    = (PBYTE)pbLXHeader + plxheaderTemplate->ulFPageTabOfs;
    ulDataLen = ( plxheaderTemplate->ulMpages + 1 ) * sizeof( ULONG );
    WRITEMEMORY( hfileAnimation, pbData, ulDataLen );

    // write new entries
    ulData = *((PULONG)pbData + plxheaderTemplate->ulMpages );
    for( i = 0; i < ulNewPages; i++ )
    {
      WRITEMEMORY( hfileAnimation, &ulData, sizeof( ULONG ));
    }

    // *** write fixup records
    pbData    = (PBYTE)pbLXHeader + plxheaderTemplate->ulFRecTabOfs;
    ulDataLen = plxheaderTemplate->ulFixupSize - ulDataLen;
    WRITEMEMORY( hfileAnimation, pbData, ulDataLen );

    // *** write memory objects
    // *** move to first page
    QUERYFILEPTR( hfileAnimation );
    if( lxheader.ulDataPageOfs < ulFilePtr ) {
      rc = ERROR_INVALID_FUNCTION;
      break;
    }
    SETFILEPTR( hfileAnimation, lxheader.ulDataPageOfs );

    // write old pages
    pbData    = (PBYTE)pdosexeheaderTemplate + plxheaderTemplate->ulDataPageOfs;
    ulDataLen = plxheaderTemplate->ulNResTabOfs -
                plxheaderTemplate->ulDataPageOfs;
    WRITEMEMORY( hfileAnimation, pbData, ulDataLen );

    // write new pages
    WRITEMEMORY( hfileAnimation, pbResourceData, ulResourceLen );

    // move to non resident name table
    SETFILEPTR( hfileAnimation, lxheader.ulNResTabOfs );

    // write non resident name table
    WRITEMEMORY( hfileAnimation, pbNonResNameTable, ulNonResNameTableLen );
  } while( FALSE );


  // cleanup
  if( hfileAnimation ) {
    DosClose( hfileAnimation );
  }
  if( pbResourceTable ) {
    free( pbResourceTable );
  }
  if( pbStringTable ) {
    free( pbStringTable );
  }
  if( pbResourceData ) {
    free( pbResourceData );
  }
  if( pbNonResNameTable ) {
    free( pbNonResNameTable );
  }
  if( ppszStringTable ) {
    for( i = 0; i < ulStringCount; i++ )
    {
      PPSZ ppszValue = ppszStringTable + i;
      if( ppszValue ) {
        free( ppszValue );
      }
    }
    free( ppszStringTable );
  }

  return rc;
}

/*Ŀ
 * Name      : QueryAnimouseFileDetails                                   
 * Comment   :                                                            
 * Author    : C.Langanke                                                 
 * Date      : 21.05.1998                                                 
 * Update    : 21.05.1998                                                 
 * called by : app                                                        
 * calls     : Dos*                                                       
 * Input     : PSZ    - Name of File                                      
 *             PVOID  - pointer to result buffer                          
 *             ULONG  - buffer len                                        
 *             PULONG - needed buffer size                                
 * Tasks     : - read file details                                        
 * returns   : APIRET - OS/2 error code                                   
 *
 */

APIRET
QueryAnimouseFileDetails( PSZ pszAnimationFileName, ULONG ulInfoLevel, PVOID pvData, ULONG ulBuflen, PULONG pulSize )
{
  APIRET       rc     = NO_ERROR;
  PBYTE        pbData = pvData;
  ULONG        i;
  HFILE        hfileAnimation = NULLHANDLE;
  HMODULE      hmodResource   = NULLHANDLE;
  ULONG        ulAction;
  ULONG        ulBytesRead, ulFilePtr;
  CHAR         szFullName[ _MAX_PATH];
  ANDFILEINFO  andfileinfo;
  DOSEXEHEADER dosexeheader;
  LXHEADER     lxheader;
  ULONG        ulLXOffset;

  PBYTE        pbResTable    = NULL;
  PBYTE        pbObjTable    = NULL;
  PBYTE        pbObjMapTable = NULL;
  PBYTE        pbNonResNameTable = NULL;

  POBJECT      pobject;
  POBJECTMAP   pobjectmap;
  POBJECTMAP   pobjectmapCurrent;
  PRESOURCE    presource;

  PFRAMEINFO   pframeinfo;
  PFRAMEINFO   pframeinfoDuplicate;
  ULONG        ulPtrIndex;

  ULONG        ulResTableOfs;
  ULONG        ulResCount;

  ULONG        ulTotalMapCount = 0;
  PULONG       paulObjOfs;
  ULONG        ulFilePageSize;
  ULONG        ulDataLen;
  ULONG        ulStringId = 0;

  HAB          hab = WinInitialize( 0 );
  BOOL         fOwnHab = ( hab != NULL );
  CHAR         szError[20];
  CHAR         szValue[20];

  ULONG        ulSourceInfoLen;
  SOURCEINFO   sourceinfo;
  PSOURCEINFO  psourceinfo;
  PBYTE        pbEntry;
  USHORT       usOrdinal;

  do
  {
    // get external hab
    if( !fOwnHab ) {
      hab = WinQueryAnchorBlock( HWND_DESKTOP );
    }

    // check parameters
    if(( pszAnimationFileName == NULL ) ||
       ( pulSize              == NULL )) {
      rc = ERROR_INVALID_PARAMETER;
      break;
    }

    // initialize
    memset( &andfileinfo, 0, sizeof( ANDFILEINFO ));
    if( pvData ) {
      memset( pvData, 0, ulBuflen );
    }

    // load DLL for string load
    rc = DosQueryPathInfo( pszAnimationFileName,
                           FIL_QUERYFULLNAME,
                           szFullName,
                           sizeof( szFullName ));
    if( rc != NO_ERROR ) {
      break;
    }

    rc = DosLoadModule( szError,
                        sizeof( szError ),
                        szFullName,
                        &hmodResource );
    if( rc != NO_ERROR ) {
      break;
    }

    // Open and read the lxheader
    rc = DosOpen( pszAnimationFileName,
                  &hfileAnimation,
                  &ulAction,
                  0,
                  0,
                  OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                  OPEN_SHARE_DENYWRITE | OPEN_ACCESS_READONLY,
                  NULL );
    if( rc != NO_ERROR ) {
      break;
    }

    // read DOS EXE header
    rc = DosRead( hfileAnimation,
                  &dosexeheader,
                  sizeof( DOSEXEHEADER ),
                  &ulBytesRead );
    if( rc != NO_ERROR ) {
      break;
    }

    if( dosexeheader.usSig != EXEID ) {
      rc = ERROR_INVALID_DATA;
      break;
    }

    // read lx header
    ulLXOffset = dosexeheader.ulOfsExtHeader;
    SETFILEPTR( hfileAnimation, ulLXOffset );
    rc = DosRead( hfileAnimation,
                  &lxheader,
                  sizeof( LXHEADER ),
                  &ulBytesRead );
    if( rc != NO_ERROR ) {
      break;
    }

    if( lxheader.usSig != SIG_LXHDR ) {
      rc = ERROR_INVALID_DATA;
      break;
    }

    // read resource table
    ulResTableOfs = ulLXOffset + lxheader.ulRsrcTabOfs;
    ulResCount    = lxheader.ulRsrcCnt;
    if( ulResCount ) {
      // resource object table laden
      ulDataLen = ulResCount * sizeof( RESOURCE );
      pbResTable = malloc( ulDataLen );
      if( !pbResTable ) {
        rc = ERROR_NOT_ENOUGH_MEMORY;
        break;
      }

      // read objects
      SETFILEPTR( hfileAnimation, ulResTableOfs );
      rc = DosRead( hfileAnimation,
                    pbResTable,
                    ulDataLen,
                    &ulBytesRead );
      if( rc != NO_ERROR ) {
        break;
      }
    } else {
      // no resources found
      rc = ERROR_INVALID_DATA;
      break;
    }

    // determine numbers of pointers
    for( i = 0, presource = (PRESOURCE)pbResTable; i < ulResCount; i++, presource++ )
    {
      if( presource->usResType == RT_POINTER ) {
        andfileinfo.ulPointerCount++;
      }
    }

    // read non resident name table
    if( lxheader.ulCbNResTab > 0 ) {
      // get memory
      if(( pbNonResNameTable = malloc( lxheader.ulCbNResTab )) == NULL ) {
        rc = ERROR_NOT_ENOUGH_MEMORY;
        break;
      }

      // read it
      SETFILEPTR( hfileAnimation, lxheader.ulNResTabOfs );
      rc = DosRead( hfileAnimation,
                    pbNonResNameTable,
                    lxheader.ulCbNResTab,
                    &ulBytesRead );
      if( rc != NO_ERROR ) {
        break;
      }

      // now determine size of needed entries
      memset( &sourceinfo, 0, sizeof( sourceinfo ));
      pbEntry = pbNonResNameTable;
      ulSourceInfoLen = 0;
      while( *pbEntry != 0 )
      {
        // check the ordinal no
        usOrdinal = *((PUSHORT)( pbEntry + *pbEntry + 1 ));
        switch( usOrdinal )
        {
          case ORDINAL_INFONAME:
            sourceinfo.pszInfoName = malloc( *pbEntry + 1 );
            if( sourceinfo.pszInfoName ) {
              memcpy( sourceinfo.pszInfoName, pbEntry + 1, *pbEntry );
              *( sourceinfo.pszInfoName + *pbEntry ) = 0;
              ulSourceInfoLen += *pbEntry + 1;
            }
            break;

          case ORDINAL_INFOARTIST:
            sourceinfo.pszInfoArtist = malloc( *pbEntry + 1 );
            if( sourceinfo.pszInfoArtist ) {
              memcpy( sourceinfo.pszInfoArtist, pbEntry + 1, *pbEntry );
              *( sourceinfo.pszInfoArtist + *pbEntry ) = 0;
              ulSourceInfoLen += *pbEntry + 1;
            }
            break;
        }

        // adress next entry
        pbEntry += *pbEntry + 3;
      }
    }

    // enough space provided ?
    switch( ulInfoLevel )
    {
      case FI_SOURCEINFO:
        *pulSize = ulSourceInfoLen;
        if( *pulSize > 0 ) {
          *pulSize += sizeof( SOURCEINFO );
        }
        break;

      case FI_DETAILINFO:
        *pulSize = sizeof( ANDFILEINFO ) + ( andfileinfo.ulPointerCount * sizeof( FRAMEINFO ));
        break;
    }

    // no data available
    if( *pulSize == 0 ) {
      break;
    }

    // buffer large enough ?
    if( ulBuflen < *pulSize ) {
      rc = ERROR_MORE_DATA;
      break;
    }

    // --------------------------------------------
    // read non resident table entries only
    if( ulInfoLevel == FI_SOURCEINFO ) {
      // hand over result
      psourceinfo = (PSOURCEINFO)pbData;
      MEMCOPY( pbData, &sourceinfo, sizeof( SOURCEINFO ));

      // copy animation name and artist name
      if( sourceinfo.pszInfoName ) {
        psourceinfo->pszInfoName = (PSZ)pbData;
        MEMCOPY( pbData, sourceinfo.pszInfoName, strlen( sourceinfo.pszInfoName ) + 1 );
        *( pbData - 1 ) = 0;
      } else {
        psourceinfo->pszInfoName = NULL;
      }

      if( sourceinfo.pszInfoArtist ) {
        psourceinfo->pszInfoArtist = (PSZ)pbData;
        MEMCOPY( pbData, sourceinfo.pszInfoArtist, strlen( sourceinfo.pszInfoArtist ) + 1 );
        *( pbData - 1 ) = 0;
      } else {
        psourceinfo->pszInfoArtist = NULL;
      }

      // job done
      break;
    }

    // address frame info table
    andfileinfo.apframeinfo = (PFRAMEINFO)((PBYTE)pvData + sizeof( ANDFILEINFO ));
    pframeinfo = andfileinfo.apframeinfo;

    // determine file page size
    ulFilePageSize = 1 << lxheader.ulPageShift;

    // determine offset table for memory objects
    paulObjOfs = malloc( lxheader.ulObjCnt * sizeof( ULONG ));
    if( !paulObjOfs ) {
      rc = ERROR_NOT_ENOUGH_MEMORY;
      break;
    }

    // read object table
    ulDataLen = lxheader.ulObjCnt * sizeof( OBJECT );
    pbObjTable = malloc( ulDataLen );
    if( !pbObjTable ) {
      rc = ERROR_NOT_ENOUGH_MEMORY;
      break;
    }

    // read object table
    SETFILEPTR( hfileAnimation, ulLXOffset + lxheader.ulObjTabOfs );
    rc = DosRead( hfileAnimation,
                  pbObjTable,
                  ulDataLen,
                  &ulBytesRead );
    if( rc != NO_ERROR ) {
      break;
    }

    // scan object table: determine object map count
    for( i = 0, pobject = (POBJECT)pbObjTable; i < lxheader.ulObjCnt; i++, pobject++ )
    {
      if( pobject->ulMapSize > 0 ) {
        ulTotalMapCount += pobject->ulMapSize;
      }
    }

    // object map table laden
    ulDataLen = ulTotalMapCount * sizeof( OBJECTMAP );
    pbObjMapTable = malloc( ulDataLen );
    if( !pbObjMapTable ) {
      rc = ERROR_NOT_ENOUGH_MEMORY;
      break;
    }

    // read objects
    SETFILEPTR( hfileAnimation, ulLXOffset + lxheader.ulObjMapOfs );
    rc = DosRead( hfileAnimation,
                  pbObjMapTable,
                  ulDataLen,
                  &ulBytesRead );
    if( rc != NO_ERROR ) {
      break;
    }
    pobjectmap = (POBJECTMAP)pbObjMapTable;

    // scan object table
    for( i = 0, pobject = (POBJECT)pbObjTable; i < lxheader.ulObjCnt; i++, pobject++ )
    {
      if( pobject->ulMapSize > 0 ) {
        pobjectmapCurrent = pobjectmap + pobject->ulPageMap - 1;
        *( paulObjOfs + i ) = lxheader.ulDataPageOfs +
                              ( pobjectmapCurrent->ulPageDataOffset * ulFilePageSize );
      } else {
        *( paulObjOfs + i ) = 0;
      }
    }

    // count pointer resources
    if( ulResCount ) {
      for( i = 0, presource = (PRESOURCE)pbResTable; i < lxheader.ulRsrcCnt; i++, presource++ )
      {
        if( presource->usResType == RT_POINTER ) {
          // get ptr index
          ulPtrIndex = GetSystemPointerIndexFromAnimouseId( presource->usResName );
          if( ulPtrIndex > NUM_OF_SYSCURSORS ) {
            continue;
          }

          // store first id for sys pointer
          if( presource->usResName % 100 == 1 ) {
            andfileinfo.apframeinfo1st[ ulPtrIndex] = pframeinfo;
            ulStringId++;
          }

          if( presource->usResName % 100 ) {
            // pointer in animation
            andfileinfo.aulFrameCount[ ulPtrIndex]++;
          } else {
            // static pointer
            andfileinfo.apframeinfoStatic[ ulPtrIndex] = pframeinfo;
          }

          // add to frame info table
          pframeinfo->usResId      = presource->usResName;
          pframeinfo->ulResSize    = presource->ulResSize;
          pframeinfo->ulPtrIndex   = ulPtrIndex;
          pframeinfo->ulResOfs     = presource->ulResOffs +
                                     paulObjOfs[ presource->usResObj - 1];

          // check for duplicates
          for( pframeinfoDuplicate = andfileinfo.apframeinfo;
               pframeinfoDuplicate < pframeinfo;
               pframeinfoDuplicate++ )
          {
            if( pframeinfoDuplicate->ulResOfs == pframeinfo->ulResOfs ) {
              pframeinfo->pvDuplicate = pframeinfoDuplicate;
              break;
            }
          }

          // adress next table entry
          pframeinfo++;
          ulStringId++;
        }     // if (presource->usResType == RT_POINTER)
      }    // for (i = 0, presource = (PRESOURCE) pbResTable; i < lxheader.ulRsrcCnt; i++, presource++)

      // load timeout values
      for( pframeinfo = andfileinfo.apframeinfo;
           pframeinfo < andfileinfo.apframeinfo + andfileinfo.ulPointerCount;
           pframeinfo++ )
      {
        ulStringId = GetAnimouseStringId( hmodResource, pframeinfo->usResId );
        LOADSTRING( ulStringId, szValue );
        pframeinfo->ulTimeout = atol( szValue );
      }
    }

    // hand over result
    memcpy( pvData, &andfileinfo, sizeof( andfileinfo ));
  } while( FALSE );

  // cleanup
  if( hfileAnimation ) {
    DosClose( hfileAnimation );
  }
  if( hmodResource ) {
    DosFreeModule( hmodResource );
  }
  if( pbResTable ) {
    free( pbResTable );
  }
  if( pbObjTable ) {
    free( pbObjTable );
  }
  if( pbObjMapTable ) {
    free( pbObjMapTable );
  }
  if( pbNonResNameTable ) {
    free( pbNonResNameTable );
  }

  if(( fOwnHab ) && ( hab )) {
    WinTerminate( hab );
  }
  return rc;
}

/*Ŀ
 * Name      : SetAnimouseFileInfo                                        
 * Comment   :                                                            
 * Author    : C.Langanke                                                 
 * Date      : 13.04.1998                                                 
 * Update    : 13.04.1998                                                 
 * called by : app                                                        
 * calls     : Dos*                                                       
 * Input     : ###                                                        
 * Tasks     : - write animouse file info                                 
 * returns   : APIRET - OS/2 error code                                   
 *
 */

PBYTE
_WriteNewNameEntry( PBYTE pbNewEntry, PSZ pszName, ULONG ulOrdinal )

{
  *pbNewEntry = strlen( pszName );
  pbNewEntry++;
  MEMCOPY( pbNewEntry, pszName, *( pbNewEntry - 1 ));
  *((PUSHORT)pbNewEntry ) = ulOrdinal;
  pbNewEntry += 2;
  return pbNewEntry;
}

APIRET
SetAnimouseFileInfo( PSZ pszAnimationFileName, PSOURCEINFO psourceinfo )
{
  APIRET       rc             = NO_ERROR;
  HFILE        hfileAnimation = NULLHANDLE;
  ULONG        ulAction;
  ULONG        ulBytesWritten, ulBytesRead, ulFilePtr;

  DOSEXEHEADER dosexeheader;
  LXHEADER     lxheader;
  ULONG        ulLXOffset;

  PBYTE        pbNonResNameTable    = NULL;
  PBYTE        pbNewNonResNameTable = NULL;
  PBYTE        pbEntry;
  PBYTE        pbNewEntry;
  ULONG        ulSourceInfoLen;
  USHORT       usOrdinal;

  BOOL fInfoNameFound     = FALSE;
  BOOL fInfoArtistFound   = FALSE;

  BOOL fInfoNameWritten   = FALSE;
  BOOL fInfoArtistWritten = FALSE;

  BOOL fUseEntry;

  do
  {
    // check parameters
    if(( pszAnimationFileName  == NULL ) ||
       ( *pszAnimationFileName == 0 )    ||
       ( psourceinfo           == NULL ))
    {
      rc = ERROR_INVALID_PARAMETER;
      break;
    }

    // Datei ffnen
    rc = DosOpen( pszAnimationFileName,
                  &hfileAnimation,
                  &ulAction,
                  0, 0,
                  OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                  OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_READWRITE,
                  NULL );

    if( rc != NO_ERROR ) {
      break;
    }

    // read DOS EXE header
    rc = DosRead( hfileAnimation,
                  &dosexeheader,
                  sizeof( DOSEXEHEADER ),
                  &ulBytesRead );
    if( rc != NO_ERROR ) {
      break;
    }

    if( dosexeheader.usSig != EXEID ) {
      rc = ERROR_INVALID_DATA;
      break;
    }

    // read lx header
    ulLXOffset = dosexeheader.ulOfsExtHeader;
    SETFILEPTR( hfileAnimation, ulLXOffset );
    rc = DosRead( hfileAnimation,
                  &lxheader,
                  sizeof( LXHEADER ),
                  &ulBytesRead );
    if( rc != NO_ERROR ) {
      break;
    }

    if( lxheader.usSig != SIG_LXHDR ) {
      rc = ERROR_INVALID_DATA;
      break;
    }

    // read non resident name table
    if( lxheader.ulCbNResTab > 0 ) {
      // get memory
      if(( pbNonResNameTable = malloc( lxheader.ulCbNResTab )) == NULL ) {
        rc = ERROR_NOT_ENOUGH_MEMORY;
        break;
      }

      // read it
      SETFILEPTR( hfileAnimation, lxheader.ulNResTabOfs );
      rc = DosRead( hfileAnimation,
                    pbNonResNameTable,
                    lxheader.ulCbNResTab,
                    &ulBytesRead );
      if( rc != NO_ERROR ) {
        break;
      }

      // now determine size of changed table
      pbEntry = pbNonResNameTable;
      ulSourceInfoLen = 0;
      while( *pbEntry != 0 )
      {
        fUseEntry = TRUE;

        // check the ordinal no
        usOrdinal = *((PUSHORT)( pbEntry + *pbEntry + 1 ));

        switch( usOrdinal )
        {
          case ORDINAL_INFONAME:
            if( psourceinfo->pszInfoName ) {
              ulSourceInfoLen += strlen( psourceinfo->pszInfoName );
              if( *psourceinfo->pszInfoName == 0 ) {
                fUseEntry = FALSE;
              }
            } else {
              ulSourceInfoLen += *pbEntry;
            }
            fInfoNameFound = TRUE;
            break;

          case ORDINAL_INFOARTIST:
            if( psourceinfo->pszInfoArtist ) {
              ulSourceInfoLen += strlen( psourceinfo->pszInfoArtist );
              if( *psourceinfo->pszInfoArtist == 0 ) {
                fUseEntry = FALSE;
              }
            } else {
              ulSourceInfoLen += *pbEntry;
            }
            fInfoArtistFound = TRUE;
            break;

          default:
            ulSourceInfoLen += *pbEntry;
            break;
        }

        // adress next entry
        if( fUseEntry ) {
          ulSourceInfoLen += 3;
        }
        pbEntry += *pbEntry + 3;
      }

      ulSourceInfoLen++; // add end zero byte
    }

    // add length of new entries
    if( !fInfoNameFound ) {
      if( psourceinfo->pszInfoName ) {
        ulSourceInfoLen += strlen( psourceinfo->pszInfoName ) + 3;
      }
    }

    if( !fInfoArtistFound ) {
      if( psourceinfo->pszInfoArtist ) {
        ulSourceInfoLen += strlen( psourceinfo->pszInfoArtist ) + 3;
      }
    }

    // get memory for new table
    if(( pbNewNonResNameTable = malloc( ulSourceInfoLen )) == NULL ) {
      rc = ERROR_NOT_ENOUGH_MEMORY;
      break;
    }

    // assemble new table
    pbEntry = pbNonResNameTable;
    pbNewEntry = pbNewNonResNameTable;

    // modify table
    if( lxheader.ulCbNResTab > 0 ) {
      while( *pbEntry != 0 )
      {
        // check the ordinal no
        usOrdinal = *((PUSHORT)( pbEntry + *pbEntry + 1 ));

        // write lower ordinals
        if( usOrdinal < ORDINAL_FIRST ) {
          MEMCOPY( pbNewEntry, pbEntry, *pbEntry + 3 );
        }

        // write own ordinals
        if( usOrdinal >= ORDINAL_INFONAME ) {
          if( !fInfoNameWritten ) {
            if( psourceinfo->pszInfoName != NULL ) {
              if( *psourceinfo->pszInfoName != 0 ) {
                pbNewEntry = _WriteNewNameEntry( pbNewEntry, psourceinfo->pszInfoName, ORDINAL_INFONAME );
              }
            } else {
              if( usOrdinal == ORDINAL_INFONAME ) {
                MEMCOPY( pbNewEntry, pbEntry, *pbEntry + 3 );
              }
            }
            fInfoNameWritten = TRUE;
          }
        }

        if( usOrdinal >= ORDINAL_INFOARTIST ) {
          if( !fInfoArtistWritten ) {
            if( psourceinfo->pszInfoArtist != NULL ) {
              if( *psourceinfo->pszInfoArtist != 0 ) {
                pbNewEntry = _WriteNewNameEntry( pbNewEntry, psourceinfo->pszInfoArtist, ORDINAL_INFOARTIST );
              }
            } else {
              if( usOrdinal == ORDINAL_INFOARTIST ) {
                MEMCOPY( pbNewEntry, pbEntry, *pbEntry + 3 );
              }
            }
            fInfoArtistWritten = TRUE;
          }
        }

        // write higher ordinals
        if( usOrdinal > ORDINAL_LAST ) {
          MEMCOPY( pbNewEntry, pbEntry, *pbEntry + 3 );
        }

        // adress next entry
        pbEntry += *pbEntry + 3;
      }
    }

    // write onw entries
    if( !fInfoNameWritten ) {
      if( psourceinfo->pszInfoName ) {
        pbNewEntry = _WriteNewNameEntry( pbNewEntry, psourceinfo->pszInfoName, ORDINAL_INFONAME );
      }
    }

    if( !fInfoArtistWritten ) {
      if( psourceinfo->pszInfoArtist ) {
        pbNewEntry = _WriteNewNameEntry( pbNewEntry, psourceinfo->pszInfoArtist, ORDINAL_INFOARTIST );
      }
    }

    // write end-of-table zero byte
    *pbNewEntry = 0;

    // write new table size to lxheader
    SETFILEPTR( hfileAnimation, ulLXOffset + FIELDOFFSET( LXHEADER, ulCbNResTab ));
    rc = DosWrite( hfileAnimation,
                   &ulSourceInfoLen,
                   sizeof( ulSourceInfoLen ),
                   &ulBytesWritten );
    if( rc != NO_ERROR ) {
      break;
    }

    // write new table
    SETFILEPTR( hfileAnimation, lxheader.ulNResTabOfs );
    rc = DosWrite( hfileAnimation,
                   pbNewNonResNameTable,
                   ulSourceInfoLen,
                   &ulBytesWritten );
    if( rc != NO_ERROR ) {
      break;
    }

    // cut of file size if new table is smaller
    if( lxheader.ulCbNResTab > ulSourceInfoLen ) {
      FILESTATUS3 fs3;

      // get file size
      rc = DosQueryFileInfo( hfileAnimation,
                             FIL_STANDARD,
                             &fs3,
                             sizeof( fs3 ));

      fs3.cbFile -= lxheader.ulCbNResTab - ulSourceInfoLen;
      // set file size
      rc = DosSetFileSize( hfileAnimation,
                           fs3.cbFile );
    }
  } while( FALSE );

  // cleanup
  if( hfileAnimation ) {
    DosClose( hfileAnimation );
  }
  if( pbNonResNameTable ) {
    free( pbNonResNameTable );
  }
  if( pbNewNonResNameTable ) {
    free( pbNewNonResNameTable );
  }
  return rc;
}

