/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       dmatrixc.c
**     SYSTEM   NAME:       beholder
**     ORIGINAL AUTHOR(S):  Nitin Ramlal
**     VERSION  NUMBER:     0.99
**     CREATION DATE:       1992/9/29
**
** DESCRIPTION: matrix group of the RMON MIB: collector
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif

#include <stdlib.h>
#include <memory.h>
#include <dnpap.h>
#include <config.h>
#include <message.h>
#include <mac.h>
#include <hash.h>
#include <sys.h>
#include <protocol.h>

#include "dmatrixd.h"
#include "dmatrixe.h"
#include "dmatrixc.h"


static WORD MatrixMaxNrSrcDsts = 8000;


static VOID MatrixCallback(MAC_COLL *collector, PROT_PKT *pkt);

static BOOLEAN MatrixAddSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *host);
static BOOLEAN MatrixRemoveSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *host);

static SRCDST* MatrixAddLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst);
static BOOLEAN MatrixUpdateLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst);
static BOOLEAN MatrixRemoveLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst);

static VOID DelSrcDsts(MATRIX_CONTROL *matrixcontrol);





static BOOLEAN UpdateSourceTop(MATRIX_CONTROL *matrixcontrol, HostInfo *src);
static BOOLEAN UpdateDestTop(MATRIX_CONTROL *matrixcontrol,HostInfo *dest);
static BOOLEAN AddSourceTop (MATRIX_CONTROL *matrixcontrol, HostInfo *src);
static BOOLEAN AddSourceLex ( MATRIX_CONTROL *matrixcontrol, HostInfo *src);
static BOOLEAN AddDestTop (MATRIX_CONTROL *matrixcontrol, HostInfo *dst);
static BOOLEAN AddDestLex ( MATRIX_CONTROL *matrixcontrol, HostInfo *dst);
static HostInfo* SourceSearch(MATRIX_CONTROL *matrixcontrol, HostInfo *searchsrc);
static HostInfo* DstSearch(MATRIX_CONTROL *matrixcontrol, HostInfo *searchdest);
static BOOLEAN MatrixAddDest ( MATRIX_CONTROL *matrixcontrol, DEST *dst);
static BOOLEAN MatrixAddSource ( MATRIX_CONTROL *matrixcontrol, SOURCE *src);
 



BOOLEAN MatrixConfigInit(VOID)
{
	ConfigGetShort("beholder.matrix.maxnrsrcdsts", &MatrixMaxNrSrcDsts);

	if (MatrixMaxNrSrcDsts < 2)
    {
        DnpapMessage(DMC_WARNING, MATRIX_MAX, "matrixcontrol: beholder.matrix.maxnrsrcdsts < 2, setting it to 2");
        MatrixMaxNrSrcDsts = 2;
    }

	return TRUE;
}


BOOLEAN MatrixCInit(MATRIX_CONTROL *matrixcontrol)
{
LONG source[] = {1,3,6,1,2,1,2,2,1,1,1};

    memcpy(matrixcontrol->Source, source, sizeof(source));
    matrixcontrol->SourceLen = sizeof(source)/sizeof(source[0]);
    matrixcontrol->TableSize = 0;
    matrixcontrol->LastDeleteTime = 0;
    matrixcontrol->Owner[0] = '\0';
    matrixcontrol->OwnerLen = 0;
    matrixcontrol->Status = SNMP_INVALID;
    
    if ((matrixcontrol->Iface =
        MacIfaceGet((WORD) matrixcontrol->Source[matrixcontrol->SourceLen-1])) == NULL)
    {
        DnpapMessage(DMC_ERROR, MATRIX_NETINIT, "matrixcontrol: network initialisation failed");
        return (FALSE);
    }
	
    matrixcontrol->Table = NULL;
    matrixcontrol->SrcDstList = NULL;
    matrixcontrol->DstSrcList = NULL;

    matrixcontrol->LRUList = NULL;
    matrixcontrol->LRULast = NULL;

    matrixcontrol->SourceList = NULL;
    matrixcontrol->DestList = NULL;

	return TRUE;
}


BOOLEAN MatrixCStart(MATRIX_CONTROL *matrixcontrol)
{
    matrixcontrol->Coll.Rcve       = MatrixCallback;
    matrixcontrol->Coll.specific   = matrixcontrol;
    if (!MacCollRegister(&(matrixcontrol->Coll)))
    {
        DnpapMessage(DMC_ERROR, MATRIX_NETERR, "matrixcontrol: network initialisation failed");
        return FALSE;
    }

    if ((matrixcontrol->Table = NewHash(5011, NULL)) == NULL)
    {
        DnpapMessage(DMC_ERROR, MATRIX_HASHERR, "matrixcontrol: can not create hashtable");
        return FALSE;
    }

    return TRUE;
}


BOOLEAN MatrixCStop(MATRIX_CONTROL *matrixcontrol)
{
    MacCollRemove(&(matrixcontrol->Coll));

    DelHash(matrixcontrol->Table);

    DelSrcDsts(matrixcontrol);

    return TRUE;
}


VOID MatrixCallback(MAC_COLL *collector, PROT_PKT *pkt)
{
MATRIX_CONTROL *matrixcontrol = collector->specific;
SRCDST *srcdst = NULL, *oldsrcdst = NULL;
BYTE broadcast[HOST_SZEADDR] = {0xff,0xff,0xff,0xff,0xff,0xff};
BYTE srcdstaddr[2*HOST_SZEADDR];
BYTE srcaddr[HOST_SZEADDR];
BYTE dstaddr[HOST_SZEADDR];
PROT_OBJ Interface = {0, {1,2}};
PROT_OBJ Size = {0, {1,4}};
PROT_OBJ Dst = {1, {2,1}};
PROT_OBJ Src = {1, {2,2}};
LWORD size;
SOURCE *src;
DEST *dst;
HostInfo *destin;
HostInfo *sourcein;

HostInfo *insertdest; /* the destination that can be inserted */
HostInfo *insertsource; /* the source that can be inserted */

    if (ProtGetField(pkt,&Interface) == TRUE &&
        Interface.Syntax.LngInt == matrixcontrol->Source[matrixcontrol->SourceLen-1])
    {
        if (ProtGetField(pkt,&Size) == TRUE &&
            ProtGetField(pkt,&Src) == TRUE &&
            ProtGetField(pkt,&Dst) == TRUE)
        {
            size = Size.Syntax.LngUns + 4L;
            if (Src.SyntaxLen != HOST_SZEADDR || Dst.SyntaxLen != HOST_SZEADDR)
            {
                DnpapMessage(DMC_ERROR, MATRIX_INVPACKET, "matrixcontrol: invalid packet");
                return;
            }

            memcpy(srcdstaddr, Src.Syntax.BufChr, HOST_SZEADDR);
            memcpy(srcdstaddr+HOST_SZEADDR, Dst.Syntax.BufChr, HOST_SZEADDR);
			
            if ((srcdst = HashSearch(matrixcontrol->Table, srcdstaddr, 2*HOST_SZEADDR)) == NULL)
            {
                /*  first try to add the new srcdst  */
                if ((srcdst = DnpapMalloc(sizeof(SRCDST))) != NULL)
                {
                    memset(srcdst, 0, sizeof(SRCDST));
                    memcpy(srcdst->SrcDst, srcdstaddr, 2*HOST_SZEADDR);
                    if (HashAdd(matrixcontrol->Table, srcdst->SrcDst, 2*HOST_SZEADDR, srcdst) == NULL)
                    {
                        DnpapFree(srcdst);
                        srcdst = NULL;
                    }
                    else
                    {
                        MatrixAddSrcDstList(matrixcontrol, srcdst);
                        oldsrcdst = MatrixAddLRUList(matrixcontrol, srcdst);
                        matrixcontrol->TableSize++;

                        if (matrixcontrol->TableSize > (LONG)MatrixMaxNrSrcDsts)
                        {
                            MatrixRemoveLRUList(matrixcontrol, oldsrcdst);
                            MatrixRemoveSrcDstList(matrixcontrol, oldsrcdst);
                            HashRemove(matrixcontrol->Table, oldsrcdst->SrcDst, HOST_SZEADDR);
                            DnpapFree(oldsrcdst);
                            matrixcontrol->LastDeleteTime = SysTime();
                            matrixcontrol->TableSize--;
                        }
                    }
                }
            }

           /* add source and destination */

            memcpy(srcaddr,Src.Syntax.BufChr,HOST_SZEADDR);
            memcpy(dstaddr,Dst.Syntax.BufChr,HOST_SZEADDR);

            /* actions for the source */

            if ((src=HashSearch(matrixcontrol->Table,srcaddr,HOST_SZEADDR)) == NULL)   /* if source not found, try to add the new source  */

            {
                if (( src = DnpapMalloc(sizeof(SOURCE))) != NULL)
                {
                    memset(src,0,sizeof(SOURCE));
                    memcpy(src->SourceAddr,srcaddr,HOST_SZEADDR);


                    if (HashAdd(matrixcontrol->Table,src->SourceAddr,HOST_SZEADDR,src)== NULL)   /* cannot add source */
                    { 
  	                    DnpapFree(src);
	                    src = NULL;
	                }
	                else
	                    MatrixAddSource(matrixcontrol,src);  /* add source to list */

                }
            }

            /* actions for the destination */

            if ((dst=HashSearch(matrixcontrol->Table,dstaddr,HOST_SZEADDR)) == NULL)   /* if source not found, try to add the new destination  */

            {
                if (( dst = DnpapMalloc(sizeof(DEST))) != NULL)
                {
                    memset(dst,0,sizeof(DEST));
                    memcpy(dst->DestAddr,dstaddr,HOST_SZEADDR);


                    if (HashAdd(matrixcontrol->Table,dst->DestAddr,HOST_SZEADDR,src)== NULL)   /* cannot add destination */
                    { 
  	                    DnpapFree(dst);
	                    dst = NULL;
	                }
	                else
	                    MatrixAddDest(matrixcontrol,dst);  /* add destination to list */

                }
            }
                              


                            
            if (src != NULL && dst != NULL)
            {
                if (( insertsource = DnpapMalloc(sizeof(HostInfo))) != NULL)
                {

                    memset(insertsource,0,sizeof(HostInfo));
                    memcpy(insertsource->ConnAddr,srcaddr,HOST_SZEADDR);
                }
                else
                {
                    DnpapFree(insertsource);
                    insertsource = NULL;
                }


                if (( insertdest = DnpapMalloc(sizeof(HostInfo))) != NULL)
                {

                    memset(insertdest,0,sizeof(HostInfo));
                    memcpy(insertdest->ConnAddr,dstaddr,HOST_SZEADDR);
                }

                else
                {
                    DnpapFree(insertdest);
                    insertdest = NULL;
                }


                /* for the source */

                destin = DstSearch(matrixcontrol,insertdest);

                if (destin == NULL) /* destination does not occur */
                {
                    src->NumberConnect ++;
                    AddDestLex(matrixcontrol,insertdest);
                    AddDestTop(matrixcontrol,insertdest);
                }


                /* for the destination */
                sourcein = SourceSearch(matrixcontrol,insertsource);

                if (sourcein == NULL)  /* source does not occur */
                {
                    dst->NumberConnect ++;
                    AddSourceLex(matrixcontrol,insertsource);
                    AddSourceTop(matrixcontrol,insertsource);

                }


                insertdest->TotalPkts++;
                insertdest->Octets++;

                insertsource->TotalPkts++;
                insertsource->Octets++;

                UpdateDestTop(matrixcontrol,insertdest); /* update the destination list of the source */
                UpdateSourceTop(matrixcontrol,insertsource); /* update the source list of the destination */

            }     
            /* end */
         

        

            if (srcdst != NULL)
            {
                srcdst->Pkts++;
                srcdst->Octets += size;
                
                MatrixUpdateLRUList(matrixcontrol, srcdst);
            }
        }
    }


    return;
}


BOOLEAN MatrixAddSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
SRCDST *p, *q;
BYTE dstsrc1[2*HOST_SZEADDR], dstsrc2[2*HOST_SZEADDR];

    if (matrixcontrol->SrcDstList == NULL && matrixcontrol->DstSrcList == NULL)
    {
        matrixcontrol->SrcDstList = srcdst;
        srcdst->SrcDstPrev = NULL;
        srcdst->SrcDstNext = NULL;
        
        matrixcontrol->DstSrcList = srcdst;
        srcdst->DstSrcPrev = NULL;
        srcdst->DstSrcNext = NULL;
        
        return TRUE;
    }


    /*  update SrcDstList  */
    for (p = matrixcontrol->SrcDstList, q = NULL; p != NULL; q = p, p = p->SrcDstNext)
    {
        if (memcmp(p->SrcDst, srcdst->SrcDst, 2*HOST_SZEADDR) >= 0)
            break;
    }

    if (q != NULL)
        q->SrcDstNext = srcdst;
    else
        matrixcontrol->SrcDstList = srcdst;

    if (p != NULL)
        p->SrcDstPrev = srcdst;

    srcdst->SrcDstPrev = q;
    srcdst->SrcDstNext = p;


    /*  update DstSrcList  */
    memcpy(dstsrc2, srcdst->SrcDst+HOST_SZEADDR, HOST_SZEADDR);
    memcpy(dstsrc2+HOST_SZEADDR, srcdst->SrcDst, HOST_SZEADDR);
    for (p = matrixcontrol->DstSrcList, q = NULL; p != NULL; q = p, p = p->DstSrcNext)
    {
    	memcpy(dstsrc1, p->SrcDst+HOST_SZEADDR, HOST_SZEADDR);
    	memcpy(dstsrc1+HOST_SZEADDR, p->SrcDst, HOST_SZEADDR);
        if (memcmp(dstsrc1, dstsrc2, 2*HOST_SZEADDR) >= 0)
            break;
    }

    if (q != NULL)
        q->DstSrcNext = srcdst;
    else
        matrixcontrol->DstSrcList = srcdst;

    if (p != NULL)
        p->DstSrcPrev = srcdst;

    srcdst->DstSrcPrev = q;
    srcdst->DstSrcNext = p;
        

    return TRUE;
}


SRCDST* MatrixAddLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (matrixcontrol->LRUList == NULL)
    {
        matrixcontrol->LRUList = srcdst;
        matrixcontrol->LRULast = srcdst;
        srcdst->LRUPrev = NULL;
        srcdst->LRUNext = NULL;
        
        return NULL;
    }

    srcdst->LRUNext = matrixcontrol->LRUList;
    srcdst->LRUPrev = NULL;
    matrixcontrol->LRUList->LRUPrev = srcdst;
    matrixcontrol->LRUList = srcdst;

    return matrixcontrol->LRULast;
}


BOOLEAN MatrixUpdateLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (matrixcontrol->TableSize > 1 && matrixcontrol->LRUList != srcdst)
    {          
        MatrixRemoveLRUList(matrixcontrol, srcdst);
        MatrixAddLRUList(matrixcontrol, srcdst);
    }
    return TRUE;
}


BOOLEAN MatrixRemoveSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (srcdst->SrcDstPrev != NULL)
        srcdst->SrcDstPrev->SrcDstNext = srcdst->SrcDstNext;
    else
        matrixcontrol->SrcDstList = srcdst->SrcDstNext;
    if (srcdst->SrcDstNext != NULL)
        srcdst->SrcDstNext->SrcDstPrev = srcdst->SrcDstPrev;

    if (srcdst->DstSrcPrev != NULL)
        srcdst->DstSrcPrev->DstSrcNext = srcdst->DstSrcNext;
    else
        matrixcontrol->DstSrcList = srcdst->DstSrcNext;
    if (srcdst->DstSrcNext != NULL)
        srcdst->DstSrcNext->DstSrcPrev = srcdst->DstSrcPrev;

    return TRUE;
}


BOOLEAN MatrixRemoveLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (srcdst->LRUPrev != NULL)
        srcdst->LRUPrev->LRUNext = srcdst->LRUNext;
    else
        matrixcontrol->LRUList = srcdst->LRUNext;
    if (srcdst->LRUNext != NULL)
        srcdst->LRUNext->LRUPrev = srcdst->LRUPrev;
    else
        matrixcontrol->LRULast = srcdst->LRUPrev;

    return TRUE;
}


VOID DelSrcDsts(MATRIX_CONTROL *matrixcontrol)
{
SRCDST *srcdst1, *srcdst2;

    for (srcdst2 = matrixcontrol->SrcDstList; srcdst2 != NULL; srcdst2 = srcdst1)
    {
        srcdst1 = srcdst2->SrcDstNext;
        DnpapFree(srcdst2);
        matrixcontrol->TableSize--;
    }
    matrixcontrol->SrcDstList = NULL;
    matrixcontrol->DstSrcList = NULL;
}


SRCDST* MatrixHashSearch(MATRIX_CONTROL *matrixcontrol, BYTE *srcdst, WORD len, BOOLEAN SDorder)
{
SRCDST* p;
BYTE srcdst2[2*HOST_SZEADDR], dstsrc[2*HOST_SZEADDR];

	if (SDorder == TRUE)
	{
	    if (len == 2*HOST_SZEADDR &&
	        (p = HashSearch(matrixcontrol->Table, srcdst, 2*HOST_SZEADDR)) != NULL)
	        return p;
	    else
	    {
	        for (p = matrixcontrol->SrcDstList; p != NULL; p = p->SrcDstNext)
	        {
	            if (memcmp(p->SrcDst, srcdst, len) >= 0)
	                break;
	        }
	        return p;
	    }
	    return NULL;
    }
    else
    {   /*  srcdst contains destination and source in that order, so switch them  */
    	memcpy(srcdst2, srcdst+HOST_SZEADDR, HOST_SZEADDR);
    	memcpy(srcdst2+HOST_SZEADDR, srcdst, HOST_SZEADDR);
	    if (len == 2*HOST_SZEADDR &&
	        (p = HashSearch(matrixcontrol->Table, srcdst2, 2*HOST_SZEADDR)) != NULL)
	        return p;
	    else
	    {
	        for (p = matrixcontrol->DstSrcList; p != NULL; p = p->DstSrcNext)
	        {
                /*  switch source and destination of the srcdst(p) under examination  */
		    	memcpy(dstsrc, p->SrcDst+HOST_SZEADDR, HOST_SZEADDR);
		    	memcpy(dstsrc+HOST_SZEADDR, p->SrcDst, HOST_SZEADDR);
		    
                /*  compare destination and source ordered 'addresses'  */
	            if (memcmp(dstsrc, srcdst, len) >= 0)
	                break;
	        }
	        return p;
	    }
	    return NULL;
	}
}


SRCDST* MatrixHashSearch2(MATRIX_CONTROL *matrixcontrol, SNMP_OBJECT *obj, WORD idlen, BOOLEAN SDorder)
{
WORD i;
BYTE srcdst[2*HOST_SZEADDR];

	if (SDorder == TRUE)
	{
	    for (i = 0; i < 2*HOST_SZEADDR; i++)
	        srcdst[i] = (BYTE)obj->Id[idlen+1+i];
	    return HashSearch(matrixcontrol->Table, srcdst, 2*HOST_SZEADDR);
    }
    else
    {
	    for (i = 0; i < HOST_SZEADDR; i++)
        {
	        srcdst[i] = (BYTE)obj->Id[idlen+1+HOST_SZEADDR+i];
	        srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[idlen+1+i];
        }
	    return HashSearch(matrixcontrol->Table, srcdst, 2*HOST_SZEADDR);
    }
}




/* new functions */


/*******************************************************************

** NAME : MatrixAddSource
** SYNOPSIS : BOOLEAN MatrixAddSource(MATRIX_CONTROL *matrixcontrol,
              SOURCE *src)
** PARAMETERS :
** DESCRIPTION : Each time a new source arrives, the source is added
                 to the list of sources

*******************************************************************/



BOOLEAN MatrixAddSource ( MATRIX_CONTROL *matrixcontrol, SOURCE *src)

{

SOURCE *a, *b;

if (matrixcontrol->SourceList == NULL)  /* no sources */
{
  matrixcontrol->SourceList = src;
  src->NextSource = NULL;
  return TRUE;
}

for (a = matrixcontrol->SourceList,b = NULL;a != NULL;b=a,a=a->NextSource)
{
	if (memcmp(a->SourceAddr,src->SourceAddr,HOST_SZEADDR) >= 0)
           break;
}

if (b != NULL)
   b->NextSource = src;
else
   matrixcontrol->SourceList = src;

src->NextSource = a;

return TRUE;

}


/*******************************************************************

** NAME : MatrixAddDest
** SYNOPSIS : BOOLEAN MatrixAddDest(MATRIX_CONTROL *matrixcontrol,
              DEST *dst)
** PARAMETERS :
** DESCRIPTION : Each time a new destination arrives, the destination
                 source is added to the list of destinations

*******************************************************************/



BOOLEAN MatrixAddDest ( MATRIX_CONTROL *matrixcontrol, DEST *dst)

{

DEST *a, *b;

if (matrixcontrol->DestList == NULL)  /* no destination */
{
  matrixcontrol->DestList = dst;
  dst->NextDest = NULL;
  return TRUE;
}

for (a = matrixcontrol->DestList,b = NULL;a != NULL;b=a,a=a->NextDest)
{
	if (memcmp(a->DestAddr,dst->DestAddr,HOST_SZEADDR) >= 0)
           break;
}

if (b != NULL)
   b->NextDest = dst;
else
   matrixcontrol->DestList = dst;

dst->NextDest = a;

return TRUE;

}


/*******************************************************************

** NAME : DstSearch
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : Dstsearch searches a destination in the source table 
*******************************************************************/



HostInfo* DstSearch(MATRIX_CONTROL *matrixcontrol, HostInfo *searchdest)

{
HostInfo* a;


if (matrixcontrol->SourceList->Info == NULL)  /* there is no destination */
    return NULL;
else
{


    for (a=matrixcontrol->SourceList->Info;a!=NULL;a=a->LexNextList)

    {

        if (memcmp(a->ConnAddr,searchdest->ConnAddr,HOST_SZEADDR) == 0)
            break;
    }
    return a;
}
return NULL;
}



/*******************************************************************

** NAME : SourceSearch
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : SourceSearch searches a source in the destination table 
*******************************************************************/



                
HostInfo* SourceSearch(MATRIX_CONTROL *matrixcontrol,HostInfo *searchsrc)
{
HostInfo* a;


if (matrixcontrol->DestList->Info == NULL)  /* there is no source*/
    return NULL;
else
{


    for (a=matrixcontrol->DestList->Info;a!=NULL;a=a->LexNextList)

    {

        if (memcmp(a->ConnAddr,searchsrc->ConnAddr,HOST_SZEADDR) == 0)
            break;
    }
    return a;
}
return NULL;
}



/*******************************************************************

** NAME : AddDestLex
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : AddDestLex adds a new destination to the source 
*******************************************************************/




BOOLEAN AddDestLex ( MATRIX_CONTROL *matrixcontrol, HostInfo *dst)
{
HostInfo  *s, *t;


/* sort the destinations */

if (matrixcontrol->SourceList->Info == NULL)  /* there is not dest */
   {
    matrixcontrol->SourceList->Info = dst;
    dst->LexNextList = NULL;
    dst->LexPrevList = NULL;
    matrixcontrol->SourceList->LastDest = NULL;
    return TRUE;
   }

   
for (s=matrixcontrol->SourceList->Info,t=NULL;s!=NULL;t=s,s=s->LexNextList)
 
   {    /* if destaddr > s */
	if (memcmp(s->ConnAddr,dst->ConnAddr,HOST_SZEADDR) >= 0) 
           break;
   }

if (t != NULL)
   t->LexNextList = dst;
else
   matrixcontrol->SourceList->Info = dst;

if (s != NULL)
   s->LexPrevList = dst;


dst->LexPrevList = t;
dst->LexNextList = s;
/*matrixcontrol->SourceList->LastDest = dst;*/



return TRUE;

}




/*******************************************************************

** NAME : AddDestTop
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : AddDestTop adds a new destination to the end of the list of sources
*******************************************************************/


BOOLEAN AddDestTop (MATRIX_CONTROL *matrixcontrol, HostInfo *dst)
{

HostInfo *h;

if (matrixcontrol->SourceList->LastDest == NULL)
{
    matrixcontrol->SourceList->LastDest = dst;
    dst->TopNextList = NULL;
    dst->TopPrevList = NULL;
    return TRUE;
}


h = matrixcontrol->SourceList->LastDest;
h->TopNextList = dst;
dst->TopPrevList = h;
matrixcontrol->SourceList->LastDest = dst;
dst->TopNextList = NULL;
return TRUE;
}
  


/*******************************************************************

** NAME : AddSourceLex
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : AddSourceLex adds a new source to the destination 
*******************************************************************/




BOOLEAN AddSourceLex ( MATRIX_CONTROL *matrixcontrol, HostInfo *src)

{
 HostInfo  *s, *t;


/* sort the destinations */

if (matrixcontrol->DestList->Info == NULL)  /* there is no source */
   {
    matrixcontrol->DestList->Info = src;
    src->LexNextList = NULL;
    src->LexPrevList = NULL;
    matrixcontrol->DestList->LastSource = NULL;
    return TRUE;
   }

   
for (s=matrixcontrol->DestList->Info,t = NULL;s != NULL;t=s,s=s->LexNextList)
 
   {    /* if destaddr > s */
	if (memcmp(s->ConnAddr,src->ConnAddr,HOST_SZEADDR) >= 0) 
           break;
   }

if (t != NULL)
   t->LexNextList = src;
else
   matrixcontrol->DestList->Info = src;

if (s != NULL)
   s->LexPrevList = src;


src->LexPrevList = t;
src->LexNextList = s;
/*matrixcontrol->HostList->LastDest = dst;*/



return TRUE;

}



/*******************************************************************

** NAME : AddSourceTop
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : AddSourceTop adds a new source to the end of the list of destinations
*******************************************************************/


BOOLEAN AddSourceTop (MATRIX_CONTROL *matrixcontrol, HostInfo *src)
{

HostInfo *h;

if (matrixcontrol->DestList->LastSource == NULL)
{
    matrixcontrol->DestList->LastSource = src;
    src->TopNextList = NULL;
    src->TopPrevList = NULL;
    return TRUE;
}


h = matrixcontrol->DestList->LastSource;
h->TopNextList = src;
src->TopPrevList = h;
matrixcontrol->DestList->LastSource = src;
src->TopNextList = NULL;
return TRUE;
}
  



 

/*******************************************************************

** NAME : UpdateDestTop
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : UpdateDestTop updates the destination list of a source
*******************************************************************/


/* updates the destination list of a source */

BOOLEAN UpdateDestTop(MATRIX_CONTROL *matrixcontrol,HostInfo *dest)
{
HostInfo *a, *b;



for (a=matrixcontrol->SourceList->Info,b=NULL;a!=NULL;b=a,a=a->TopNextList)
{

   if (a->TotalPkts < dest->TotalPkts)
      break;

}

if (b != NULL)
    b->TopNextList = dest;

else
   matrixcontrol->SourceList->Info->TopNextList = dest;

if (a != NULL)
    a->TopPrevList = dest;


dest->TopPrevList = b;
dest->TopNextList = a;

return TRUE;
} 




/*******************************************************************

** NAME : UpdateSourceTop
** SYNOPSIS : 
** PARAMETERS :
** DESCRIPTION : UpdateSourceTop updates the source list of a destination
*******************************************************************/





BOOLEAN UpdateSourceTop(MATRIX_CONTROL *matrixcontrol, HostInfo *src)
{
HostInfo *a, *b;



for (a=matrixcontrol->DestList->Info,b=NULL;a!=NULL;b=a,a=a->TopNextList)
{

   if (a->TotalPkts < src->TotalPkts)
      break;

}

if (b != NULL)
    b->TopNextList = src;

else
   matrixcontrol->DestList->Info->TopNextList = src;

if (a != NULL)
    a->TopPrevList = src;


src->TopPrevList = b;
src->TopNextList = a;

return TRUE;
}



/*******************************************************************

** NAME : SourceDestHashSearch
** SYNOPSIS : 
** PARAMETERS : matrixcontrol, srcdst, len
** DESCRIPTION : SourceDestHashSearch searches a destination for a 
** source, it also accounts for situations when the length is not
** fully specified 
*******************************************************************/




HostInfo* SourceDestHashSearch(MATRIX_CONTROL *matrixcontrol, SNMP_OBJECT *obj,WORD len, BOOLEAN SDorder)
{
HostInfo *t;
SOURCE* q;
INT i;
BYTE src[HOST_SZEADDR];
BYTE dst[HOST_SZEADDR];
BYTE srcdst[2*HOST_SZEADDR];


    if (SDorder == TRUE)
    {
        for (i=0;i<2*HOST_SZEADDR;i++)
            srcdst[i] = (BYTE)obj->Id[len+1+i];

    }
    else
    {
        for (i=0;i<HOST_SZEADDR;i++)
        {
            srcdst[i] = (BYTE)obj->Id[len+1+HOST_SZEADDR+i];
            srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[len+1+i];
        }
    }

    for (i=0;i<HOST_SZEADDR;i++)
    {
        src[i] = srcdst[i];
        dst[i] = srcdst[i+HOST_SZEADDR];
    }
    if (len >= HOST_SZEADDR && len <= 2*HOST_SZEADDR)
    {


        if ((q = HashSearch(matrixcontrol->Table, src, HOST_SZEADDR)) != NULL)
	    {
            for (t=q->Info;t!=NULL;t=t->LexNextList)
            {
                if (memcmp(t->ConnAddr, dst,HOST_SZEADDR) >= 0)
                break;
            }
            return t;
        }
        else  
        {
            for (q=matrixcontrol->SourceList;q!=NULL;q=q->NextSource)
            {
                if (memcmp(q->SourceAddr,src,HOST_SZEADDR) >= 0)
                    break;
            }

            if (q == NULL)  /* no appropriate source found */
                return NULL;
            else
            {
    
                for (t=q->Info;t!=NULL;t=t->LexNextList)
                {
                    if (memcmp(t->ConnAddr, dst,HOST_SZEADDR) >= 0)
                        break;
                }
                return t;
            }
        }
    
    }
    else    /* thus len < HOST_SZEADDR */
    {
        for (q=matrixcontrol->SourceList;q!=NULL;q=q->NextSource)
        {
            if (memcmp(q->SourceAddr,src,HOST_SZEADDR) >= 0)
                break;
        }

        if (q == NULL)  /* no appropriate source found */
        return NULL;
        else
        {
    
            for (t=q->Info;t!=NULL;t=t->LexNextList)
            {
                if (memcmp(t->ConnAddr,dst,HOST_SZEADDR) >= 0)
                    break;
            }
            return t;
        }
    }
}


/*******************************************************************

** NAME : DestSourceHashSearch
** SYNOPSIS : 
** PARAMETERS : matrixcontrol, srcdst, len
** DESCRIPTION : DestSourceHashSearch searches a source for a 
** destination, it also accounts for situations when the length is not
** fully specified 
*******************************************************************/


HostInfo* DestSourceHashSearch(MATRIX_CONTROL *matrixcontrol,SNMP_OBJECT *obj, WORD len,BOOLEAN SDorder)
{
HostInfo* t;
DEST *q;
INT i;
BYTE src[HOST_SZEADDR];
BYTE dst[HOST_SZEADDR];
BYTE srcdst[2*HOST_SZEADDR];


    if (SDorder == TRUE)
    {
        for (i=0;i<2*HOST_SZEADDR;i++)
            srcdst[i] = (BYTE)obj->Id[len+1+i];

    }
    else
    {
        for (i=0;i<HOST_SZEADDR;i++)
        {
            srcdst[i] = (BYTE)obj->Id[len+1+HOST_SZEADDR+i];
            srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[len+1+i];
        }
    }

    for (i=0;i<HOST_SZEADDR;i++)
        dst[i] = srcdst[i+HOST_SZEADDR];

    for (i=0;i<HOST_SZEADDR;i++)
        src[i] = srcdst[i];

    if (len >= HOST_SZEADDR && len <= 2*HOST_SZEADDR)
    {

        if ((q = HashSearch(matrixcontrol->Table, dst, HOST_SZEADDR)) != NULL)
	    {
            for (t=q->Info;t!=NULL;t=t->LexNextList)
            {
                if (memcmp(t->ConnAddr,src,HOST_SZEADDR) >= 0)
                break;
            }
            return t;
        }
        else  /* if source not found */
        {
            for (q=matrixcontrol->DestList;q!=NULL;q=q->NextDest)
            {
                if (memcmp(q->DestAddr,src,HOST_SZEADDR) >= 0)
                    break;
            }

            if (q == NULL)  /* no appropriate source found */
                return NULL;
            else
            {
    
                for (t=q->Info;t!=NULL;t=t->LexNextList)
                {
                    if (memcmp(t->ConnAddr, src,HOST_SZEADDR) >= 0)
                        break;
                }
                return t;
            }
        }
    
    }
    else    
    {
        for (q=matrixcontrol->DestList;q!=NULL;q=q->NextDest)
        {
            if (memcmp(q->DestAddr,src,HOST_SZEADDR) >= 0)
                break;
        }

        if (q == NULL)  /* no appropriate source found */
        return NULL;
        else
        {
    
            for (t=q->Info;t!=NULL;t=t->LexNextList)
            {
                if (memcmp(t->ConnAddr, src,HOST_SZEADDR) >= 0)
                    break;
            }
            return t;
        }
        return NULL;

    }
}



/*******************************************************************

** NAME : SourceDestHashSearch2
** SYNOPSIS : 
** PARAMETERS : matrixcontrol, obj , idlen
** DESCRIPTION : SourceDestHashSearch simply searches a destination for a 
** source
*******************************************************************/
                                                            
HostInfo* SourceDestHashSearch2 (MATRIX_CONTROL *matrixcontrol, SNMP_OBJECT *obj, WORD idlen, BOOLEAN SDorder)
{
WORD i;
BYTE srcdst[2*HOST_SZEADDR];
SOURCE *q;
HostInfo *t;
BYTE src[HOST_SZEADDR];
BYTE dst[HOST_SZEADDR];

    if (SDorder == TRUE)
    {
        for (i=0;i<2*HOST_SZEADDR;i++)
            srcdst[i] = (BYTE)obj->Id[len+1+i];

    }
    else
    {
        for (i=0;i<HOST_SZEADDR;i++)
        {
            srcdst[i] = (BYTE)obj->Id[len+1+HOST_SZEADDR+i];
            srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[len+1+i];
        }
    }



    for (i=0;i<HOST_SZEADDR;i++)
    {
        src[i] = srcdst[i];
        dst[i] = srcdst[i+HOST_SZEADDR];
    }

    q = HashSearch(matrixcontrol->Table,src,HOST_SZEADDR);
    if (q!=NULL)
    {
        for (t=q->Info;t!=NULL;t=t->LexNextList)
        {
            if (memcmp(t->ConnAddr,dst,HOST_SZEADDR) == 0)
               break;
        }
        return t;

    }
    else
        return NULL;
}


/*******************************************************************

** NAME : DestSourceHashSearch2
** SYNOPSIS : 
** PARAMETERS : matrixcontrol, srcdst, len
** DESCRIPTION : DestSourceHashSearch simply searches a source for a 
** destination, if not found then NULL
*******************************************************************/

    if (SDorder == TRUE)
    {
        for (i=0;i<2*HOST_SZEADDR;i++)
            srcdst[i] = (BYTE)obj->Id[len+1+i];

    }
    else
    {
        for (i=0;i<HOST_SZEADDR;i++)
        {
            srcdst[i] = (BYTE)obj->Id[len+1+HOST_SZEADDR+i];
            srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[len+1+i];
        }
    }
HostInfo* DestSourceHashSearch2 (MATRIX_CONTROL *matrixcontrol, SNMP_OBJECT *obj, WORD idlen,BOOLEAN SDorder)
{
WORD i;
BYTE srcdst[2*HOST_SZEADDR];
BYTE src[HOST_SZEADDR];
BYTE dst[HOST_SZEADDR];
DEST *p;
HostInfo *t;



    if (SDorder == TRUE)
    {
        for (i=0;i<2*HOST_SZEADDR;i++)
            srcdst[i] = (BYTE)obj->Id[len+1+i];

    }
    else
    {
        for (i=0;i<HOST_SZEADDR;i++)
        {
            srcdst[i] = (BYTE)obj->Id[len+1+HOST_SZEADDR+i];
            srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[len+1+i];
        }
    }

    for (i=0;i<2*HOST_SZEADDR;i++)
    {
        srcdst[i] = (BYTE)obj->Id[idlen+1+HOST_SZEADDR+i];
        srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[idlen+1+i];
    }


    for (i=0;i<HOST_SZEADDR;i++)
    {
        dst[i] = srcdst[i+HOST_SZEADDR];
        src[i] = srcdst[i];
    }

    p = HashSearch(matrixcontrol->Table,dst,HOST_SZEADDR);
    if (p!=NULL)
    {
        for (t=p->Info;t!=NULL;t=t->LexNextList)
        {
            if (memcmp(t->ConnAddr,src,HOST_SZEADDR) == 0)
                break;
        }
        return t;
    }
    else
        return NULL;
}




/*******************************************************************

** NAME : SourceDestTopSearch
** SYNOPSIS : 
** PARAMETERS : matrixcontrol, srcdst, len
** DESCRIPTION : DestSourceHashSearch simply searches a source for a 
** destination, if not found then NULL
*******************************************************************/

HostInfo* SourceDestTopSearch(MATRIX_CONTROL *matrixcontrol,SNMP_OBJECT *obj,WORD idlen)
{
WORD i;
BYTE srcdst[2*HOST_SZEADDR];
SOURCE *q;
HostInfo *t;
BYTE src[HOST_SZEADDR];
BYTE dst[HOST_SZEADDR];

 
    for (i=0;i<2*HOST_SZEADDR;i++)
        srcdst[i] = (BYTE)obj->Id[idlen+1+i];


    for (i=0;i<HOST_SZEADDR;i++)
    {
        src[i] = srcdst[i];
        dst[i] = srcdst[i+HOST_SZEADDR];
    }

    q = HashSearch(matrixcontrol->Table,src,HOST_SZEADDR);
    if (q!=NULL)
    {
        for (t=q->Info;t!=NULL;t=t->TopNextList)
        {
            if (memcmp(t->ConnAddr,dst,HOST_SZEADDR) == 0)
               break;
        }
        return t;

    }

}



/*******************************************************************

** NAME : DestSourceTopSearch
** SYNOPSIS : 
** PARAMETERS : matrixcontrol, srcdst, len
** DESCRIPTION : DestSourceHashSearch simply searches a source for a 
** destination, if not found then NULL
*******************************************************************/

HostInfo* DestSourceTopSearch(MATRIX_CONTROL *matrixcontrol,SNMP_OBJECT *obj,WORD idlen)
{
WORD i;
BYTE srcdst[2*HOST_SZEADDR];
BYTE src[HOST_SZEADDR];
BYTE dst[HOST_SZEADDR];
DEST *p;
HostInfo *t;


    for (i=0;i<2*HOST_SZEADDR;i++)
    {
        srcdst[i] = (BYTE)obj->Id[idlen+1+HOST_SZEADDR+i];
        srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[idlen+1+i];
    }


    for (i=0;i<HOST_SZEADDR;i++)
    {
        dst[i] = srcdst[i+HOST_SZEADDR];
        src[i] = srcdst[i];
    }

    p = HashSearch(matrixcontrol->Table,dst,HOST_SZEADDR);
    if (p!=NULL)
    {
        for (t=p->Info;t!=NULL;t=t->TopNextList)
        {
            if (memcmp(t->ConnAddr,src,HOST_SZEADDR) == 0)
                break;
        }
        return t;
    }
    else
        return NULL;

}


