/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          timer.c
** MODULE NAME:        timer
** ORIGINAL AUTHOR(S): Dirk Wisse
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/1/21 (y/m/d)
** DESCRIPTION:        is a module for timing purposes.
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************
** COPYRIGHT:   CARDIT (c) 1991
**              Department Of Computer Architecture and Digital
**              Techniques
**              Electrotechnical Faculty
**              Delft University of Technology
**              All rights reserved.
************************************************************************/
#include <time.h>
#ifdef OS2
#include <process.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <limits.h>
#include <dnpap.h>
#include "timer.h"

#ifndef UNIX
#if CLK_TCK==1000
    #define TIMER_CLOCK (clock())
#else
    #define TIMER_CLOCK ((clock()/CLK_TCK)*1000L)
#endif
#else
    #define TIMER_CLOCK (clock()/1000L)
#endif


TIMER_DESCR    *timerList=NULL;
ULONG           timerOffset;

static VOID     Insert(TIMER_DESCR *timer);
static VOID     Delete(TIMER_DESCR *timer);

/*****************************************************************
** NAME:        TimerInit
** SYNOPSIS:    BOOLEAN TimerInit(VOID)  
** DESCRIPTION: initializes timer module.
**              Finds the global info segment and starts a thread
**              that periodically generates TIMER_EVENT_EXPIRE
**              events.
**              You can call TimerInit as often as you want, it
**              will only initialize the timer if it is not
**              already done.
**              So you could use TimerInit to make sure that
**              the timer was initialized.
** RETURNS:     BOOLEAN
*******************************************************************/
BOOLEAN TimerInit(VOID)
{
    static BOOLEAN init=FALSE;
    
    if (!init)
    {
        timerOffset=TIMER_CLOCK;    
        init=TRUE;
    }
    return init;
}                                



/*****************************************************************
** NAME:        TimerRegister
** SYNOPSIS:    TIMER_DESCR *TimerRegister(
**                  TIMER_CALLBACK callback,
**                  VOID  *parm;
**                  ULONG msecs,
**                  ULONG count,
**                  int type)
** PARAMETERS:  - TIMER_CALLBACK callback
**              The callback function should have the following
**              synopsis:
**              VOID FAR *callback(TIMER_DESCR *timer,
**                                      ULONG now, VOID *parm).
**              - VOID *parm
**              Optional parmater for callback function.
**              - ULONG msecs
**              Timer period in milliseconds.
**              - ULONG count
**              Number of periods before the timer is removed or
**              TIMER_FOREVER if the timer should run until it is
**              removed with TimerRemove.
**              - int type
**              Type which specifies what to do if we have missed
**              some periods. TIMER_TYPE_SKIP means that these periods
**              are skiped. TIMER_TYPE_RECOVER means that we try to
**              recover from this situation by some shorter
**              periods so the mean length of a period remains
**              always the same. 
** DESCRIPTION: registers a callback which is called periodically.
**              - A one shot timer can be registered with:
**              TimerRegister(callback, 1000L, 1, TIMER_SKIP)
**              - A normal timer is registered with:
**              TimerRegister(callback, 1000L,
**                                  TIMER_FOREVER, TIMER_RECOVER)
** RETURNS:     TIMER_DESCR *timer.
** SEE ALSO:    TimerRemove.
*******************************************************************/
TIMER_DESCR *TimerRegister(TIMER_CALLBACK callback, VOID *parm,
                    ULONG msecs, ULONG count, int type)
{
	TIMER_DESCR *timer;

	timer=(TIMER_DESCR *)DnpapMalloc(sizeof(TIMER_DESCR));
	if (timer==NULL)
    {
        DnpapMessage(DMC_FATAL, TIMER_ERROR,"Malloc failed");
        DnpapExit(1);
    }
	if (msecs==0)
		msecs=1;
    timer->callback=callback;
    timer->parm=parm;
	timer->msecs=msecs;
    timer->type=type;
    timer->count=count;
	timer->alarm=TIMER_CLOCK+msecs-31;
	Insert(timer);
	return timer;
}

/*****************************************************************
** NAME:        TimerRemove
** SYNOPSIS:    ULONG TimerRemove(TIMER_DESCR *timer)
** PARAMETERS:  - TIMER_DESCR *timer
**              Timer handle obtained by TimerRegister.
** DESCRIPTION: removes a timer registered with TimerRegister.
** RETURNS:     ULONG remaining milliseconds before timer expires.
** SEE ALSO:    TimerRegister.
*******************************************************************/
ULONG TimerRemove(TIMER_DESCR *timer)
{
    ULONG msecs;

    if (timer==0)
        return 0;
	msecs=timer->alarm-TIMER_CLOCK;
	Delete(timer);
	DnpapFree(timer);
	return msecs;
}



/*****************************************************************
** NAME:        TimerRemaining
** SYNOPSIS:    LONG TimerRemaining(VOID)
** DESCRIPTION: Returns time till one off the timers expires.
**              After this period of time you should call
**              TimerCheck to handle the expired timers.
** RETURNS:     ULONG remaining time.
**              <0 -> No timers.
**               0 -> No time remaining. Call TimerCheck as soon
**                    as possible.
**              >0 -> Remaining time of timer to expire first.
** SEE ALSO:    TimerCheck.
*******************************************************************/
LONG TimerRemaining(VOID)
{
    ULONG dt;

    if (timerList==NULL)
        return -1;
    dt = timerList->alarm-TIMER_CLOCK;
    if (dt > LONG_MAX)
        return 0;
    return (LONG)dt;
}







/*****************************************************************
** NAME:        TimerCheck
** SYNOPSIS:    BOOLEAN TimerCheck(VOID)
** DESCRIPTION: Main function to get the timers going. Call this
**              function regularly.
** RETURNS:     BOOLEAN success.
*******************************************************************/
BOOLEAN TimerCheck(VOID)
{
    TIMER_DESCR *timer,**p;

	while (timerList != NULL && TIMER_CLOCK-timerList->alarm<LONG_MAX)
    {
        timer=timerList;
        timer->callback(timer,TIMER_CLOCK-timerOffset,timer->parm);
        if (timer==timerList)    
        {
	        timerList=timerList->next;
            if (timer->count==1)
            {
	            DnpapFree(timer);
            }
            else
            {
                if (timer->count!=TIMER_FOREVER)
                    timer->count--;
                if (timer->type==TIMER_TYPE_SKIP)
                    timer->alarm=TIMER_CLOCK;
                timer->alarm+=timer->msecs;
	            for (p=&timerList;
		            *p!=NULL && (timer->alarm-(*p)->alarm)<LONG_MAX;
		            p=&(*p)->next);
	            timer->next=*p;
	            *p=timer;
            }
        }
    }
    return TRUE;
}
                              
/*****************************************************************
** NAME:        TimerNow
** SYNOPSIS:    ULONG TimerNow(VOID)
** DESCRIPTION: Returns the time in milliseconds sinds TimerInit.
** RETURNS:     ULONG time in milliseconds sinds TimerInit.
*******************************************************************/
ULONG TimerNow(VOID)
{
    return TIMER_CLOCK-timerOffset;
}



/*****************************************************************
** NAME:        TimerChange
** SYNOPSIS:    BOOLEAN TimerChange(TIMER_DESCR *timer,
**                      ULONG msecs)
** PARAMETERS:  - TIMER_DESCR *timer
**              Timer handle obtained by TimerRegister.
**              - ULONG msecs
**              New timer period. The change takes place after
**              the next expiration.
** DESCRIPTION: changes period of a timer.
** RETURNS:     BOOLEAN success.
** SEE ALSO:    TimerRegister, TimerRemove.
*******************************************************************/
BOOLEAN TimerChange(TIMER_DESCR *timer, ULONG msecs)
{
    if (timer==0)
        return FALSE;
	if (msecs==0)
		msecs=1;
	timer->msecs=msecs;
	return TRUE;
}



static VOID Insert(TIMER_DESCR *timer)
{
	TIMER_DESCR **p;

	for (p=&timerList;
      	  *p!=NULL && (timer->alarm-(*p)->alarm)<LONG_MAX;
		  p=&(*p)->next);
	timer->next=*p;
	*p=timer;
}


static VOID Delete(TIMER_DESCR *timer)
{
	TIMER_DESCR **p;
	
	for (p=&timerList;
		  *p!=NULL && *p!=timer;
		  p=&(*p)->next);
	if (*p!=NULL)
	{
		*p=(*p)->next;
	}
}



