/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          stat.c
** SYSTEM NAME:        beholder
** MODULE NAME:        stat\view
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/14
** DESCRIPTION:        etherStats group: viewer
*************************************************************************/

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <dnpap.h>
#include <maxmem.h>
#include <config.h>
#include <message.h>
#include <snmp.h>
#include <mgmt.h>
#include <vip.h>                                        

#define MAXCOLLECTORS   1000

#define TOTALVAR        17
#define INDEXPOS        11

static LONG             iface = 0;

static LONG             collIndex[MAXCOLLECTORS];

static LONG             currIndex = 0;
static LONG             statIndex = 0;
static BYTE             statOwner[SNMP_SIZE_BUFCHR];
static WORD             statOwnerLen;
static LONG             statSource[] = {1,3,6,1,2,1,2,2,1,1,1};
static VIPTIMER         *statTimer = NULL;
static VIPINFO          *statFrame, *mainFrame, *modeFrame, *menuFrame;
static VIPINFO          *graph1, *graph2, *graph3, *varFrame;
static VIPINFO          *configList;
static LWORD            statInterval, statClockOld, statVarOld[TOTALVAR];
static BOOLEAN          showGraph = FALSE, configMode = TRUE;

static MGMT             *mgmt = NULL;

static SNMP_OBJECT  varObj[] =
{
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,3,1}, 12, SNMP_COUNTER, 0},     /* dropevents */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,4,1}, 12, SNMP_COUNTER, 0},     /* octets */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,5,1}, 12, SNMP_COUNTER, 0},     /* pkts */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,6,1}, 12, SNMP_COUNTER, 0},     /* broadcast */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,7,1}, 12, SNMP_COUNTER, 0},     /* multicast */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,8,1}, 12, SNMP_COUNTER, 0},     /* CRCAlign */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,9,1}, 12, SNMP_COUNTER, 0},     /* undersize */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,10,1}, 12, SNMP_COUNTER, 0},    /* oversize */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,11,1}, 12, SNMP_COUNTER, 0},    /* fragments */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,12,1}, 12, SNMP_COUNTER, 0},    /* jabbers */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,13,1}, 12, SNMP_COUNTER, 0},    /* collisions */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,14,1}, 12, SNMP_COUNTER, 0},    /* 64 */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,15,1}, 12, SNMP_COUNTER, 0},    /* 65-127 */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,16,1}, 12, SNMP_COUNTER, 0},    /* 128-255 */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,17,1}, 12, SNMP_COUNTER, 0},    /* 256-511 */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,18,1}, 12, SNMP_COUNTER, 0},    /* 512-1023 */
    {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,19,1}, 12, SNMP_COUNTER, 0}     /* 1024-1518 */
};


static VOID StatConfig (VOID);                         /* reads config file */
static VOID StatInit (VOID);              /* initializes collector & viewer */
static VOID StatMenu(VIPINFO *wip, VOID *ptr, int elem);            /* menu */
static VOID TimerCall (VIPTIMER *timer, VOID *parm); /* requests data collector */
static VOID ConfigCall (VOID);        /* frame with all collector instances */
static VOID ViewCall (VOID);                             /* frame with data */
static VOID QuitCall (VOID);                                /* quits viewer */
static VOID ResetCall (VOID);                              /* resets viewer */
static VOID ModeCall (VOID);                      /* changes text <-> graph */
static VOID DrawConfig (SNMP_OBJECT *obj, WORD ind);
static VOID DrawGraph (VOID);
static VOID DrawText (VOID);
static VOID UpdateGraph (ULONG now);
static VOID UpdateText (ULONG now);
static VOID LineSelect(VIPINFO *wip, VOID *p, INT line);

static VOID SignalHandler(INT signum);

static VOID Usage (VOID);


VOID main (int argc, char *argv[])
{
    int argi;
    INT i;

    if (ConfigInit(argc, argv) == FALSE || ConfigLoad() == FALSE)
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "stat: ConfigInit failed");

    MemoryInit(8000000L, 32000L);

    if (DnpapSignal(SIGINT, SignalHandler) == SIG_ERR)
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "stat: signal handler initialization failed");
    
    MessageInit("stdio.out", MSGFILE, MSGVIP, NULL);
    MessageConfig(STATISTICS_ERROR, "Stat");

	for (argi = 1; argi < argc; argi++)
	{				 
		if (argv[argi][0] == '-')
        {
			for (i = 1; i < (INT)strlen(argv[argi]); i++)
				switch (argv[argi][i])
				{
				case 'h':
					Usage();
					DnpapExit(0);
				default:
					DnpapMessage(DMC_FATAL, STATISTICS_ERROR, "stat: unrecognized option");
					Usage();
					DnpapExit(1);
				}
        }
		else
		{
			if ((iface = atoi(argv[argi])) < 1)
            {
				DnpapMessage(DMC_FATAL, STATISTICS_ERROR, "stat: invalid iface number");
				DnpapExit(1);
            }
		}
	}

    if (VipInit (argc, argv) == FALSE)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "stat: VipInit failed");
        DnpapExit(1);
    }
    if (MgmtInit() == FALSE)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "stat: MgmtInit failed");
        DnpapExit(1);
    }
    StatConfig();
    if ((mgmt = MgmtOpen (0, MGMT_TYPE_DEFAULT)) == NULL)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "stat: MgmtOpen failed");
        DnpapExit(1);
    }
    StatInit();
    VipCheck(TRUE);
    MgmtClose (mgmt);
    DnpapExit(0);
}


VOID StatConfig (VOID)
{
    BYTE *p;
    LONG l;

    if (ConfigGetLong("stat.interval", &l))
        statInterval = l;
    else
        statInterval = 2000L;
    if (iface > 0)
        statSource[sizeof(statSource)/sizeof(statSource[0])-1]=iface;
    else
    if (ConfigGetLong("stat.iface",&l))
        statSource[sizeof(statSource)/sizeof(statSource[0])-1]=l;
    else
        statSource[sizeof(statSource)/sizeof(statSource[0])-1]=1;
    if (!ConfigGetString("stat.owner", &p) || strlen(p) > SNMP_SIZE_BUFCHR)
        p="monitorEtherStats";
    strcpy(statOwner, p);
    statOwnerLen = strlen(p);
    
    return;
}


VOID StatInit (VOID)
{
    WORD status, index;

    SNMP_OBJECT obj[] =
    {
        {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12},    /* status */
        {SNMP_PDU_SET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12, SNMP_INTEGER, SNMP_CREATEREQUEST},
        {SNMP_PDU_SET, {1,3,6,1,2,1,16,1,1,1,20,1}, 12, SNMP_DISPLAYSTR},
        {SNMP_PDU_SET, {1,3,6,1,2,1,16,1,1,1,2,1}, 12, SNMP_OBJECTID}
    };

    strcpy (obj[2].Syntax.BufChr, statOwner);
    obj[2].SyntaxLen = statOwnerLen;
    memcpy(obj[3].Syntax.BufInt, statSource, sizeof(statSource));
    obj[3].SyntaxLen = sizeof(statSource)/sizeof(statSource[0]);

    statFrame = VipOpenFrame (NULL, 5, 150, 545, 670);
    mainFrame = VipOpenFrame (statFrame, 0, 0, 1000, 1000);
    VipSetSystemMenu(statFrame, FALSE);

    status = SNMP_NOERROR;
    statIndex = 0;

    obj[0].Id[INDEXPOS] = 0;

    while (status != SNMP_NOSUCHNAME)
    {
        if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_NEXT, &obj[0], 1, &status, &index, 0, 0) == FALSE)
        {
            DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"stat: MgmtRequest failed");
            QuitCall();
            return;
        }
    }

    statIndex = obj[0].Id[INDEXPOS]+1;

    obj[1].Id[INDEXPOS] = statIndex;
    obj[2].Id[INDEXPOS] = statIndex;
    obj[3].Id[INDEXPOS] = statIndex;

    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_SET, &obj[1], 3, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"stat: can't initialize");
        QuitCall();
        return;
    }

    currIndex = statIndex;

    VipShow (statFrame);
    ConfigCall();

    return;
}


VOID StatMenu(VIPINFO *wip, VOID *ptr, int elem)
{
    switch(elem)
    {
    case 0: QuitCall(); break;
    case 1:
        configMode = 1 - configMode;
        if (configMode == TRUE)
            ConfigCall();
        else
            ViewCall();
        break;
    case 2: ModeCall(); break;
    case 3: ResetCall();
    }
}


VOID ConfigCall (VOID)
{
    static int configTabs[]={150,600};

    LONG                indexId[]= {1,3,6,1,2,1,16,1,1,1,1};
    WORD                status, index;
    WORD                ind = 0, IdLen = 11;
    VIPINFO             *configFrame, *configScrol, *configText, *configVar;

    SNMP_OBJECT          obj[] =
    {
        {SNMP_PDU_NEXT, {1,3,6,1,2,1,16,1,1,1,1}, 11},     /* index */
        {SNMP_PDU_NEXT, {1,3,6,1,2,1,16,1,1,1,20}, 11},    /* owner */
        {SNMP_PDU_NEXT, {1,3,6,1,2,1,16,1,1,1,21}, 11}    /* status */
    };

    if (statTimer != NULL)
        VipDestroyTimer (statTimer);
    VipDestroy(mainFrame);

    /* Open all windows */
    mainFrame = VipOpenFrame (statFrame, 0, 0, 1000, 1000);
    menuFrame = VipOpenMenu (mainFrame, 0, 0, 1000, 0);
    configFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 1000);

    configScrol = VipOpenVScroll (configFrame, 0, 0, 0, 1000);
    configVar   = VipOpenFrame (configFrame, 0,0, 1000,1000);

    configText = VipOpenText (configVar, 0, 0, 1000, 0);
    configList = VipOpenList (configVar, 0, 0, 1000, 1000);
    VipSetListCallBack(configList, LineSelect, NULL);


    /* Configure all windows */
    VipSetTitle (statFrame, "stat: Configuration Window");

    VipSetMenuType (menuFrame, VIP_HMENU|VIP_FIXMENU);
    VipSetMenuCallBack (menuFrame, StatMenu);
    VipSetMenuEntry(menuFrame, "~Quit!", 0, 0, NULL);
    VipSetMenuEntry(menuFrame, "~View!", 1, 0, NULL);
    VipSetMenuEntry(menuFrame, NULL,    2, VIP_MENUBAR, NULL);
    VipSetMenuEntry(menuFrame, NULL,    3, VIP_MENUBAR, NULL);

    VipSetTextTabs (configText, 2, configTabs, VIP_TABLINES);
    VipSetTextLine (configText, "Index\tOwner\tStatus", 0, VIP_LEFT);

    VipSetTextTabs (configList, 2, configTabs, VIP_TABLINES);
    VipSetTextScrollbars(configList, configScrol, NULL);

    /* Position all windows */
    VipSetFrameSplit (mainFrame, menuFrame, VIP_HSPLIT|VIP_FIXTOP);
    VipSetFrameSplit (configFrame, configScrol, VIP_VSPLIT);
    VipSetFrameSplit (configVar, configText, VIP_HSPLIT|VIP_FIXTOP);

    VipShow (mainFrame);

    while (MgmtRequest(mgmt, 0, 0, SNMP_PDU_NEXT, obj, 3, &status, &index, 0, 0) == TRUE &&
        status == SNMP_NOERROR &&
        !memcmp(obj[0].Id, indexId, IdLen * sizeof(LONG)))
        DrawConfig (obj, ind++);

    return;
}


VOID ViewCall (VOID)
{
    WORD    status, index;

    SNMP_OBJECT obj[] =
    {
        {SNMP_PDU_GET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12},
        {SNMP_PDU_SET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12, SNMP_INTEGER, SNMP_VALID}
    };

    VipDestroy(mainFrame);

    mainFrame = VipOpenFrame (statFrame, 0, 0, 1000, 1000);
    menuFrame = VipOpenMenu (mainFrame, 0, 0, 1000, 0);
    modeFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 1000);
    VipSetMenuType (menuFrame, VIP_HMENU|VIP_FIXMENU);

    VipSetMenuCallBack(menuFrame, StatMenu);
    VipSetMenuEntry(menuFrame, "~Quit!", 0, 0, NULL);
    VipSetMenuEntry(menuFrame, "~Config!", 1, 0, NULL);
    VipSetMenuEntry(menuFrame, "~Reset!",  3, 0, NULL);

    VipShow (mainFrame);

    if (currIndex > 0)
    {
        obj[0].Id[INDEXPOS] = currIndex;
        obj[1].Id[INDEXPOS] = currIndex;
    }

    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &obj[0], 1, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"stat: can't get status");
        QuitCall();
        return;
    }
    if (status == SNMP_NOERROR && obj[0].Syntax.LngInt == SNMP_UNDERCREATION)
    {
        if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_SET, &obj[1], 1, &status, &index, 0, 0) == FALSE ||
            status != SNMP_NOERROR)
        {
            DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"stat: can't set status to valid");
            QuitCall();
            return;
        }
    }
    if ((statTimer = VipCreateTimer (statInterval, TimerCall, NULL)) == NULL)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"stat: can't register timer");
        QuitCall();
        return;
    }

    ResetCall ();
    return;
}


VOID QuitCall (VOID)
{
    WORD    status, index;

    SNMP_OBJECT obj[] =
    {
        {SNMP_PDU_SET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12, SNMP_INTEGER, SNMP_INVALID}
    };


    if (statIndex > 0)
    {
        obj[0].Id[INDEXPOS] = statIndex;

        if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_SET, &obj[0], 1, &status, &index, 0, 0) == FALSE ||
            status != SNMP_NOERROR)
            DnpapMessage(DMC_WARNING, INFO_ERROR,"stat: can't set status to invalid");
    }

    VipClose (statFrame);
}


VOID ResetCall (VOID)
{
    WORD status, index;
    WORD ind;

    VipDestroy (modeFrame);
    modeFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 1000);
    VipSetFrameSplit (mainFrame, menuFrame, VIP_HSPLIT|VIP_FIXTOP);
    if (showGraph == TRUE)
        DrawGraph();
    else
        DrawText();
    for (ind = 0; ind < TOTALVAR; ind++)
        varObj[ind].Id[INDEXPOS] = currIndex;
    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &varObj[0], 8, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR ||
        MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &varObj[8], 9, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"stat: can't get data");
        return;
    }
    for (ind = 0; ind < TOTALVAR; ind++)
        statVarOld[ind] = varObj[ind].Syntax.LngUns;
    statClockOld = clock()/1000L;
    VipUpdate (menuFrame, TRUE);
    VipShow (modeFrame);
    return;
}


VOID ModeCall (VOID)
{
    showGraph = 1 - showGraph;
    ResetCall ();
    return;
}


VOID TimerCall (VIPTIMER *timer, VOID *parm)
{
    WORD status, index;
    WORD ind;
    LWORD now;

    for (ind = 0; ind < TOTALVAR; ind++)
        varObj[ind].Id[INDEXPOS] = currIndex;
    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &varObj[0], 8, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR ||
        MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &varObj[8], 9, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"stat: can't get data");
        return;
    }
    now = clock()/1000L;
    if (showGraph == TRUE)
        UpdateGraph(now);
    else
        UpdateText(now);
    return;
}


VOID UpdateText (ULONG now)
{
    WORD            ind;
    LWORD           current;
    BYTE            str[SNMP_SIZE_BUFCHR];

    for (ind = 0; ind < TOTALVAR; ind++)
    {
        current = varObj[ind].Syntax.LngUns - statVarOld[ind];
        sprintf(str, "..............................................................%lu", current);
        VipSetTextLine (varFrame, str, ind, VIP_RIGHT);
    }
    VipUpdate(varFrame, TRUE);
    return;
}


VOID UpdateGraph (ULONG now)
{
    WORD ind;
    LWORD current, sec;

    sec = 1 + (now - statClockOld) / 1000L;
    statClockOld = now;
    for (ind = 0; ind < TOTALVAR; ind++)
    {
        current = varObj[ind].Syntax.LngUns - statVarOld[ind];
        statVarOld[ind] = varObj[ind].Syntax.LngUns;
        if (ind > 0)
        {
            if (ind < 5)
                VipSetGraphLast(graph2, ind - 1, (long) (current/sec));
            else if (ind < 11)
                VipSetGraphLast(graph1, ind - 4, (long) (current/sec));
            else
                VipSetGraphLast(graph3, ind - 11, (long) (current/sec));
        }
        else
            VipSetGraphLast(graph1, 0, (long) (current/sec));
    }
    VipUpdate(modeFrame, FALSE);
    return;
}


VOID DrawGraph (VOID)
{
    VipSetTitle (statFrame, "stat: Graph Window");
    VipSetMenuEntry(menuFrame, "~Text!",   2, 0, NULL);
    graph1 =      VipOpenGraph (modeFrame, 0, 600, 1000, 400);
        /* Drop, CRC, Under, Over, Fragment, Jabber, Collision */
    graph2 =       VipOpenGraph (modeFrame, 0, 300, 1000, 300);
        /* Pkt, Octet, Brc, Multi */
    graph3 =        VipOpenGraph (modeFrame, 0, 0, 1000, 300);
        /* 64, 128, 256, 512, 1024 */

    VipSetBackground (graph1, VIP_WHITE);
    VipSetBackground (graph2, VIP_WHITE);
    VipSetBackground (graph3, VIP_WHITE);
    VipSetGraphDataSet (graph1, 6, 50, 0, 100, "Collision", VIP_PINK);
    VipSetGraphDataSet (graph1, 5, 50, 0, 100, "Jabber", VIP_BLUE);
    VipSetGraphDataSet (graph1, 4, 50, 0, 100, "Fragment", VIP_CYAN);
    VipSetGraphDataSet (graph1, 3, 50, 0, 100, "OverSize", VIP_GREEN);
    VipSetGraphDataSet (graph1, 2, 50, 0, 100, "UnderSize", VIP_BROWN);
    VipSetGraphDataSet (graph1, 1, 50, 0, 100, "CRCAlign", VIP_DARKRED);
    VipSetGraphDataSet (graph1, 0, 50, 0, 500, "DropEvent", VIP_RED);
    VipSetGraphDataSet (graph2, 3, 50, 0, 500, "Multicast", VIP_PINK);
    VipSetGraphDataSet (graph2, 2, 50, 0, 500, "Broadcast", VIP_RED);
    VipSetGraphDataSet (graph2, 1, 50, 0, 500, "Pkt", VIP_DARKRED);
    VipSetGraphDataSet (graph2, 0, 50, 0, 250000L, "Octet", VIP_BLUE);
    VipSetGraphDataSet (graph3, 5, 50, 0, 200, "1024-1518Pkt", VIP_BROWN);
    VipSetGraphDataSet (graph3, 4, 50, 0, 200, "512-1023Pkt", VIP_CYAN);
    VipSetGraphDataSet (graph3, 3, 50, 0, 200, "256-511Pkt", VIP_PINK);
    VipSetGraphDataSet (graph3, 2, 50, 0, 200, "128-255Pkt", VIP_DARKRED);
    VipSetGraphDataSet (graph3, 1, 50, 0, 200, "65-127Pkt", VIP_RED);
    VipSetGraphDataSet (graph3, 0, 50, 0, 200, "64Pkt", VIP_BLUE);
    return;
}


VOID DrawText (VOID)
{
    WORD        ind;
    VIPINFO     *textFrame;

    VipSetTitle (statFrame, "stat: Text Window");                                                 
    VipSetMenuEntry(menuFrame, "~Graph!",  2, 0, NULL);
    textFrame = VipOpenText (modeFrame, 0, 0, 1000, 1000);
    varFrame = VipOpenText (modeFrame, 700, 0, 300, 1000);
    VipSetTextLine (textFrame, "DropEvents...............................................................................................................................", 0, VIP_LEFT);
    VipSetTextLine (textFrame, "Octets...................................................................................................................................", 1, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts.....................................................................................................................................", 2, VIP_LEFT);
    VipSetTextLine (textFrame, "BroadcastPkts............................................................................................................................", 3, VIP_LEFT);
    VipSetTextLine (textFrame, "MulticastPkts............................................................................................................................", 4, VIP_LEFT);
    VipSetTextLine (textFrame, "CRCAlignErrors...........................................................................................................................", 5, VIP_LEFT);
    VipSetTextLine (textFrame, "UndersizePkts............................................................................................................................", 6, VIP_LEFT);
    VipSetTextLine (textFrame, "OversizePkts.............................................................................................................................", 7, VIP_LEFT);
    VipSetTextLine (textFrame, "Fragments................................................................................................................................", 8, VIP_LEFT);
    VipSetTextLine (textFrame, "Jabbers..................................................................................................................................", 9, VIP_LEFT);
    VipSetTextLine (textFrame, "Collisions...............................................................................................................................", 10, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts64Octets.............................................................................................................................", 11, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts65to127Octets........................................................................................................................", 12, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts128to255Octets.......................................................................................................................", 13, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts256to511Octets.......................................................................................................................", 14, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts512to1023Octets......................................................................................................................", 15, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts1024to1518Octets.....................................................................................................................", 16, VIP_LEFT);
    for (ind = 0; ind < TOTALVAR; ind++)
    {
        VipSetTextLine (varFrame, "......................................................................................................................................0", ind, VIP_RIGHT);
    }
    return;
}


VOID DrawConfig (SNMP_OBJECT *obj, WORD ind)
{
    BYTE str[SNMP_SIZE_BUFCHR];

    switch (obj[2].Syntax.LngInt)
    {
        case SNMP_INVALID:
            sprintf(str, "%ld\t%s\tINVALID", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
            break;
        case SNMP_VALID:
            sprintf(str, "%ld\t%s\tVALID", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
            break;
        case SNMP_UNDERCREATION:
            sprintf(str, "%ld\t%s\tUNDERCREATION", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
            break;
        default:
            sprintf(str, "%ld\t%s\tUNKNOWN", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
    }

    VipSetTextLine (configList, str, ind, VIP_LEFT);

    if (ind < MAXCOLLECTORS)
        collIndex[ind] = obj[0].Id[INDEXPOS];

    return;
}


VOID LineSelect(VIPINFO *wip, VOID *p, INT line)
{
    if (line < MAXCOLLECTORS)
        currIndex = collIndex[line];
    else
        DnpapMessage(DMC_WARNING, STATISTICS_ERROR, "stat: collector index array too small");
}


VOID SignalHandler(INT signum)
{
    QuitCall();
    DnpapExit(0);
}


VOID Usage (VOID)
{
	DnpapMessage(DMC_MESSAGE, STATISTICS_ERROR, "usage: stat [-h] iface");
}
