/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vipprint.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/6/15
**
** DESCRIPTION: 
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#define LIBRARY
#define INCL_WINSHELLDATA
#define INCL_WINERRORS
#include "vipinc.h"

/*--------------------- PM Printer Management ---------------------*/
typedef struct _PRINTER
    {
        DRIVDATA *Data;
        HDC hdc;
        HPS hps;
        char *Driver;
        char *Model;
        char *Queue;
        char *LogDev;
        char *Port;
        char Printer[32];
        char Details[256];
        LONG Type;
    } PRINTER;


PRIVAT BOOLEAN VipInitPrinter(void);
PRIVAT void VipExitPrinter(void);
PRIVAT HPS  VipOpenPrinter(CONST char *DocName);
PRIVAT void VipClosePrinter(void);

PRIVAT void PrintChildren(VIPINFO *wip, MATRIXLF *matrix);
PRIVAT BOOLEAN QueryPrinterSetup(void);
PRIVAT BOOLEAN QueryPrinter(void);
PRIVAT LONG PrinterError(LONG ec, char *msg);

PRIVAT PRINTER *printerInfo = NULL;

EXPORT LONG vipPrintScale;
EXPORT BOOLEAN vipPrinting=FALSE;

IMPORT HAB vipHab;


/**************************************************************
** NAME:        VipInitPrinter                            
** SYNOPSIS:    BOOLEAN VipInitPrinter()
** DESCRIPTION: Inits a printer environment.
** RETURNS:     TRUE, success
**              FALSE, failure
**************************************************************/
PRIVAT BOOLEAN VipInitPrinter()
{
    printerInfo=DnpapMalloc(sizeof(PRINTER));
    if (!printerInfo)
        return FALSE;
    printerInfo->Data=NULL;
    DnpapAtExit(VipExitPrinter);

    if (!QueryPrinterSetup())
        return (BOOLEAN)PrinterError(VIPERR_PRTSET,
                "Failed to get printer setup");
    return TRUE;
}

PRIVAT void VipExitPrinter()
{
    if (printerInfo)
    {
        if (printerInfo->Data)
            DnpapFree(printerInfo->Data);
        DnpapFree(printerInfo);
        printerInfo=NULL;
    }
}

/**************************************************************
** NAME:        VipPrintWindow                            [API]
** SYNOPSIS:    BOOLEAN VipPrintWindow(
**                  VIPINFO *vip, BOOLEAN all);
** DESCRIPTION: Prints the contents of a window.
**              if <all> is TRUE, all its children are also
**              printed. Currently the printing (spooling)
**              is done in the main thread and will take some
**              while (in the order of seconds) before this
**              function returns. 
** RETURNS:     TRUE, success
**              FALSE, failure
**************************************************************/
BOOLEAN
VipPrintWindow(VIPINFO *wip, BOOLEAN all)
{
    SIZEL sizl;             /* page size */
    MATRIXLF matrix;
    LONG fy;

    if (!wip) return FALSE;

    if (!WinIsWindowVisible(wip->win)) return FALSE;

    if (!VipOpenPrinter(wip->title?wip->title:"VIP"))
        return FALSE;

    VipSetBusy(TRUE);

    DevQueryCaps(printerInfo->hdc, CAPS_WIDTH, 2L, (PLONG) &sizl);
    GpiConvert(printerInfo->hps, CVTC_DEVICE, CVTC_WORLD, 1L, (PPOINTL) &sizl);

    /* First get the current transformation matrix. */
    GpiQueryDefaultViewMatrix(printerInfo->hps, 9, &matrix);

    /* Change the x- and y-translation elements. */
    vipPrintScale=((65536L*5L/6L)*sizl.cx)/wip->cx;
    fy=((65536L*5L/6L)*sizl.cy)/wip->cy;

    if (fy<vipPrintScale) vipPrintScale=fy;   /* Choose smallest size factor */
    matrix.fxM11 = vipPrintScale;
    matrix.fxM22 = vipPrintScale;

    matrix.lM31 += (sizl.cx - FIXEDINT(vipPrintScale*(LONG)wip->cx+32768L))/2;
    matrix.lM32 += (sizl.cy - FIXEDINT(vipPrintScale*(LONG)wip->cy+32768L))/2;

    /* Call GpiSetDefaultViewMatrix to translate image to page. */
    GpiSetDefaultViewMatrix(printerInfo->hps, 9, &matrix, TRANSFORM_REPLACE);

    vipPrinting=TRUE;
    wip->update(wip,printerInfo->hps,1);
    if (all)
        PrintChildren(wip, &matrix);
    vipPrinting=FALSE;
    VipClosePrinter();

    VipSetBusy(FALSE);
    return TRUE;
}


PRIVAT void
PrintChildren(VIPINFO *wip, MATRIXLF *matrix)
{
    MATRIXLF matrix2;
    VIPINFO *child;
    LONG cx, cy;

    cx=((LONG)wip->cx*vipPrintScale+500)/1000L;
    cy=((LONG)wip->cy*vipPrintScale+500)/1000L;
  	for (child=wip->child;child; child=child->sibling)
        if (WinIsWindowVisible(child->win))
        {
            matrix2=*matrix;
            matrix2.lM31 += FIXEDINT((LONG)child->x*cx+32768L);
            matrix2.lM32 += FIXEDINT((LONG)child->y*cy+32768L);
            GpiSetDefaultViewMatrix(printerInfo->hps, 9, &matrix2, TRANSFORM_REPLACE);
            child->update(child, printerInfo->hps, 1);
            PrintChildren(child, &matrix2);
        }
}

/**************************************************************
** NAME:        VipOpenPrinter                            
** SYNOPSIS:    HPS VipOpenPrinter(CONST char *DocName)
** DESCRIPTION: Opens a printer environment. And starts a new
**              document frame.
** RETURNS:     NULL, Error occured (is reported to the user)
**              handle to presentation space, otherwise
** SEE ALSO:    VipClosePrint
**************************************************************/
PRIVAT HPS
VipOpenPrinter(CONST char *DocName)
{
    DEVOPENSTRUC dop;
    SIZEL        sizl;
    LONG         lrc;

    if (!printerInfo)
        if (!VipInitPrinter())
            return NULL;

    /* create a Device Context for the printer */
    memset(&dop,0,sizeof(DEVOPENSTRUC));
    dop.pszLogAddress = printerInfo->LogDev;
    dop.pszDriverName = printerInfo->Driver;
    dop.pdriv         = printerInfo->Data;            
    dop.pszDataType   = "PM_Q_STD";
    /*if (printerInfo->Type==OD_QUEUED) */
        dop.pszComment = "WindowDump by VIP";

    printerInfo->hdc = DevOpenDC( vipHab,
                         printerInfo->Type,       /* print in a queued fashion */
                         "*",                               /* no device info */
                         5L,                            /* number of elements */
                         (PDEVOPENDATA) &dop,                  /* device info */
                         NULL);                  /* DC compatible with screen */
                                        
    if (printerInfo->hdc == DEV_ERROR)
        return (HPS)PrinterError(VIPERR_PRTDEV,"Failed to open device context");
    /* Tell the printer that a new print job is started */
    lrc = DevEscape( printerInfo->hdc,
                     DEVESC_STARTDOC,
                     strlen(DocName), (char*)DocName, 
                     NULL,
                     NULL);
    if (lrc==DEVESC_ERROR)
    {
        DevCloseDC(printerInfo->hdc);        
        return (HPS)PrinterError(VIPERR_PRTSTART,"Failed to start the printer");
    }

    sizl.cx = 0L;
    sizl.cy = 0L;
    printerInfo->hps = GpiCreatePS( vipHab,
                           printerInfo->hdc,
                           &sizl,
                           PU_LOENGLISH | GPIF_LONG |
                           GPIT_NORMAL | GPIA_ASSOC);

    if (printerInfo->hps == NULL)
    {
        DevCloseDC(printerInfo->hdc);        
        return (HPS)PrinterError(VIPERR_PRTHPS,"Failed to create presentation space for the printer");
    }
    return printerInfo->hps;
}


/**************************************************************
** NAME:        VipClosePrinter                           
** SYNOPSIS:    void VipClosePrinter();
** DESCRIPTION: Closes the presentation space to the printer.
** RETURNS:     void
** SEE ALSO:    VipOpenPrint
**************************************************************/
void
VipClosePrinter()
{
    /* Tell the printer that this print job is finished */
    DevEscape(printerInfo->hdc, DEVESC_ENDDOC, 0L, NULL, NULL, NULL);
    GpiAssociate(printerInfo->hps,NULL); /* Release printer context */
    DevCloseDC(printerInfo->hdc);        
    GpiDestroyPS(printerInfo->hps);
}


/**************************************************************
** NAME:        VipPrinterQuery                             [API]
** SYNOPSIS:    BOOLEAN VipPrinterQuery(VIPINFO *win)
** DESCRIPTION: Displays the printer setup.
** RETURNS:     TRUE, okay.
**              FALSE, no memory for window.
**************************************************************/
BOOLEAN
VipPrinterQuery(VIPINFO *win)
{
    char buf[256];
    if (!printerInfo)
        if (!VipInitPrinter())
            return FALSE;

    sprintf(buf,"Printer: %s\nDriver: %s\nModel: %s\nPort: %s\nAccess: %s",
                printerInfo->Printer, printerInfo->Driver,
                printerInfo->Model, printerInfo->Port, 
                printerInfo->Type==OD_DIRECT?"direct":"queue");
    VipMessage(win, "Printer Info", buf);
    return TRUE;
}


/**************************************************************
** NAME:        VipPrinterSetup                           [API]
** SYNOPSIS:    BOOLEAN VipPrinterSetup()
** DESCRIPTION: Displays printer setup dialog. Doesn't store
**              the changes in the OS2.INI file.
** RETURNS:     TRUE, success
**              FALSE, failure
**************************************************************/
BOOLEAN
VipPrinterSetup()
{
    if (!printerInfo)
        if (!VipInitPrinter())
            return FALSE;

    if (printerInfo->Data)
    {
        if (DevPostDeviceModes( vipHab,
                                (PDRIVDATA)printerInfo->Data,
                                printerInfo->Driver,    /* points to 'PSCRIPT' */
                                printerInfo->Model,     /* Points to 'Apple...' */
                                printerInfo->Printer,   /* Use config as stored in PDRIVDATA */
                                DPDM_POSTJOBPROP)==DEV_OK)
            return TRUE;
    }
    else
        DnpapMessage(DMC_WARNING,VIPERR_PRTCONFIG,"Printer can not be configurated");
    WinGetLastError(vipHab);   /* Clear any errors... */
    return FALSE;
}


PRIVAT BOOLEAN
QueryPrinterSetup()
{                                      
    char *ptr;
    LONG len;

    if (!QueryPrinter())
        return (BOOLEAN)PrinterError(VIP_ERROR,
           "Can't find printer configuration in OS2.INI\n"
            "Please install a printer driver");

    /* Details:LPT1;PSCRIPT.Apple LaserWriter II NTX;PSCRIPT;; */
    /* parse the string for the correct info */
    printerInfo->Port= printerInfo->Details;
    printerInfo->Driver = strchr(printerInfo->Details,';')+1;
    printerInfo->Driver[-1]='\0';
    printerInfo->Queue = strchr(printerInfo->Driver,';')+1;
    printerInfo->Queue[-1] = '\0';
    *(strchr(printerInfo->Queue,';')) = '\0';

    /* Discard extra junk */
    ptr = strchr(printerInfo->Driver,'.');
    if (ptr)
    {
        *ptr='\0';
        printerInfo->Model = ptr+1;
    }
    else
        printerInfo->Model = NULL;
    ptr = strchr(printerInfo->Queue,',');
    if (ptr) *ptr = '\0';
    ptr = strchr(printerInfo->Queue,'.');
    if (ptr) *ptr = '\0';

    if (printerInfo->Queue[0]!='\0')                     /* Queue is defined ? */
    {
        printerInfo->Type = OD_QUEUED;
        printerInfo->LogDev = printerInfo->Queue;
    }
    else
    {
        printerInfo->Type = OD_DIRECT;
        printerInfo->LogDev = printerInfo->Port;
    }

    /* Ask length of Printer Data structure */
    len=DevPostDeviceModes( vipHab,
                            NULL,                 /* NULL to request size */
                            printerInfo->Driver,  /* points to 'PSCRIPT' */
                            printerInfo->Model,   /* points to 'Apple ...' */
                            printerInfo->Printer, /* Points to 'Matrixje' */
                            0L);                  /* request DataDriv length */
    if (printerInfo->Data)
        DnpapFree(printerInfo->Data);
    printerInfo->Data = NULL;
    if (len==DPDM_NONE)
        return TRUE;    /* No settable options */
    if (len==DPDM_ERROR)
        return (BOOLEAN)PrinterError(VIP_ERROR,
           "Request for the size of the 'Printer Data' structure failed");
    printerInfo->Data=DnpapMalloc((size_t)len);
    if (!printerInfo->Data)
        return (BOOLEAN)PrinterError(VIP_ERROR,
           "Request for memory for the 'Printer Data' structure failed");
    if (DevPostDeviceModes( vipHab,
                            (PDRIVDATA)printerInfo->Data,
                            printerInfo->Driver,       
                            printerInfo->Model,    
                            printerInfo->Printer,   
                            DPDM_QUERYJOBPROP)==DEV_OK)
        return TRUE;
    DnpapFree(printerInfo->Data);
    printerInfo->Data=NULL;
    return (BOOLEAN)PrinterError(VIP_ERROR,
        "Request for the contents of the 'Printer Data' structure failed");
}


PRIVAT BOOLEAN
QueryPrinter()
{
#ifdef OS2EMX
    return FALSE;
#else
    ULONG cb;
    cb = WinQueryProfileString(
            vipHab,
            "PM_SPOOLER",                          /* section name of OS2.INI */
            "PRINTER",                                             /* keyname */
            "",                                              /* default value */
            printerInfo->Printer,                      /* buffer to store info */
            sizeof(printerInfo->Printer));                   /* size of buffer */
    if (cb==0 || printerInfo->Printer[0]==0)
        return FALSE;
    printerInfo->Printer[cb-2] = 0;                     /* remove trailing ';' */
    return (WinQueryProfileString(
            vipHab,
            "PM_SPOOLER_PRINTER",
            printerInfo->Printer,                        /* Keyname in OS2.ini */
            "",                                                    /* default */
            printerInfo->Details,                      /* buffer to store info */
            sizeof(printerInfo->Details))!=0);               /* size of buffer */
    /* Details:LPT1;PSCRIPT.Apple LaserWriter II NTX;PSCRIPT;; */
#endif
}


PRIVAT LONG
PrinterError(LONG ec, char *msg)
{
#ifdef OS2EMX
    DnpapMessage(DMC_ERROR, ec, "%s.", msg);
#else
    ERRORID id;
    id=WinGetLastError(vipHab);
    if (id)
        DnpapMessage(DMC_ERROR, ec, "%s.\n\tErrorcode: %d, severity: %d", msg,
            ERRORIDERROR(id), ERRORIDSEV(id) );
    else
        DnpapMessage(DMC_ERROR, ec, "%s.", msg);
#endif
    return 0L;
}

