/**************************************************************************/
/* rxCalcNet.cmd - The Subnet Calculator allows to calculate a address,   */
/*                 a size and a subnet mask.                              */
/*                                                                        */
/* Syntax:                                                                */
/*    rxCalcNet <subnet/mask_size>                                        */
/*    rxCalcNet <subnet netmask>                                          */
/*    rxCalcNet <address1-address2>                                       */
/*                                                                        */
/* Parameters:                                                            */
/*    subnet/mask_size  - the subnet address and the number of the bits   */
/*                        in a subnet mask                                */
/*    subnet netmask    - the subnet address and the subnet mask          */
/*    address1-address2 - starting and finishing addresses of a subnet    */
/*                                                                        */
/* Example:                                                               */
/*    rxCalcNet 192.168.2.0/24                                            */
/*    rxCalcNet 192.168.2.0 255.255.255.0                                 */
/*    rxCalcNet 192.168.2.0-192.168.2.255                                 */
/*========================================================================*/
/* (c) VicTor Smirnoff, 2010                            iamvic@rambler.ru */
/*------------------------------------------------------------------------*/
trace off
numeric digits 12

mr.Version    = 'rxCalcNet v0.02'
mr.CopyLeft   = '(C)opyLeft VicTor Smirnoff, 2010'
mr.Email      = 'iamvic@rambler.ru'

rc = xGetAnsiValue()

say mr.c.Reset
say mr.b.W||mr.Version||'   '||mr.CopyLeft||'     '||mr.Email
say mr.c.Reset

parse arg pParam

xMode = xTestParam( pParam )
if xMode = 0 then do
   say mr.b.R||'Invalid or missing parameters.'||mr.c.Reset
   rc = xHowUse()
end

mr.SubNet = ''
mr.MaskSize = 0
mr.Mask = ''
mr.StartAddr = ''
mr.FinishAddr = ''

rc = xGetParam( xMode, pParam )
if rc \= '' then do
   say mr.b.R||rc||mr.c.Reset
   rc = xHowUse()
end

say '      Subnet: '||mr.b.Y||mr.SubNet||'/'||mr.MaskSize||mr.c.Reset
say ' Subnet Mask: '||mr.b.Y||mr.Mask||mr.c.Reset
say 'Subnet Range: '||mr.b.Y||mr.StartAddr||'-'||mr.FinishAddr||mr.c.Reset

exit



/**************************************************************************/
/* xTestParam                                                             */
/*========================================================================*/
/* Description: check the parameter string, select the processing mode    */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xTestParam( pParam )                                           */
/*                                                                        */
/* Parameters:                                                            */
/*    pParam - the parameter string                                       */
/*                                                                        */
/* Return:                                                                */
/*    rc - the processing mode                                            */
/*       = 0   error, the mode is undefined                               */
/*       = 1   the subnet address and the mask size                       */
/*       = 2   the subnet address and the netmask                         */
/*       = 3   the range of a subnet                                      */
/*------------------------------------------------------------------------*/
xTestParam: procedure
   pParam = Strip( arg( 1 ) )

   select
      when Pos( '/', pParam ) \= 0 then do
         rc = 1
      end
      when Pos( '-', pParam ) \= 0 then do
         rc = 3
      end
      when Pos( ' ', pParam ) \= 0 then do
         rc = 2
      end
      otherwise do
         rc = 0
      end
   end

   return rc
/*------------------------------------------------------------------------*/
/* end of function xTestParam()                                           */
/**************************************************************************/



/**************************************************************************/
/* xGetParam                                                              */
/*========================================================================*/
/* Description: get the parameters from the parameter string              */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xGetParam( pMode, pParam )                                     */
/*                                                                        */
/* Parameters:                                                            */
/*    pMode  - the processing mode                                        */
/*       = 1   the subnet address and the mask size                       */
/*       = 2   the subnet address and the netmask                         */
/*       = 3   the range of a subnet                                      */
/*    pParam - the parameter string                                       */
/*                                                                        */
/* Return:                                                                */
/*    rc - the blank string                                               */
/*         or the error message                                           */
/*------------------------------------------------------------------------*/
xGetParam: procedure expose mr.
   pMode = arg( 1 )
   pParam = Strip( arg( 2 ) )

   select
/* subnet/mask_size */
      when pMode = 1 then do
         rc = ''

         parse var pParam xSubNet '/' xMaskSize
         if (xSubNet='' | xMaskSize='') then do
            rc = 'Missing obligatory parameter.'
            return rc
         end
         if (\ xCheckIpAddr( xSubNet )) then do
            rc = 'Invalid IP-address (range of each byte: 0-255).'
            return rc
         end
         if (\ xCheckMaskSize( xMaskSize )) then do
            rc = 'Invalid subnet mask size (range: 0-32).'
            return rc
         end

         mr.Subnet = xSubnet
         mr.MaskSize = xMaskSize
         mr.Mask = xHexToDot( xGetHexMask( mr.MaskSize ) )
      end

/* subnet mask */
      when pMode = 2 then do
         rc = ''

         parse var pParam xSubNet xMask
         if (xSubNet='' | xMask='') then do
            rc = 'Missing obligatory parameter.'
            return rc
         end
         if (\ xCheckIpAddr( xSubNet )) then do
            rc = 'Invalid IP-address (range of each byte: 0-255).'
            return rc
         end
         if (\ xCheckIpAddr( xMask )) then do
            rc = 'Invalid subnet mask (range of each byte: 0-255).'
            return rc
         end

         mr.Subnet = xSubnet
         mr.Mask = xCalcMask( xMask )
         mr.MaskSize = xGetMaskSize( mr.Mask )
      end

/* start-finish */
      when pMode = 3 then do
         rc = ''

         parse var pParam xStart '-' xFinish
         if (xStart='' | xFinish='') then do
            rc = 'Missing obligatory parameter.'
            return rc
         end
         if (\ xCheckIpAddr( xStart )) | ,
            (\ xCheckIpAddr( xFinish )) then do
            rc = 'Invalid IP-address (range of each byte: 0-255).'
            return rc
         end

         x0011 = BitXOR( x2b(xDotToHex( xStart )), x2b(xDotToHex( xFinish )) )
         x1100 = BitXOR( x2b('FFFFFFFF'), x0011 )
         xPos = Pos( '0', x1100 )
         if xPos = 0 then do
            mr.MaskSize = 32
         end
         else do
            mr.MaskSize = xPos - 1
         end
         mr.Mask = xHexToDot( xGetHexMask( mr.MaskSize ) )

         xHexStart = xDotToHex( xStart )
         xHexFinish = xDotToHex( xFinish )
         if xHexStart > xHexFinish then do
            mr.SubNet = xFinish
         end
         else do
            mr.SubNet = xStart
         end
      end

/* unkown mode */
      otherwise do
         rc = 'Unknown parameter processing mode.'
         return rc
      end
   end

   xHexAddr = xDotToHex( mr.SubNet )
   xHexMask = xDotToHex( mr.Mask )
   xHexSubNet = b2x( BitAND( x2b(xHexAddr), x2b(xHexMask) ) )
   mr.SubNet = xHexToDot( xHexSubNet )
   mr.StartAddr = mr.SubNet
   xHexSubNetRange = b2x( BitOR( x2b(xHexSubNet), BitXOR( x2b('FFFFFFFF'), x2b(xHexMask) ) ) )
   mr.FinishAddr = xHexToDot( xHexSubNetRange )
   return rc
/*------------------------------------------------------------------------*/
/* end of function xGetParam()                                            */
/**************************************************************************/



/**************************************************************************/
/* xDotToHex()                                                            */
/*========================================================================*/
/* Description: convert the address from DOT- to HEX-notation             */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xDotToHex( pDotAddr )                                          */
/*                                                                        */
/* Parameters:                                                            */
/*    pDotAddr - the IP-address in the DOT-notation                       */
/*                                                                        */
/* Return:                                                                */
/*    rc - the IP-address in the HEX-notation                             */
/*------------------------------------------------------------------------*/
xDotToHex: procedure
   pDotAddr = arg( 1 )

   parse var pDotAddr x.3'.'x.2'.'x.1'.'x.0
   rc = d2x( x.3, 2 )||d2x( x.2, 2 )||d2x( x.1, 2 )||d2x( x.0, 2 )
   return rc
/*------------------------------------------------------------------------*/
/* end of function xDotToHex()                                            */
/**************************************************************************/



/**************************************************************************/
/* xHexToDot()                                                            */
/*========================================================================*/
/* Description: convert the address from HEX- to DOT-notation             */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xHexToDot( pHexAddr )                                          */
/*                                                                        */
/* Parameters:                                                            */
/*    pHexAddr - the IP-address in the HEX-notation                       */
/*                                                                        */
/* Return:                                                                */
/*    rc - the IP-address in the DOT-notation                             */
/*------------------------------------------------------------------------*/
xHexToDot: procedure
   pHexAddr = arg( 1 )

   do i = 0 to 3
      x.i = substr( pHexAddr, 7-i*2, 2 )
   end
   rc = x2d( x.3, 4 )||'.'||x2d( x.2, 4 )||'.'||x2d( x.1, 4 )||'.'||x2d( x.0, 4 )
   return rc
/*------------------------------------------------------------------------*/
/* end of function xHexToDot()                                            */
/**************************************************************************/



/**************************************************************************/
/* xCheckIpAddr()                                                         */
/*========================================================================*/
/* Description: check the IP-address                                      */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xCheckIpAddr( pAddr )                                          */
/*                                                                        */
/* Parameters:                                                            */
/*    xAddr - the IP-address                                              */
/*                                                                        */
/* Return:                                                                */
/*    rc = 0 - invalid                                                    */
/*       = 1 - valid                                                      */
/*------------------------------------------------------------------------*/
xCheckIpAddr: procedure
   pAddr = arg(1)

   x.0 = 4
   parse var pAddr x.4'.'x.3'.'x.2'.'x.1 .
   do i=1 to x.0
      if (\ xCheckIpAddrByte( x.i )) then do
         return 0
      end
   end
   return 1
/*------------------------------------------------------------------------*/
/* end of function xCheckIpAddr()                                         */
/**************************************************************************/



/**************************************************************************/
/* xCheckIpAddrByte()                                                     */
/*========================================================================*/
/* Description: check the bytes of IP-address                             */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xCheckIpAddrByte( pByte )                                      */
/*                                                                        */
/* Parameters:                                                            */
/*    pByte - the byte of the IP-address                                  */
/*                                                                        */
/* Return:                                                                */
/*    rc = 0 - invalid                                                    */
/*       = 1 - valid                                                      */
/*------------------------------------------------------------------------*/
xCheckIpAddrByte: procedure
   pByte = arg(1)

   if DataType( pByte, 'W' ) then do
      if (pByte>=0) & (pByte<=255) then do
         return 1
      end
   end
   return 0
/*------------------------------------------------------------------------*/
/* end of function xCheckIpAddrByte()                                     */
/**************************************************************************/



/**************************************************************************/
/* xGetHexMask()                                                          */
/*========================================================================*/
/* Description: build the netmask in HEX-notation                         */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xGetHexMask( pMask )                                           */
/*                                                                        */
/* Parameters:                                                            */
/*    pMask - the number of netmask bits                                  */
/*                                                                        */
/* Return:                                                                */
/*    rc - the subnet mask in the HEX-notation                            */
/*------------------------------------------------------------------------*/
xGetHexMask: procedure
   pMask = arg(1)

   rc = b2x( Copies( '1', pMask )||Copies( '0', 32 - pMask ) )
   return rc
/*------------------------------------------------------------------------*/
/* end of function xGetHexMask()                                          */
/**************************************************************************/



/**************************************************************************/
/* xCalcMask()                                                            */
/*========================================================================*/
/* Description: calculate the netmask                                     */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xCalcMask( pMask )                                             */
/*                                                                        */
/* Parameters:                                                            */
/*    pMask - the netmask                                                 */
/*                                                                        */
/* Return:                                                                */
/*    rc - the recalculated netmask                                       */
/*------------------------------------------------------------------------*/
xCalcMask: procedure
   pMask = arg(1)

   xPos = xGetMaskSize( pMask )
   xHex = b2x( Copies( '1', xPos )||Copies( '0', 32 - xPos ) )
   xDot = xHexToDot( xHex )

   return xDot
/*------------------------------------------------------------------------*/
/* end of function xCalcMask()                                            */
/**************************************************************************/



/**************************************************************************/
/* xGetMaskSize()                                                         */
/*========================================================================*/
/* ᠭ: calculate the mask size                                      */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xGetMaskSize( pMask )                                          */
/*                                                                        */
/* Parameters:                                                            */
/*    pMask - the netmask                                                 */
/*                                                                        */
/* Return:                                                                */
/*    rc - the netmask size                                               */
/*------------------------------------------------------------------------*/
xGetMaskSize: procedure
   pMask = arg(1)

   xHex = xDotToHex( pMask )
   xPos = Pos( '0', x2b( xHex ) ) - 1
   if xPos < 0 then do
      xPos = 32
   end

   return xPos
/*------------------------------------------------------------------------*/
/* end of function xGetMaskSize()                                         */
/**************************************************************************/



/**************************************************************************/
/* xCheckMaskSize()                                                       */
/*========================================================================*/
/* Description: check the mask size                                       */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xCheckMaskSize( pSize )                                        */
/*                                                                        */
/* Parameters:                                                            */
/*    pMask - the number of netmask bits                                  */
/*                                                                        */
/* Return:                                                                */
/*    rc = 0 - invalid                                                    */
/*       = 1 - valid                                                      */
/*------------------------------------------------------------------------*/
xCheckMaskSize: procedure
   pSize = arg(1)

   if DataType( pSize, 'W' ) then do
      if (pSize>=0) & (pSize<=32) then do
         return 1
      end
   end
   return 0
/*------------------------------------------------------------------------*/
/* end of function xCheckMaskSize()                                       */
/**************************************************************************/



/**************************************************************************/
/* xGetAnsiValue()                                                        */
/*========================================================================*/
/* Description: initialize the ANSI-sequences                             */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xGetAnsiValue()                                                */
/*                                                                        */
/* Parameters: (none)                                                     */
/*                                                                        */
/* Return:                                                                */
/*    rc = 0 - initialized                                                */
/*------------------------------------------------------------------------*/
xGetAnsiValue: procedure expose mr.
   xEsc=x2c('1B')

/**************************************************************************/
/* The foreground color                                                   */
/**************************************************************************/
   mr.n.K = xEsc||'[0;30m'      /* Black   */
   mr.n.R = xEsc||'[0;31m'      /* Red     */
   mr.n.G = xEsc||'[0;32m'      /* Green   */
   mr.n.Y = xEsc||'[0;33m'      /* Yellow  */
   mr.n.B = xEsc||'[0;34m'      /* Blue    */
   mr.n.M = xEsc||'[0;35m'      /* Magenta */
   mr.n.C = xEsc||'[0;36m'      /* Cyan    */
   mr.n.W = xEsc||'[0;37m'      /* White   */

/**************************************************************************/
/* The foreground color (bright)                                          */
/**************************************************************************/
   mr.b.K = xEsc||'[1;30m'      /* Black   */
   mr.b.R = xEsc||'[1;31m'      /* Red     */
   mr.b.G = xEsc||'[1;32m'      /* Green   */
   mr.b.Y = xEsc||'[1;33m'      /* Yellow  */
   mr.b.B = xEsc||'[1;34m'      /* Blue    */
   mr.b.M = xEsc||'[1;35m'      /* Magenta */
   mr.b.C = xEsc||'[1;36m'      /* Cyan    */
   mr.b.W = xEsc||'[1;37m'      /* White   */

/**************************************************************************/
/* The background color                                                   */
/*************************************************************************/
   mr.g.K = xEsc||'[0;40m'      /* Black   */
   mr.g.R = xEsc||'[0;41m'      /* Red     */
   mr.g.G = xEsc||'[0;42m'      /* Green   */
   mr.g.Y = xEsc||'[0;43m'      /* Yellow  */
   mr.g.B = xEsc||'[0;44m'      /* Blue    */
   mr.g.M = xEsc||'[0;45m'      /* Magenta */
   mr.g.C = xEsc||'[0;46m'      /* Cyan    */
   mr.g.W = xEsc||'[0;47m'      /* White   */

/**************************************************************************/
/* The background color (bright) or the blinking (Full Screen)            */
/**************************************************************************/
   mr.f.K = xEsc||'[5;40m'      /* Black   */
   mr.f.R = xEsc||'[5;41m'      /* Red     */
   mr.f.G = xEsc||'[5;42m'      /* Green   */
   mr.f.Y = xEsc||'[5;43m'      /* Yellow  */
   mr.f.B = xEsc||'[5;44m'      /* Blue    */
   mr.f.M = xEsc||'[5;45m'      /* Magenta */
   mr.f.C = xEsc||'[5;46m'      /* Cyan    */
   mr.f.W = xEsc||'[5;47m'      /* White   */

/**************************************************************************/
/* Attributes                                                             */
/**************************************************************************/
   mr.a.O = xEsc||'[0m'         /* All off    */
   mr.a.B = xEsc||'[1m'         /* Bold       */
   mr.a.N = xEsc||'[2m'         /* Normal     */
   mr.a.L = xEsc||'[5m'         /* Blind      */
   mr.a.I = xEsc||'[7m'         /* Inverse    */
   mr.a.V = xEsc||'[8m'         /* Invisible  */

/**************************************************************************/
/* Commands                                                               */
/**************************************************************************/
   mr.c.Cs = xEsc||'[2J'        /* Clear Screen             */
   mr.c.Cl = xEsc||'[0K'        /* Clear String from Cursor */
   mr.c.Mu = xEsc||'[1A'        /* Cursor Up 1 Row          */
   mr.c.Md = xEsc||'[1B'        /* Cursor Down 1 Row        */
   mr.c.Mr = xEsc||'[1C'        /* Cursor Right 1 Column    */
   mr.c.Ml = xEsc||'[1D'        /* Cursor Left 1 Column     */
   mr.c.Ps = xEsc||'[1s'        /* Save Cursor Position     */
   mr.c.Pr = xEsc||'[1u'        /* Restore Cursor Position  */

   mr.c.Reset = mr.a.O||mr.g.K||mr.n.W
   return 0
/*------------------------------------------------------------------------*/
/* end of function xGetAnsiValue()                                        */
/**************************************************************************/



/**************************************************************************/
/* xHowUse()                                                              */
/*========================================================================*/
/* Description: display the command line sintax                           */
/*                                                                        */
/* Syntax:                                                                */
/*    rc = xHowUse()                                                      */
/*                                                                        */
/* Parameters: (none)                                                     */
/*                                                                        */
/* Return: (none)                                                         */
/*------------------------------------------------------------------------*/
xHowUse: procedure
   say ''
   say 'Usage: rxCalcNet subnet/mask_size'
   say '       rxCalcNet subnet mask'
   say '       rxCalcNet address1-address2'
   say ''
   say ' i.e.: rxCalcNet 192.168.2.0/24'
   say '       rxCalcNet 192.168.2.0 255.255.255.0'
   say '       rxCalcNet 192.168.2.0-192.168.2.255'
   exit 1;
/*------------------------------------------------------------------------*/
/* end of function xHowUse()                                              */
/**************************************************************************/



/*------------------------------------------------------------------------*/
/* end of file rxCalcNet.cmd                                              */
/**************************************************************************/
