/*
    Script to create a maintenance partition or a bootable CD

    Written by Hayo Baan (hrbaan@xs4all.nl)
    http://www.xs4all.nl/~hrbaan

    Please see bootIT.txt for instructions and version history

    Version 2.0 (29/08/2002)
*/

"@echo off"

/* Load the rexUtil functions                                               */
call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
call SysLoadFuncs

/* String variable to insert a newline in a character string                */
CRLF=d2c(13)||d2c(10)

/* Read the configuration file(s)                                           */
rc=readConfig(ARG(1))

/* Determine if command output should be shown                              */
select
    when VERBOSE=9 then IGNOREOUTPUT=""
    when VERBOSE=3 then IGNOREOUTPUT="> /dev/nul"
    otherwise IGNOREOUTPUT="> /dev/nul 2>&1"
end

/* Create temp directory                                                    */
TEMPDIR=value("TEMP",,"OS2ENVIRONMENT")
if TEMPDIR="" then TEMPDIR=value("TMP",,"OS2ENVIRONMENT")
if right(TEMPDIR,1)<>"\" then TEMPDIR=TEMPDIR"\"
TEMPDIR=TEMPDIR"bootIT.$$$"
CLEANUP_=CLEANUP
CLEANUP="NO"
rc=createDir(TEMPDIR)
CLEANUP=CLEANUP_


/****************************************************************************/
/**** Bootable CD creation **************************************************/
/****************************************************************************/

rc=prepareImage(1)
rc=copyImageFiles(1)
rc=createConfigSys()

/* Bootmessages? */
if isSet(BOOTMESSAGES) then
    rc=createFile(VIRTUALDRIVE"\OS2\BOOT\ALTF2ON.$$$", "")

if isSet(TWODISKBOOT) then do
    rc=writeImage(1)
    rc=prepareImage(2)
end

rc=copyImageFiles(2)

if isSet(TWODISKBOOT) then
    rc=writeImage(2)
else
    rc=writeImage(1)

rc=copyCDFiles()

if isSet(WPS) then
    rc=createINIFiles()

call progress 1, "Done creating bootable CD"
exit cleanUp()


/****************************************************************************/
/**** Functions *************************************************************/
/****************************************************************************/

/* Determine if an option is set ------------------------------------------ */
isSet: procedure
    return (left(translate(ARG(1)),1)="Y")


/* Inline version of the if clause ---------------------------------------- */
ifClause: procedure
    if (ARG(1)) then
        return ARG(2)
    else
        return ARG(3)


/* Combination of ifClause and isSet -------------------------------------- */
ifSet: procedure
    if (isSet(ARG(1))) then
        return ARG(2)
    else
        return ARG(3)


/* Returns the file argument if it exists, an empty string if not --------- */
fileExists: procedure
    return stream(ARG(1), "C", "QUERY EXISTS")


/* Returns the first file argument if it exists, the second if not -------- */
ifExists: procedure
    if fileExists(ARG(1))<>"" then
        return ARG(1)
    else
        return ARG(2)


/* Return file name portion of a filespec --------------------------------- */
fileName: procedure
    if ARG(2) <> "" then do
        FS=filespec("NAME", ARG(1))
        FSDOT=pos(".", FS)
        if FSDOT>0 then do
            FSNAME=left(FS,min(FSDOT-1,8))
            FSEXT=right(FS,length(FS)-FSDOT)
        end
        else do
            FSNAME=left(FS,min(length(FS),8))
            FSEXT=""
        end
        return FSNAME||ifClause(FSDOT>0,".")||FSEXT
    end
    else
        return filespec("NAME", ARG(1))


/* Display progress when in given verbose mode ---------------------------- */
progress: procedure expose VERBOSE
    if VERBOSE>=ARG(1) then say ARG(2)
    return rc


/* Write an error message to stderr and exit program ---------------------- */
error: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    call errorMessage ARG(1)
    rc=cleanUp()
    exit 1


/* Write an error message to stderr --------------------------------------- */
errorMessage: procedure
    call lineout "STDERR", "ERROR:" ARG(1)
    return 0


/* Cleanup ---------------------------------------------------------------- */
cleanUp: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    if \ isSet(CLEANUP) then return 0
    call progress 1, "Cleaning up"
    rc=sysFileTree(TEMPDIR"\*", FILETREE, "FO")
    do I=1 to FILETREE.0 while rc=0
        "attrib -r -s -h" FILETREE.I IGNOREOUTPUT
        rc=sysFileDelete(FILETREE.I)
    end
    if rc=0 then
        rc=sysRmDir(TEMPDIR)
    if rc<>0 then
        call errorMessage "Failed to clean up" TEMPDIR
    return rc


/* Create a file with given content --------------------------------------- */
createFile: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    FILE=ARG(1)
    TXT=ARG(2)
    rc=charout(FILE,TXT)
    if rc=0 then
        rc=charout(FILE)
    if rc<>0 then
        call error "Failed to create" FILE
    return 0


/* Copy a list of files --------------------------------------------------- */
copyFiles: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    SOURCEFILES=ARG(1)
    DESTINATION=ARG(2)
    TRUNCATE=ARG(3)
    if right(DESTINATION,1)<>"\" then DESTINATION=DESTINATION"\"
    call progress 1, "Copying files to" DESTINATION
    do while words(SOURCEFILES)>0
        drop FILETREE
        rc=sysFileTree(word(SOURCEFILES,1), FILETREE, "F")
        if rc=0 then do
            do I=1 to FILETREE.0
                rc=copyFile(FILETREE.I, DESTINATION, TRUNCATE)
            end I
            if FILETREE.0=0 then
                call error "File not found" word(SOURCEFILES,1)
        end
        else
            call error "Not enough memory copying" word(SOURCEFILES,1)
        SOURCEFILES=delword(SOURCEFILES,1,1)
    end
    return 0


/* Copy a file without extended attributes -------------------------------- */
copyFile: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    parse value ARG(1) with dummy dummy dummy ATTRIBUTES SOURCEFILESPEC
    ATTRIBUTES=space(translate(ATTRIBUTES, " ", "-"),0)
    SOURCEFILESPEC=strip(SOURCEFILESPEC)
    SOURCEDIR=fileSpec("PATH", SOURCEFILESPEC)
    SOURCEFILE=fileName(SOURCEFILESPEC)
    DESTDIR=ARG(2)
    DESTFILE=fileName(SOURCEFILE, ARG(3))
    if fileExists(DESTDIR||DESTFILE)<>"" then do
        call progress 2, "Ignored" left(ATTRIBUTES,4) SOURCEFILESPEC "(file already copied)"
        return 0
    end
    call progress 2, "Copying" left(ATTRIBUTES,4) SOURCEFILESPEC "=>" DESTDIR||DESTFILE
    "xcopy /h/t/r" SOURCEFILESPEC TEMPDIR IGNOREOUTPUT
    select
        when rc=1 then ERR="File not found"
        when rc=2 then ERR="File errors"
        when rc=3 then ERR="Interrupted"
        when rc=4 then ERR="Not enough room for file"
        otherwise ERR=""
    end
    if rc=0 then do
        "attrib -a -h -r -s" TEMPDIR"\"SOURCEFILE IGNOREOUTPUT
        if rc<>0 then ERR="Resetting attributes failed ("rc")"
    end
    if rc=0 then do
        "eautil /s /r" TEMPDIR"\"SOURCEFILE TEMPDIR"\eautil.$$$" IGNOREOUTPUT
        if rc<>0 then ERR="Removing extended attributes failed"
    end
    if rc=0 then do
        "copy" TEMPDIR"\"SOURCEFILE DESTDIR||DESTFILE IGNOREOUTPUT
        if rc<>0 then ERR="Copy to" DESTDIR||DESTFILE "failed ("rc")"
    end
    if rc=0 then do
        "del" TEMPDIR"\"SOURCEFILE IGNOREOUTPUT
        if rc<>0 then ERR="Deletion of" TEMPDIR"\"SOURCEFILE "failed ("rc")"
    end
    if rc=0 & ATTRIBUTES<>"" then do
        do I=1 to length(ATTRIBUTES)
            SETATTRIBUTES=" +"||substr(ATTRIBUTES,I,1)
        end
        "attrib" SETATTRIBUTES DESTDIR||DESTFILE IGNOREOUTPUT
        if rc<>0 then ERR="Setting" ATTRIBUTES "attributes failed ("rc")"
    end
    if rc<>0 then
        call error "Error copying" SOURCEFILESPEC||":" ERR
    return rc


/* Create a directory ----------------------------------------------------- */
createDir: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    DIRNAME=ARG(1)
    rc=sysMkDir(DIRNAME)
    select
        when rc=2   then ERR="File not found"
        when rc=3   then ERR="Path not found"
        when rc=5   then ERR="Access denied"
        when rc=26  then ERR="Not a DOS disk"
        when rc=87  then ERR="Invalid parameter"
        when rc=108 then ERR="Drive locked"
        when rc=206 then ERR="Filename exceeds range"
        otherwise ERR=""
    end
    if rc<>0 then
        call error "Error creating directory" DIRNAME||":" ERR
    return rc


/* Write a line to the config.sys file ------------------------------------ */
configLine: procedure expose VIRTUALDRIVE
    rc=lineout(VIRTUALDRIVE"\config.sys",ARG(1))
    return rc


/* Read and parse the configuration file(s) ------------------------------- */
readConfig:
    ARGS=ARG(1)
    I=0
    do while length(ARGS)>0
        I=I+1
        parse value ARGS with CONFIGFILES.I ARGS
    end
    CONFIGFILES.0=I

    /* Process all specified configuration files */
    do I=1 to CONFIGFILES.0
        rc=readConfigFile(CONFIGFILES.I)
    end
    /* Process the defaults */
    rc=readConfigFile("bootIT.cfg", "YES")

    /* Make sure the RESERVEDRIVELETTER has no trailing ':' */
    RESERVEDRIVELETTER=left(RESERVEDRIVELETTER,1)

    /* Force USB settings */
    if (USBOHCI>0 | USBUHCI>0 | USBEHCI>0) then
        USB="YES"
    else do
        USB="NO"
        USBKEYB="NO"
        USBCOM="NO"
        USBMOUSE="NO"
        USBPRINTER="NO"
        USBCDROM="NO"
        USBSTORAGE="NO"
    end

    /* Force settings when WPS is set */
    if isSet(WPS) then do
        if isSet(CREATEBOOTCD) then
            RAMDISK="YES"
        MOUSE="YES"
    end
    else
        WPSAPPS="NO"

    /* Force settings when creating a maintenance partition */
    if \ isSet(CREATEBOOTCD) then do
        VIRTUALDRIVE=WRITERDRIVE
        CDROMDRIVE=WRITERDRIVE
        MAKEROOM="NO"
        TWODISKBOOT="NO"
    end

    /* Build up ini-file template parsing replacement table */
    RCREPLACE.0=7
    RCREPLACE.1="$RAMDISKDRIVE:\";  RCREPLACEWITH.1=ifSet(CREATEBOOTCD,RAMDISKDRIVE,WRITERDRIVE)"\"
    RCREPLACE.2="$CDROMDRIVE:\";    RCREPLACEWITH.2=CDROMDRIVE"\"
    RCREPLACE.3="$VIDEODRIVERNAME"; RCREPLACEWITH.3=VIDEODRIVERNAME.VIDEODRIVER
    RCREPLACE.4="$CODEPAGE";        RCREPLACEWITH.4=CODEPAGE
    RCREPLACE.5="$COUNTRYCODE";     RCREPLACEWITH.5=strip(COUNTRY,"L","0")
    RCREPLACE.6="$WPSAPPS";         RCREPLACEWITH.6=ifSet(WPSAPPS,"","/*")
    RCREPLACE.7="$APM";             RCREPLACEWITH.7=ifSet(APM,    "","/*")

    return 0


/* Read and parse an individual configuration file ------------------------ */
readConfigFile:
    CONFIGFILE=ARG(1)
    SETDEFAULTS=ARG(2)
    if filename(CONFIGFILE)=CONFIGFILE then
        CONFIGFILE=SysSearchPath("DPATH",CONFIGFILE)
    select
        when CONFIGFILE="" then rc=1
        when fileExists(CONFIGFILE)="" then rc=1
        otherwise rc=0
    end
    if rc<>0 then
        call error "Configuration file" ARG(1) "not found"
    STATEMENT=""
    LINENR=0
    do while lines(CONFIGFILE)>0
        LINE=strip(linein(CONFIGFILE))
        LINENR=LINENR+1
        CONT=0
        if right(LINE,1)="," then do
            CONT=1
            LINE=left(LINE,length(LINE)-1)
        end
        if length(LINE)>0 & left(LINE,2)<>"/*" then
            STATEMENT=strip(STATEMENT LINE)
        if CONT=0 then do
            DOPARSE="YES"
            if isSet(SETDEFAULTS) then do
                EQUALSIGN=pos("=", STATEMENT)
                if (EQUALSIGN>0) then do
                    VARNAME=left(STATEMENT,EQUALSIGN-1)
                    if translate(VARNAME)<>value(VARNAME) then
                        DOPARSE="NO"
                end
            end
            if isSet(DOPARSE) then
                interpret STATEMENT
            STATEMENT=""
        end
    end
    rc=stream(CONFIGFILE, "C", "CLOSE")
    if rc<>"READY:" then
        call error "Failed to close" CONFIGFILE "rc="rc

    return 0


/* List of MAKEROOM files for the os2 directory --------------------------- */
makeRoomFiles:
    FILES_=VITALFILES_ SHELLFILE ADDITIONALSHELLFILES PROTSHELLFILE
    if isSet(WPS)          then FILES_=FILES_ WPSRUNWORKPLACEFILE
    if isSet(CDFS)         then FILES_=FILES_ CDFSFILES
    if isSet(FAT32)        then FILES_=FILES_ FAT32FILES
    if isSet(HPFS)         then FILES_=FILES_ HPFSFILES
    if isSet(JFS)          then FILES_=FILES_ JFSFILES
    if isSet(UDF)          then FILES_=FILES_ UDFFILES
    if isSet(RAMDISK)      then FILES_=FILES_ RAMDISKFILE
    if isSet(LVM)          then FILES_=FILES_ LVMFILES
    if isSet(APM)          then FILES_=FILES_ APMFILE APMDAEMONFILE
    if isSet(PARALLELPORT) then FILES_=FILES_ PARALLELPORTFILE
    if isSet(SERIALPORT)   then FILES_=FILES_ SERIALPORTFILE
    if isSet(MOUSE)        then FILES_=FILES_ MOUSEPOINTDDFILE MOUSEFILE MOUSEDEVFILE
    if isSet(USBKEYB)      then FILES_=FILES_ USBKEYBFILE
    if isSet(USBCOM)       then FILES_=FILES_ USBCOMFILE
    if isSet(USBMOUSE)     then FILES_=FILES_ USBMOUSEFILE
    if isSet(USBPRINTER)   then FILES_=FILES_ USBPRINTERFILE
    return FILES_


/* List of MAKEROOM files for the os2\dll directory ----------------------- */
makeRoomDLLs:
    DLLS_=VITALDLLFILES_
    if isSet(CDFS)    then DLLS_=DLLS_ CDFSDLLS
    if isSet(FAT32)   then DLLS_=DLLS_ FAT32DLLS
    if isSet(HPFS)    then DLLS_=DLLS_ HPFSDLLS
    if isSet(JFS)     then DLLS_=DLLS_ JFSDLLS
    if isSet(UDF)     then DLLS_=DLLS_ UDFDLLS
    if isSet(LVM)     then DLLS_=DLLS_ LVMDLLS
    if isSet(UNICODE) then DLLS_=DLLS_ UNICODEDLLS
    return DLLS_
    

/* Prepare a boot image --------------------------------------------------- */
prepareImage:
    IMAGENR=ARG(1)

    /* Format the (virtual) disk */
    call progress 1, "Formatting boot image" IMAGENR
    if isSet(CREATEBOOTCD) | isSet(FORMATMAINTENANCE) then do
        LABEL=ifSet(CREATEBOOTCD,"CDBOOT_"IMAGENR,"MAINTENANCE")
        "label" VIRTUALDRIVE||"FAILED" IGNOREOUTPUT
        if rc<>0 then
            call error "Failed to reset the label of" VIRTUALDRIVE
        rc=lineOut(TEMPDIR"\response", "FAILED"||CRLF||"Y")
        if rc=0 then
            rc=lineout(TEMPDIR"\response")
        if rc<> 0 then
            call error "Failed to create format response file"
        "format" VIRTUALDRIVE "/fs:"ifSet(CREATEBOOTCD,"fat","hpfs") "/once /v:"LABEL "<" TEMPDIR"\response" IGNOREOUTPUT
        /* Somehow format does not always return non-zero rc in case of error... */
        if rc<>0 | word(sysDriveInfo(VIRTUALDRIVE),4)<>LABEL then
            call error "Error formatting boot image" IMAGENR "("VIRTUALDRIVE")"
    end

    /* Install system files */
    call progress 1, "Transfering system files to boot image" IMAGENR
    SYSINSTX VIRTUALDRIVE IGNOREOUTPUT
    if rc<>0 then
        call error "Error transfering system files to boot image" IMAGENR

    return 0


/* Write a boot image to CD ----------------------------------------------- */
writeImage:
    if \ isSet(CREATEBOOTCD) then return 0

    IMAGENR=ARG(1)   

    if isSet(PAUSEBEFOREBOOTIMAGE) then do
        say "Pausing before writing boot image" IMAGENR
        say "Continue with script? ([Y]es/[N]o/[Q]uit without cleaning up)"
        rc=1
        do until rc=0
            KEY=translate(sysGetKey("NOECHO"))
            if pos(KEY,"YNQ")>0 then
                rc=0
            else
                call beep 524,100
        end
        say KEY
        if KEY<>"Y" then do
            if KEY="Q" then
                CLEANUP="NO"
            call error "Script aborted"
        end
    end

    call progress 1, "Writing boot image" IMAGENR
    "savedskf /a/d" VIRTUALDRIVE WRITERDRIVE"\CDBOOT_"IMAGENR".IMG" IGNOREOUTPUT
    if rc<>0 then
        call error "Error writing boot image" IMAGENR

    /* The first image will be marked bootable */
    if IMAGENR=1 & \ isSet(CDRECORD) then do
        "cdboot" WRITERDRIVE"\CDBOOT_"IMAGENR".img" IGNOREOUTPUT
        if rc<>0 then
            call error "Error marking boot image"
    end

    return 0


/* Copy files to boot image ----------------------------------------------- */
copyImageFiles:
    IMAGENR=ARG(1)

    call progress 1, "Copying files to boot image" IMAGENR

    /* Files to copy to root directory of boot image */
    if IMAGENR=1 | isSet(TWODISKBOOT) then
        rc=copyFiles(VITALROOTFILES.IMAGENR, VIRTUALDRIVE"\", "T")

    /* Files to copy to os2 directory of boot image */
    if IMAGENR=1 | isSet(TWODISKBOOT) then do
        rc=createDir(VIRTUALDRIVE"\OS2")
    end
    if IMAGENR=2 then do
        FILES=VITALFILES
        if isSet(CDROM)   then FILES=FILES CDROMDMDFILE
        if isSet(CDFS)    then FILES=FILES CDFSIFSFILE
        if isSet(FAT32)   then FILES=FILES FAT32IFSFILE
        if isSet(HPFS)    then FILES=FILES HPFSIFSFILE
        if isSet(JFS)     then FILES=FILES JFSIFSFILE
        if isSet(UDF)     then FILES=FILES UDFIFSFILE
        if isSet(RAMDISK) then FILES=FILES RAMDISKIFSFILE
        if isSet(UNICODE) then FILES=FILES UNICODEFILE
        if \ isSet(MAKEROOM) then FILES=FILES makeRoomFiles()
        rc=copyFiles(FILES, VIRTUALDRIVE"\os2", "T")
    end
    
    /* Files to copy to os2\boot directory of boot image */
    if IMAGENR=1 then do
        rc=createDir(VIRTUALDRIVE"\OS2\BOOT")
        BOOTFILES=VITALBOOTFILES BOOTFLTFILE
        if isSet(WPS)          then BOOTFILES=BOOTFILES WPSBOOTFILES
        if isSet(FLOPPY)       then BOOTFILES=BOOTFILES FLOPPYADDFILE
        if isSet(DASDFALLBACK) then BOOTFILES=BOOTFILES DASDFALLBACKFILE
        if isSet(IDE)          then BOOTFILES=BOOTFILES IDEADDFILE
        if isSet(ATAPI)        then BOOTFILES=BOOTFILES ATAPIFLTFILE
        if isSet(ASPI)         then BOOTFILES=BOOTFILES ASPIDMDFILE
        if isSet(LVM)          then BOOTFILES=BOOTFILES LVMDMDFILE
        do I=1 to SCSI
            BOOTFILES=BOOTFILES SCSIADDFILE.I
        end
        if isSet(USB) then do
            BOOTFILES=BOOTFILES USBDFILE USBHIDFILE
            if USBOHCI>0 then BOOTFILES=BOOTFILES USBOHCIFILE
            if USBUHCI>0 then BOOTFILES=BOOTFILES USBUHCIFILE
            if USBEHCI>0 then BOOTFILES=BOOTFILES USBEHCIFILE
            if isSet(USBCDROM)   then BOOTFILES=BOOTFILES USBCDROMFILE
            if isSet(USBSTORAGE) then BOOTFILES=BOOTFILES USBSTORAGEFILE
        end
        rc=copyFiles(BOOTFILES, VIRTUALDRIVE"\os2\boot", "T")
    end
    if IMAGENR=2 & isSet(APM) & APMGSVDAEMONFILE<>"" then do
        if isSet(TWODISKBOOT) then
            rc=createDir(VIRTUALDRIVE"\OS2\BOOT")
        rc=copyFiles(APMGSVDAEMONFILE, VIRTUALDRIVE"\OS2\BOOT", "T")
    end

    /* Files to copy to os2\dll directory of boot image */
    if IMAGENR=2 then do
        rc=createDir(VIRTUALDRIVE"\OS2\DLL")
        DLLS=VITALDLLFILES
        if \ isSet(MAKEROOM) then DLLS=DLLS makeRoomDLLs()
        rc=copyFiles(DLLS, VIRTUALDRIVE"\os2\dll", "T")
    end
    
    /* Unicode support files */
    if IMAGENR=2 & isSet(UNICODE) then do
        /* Language directory tree */
        rc=createDir(VIRTUALDRIVE"\LANGUAGE")
        rc=createDir(VIRTUALDRIVE"\LANGUAGE\CODEPAGE")
        rc=createDir(VIRTUALDRIVE"\LANGUAGE\"UNILANGDIR)
        rc=createDir(VIRTUALDRIVE"\LANGUAGE\FONT")
        rc=createDir(VIRTUALDRIVE"\LANGUAGE\KEYBOARD")
        rc=createDir(VIRTUALDRIVE"\LANGUAGE\LOCALE")
        rc=copyFiles(UNICODECODEPAGEFILE,  VIRTUALDRIVE"\language\codepage", "T")
        rc=copyFiles(UNICODELANGUAGEFILES, VIRTUALDRIVE"\language\"UNILANGDIR, "T")
        rc=copyFiles(UNICODEKEYBOARDFILE,  VIRTUALDRIVE"\language\keyboard", "T")
        rc=copyFiles(UNICODEUSERFILE,      VIRTUALDRIVE"\language\locale", "T")
    end
    
    return rc


/* Copy files to CD ------------------------------------------------------- */
copyCDFiles:
    call progress 1, "Writing files to CD"

    /* Files to copy to os2 directory of CD */
    if isSet(CREATEBOOTCD) then rc=createDir(WRITERDRIVE"\OS2")
    FILES=EXTRAFILES
    if isSet(OS2APPS)  then FILES=FILES OS2APPFILES
    if isSet(MAKEROOM) then FILES=FILES makeRoomFiles()
    if isSet(WPS)      then FILES=FILES WPSFILES VIDEODRIVERFILES.VIDEODRIVER
    rc=copyFiles(FILES, WRITERDRIVE"\os2")

    /* Files to copy to os2\dll directory of CD */
    if isSet(CREATEBOOTCD) then rc=createDir(WRITERDRIVE"\OS2\DLL")
    DLLS=EXTRADLLS
    if isSet(OS2APPS)  then DLLS=DLLS OS2APPDLLS
    if isSet(MAKEROOM) then DLLS=DLLS makeRoomDLLs()
    if isSet(WPS)      then DLLS=DLLS WPSDLLFILES VIDEODRIVERDLLS.VIDEODRIVER
    rc=copyFiles(DLLS, WRITERDRIVE"\os2\dll")

    /* Files to copy to os2\help directory of CD */
    if isSet(WPS) then do
        rc=createDir(WRITERDRIVE"\OS2\HELP")
        HELPFILES=WPSHELPFILES VIDEODRIVERHELP.VIDEODRIVER
        if isSet(WPSAPPS)    then HELPFILES=HELPFILES WPSAPPSHELPFILES
        rc=copyFiles(HELPFILES, WRITERDRIVE"\os2\help")
    end

    /* Files to copy to os2\apps directory of CD */
    if isSet(WPSAPPS) then do
        rc=createDir(WRITERDRIVE"\OS2\APPS")
        rc=copyFiles(WPSAPPSFILES, WRITERDRIVE"\OS2\APPS")
    end

    /* Files to copy to os2\apps\dll directory of CD */
    if isSet(WPSAPPS) then do
        rc=createDir(WRITERDRIVE"\OS2\APPS\DLL")
        rc=copyFiles(WPSAPPSDLLFILES, WRITERDRIVE"\OS2\APPS\DLL")
    end

    return rc


/* Create config.sys file ------------------------------------------------- */
createConfigSys:
    call progress 1, "Creating config.sys file on boot image"
    
    /* Drive letter for files that can reside on boot image or on CD */
    if isSet(MAKEROOM) | \ isSet(CREATEBOOTCD) then
        DRIVELETTER=CDROMDRIVE
    else
        DRIVELETTER="a:"

    /* Truncate file names only if they are written to the boot image */
    if isSet(MAKEROOM) then
        TRUNCATE=""
    else
        TRUNCATE="T"

    /* Reservedriveletter */
    rc=configLine("RESERVEDRIVELETTER="RESERVEDRIVELETTER)

    /* Paths */
    rc=configLine("LIBPATH=.;"ifSet(CREATEBOOTCD,"a:\os2\dll;")||CDROMDRIVE"\os2\dll;"||ifSet(WPSAPPS,CDROMDRIVE"\os2\apps\dll;"))
    rc=configLine("SET PATH="ifSet(CREATEBOOTCD,"a:\os2;")||CDROMDRIVE"\os2;"||ifSet(WPSAPPS,CDROMDRIVE"\os2\apps;"))
    rc=configLine("SET DPATH="ifSet(CREATEBOOTCD,"a:\os2;")||CDROMDRIVE"\os2;"||ifSet(WPSAPPS,CDROMDRIVE"\os2\apps;"))

    /* File systems */
    if isSet(CDFS)    then rc=configLine("IFS=\os2\"fileName(CDFSIFSFILE,"T")  CDFSOPTIONS)
    if isSet(FAT32)   then rc=configLine("IFS=\os2\"fileName(FAT32IFSFILE,"T") FAT32OPTIONS)
    if isSet(HPFS)    then rc=configLine("IFS=\os2\"fileName(HPFSIFSFILE,"T")  HPFSOPTIONS)
    if isSet(JFS)     then rc=configLine("IFS=\os2\"fileName(JFSIFSFILE,"T")   JFSOPTIONS)
    if isSet(UDF)     then rc=configLine("IFS=\os2\"fileName(UDFIFSFILE,"T")   UDFOPTIONS)

    /* Ramdisk */    
    if isSet(RAMDISK) then do
        rc=configLine("IFS=\os2\"fileName(RAMDISKIFSFILE,"T") RAMDISKOPTIONS)
        rc=configLine("RUN="DRIVELETTER"\os2\"filename(RAMDISKFILE,TRUNCATE) RAMDISKDRIVE "RAMDISK")
        rc=configLine("CALL="filename(SHELLFILE,TRUNCATE) "/c mkdir" RAMDISKDRIVE"\Temp > /dev/nul" )
        rc=configLine("SET TEMP="RAMDISKDRIVE"\Temp")
        rc=configLine("SET TMP="RAMDISKDRIVE"\Temp")
        rc=configLine("SUPPRESSPOPUPS="left(RAMDISKDRIVE,1))
    end

    /* Base device driver for keyboard */
    rc=configLine("BASEDEV=ibmkbd.sys")

    /* Base device driver for storage */
    rc=configLine("BASEDEV=os2dasd.dmd")

    /* Base device drivers for storage devices */
    if isSet(FLOPPY) then rc=configLine("BASEDEV="fileName(FLOPPYADDFILE,"T") FLOPPYOPTIONS)
    if isSet(IDE) & \ isSet(SCSIBEFOREIDE) then rc=configLine("BASEDEV="fileName(IDEADDFILE,"T") IDEOPTIONS)
    do I=1 to SCSI
        rc=configLine("BASEDEV="fileName(SCSIADDFILE.I,"T") SCSIOPTIONS.I)
    end
    if isSet(IDE) & isSet(SCSIBEFOREIDE) then rc=configLine("BASEDEV="fileName(IDEADDFILE,"T") IDEOPTIONS)
    if isSet(ATAPI)        then rc=configLine("BASEDEV="fileName(ATAPIFLTFILE,"T")     ATAPIOPTIONS)
    if isSet(DASDFALLBACK) then rc=configLine("BASEDEV="fileName(DASDFALLBACKFILE,"T") DASDFALLBACKOPTIONS)
    if isSet(LVM)          then rc=configLine("BASEDEV="fileName(LVMDMDFILE,"T")       LVMOPTIONS)

    /* Extra base device drivers */
    do I=1 to EXTRABASEDEV
        rc=configLine("BASEDEV="fileName(EXTRABASEDEVFILE.I,"T") EXTRABASEDEVOPTIONS.I)
    end
    
    /* Base device driver providing bootable CD support */
    if isSet(CREATEBOOTCD) then rc=configLine("BASEDEV="fileName(BOOTFLTFILE,"T") BOOTOPTIONS)

    /* Codepage, display, and keyboard layout */
    rc=configLine("COUNTRY="COUNTRY",\os2\country.sys")
    rc=configLine("CODEPAGE="CODEPAGE)
    rc=configLine("DEVINFO=KBD,"KEYBOARD",\os2\keyboard.dcp")
    rc=configLine("DEVINFO=SCR,VGA,\os2\viotbl.dcp")

    /* Unicode */
    if isSet(UNICODE) then do
      rc=configLine("SET LANG="UNILANG)
      rc=configLine("SET ULSPATH=\language")
      rc=configLine("DEVICE=\os2\"fileName(UNICODEFILE,"T"))
    end

    /* CDROM support */
    if isSet(CDROM) then rc=configLine("DEVICE=\os2\"fileName(CDROMDMDFILE,"T"))

    /* ASPI support */
    if isSet(ASPI)  then rc=configLine("DEVICE=\os2\"fileName(ASPIFILE,"T"))

    /* Test device driver (required?) */
    rc=configLine("DEVICE="DRIVELETTER"\os2\testcfg.sys")

    /* Mouse support */
    if isSet(MOUSE) then do
        rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(MOUSEPOINTDDFILE,TRUNCATE))
        if MOUSEDEVFILE<>"" then rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(MOUSEDEVFILE,TRUNCATE) MOUSEDEVFILEOPTIONS)
        rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(MOUSEFILE,TRUNCATE) MOUSEFILEOPTIONS)
    end
    
    /* USB */
    if isSet(USB) then do
        do USBOHCI
            rc=configLine("BASEDEV="fileName(USBOHCIFILE,"T") USBOHCIOPTIONS)
        end
        do USBUHCI
            rc=configLine("BASEDEV="fileName(USBUHCIFILE,"T") USBUHCIOPTIONS)
        end
        do USBEHCI
            rc=configLine("BASEDEV="fileName(USBEHCIFILE,"T") USBEHCIOPTIONS)
        end
        rc=configLine("BASEDEV="fileName(USBDFILE,  "T") USBDOPTIONS)
        rc=configLine("BASEDEV="fileName(USBHIDFILE,"T") USBHIDOPTIONS)
        if isSet(USBKEYB)    then rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(USBKEYBFILE,   TRUNCATE) USBKEYBOPTIONS)
        if isSet(USBCOM)     then rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(USBCOMFILE,    TRUNCATE) USBCOMOPTIONS)
        if isSet(USBMOUSE)   then rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(USBMOUSEFILE,  TRUNCATE) USBMOUSEOPTIONS)
        if isSet(USBPRINTER) then rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(USBPRINTERFILE,TRUNCATE) USBPRINTEROPTIONS)
        if isSet(USBCDROM)   then rc=configLine("BASEDEV="fileName(USBCDROMFILE,"T")   USBCDROMOPTIONS)
        if isSet(USBSTORAGE) then rc=configLine("BASEDEV="fileName(USBSTORAGEFILE,"T") USBSTORAGEOPTIONS)
    end
    
    /* Serial port support */
    if isSet(SERIALPORT) then
        rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(SERIALPORTFILE,TRUNCATE) SERIALPORTOPTIONS)

    /* Parallel port support */
    if isSet(PARALLELPORT) then do
        rc=configLine("DEVICE="DRIVELETTER"\os2\"fileName(PARALLELPORTFILE,TRUNCATE) PARALLELPORTOPTIONS)
        rc=configLine("PRINTMONBUFSIZE=134,134,134")
    end

    /* APM */
    if isSet(APM) then do
        rc=configLine("DEVICE="DRIVELETTER"\os2\"filename(APMFILE,TRUNCATE) APMOPTIONS)
        if APMDAEMONFILE<>"" then rc=configLine("RUN="DRIVELETTER"\os2\"filename(APMDAEMONFILE,TRUNCATE) APMDAEMONOPTIONS)
    end

    /* WPS */
    if isSet(WPS) then do
        rc=configLine("BASEDEV=timer0.sys")
        rc=configLine("DEVICE="DRIVELETTER"\os2\pmdd.sys")
        if isSet(CREATEBOOTCD) then do
            rc=configLine("CALL="filename(SHELLFILE,TRUNCATE) "/c mkdir" RAMDISKDRIVE"\OS2 > /dev/nul")
            rc=configLine("CALL="filename(SHELLFILE,TRUNCATE) "/c copy" CDROMDRIVE"\os2\*.ini" RAMDISKDRIVE"\OS2 > /dev/nul")
        end
        rc=configLine("SET USER_INI="ifSet(CREATEBOOTCD,RAMDISKDRIVE,CDROMDRIVE)"\os2\os2.ini")
        rc=configLine("SET SYSTEM_INI="ifSet(CREATEBOOTCD,RAMDISKDRIVE,CDROMDRIVE)"\os2\os2sys.ini")
        rc=configLine("SET AUTOSTART=PROGRAMS,TASKLIST,FOLDERS,WARPCENTER")
        rc=configLine("SET RESTARTOBJECTS=STARTUPFOLDERSONLY")
        rc=configLine("SET DISPLAYTYPE=VGA")
        rc=configLine("SET HELP="CDROMDRIVE"\os2\help")
        rc=configLine("SET BOOKSHELF="CDROMDRIVE"\os2\help")
        if isSet(WPSAPPS) then rc=configLine("SET EPMPATH="CDROMDRIVE"\os2\apps")
    end

    /* System settings */
    rc=configLine("MEMMAN=NOSWAP,PROTECT")
    rc=configLine("IOPL=YES")
    rc=configLine("PROTECTONLY=YES")
    rc=configLine("THREADS=96")
    rc=configLine("DISKCACHE=48,LW")
    rc=configLine("MAXWAIT=1")

    /* Display settings */    
    if isSet(WPS) then rc=configLine("SET DISPLAYTYPE=VGA")
    select
        when isSet(WPS) & VIDEODRIVER=2 then do
            rc=configLine("SET VIDEO_DEVICES=VIO_SVGA")
            rc=configLine("SET VIO_SVGA=DEVICE(BVHVGA,BVHSVGA)")
            rc=configLine("SET C1=GENGRADD")
            rc=configLine("SET GRADD_CHAINS=C1")
            rc=configLine("DEVICE="CDROMDRIVE"\os2\gradd.sys")
        end
        otherwise do
            rc=configLine("SET VIDEO_DEVICES=VIO_VGA")
            rc=configLine("SET VIO_VGA=DEVICE(BVHVGA)")
        end
    end

    /* Command environment */
    TZ=value("TZ",,"OS2ENVIRONMENT")
    if TZ<>"" then rc=configLine("SET TZ="TZ)
    rc=configLine("SET PROMPT=$P$$$s")
    rc=configLine("SET KEYS=ON")

    /* Shell */
    rc=configLine("PROTSHELL="DRIVELETTER"\os2\"fileName(PROTSHELLFILE,TRUNCATE) PROTSHELLFILEOPTIONS)
    if isSet(WPS) then do
        rc=configLine("SET RUNWORKPLACE="DRIVELETTER"\os2\"fileName(WPSRUNWORKPLACEFILE,TRUNCATE) WPSRUNWORKPLACEFILEOPTIONS)
    end
    rc=configLine("SET OS2_SHELL="DRIVELETTER"\os2\"fileName(SHELLFILE,TRUNCATE) SHELLFILEOPTIONS)
    rc=configLine("SET COMSPEC="DRIVELETTER"\os2\"fileName(SHELLFILE,TRUNCATE))

    /* Additional config.sys lines */
    rc=configLine(ADDITIONALCFG)
    
    /* Close config.sys */
    rc=lineout(VIRTUALDRIVE"\config.sys")
    
    return rc


/* Create INI files on the CD --------------------------------------------- */
createINIFiles:
    call progress 1, "Creating INI files"

    rc=parseINIRC(WPSUSERINIRC, TEMPDIR"\ini.rc")
    rc=parseINIRC(WPSSYSINIRC,  TEMPDIR"\inisys.rc")

    /* Create the ini files based on the templates */
    "makeini" TEMPDIR"\os2.ini" TEMPDIR"\ini.rc" IGNOREOUTPUT
    if rc<>0 then
        call error "Failed to create os2.ini file using template" WPSUSERINIRC
    "makeini" TEMPDIR"\os2sys.ini" TEMPDIR"\inisys.rc" IGNOREOUTPUT
    if rc<>0 then
        call error "Failed to create os2sys.ini file using template" WPSSYSINIRC

    /* Add display resolution setting */
    if VIDEODRIVER<>1 and VIDEORESOLUTION<>0 then do
        /* Video resolution and color depth are encoded in 32bit Low-Endian notation */
        select
            when VIDEORESOLUTION=2 then VIDEORESOLUTIONSTRING="20030000 58020000" /*  800x 600 */
            when VIDEORESOLUTION=3 then VIDEORESOLUTIONSTRING="00040000 00030000" /* 1024x 768 */
            when VIDEORESOLUTION=4 then VIDEORESOLUTIONSTRING="00050000 00040000" /* 1280x1024 */
            when VIDEORESOLUTION=5 then VIDEORESOLUTIONSTRING="40060000 b0040000" /* 1600x1200 */
            otherwise                   VIDEORESOLUTIONSTRING="80020000 e0010000" /*  640x 480 */
        end
        select
            when VIDEOCOLORDEPTH=2 then VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING "00800000" /* 32K colors */ 
            when VIDEOCOLORDEPTH=3 then VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING "00000100" /* 64K colors */
            when VIDEOCOLORDEPTH=4 then VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING "00000001" /* 16M colors */
            otherwise                   VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING "00010000" /* 256 colors */
        end
        /* Two extra 32bit integers don't know what they are for (refresh rate?) */
        VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING "01000000 40000000"
        rc=sysIni(TEMPDIR"\os2.ini", "PM_DISPLAYDRIVERS", "DEFAULTSYSTEMRESOLUTION", X2C(VIDEORESOLUTIONSTRING))
        if rc<>"" then
            call error "Failed to add default system resolution to os2.ini" "RC="rc
    end

    /* Move the ini files to CD */
    "copy" TEMPDIR"\os2.ini" WRITERDRIVE"\OS2" IGNOREOUTPUT
    if rc<>0 then ERR="Error copying" TEMPDIR"\os2.ini to" WRITERDRIVE"\OS2 rc("rc")"
    if rc=0 then do
        "del" TEMPDIR"\os2.ini" IGNOREOUTPUT
        if rc<>0 then ERR="Error deleting" TEMPDIR"\os2.ini rc("rc")"
    end
    if rc=0 then do
        "copy" TEMPDIR"\os2sys.ini" WRITERDRIVE"\OS2" IGNOREOUTPUT
        if rc<>0 then ERR="Error copying" TEMPDIR"\os2sys.ini to" WRITERDRIVE"\OS2 rc("rc")"
    end
    if rc=0 then do
        "del" TEMPDIR"\os2sys.ini" IGNOREOUTPUT
        if rc<>0 then ERR="Error deleting" TEMPDIR"\os2sys.ini rc("rc")"
    end
    if rc<>0 then
        call error(ERR)

    return 0


/* Parse INI file template ------------------------------------------------ */
parseINIRC:
    SOURCERC=ARG(1)
    DESTINATIONRC=ARG(2)

    call progress 1, "Parsing INI file template" SOURCERC

    do while lines(SOURCERC)>0
        RCLINE=linein(SOURCERC)
        do I=1 to RCREPLACE.0
            STARTPOS=1
            do until STARTPOS=0
                STARTPOS=pos(RCREPLACE.I, RCLINE, STARTPOS)
                if (STARTPOS<>0) then do
                    RCLINE=left(RCLINE,STARTPOS-1)||RCREPLACEWITH.I||right(RCLINE,length(RCLINE)-STARTPOS-length(RCREPLACE.I)+1)
                    STARTPOS=STARTPOS+length(RCREPLACEWITH.I)
                end
            end
        end
        if left(strip(RCLINE),2)<>"/*" then
            rc=lineout(DESTINATIONRC, RCLINE)
    end
    
    /* Close files */
    rc=lineout(DESTINATIONRC)
    if rc<>0 then
        call error "Failed to close file" DESTINATIONRC "rc("rc")"
    rc=stream(SOURCERC, "C", "CLOSE")
    if rc<>"READY:" & rc<>"" then
        call error "Failed to close file" SOURCERC "rc("rc")"

    return 0

