#include <stdio.h>
#include <functional>
#include <type_traits>
#include <typeinfo>

#define WR         /*read-write*/
#define RC         /*recursive*/
#define API        /*namespace-api*/

#define interface  class
#define mixin      class

#define extends    public
#define implements public
#define imports    private
#define export     using
#define transitive /*export-not-required*/
#define DI         /*dependency-injection*/

#define var        auto /*new-heap-var*/
#define let        /*new-heap-let-type*/

#define with       std::function        // Method with External Method
#define self_ptr   this                 // Calling Chain Link to Self
#define mix        (*this)              // Calling to Mix-in

#define at         class
#define on         class

#define type       typename
#define type_of    enable_if_same_type

#define include    if constexpr
#define PURE_C     1
#define STD_API    1
#define OS2_API    1
#define X11_API    1

#define size_of(x) ( sizeof (x) )
#define array_size(x) ( sizeof (x) )
#define array_length(x) ( sizeof (x) / sizeof (x[0]) )

#define free(ptr) ( delete *ptr, *ptr = NULL )

#define t_id(x) ( typeid (x).hash_code () )
#define t_name(x) ( typeid (x).name () )

#define error(ec) ( exit (ec + (fprintf (stderr, "Error: %d, %s(), %s, line %d." "\n", \
                                                 ec, __FUNCTION__, __FILE__, __LINE__) * 0)) )

typedef void struct_ptr;

typedef unsigned char      byte;
typedef unsigned long      ulong;
typedef unsigned short     ushort;
typedef unsigned long long ulong64;

typedef unsigned long      hfile;
typedef unsigned long      hmq;
typedef unsigned long      hwnd;

template <type T, type U> inline constexpr
     int enable_if_same_type = std::is_same <T, U>::value;

/* * */

namespace list_item
{
  template <type T> interface IMeasurable
  {
    public:
      virtual float get_float_value () = 0;
    ;
  };
}

export list_item::IMeasurable;

/* * */

namespace list
{
  template <type T> interface IContainer
  {
    public:
      virtual WR IContainer <T> * add (T *) API = 0;
    ;
  };

  template <type T> interface IEnumerable
  {
    public:
      virtual void for_each (with <void (T *)>) API = 0;
    ;
  };

  template <type T> interface IFreezable
  {
    public:
      virtual WR void freeze () API = 0;
    ;
  };
}

export list::IContainer;
export list::IEnumerable;
export list::IFreezable;

/* * */

namespace list
{
  #ifndef ERR_BOUNDS
  #define ERR_BOUNDS -1
  #endif
  #ifndef ERR_FROZEN
  #define ERR_FROZEN -2
  #endif

  template <type T> class RC WR ListItem : implements IMeasurable <T>
  {
    public:
      ListItem (T *ptr, with <void (let ListItem <T> WR *)> init = NULL)
      {
        printf ("- %d" "\t" "%s" "\n", t_id (T), t_name (T));

        pointer = ptr;

        if (init) init (self_ptr);
      }

      virtual ~ListItem () {}

      float get_float_value ();
    ;

    public:
      ListItem <T> *next_item = NULL;
      T *pointer = NULL;
    ;
  };

  template <type T> float ListItem <T>::get_float_value ()
  {
    if constexpr (type_of <T, int> || type_of <T, float> || type_of <T, ulong> || type_of <T, ulong64>)
    {
      return (float) *pointer;
    }
    else
    {
      return (float) pointer->value;
    }
  }

  template <type T> class WR ArrayList : implements IContainer <T>,
                                         implements IEnumerable <T>,
                                         implements IFreezable <T>
  {
    public:
      ArrayList (int capacity, with <void (let ArrayList <T> WR *)> init = NULL)
      {
        printf ("- %d" "\t" "%s" "\n", t_id (T), t_name (T));

        max_index = capacity;
        pointers = new T * [capacity] {};

        if (init) init (self_ptr);
      }

      virtual ~ArrayList ()
      {
        free (&pointers);
      }

      ArrayList <T> * add (T *);
      void freeze ();

      void for_each (with <void (T *)>);
    ;

    private:
      int index = 0; int max_index = 0;
      T * *pointers = NULL;

      int frozen = 0;
    ;
  };

  template <type T> ArrayList <T> * ArrayList <T>::add (T *pointer)
  {
    if (!frozen)
    {
      if (index < max_index)
      {
        pointers[index] = pointer; index ++;
      }
      else
      {
        error (ERR_BOUNDS);
      }
    }
    else
    {
      error (ERR_FROZEN);
    }

    return self_ptr;
  }

  template <type T> void ArrayList <T>::freeze ()
  {
    frozen = 1;
  }

  template <type T> void ArrayList <T>::for_each (with <void (T *)> process_item)
  {
    for (int i = 0; i < index; i ++)
    {
      process_item (pointers[i]);
    }
  }

  /* * */

  template <type T> class WR LinkedList : implements IContainer <T>,
                                          implements IEnumerable <T>,
                                          implements IFreezable <T>
  {
    public:
      LinkedList (with <void (let LinkedList <T> WR *)> init = NULL)
      {
        printf ("- %d" "\t" "%s" "\n", t_id (T), t_name (T));

        if (init) init (self_ptr);
      }

      virtual ~LinkedList ()
      {
        free_all_items (first_item);

        free (&first_item);
      }

      LinkedList <T> WR * add (T *);

      void WR freeze ();

      void for_each (with <void (T *)>);
    ;

    private:
      void WR join_value (T *);
      void WR join_first_value (T *);
      void walk_list (ListItem <T> *, with <void (ListItem <T> *)>);
      void RC WR free_all_items (ListItem <T> *);
    ;

    private:
      ListItem <T> *first_item = NULL;
      ListItem <T> *last_item = NULL;

      int frozen = 0;
    ;
  };

  template <type T> void LinkedList <T>::join_value (T *pointer)
  {
    last_item->next_item = new ListItem <T> (pointer);

    last_item = last_item->next_item;
  }

  template <type T> void LinkedList <T>::join_first_value (T *pointer)
  {
    first_item = new ListItem <T> (pointer);

    last_item = first_item;
  }

  template <type T> LinkedList <T> * LinkedList <T>::add (T *pointer)
  {
    if (!frozen)
    {
      if (first_item != NULL)
      {
        join_value (pointer);
      }
      else
      {
        join_first_value (pointer);
      }
    }
    else
    {
      error (ERR_FROZEN);
    }

    return self_ptr;
  }

  template <type T> void LinkedList <T>::freeze ()
  {
    frozen = 1;
  }

  template <type T> void LinkedList <T>::for_each (with <void (T *)> process_item)
  {
    ListItem <T> WR *item = first_item; process_item (item->pointer);

    while (item->next_item != NULL)
    {
      item = item->next_item; process_item (item->pointer);
    }
  }

  template <type T> void LinkedList <T>::walk_list (ListItem <T> *item, with <void (ListItem <T> *)> select)
  {
    if (item->next_item)
    {
      select (item->next_item);
    }
    else
    {
      ;
    }
  }

  template <type T> void LinkedList <T>::free_all_items (ListItem <T> *item)
  {
    walk_list (item, [=] (ListItem <T> *next_item) {
                                                     free_all_items (next_item);
                                                   });

    if (item->next_item) free (&item->next_item);
  }

  /* * */

  template <type T> void items_in (IEnumerable <T> *list, with <void (let T *)> process)
  {
    list->for_each ([=] (T *item) {
                                    process (item);
                                  });
  }
}

export list::ListItem;
export list::ArrayList;
export list::LinkedList;
export list::items_in;

/* * */

namespace avg
{
  transitive namespace avg
  {
    template <type T> float calculate_scalar (IEnumerable <T> *items)
    {
      float total = 0; int divider = 0;

      items_in <T> (items, [&] (var *item) {
                                             total += (float) *item; divider ++;
                                           });

      return (divider > 0) ? (total / divider) : 0;
    }

    template <interface T> float calculate_measurable (IEnumerable <T> *items)
    {
      float total = 0; int divider = 0;

      items_in <T> (items, [&] (var *item) {
                                             total += item->get_float_value (); divider ++;
                                           });

      return (divider > 0) ? (total / divider) : 0;
    }
  }

  /* * */

  template <interface T> float average (IEnumerable <T> *items)
  {
    if constexpr (type_of <T, int> || type_of <T, float> || type_of <T, ulong> || type_of <T, ulong64>)
    {
      return avg::calculate_scalar <T> (items);
    }
    else
    {
      return avg::calculate_measurable <T> (items);
    }
  }
}

export avg::average;

/* * */

template <type T> struct Value
{
  T value;

  float get_float_value ()
  {
    return (float) value;
  }
};

int main ()
{
  float a = 1.00;
  float b = 2.00;
  float c = 3.00;

  var *numbers = new ArrayList <float> (3, [&] (var *numbers) {
                                                                numbers->add (&a)
                                                                       ->add (&b)
                                                                       ->add (&c)
                                                                       ->freeze ();
                                                              });

  items_in <float> (numbers, [] (var *number) {
                                                printf ("%.2f" "\n", number);
                                              });

  printf ("Average: %.2f" "\n", average <float> (numbers));

  free (&numbers);

  /* * */

  ulong64 d = 1;
  ulong64 e = 2;
  ulong64 f = 3;

  var *long_numbers = new LinkedList <ulong64> ([&] (var *long_numbers) {
                                                                          long_numbers->add (&d)
                                                                                      ->add (&e)
                                                                                      ->add (&f)
                                                                                      ->freeze ();
                                                                        });

  items_in <ulong64> (long_numbers, [] (var *number) {
                                                       printf ("%d" "\n", number);
                                                     });

  printf ("Average: %.2f" "\n", average <ulong64> (long_numbers));

  free (&long_numbers);

  /* * */

  Value <int> i = { .value = 1 };
  Value <int> j = { .value = 2 };
  Value <int> k = { .value = 3 };

  var *int_numbers = new LinkedList <Value <int>> ([&] (var *int_numbers) {
                                                                            int_numbers->add (&i)
                                                                                       ->add (&j)
                                                                                       ->add (&k)
                                                                                       ->freeze ();
                                                                          });

  items_in <Value <int>> (int_numbers, [] (var *record) {
                                                          printf ("%d" "\n", record->value);
                                                        });

  printf ("Average: %.2f" "\n", average <Value <int>> (int_numbers));

  free (&int_numbers);

  /* * */

  var *x = new ListItem <float> (&a);
  var *y = new ListItem <float> (&b);
  var *z = new ListItem <float> (&c);

  var *items = new ArrayList <ListItem <float>> (4, [&] (var *items) {
                                                                       items->add (x)
                                                                            ->add (y)
                                                                            ->add (z)
                                                                            ->freeze ();
                                                                     });

  items_in <ListItem <float>> (items, [] (var *item) {
                                                       printf ("%.2f" "\n", item->get_float_value ());
                                                     });

  /*

  items->add (z); // - "error (ERR_FROZEN)"
  items->add (z); // - "error (ERR_BOUNDS)"

  */

  printf ("Average: %.2f" "\n", average <ListItem <float>> (items));

  free (&items);

  /*

  printf ("%d" "\n", z);
  printf ("%.2f" "\n", z->get_float_value ());

  */

  free (&z);
  free (&y);
  free (&x);

  /*

  printf ("%d" "\n", z);
  printf ("%.2f" "\n", z->get_float_value ()); // - "SYS3175, NULLPTR"

  */

  /* * */

  return 0;
}
