#ifndef LINEA_H
#define LINEA_H
/*
 * This header file describes a C interface
 * to the Atari ST assembler level graphics routines.
 *
 * Each of the following linea*() functions consists of only a few
 * lines of assembler code which save registers, load parameters,
 * execute one of the unimplemented line 'a' instructions, restore
 * registers, and return.
 *
 * Most of the linea*() functions pass their parameters through the
 * la_data structure (described below) which is referenced through
 * the pointer in the la_init structure (also described below) which
 * is initialized by the linea0() function.
 *
 * The bitblt and sprite functions are exceptions to this rule.
 */
extern linea0();	/* () Initialize */
extern linea1();	/* () Put pixel */
extern linea2();	/* () Get pixel */
extern linea3();	/* () Draw line */
extern linea4();	/* () Horizontal line */
extern linea5();	/* () Filled rectangle */
extern linea6();	/* () Filled polygon */
extern linea7();	/* (struct la_blit *bp) BitBlt */
extern linea8();	/* () TextBlt */
extern linea9();	/* () Show mouse */
extern lineaa();	/* () Hide mouse */
extern lineab();	/* () Transform mouse */
extern lineac();	/* (char *sp) Undraw sprite */
extern linead();	/* (long x,y; char *p,*q) Draw sprite */
extern lineae();	/* () Copy raster form */
extern lineaf();	/* () Seedfill */

/*
 * Calling linea0() will fetch the following structure from
 * the system and store it in external memory.
 */
struct linea_init {
	/* Early versions of TOS don't implement all the possible functions */
	long li_d0;		/* 0 if old system bugs persist */
	struct la_data *li_a0;	/* linea data structure pointer */
	struct la_font **li_a1;	/* system font vector */
	long (*li_a2)();	/* linea function vector */
};

extern struct linea_init la_init;

/*
 * The system data structure for implementing graphics.
 * Most of this is simply here to confuse you.
 */
struct la_data {
	int	ld_vplanes;	/* number of video planes: 1, 2, or 4 */
	int	ld_vwrap;	/* number of bytes/video line */
	int	*ld_contrl;	/* pointer to CONTRL array */
	int	*ld_intin;	/* pointer to INTIN array */
	int	*ld_ptsin;	/* pointer to PTSIN array */
	int	*ld_intout;	/* pointer to INTOUT array */
	int	*ld_ptsout;	/* pointer to PTSOUT array */
	int	ld_colbit[4];	/* color bit-plane[i] value */
	int	ld_lstlin;	/* -1 */
	int	ld_lnmask;	/* line style mask */
	int	ld_wmode;	/* writing mode 0:replace, 1:transparent,
				 * 2:exclusive or, 3:inverse transparent */
	int	ld_x1;		/* x1 coordinate */
	int	ld_y1;		/* y1 coordinate */
	int	ld_x2;		/* x2 coordinate */
	int	ld_y2;		/* y2 coordinate */
	int	*ld_patptr;	/* fill pattern pointer */
	int	ld_patmsk;	/* fill pattern mask */
	int	ld_mfill;	/* multi-plane fill flag */
	int	ld_clip;	/* clipping flag */
	int	ld_xmincl;	/* minimum x clipping value */
	int	ld_ymincl;	/* minimum y clipping value */
	int	ld_xmaxcl;	/* maximum x clipping value */
	int	ld_ymaxcl;	/* maximum y clipping value */
	int	ld_xdda;	/* accumulator for textblt dda */
	int	ld_ddainc;	/* fixed point scale factor */
	int	ld_scaldir;	/* scale direction flag */
	int	ld_mono;	/* current font is monospaced */
	int	ld_srcx;	/* x coord of character in font */
	int	ld_srcy;	/* y coord of character in font */
	int	ld_dstx;	/* x coord of character on screen */
	int	ld_dsty;	/* y coord of character on screen */
	int	ld_delx;	/* width of character */
	int	ld_dely;	/* height of character */
	int	*ld_fbase;	/* pointer to start of font form */
	int	ld_fwidth;	/* width of font form */
	int	ld_style;	/* textblt special effects flags */
	int	ld_litemsk;	/* lightening mask */
	int	ld_skewmsk;	/* skewing mask */
	int	ld_weight;	/* thickening factor */
	int	ld_roff;	/* skew offset above baseline */
	int	ld_loff;	/* skew offset below baseline */
	int	ld_scale;	/* scaling flag */
	int	ld_chup;	/* character rotation vector */
	int	ld_textfg;	/* text foreground color */
	int	*ld_scrtchp;	/* text special effects buffer */
	int	ld_scrpt2;	/* offset to scaling buffer from above */
	int	ld_textbg;	/* text background color */
	int	ld_copytran;	/* copy raster form type flag */
	int	(*ld_seedabort)();	/* seedfill end detect function */
};

/*
 * Macro definitions using la_init and la_data to make
 * MACRO names as used in the linea.doc description.
 */
#define VPLANES	la_init.li_a0->ld_vplanes
#define VWRAP	la_init.li_a0->ld_vwrap
#define CONTRL	la_init.li_a0->ld_contrl
#define INTIN	la_init.li_a0->ld_intin
#define PTSIN	la_init.li_a0->ld_ptsin
#define INTOUT	la_init.li_a0->ld_intout
#define PTSOUT	la_init.li_a0->ld_ptsout
#define COLBIT0	la_init.li_a0->ld_colbit[0]
#define COLBIT1	la_init.li_a0->ld_colbit[1]
#define COLBIT2	la_init.li_a0->ld_colbit[2]
#define COLBIT3	la_init.li_a0->ld_colbit[3]
#define LSTLIN	la_init.li_a0->ld_lstlin
#define LNMASK	la_init.li_a0->ld_lnmask
#define WMODE	la_init.li_a0->ld_wmode
#define X1	la_init.li_a0->ld_x1
#define Y1	la_init.li_a0->ld_y1
#define X2	la_init.li_a0->ld_x2
#define Y2	la_init.li_a0->ld_y2
#define PATPTR	la_init.li_a0->ld_patptr
#define PATMSK	la_init.li_a0->ld_patmsk
#define MFILL	la_init.li_a0->ld_mfill
#define CLIP	la_init.li_a0->ld_clip
#define XMINCL	la_init.li_a0->ld_xmincl
#define YMINCL	la_init.li_a0->ld_ymincl
#define XMAXCL	la_init.li_a0->ld_xmaxcl
#define YMAXCL	la_init.li_a0->ld_ymaxcl
#define XDDA	la_init.li_a0->ld_xdda
#define DDAINC	la_init.li_a0->ld_ddainc
#define SCALDIR	la_init.li_a0->ld_scaldir
#define MONO	la_init.li_a0->ld_mono
#define SRCX	la_init.li_a0->ld_srcx
#define SRCY	la_init.li_a0->ld_srcy
#define DSTX	la_init.li_a0->ld_dstx
#define DSTY	la_init.li_a0->ld_dsty
#define DELX	la_init.li_a0->ld_delx
#define DELY	la_init.li_a0->ld_dely
#define FBASE	la_init.li_a0->ld_fbase
#define FWIDTH	la_init.li_a0->ld_fwidth
#define STYLE	la_init.li_a0->ld_style
#define LITEMSK	la_init.li_a0->ld_litemsk
#define SKEWMSK	la_init.li_a0->ld_skewmsk
#define WEIGHT	la_init.li_a0->ld_weight
#define ROFF	la_init.li_a0->ld_roff
#define LOFF	la_init.li_a0->ld_loff
#define SCALE	la_init.li_a0->ld_scale
#define CHUP	la_init.li_a0->ld_chup
#define TEXTFG	la_init.li_a0->ld_textfg
#define SCRTCHP	la_init.li_a0->ld_scrtchp
#define SCRPT2	la_init.li_a0->ld_scrpt2
#define TEXTBG	la_init.li_a0->ld_textbg
#define COPYTRAN	la_init.li_a0->ld_copytran
#define SEEDABORT	la_init.li_a0->ld_seedabort

/*
 * Macros for the most reasonable linea functions.
 * All the rest require a long discussion.
 */
#define putpixel(x,y,v)	((PTSIN[0]=x),(PTSIN[1]=y),(INTIN[0]=v),linea1())
#define getpixel(x,y)	((PTSIN[0]=x),(PTSIN[1]=y),linea2())
#define showmouse()	(linea9())
#define hidemouse()	(lineaa())

/*
 * linea7() = BitBlt parameter block.
 * Source and destination description blocks
 */
struct la_blk {
	int	bl_xmin;		/* Minimum x */
	int	bl_ymin;		/* Minimum y */
	int	*bl_form;		/* Word aligned memory form */
	int	bl_nxwd;		/* Offset to next word in line */
	int 	bl_nxln;		/* Offset to next line in plane */
	int 	bl_nxpl;		/* Offset to next plane */
};

struct la_blit {
	int	bb_b_wd;		/* width of block in pixels */
	int	bb_b_ht;		/* height of block in pixels */
	int	bb_plane_ct;		/* number of planes to blit */
	int	bb_fg_col;		/* foreground color */
	int	bb_bg_col;		/* background color */
	char	bb_op_tab[4];		/* logic for fg x bg combination */
	struct la_blk bb_s;		/* source info block */
	struct la_blk bb_d;		/* destination info block */
	int	*bb_p_addr;		/* pattern buffer address */
	int	bb_p_nxln;		/* offset to next line in pattern */
	int	bb_p_nxpl;		/* offset to next plane in pattern */
	int	bb_p_mask;		/* pattern index mask */
	char	bb_fill[24];		/* work space */
};
/* Offsets to next word in plane */
#define HIGH_NXWD	2
#define MED_NXWD	4
#define LOW_NXWD	8
/* Scan line widths of the screen */
#define HIG
#define MED_NXLN	160
#define LOW_NXLN	160
/* Offsets between planes */
#define NXPL		2

/*
 * linead() = Draw sprite, sprite definition block.
 *	sprite save block is 10+VPLANES*64 bytes big.
 */
struct la_sprite {
	int	ls_xhot;	/* x hot spot offset */
	int	ls_yhot;	/* y hot spot offset */
	int	ls_form;	/* 1 for VDI, -1 for XOR */
	int	ls_bgcol;	/* background color index */
	int	ls_fgcol;	/* foreground color index */
	int	ls_image[32];	/* background/foreground interleaved image */
};

/*
 * Font Header. Taken from GEM VDI Programmers Guide G-2
 */

struct la_font {
	int	font_id;	/* face identifier */
	int	font_size;	/* font size in points */
	char	font_name[32];	/* face name */
	int	font_low_ade;	/* lowest ascii value in face */
	int	font_hi_ade;	/* highest ascii value in face */
		/* distance measured from char baseline */
	int	font_top_dst;	/* top line distance */
	int	font_ascent_dst; /* ascent line distance */
	int	font_half_dst;	/* half line distance */
	int	font_descent_dst; /* descent line distance */
	int	font_bottom_dist; /* bottom line distance */
	int	font_fatest;	/* width of widest char in font */
	int	font_fat_cell;	/* width of widest char cell in font */
	int	font_left_off;	/* left offset */
	int	font_right_off;	/* right offset */
	int	font_thickening; /* number of pixles to widen chars */
	int	font_underline;	/* width in pixles of underline */
	int	font_lightening; /* mask used to drop pixles out */
	int	font_skewing;	/* mask used to determine skewing */
	unsigned default_font:1; /* set if default system font */
	unsigned horiz_ofset:1;	/* horizontal ofset tables should be used */
	unsigned byte_swap:1;	/* byte swap flag */
	unsigned mono_space:1;	/* mono spaced font */
	int	*font_horiz_off; /* pointer to horizontal offset table */
	int	*font_char_off;	/* pointer to char offset table */
	char	*font_data;	/* pointer to font data */
	int	font_width;	/* font width */
	int	font_height;	/* font height */
	char	*font_next;	/* pointer to next font */
};
#endif
