	title	"PC AT/PS/2 to XT keycode translator."
	list	p=12F629
	radix	dec
;	 noexpand			 ; I like neat listings

	include "p12f629.inc"
	include "maclib.inc"

	org	2100h			; EPROM area
	DE	"[ATXTKEY - eXtended RT 0.1d]"


;*	AT/PS2 to XT Keyboard Translator.
;	---------------------------------
;
;	Version 0.90  Chuck Guzis, June, 2009
;	    -- initial working release
;	Version 0.92, September, 2009
;	    -- cleans up host-to-keyboard timing a bit
;	Version 0.93, August, 2010
;	     -- Change to pass auxiliary code 0xE0 through
;		to host without translation.
;	Version 0.94, January, 2011
;	     -- Changed translation table length to 144 bytes to handle
;		erroneous F7 (scan code 83h) conversion.  Main routine
;		modified to perform threshold check instead of bit 7 check
;		to filter special codes.
;	eXtended RT 0.1d, June, 2015
;		-- many little changes:
;			- able to pass through 0xE1 too
;			- optimized timing with oscilloscope and added missing edge
;			- filter leading 0xE0/0xE1 if invalid byte follows
;			- make SysReq behave more realistic
;			- LockKey-LEDs no longer affected by CTRL or BREAK
;			- fixed buffer to do something usefull (rKeyQIn -> FSR)
;			- killed several interrupt problems
;			- soft-reset checking in SendXTByte routine
;			- ...
;
;	Copyright by Charles P. Guzis, all rights reserved.
;
;	Chuck Guzis retains all rights to this code, but gives permission
;	to use and modify it for non-commercial, non-profit applications,
;	provided that this copyright notice is reproduced in its entirety.
;
;	This translator uses a small PIC12F629 microcontroller to translate
;	an AT or PS/2 keyboard interface to that of the PC XT.	Shift-lock
;	status is noted and the keyboard LEDs are illuminated accordingly.
;
;	The electrical interface to the 12F629 is very simple:
;
;	Pin 1 = +5 supply (should have a 47 uF decoupling capacitor to Gnd
;	Pin 2 = Clock from the PC XT host
;	Pin 3 = Data from the PC XT host
;	Pin 4 = unused, for HW-Reset by Host
;	Pin 5 = AT keyboard clock  (pulled high by 4.7K to +5)
;	Pin 6 = AT keyboard data   (pulled high by 4.7K to +5)
;	Pin 7 = AT clock pulldown (a BAT85 diode is connected between
;		Pin 7 (cathode end) and pin 5 (anode end)
;	Pin 8 = Gnd
;
;	The diode isn't critical, but low-voltage drop will give better
;	signal-to-noise ratio.  It's there to make GP0 behave as an open-
;	drain driver.
;
;	The XT reset line is not used, as the XT BIOS asserts reset by
;	holding the host clock low for at least 20 mS.
;
;	Just so you don't have to look it up, here are the pinouts
;	for both the XT (5 pin DIN) and PS/2 (6 pin mini-DIN) conectors:
;
;	PS/2 connector:
;		1 - Data
;		3 - Gnd
;		4 - +5
;		5 - Clock
;
;	XT Connector:
;		1 - Clock
;		2 - Data
;		3 - Reset, not used
;		4 - Gnd
;		5 - +5
;
;	A 32-byte buffer is implemented for received keyboard data.
;	This is actually twice the size of the AT keyboard buffer.
;

	__CONFIG	_MCLRE_OFF & _CP_OFF & _WDT_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT  ;Internal osc.
	subtitle	"Symbol definitions"
	page

;	Our GPIO bit assignments by position.

bnKeyClockPD	equ	0	; AT key pulldown (GPIO 0, pin 7)
bnKeyData		equ	1	; AT keyboard data (GPIO 1, pin 6)
bnKeyClock		equ	2	; AT Keyboard clock (GPIO 2, pin 5)
bnPassE0E1		equ	3	; MCLEAR, unused (GPIO 3, pin 4)
bnHostData		equ	4	; XT host data	(GPIO 4, pin 3)
bnHostClock		equ	5	; XT host clock	(GPIO 5, pin 2)

;	Same as the above, but by bit values.

bvKeyClockPD	equ	0<<bnKeyClockPD ; AT key pulldown (GPIO 0, pin 7)
bvKeyData		equ	1<<bnKeyData	; AT keyboard data (GPIO 1, pin 6)
bvKeyClock		equ	1<<bnKeyClock	; AT keyboard clock (GPIO 2, pin 5)
bvPassE0E1		equ	1<<bnPassE0E1	; MCLEAR, unused (GPIO 3, pin 4)
bvHostData		equ	1<<bnHostData	; XT host data (GPIO 4, pin 3)
bvHostClock		equ	1<<bnHostClock	; XT host clock (GPIO 5, pin 2)

;	Bits relating to keyboard state in rATFlags.
;
;	Observe that the LED (shift lock) keys are in the precise
;	position that the AT keyboard ED command expects them.

bnATScrollLock	equ	0	; scroll lock
bnATNumLock		equ	1	; numeric lock
bnATCapsLock	equ	2	; caps lock

bvATScrollLock	equ	1<<bnATScrollLock	; scroll lock
bvATNumLock		equ	1<<bnATNumLock		; numeric lock
bvATCapsLock	equ	1<<bnATCapsLock		; caps are locked

;	Register holding flags for the "send byte" code in the
;	interrupt service code.

bnATSendReq		equ	0	; if set, there's a character in
						; rATSendByte that needs to go
bnACKSeen		equ	1	; ACK isn't put into the queue;
						; rather when received, this flag
						; is set.
bnATXTCTRL		equ	2	; remember if CTRL-Key active
					
;	Register holding flags for delayed transmission.
;	Should be suppress if invalid bytes follow.

bnXTE0			equ 0
bnXTE1			equ 1
bnXTF0			equ	2
					
;	Input queue area starts at 0x40 and goes through 0x5f

QUEUE_SIZE		equ	32		; cells to use in queue

;	XT scancodes for special keys.

XTSC_CAPSLOCK	equ	0x3a	; Caps lock released
XTSC_SCRLOCK	equ	0x46	; Scroll lock released
XTSC_NUMLOCK	equ	0x45	; Numeric lock released
XTSC_OKAY		equ	0xaa	; Scan code for "Diagnostics Passed"

;	AT command codes.

ATSC_SETLED		equ	0xed	; set LEDs
ATSC_RESET		equ	0xff	; reset
ATSC_ACK		equ	0xfa	; acknowledge
ATSC_OK			equ	0xaa	; diagnostics okay


	subtitle	"Data area (register) layout"
	page

;	Register file definitions.

	cblock	0x20

	rXTByte		:1		; byte to send - used by SendXTByte
	rXTBitCount	:1		; bits left to send
	rXTdelayed	:1		; handler for delayed bytes
	rXTlast		:1		; last byte send
	rXTlast2	:1		; 2nd last byte send
	rXTlast3	:1		; 3rd last byte send
	rXTlast4	:1		; 4th last byte send

	rATChar		:1		; character returned by ReadKey
	rATFlags	:1		; flags set by AT codes

	rTemp1		:1		; Scratch reg 1

;	Interrupt routine registers.

	rIntSaveW	:1		; Savearea for W
	rIntSaveS	:1		; Savearea for status word
	rIntKCount	:1		; keyboard data bit counter
	rIntKData	:1		; keyboard code accumulator
	rIntTemp	:1		; temporary for the ISR

;	Code shared by interrupt and mainline routines.
;	Be careful to observe who reads and writes to avoid race conditions!

	rATSendByte		:1	; byte to send to keyboard
	rATSendFlags	:1	; flags for the send routine

;	Keyboard buffer pointers.

	rKeyQIn		:1		; in
	rKeyQOut	:1		; out

	endc

;	The keyboard buffer.

	cblock	0x40

	rKeyQFirst	:QUEUE_SIZE	; circular queue

	endc

;*	Boot entry.
;	==========


Start	org	0x0000		;program starts at location 000
	call	SetUp
	goto	Main		; start the routine

	subtitle	"Interrupt servicer"
	page

;	Interrupt servicing routine.
;	----------------------------
;
;	We interrupt on the AT keyboard clock bit high-to-low transition and nothing else.
;

	org	0x0004

	proc	IntServ
	movwf	rIntSaveW
	swapf	STATUS,w			; movfw changes the Z status so can't use
    bank0						; rIntSaveS in Bank 0
	movwf	rIntSaveS			; save the status word
	
;	When we come in here, we've triggered on a high-to-low transition of the
;	AT keyboard clock.  If the keyboard is sending data to us, the clock
;	will be low and the data line will be low also (start bit).  If we're
;	sending data to the keyboard, the data line will be low because we
;	pulled it low.

	btfss	GPIO,bnKeyData		; if keyboard data high, ignore
	btfsc	GPIO,bnKeyClock		; if keyboard clock high, ignore
	goto	IntServExit			; ignore the interrupt

;	See if we're asking to send a byte.  If we get a collision,
;	(a byte coming in when we're trying to send), it's not fatal.

	bbs		rATSendFlags,bnATSendReq,IntServ20	; if sending

	movif	8,rIntKCount		; bit counter
	clrc						; clear carry
	clrf	rIntKData			; clear accumulated bytes

;	Wait for the clock to transit high, then low (skip the start bit).

IntServ2:
	stallc	GPIO,bnKeyClock		; wait for clock to  rise
	stalls	GPIO,bnKeyClock		; and wait for clock to fall again

;	Get the data bits

	btfsc	GPIO,bnKeyData		; skip if a 0 bit
	setc						; set a 1 bit
	rrf		rIntKData,f			; the keyboard data bits
	dbnz	rIntKCount,IntServ2	; for next bit

;	We still have two bits to go, parity and stop.	We skip them.

	stallc	GPIO,bnKeyClock		; wait for clock to rise
	stalls	GPIO,bnKeyClock		; and fall again  (skip parity bit)
	stallc	GPIO,bnKeyClock		; wait for clock rise
	stalls	GPIO,bnKeyClock		; and fall again (skip stop bit)

;	We special-case ACK.  Because it occurs after a command has been
;	sent, we can't put it into the queue or the routine sending the
;	command may never see it.

	movfw	rIntKData			; get data just received
	xorlw	ATSC_ACK			; see if ACK
	bnz		IntServ14			; skip.
	bsf		rATSendFlags,bnACKSeen	; say we saw it
	goto	IntServExit			; don't put it in the queue

;	Now, add the assembled byte to the buffer.  Note that if IN+1 == OUT
;	there's no room in the buffer, so we toss the keystroke.

IntServ14:
	incf	rKeyQIn,w			; advance
	andlw	0x5f				; enforce a wrap-around
	movwf	rIntTemp			; save it
	sublw	rKeyQOut			; see if equal
	bz		IntServExit			; if queue full, drop the character

;	N.B.:  If we see that we have problems with overflow, we can drive the
;	keyboard clock low and essentially shut things off until we have a
;	chance to empty the queue.

	movfw	rKeyQIn
	movwf	FSR
	movff	rIntKData,INDF		; store the byte
	movff	rIntTemp,rKeyQIn	; update the "in" pointer
	goto	IntServExit			; all done

;	This is where we send code to the AT Keyboard.
;	At entry here, the clock line has just been released and
;	the edge of the keyboard-generated clock brings us here.

IntServ20:
	bcf		rATSendFlags,bnACKSeen	; say we haven't seen ACK
	bcf		rATSendFlags,bnATSendReq	; clear request flag
	movif	8,rIntKCount		; how many bits to send out
	clrf	rIntTemp			; parity counter
IntServ22:
	rrf		rATSendByte,f		; get a bit
	bc		IntServ24			; if a 1
	bcf		GPIO,bnKeyData		; set a zero
	goto	IntServ26			; keep going

IntServ24:
	bsf		GPIO,bnKeyData		; set a 1
	incf	rIntTemp,f			; count parity
IntServ26:
	stallc	GPIO,bnKeyClock		; wait for clock to rise

;	The keyboard samples the bit while the clock is high.

	stalls	GPIO,bnKeyClock		; wait for clock to fall
	nop
	nop
	nop
	nop							; kill a few microseconds
	dbnz	rIntKCount,IntServ22	; for next bit

;	We need to add the parity bit.	If we have an even number of
;	1 bits in the data stream, we add a 1 bit, otherwise, a zero.

	btfsc	rIntTemp,0			; skip if even number of bits
	bcf		GPIO,bnKeyData		; add 1
	btfss	rIntTemp,0			; skip if odd number of bits
	bsf		GPIO,bnKeyData		; add 0
	stallc	GPIO,bnKeyClock		; wait for clock to rise

;	The parity bit is sampled here.

	stalls	GPIO,bnKeyClock		; wait for clock to fall

;	Now, float the data line.  The keyboard will drop the data
;	line after the clock rises to signal acceptance.

	nop
	nop							; kill 2 uS

	bank1
	bsf		TRISIO,bnKeyData	; set data line to input
	bank0

	stallc	GPIO,bnKeyClock		; wait for high clock

;	Wait for the device to bring data low.

IntServ32:
	bbs		GPIO,bnKeyData,IntServ32	; stall until data low

;	Data is low, wait for clock to fall.

IntServ34:
	bbs		GPIO,bnKeyClock,IntServ34	; wait around

;	Now wait for a high clock.

IntServ36:
	bbc		GPIO,bnKeyClock,IntServ36	; wait for rising clock

;	The keyboard responded as expected, if we get here.
;	Else next key pressed must finish this. No timeout added, reprogramming of hardware-counter
;	in interrupt routine could cause more trouble.

IntServ40:

;	Return from interrupt.

IntServExit:
	bcf		INTCON,INTF			; clear the old interrupt
	swapf	rIntSaveS,w
	movwf	STATUS				; restore status
	swapf	rIntSaveW,f
	swapf	rIntSaveW,w
	
	retfie						; return and re-enable interrupts
	endproc IntServ


	subtitle	"Read/write key code routines"
	page

;*	ReadKey - Read a key from the queue.
;	------------------------------------
;
;	If there are no keystrokes in the buffer, stall until
;	some arrive.  Returns next keystroke in rATChar and Z clear,
;	or Z set if no character.
;

	proc	ReadKey
	
	movfw	rKeyQOut
	xorwf	rKeyQIn,w
	bz		ReadKey4			; return Z set if empty

;	Retrieve the keystroke.

	movfw	rKeyQOut
	bcf		INTCON,GIE			; disable interrupts
	movwf	FSR					; prepare for fetching
	movff	INDF,rATChar		; get the key
	bsf		INTCON,GIE			; enable interrupts
	
	incf	rKeyQOut,w			; advance
	andlw	0x5f				; wrap-around
	movwf	rKeyQOut			; update pointer
	
	clrz						; say we have one

ReadKey4:
	return
	endproc ReadKey

;*     SendXTBit - Send a bit to the XT keyboard port.
;      -----------------------------------------------
;
;	Note that bits are clocked by the XT on the high-to-low
;	transition of the clock. On entry, the data bit is in C (carry).
;

	proc	SendXTBit
	skpnc
	bsf		GPIO,bnHostData		; if send a 1
	skpc
	bcf		GPIO,bnHostData		; if send a 0
	nop
	bsf		GPIO,bnHostClock	; raise clock
	movlw	48/4				; to reach 66 usec
	call	ShortDelay
	bcf		GPIO,bnHostClock	; drop the clock line
	
	return
	endproc SendXTBit

;*	SendXTByte - Send a byte to the XT.
;	-----------------------------------
;
;	Inserts a start bit, waits for XTClock and XTData to go high before
;	sending.
;
;	Byte to send is in rXTByte.
;
;	We need to frame things accurately, so data bit transitions are made
;	in the center of the clock-low bit cell.
;
;	The XT interface is very simple.  Bits are shifted into an 74LS322 shift
;	register.  When the start bit reaches the high-order bit of the shift
;	register, it sets an interrupt, disables shifting and pulls data low one
;	shift time later.  In other words, the start bit is shifted out.
;	The bit order is LSB first for 8 data bits.  After that, the interface
;	is essentially blind until the PC has picked up the character.
;
;	Many keyboards start out by strobing a low "pseudo-start"  bit, then
;	follow with a high "real start".  Apparently, this results in more
;	stable performance.  We'll do the same thing.
;
;	Clock is held low by the PC as an inhibit--the keyboard should (and
;	cannot) send data until both data and clock lines have been allowed
;	to go high.
;
;	The actual data rate only needs to be slower than the processor clock,
;	as a conditioning circuit formed by two flip-flops is clocked from
;	that to avoid any problems that might arise with ringing.
;

	proc	SendXTByte
	
	call	CheckLockKey		; check for lock key here
	
;	At this point sending E0/E1 will be skipped if necessary.
	
	bbs		GPIO,bnPassE0E1,SendXTByte2	; see if we need to pass E0/E1 code
	movfw	rXTByte
	xorlw	0xe0				; check for E0
	skpz
	xorlw	0xe0 ^ 0xe1			; restore and check for E1
	skpnz
	goto	SendXTByte6			; exit
	
;	Wait for high level on XTClock--and XTData.

SendXTByte2:
	call	PollHost			; check if clock low and reset if too long
	btfss	GPIO,bnHostData		; loop if data low
	goto	SendXTByte2

;	Put XTClock and XTData into output mode with clock and data high.

	bsf		GPIO,bnHostClock
	bsf		GPIO,bnHostData
	bank1
	bcf		TRISIO,bnHostClock	; enable clock output
	bcf		TRISIO,bnHostData	; enable data output
	bank0
	
;	up from here timing is important, so disable interrupts

	bcf		INTCON,GIE			; disable interrupts
	bcf		GPIO,bnKeyClockPD	; pull the clock low

;	Start off with a 0 bit, then a 1 bit.

	clrc
	call	SendXTBit
	movlw	0/4					; to reach >5 usec
	call	ShortDelay
	bsf		GPIO,bnHostData
	movlw	68/4				; reach 120 usec
	call	ShortDelay
	setc
	call	SendXTBit

;	Now, send the remaining 8 bits of the scan code.

	movif	8,rXTBitCount		; number of bits to send
	
SendXTByte4:
	movlw	0/4					; reach 30 usec
	call	ShortDelay
	rrf		rXTByte,f
	call	SendXTBit			; send, starting with the LSbit
	dbnz	rXTBitCount,SendXTByte4	; loop

	movlw	8/4					; reach 30 usec
	call	ShortDelay
	bsf		GPIO,bnHostClock	; raise clock

;	At this point, the PC will be driving data low until the interrupt
;	has been serviced, so set data and clock to input.

	bank1
	bsf		TRISIO,bnHostData	; disable data
	bsf		TRISIO,bnHostClock	; disable clock output
	bank0
	
;	Lets have a brake. Some XT-relatives are unable to pull clock down.
;	Don't want to destroy content of arduous filled shift-register, while it is read.

	clrf	TMR1L
	movif	(256-3),TMR1H		; set >10 msec
	bcf		PIR1,TMR1IF			; clear overflow flag
	
;	Enable interrupts again, but not before timer is running.

	bcf		INTCON,INTF			; clear the self-made interrupt
	bsf		INTCON,GIE			; enable interrupts
	bsf		GPIO,bnKeyClockPD	; let keyboard clock float high	
	
SendXTByte5:
	bbc		PIR1,TMR1IF,SendXTByte5	; loop until overflow

SendXTByte6:
	return
	endproc SendXTByte

;*	SendATByte - Send a byte to the keyboard.
;	-----------------------------------------
;
;	This is tricky.	 We have to take the AT keybaord
;	clock low for more than 60 uS, drop the data line, then
;	let the clock go.
;
;	The interrupt routine will pick up when the next clock hits,
;	as long as rATSendByte is nonzero--whence our kludge.
;
;	W has the data we want to send.
;

	proc	SendATByte

	movwf	rATSendByte			; save it for the interrupt routine
	bcf		INTCON,GIE			; disable interrupt
	bcf		GPIO,bnKeyClockPD	; pull the clock low

;	Stall with the clock low to prevent keyboard from starting a new
;	character.  We also have to mask the clock interrupt, since we're
;	pulling the clock low.

	movlw	100/4
	call	ShortDelay			; stall for >100 uS

	bank1
	bcf		TRISIO,bnKeyData	; set key data to output
	bank0

	bcf		GPIO,bnKeyData		; bring data low
	movlw	72/4				; more than 1 bit time
	call	ShortDelay
	bcf		rATSendFlags,bnACKSeen	; clear the "ACK seen"	flag
	bsf		rATSendFlags,bnATSendReq	; say we have a byte
	
;	We stall until the byte gets sent.  We wait for an ACK--if we
;	don't get it within 8 milliseconds, we proceed as if we got it
;	anyway.

	clrf	TMR1L
	movif	(256-2),TMR1H		; set about 8 msec
	bcf		PIR1,TMR1IF			; clear overflow flag
	
;	Enable interrupts again, but not before timer is running.
	
	bcf		INTCON,INTF			; clear the self-made interrupt
	bsf		INTCON,GIE			; enable interrupt
	bsf		GPIO,bnKeyClockPD	; let keyboard clock float high
	
SendATByte4:
	bbs		PIR1,TMR1IF,SendATByte8	; leave if timeout
	bbc		rATSendFlags,bnACKSeen,SendATByte4	; loop until ACK set

;	We timed out here, or we got an ACK. so we set the data to input.

SendATByte8:
	bank1
	bsf		TRISIO,bnKeyData	; set key data back to input
	bank0
	
	return						; all done
	endproc SendATByte

	subtitle	"Miscellaneous routines."
	page

;*	ShortDelay - Delay less than 1 msec.
;	------------------------------------
;
;	On entry, W has delay time in about 4 microsecond units.
;	Resolution is about 4-8 uS.
;

	proc	ShortDelay

	sublw	255
	movwf	TMR0
	bcf		INTCON,T0IF		; clear interrupt flag
	stallc	INTCON,T0IF		; stall until interrupt set
	
	return
	endproc ShortDelay


;*	CheckLockKey - Check for Caps, Num and Scroll Lock.
;	---------------------------------------------------
;
;	Just toggles the state of the LEDs and builds some history. The XT scan code (make)
;	is in W.
;

	proc	CheckLockKey

;	Because BREAK is realised as CTRL+ScrollLock and PAUSE as
;	CTRL+NumLock on old XT, we've got to do some filtering.
	
	movfw	rXTByte
	xorlw	0x1d			; check for CTRL, BREAK, CTRL+BREAK
	skpnz
	bsf		rATSendFlags,bnATXTCTRL	; remember state of CTRL/BREAK
	xorlw	0x1d ^ 0x9d				; restore byte and check if released
	skpnz
	bcf		rATSendFlags,bnATXTCTRL	; clear if released
	bbs		rATSendFlags,bnATXTCTRL,CheckLockKey10
	
	movlw	XTSC_CAPSLOCK
	xorwf	rXTByte,w
	bnz		CheckLockKey2	; if not capslock
	movlw	bvATCapsLock
	xorwf	rATFlags,f		; toggle caps lock
	goto	UpdateLED		; go update LED status

CheckLockKey2:
	movlw	XTSC_SCRLOCK
	xorwf	rXTByte,w
	bnz		CheckLockKey4	; if not scroll lock
	movlw	bvATScrollLock
	xorwf	rATFlags,f		; toggle scroll lock
	goto	UpdateLED		; update the LED status

CheckLockKey4:
	movlw	XTSC_NUMLOCK
	xorwf	rXTByte,w
	bnz		CheckLockKey10	; if not numlock
	movlw	bvATNumLock
	xorwf	rATFlags,f		; toggle num lock
	goto	UpdateLED		; update the LED status

;	Let us keep some history, will be usefull for other routines too.

CheckLockKey10:
	movfw	rXTlast3
	movwf	rXTlast4		; remember 4th last byte for next cycle	
	movfw	rXTlast2
	movwf	rXTlast3		; remember 3rd last byte
	movfw	rXTlast
	movwf	rXTlast2		; remember 2nd last byte
	movfw	rXTByte
	movwf	rXTlast			; remember last byte
	
	return					; all done
	endproc CheckLockKey

;*	UpdateLED - Update the LEDs.
;	---------------------------
;
;	Done all at once.
;

	proc	UpdateLED
	movlw	ATSC_SETLED		; set/reset LED
	call	SendATByte		; send the command
	movfw	rATFlags		; get LED status
	andlw	7				; isolate
	call	SendATByte		; return state
	
	return					; exit
	endproc UpdateLED


;*	PollHost - See what the XT is doing.
;	------------------------------------
;
;	At some point, the XT host will drop the keyboard clock
;	line for over 20 ms. When this happens, we need to
;	simulate a keyboard reset.
;
;	We reset the AT keyboard and re-initialize PIC.
;

	proc	PollHost
	bbs		GPIO,bnHostClock,PollHost2	; exit if clock high

;	Clock has gone low; time it.

	bcf		INTCON,GIE				; disable interrupts
	clrf	TMR1L
	movif	(256-5),TMR1H			; set about 20 msec
	bcf		PIR1,TMR1IF				; clear overflow flag
	bsf		INTCON,GIE				; enable interrupts
		
	stallc	GPIO,bnHostClock		; stall until XT Clock high
 
;	Clock is high again--see for how long.

	bbc		PIR1,TMR1IF,PollHost2	; not long enough, just exit

;	Reset the AT keyboard.

	movlw	ATSC_RESET
	call	SendATByte				; send a reset to the AT keyboard
	
;	Ignore interrupt, mute AT-keybaord and jump into water.

	bcf		INTCON,GIE				; disable interrupts
	bcf		GPIO,bnKeyClockPD		; pull the clock low
	goto	Start
		
PollHost2:
	return
	endproc PollHost

	subtitle	"Scan code translation"
	page

;*	Keyboard translation lookup.
;
;	We put the translation subroutine here, as there's no chance of
;	crossing a page boundary.

	include "xttrans.inc"			; keycode translator

	subtitle	"Main routine"
	page

;*	Main Loop.
;	----------
;
;	Here's the control flow:
;
;	a.  Check the host--has the clock been pulled low for more than
;	    20 mS.?  If so, issue a reset to the AT keyboard and re-init.
;	b.  See if an AT keystroke is available. If so, look at it.
;		Note the occurrence of E0, E1, F0 (key up) and other special
;		codes.  Translate to something usefull.
;	c.  Send the keystroke (with key-up modifier) to the host.
;	e.  Go to (a)
;

	proc	Main

Main2:
	call	PollHost		; poll the host interface

;	Main loop.  Sit around and cycle until there's data in the
;	AT keyboard receive buffer.

Main4:
	call	ReadKey			; read something from the buffer
	bz		Main2			; no data
	
	movfw	rATChar			; get the key again
	addlw	256-KEY_TABLE_LENGTH	; check to see if less than 90h
	bnc		Main5			; if less
	
	movfw	rXTdelayed
	movwf	rTemp1			; save the flags
	
	movfw	rATChar			; get the byte again
	xorlw	0xe0			; check for E0
	skpnz
	bsf		rXTdelayed,bnXTE0
	xorlw	0xe0 ^ 0xe1		; restore and check for E1
	skpnz
	bsf		rXTdelayed,bnXTE1
	xorlw	0xe1 ^ 0xf0		; restore and check for F0
	skpnz
	bsf		rXTdelayed,bnXTF0	; a key is released
	xorlw	0xf0 ^ ATSC_OK	; restore and check for AA
	skpnz
	call	UpdateLED		; kbd reconnected, update LEDs
	
	movfw	rXTdelayed
	xorwf	rTemp1,w		; compare with old flags
	skpnz
	clrf	rXTdelayed		; discard if nothing changed
	
	goto	Main4			; get next keystroke

;	Something resembling a normal keycode has arrived.

Main5:
	movfw	rATChar
	call	TransATXT		; translate
	iorlw	0				; check for zero
	bz		Main10			; skip
	movwf	rTemp1			; save the code

	bbc		rXTdelayed,bnXTE0,Main7
	movif	0xe0,rXTByte
	call	SendXTByte		; send missing E0
	
Main7:
	bbc		rXTdelayed,bnXTE1,Main8
	movif	0xe1,rXTByte
	call	SendXTByte		; send missing E1

; Next we try to fix behaviour of [SYSReq]-Key, because AT-keyboards report an active
; [ALT] or [ALTGR] in front. Unfortunately this can't be delayed. 

Main8:
	movlw	0x54
	xorwf	rTemp1,w
	bnz		Main9			; skip if no SysRequest
	bbs		rXTdelayed,bnXTF0,Main10	; jump if released one
	movlw	0xe0
	movwf	rXTByte
	xorwf	rXTlast2,w		; with ALTGR?
	skpnz					; yes?
	call	SendXTByte		; send it
	movif	0xB8,rXTByte
	call	SendXTByte
	movif	0x54,rXTByte
	call	SendXTByte
	movif	0xD4,rXTByte
	call	SendXTByte
	movlw	0xe0
	movwf	rXTByte
	xorwf	rXTlast4,w
	skpnz
	call	SendXTByte
	movif	0x38,rXTByte
	call	SendXTByte
	goto	Main10

Main9:
	movf	rTemp1,w
	btfsc	rXTdelayed,bnXTF0
	iorlw	128				; add the release flag	
	movwf	rXTByte
	call	SendXTByte		; send it

Main10:
	clrf	rXTdelayed		; clear flags
	goto	Main2			; keep going

	endproc Main

	subtitle	"Initialization code"
	page

;*	Setup - Boot setup tasks.
;	-------------------------
;
;	Set up I/O ports, timers and interrupts.
;	Initialize variables.
;

	proc	SetUp

	bank1

;	Option Register settings:
;		Weak pullup on GPIO enabled
;		Interrupt on falling edge of GP2
;		Use CPU clock
;		Increment on high-to-low transistion
;		Assign prescaler to TMR0
;		Prescale clock by 4

OPTIONS equ	(1<<T0SE)+(1<<PS0)

	movif	OPTIONS,OPTION_REG	; /4 counter, internal clock
					; to TMR, enable pullups, GP2INT falling

;	Just in case there's no keyboard or PC attached, we enable weak
;	pullups on all pins to keep them from floating.

	movif	(bvKeyClock+bvKeyData+bvPassE0E1+bvHostClock+bvHostData),WPU

;	Initially, the only outputs that are enabled are for the AT clock
;	pulldown. Everything else is input.

	movif	(bvKeyClock+bvKeyData+bvHostData+bvHostClock),TRISIO

;	Calibrate the internal oscillator
 
	call	0x3ff			; get the calibration value
	movwf	OSCCAL			; calibrate oscillator

;	The following line should be uncommented if a 12F675 is being used.
;	It disables the AD converter and sets GP2 to digital mode.

;	clrf	 0x1f			; ANSEL (12F675 only)

	bank0

	movif	7,CMCON			; disable comparator
	
;	Set the outputs to a known state.

	movif	bvKeyClockPD+bvHostData+bvHostClock,GPIO

;	Clear up all at once and initialize what's missing

	movlw	0x20
	movwf	FSR				; put condense of W register in memory location "FSR"       
CleanUp:
	clrf	INDF			; clear location pointed at by the value in W register. 
	incf	FSR,F			; increment the content of the FSR
	btfss	FSR,7			; test bit7 in the FSR and skip if set (exiting the loop)
	goto	CleanUp			; loop

	movif	rKeyQFirst,rKeyQIn
	movwf	rKeyQOut		; set IN=OUT on keyboard queue

;	Timer0 is used as a sub-1 mS timer, while timer1 is used for
;	millisecond intervals (mostly as a deadman).

;	Timer 0 Control Settings:
;
;		Prescale by 4
;		Use internal clock
;		Timer 1 enabled

TIMER1_CONTROL	equ	(1<<T1CKPS1)

	movif	TIMER1_CONTROL,T1CON		; TIMER1 internal, osc, /4, enabled
	bsf		T1CON,TMR1ON				; start timer 1
	
	bcf		INTCON,INTF					; clear GP2 interrupts
	movif	(1<<GIE)+(1<<INTE),INTCON	; enable GP2 interrupts

;	Send out a "Diagnostics passed byte" (and let keyboard clock float high)

	movif	XTSC_OKAY,rXTByte
	call	SendXTByte					; send it
	
	return								; exit
	endproc Setup

	end
