#ifndef LINT
#ifdef RCS_ID
static char *rcsid = "$Header: /nwd/tools/media/X11/XP/src/xmodmap/RCS/handle.c,v 1.14 1993/04/23 22:38:51 houchin Exp $";
#endif /* RCS_ID */
#endif /* LINT */
/*
 * xmodmap - program for loading keymap definitions into server
 *
 * $XConsortium: handle.c,v 1.21 89/12/10 19:48:04 jim Exp $
 *
 * Copyright 1988 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * Author:  Jim Fulton, MIT X Consortium
 */

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xlibint.h>

#if (! defined M4300) && (! defined M4800)
#include "Xkbd.h"	/* for KBCompose routines stolen from Tek Xlib */
#endif

#include <X11/keysym.h>
#include <stdio.h>
#include <ctype.h>
#include "xmodmap.h"
#include "wq.h"

static XModifierKeymap *map = NULL;


/*
 * The routines in this file manipulate a queue of intructions.  Instead of
 * executing each line as it is entered, we build up a list of actions to 
 * take and execute them all at the end.  This allows us to find all errors
 * at once, and to preserve the context in which we are looking up keysyms.
 */

struct wq work_queue = {NULL, NULL};


/*
 * common utility routines
 */

char *copy_to_scratch (s, len)
    char *s;
    int len;
{
    static char *buf = NULL;
    static int buflen = 0;

    if (len > buflen) {
	if (buf) free (buf);
	buflen = (len < 40) ? 80 : (len * 2);
	buf = (char *) calloc (1,buflen+1);
    }
    if (len > 0)
      strncpy (buf, s, len);
    else 
      len = 0;

    buf[len] = '\0';
    return (buf);
}

badheader ()
{
    fprintf (stderr, "%s:  %s:%d:  bad ", ProgramName, inputFilename, lineno);
}

#define badmsg(what,arg) { badheader(); fprintf (stderr, what, arg); \
			   putc ('\n', stderr); }

#define badmsg2(what,arg1,arg2) { badheader(); fprintf (stderr, what, arg1, arg2); \
				  putc ('\n', stderr); }

#define badmsgn(what,s,len) badmsg (what, copy_to_scratch (s, len))

void initialize_map ()
{
    map = XGetModifierMapping (dpy);
    return;
}

static int do_keycode(), do_keysym(), finish_keycode(), get_keysym_list();
static int do_add(), do_remove(), do_clear(), do_pointer();
static int do_remove_compose(), do_compose(), do_execute();

int skip_word(), skip_space(), skip_chars();

static struct dt {
    char *command;			/* name of input command */
    int length;				/* length of command */
    int (*proc)();			/* handler */
} dispatch_table[] = {
    { "keycode", 7, do_keycode },
    { "keysym", 6, do_keysym },
    { "add", 3, do_add },
    { "remove", 6, do_remove },
    { "clear", 5, do_clear },
    { "pointer", 7, do_pointer },
    { "compose", 7, do_compose },
    { "execute", 7, do_execute },
    { NULL, 0, NULL }};

/*
 * handle_line - this routine parses the input line (which has had all leading
 * and trailing whitespace removed) and builds up the work queue.
 */

int handle_line (line, len)
    char *line;				/* string to parse */
    int len;				/* length of line */
{
    int n;
    struct dt *dtp;
    int status;

    n = skip_chars (line, len);
    if (n < 0) {
	badmsg ("input line '%s'", line);
	return (-1);
    }

    for (dtp = dispatch_table; dtp->command != NULL; dtp++) {
	if (n == dtp->length &&
	    strncmp (line, dtp->command, dtp->length) == 0) {

	    n += skip_space (line+n, len-n);
	    line += n, len -= n;

	    status = (*(dtp->proc)) (line, len);
	    return (status);
	}
    }

    fprintf (stderr, "%s:  unknown command on line %s:%d\n",
	     ProgramName, inputFilename, lineno);
    return (-1);
}

/*
 * the following routines are useful for parsing
 */ 

int skip_word (s, len)
    register char *s;
    register int len;
{
    register int n;

    n = skip_chars (s, len);
    return (n + skip_space (s+n, len-n));
}

int skip_chars (s, len)
    register char *s;
    register int len;
{
    register int i;

    if (len <= 0 || !s || *s == '\0') return (0);

    for (i = 0; i < len; i++) {
	if (isspace(s[i])) break;
    }
    return (i);
}

int skip_space (s, len)
    register char *s;
    register int len;
{
    register int i;

    if (len <= 0 || !s || *s == '\0') return (0);

    for (i = 0; i < len; i++) {
	if (!s[i] || !isspace(s[i])) break;
    }
    return (i);
}


int skip_until_char (s, len, c)
    register char *s;
    register int len;
    register char c;
{
    register int i;

    for (i = 0; i < len; i++) {
	if (s[i] == c) break;
    }
    return (i);
}

int skip_until_chars (s, len, cs, cslen)
    char *s;
    int len;
    register char *cs;
    register int cslen;
{
    int i;

    for (i = 0; i < len; i++) {
	register int j;
	register char c = s[i];

	for (j = 0; j < cslen; j++) {
	    if (c == cs[j]) goto done;
	}
    }
  done:
    return (i);
}

/*
 * The action routines.
 *
 * This is where the real work gets done.  Each routine is responsible for
 * parsing its input (note that the command keyword has been stripped off)
 * and adding to the work queue.  They are also in charge of outputting
 * error messages and returning non-zero if there is a problem.
 *
 * The following global variables are available:
 *     dpy                the display descriptor
 *     work_queue         linked list of opcodes
 *     inputFilename      name of the file being processed
 *     lineno             line number of current line in input file
 */

add_to_work_queue (p)			/* this can become a macro someday */
    union op *p;
{
    if (work_queue.head == NULL) {	/* nothing on the list */
	work_queue.head = work_queue.tail = p;	/* head, tail, no prev */
    } else {
	work_queue.tail->generic.next = p;  /* head okay, prev */
	work_queue.tail = p;		/* tail */
    }
    p->generic.next = NULL;

    if (verbose) {
	print_opcode (p);
    }
    return;
}

char *copystring (s, len)
    char *s;
    int len;
{
    char *retval;

    retval = (char *) calloc (1,len+1);
    if (retval) {
	strncpy (retval, s, len);
	retval[len] = '\0';
    }
    return (retval);
}

static Bool parse_number (str, val)
    char *str;
    unsigned long *val;
{
    char *fmt = "%ld";

    if (*str == '0') {
	str++;
	fmt = "%lo";
	if (*str == '\0') {
	    *val = 0;
	    return 1;
	}
	if (*str == 'x' || *str == 'X') {
	    str++;
	    fmt = "%lx";
	}
    }
    return (sscanf (str, fmt, val) == 1);
}

static Bool parse_keysym (line, n, name, keysym)
    char *line;
    int n;
    char **name;
    KeySym *keysym;
{
    *name = copy_to_scratch (line, n);
    if (!strcmp(*name, "NoSymbol")) {
	*keysym = NoSymbol;
	return (True);
    }
    *keysym = XStringToKeysym (*name);
    if (*keysym == NoSymbol && '0' <= **name && **name <= '9')
	return parse_number(*name, keysym);
    return (*keysym != NoSymbol);
}

/*
 * do_keycode - parse off lines of the form
 *
 *                 "keycode" number "=" [keysym ...]
 *                           ^
 *
 * where number is in decimal, hex, or octal.  Any number of keysyms may be
 * listed.
 */


static int parse_keycode (line, len, keycode, err_type)
    char *line;
    int len;
    KeyCode *keycode;
    char *err_type;
{
    int dummy;
    char *fmt = "%d";

    if (*line == '0') line++, len--, fmt = "%o";
    if (*line == 'x' || *line == 'X') line++, len--, fmt = "%x";

    dummy = 0;
    if (sscanf (line, fmt, &dummy) != 1 || dummy == 0) {
	badmsg2 ("%s (syntax error) at\n    \"%s\"", err_type, line);
	return (-1);
    }
    *keycode = (KeyCode) dummy;
    if (*keycode < min_keycode || *keycode > max_keycode) {
	badmsg2 ("%s (keycode %d out of range)", err_type, *keycode);
	return (-1);
    }
    return (0);
}


static int do_keycode (line, len)
    char *line;
    int len;
{
    KeyCode keycode;

    if (len < 3 || !line || *line == '\0') {  /* 5=a minimum */
	badmsg ("keycode input line", NULL);
	return (-1);
    }
    if (parse_keycode (line, len, &keycode, "keycode expression value"))
	return (-1);
    return (finish_keycode (line, len, keycode));
}

/*
 * do_keysym - parse off lines of the form
 *
 *                 "keysym" keysym "=" [keysym ...]
 *                          ^
 *
 * The left keysyms has to be checked for validity and evaluated.
 */

static int do_keysym (line, len)
    char *line;
    int len;
{
    int n;
    KeyCode keycode;
    KeySym keysym;
    char *tmpname;

    if (len < 3 || !line || *line == '\0') {  /* a=b minimum */
	badmsg ("keysym input line", NULL);
	return (-1);
    }

    n = skip_chars (line, len);
    if (n < 1) {
	badmsg ("target keysym name", NULL);
	return (-1);
    }
    if (!parse_keysym(line, n, &tmpname, &keysym)) {
	badmsg ("keysym target key symbol '%s'", tmpname);
	return (-1);
    }

    keycode = XKeysymToKeycode (dpy, keysym);
    if (verbose) {
	printf ("! Keysym %s (0x%lx) corresponds to keycode 0x%x\n", 
		tmpname, (long) keysym, keycode);
    }
    if (keycode < min_keycode || keycode > max_keycode) {
	if (!verbose) {
	    printf ("! Keysym %s (0x%lx) corresponds to keycode 0x%x\n", 
		    tmpname, (long) keysym, keycode);
	}
	badmsg ("keysym target keysym '%s', out of range", tmpname);
	return (-1);
    }

    return (finish_keycode (line, len, keycode));
}

static int finish_keycode (line, len, keycode)
    char *line;
    int len;
    KeyCode keycode;
{
    int n;
    KeySym *kslist;
    union op *uop;
    struct op_keycode *opk;
   
    n = skip_until_char (line, len, '=');
    line += n, len -= n;
    
    if (len < 1 || *line != '=') {	/* = minimum */
	badmsg ("keycode command (missing keysym list),", NULL);
	return (-1);
    }
    line++, len--;			/* skip past the = */

    n = skip_space (line, len);
    line += n, len -= n;

    /* allow empty list */
    if (get_keysym_list (line, len, &n, &kslist) < 0 || n < 0) {
	badmsg ("keycode keysym list", NULL);
	return (-1);
    }

    uop = AllocStruct (union op);
    if (!uop) {
	badmsg ("attempt to allocate a %ld byte keycode opcode",
		(long) sizeof (struct op_keycode));
	return (-1);
    }
    opk = &uop->keycode;

    opk->type = doKeycode;
    opk->target_keycode = keycode;
    opk->count = n;
    opk->keysyms = kslist;

    add_to_work_queue (uop);

#ifdef later
    /* make sure we handle any special keys */
    check_special_keys (keycode, n, kslist);
#endif

    return (0);
}


/*
 * parse_modifier - convert a modifier string name to its index
 */

struct modtab modifier_table[] = {	/* keep in order so it can be index */
    { "shift", 5, 0 },
    { "lock", 4, 1 },
    { "control", 7, 2 },
    { "mod1", 4, 3 },
    { "mod2", 4, 4 },
    { "mod3", 4, 5 },
    { "mod4", 4, 6 },
    { "mod5", 4, 7 },
    { "ctrl", 4, 2 },
    { "s", 5, 0 },
    { "l", 4, 1 },
    { "c", 7, 2 },
    { "m1", 4, 3 },
    { "m2", 4, 4 },
    { "m3", 4, 5 },
    { "m4", 4, 6 },
    { "m5", 4, 7 },
    { NULL, 0, 0 }};

static int parse_modifier (line, n)
    register char *line;
    register int n;
{
    register int i;
    struct modtab *mt;

    /* lowercase for comparison against table */
    for (i = 0; i < n; i++) {
	if (isupper (line[i])) line[i] = tolower (line[i]);
    }

    for (mt = modifier_table; mt->name; mt++) {
	if (n == mt->length && strncmp (line, mt->name, n) == 0)
	  return (mt->value);
    }
    return (-1);
}


static int parse_modifier_or_none (line, n)
    register char *line;
    register int n;
{
    int modifier;

    if ((modifier = parse_modifier (line, n)) != -1)
	return (modifier);
    else if (n == 4 && strncmp (line, "none", n) == 0)
	return (-2);
    return (-1);
}

static char * print_modifiers(modmask)
    unsigned short modmask;
{
    static char buf[256];
    char * bufptr = buf;
    static char * anym = "AnyModifier, ";
    Bool havemods = False;
    int i;

    buf[0] = NULL;
    if (modmask & AnyModifier) {
	sprintf (bufptr, anym);
	bufptr += strlen(anym);
	havemods = True;
	modmask ^= AnyModifier;
    }
    for (i = 0; modmask != 0; modmask = modmask >> 1, i++)
	if (modmask & 1) {
	    sprintf (bufptr, modifier_table[i].name);
	    bufptr += modifier_table[i].length;
	    sprintf (bufptr, ", ");
	    bufptr += 2;
	    havemods = True;
	}
    if (!havemods) {
	sprintf (bufptr, "<None>, ");
    }
	
    return (buf);
}

/*
 * do_add - parse off lines of the form
 *
 *                 add MODIFIER = keysym ...
 *                     ^
 * where the MODIFIER is one of Shift, Lock, Control, Mod[1-5] where case
 * is not important.  There should also be an alias Ctrl for control.
 */

static int do_add (line, len)
    char *line;
    int len;
{
    int n;
    int modifier;
    KeySym *kslist;
    union op *uop;
    struct op_addmodifier *opam;

    if (len < 6 || !line || *line == '\0') {  /* Lock=a minimum */
	badmsg ("add modifier input line", NULL);
	return (-1);
    }

    n = skip_chars (line, len);
    if (n < 1) {
	badmsg ("add modifier name %s", line);
	return (-1);
    }

    modifier = parse_modifier (line, n);
    if (modifier < 0) {
	badmsgn ("add modifier name '%s', not allowed", line, n);
	return (-1);
    }

    line += n, len -= n;
    n = skip_until_char (line, len, '=');
    if (n < 0) {
	badmsg ("add modifier = keysym", NULL);
	return (-1);
    }

    n++;				/* skip = */
    n += skip_space (line+n, len-n);
    line += n, len -= n;

    if (get_keysym_list (line, len, &n, &kslist) < 0 || n <= 0) {
	badmsg ("add modifier keysym list", NULL);
	return (-1);
    }

    uop = AllocStruct (union op);
    if (!uop) {
	badmsg ("attempt to allocate %ld byte addmodifier opcode",
		(char*) sizeof (struct op_addmodifier));
	return (-1);
    }
    opam = &uop->addmodifier;

    opam->type = doAddModifier;
    opam->modifier = modifier;
    opam->count = n;
    opam->keysyms = kslist;

    add_to_work_queue (uop);
    
    return (0);
}

#ifdef AUTO_ADD_REMOVE
/*
 * make_add - stick a single add onto the queue
 */
static void make_add (modifier, keysym)
    int modifier;
    KeySym keysym;
{
    union op *uop;
    struct op_addmodifier *opam;

    uop = AllocStruct (union op);
    if (!uop) {
	badmsg ("attempt to allocate %ld byte addmodifier opcode",
		(char*) sizeof (struct op_addmodifier));
	return;
    }
    opam = &uop->addmodifier;

    opam->type = doAddModifier;
    opam->modifier = modifier;
    opam->count = 1;
    opam->keysyms = (KeySym *) calloc (1,sizeof (KeySym));
    if (!opam->keysyms) {
	badmsg ("attempt to allocate %ld byte KeySym", (char*) sizeof (KeySym));
	free ((char *) opam);
	return;
    }
    opam->keysyms[0] = keysym;

    add_to_work_queue (uop);
    return;
}
#endif /* AUTO_ADD_REMOVE */


/*
 * do_remove - parse off lines of the form
 *
 *                 remove MODIFIER = oldkeysym ...
 *                        ^
 * where the MODIFIER is one of Shift, Lock, Control, Mod[1-5] where case
 * is not important.  There should also be an alias Ctrl for control.
 *		  remove compose ANYORALLMODNAMES KEYCODELIST
 *		  remove compose ANYORALLMODNAMES keysyms KEYSYMNAMELIST
 *		  remove compose numeric HEXDIGIT = KEYCODE
 *		  remove compose numeric HEXDIGIT = KEYSYM
 *		  remove compose cancel KEYCODE
 */

static int do_remove (line, len)
    char *line;
    int len;
{
    int n;
    int nc;
    int i;
    int modifier;
    KeySym *kslist;
    KeyCode *kclist;
    union op *uop;
    struct op_removemodifier *oprm;

    if (len < 6 || !line || *line == '\0') {  /* Lock=a minimum */
	badmsg ("remove input line", NULL);
	return (-1);
    }

    n = skip_chars (line, len);
    if (n < 1) {
	badmsg ("remove name %s", line);
	return (-1);
    }

    /* now, is this a modifier or a compose specification? */
    if ((n == 7) && !strncmp(line, "compose", 7))
	return (do_remove_compose(line+n, len-n));
    /* failed, this is a modifier */

    modifier = parse_modifier (line, n);
    if (modifier < 0) {
	badmsgn ("remove modifier name '%s', not allowed", line, n);
	return (-1);
    }

    line += n, len -= n;
    n = skip_until_char (line, len, '=');
    if (n < 0) {
	badmsg ("remove modifier = keysym", NULL);
	return (-1);
    }

    n++;
    n += skip_space (line+n, len-n);
    line += n, len -= n;

    if (get_keysym_list (line, len, &n, &kslist) < 0 || n <= 0) {
	badmsg ("remove modifier keysym list", NULL);
	return (-1);
    }

    /*
     * unlike the add command, we have to now evaluate the keysyms
     */

    kclist = (KeyCode *) calloc (1,n * sizeof (KeyCode));
    if (!kclist) {
	badmsg ("attempt to allocate %ld byte keycode list",
		(char*) (n * sizeof (KeyCode)));
	free ((char *) kslist);
	return (-1);
    }

    nc = 0;
    for (i = 0; i < n; i++) {
	KeyCode kc = XKeysymToKeycode (dpy, kslist[i]);
	if (verbose) {
	    char *tmpname = XKeysymToString (kslist[i]);
	    printf ("! Keysym %s (0x%lx) corresponds to keycode 0x%x\n", 
		    tmpname ? tmpname : "?", (long) kslist[i], kc);
	}
	if (kc < min_keycode || kc > max_keycode) {
	    char *tmpname = XKeysymToString (kslist[i]);
	    printf ("! Keysym %s (0x%lx), keycode 0x%x, ",
		    tmpname ? tmpname : "?", (long) kslist[i], kc);
	    printf ("out of range [0x%x, 0x%x]\n", 
		    min_keycode, max_keycode);
	    badmsg ("keycode value 0x%lx in remove modifier list",
		    (char*) kc);
	    continue;
	}
	kclist[nc++] = kc;		/* okay, add it to list */
    }

    free ((char *) kslist);		/* all done with it */

    uop = AllocStruct (union op);
    if (!uop) {
	badmsg ("attempt to allocate %ld byte removemodifier opcode",
		(char*) sizeof (struct op_removemodifier));
	return (-1);
    }
    oprm = &uop->removemodifier;

    oprm->type = doRemoveModifier;
    oprm->modifier = modifier;
    oprm->count = nc;
    oprm->keycodes = kclist;

    add_to_work_queue (uop);
    
    return (0);
}


static XKBCompose ComposeTable;
#define DEF_MAX_SEQ_LEN 2
/* sequences must be at least two keycodes */
#define INIT_SEQ_SPACE 100
static int seq_space = INIT_SEQ_SPACE;
static int cancel_space;
int new_seq_count = 0;	/* count of new compose sequences specified */
Bool have_compose_chg = False;
static int max_compose_keycodes = DEF_MAX_SEQ_LEN;

#define KEYCODES_PER_HEX_CHAR 3 
void initialize_compose ()
{
    int i, j, k;
    KeyCode *temp_cancel_keycodes;
    KeyCode Delete_key;
    KeySym none;
    static Bool cleared = False;

    if (!cleared && (noCompose || resetCompose)) {
	/* delete any existing Compose prop before initializing.  Make sure
	   it happens as soon as option is executed.  XXX A -resetcompose
	   after a -e "compose" -e "execute" will have no effect. */
	XClearKBCompose(dpy);
    /* the kludge - to make clients call XRefreshKeyboarMapping which will
       reread compose property, call XChangeKeyboardMapping.  XXX Note that
       this overhead makes it imperative that we process all real mapping
       changes at once!  Eventually, we will call this once for both Compose
       and the keyboard rearchitecture extension, which is why it is done here
       and not in XSetKBCompose. */
    	XChangeKeyboardMapping (dpy, dpy->min_keycode, 1, &none, 0);
	cleared = True;
    }

    if (compose_initialized || (!resetCompose && (have_compose_chg == False)))
	/* optimization - don't waste time reading/writing compose
	   prop if we have nothing to do (yet).  */
	return;

    compose_initialized = True;

    if (XGetKBCompose (dpy, DefaultRootWindow(dpy), &ComposeTable)) {
	seq_space = ComposeTable.num_composes;
	cancel_space = ComposeTable.num_cancel_keycodes;
	return;
    }
    /* no property; initialize struct to default values */
    ComposeTable.num_composes = 0;	/* no compose sequences initially */
    ComposeTable.max_keycodes = DEF_MAX_SEQ_LEN;
    for (i = 0; i < 10; i++) {
	ComposeTable.numeric_keycodes[i * KEYCODES_PER_HEX_CHAR ] = 
					    XKeysymToKeycode(dpy, XK_0 + i);
	ComposeTable.numeric_keycodes[1 + (i * KEYCODES_PER_HEX_CHAR )] = 
					    XKeysymToKeycode(dpy, XK_KP_0 + i);
	ComposeTable.numeric_keycodes[2 + (i * KEYCODES_PER_HEX_CHAR )] = 0;

    }
    for (i = 0; i < 6; i++) {
	ComposeTable.numeric_keycodes[(i + 10) * KEYCODES_PER_HEX_CHAR ] = 
					    XKeysymToKeycode(dpy, XK_A + i);
	ComposeTable.numeric_keycodes[1 + ((i + 10) * KEYCODES_PER_HEX_CHAR )]
					    = 0;
	ComposeTable.numeric_keycodes[2 + ((i + 10) * KEYCODES_PER_HEX_CHAR )]
					    = 0;
    }
    ComposeTable.modifiers = (unsigned short *)
	calloc(1,sizeof(unsigned short) * INIT_SEQ_SPACE * DEF_MAX_SEQ_LEN);
    ComposeTable.sequences = (KeyCode *)
	calloc(1,sizeof(KeyCode) * INIT_SEQ_SPACE * DEF_MAX_SEQ_LEN);
    ComposeTable.output_modifiers = (unsigned int *)
			    calloc(1,sizeof(unsigned int) * INIT_SEQ_SPACE);
    ComposeTable.output_keycodes = (KeyCode *)
			    calloc(1,sizeof(KeyCode) * INIT_SEQ_SPACE);
	    /* really need to use Xmalloc, so XFreeKBCompose can XFree */

    temp_cancel_keycodes = (KeyCode *) calloc(1,sizeof(KeyCode) * 
				   (dpy->max_keycode - dpy->min_keycode + 1));
    j = 0;
    for (i = dpy->min_keycode; i <= dpy->max_keycode; i++) {
	/* check for a keysym of 0xFFxx first in the list */
	/* Note that this picks up XK_Delete */
	k = (i - dpy->min_keycode) * dpy->keysyms_per_keycode;
	if ((dpy->keysyms[k] & 0xff00) == 0xff00)
	    temp_cancel_keycodes[j++] = i;
        /* Note - we can get away with this because the call to
	   XKeysymToKeycode() initializes the keysym table */
    }
    /* Note - assumes this is a real keyboard, with at least one "cancel"
       key! */
    ComposeTable.cancel_keycodes = (KeyCode *)calloc(1,sizeof(KeyCode) * j);
    bcopy (temp_cancel_keycodes, ComposeTable.cancel_keycodes, 
	   sizeof(KeyCode) * j);
    free(temp_cancel_keycodes);
    cancel_space = j;
    ComposeTable.num_cancel_keycodes = j;
    /* Currently, Delete is hardwired to be silently discarded */
    if (Delete_key = XKeysymToKeycode(dpy, XK_Delete)) {
	/* in case there is no Delete key */
	ComposeTable.abort_keycodes = (KeyCode *)calloc(1,sizeof(KeyCode));
	ComposeTable.abort_keycodes[0] = Delete_key;
	ComposeTable.num_abort_keycodes = 1;
    }
    else {
	ComposeTable.abort_keycodes = (KeyCode *)NULL;
	ComposeTable.num_abort_keycodes = 0;
    }
    return;
}

static int
parse_numeric(line, len, keycode)
    char *line;
    int len;
    KeyCode *keycode;
{
    if ((len >= 7) && !strncmp(line, "decimal", 7)) {
	*keycode = DECIMAL_COMPOSE;
    }
    else if ((len >= 6) && !strncmp(line, "binary", 6)) {
	*keycode = BINARY_COMPOSE;
    }
    else if ((len >= 5) && !strncmp(line, "octal", 5)) {
	*keycode = OCTAL_COMPOSE;
    }
    else if ((len >= 3) && !strncmp(line, "hex", 3)) {
	*keycode = HEX_COMPOSE;
    }
    else
	return (-1);
    return (0);
}

parse_compose_sequence(lineptr, lenptr, keycode_count, keycodelist_return, 
		       modifierlist_return, isnumeric)
    char **lineptr;
    int *lenptr;
    int *keycode_count;
    KeyCode **keycodelist_return;
    unsigned short **modifierlist_return;
    Bool *isnumeric;

{
    char *line;
    int len;
    int n = 0;
    /* keep lint quiet */
    KeyCode keycode;
    KeySym keysym;
    char temp[256];
    char *tmpname;
    int modifier = -3;
    Bool ignore_keysym = False;
    int havesofar = 0;
    KeyCode *keycodelist;
    unsigned short *modifierlist;
    int column;

    line = *lineptr; len = *lenptr;
    keycodelist = (KeyCode *) calloc (1,max_compose_keycodes * sizeof (KeyCode));
    if (!keycodelist) {
	badmsg ("attempt to allocate %ld byte initial keycodelist",
		(char *) (max_compose_keycodes * sizeof (KeyCode)));
	return (-1);
    }
    modifierlist = (unsigned short *) calloc (1,max_compose_keycodes * 
				       sizeof (unsigned short));
    if (!modifierlist) {
	badmsg ("attempt to allocate %ld byte initial modifierlist",
		(char *) (max_compose_keycodes * sizeof (unsigned short)));
	return (-1);
    }

    /* first collect initial modifiers, if any */
    modifierlist[0] = 0;
    do {
	if (modifier >= 0) {
	    n = n + skip_space (line+n, len-n);
	    line += n, len -= n;
	    modifierlist[0] |= 1 << modifier;
	    /* convert to Mod1Mask, etc. */
	}
	else if (modifier == -1) {
	    /* must be anymodifier */
	    n = n + skip_space (line+n, len-n);
	    line += n, len -= n;
	    modifierlist[0] |= AnyModifier;
	}
	else if (modifier == -2) {
	    /* "none" */
	    n = n + skip_space (line+n, len-n);
	    line += n, len -= n;
	    /* wipes out any prior modifiers */
	    modifierlist[0] = 0;
	    /* don't take any modifiers from the keysym column */
	    ignore_keysym = True;
	}
	/* else first time through */
	n = skip_chars(line, len);
	strncpy(temp, line, n);
	temp[n] = 0;
    } while (((modifier = parse_modifier_or_none(temp, n)) != -1) || 
	     !strcmp("anymodifier", temp));
    /* note - parse_modifier lowercases temp, so anymodifier is any case */

    /* now, is this a keysym, keycode, or numeric specification? */
    if ((len < 6) || strncmp(line, "keysym", 6)) {
	/* failed, this is a keycode list or KEYCODE NUMERIC */
	while ((len > 0) && strncmp(line, "=", 1)) {
	    /* compose ends with '=', remove compose ends with newline */
	    if (!(*isnumeric = !parse_numeric(line, len, &keycode))) {
		/* not numeric, so go for keycode */
	        /* this does not catch bogus syntaxes intermingling keycodes
		   and NUMERICs */
		if (parse_keycode (line, len, &keycode,
		    "compose expression sequence keycode value"))
		    return (-1);
	    }
	    /* stash keycode */
	    /* grow the list bigger if necessary */
	    if (havesofar >= max_compose_keycodes) {
		max_compose_keycodes++;
		/* track largest keycode count */
		keycodelist = (KeyCode *) realloc (keycodelist,
			       max_compose_keycodes * sizeof (KeyCode));
		if (!keycodelist) {
		    badmsg ("attempt to grow keycode list to %ld bytes",
			    (char *) (max_compose_keycodes * sizeof (KeyCode)));
		    return (-1);
		}
		modifierlist = (unsigned short *) realloc (modifierlist,
			       max_compose_keycodes * sizeof (unsigned short));
		if (!modifierlist) {
		    badmsg ("attempt to grow modifier list to %ld bytes",
		     (char *) (max_compose_keycodes * sizeof (unsigned short)));
		    return (-1);
		}
	    }
	    if (havesofar)
		modifierlist[havesofar] = AnyModifier;
	    /* first keycode's modifiers already specified explicitly */
	    /* keycode spec on subsequent keys allows any modifier combo */
	    keycodelist[havesofar++] = keycode;
	    n = skip_word(line, len);
	    line += n; len -= n;
	}
    }
    else {
	if ((len >= 7) && (line[6] == 's'))
	    /* this is a plain keysyms list */
	    *isnumeric = False;
	else
	    /* keysym KEYSYMNAME NUMERIC */
	    *isnumeric = True;
	/* skip "keysym[s]" keyword */
	n = skip_word(line, len);
	line += n; len -= n;
	while ((len > 0) && strncmp(line, "=", 1)) {
	    if (*isnumeric && (havesofar == 1)) {
		/* must have exactly one KEYSYMNAME; second token should be
		   NUMERIC */
		if (parse_numeric(line, len, &keycode)) {
		    badmsg ("compose expression sequence: bad NUMERIC token, should be \"binary\", \"octal\", \"decimal\", or \"hex\"", NULL);
		    return (-1);
		}
	    }
	    else if (*isnumeric && (havesofar == 2)) {
		badmsg ("compose expression sequence: bogus extra token after NUMERIC", NULL);
		return (-1);
	    }
	    else {
		n = skip_chars(line, len);
		if (!parse_keysym(line, n, &tmpname, &keysym)) {
		    badmsg ("(invalid) compose expression sequence keysym '%s'", 
			     tmpname);
		    return (-1);
		}
		keycode = XKeysymToKeycodeAndColumn(dpy, keysym, &column);
		if (verbose) {
		    printf ("! Keysym %s (0x%lx) corresponds to keycode 0x%x, column %d\n", 
			    tmpname, (long) keysym, keycode, column);
		}
		if (keycode < min_keycode || keycode > max_keycode) {
		    badmsg ("(unassigned) compose expression sequence keysym '%s', not found in mapping table", 
			    tmpname);
		    return (-1);
		}
	    }
	    /* stash keycode */
	    /* grow the list bigger if necessary */
		if (havesofar >= max_compose_keycodes) {
		    max_compose_keycodes++;
		    /* track largest keycode count */
		    keycodelist = (KeyCode *) realloc (keycodelist,
				   max_compose_keycodes * sizeof (KeyCode));
		    if (!keycodelist) {
			badmsg ("attempt to grow keycode list to %ld bytes",
			    (char *) (max_compose_keycodes * sizeof (KeyCode)));
			return (-1);
		    }
		    modifierlist = (unsigned short *) realloc (modifierlist,
			       max_compose_keycodes * sizeof (unsigned short));
		    if (!modifierlist) {
			badmsg ("attempt to grow modifier list to %ld bytes",
		      (char *) (max_compose_keycodes * sizeof (unsigned short)));
			return (-1);
		    }
		}
		if (havesofar) {
		/* Set modifiers for subsequent keycodes to keysym column
		   (currently just Shift).  Note that xmodmap currently does
		   not support explicit modifiers on any but the first key in
		   the sequence. */
		    if (XColumnToModifier(dpy, column, &modifier))
			modifierlist[havesofar] = modifier;
		    else
			modifierlist[havesofar] = 0;
			/* if column > 1, behavior currently undefined.  If a
			   language group protocol is adopted, we would need
			   to include the language modifier. */
		}
		else
		    /* first keycode's modifiers already specified
		       explicitly, just or in Shift unless explicit modifier
		       was "none" */
		    if (!ignore_keysym && 
			XColumnToModifier(dpy, column, &modifier))
			modifierlist[havesofar] |= modifier;

		keycodelist[havesofar++] = keycode;

	    n = skip_word(line, len);
	    line += n; len -= n;
	}
    }
    *lineptr = line; *lenptr = len;
    if (havesofar < 1) {
	badmsg ("compose sequence must have at least one keycode\n", NULL);
	return (-1);
    }
	
    *keycode_count = havesofar;
    *keycodelist_return = keycodelist;
    *modifierlist_return = modifierlist;
    return (0);
}

static int
parse_hex_char(lineptr, lenptr, hex_return, err_type)
    char **lineptr;
    int *lenptr;
    int *hex_return;
    char *err_type;
{
    char msg[256];
    char *tmp = "1";
    int n;

    tmp[0] = tolower(**lineptr);
    n = skip_word(*lineptr, *lenptr);
    strcpy (msg, err_type);

    if (isdigit(tmp[0]) || (tmp[0] >= 'a' && tmp[0] <= 'f')) {
	sscanf (tmp, "%x", hex_return);
	*lineptr += n, *lenptr -= n;
	return (0);
    }
    strcat (msg, " has bogus HEXCHAR %c");
    badmsg (msg, (char *) **lineptr);
    /* yes, we still have to fool lint! */
    *lineptr += n, *lenptr -= n;
    return (-1);
}

static int
parse_key(line, len, keycode_return, column_return, err_type)
    char **line;
    int *len;
    KeyCode *keycode_return;
    char *err_type;
    int *column_return;
{
    char msg[256];
    KeySym keysym;
    int n;
    char *tmpname;

    strcpy (msg, err_type);

    if ((*len < 6) || strncmp(*line, "keysym", 6)) {
	/* failed, this is a keycode */
	strcat (msg, " keycode value");
	if (parse_keycode (*line, *len, keycode_return, msg))
	    return (-1);
	*column_return = 0;
	n = skip_word(*line, *len);
    }
    else {
	/* this is a keysym */
	/* skip "keysym" keyword */
    	n = skip_word(*line, *len);
	*line += n; *len -= n;
	n = skip_chars (*line, *len);
	if (!parse_keysym(*line, n, &tmpname, &keysym)) {
	    strcat (msg, " keysym '%s'");
	    badmsg (msg, tmpname);
	    return (-1);
	}
	*keycode_return = XKeysymToKeycodeAndColumn(dpy, keysym,
						    column_return);
	if (verbose) {
	    printf ("! Keysym %s (0x%lx) corresponds to keycode 0x%x\n", 
		    tmpname, (long) keysym, *keycode_return);
	}
	if (*keycode_return < min_keycode || *keycode_return > max_keycode) {
	    strcat (msg, " keysym '%s', not found in mapping table");
	    badmsg (msg, tmpname);
	    return (-1);
	}

	n = n + skip_space(*line+n, *len-n);
    }
    if (*len != n)
	badmsg ("%s has trailing garbage; ignored", err_type);
    return (0);
}

static int do_remove_compose (line, len)
    char *line;
    int len;
{
    int n;
    int keycode_count;
    KeyCode *keycodelist;
    unsigned short *modifierlist;
    int hexdigit;
    KeyCode keycode;
    struct op_remove_compose_sequence *oprmc;
    struct op_remove_compose_numeric *oprmcn;
    struct op_remove_compose_cancel *oprmcc;
    Bool isnumeric;
    int column;	    /* junk */

    n = skip_space (line, len);
    if (n < 1) {
	badmsg ("no space after \"remove compose\" in line %s", line);
	return (-1);
    }
    have_compose_chg = True;
    line += n, len -= n;
    /* now, is this a numeric, cancel, or sequence remove? */
    if ((len >= 7) && !strncmp(line, "numeric", 7)) {
	/* skip "numeric" keyword */
	n = skip_word(line, len);
	line += n; len -= n;
	
	if (parse_hex_char(&line, &len, &hexdigit,
			   "remove compose numeric expression"))
	    return (-1);

	if (line[0] != '=') {
	    badmsg ("remove compose numeric expression, missing equal sign", NULL);
	    return (-1);
        }
	line++, len--;			/* skip '=' */
	n = skip_space (line, len);
	line += n, len -= n;

	if (parse_key(&line, &len, &keycode, &column, 
		      "remove compose numeric expression"))
	    return (-1);

        oprmcn = AllocStruct (struct op_remove_compose_numeric);
        if (!oprmcn) {
	    badmsg ("attempt to allocate a %ld byte remove compose numeric opcode",
		    (char *) sizeof (struct op_remove_compose_numeric));
	    return (-1);
	}
	oprmcn->type = doRemoveComposeNumeric;
	oprmcn->hexdigit = hexdigit;
	oprmcn->keycode = keycode;
	add_to_work_queue ((union op *)oprmcn);
    }
    else if ((len >= 6) && !strncmp(line, "cancel", 6)) {
	/* skip "cancel" keyword */
	n = skip_word(line, len);
	line += n; len -= n;
	if (parse_keycode (line, len, &keycode,
			   "remove compose cancel expression keycode value"))
	    return (-1);
        oprmcc = AllocStruct (struct op_remove_compose_cancel);
        if (!oprmcc) {
	    badmsg ("attempt to allocate a %ld byte remove compose cancel opcode",
		    (char *) sizeof (struct op_remove_compose_cancel));
	    return (-1);
	}
	oprmcc->type = doRemoveComposeCancel;
	oprmcc->keycode = keycode;
	add_to_work_queue ((union op *)oprmcc);
    }
    else {
	if (parse_compose_sequence(&line, &len, &keycode_count, &keycodelist, 
				&modifierlist, &isnumeric))
	    return (-1);
        oprmc = AllocStruct (struct op_remove_compose_sequence);
        if (!oprmc) {
	    badmsg ("attempt to allocate a %ld byte remove compose opcode",
		    (char *) sizeof (struct op_remove_compose_sequence));
	    return (-1);
	}
	oprmc->type = doRemoveComposeSequence;
	oprmc->modifiers = modifierlist;
	oprmc->count = keycode_count;
	oprmc->keycodes = keycodelist;

	add_to_work_queue ((union op *)oprmc);
/*	new_seq_count--; we currently don't reuse space */
    }
    return (0);
}

#ifdef AUTO_ADD_REMOVE
/*
 * make_remove - stick a single remove onto the queue
 */
static void make_remove (modifier, keycode)
    int modifier;
    KeyCode keycode;
{
    union op *uop;
    struct op_removemodifier *oprm;

    uop = AllocStruct (union op);
    if (!uop) {
	badmsg ("attempt to allocate %ld byte removemodifier opcode",
		(char*) sizeof (struct op_removemodifier));
	return;
    }
    oprm = &uop->removemodifier;

    oprm->type = doRemoveModifier;
    oprm->modifier = modifier;
    oprm->count = 1;
    oprm->keycodes = (KeyCode *) calloc (1,sizeof (KeyCode));
    if (!oprm->keycodes) {
	badmsg ("attempt to allocate %ld byte KeyCode",
		(char*) sizeof (KeyCode));
	free ((char *) oprm);
	return;
    }
    oprm->keycodes[0] = keycode;

    add_to_work_queue (uop);
    return;
}
#endif /* AUTO_ADD_REMOVE */


/*
 * do_clear - parse off lines of the form
 *
 *                 clear MODIFIER
 *                 clear compose ...
 *                       ^
 */

static int do_clear (line, len)
    char *line;
    int len;
{
    int n;
    int modifier;
    union op *uop;
    struct op_clearmodifier *opcm;
    struct op_clear_compose *opcc;

    if (len < 4 || !line || *line == '\0') {  /* Lock minimum */
	badmsg ("clear modifier input line", NULL);
	return (-1);
    }

    n = skip_chars (line, len);


    /* now, is this a modifier or a clear compose ? */
    if ((n == 7) && !strncmp(line, "compose", 7)) {
        have_compose_chg = True;
	n += skip_space (line+n, len-n);
	line += n, len -= n;
	opcc = AllocStruct (struct op_clear_compose);
	if (!opcc) {
	    badmsg ("attempt to allocate a %ld byte clear compose opcode",
		(char *) sizeof (struct op_clear_compose));
	    return (-1);
	}
    /* now, is this a numeric, cancel, or sequence clear? */
	if ((len >= 9) && !strncmp(line, "sequences", 9)) {
	    opcc->type = doClearComposeSequence;
	    add_to_work_queue ((union op *)opcc);
	    new_seq_count = 0;
	    /* we certainly won't have to enlarge the table! */
	}
        else if ((len >= 7) && !strncmp(line, "numeric", 7)) {
	    opcc->type = doClearComposeNumeric;
	    add_to_work_queue ((union op *)opcc);
	}
        else if ((len >= 6) && !strncmp(line, "cancel", 6)) {
	    opcc->type = doClearComposeCancel;
	    add_to_work_queue ((union op *)opcc);
	}
	else {
	    /* bogus keyword */
	    badmsg ("clear compose keyword \"%s\" is invalid ", line);
	    return (-1);
	}
	/* skip keyword */
	n = skip_word(line, len);
	if (len != n)
	    badmsg ("clear compose has trailing garbage; ignored", NULL);
	return (0);
    }
    /* failed, this is a modifier */

    modifier = parse_modifier (line, n);
    if (modifier < 0) {
	badmsgn ("clear modifier name '%s'", line, n);
	return (-1);
    }
    n += skip_space (line+n, len-n);
    if (n != len) {
	badmsgn ("extra argument '%s' to clear modifier", line+n, len-n);
	/* okay to continue */
    }

    uop = AllocStruct (union op);
    if (!uop) {
	badmsg ("attempt to allocate %d byte clearmodifier opcode",
		(char*) sizeof (struct op_clearmodifier));
	return (-1);
    }
    opcm = &uop->clearmodifier;

    opcm->type = doClearModifier;
    opcm->modifier = modifier;

    add_to_work_queue (uop);

    return (0);
}

static int strncmp_nocase (a, b, n)
    char *a, *b;
    int n;
{
    int i;
    int a1, b1;

    for (i = 0; i < n; i++, a++, b++) {
	if (!*a) return -1;
	if (!*b) return 1;

	if (*a != *b) {
	    a1 = (isascii(*a) && isupper(*a)) ? tolower(*a) : *a;
	    b1 = (isascii(*b) && isupper(*b)) ? tolower(*b) : *b;
	    if (a1 != b1) return b1 - a1;
	}
    }
    return 0;
}


/*
 * do_pointer = get list of numbers of the form
 *
 *                 buttons = NUMBER ...
 *                         ^
 */

static int do_pointer (line, len)
    char *line;
    int len;
{
    int n;
    int i;
    unsigned long val;
    union op *uop;
    struct op_pointer *opp;
    unsigned char buttons[MAXBUTTONCODES];
    int nbuttons;
    char *strval;
    Bool ok;

    if (len < 2 || !line || *line == '\0') {  /* =1 minimum */
	badmsg ("buttons input line", NULL);
	return (-1);
    }

    nbuttons = XGetPointerMapping (dpy, buttons, MAXBUTTONCODES);

    n = skip_space (line, len);
    line += n, len -= n;

    if (line[0] != '=') {
	badmsg ("buttons pointer code list, missing equal sign", NULL);
	return (-1);
    }

    line++, len--;			/* skip = */
    n = skip_space (line, len);
    line += n, len -= n;

    i = 0;
    if (len < 7 || strncmp_nocase (line, "default", 7) != 0) {
	while (len > 0) {
	    n = skip_space (line, len);
	    line += n, len -= n;
	    if (line[0] == '\0') break;
	    n = skip_word (line, len);
	    if (n < 1) {
		badmsg ("skip of word in buttons line:  %s", line);
		return (-1);
	    }
	    strval = copy_to_scratch(line, n);
	    ok = parse_number (strval, &val);
	    if (!ok || val >= MAXBUTTONCODES) {
		badmsg ("value %s given for buttons list", strval);
		return (-1);
	    }
	    buttons[i++] = (unsigned char) val;
	    line += n, len -= n;
	}
    }
    
    if (i > 0 && i != nbuttons) {
	badheader ();
	fprintf (stderr, "number of buttons, must have %d instead of %d\n",
		 nbuttons, i);
	return (-1);
    }

    uop = AllocStruct (union op);
    if (!uop) {
	badmsg ("attempt to allocate a %ld byte pointer opcode",
		(char*) sizeof (struct op_pointer));
	return (-1);
    }
    opp = &uop->pointer;

    opp->type = doPointer;
    opp->count = i;
    for (i = 0; i < opp->count; i++) {
	opp->button_codes[i] = buttons[i];
    }

    add_to_work_queue (uop);

    return (0);
}



/*
 * do_compose - parse off lines of the form
 *
 * compose ANYORALLMODNAMES KEYCODELIST "=" ANYMODNAMES KEYCODE
 * compose ANYORALLMODNAMES keysyms KEYSYMNAMELIST "=" ANYMODNAMES KEYCODE
 * compose ANYORALLMODNAMES KEYCODELIST "=" ANYMODNAMES keysym KEYSYMNAME
 * compose ANYORALLMODNAMES keysyms KEYSYMNAMELIST "=" ANYMODNAMES keysym KEYSYMNAME
 *
 */

static int do_compose (line, len)
    char *line;
    int len;
{
    int n;
    char temp[256];
    int modifier = -3;
    Bool ignore_keysym = False;
    unsigned short *modifierlist;
    int output_modmask = 0;
    KeyCode output_keycode;
    int keycode_count;
    KeyCode *keycodelist;
    struct op_compose_sequence *opk;
    struct op_remove_compose_sequence *oprmc;
    int hexdigit;
    KeyCode keycode;
    Bool isnumeric;
    int column;	    /* junk */
    struct op_compose_numeric *opcn;
    struct op_compose_cancel *opcc;

    if (len < 3 || !line || *line == '\0') {  /* 3=a minimum */
	badmsg ("compose input line", NULL);
	return (-1);
    }
    
    have_compose_chg = True;
    if ((len >= 7) && !strncmp(line, "numeric", 7)) {
	/* skip "numeric" keyword */
	n = skip_word(line, len);
	line += n; len -= n;
	
	if (parse_hex_char(&line, &len, &hexdigit,
			   "compose numeric expression"))
	    return (-1);

	if (line[0] != '=') {
	    badmsg ("compose numeric expression, missing equal sign", NULL);
	    return (-1);
        }
	line++, len--;			/* skip '=' */
	n = skip_space (line, len);
	line += n, len -= n;

	if (parse_key(&line, &len, &keycode, &column,
		      "compose numeric expression"))
	    return (-1);

        opcn = AllocStruct (struct op_compose_numeric);
        if (!opcn) {
	    badmsg ("attempt to allocate a %ld byte compose opcode",
		    (char *) sizeof (struct op_compose_numeric));
	    return (-1);
	}
        opcn->type = doComposeNumeric;
	opcn->hexdigit = hexdigit;
	opcn->keycode = keycode;
	add_to_work_queue ((union op *)opcn);
	return (0);
    }
    else if ((len >= 6) && !strncmp(line, "cancel", 6)) {
	/* skip "cancel" keyword */
	n = skip_word(line, len);
	line += n; len -= n;
	if (parse_keycode (line, len, &keycode,
			   "compose cancel expression keycode value"))
	    return (-1);
        opcc = AllocStruct (struct op_compose_cancel);
        if (!opcc) {
	    badmsg ("attempt to allocate a %ld byte compose cancel opcode",
		    (char *) sizeof (struct op_compose_cancel));
	    return (-1);
	}
	opcc->type = doComposeCancel;
	opcc->keycode = keycode;
	add_to_work_queue ((union op *)opcc);
	return (0);
    }

    if (parse_compose_sequence(&line, &len, &keycode_count, &keycodelist, 
			       &modifierlist, &isnumeric))
	return (-1);

    if (!isnumeric) {
        if (len < 1) {
	    badmsg ("compose expression equal sign missing", NULL);
	    return (-1);
	}
        /* if no keycodes found, error */

	n = skip_space (line+1, len-1);	/* skip over '=' */
	line += n + 1; len -= n + 1;

    	/* look for output modifiers, if any */
	do {
	    if (modifier >= 0) {
		n = n + skip_space (line+n, len-n);
		line += n; len -= n;
		output_modmask |= 1 << modifier;
	    }
	    else if (modifier == -2) {
		/* "none" */
		ignore_keysym = True;
		output_modmask = 0;
	    }
	    /* else first time through */
	    n = skip_chars(line, len);
	    strncpy(temp, line, n);
	    temp[n] = 0;
	} while ((modifier = parse_modifier_or_none(temp, n)) != -1);

	/* look for output keycode or keysym */
	if (parse_key(&line, &len, &output_keycode, &column,
	    "compose expression output"))
	    return (-1);
	/* adjust modifiers for keysym column (currently just Shift) */
	if (!ignore_keysym && XColumnToModifier(dpy, column, &modifier))
	    output_modmask |= modifier;
    }

    oprmc = AllocStruct (struct op_remove_compose_sequence);
    if (!oprmc) {
	badmsg ("attempt to allocate a %ld byte compose opcode",
		(char *) sizeof (struct op_remove_compose_sequence));
	return (-1);
    }
    opk = AllocStruct (struct op_compose_sequence);
    if (!opk) {
	badmsg ("attempt to allocate a %ld byte compose opcode",
		(char *) sizeof (struct op_compose_sequence));
	return (-1);
    }

    /* First, delete any existing conflicting compose sequences.  Could
	be a bit faster if we pre-pended new Compose sequences, but the 
	property would grow */
    if (!fast) {
	/* goes much faster if we are sure there is none to delete */
	oprmc->type = doRemoveComposeSequence;
	oprmc->modifiers = modifierlist;
	oprmc->count = keycode_count;
	oprmc->keycodes = keycodelist;

        add_to_work_queue ((union op *)oprmc);
/*	new_seq_count--; we currently don't reuse space */
	
    }

    opk->type = doComposeSequence;
    opk->modifiers = modifierlist;
    opk->count = keycode_count;
    opk->keycodes = keycodelist;
    if (isnumeric) {
	opk->output_modmask = opk->output_keycode = 0;
	/* don't care */
    }
    else {
	opk->output_modmask = output_modmask;
	opk->output_keycode = output_keycode;
    }

    add_to_work_queue ((union op *)opk);
    new_seq_count++;
    return (0);
}



/*
 * do_execute - parse off lines of the form
 *
 *           execute
 *                  ^
 *
 */

static int do_execute (line, len)
    /*ARGSUSED*/
    char *line;
    /*ARGSUSED*/
    int len;
{
    int status = False;

    if (dontExecute) {
	print_work_queue ();
	/* print what we got so far */
    }
    else if (errors == 0) {
	initialize_compose();
	/* will initialize the first time we have a compose */
	status = execute_work_queue ();
	/* compose properties do not need to be updated until end */
	refreshMapping();
	/* but we must make sure that the keycode mapping gets updated.  This
	   will cause an extra XSync if execute occurs after a non-keycode
	   expression - tough. */
    }
    /* do nothing if errors; message will be printed at final execute */
    work_queue.head = work_queue.tail = NULL;
    return (status);
}

/*
 * get_keysym_list - parses the rest of the line into a keysyms assumes
 * that the = sign has been parsed off but there may be leading whitespace
 *
 *                 keysym ...
 *                 ^
 *
 * this involves getting the word containing the keysym, checking its range,
 * and adding it to the list.
 */

static int get_keysym_list (line, len, np, kslistp)
    char *line;
    int len;
    int *np;
    KeySym **kslistp;
{
    int havesofar, maxcanhave;
    KeySym *keysymlist;

    *np = 0;
    *kslistp = NULL;

    if (len == 0) return (0);		/* empty list */

    havesofar = 0;
    maxcanhave = 4;			/* most lists are small */
    keysymlist = (KeySym *) calloc (1,maxcanhave * sizeof (KeySym));
    if (!keysymlist) {
	badmsg ("attempt to allocate %ld byte initial keysymlist",
		(char*) (maxcanhave * sizeof (KeySym)));
	return (-1);
    }

    while (len > 0) {
	KeySym keysym;
	int n;
	char *tmpname;
	Bool ok;

	n = skip_space (line, len);
	line += n, len -= n;

	n = skip_chars (line, len);
	if (n < 0) {
	    badmsg ("keysym name list", NULL);
	    return (-1);
	}

	ok = parse_keysym (line, n, &tmpname, &keysym);
	line += n, len -= n;
	if (!ok) {
	    badmsg ("keysym name '%s' in keysym list", tmpname);
	    /* do NOT return here, look for others */
	    continue;
	}

	/*
	 * Do NOT test to see if the keysym translates to a keycode or you
	 * won't be able to assign new ones....
	 */

	/* grow the list bigger if necessary */
	if (havesofar >= maxcanhave) {
	    maxcanhave *= 2;
	    keysymlist = (KeySym *) realloc (keysymlist,
					     maxcanhave * sizeof (KeySym));
	    if (!keysymlist) {
		badmsg ("attempt to grow keysym list to %ld bytes",
			(char*) (maxcanhave * sizeof (KeySym)));
		return (-1);
	    }
	}

	/* and add it to the list */
	keysymlist[havesofar++] = keysym;
    }

    *kslistp = keysymlist;
    *np = havesofar;
    return (0);
}


#ifdef later
/*
 * check_special_keys - run through list of keysyms and generate "add" or
 * "remove" commands for for any of the key syms that appear in the modifier
 * list.  this involves running down the modifier map which is an array of
 * 8 by map->max_keypermod keycodes.
 */

static void check_special_keys (keycode, n, kslist)
    KeyCode keycode;
    int n;
    KeySym *kslist;
{
    int i;				/* iterator variable */
    KeyCode *kcp;			/* keycode pointer */

    /*
     * walk the modifiermap array.  since its dimensions are not known at
     * compile time, we have to walk it by hand instead of indexing.  this
     * is why it is initialized outside the loop, but incremented inside the
     * second loop.
     */

    kcp = map->modifiermap;		/* start at beginning and iterate */
    for (i = 0; i < 8; i++) {		/* there are 8 modifier keys */
	int j;

	for (j = 0; j < map->max_keypermod; j++, kcp++) {
	    KeySym keysym;
	    int k;

	    if (!*kcp) continue;	/* only non-zero entries significant */

	    /*
	     * check to see if the target keycode is already a modifier; if so,
	     * then we have to remove it
	     */
	    if (keycode == *kcp) {
		make_remove (i, keycode);
	    }

	    /*
	     * now, check to see if any of the keysyms map to keycodes
	     * that are in the modifier list
	     */
	    for (k = 0; k < n; k++) {
		KeyCode kc = XKeysymToKeycode (dpy, kslist[k]);

		if (kc == *kcp) {	/* yup, found one */
		    /*
		     * have to generate a remove of the CURRENT keycode
		     * and then an add of the new KEYCODE
		     */
		    make_remove (i, kc);  /* modifier, keycode */
		    make_add (i, kslist[k]);  /* modifier, keysym */
		}
	    }
	}
    }
    return;
}
#endif

/*
 * print_work_queue - disassemble the work queue and print it on stdout
 */

void print_work_queue ()
{
    union op *op;

    printf ("! dump of work queue\n");
    for (op = work_queue.head; op; op = op->generic.next) {
	print_opcode (op);
    }
    return;
}

void print_opcode (op)
    union op *op;
{
    int i;

    printf ("        ");
    switch (op->generic.type) {
      case doKeycode:
	printf ("keycode 0x%lx =", (long) op->keycode.target_keycode);
	for (i = 0; i < op->keycode.count; i++) {
	    char *name = XKeysymToString (op->keycode.keysyms[i]);

	    printf (" %s", name ? name : "BADKEYSYM");
	}
	printf ("\n");
	break;
      case doAddModifier:
	printf ("add %s =", modifier_table[op->addmodifier.modifier].name);
	for (i = 0; i < op->addmodifier.count; i++) {
	    char *name = XKeysymToString (op->addmodifier.keysyms[i]);
	    printf (" %s", name ? name : "BADKEYSYM");
	}
	printf ("\n");
	break;
      case doRemoveModifier:
	printf ("remove %s = ",
		modifier_table[op->removemodifier.modifier].name);
	for (i = 0; i < op->removemodifier.count; i++) {
	    printf (" 0x%lx", (long) op->removemodifier.keycodes[i]);
	}
	printf ("\n");
	break;
      case doClearModifier:
	printf ("clear %s\n", modifier_table[op->clearmodifier.modifier].name);
	break;
      case doComposeSequence:
	printf ("compose sequence: keycodes/modifiers ");
	for (i = 0; i < op->compose_sequence.count; i++) {
	    printf ("0x%lx/", (long) op->compose_sequence.keycodes[i]);
	    printf (print_modifiers(op->compose_sequence.modifiers[i]));
	}
	printf ("\n        output modifier %soutput keycode 0x%lx\n",
		print_modifiers(op->compose_sequence.output_modmask),
		(long) op->compose_sequence.output_keycode);
	break;
      case doRemoveComposeSequence:
	printf ("remove compose sequence: keycodes/modifiers ");
	for (i = 0; i < op->remove_compose_sequence.count; i++) {
	    printf("0x%lx/", (long) op->remove_compose_sequence.keycodes[i]);
	    printf(print_modifiers(op->remove_compose_sequence.modifiers[i]));
	}
	printf ("\n");
	break;
      case doComposeNumeric:
	printf ("compose numeric: hex digit %x, keycode %d\n",
		op->compose_numeric.hexdigit, op->compose_numeric.keycode);
	break;
      case doRemoveComposeNumeric:
	printf ("remove compose numeric: hex digit %x, keycode %d\n",
		op->remove_compose_numeric.hexdigit, 
		op->remove_compose_numeric.keycode);
	break;
      case doComposeCancel:
	printf ("compose cancel: keycode %d\n",
		op->compose_cancel.keycode);
	break;
      case doRemoveComposeCancel:
	printf ("remove compose cancel: keycode %d\n",
		op->remove_compose_cancel.keycode);
	break;
      case doClearComposeSequence:
	printf ("clear compose sequences\n");
	break;
      case doClearComposeNumeric:
	printf ("clear compose numeric\n");
	break;
      case doClearComposeCancel:
	printf ("clear compose cancel\n");
	break;
      default:
	printf ("! unknown opcode %d\n", op->generic.type);
	break;
    }				/* end switch */
    return;
}

/*
 * execute_work_queue - do the real meat and potatoes now that we know what
 * we need to do and that all of the input is correct.
 */

static int exec_keycode(), exec_add(), exec_remove(), exec_clear();
static int exec_pointer();
static int exec_compose(), exec_remove_compose_sequence();
static int exec_compose_numeric(), exec_remove_compose_numeric();
static int exec_compose_cancel(), exec_remove_compose_cancel();
static int exec_clear_compose();

static int refreshMapping()
{
    XSync (dpy, 0);
    while (XEventsQueued (dpy, QueuedAlready) > 0) {
	XEvent event;
	XNextEvent (dpy, &event);
	if (event.type == MappingNotify) {
	    /* read all MappingNotify events */
	    while (XCheckTypedEvent (dpy, MappingNotify, &event)) ;
	    XRefreshKeyboardMapping (&event);
	} else {
	    fprintf (stderr, "%s:  unknown event %ld\n", 
	    	     ProgramName, (long) event.type);
	}
    }
}

int execute_work_queue ()
{
    union op *op;
    int errors;
    Bool update_map = False;
    enum opcode lastop;
    /* If we did a do_execute after a keycode event, this avoids an extra
       refreshMapping() here. */

    if (verbose) {
	printf ("!\n");
	printf ("! executing work queue\n");
	printf ("!\n");
    }

    errors = 0;
    lastop = doClearModifier;				/* fake it for op test */

    for (op = work_queue.head; op; op = op->generic.next) {
	if (verbose) print_opcode (op);

	/* check to see if we have to update the keyboard mapping */
	if (lastop == doKeycode && op->generic.type != doKeycode) {
	  refreshMapping();
	}

	switch (op->generic.type) {
	  case doKeycode:
	    if (exec_keycode ((struct op_keycode *) op) < 0) errors++;
	    break;
	  case doAddModifier:
	    if (exec_add ((struct op_addmodifier *) op) < 0) errors++;
	    else update_map = True;
	    break;
	  case doRemoveModifier:
	    if (exec_remove ((struct op_removemodifier *) op) < 0) errors++;
	    else update_map = True;
	    break;
	  case doClearModifier:
	    if (exec_clear ((struct op_clearmodifier *) op) < 0) errors++;
	    else update_map = True;
	    break;
	  case doPointer:
	    if (exec_pointer ((struct op_pointer *) op) < 0) errors++;
	    break;
	  case doComposeSequence:
	    if (exec_compose ((struct op_compose_sequence *) op) < 0) errors++;
	    break;
	  case doRemoveComposeSequence:
	    if (exec_remove_compose_sequence (
		(struct op_remove_compose_sequence *) op) < 0) errors++;
	    break;
	  case doComposeNumeric:
	    if (exec_compose_numeric (
		(struct op_compose_numeric *) op) < 0) errors++;
	    break;
	  case doRemoveComposeNumeric:
	    if (exec_remove_compose_numeric (
		(struct op_remove_compose_numeric *) op) < 0) errors++;
	    break;
	  case doComposeCancel:
	    if (exec_compose_cancel (
		(struct op_compose_cancel *) op) < 0) errors++;
	    break;
	  case doRemoveComposeCancel:
	    if (exec_remove_compose_cancel (
		(struct op_remove_compose_cancel *) op) < 0) errors++;
	    break;
	  case doClearComposeSequence:
	  case doClearComposeNumeric:
	  case doClearComposeCancel:
	    if (exec_clear_compose ((struct op_clear_compose *) op) < 0) 
		errors++;
	    break;
	  default:
	    fprintf (stderr, "%s:  unknown opcode %d\n", 
		     ProgramName, op->generic.type);
	    break;
	}
	lastop = op->generic.type;

    }

    if (update_map) {
	if (UpdateModifierMapping (map) < 0) errors++;
    }

    return (errors > 0 ? -1 : 0);
}

static int exec_keycode (opk)
    struct op_keycode *opk;
{
    if (opk->count == 0) {
	KeySym dummy = NoSymbol;
	XChangeKeyboardMapping (dpy, opk->target_keycode, 1,
				&dummy, 1);
    } else {
	XChangeKeyboardMapping (dpy, opk->target_keycode, opk->count, 
				opk->keysyms, 1);
    }
    return (0);
}

static int exec_add (opam)
    struct op_addmodifier *opam;
{
    int i;
    int status;

    status = 0;
    for (i = 0; i < opam->count; i++) {
	KeyCode kc = XKeysymToKeycode (dpy, opam->keysyms[i]);

	if (AddModifier (&map, kc, opam->modifier) < 0)
	  status = -1;
    }
    return (status);
}

static int exec_remove (oprm)
    struct op_removemodifier *oprm;
{
    int i;
    int status;

    status = 0;
    for (i = 0; i < oprm->count; i++) {
	if (RemoveModifier (&map, oprm->keycodes[i], oprm->modifier) < 0)
	  status = -1;
    }
    return (status);
}

static int exec_clear (opcm)
    struct op_clearmodifier *opcm;
{
    return (ClearModifier (&map, opcm->modifier));
}


static int exec_pointer (opp)
    struct op_pointer *opp;
{
    return (SetPointerMap (opp->button_codes, opp->count));
}


void LengthenComposeTable(new_seq_space)
    unsigned int new_seq_space;
{
    if (seq_space) {
	ComposeTable.modifiers = (unsigned short *)realloc(
	ComposeTable.modifiers, 
	  sizeof(unsigned short) * new_seq_space * ComposeTable.max_keycodes);
	ComposeTable.sequences = (KeyCode *)realloc(
	ComposeTable.sequences,
	    sizeof(KeyCode) * new_seq_space * ComposeTable.max_keycodes);
	ComposeTable.output_modifiers = (unsigned int *)realloc(
	ComposeTable.output_modifiers, sizeof(unsigned int) * new_seq_space);
	ComposeTable.output_keycodes = (KeyCode *)realloc(
	ComposeTable.output_keycodes, sizeof(KeyCode) * new_seq_space);
    }
    else {
	ComposeTable.modifiers = (unsigned short *)calloc(1,
				    sizeof(unsigned short) * new_seq_space);
	ComposeTable.sequences = (KeyCode *)calloc(1,
				    sizeof(KeyCode) * new_seq_space);
	ComposeTable.output_modifiers = (unsigned int *)calloc(1,
				    sizeof(unsigned int) * new_seq_space);
	ComposeTable.output_keycodes = (KeyCode *)calloc(1,
				    sizeof(KeyCode) * new_seq_space);
    }
    seq_space = new_seq_space;    
}

void WidenComposeTable(added_keycodes)
    unsigned int added_keycodes;
{
    int i, j;
    KeyCode *new_sequences;
    unsigned short *new_modifiers;
    int new_width = ComposeTable.max_keycodes + added_keycodes;

    new_sequences = (KeyCode *)calloc(1,sizeof(KeyCode) *
					     seq_space * new_width);
    new_modifiers = (unsigned short *)calloc(1,sizeof(unsigned short) * 
					     seq_space * new_width);
	    /* really need to use Xmalloc, so XFreeKBCompose can XFree */
    for (i = 0; i < ComposeTable.num_composes; i++)
	for (j = 0; j < ComposeTable.max_keycodes; j++) {
	    new_sequences[(i * new_width) + j] = 
		ComposeTable.sequences[(i * ComposeTable.max_keycodes) + j];
	    new_modifiers[(i * new_width) + j] = 
		ComposeTable.modifiers[(i * ComposeTable.max_keycodes) + j];
	}
    for (i = 0; i < ComposeTable.num_composes; i++)
	for (j = ComposeTable.max_keycodes; j < new_width; j++) {
	    new_sequences[(i * new_width) + j] = (KeyCode)NULL;
	    new_modifiers[(i * new_width) + j] = (unsigned int)NULL;
	}
    ComposeTable.max_keycodes += added_keycodes;
/*    XFree(ComposeTable.sequences); */
    ComposeTable.sequences = new_sequences;
    ComposeTable.modifiers = new_modifiers;
}

static int exec_compose (opk)
    struct op_compose_sequence *opk;
{
    int i, next_seq;

    if ((new_seq_count + ComposeTable.num_composes) > seq_space)
	LengthenComposeTable(new_seq_count + ComposeTable.num_composes);
    if (max_compose_keycodes > ComposeTable.max_keycodes)
	WidenComposeTable(max_compose_keycodes - ComposeTable.max_keycodes);
/* Note - above tests could be done once before calling execute_work_queue */

    next_seq = ComposeTable.num_composes * ComposeTable.max_keycodes;
    bcopy(opk->modifiers, &ComposeTable.modifiers[next_seq], 
	  opk->count * sizeof (ComposeTable.modifiers[0]));
    bcopy(opk->keycodes, &ComposeTable.sequences[next_seq], 
	  opk->count * sizeof (KeyCode));
    for (i = opk->count; i < ComposeTable.max_keycodes; i++)
	ComposeTable.sequences[next_seq + i] = (KeyCode)NULL;
    ComposeTable.output_modifiers[ComposeTable.num_composes] = 
	    opk->output_modmask;
    ComposeTable.output_keycodes[ComposeTable.num_composes] = 
	    opk->output_keycode;
    ComposeTable.num_composes++;
        
    return (0);
}

static int exec_remove_compose_sequence (opk)
    struct op_remove_compose_sequence *opk;
{
    int i, j;

    /* eliminate any sequences that start with the specified sequence */
    /* it would be nice to compress them out of the table, but why bother? */
    for (i = 0; i < (ComposeTable.num_composes * ComposeTable.max_keycodes);
	 i += ComposeTable.max_keycodes) {
	/* check all entries */
	for (j = 0; j < opk->count; j++) {
	    if (((ComposeTable.modifiers[i + j] != opk->modifiers[j]) &&
		 (ComposeTable.modifiers[i + j] != AnyModifier) &&
		 (opk->modifiers[j] != AnyModifier)) ||
		(ComposeTable.sequences[i + j] != opk->keycodes[j]))
		break;
	}
	if (j == opk->count)
	    /* all keys matched */
	    ComposeTable.sequences[i] = 0;
	    /* zero out first keycode; sufficient to kill it */
    }	     

    return (0);
}

static int exec_compose_numeric (opk)
    struct op_compose_numeric *opk;
{
    int i;
    char msg[500];

    for (i = 0; i < KEYCODES_PER_HEX_CHAR; i++) 
	if (ComposeTable.numeric_keycodes[
		(opk->hexdigit * KEYCODES_PER_HEX_CHAR) + i] == 0)
	    break; 
    if (i == KEYCODES_PER_HEX_CHAR) {
	(void) sprintf (msg, "numeric compose expression;\nlimit of %d numeric keycodes exceeded;\nmust delete one of the keycodes %d, %d, or %d for numeric character '%x'\nbefore adding a new one\n",
   KEYCODES_PER_HEX_CHAR,
   ComposeTable.numeric_keycodes[(opk->hexdigit * KEYCODES_PER_HEX_CHAR)],
   ComposeTable.numeric_keycodes[(opk->hexdigit * KEYCODES_PER_HEX_CHAR) + 1],
   ComposeTable.numeric_keycodes[(opk->hexdigit * KEYCODES_PER_HEX_CHAR) + 2],
   opk->hexdigit); 
        badmsg (msg, NULL);
	return (-1);
    }
    ComposeTable.numeric_keycodes[
	    (opk->hexdigit * KEYCODES_PER_HEX_CHAR) + i] = opk->keycode;
    return (0);
}

static int exec_remove_compose_numeric (opk)
    struct op_remove_compose_numeric *opk;
{
    int i;

    for (i = 0; i < KEYCODES_PER_HEX_CHAR; i++) 
	/* check all entries */
	if (ComposeTable.numeric_keycodes[
		(opk->hexdigit * KEYCODES_PER_HEX_CHAR) + i] == opk->keycode)
	    ComposeTable.numeric_keycodes[
		(opk->hexdigit * KEYCODES_PER_HEX_CHAR) + i] = 0;
    /* silently ignore if it does not exist */
    return (0);
}

static int exec_compose_cancel (opk)
    struct op_compose_cancel *opk;
{
    if (ComposeTable.num_cancel_keycodes >= cancel_space) { 
	/* arbitrarily double space */
	if (cancel_space == 0) {
	    cancel_space = 1;
	    ComposeTable.cancel_keycodes = (KeyCode *)calloc(1,sizeof(KeyCode));
	}
	else {
	    cancel_space = ComposeTable.num_cancel_keycodes << 1;
	    ComposeTable.cancel_keycodes = (KeyCode *) realloc(
			    ComposeTable.cancel_keycodes,
		       sizeof(KeyCode) * cancel_space); 
	    if (!ComposeTable.cancel_keycodes) {
		badmsg ("attempt to reallocate cancel keycode space to %ld", 
			(char *) (cancel_space * sizeof (KeyCode)));
		return (-1); 
	    }
	}
    }
    ComposeTable.cancel_keycodes[ComposeTable.num_cancel_keycodes++] = 
	    opk->keycode; 
    return (0);
}

static int exec_remove_compose_cancel (opk)
    struct op_remove_compose_cancel *opk;
{
    int i;

    for (i = 0; i < ComposeTable.num_composes; i++)
	if (ComposeTable.cancel_keycodes[i] = opk->keycode)
	    ComposeTable.cancel_keycodes[i] = 0; 
	    /* non-keycode, no effect; eliminate any and all */
    return (0);
}

static int exec_clear_compose (opk)
    struct op_clear_compose *opk;
{

    switch (opk->type) {
      case doClearComposeSequence:
	ComposeTable.num_composes = 0;
	/* space remains, all composes processed to this point are eliminated */
	break;
      case doClearComposeNumeric:
        bzero(ComposeTable.numeric_keycodes, 48);
	break;
      case doClearComposeCancel:
	ComposeTable.num_cancel_keycodes = 0;
	break;
    }        
    return (0);
}

UpdateComposeProperty ()
{
    KeySym none;

    /* the kludge - to make clients call XRefreshKeyboarMapping which will
       reread compose property, call XChangeKeyboardMapping.  XXX Note that
       this overhead makes it imperative that we process all real mapping
       changes at once!  Eventually, we will call this once for both Compose
       and the keyboard rearchitecture extension, which is why it is done here
       and not in XSetKBCompose. */
    XSetKBCompose(dpy, DefaultRootWindow(dpy), &ComposeTable);
    XChangeKeyboardMapping (dpy, dpy->min_keycode, 1, &none, 0);
}

void print_modifier_map ()
{
    PrintModifierMapping (map, stdout);
    return;
}

void print_key_table ()
{
    PrintKeyTable (stdout);
    return;
}

void print_pointer_map ()
{
    PrintPointerMap (stdout);
    return;
}
