-- Skeleton of an options processor

indexing
	
	type: skeleton;
	author: "$Author: Neil_Wilson $", "(c) 1993";
	date: "$Date: 1993/06/24 19:30:11 $";
	revision: "$Revision: 1.4 $";
	licence: "GNU Library GPL - see file README.LIB";
	log: 
	-- $Log: opt_proc.e%v $
	-- Revision 1.4  1993/06/24  19:30:11  Neil_Wilson
	-- Generalised due to multiple option matcher
	--
	-- Revision 1.3  1993/06/20  12:17:33  Neil_Wilson
	-- Decoupled matcher from processor
	--
	-- Revision 1.2  1993/06/10  20:36:44  Neil_Wilson
	-- Generalised inheritance structure
	--
	-- Revision 1.1  1993/06/07  21:24:31  Neil_Wilson
	-- Initial revision
	--
	
deferred class OPTION_PROCESSOR

inherit

	OPTION_ATTRIBUTES;

feature -- storage areas

	arguments: COLLECTION [STRING];
		-- Processed strings that weren't options.

feature -- Processor engine controls.

	process (arg: STRING) is
		-- Run 'arg' through the processor.
	require
		processable: arg /= Void;
		valid_state: not has_invalid_option
	do
		running := true;
		scan_for (arg);
		if is_option_string then
			option_transition;
		end;
		inspect
			state
		when Argument then
			process_argument (arg)
		when Option then
			process_option (arg);
		when Parameter, Optional then
			process_parameter (arg);
		when Invalid then
			;
		end;
	ensure 
		processed:
			has_invalid_option /= old has_invalid_option or else
			arguments.count /= old arguments.count or else
			not deep_equal(options, old options);
		processor_on: running;
	end -- process

	running : BOOLEAN;
		-- Is the processor running?

	stop is
		-- Signal the end of the process run
	require
		processor_on: running
		valid_state: not has_invalid_option
	do
		inspect (state)
			when Optional then
				options.put (Void, hold_option);
				state := Argument;
			when Parameter then
				has_invalid_option := true;
				invalid_option := clone(Parameter_missing);
				state := Invalid;
			else
				state := Argument;
		end;
		running := false;
	ensure
		valid_state: state = Argument or state = Invalid
		processor_off: not running
	end -- stop
		
	reset is
		-- Reset the processor.
	do
		arguments.make (false);
		options.make;
		has_invalid_option := false;
		running := false;
		state := Argument;
	ensure
		arguments.empty;
		options_empty: options.count = 0;
		valid_state: not has_invalid_option and state = Argument;
		processor_off: not running;
	end -- reset

feature {NONE} -- Processor state controls

	Argument, Invalid, Option, Optional, Parameter: INTEGER is unique;
		-- Possible states of the processor;

	state: INTEGER;
		-- Current state of the processor

	match: OPTION_MATCHER
		-- What an option looks like and how to deal with it.

	process_option (arg: STRING) is
	require
		argument_not_void: arg /= Void
		processor_running: running;
		correct_state: state = Option
		correct_matcher: match.is_option_string (arg);
	do
		from
			match.search (match.remove_option_marker (arg));
			alter_state
		variant
			search_loop: match.remaining.count
		until
			match.remaining.empty or has_invalid_option
		loop
			option_transition
			match.search (match.remaining);
			alter_state
		end
	ensure
		valid_new_state: state /= Option;
	end -- process_option;

	process_parameter (arg: STRING) is
	require
		processor_running: running;
		correct_state: state = Optional or state = Parameter
	do
		options.put (arg, hold_option);
		state := Argument;
	ensure
		valid_new_state: state = Argument
	end -- process_parameter

	process_argument (arg: STRING) is
	require
		processor_running: running;
		correct_state: state = Argument
	do
		arguments.add (arg);
	ensure
		valid_new_state: state = Argument
	end -- process_argument;

	option_transition is
		-- Moves the system into the Option state
	require
		valid_state: state /= Invalid
	do
		inspect
			state
		when Argument, Option then
			state := Option
		when Optional then
			options.put (Void, hold_option);
			state := Option
		when Parameter then
			has_invalid_option := true;
			invalid_option := clone (Parameter_missing);
			state := Invalid;
		end;
	ensure
		valid_new_state: state = Option or else state = Invalid
	end -- option_transition

	alter_state is
		-- Changes the state from Option using the matcher
	require
		in_option_state: state = Option
	do
		if match.simple_found then
			options.put (Void, match.last_search);
			state := Argument
		elseif match.optional_found then
			state := Optional
			hold_option := match.last_search
		elseif match.parameter_found then
			state := Parameter
			hold_option := match.last_search
		else
			has_invalid_option := true;
			invalid_option := clone (Invalid_message);
			invalid_option.append (match.add_option_marker (match.last_search))
			state := Invalid
		end
	ensure
		state /= Option
		held_option: (state = Optional or state = Parameter) implies
		    hold_option = match.last_search
	end -- alter_state

	hold_option: STRING
		-- The last matched option string.

feature {NONE} -- option matcher manipulators

	scan_for (arg: STRING) is
		-- Look for an option match and make the matcher available.
	require
		arg_available: arg /= Void
	deferred
	ensure
		scan_consistent: match = Void implies not is_option_string;
		scan_consistent: match /= Void implies
		    (is_option_string = match.is_option_string (arg))
	end -- scan_for

	is_option_string: BOOLEAN;
		-- Result of last scan for.

feature {NONE} -- Error messages.

	Parameter_missing: STRING is "Required parameter missing";

	Invalid_message: STRING is "Invalid option "
	
invariant
	
	arguments_available: arguments /= Void
	
	valid_state: state >= Argument and state <= Parameter;

	consistent_invalid_state: state = Invalid implies has_invalid_option;
	consistent_valid_state: state /= Invalid implies not has_invalid_option

	Error_messages_protected: invalid_option /= Invalid_message and then
		invalid_option /= Parameter_missing
		-- This may seem a strange assertion, but I want to make
		-- sure that the 'constant' message objects are not
		-- accidentally passed to the outside world (where they
		-- could get mangled).

end -- class OPTION_PROCESSOR
