class   STREAMIO

   inherit
      EXCEPTION
         end

   creation
      open

      -----------------------------------------------------------

   feature {ANY}

      undefined, read_only,
      write_only, read_write,   : INTEGER is unique

      stdout                    : INTEGER is 1
      stdin                     : INTEGER is 2
      stderr                    : INTEGER is 3
      stdio                     : INTEGER is 4
      file                      : INTEGER is 5      

      -----------------------------------------------------------

      open (f_name, f_mode: STRING) is   
      -- open file with attributes, special handling for       
      -- "stderr", "stdout", "stdin" and "stdio" (bidirectional,          
      -- both stdio and stdout on one file (standard console)  

         require
            parameter_void: f_name /= void and then f_mode /= void
            already_open: not is_open

         do
            if f_name.is_equal ("stdout") then
               name := stdout
               direction := write_only
            elseif f_name.is_equal ("stdio") then
               name := stdio
               direction := read_write
            elseif f_name.is_equal ("stderr") then
               name := stderr
               direction := write_only
            elseif f_name.is_equal ("stdin") then
               name := stdin
               direction := read_only
            else
               name := file
               if index (f_mode, '+') > 0 then
                  direction := read_write
               elseif index (f_mode, 'r') > 0 then
                  direction := read_only
               elseif index (f_mode, 'w') > 0 or else 
                  index (f_mode, 'a') > 0 then
                  direction := write_only
               else
                  raise ("streamio", "wrong mode", 0, current)
               end -- if        
            end -- if        

            file_name := clone (f_name)
            file_ptr := open_file (name, f_name, f_mode)
            is_open := true

         ensure
            direction /= undefined
         end -- make      

   -----------------------------------------------------------

      close is
      -- close file, special handling for
      -- "stderr", "stdout", "stdin" and "stdio"    

         require
            not_open: is_open

         do
            if file_name.is_equal ("stdout") or else
               file_name.is_equal ("stdio") or else
               file_name.is_equal ("stderr") or else
               file_name.is_equal ("stdin") then
               -- do nothing
            else
               close_file
            end -- if        
            is_open := false
         end -- close     

   -----------------------------------------------------------

      last_char : CHARACTER 
      -- contains last character read by get_char   

      -----------------------------------------------------------

      get_char is
      -- Read a char and assign it to `last_char'.  

         require
            file_not_open: is_open = true
            wrong_direction: direction /= write_only 
         external "C"
            alias    "STRM_get_char"
         end

      -----------------------------------------------------------

      last_string : string 
      -- contains last string read by get_string    

      -----------------------------------------------------------

      get_string is
      -- Read a string (terminated by NEWLINE or EOF)          
      -- and assign it to `last_string'. 
      -- NOTE: Every call will allocate a new string
      -- result is empty if the line is empty.      
      -- the result contains no NEWLINE separators  

         require
            file_not_open: is_open = true
            wrong_direction: direction /= write_only 

         local
            ti: INTEGER
            chr: CHARACTER

         do
            from
               get_char
               chr := last_char
               !!last_string.make (80)
            until
               end_of_input or else chr = '%N'
            loop
               ti := ti + 1
               if ti > last_string.count then
                  last_string.append ("         ")
               end -- if        
               last_string.put (chr, ti)
               -- get next char        
               get_char
               chr := last_char
            end -- loop      
            if ti = 0 then
               last_string := clone ("")
            else
               last_string := last_string.substring (1, ti)
            end -- if        
         end -- get_string           

   -----------------------------------------------------------

      end_of_input : BOOLEAN is
      -- Has end-of-input been reached   

         require
            file_not_open: is_open = true
         external "C"
            alias   "STRM_end_of_input"
         end

      -----------------------------------------------------------

      put_string (s : string) is
      -- Print `s' 

         require
            file_not_open: is_open = true
            wrong_direction: direction /= read_only 
         external "C"
            alias    "STRM_put_string"
         end

      -----------------------------------------------------------

      put_char (c : character) is
      -- Print `c' 

         require
            file_not_open: is_open = true
            wrong_direction: direction /= read_only 
         external "C"
            alias    "STRM_put_char"
         end

      -----------------------------------------------------------

      put_newline is
      -- Print newline character         

         require
            file_not_open: is_open = true
         external "C"
            alias    "STRM_put_newline"
         end

      -----------------------------------------------------------

      file_name:  string
      direction:  integer

      -----------------------------------------------------------

   feature {NONE}
      -- only for internal use

      file_ptr:   POINTER     -- space for FILE pointer          
      name:       INTEGER     -- internal code stdout .. file    
      is_open:    BOOLEAN     -- is file open ?       

      open_file (tp: INTEGER; n, m: STRING): POINTER is
         external "C"
            alias    "STRM_open_file"
         end

      close_file is
         external "C"
            alias    "STRM_close_file"
         end

      index (s: STRING, c: CHARACTER): INTEGER is

         require
            string_not_void: s /= void

         local
            i: INTEGER

         do
            from 
               i := 1
            until 
               Result > 0 or else i > s.count
            loop
               if s @ i = c then
                  Result := i
               end
               i := i + 1
            end -- loop      
         end -- index     

   end -- class STREAMIO       
