/*
 * @(#)BreakpointBrowser.java	1.0alpha (18 Oct 1995)
 *
 * Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
 * Unpublished work.  All Rights Reserved.
 *
 * The software contained on this media is the property of the
 * DSTC Pty Ltd.  Use of this software is strictly in accordance
 * with the license agreement in the accompanying LICENSE.DOC file.
 * If your distribution of this software does not contain a
 * LICENSE.DOC file then you have no rights to use this software
 * in any manner and should contact DSTC at the address below
 * to determine an appropriate licensing arrangement.
 *
 *     DSTC Pty Ltd
 *     Level 7, Gehrmann Labs
 *     University of Queensland
 *     St Lucia, 4072
 *     Australia
 *     Tel: +61 7 3365 4310
 *     Fax: +61 7 3365 4311
 *     Email: enquiries@dstc.edu.au
 *
 * This software is being provided "AS IS" without warranty of
 * any kind.  In no event shall DSTC Pty Ltd be liable for
 * damage of any kind arising out of or in connection with
 * the use or performance of this software.
 */

package dstc.espresso.debug;

import java.awt.*;

/** 
 * List of breakpoints
 *
 * @version	1.0alpha (18 Oct 1995)
 * @author	Ted Phelps
 */

public class BreakpointBrowser extends Frame implements PollCallback
{
    /** The debugger */
    Debugger debugger;

    /** Current list of breakpoint names */
    String breakpointNames[];

    /** The UI list of breakpoints */
    List breakpointList;

    /** The class name text */
    TextField className;

    /** The method name text */
    TextField methodName;


    /** Constructs a new breakpoint browser */
    BreakpointBrowser(Debugger debugger)
    {
	super("Breakpoint");
	this.debugger = debugger;

	breakpointNames = new String[0];
	initializeFrame();
    }

    /** Places the widgets in the frame */
    private void initializeFrame()
    {
	Panel panel, subpanel, minipanel;

	setLayout(new GridLayout(1, 2));

	breakpointList = new List();
	add(breakpointList);

	panel = new Panel();
	panel.setLayout(new BorderLayout());

	subpanel = new Panel();
	subpanel.setLayout(new GridLayout(2, 1));

	minipanel = new Panel();
	minipanel.setLayout(new FlowLayout());
	minipanel.add(new Label("Class"));
	minipanel.add(className = new TextField("", 20));
	subpanel.add(minipanel);

	minipanel = new Panel();
	minipanel.setLayout(new FlowLayout());
	minipanel.add(new Label("Method"));
	minipanel.add(methodName = new TextField("", 20));
	subpanel.add(minipanel);
	panel.add("Center", subpanel);

	subpanel = new Panel();
	subpanel.setLayout(new GridLayout(1, 2));
	subpanel.add(new Button("Set Breakpoint"));
	subpanel.add(new Button("Clear Breakpoint"));
	panel.add("South", subpanel);

	add(panel);
	pack();
    }


    /** Updates the list of breakpoint names if necessary */
    public void refreshBreakpointNames() throws Exception
    {
	String oldNames[] = breakpointNames;
	breakpointNames = debugger.debugger.listBreakpoints();

	if (oldNames.length != breakpointNames.length)
	{
	    updateBreakpointNames();
	    return;
	}

	for (int i = 0; i < breakpointNames.length; i++)
	{
	    if (! breakpointNames[i].equals(oldNames[i]))
	    {
		updateBreakpointNames();
		return;
	    }
	}
    }

    /** Updates the UI list to reflect a new bunch of breakpoint names */
    private void updateBreakpointNames()
    {
	debugger.updateList(breakpointList, breakpointNames, -1);
    }




    /** Connects browser to the polling mechanism when visible */
    public void show()
    {
	if (! isVisible())
	{
	    debugger.pollManager.addClient(this);
	}

	super.show();
    }

    /** Disconnects the browser from the polling mechanism when hidden */
    public void hide()
    {
	super.hide();

	if (isVisible())
	{
	    debugger.pollManager.removeClient(this);
	}
    }

    /** Implements the PollCallback interface */
    public void pollEvent()
    {
	try
	{
	    refreshBreakpointNames();
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }



    /** Handles UI events */
    public boolean handleEvent(Event event)
    {
	switch (event.id)
	{
	    case Event.WINDOW_DESTROY:
		hide();
		return true;
	}

	return super.handleEvent(event);
    }

    /** Handles button presses */
    public boolean action(Event event, Object target)
    {
	if (target.equals("Set Breakpoint"))
	{
	    debugger.setBreakpoint(className.getText(), methodName.getText());
	    debugger.pollManager.sync();
	    return true;
	}

	if (target.equals("Clear Breakpoint"))
	{
	    String name = breakpointList.getSelectedItem();

	    if (name != null)
	    {
		String cname = name.substring(0, name.indexOf(':'));
		String index = name.substring(name.indexOf(':') + 1);

		debugger.clearBreakpoint(cname, Integer.parseInt(index));
		debugger.pollManager.sync();
	    }
	    return true;
	}

	return super.action(event, target);
    }

}
