/*
 * @(#)Debugger.java	1.0alpha (18 Oct 1995)
 *
 * Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
 * Unpublished work.  All Rights Reserved.
 *
 * The software contained on this media is the property of the
 * DSTC Pty Ltd.  Use of this software is strictly in accordance
 * with the license agreement in the accompanying LICENSE.DOC file.
 * If your distribution of this software does not contain a
 * LICENSE.DOC file then you have no rights to use this software
 * in any manner and should contact DSTC at the address below
 * to determine an appropriate licensing arrangement.
 *
 *     DSTC Pty Ltd
 *     Level 7, Gehrmann Labs
 *     University of Queensland
 *     St Lucia, 4072
 *     Australia
 *     Tel: +61 7 3365 4310
 *     Fax: +61 7 3365 4311
 *     Email: enquiries@dstc.edu.au
 *
 * This software is being provided "AS IS" without warranty of
 * any kind.  In no event shall DSTC Pty Ltd be liable for
 * damage of any kind arising out of or in connection with
 * the use or performance of this software.
 */

package dstc.espresso.debug;

import java.awt.*;
import java.net.*;
import sun.tools.debug.*;

/** 
 * A graphical debugger
 *
 * @version	1.0alpha1 (9 Nov 1995)
 * @author	Ted Phelps
 */

public class Debugger extends Frame implements DebuggerCallback, PollCallback
{
    /** The remote debugger */
    public RemoteDebugger debugger;

    /** The poll manager */
    public PollManager pollManager;

    /** The thread groups being displayed in the list below */
    private RemoteThreadGroup threadGroups[];

    /** The List of thread groups */
    private List threadGroupList;

    /** The threads being displayed in the list below */
    private RemoteThread threads[];

    /** The status of the threads being displayed in the list below */
    private String threadStatus[];

    /** The List of threads */
    private List threadList;

    /** The breakpoint setting window */
    private BreakpointBrowser breakpointBrowser;

    /** The arguments to run a new thread */
    public String args[];

    /** Constructs a debugger on the local host using the cookie */
    public Debugger(String cookie) throws Exception
    {
	this(InetAddress.getLocalHost().getHostName(), cookie);
    }

    /** Constructs a debugger on the named host using the cookie */
    public Debugger(String hostName, String cookie) throws Exception
    {
	initialize(new RemoteDebugger(hostName, cookie, this, false));
    }

    /** Constructs a new local debugger */
    public Debugger() throws Exception
    {
	initialize(new RemoteDebugger("", this, false));
    }

    /** Constructs a debugger on the given RemoteDebugger */
    public void initialize(RemoteDebugger debugger) throws Exception
    {
	this.debugger = debugger;
	pollManager = new PollManager(15000);

	threadGroups = new RemoteThreadGroup[0];
	threads = new RemoteThread[0];
	threadStatus = new String[0];

	initializeFrame();
	pollManager.start();
    }

    /** Sets up the widgets for the display of the debugger */
    private void initializeFrame()
    {
	Panel panel;

	setTitle("Debugger");

	panel = new Panel();
	panel.setLayout(new GridLayout(1, 2));

	threadGroupList = new List();
	panel.add(threadGroupList);

	threadList = new List();
	panel.add(threadList);

	add("Center", panel);

	panel = new Panel();
	panel.setLayout(new GridLayout(1, 2));

	panel.add(new Button("Run"));
	panel.add(new Button("Breakpoints"));
	panel.add(new Button("Show"));
	add("South", panel);

	pack();
	resize(300, 200);
	show();
    }


    /** Implements the DebuggerCallback interface */
    public void breakpointEvent(RemoteThread thread) throws Exception
    {
	ThreadBrowser.on(this, thread).show();
	pollManager.sync();
    }

    /** Implements the DebuggerCallback interface */
    public void exceptionEvent(RemoteThread thread, String error) throws Exception
    {
	new ExceptionFrame(this, thread, error).show();
	pollManager.sync();
    }

    /** Implements the DebuggerCallback interface */
    public void quitEvent()
    {
	System.out.println("Client application has exited.");
	System.out.println("Shutting down.");
	System.exit(0);
    }

    /** Implements the DebuggerCallback interface */
    public void printToConsole(String string)
    {
	System.out.print(string);
	System.out.flush();
    }

    /** Implements the DebuggerCallback interface */
    public void threadDeathEvent(RemoteThread thread)
    {
	pollManager.sync();
    }


    /** Implements the PollCallback interface */
    public void pollEvent()
    {
	try
	{
	    refreshThreadGroups();
	    refreshThreads();
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }

    public void updateList(List list, String items[], int index)
    {
	list.hide();
	list.clear();
	for (int i = 0; i < items.length; i++)
	{
	    list.addItem(items[i]);
	}

	if (index >= 0)
	{
	    list.select(index);
	}
	list.show();
    }



    /** Loads a class into the client application */
    public void loadClass(String className)
    {
	try
	{
	    debugger.findClass(className);
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }

    /** Runs a class's main() method */
    public void run(String args[])
    {
	try
	{
	    debugger.run(args.length, args);
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }

    /** Run the default class's main() method */
    public void run()
    {
	run(args);
    }

    /** Sets a breakpoint on the named class and method */
    public void setBreakpoint(String className, String methodName)
    {
	try
	{
	    RemoteClass clazz = debugger.findClass(className);
	    if (clazz == null)
	    {
		return;
	    }

	    RemoteField method = clazz.getMethod(methodName);
	    if (method != null)
	    {
		clazz.setBreakpointMethod(method);
	    }
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }


    /** Removes a breakpoint fro the named class and method */
    public void clearBreakpoint(String className, String methodName)
    {
	try
	{
	    RemoteClass clazz = debugger.findClass(className);
	    if (clazz == null)
	    {
		return;
	    }

	    RemoteField method = clazz.getMethod(methodName);
	    if (method != null)
	    {
		clazz.clearBreakpointMethod(method);
	    }
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }

    /** Removes a breakpoint fro the named class and line number */
    public void clearBreakpoint(String className, int lineno)
    {
	try
	{
	    RemoteClass clazz = debugger.findClass(className);
	    if (clazz == null)
	    {
		return;
	    }

	    clazz.clearBreakpointLine(lineno);
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }


    /** Returns the selected thread group */
    private RemoteThreadGroup getSelectedThreadGroup()
    {
	int index = threadGroupList.getSelectedIndex();

	return index < 0 ? null : threadGroups[index];
    }


    /** Update the list of thread groups if necessary */
    private void refreshThreadGroups() throws Exception
    {
	RemoteThreadGroup old[] = threadGroups;
	RemoteThreadGroup selected = getSelectedThreadGroup();

	threadGroups = debugger.listThreadGroups(null);
	if (old.length != threadGroups.length)
	{
	    updateThreadGroups(selected);
	    return;
	}

	for (int i = 0; i < threadGroups.length; i++)
	{
	    if (old[i] != threadGroups[i])
	    {
		updateThreadGroups(selected);
		return;
	    }
	}
    }

    /** Update the list of thread groups and select the given group */
    private void updateThreadGroups(RemoteThreadGroup selected) throws Exception
    {
	String names[] = new String[threadGroups.length];
	int index = -1;

	for (int i = threadGroups.length - 1; i >= 0; i--)
	{
	    names[i] = threadGroups[i].getName();
	    if (threadGroups[i] == selected)
	    {
		index = i;
	    }
	}

	updateList(threadGroupList, names, index);
    }



    /** Returns the selected thread */
    private RemoteThread getSelectedThread()
    {
	int index = threadList.getSelectedIndex();

	return index < 0 ? null : threads[index];
    }

    /** Updates the list of threads if necessary */
    public void refreshThreads() throws Exception
    {
	RemoteThread old[] = threads;
	String oldStatus[] = threadStatus;
	RemoteThread selected = getSelectedThread();
	RemoteThreadGroup group = getSelectedThreadGroup();

	threads = group == null ? new RemoteThread[0] : group.listThreads(false);
	threadStatus = new String[threads.length];

	for (int i = 0; i < threads.length; i++)
	{
	    threadStatus[i] = threads[i].getStatus();
	}

	if (old.length != threads.length)
	{
	    updateThreads(selected);
	    return;
	}

	for (int i = 0; i < threads.length; i++)
	{
	    if (old[i] != threads[i] || (! oldStatus[i].equals(threadStatus[i])))
	    {
		updateThreads(selected);
		return;
	    }
	}
    }

    /** Update the list of threads to reflect some change */
    private void updateThreads(RemoteThread selected) throws Exception
    {
	String names[] = new String[threads.length];
	int index = -1;

	for (int i = threads.length - 1; i >= 0; i--)
	{
	    names[i] = threads[i].getName() + " (" + threadStatus[i] + ")";
	    if (threads[i] == selected)
	    {
		index = i;
	    }
	}

	updateList(threadList, names, index);
    }


    /** Connects the debugger to the polling mechanism when visible */
    public void show()
    {
	pollManager.addClient(this);
	super.show();
    }

    /** Shut everything down when the window is hidden */
    public void hide()
    {
	super.hide();
	pollManager.removeClient(this);

	try
	{
	    debugger.close();
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}

	System.exit(0);
    }

    /** Handles UI events */
    public boolean handleEvent(Event event)
    {
	switch (event.id)
	{
	    case Event.WINDOW_DESTROY:
		hide();
		return true;

	    case Event.LIST_SELECT:
	    case Event.LIST_DESELECT:
		pollManager.sync();
	}

	return super.handleEvent(event);
    }


    /** Handles button presses */
    public boolean action(Event event, Object target)
    {
	try
	{
	    if (target.equals("Show"))
	    {
		RemoteThread thread = getSelectedThread();
		if (thread != null)
		{
		    ThreadBrowser.on(this, thread).show();
		    return true;
		}
	    }

	    if (target.equals("Breakpoints"))
	    {
		if (breakpointBrowser == null)
		{
		    breakpointBrowser = new BreakpointBrowser(this);
		}

		breakpointBrowser.show();
	    }

	    if ("Run".equals(target))
	    {
		// FIX THIS: should prompt the user for alternative classes to run
		run();
	    }
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}

	return super.action(event, target);
    }
}
