/*
 * @(#)InspectorPanel.java	1.0alpha (17 Oct 1995)
 *
 * Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
 * Unpublished work.  All Rights Reserved.
 *
 * The software contained on this media is the property of the
 * DSTC Pty Ltd.  Use of this software is strictly in accordance
 * with the license agreement in the accompanying LICENSE.DOC file.
 * If your distribution of this software does not contain a
 * LICENSE.DOC file then you have no rights to use this software
 * in any manner and should contact DSTC at the address below
 * to determine an appropriate licensing arrangement.
 *
 *     DSTC Pty Ltd
 *     Level 7, Gehrmann Labs
 *     University of Queensland
 *     St Lucia, 4072
 *     Australia
 *     Tel: +61 7 3365 4310
 *     Fax: +61 7 3365 4311
 *     Email: enquiries@dstc.edu.au
 *
 * This software is being provided "AS IS" without warranty of
 * any kind.  In no event shall DSTC Pty Ltd be liable for
 * damage of any kind arising out of or in connection with
 * the use or performance of this software.
 */

package dstc.espresso.inspect;

import java.awt.*;
import java.io.*;
import java.util.*;

/** 
 * Constructs a Panel for graphically displaying an Inspector
 *
 * @see		dstc.espresso.inspect.Inspector
 * @see		dstc.espresso.inspect.InspectorFrame
 * @version	1.0alpha (17 Oct 1995)
 * @author	Ted Phelps
 */

public class InspectorPanel extends Panel
{
    /** The model is the Inspector to be displayed */
    private Inspector model;

    /**
     * The history remembers which objects and fields were inspected to
     * get to the current model
     */
    private Stack history;

    /**
     * The slots are the instance variable/array indices of the
     * object
     */
    private List slots;

    /**
     * The value is a text area displaying the contents of the
     * currently selected slot
     */
    private TextArea value;



    /** Returns an InspectorPanel on the object */
    public static InspectorPanel on(Object object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the boolean */
    public static InspectorPanel on(boolean object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the byte */
    public static InspectorPanel on(byte object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the char */
    public static InspectorPanel on(char object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the short */
    public static InspectorPanel on(short object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the int */
    public static InspectorPanel on(int object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the long */
    public static InspectorPanel on(long object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the float */
    public static InspectorPanel on(float object)
    {
	return new InspectorPanel(Inspector.on(object));
    }

    /** Returns an InspectorPanel on the double */
    public static InspectorPanel on(double object)
    {
	return new InspectorPanel(Inspector.on(object));
    }


    /** Constructs a new InspectorPanel on the Inspector */
    public InspectorPanel(Inspector inspector)
    {
	super();

	history = new Stack();
	setLayout(new GridLayout(1, 2));

	slots = new List();
	add(slots);

	value = new TextArea(10, 8);
	add(value);

	setModel(inspector);
    }

    /** Returns the title of the panel */
    public String getTitle()
    {
	if (model == null)
	{
	    return "";
	}
	else
	{
	    return model.getClassName();
	}
    }

    /**
     * Returns a new InspectorPanel on the currently selected slot,
     * or null if no slots is selected.
     */
    public InspectorPanel spawn()
    {
	int index = slots.getSelectedIndex();
	if (index < 0)
	{
	    return null;
	}
	else
	{
	    return new InspectorPanel(model.getInspector(index));
	}
    }

    /**
     * Inspects the selected slot, recording the current object and
     * slot index in expectation of a subsequent pop
     * @see #pop
     */
    public void push()
    {
	int index = slots.getSelectedIndex();
	if (index >= 0)
	{
	    history.push(model);
	    history.push(new Integer(slots.getSelectedIndex()));
	    setModel(model.getInspector(index));
	}
    }

    /**
     * Inspects the object by which the InspectorPanel came to be
     * viewing the current Inspector.  Similar to the "back" button of
     * a web browser
     * @see #push
     */
    public void pop()
    {
	if (! history.empty())
	{
	    int index = ((Integer) history.pop()).intValue();
	    Inspector inspector = (Inspector)history.pop();

	    setModel(inspector);
	    slots.select(index);
	    value.setText(model.getValue(index));
	}
    }


    /** Throw away the history and the target */
    public void reset()
    {
	history = new Stack();
	setModel(null);
    }

    /**
     * View the fields of a Inspector.  Does not record what model was
     * previously being viewed
     * @see #pop
     * @see #push
     */
    public void setModel(Inspector inspector)
    {
	String names[];

	if (inspector == null)
	{
	    slots.clear();
	    value.setText("");
	}
	else
	{
	    model = inspector;
	    value.setText(model.getTarget());
	    
	    slots.clear();
	    names = model.getNames();
	    for (int i = 0; i < names.length; i++)
	    {
		slots.addItem(names[i]);
	    }
	}
    }



    /**
     * Handles list select and deselect events to update the value
     * text area
     */
    public boolean handleEvent(Event event)
    {
	if (event.target == slots)
	{
	    switch (event.id)
	    {
		case Event.LIST_DESELECT:
		    value.setText(model.getTarget());
		    return true;

		case Event.LIST_SELECT:
		    value.setText(model.getValue(slots.getSelectedIndex()));
		    return true;
	    }
	}

	return super.handleEvent(event);
    }
}
