/*
 * @(#)ThreadBrowser.java	1.0alpha (18 Oct 1995)
 *
 * Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
 * Unpublished work.  All Rights Reserved.
 *
 * The software contained on this media is the property of the
 * DSTC Pty Ltd.  Use of this software is strictly in accordance
 * with the license agreement in the accompanying LICENSE.DOC file.
 * If your distribution of this software does not contain a
 * LICENSE.DOC file then you have no rights to use this software
 * in any manner and should contact DSTC at the address below
 * to determine an appropriate licensing arrangement.
 *
 *     DSTC Pty Ltd
 *     Level 7, Gehrmann Labs
 *     University of Queensland
 *     St Lucia, 4072
 *     Australia
 *     Tel: +61 7 3365 4310
 *     Fax: +61 7 3365 4311
 *     Email: enquiries@dstc.edu.au
 *
 * This software is being provided "AS IS" without warranty of
 * any kind.  In no event shall DSTC Pty Ltd be liable for
 * damage of any kind arising out of or in connection with
 * the use or performance of this software.
 */

package dstc.kalimantan.debug;

import java.awt.*;
import java.util.*;
import dstc.kalimantan.inspect.*;


/** 
 * A Panel for presenting an inspector with some useful buttons
 *
 * @see		dstc.kalimantan.inspect.InspectorFrame
 * @version	1.0alpha (17 Oct 1995)
 * @author	Ted Phelps
 */
public class DebugInspectorPanel extends Panel implements PollCallback
{
    /** The debugger from which we were spawned */
    private Debugger debugger;

    /** The object being inspected */
    private Inspector model;

    /** The history of the objects we've inspected to get here */
    private Stack history;

    /** The name of the class of the object being inspected */
    private Label title;

    /** The names of the fields */
    private List slotList;

    /** The actual strings displayed in the slotList */
    private String slots[];

    /** The value of the currently selected field */
    private TextArea value;

    /** Constructs a new panel */
    public DebugInspectorPanel(Debugger debugger, Inspector inspector)
    {
	Panel panel;

	this.debugger = debugger;
	history = null;
	slots = new String[0];

	setLayout(new BorderLayout());

	title = new Label();
	title.setAlignment(Label.CENTER);
	title.setText("bubba");
	add("North", title);

	panel = new Panel();
	panel.setLayout(new GridLayout(1, 2));

	slotList = new List();
	panel.add(slotList);

	value = new TextArea(10, 8);
	panel.add(value);
	add("Center", panel);

	panel = new Panel();
	panel.setLayout(new GridLayout(1, 2));
	panel.add(new Button("Back"));
	panel.add(new Button("Forward"));
	panel.add(new Button("Spawn"));
	add("South", panel);

	setModel(inspector);
    }

    /** Implements the PollCallback interface */
    public void pollEvent()
    {
	refreshTitle();
	refreshSlots();
	refreshValue();
    }

    /** @see #dstc.kalimantan.inspect.InspectorPanel.push */
    public void push()
    {
	int index = slotList.getSelectedIndex();
	if (index >= 0)
	{
	    history.push(model);
	    history.push(new Integer(index));
	    setModel(model.getInspector(index));
	}
    }

    /** @see #dstc.kalimantan.inspect.InspectorPanel.pop */
   public void pop()
    {
	if (! history.empty())
	{
	    int index = ((Integer) history.pop()).intValue();
	    Inspector inspector = (Inspector)history.pop();

	    setModel(inspector);
	    slotList.select(index);
	}
    }


    /** Changes the focus of the panel to a new inspector */
    public void setModel(Inspector inspector)
    {
	model = inspector;
    }

    /** Sets the model and erases the history */
    public void resetModel(Inspector inspector)
    {
	history = new Stack();
	model = inspector;
    }


    /** Refreshes the title of the inspector, if appropriate */
    private void refreshTitle()
    {
	String oldTitle = title.getText();
	String newTitle;

	if (model != null)
	{
	    newTitle = model.getClassName();
	}
	else
	{
	    newTitle = "";
	}

	if (oldTitle != newTitle)
	{
	    title.setText(newTitle);
	}
    }

    /** Refreshes the list of slots if appropriate */
    private void refreshSlots()
    {
	String old[] = slots;

	if (model == null)
	{
	    slots = new String[0];
	}
	else
	{
	    slots = model.getNames();
	}

	if (old.length != slots.length)
	{
	    updateSlots();
	    return;
	}

	for (int i = 0; i < slots.length; i++)
	{
	    if (! old[i].equals(slots[i]))
	    {
		updateSlots();
		return;
	    }
	}
    }

    /** Updates the list of slots to reflect a change */
    private void updateSlots()
    {
	debugger.updateList(slotList, slots, -1);
    }


    /** Updates the value displayed if necessary */
    private void refreshValue()
    {
	String old = value.getText();
	int index = slotList.getSelectedIndex();
	String v;

	if (model == null)
	{
	    v = "";
	}
	else if (index < 0)
	{
	    v = model.getTarget();
	}
	else
	{
	    v = model.getValue(index);
	}

	if (! v.equals(old))
	{
	    value.setText(v);
	}
    }

    /** Handles GUI events */
    public boolean handleEvent(Event event)
    {
	switch (event.id)
	{
	    case Event.LIST_DESELECT:
	    case Event.LIST_SELECT:
		debugger.pollManager.sync();
		return true;
	}

	return super.handleEvent(event);
    }

    /** Handles buttonpress events */
    public boolean action(Event event, Object target)
    {
	if ("Forward".equals(target))
	{
	    push();
	    debugger.pollManager.sync();
	    return true;
	}

	if ("Back".equals(target))
	{
	    pop();
	    debugger.pollManager.sync();
	    return true;
	}

	if ("Spawn".equals(target))
	{
	    int index = slotList.getSelectedIndex();
	    if (index >= 0)
	    {
		new InspectorFrame(new InspectorPanel(model.getInspector(index)));
	    }
	    return true;
	}

	return super.action(event, target);
    }
}
