/*
 * @(#)ThreadBrowser.java	1.0alpha (18 Oct 1995)
 *
 * Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
 * Unpublished work.  All Rights Reserved.
 *
 * The software contained on this media is the property of the
 * DSTC Pty Ltd.  Use of this software is strictly in accordance
 * with the license agreement in the accompanying LICENSE.DOC file.
 * If your distribution of this software does not contain a
 * LICENSE.DOC file then you have no rights to use this software
 * in any manner and should contact DSTC at the address below
 * to determine an appropriate licensing arrangement.
 *
 *     DSTC Pty Ltd
 *     Level 7, Gehrmann Labs
 *     University of Queensland
 *     St Lucia, 4072
 *     Australia
 *     Tel: +61 7 3365 4310
 *     Fax: +61 7 3365 4311
 *     Email: enquiries@dstc.edu.au
 *
 * This software is being provided "AS IS" without warranty of
 * any kind.  In no event shall DSTC Pty Ltd be liable for
 * damage of any kind arising out of or in connection with
 * the use or performance of this software.
 */

package dstc.kalimantan.debug;

import java.awt.*;
import java.util.*;
import java.io.*;
import sun.tools.debug.*;
import dstc.kalimantan.util.*;
import dstc.kalimantan.inspect.*;

/** 
 * A graphical thread browser
 *
 * @version	1.0alpha (18 Oct 1995)
 * @author	Ted Phelps
 */

public class ThreadBrowser extends Frame implements PollCallback
{
    /** The table of all existing ThreadBrowsers */
    private static Hashtable browsers = new Hashtable();

    /** The table of all source code */
    private static Hashtable sources = new Hashtable();

    /** The Debugger that spawned us */
    Debugger debugger;

    /** The thread we're browsing */
    RemoteThread thread;

    /** The frame of the stack displayed in the List below */
    RemoteStackFrame stackFrames[];

    /** The List of stack frames */
    List stackFrameList;

    /** The title of the frame */
    String name;

    /** The status of the thread */
    String status;

    /** The source string displayed in the text area below */
    IndexedString source;

    /** The TextArea in which the source code is displayed */
    TextArea sourceText;

    /** The InspectorPanel on thisInspector below */
    DebugInspectorPanel thisInspectorPanel;

    /** The inspector on the current context object */
    Inspector thisInspector;

    /** The InspectorPanel on the tempInspector below */
    DebugInspectorPanel tempInspectorPanel;

    /** The Inspector on the temporary variables */
    RemoteStackFrameInspector tempInspector;

    /** The name of the selected stack frame */
    String selectedName;

    /** Returns a possibly new ThreadBrowser on a given thread */
    public static ThreadBrowser on(Debugger debugger, RemoteThread thread) throws Exception
    {
	ThreadBrowser browser = (ThreadBrowser)browsers.get(thread);

	if (browser == null)
	{
	    browser = new ThreadBrowser(debugger, thread);
	    browsers.put(thread, browser);
	}

	return browser;
    }

    /** Returns the source file associated with the stack frame */
    public static IndexedString getSource(RemoteStackFrame sFrame) throws Exception
    {
	return getSource(sFrame.getRemoteClass());
    }

    /** Returns the source file associated with the given class */
    public static IndexedString getSource(RemoteClass clazz) throws Exception
    {
	IndexedString source = (IndexedString) sources.get(clazz);

	if (source == null)
	{
	    // FIX THIS: (workaround hack)
	    // make sure the class info is cached
	    clazz.getName();
	    InputStream stream = clazz.getSourceFile();

	    if (stream == null)
	    {
		source = new IndexedString((String) null);
	    }
	    else
	    {
		source = new IndexedString(stream);
	    }

	    sources.put(clazz, source);
	}

	return source;
    }
    

    /** Constructs a new ThreadBrowser on a given thread */
    public ThreadBrowser(Debugger debugger, RemoteThread thread) throws Exception
    {
	this.debugger = debugger;
	this.thread = thread;

	stackFrames = new RemoteStackFrame[0];
	name = thread.getName();
	source = null;
	thisInspector = null;
	tempInspector = null;
	selectedName = "";

	initializeFrame();
    }

    /** Sets up the UI for the browser */
    private void initializeFrame()
    {
	Panel panel, subPanel;

	panel = new Panel();
	panel.setLayout(new BorderLayout());
	stackFrameList = new List();
	panel.add("Center", stackFrameList);

	subPanel = new Panel();
	subPanel.setLayout(new GridLayout(1, 4));
	subPanel.add(new Button("Suspend"));
	subPanel.add(new Button("Resume"));
	panel.add("North", subPanel);

	subPanel = new Panel();
	subPanel.setLayout(new GridLayout(1, 4));
	subPanel.add(new Button("Step"));
	subPanel.add(new Button("Next"));
	subPanel.add(new Button("Continue"));
	panel.add("South", subPanel);

	add("West", panel);


	sourceText = new TextArea();
	add("Center", sourceText);

	panel = new Panel();
	panel.setLayout(new GridLayout(2, 1));

	thisInspectorPanel = new DebugInspectorPanel(debugger, thisInspector);
	panel.add(thisInspectorPanel);

	tempInspectorPanel = new DebugInspectorPanel(debugger, tempInspector);
	panel.add(tempInspectorPanel);

	add("East", panel);

	pack();
	resize(900, 500);
    }


    /** Returns the selected stack frame */
    public RemoteStackFrame getSelectedStackFrame()
    {
	int index = stackFrameList.getSelectedIndex();
	return index < 0 ? null : stackFrames[index];
    }


    /** Implements the PollCallback interface */
    public void pollEvent()
    {
	try
	{
	    refreshTitle();
	    refreshStackFrames();
	    refreshSource();
	    refreshInspectors();
	    thisInspectorPanel.pollEvent();
	    tempInspectorPanel.pollEvent();
	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}
    }


    /** Updates the title if necessary */
    private void refreshTitle() throws Exception
    {
	String oldStatus = status;

	status = thread.getStatus();
	if (! status.equals(oldStatus))
	{
	    setTitle(name + " (" + status + ")");
	}
    }


    /** Updates the stack trace if necessary */
    private void refreshStackFrames() throws Exception
    {
	RemoteStackFrame old[] = stackFrames;
	RemoteStackFrame selected = getSelectedStackFrame();

	if (isSuspended())
	{
	    stackFrames = thread.dumpStack();
	}
	else
	{
	    stackFrames = new RemoteStackFrame[0];
	}

	if (stackFrames.length > 0 &&
	    (Math.abs(stackFrames.length - old.length) <= 1 ||
	     old.length == 0))
	{
	    selected = stackFrames[0];
	}

	if (old.length != stackFrames.length)
	{
	    updateStackFrames(selected);
	    return;
	}

	for (int i = 0; i < stackFrames.length; i++)
	{
	    if (! old[i].toString().equals(stackFrames[i].toString()))
	    {
		updateStackFrames(selected);
		return;
	    }
	}
    }

    /** Updates the list of stack frames to reflect a recent change */
    public void updateStackFrames(RemoteStackFrame selected) throws Exception
    {
	int index = -1;
	String names[] = new String[stackFrames.length];

	for (int i = stackFrames.length - 1; i >= 0; i--)
	{
	    names[i] = stackFrames[i].toString();
	    if (stackFrames[i] == selected)
	    {
		index = i;
	    }
	}

	debugger.updateList(stackFrameList, names, index);
    }




    /** Updates the source code to be representative of the current stack frame */
    public void refreshSource() throws Exception
    {
	RemoteStackFrame sFrame = getSelectedStackFrame();
	IndexedString old = source;

	if (sFrame == null)
	{
	    if  (source != null)
	    {
		source = null;
		sourceText.setText("");
	    }
	}
	else
	{
	    source = getSource(sFrame);
	    String string = source.toString();

	    if (string == null)
	    {
		if (old == null || old.toString() != null)
		{
		    sourceText.setText("*** source file not available ***");
		}
	    }
	    else
	    {
		int line = sFrame.getLineNumber();

		if (old != source)
		{
		    sourceText.setText(string);
		}

		if (line-- >= 0)
		{
		    sourceText.select(source.startOfLine(line), source.endOfLine(line));
		}
	    }
	}
    }


    /** Refreshes the thisInspectorPanel and tempInspectorPanel */
    private void refreshInspectors() throws Exception
    {
	RemoteStackFrame rFrame = getSelectedStackFrame();
	String old = selectedName;

	selectedName = rFrame == null ? "" : rFrame.toString();
	if (! selectedName.equals(old))
	{
	    updateInspectors();
	}
    }

    /** Updates thisInspectorPanel and tempInspectorPanel */
    private void updateInspectors() throws Exception
    {
	RemoteStackFrame rFrame = getSelectedStackFrame();
	if (rFrame == null)
	{
	    thisInspector = null;
	    tempInspector = null;
	}
	else
	{
	    RemoteStackVariable vs[] = rFrame.getLocalVariables();
	    RemoteStackVariable v = vs.length > 0 ? vs[0] : null;

	    thisInspector = v == null ? null : Inspector.on(v.getValue());
	    tempInspector = new RemoteStackFrameInspector(vs);
	}

	thisInspectorPanel.resetModel(thisInspector);
	tempInspectorPanel.resetModel(tempInspector);
    }

    /** Connects browser to the polling mechanism when visible */
    public void show()
    {
	if (! isVisible())
	{
	    debugger.pollManager.addClient(this);
	}

	super.show();
    }

    /** Disconnects the browser from the polling mechanism when hidden */
    public void hide()
    {
	super.hide();

	if (isVisible())
	{
	    debugger.pollManager.removeClient(this);
	}

    }


    /** If the thread is suspended or at a breakpoint */
    public boolean isSuspended()
    {
	return status.equals("suspended") || status.equals("at breakpoint");
    }

    public boolean isAtBreakpoint()
    {
	return status.equals("at breakpoint");
    }



    /** Handle UI events */
    public boolean handleEvent(Event event)
    {
	switch (event.id)
	{
	    case Event.WINDOW_DESTROY:
		hide();
		return true;
	    case Event.LIST_SELECT:
	    case Event.LIST_DESELECT:
		debugger.pollManager.sync();
		return true;
	}

	return super.handleEvent(event);
    }

    /** Handles button presses */
    public boolean action(Event event, Object target)
    {
	try
	{
	    if (target.equals("Suspend"))
	    {
		thread.suspend();
		debugger.pollManager.sync();
		return true;
	    }
	    
	    if (target.equals("Resume"))
	    {
		thread.resume();
		debugger.pollManager.sync();

		return true;
	    }

	    if (target.equals("Step"))
	    {
		thread.step(true);

		return true;
	    }

	    if (target.equals("Next"))
	    {
		thread.next();

		return true;
	    }

	    if (target.equals("Continue"))
	    {
		thread.cont();
		debugger.pollManager.sync();

		return true;
	    }

	}
	catch (Exception exception)
	{
	    exception.printStackTrace();
	}

	return super.action(event, target);
    }
}
