/*
 * @(#)Hack.java	1.0alpha (17 Oct 1995)
 *
 * Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
 * Unpublished work.  All Rights Reserved.
 *
 * The software contained on this media is the property of the
 * DSTC Pty Ltd.  Use of this software is strictly in accordance
 * with the license agreement in the accompanying LICENSE.DOC file.
 * If your distribution of this software does not contain a
 * LICENSE.DOC file then you have no rights to use this software
 * in any manner and should contact DSTC at the address below
 * to determine an appropriate licensing arrangement.
 *
 *     DSTC Pty Ltd
 *     Level 7, Gehrmann Labs
 *     University of Queensland
 *     St Lucia, 4072
 *     Australia
 *     Tel: +61 7 3365 4310
 *     Fax: +61 7 3365 4311
 *     Email: enquiries@dstc.edu.au
 *
 * This software is being provided "AS IS" without warranty of
 * any kind.  In no event shall DSTC Pty Ltd be liable for
 * damage of any kind arising out of or in connection with
 * the use or performance of this software.
 */

package sun.tools.debug;

import java.io.*;
import java.net.*;


/** 
 * Modifications to the Debugger API which enable a process to be both
 * a debugging client and server.  This is used by the Inspector.
 *
 * @see		dstc.kalimantan.inspect.Inspector
 * @version	1.0alpha (17 Oct 1995)
 * @author	Ted Phelps
 */

public final class Hack extends Object implements DebuggerCallback
{
    /** instance holds the distinguished instance of this class */
    private static Hack instance;

    /** The server side of the debugger API */
    private Agent agent;

    /** The client side of the debugger API */
    private RemoteAgent remoteAgent;

    /** The server end of a pipe connection */
    DataInputStream in;

    /** The client end of a pipe connection */
    DataOutputStream out;

    /** The true System.out */
    PrintStream stdout;


    /**
     * Returns the distinguished instance of this class
     */
    public static Hack getInstance()
    {
	if (instance == null)
	{
	    instance = new Hack(3456);
	}

	return instance;
    }


    /**
     * Construct the distinguished instance of this class to
     * communicate on the given port.
     */
     private Hack(int port)
    {
	PipedInputStream pin;

	agent = null;
	remoteAgent = null;
	stdout = System.out;

	try
	{
	    startAgent(port);

	    pin = new PipedInputStream();
	    in = new DataInputStream(pin);
	    out = new DataOutputStream(new PipedOutputStream(pin));
	}
	catch (Exception e)
	{
	    e.printStackTrace();
	    System.exit(0);
	}
    }


    /**
     * Starts up a matched pair of a client and server for the
     * debugger api.
     */
    private void startAgent(int port) throws Exception
    {
	ByteArrayOutputStream stream;
	PrintStream out;
	String buffer, cookie, hostname;

	// Redirect System.out so that we can grab the cookie
	stream = new ByteArrayOutputStream(256);
	out = System.out;
	System.out = new PrintStream(stream);
	agent = new Agent(port);
	System.out = out;

	// pull the cookie out of the output string
	buffer = stream.toString();
	cookie = buffer.substring(buffer.indexOf("=") + 1, buffer.indexOf("\n"));

	// start up the agent
	new Thread(agent).start();

	// start up the remote agent
	try
	{
	    hostname = InetAddress.getLocalHost().getHostName();
	}
	catch (UnknownHostException exception)
	{
	    hostname = "localhost";
	}

	remoteAgent = new RemoteAgent(hostname, cookie, "", this, false);
    }


    /**
     * Returns a RemoteValue which represents the object.
     */
    public RemoteValue asRemoteValue(Object obj) throws Exception
    {
	if (obj instanceof Object[])
	{
	    Object target[] = (Object [])obj;
	    return asRemoteValue(target, target.length);
	}
	if (obj instanceof boolean[])
	{
	    byte target[] = (byte [])obj;
	    return asRemoteValue(target, target.length);
	}
	if (obj instanceof char[])
	{
	    char target[] = (char [])obj;
	    return asRemoteValue(target, target.length);
	}
	if (obj instanceof short[])
	{
	    short target[] = (short [])obj;
	    return asRemoteValue(target, target.length);
	}
	if (obj instanceof int[])
	{
	    int target[] = (int [])obj;
	    return asRemoteValue(target, target.length);
	}
	if (obj instanceof long[])
	{
	    long target[] = (long [])obj;
	    return asRemoteValue(target, target.length);
	}
	if (obj instanceof float[])
	{
	    float target[] = (float [])obj;
	    return asRemoteValue(target, target.length);
	}
	if (obj instanceof double[])
	{
	    double target[] = (double [])obj;
	    return asRemoteValue(target, target.length);
	}

	agent.writeObject(obj, out);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the array
     */
    public RemoteValue asRemoteValue(Object obj, int length) throws Exception
    {
	int id = 0;

	out.writeInt(9);
	if (obj == null)
	{
	    out.writeInt(0);
	}
	else
	{
	    Class clazz = obj.getClass();
	    agent.objects.put(new Integer(id = agent.objectId(clazz)), clazz);
	    out.writeInt(id);
	    agent.objects.put(new Integer(id = agent.objectId(obj)), obj);
	    out.writeInt(id);
	    out.writeInt(length);
	}

	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the boolean.
     */
    public RemoteValue asRemoteValue(boolean bool) throws Exception
    {
	out.writeInt(0);
	out.writeBoolean(bool);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the byte.
     */
    public RemoteValue asRemoteValue(byte b) throws Exception
    {
	out.writeInt(1);
	out.writeByte(b);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the char.
     */
    public RemoteValue asRemoteValue(char c) throws Exception
    {
	out.writeInt(2);
	out.writeChar(c);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the short.
     */
    public RemoteValue asRemoteValue(short sh) throws Exception
    {
	out.writeInt(3);
	out.writeShort(sh);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the int.
     */
    public RemoteValue asRemoteValue(int i) throws Exception
    {
	out.writeInt(4);
	out.writeInt(i);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the long.
     */
    public RemoteValue asRemoteValue(long l) throws Exception
    {
	out.writeInt(5);
	out.writeLong(l);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the float.
     */
    public RemoteValue asRemoteValue(float f) throws Exception
    {
	out.writeInt(6);
	out.writeFloat(f);
	return remoteAgent.readValue(in);
    }

    /**
     * Returns a RemoteValue which represents the double.
     */
    public RemoteValue asRemoteValue(double d) throws Exception
    {
	out.writeInt(7);
	out.writeDouble(d);
	return remoteAgent.readValue(in);
    }


    /** Ignores breakpoint events */
    public void breakpointEvent(RemoteThread t) throws Exception
    {}

    /** Ignores exception events */
    public void exceptionEvent(RemoteThread t, String errorText) throws Exception
    {}

    /** Ignores quit events */
    public void quitEvent()
    {}

    /** Ignores print-to-console requests */
    public void printToConsole(String str)
    {
	System.err.print(str);
	System.err.flush();
    }

    /** Ignores thread death events */
    public void threadDeathEvent(RemoteThread thread)
    {}
}
