/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.sys.*;
import dejava.util.*;
import dejava.lang.*;

import java.util.Enumeration;
import java.util.Vector;
import java.awt.*;

public class Browser extends DejavaFrame {
    protected static final String PKGLIST    = "pkgList";
    protected static final String CLASSLIST  = "clsList";
    protected static final String METHODLIST = "mtdList";

    protected Panel listArea;
    protected DejavaList pkgList;
    protected DejavaList clsList;
    protected DejavaList mtdList;
    private int pkgListIndex = -1;
    private int clsListIndex = -1;
    private int mtdListIndex = -1;

    protected int classesWidth;

    /**
     * spawn browser
     */
    public static void openOn(CodePath cp) {
	openOn(cp, cp);
    }

    /**
     * spawn browser
     */
    static void openOn(CodePath cp, CodePath select) {
	String title;
	if (cp.methodManager() != null) {
	    title = "Method";
	} else if (cp.classManager() != null) {
	    title = "Class";
	} else if (cp.packageManager() != null) {
	    title = "Package";
	} else {
	    title = "Project";
	}
	title += " Browser on: " + cp.printString();
	openOn(cp, select, title);
    }

    /**
     * spawn browser
     */
    static void openOn(CodePath cp, CodePath select, String title) {
	(new Browser(cp, select, title)).open();
    }

    /**
     * the constructor.
     */
    public Browser(CodePath cp, CodePath select, String title) {
	super(cp, title, true);
	current = new CodePath(select);
    }

    /**
     * create menubar items
     */
    protected DejavaMenuBar createMenuBar() {
	return new BrowserMenuBar();
    }

    /**
     * create display items
     */
    protected void createDisplayItems() {
	super.createDisplayItems();
	int listLines = DejavaProperties.getInteger("attribute.browsing.lines");
	if (startup.methodManager() == null) {
	    listArea = new Panel();
	    listArea.setLayout(new GridLayout(1, startup.classManager() == null ? 2 : 1));
	    Panel left = null;
	    if (startup.packageManager() == null) {	// package selection list is required
        	pkgList = new DejavaList(listLines, false);
		left = new Panel();
		left.setLayout(new GridLayout(1, 2));
		left.add(pkgList.makePanel());
		listArea.add(left);
	    }
	    if (startup.classManager() == null) {	// class selection list is required
        	clsList = new DejavaList(listLines, false);
		if (left != null) {
		    left.add(clsList.makePanel());
		} else {
		    listArea.add(clsList.makePanel());
		}
	    }
	    mtdList = new DejavaList(listLines, false);
	    listArea.add(mtdList.makePanel());

	    add(NORTH, listArea);
	}
    }

    public boolean handleEvent(Event ev) {
	switch (ev.id) {
	case Event.LIST_SELECT:
	    if (SystemManager.onWin32) {
		// Windows can't deselect
		int selectedIndex = ((Integer)ev.arg).intValue();
		if (ev.target == pkgList) {
		    if (pkgListIndex == selectedIndex) pkgList.deselect(selectedIndex);
		} else if (ev.target == clsList) {
		    if (clsListIndex == selectedIndex) clsList.deselect(selectedIndex);
		} else if (ev.target == mtdList) {
		    if (mtdListIndex == selectedIndex) mtdList.deselect(selectedIndex);
		} else {
		    break;
		}
	    }
	    // no break
	case Event.LIST_DESELECT:
	case Event.ACTION_EVENT:
	    CodePath selected = new CodePath(current);
	    if (ev.target == pkgList) {
		selected.packageManager(pkgList.getSelectedItem());
	    } else if (ev.target == clsList) {
		selected.classManager(clsList.getSelectedItem());
	    } else if (ev.target == mtdList) {
		selected.methodManager(mtdList.getSelectedItem());
	    } else {
		break;
	    }
	    if (!codeView.isChanged()) {
		current = selected;
		try {
		    if (ev.target == pkgList) {
			updateClasses();
		    } else if (ev.target == clsList) {
			updateMethods();
		    } else if (ev.target == mtdList) {
			updateCurrent();
		    }
		} catch (DejavaException e) {
		    current.classManager((ClassManager)null);
		    update();
		}
	    } else if (acceptWithConfirm()) {
                current = selected;
		update();
	    } else {
		if (ev.target == pkgList) {
		    selectItem(pkgList, current.packageName());
		} else if (ev.target == clsList) {
		    selectItem(clsList, current.className());
		} else if (ev.target == mtdList) {
		    selectItem(mtdList, current.methodPrototype());
		}
	    }
	    if (!SystemManager.onSolaris) {
		if (pkgList != null) pkgListIndex = pkgList.getSelectedIndex();
		if (clsList != null) clsListIndex = clsList.getSelectedIndex();
		if (mtdList != null) mtdListIndex = mtdList.getSelectedIndex();
	    }
	    return false;
	}
	return super.handleEvent(ev);
    }

    private void selectItem(List list, String name) {
	int num = list.countItems();
	for (int i = 0; i < num; i++) {
	    String listedName = list.getItem(i);
	    if (listedName.equals(name)) {
		if (!list.isSelected(i)) list.select(i);
		return;
	    }
	}
    }

    /**
     * menu handler of new package
     */
    void newPackage(CodePath cp) {
	if (!acceptWithConfirm()) return;
	NewPackageDialog dialog = new NewPackageDialog(this);
	if (dialog.confirm() && acceptWithConfirm()) {
	    PackageManager pm = cp.projectManager().createPackageAfter(
				dialog.getName(), cp.packageManager());
	    if (pm != null) {
		current.packageManager(pm);
		DejavaFrame.updateNotify(current);
		update();
	    }
	}
	dialog.dispose();
    }

    /**
     * new class or interface
     */
    void newClass(CodePath cp, boolean isInterface) {
	if (!acceptWithConfirm()) return;
	NewClassDialog dialog = (new NewClassDialog(this, current,
				 new ClassSpec(null, null, null, isInterface)));
	if (dialog.confirm() && acceptWithConfirm()) {
	    ClassSpec cs = dialog.getClassSpec();
	    ClassManager cm = cp.packageManager().createClass(cs);
	    if (cm != null) {
		doReset();
		current.classManager(cm);
		DejavaFrame.updateNotify(current);
		update();
	    }
	}
	dialog.dispose();
    }

    /**
     * new constracor or method
     */
    void newMethod(CodePath cp, boolean isConstractor) {
	if (!acceptWithConfirm()) return;
	try {
	    NewMethodDialog dialog = (new NewMethodDialog(this, current,
				     cp.classManager().spec(), isConstractor));
	    if (dialog.confirm() && acceptWithConfirm()) {
		codeView.setText("    " +
				 dialog.getDescription() +
				 (dialog.getMethodSpec().isAbstract() ?
					 ";\n" : " {\n    }\n"));
	    }
	    dialog.dispose();
	} catch (DejavaException e) {
	    DejavaDialog.information(this, e.getMessage());
	}

    }

    /**
     * new constracor or method
     */
    void newMethod(CodePath cp, String code) {
	if (acceptWithConfirm()) {
	    codeView.setText(code);
	}
    }

    /**
     * remove the structure
     */
    void remove(CodePath cp) {
	if (confirm("remove " + cp.printShortString() + ", really?")) {
	    doReset();
	    if (cp.getManager().remove() == null) {
		cp.setManager(null);
		DejavaFrame.updateNotify(cp);
		update();
	    }
	}
    }

    /**
     * rename the structure
     */
    void rename(CodePath cp) {
	Manager man = cp.getManager();
	String name = request("rename " + cp.printString() + " to what?");
	if (name != null && !name.equals(man.name())) {
	    try {
		if (man.rename(name) != null) {
		    DejavaFrame.updateNotify(cp);
		    update();
		}
	    } catch (DejavaException e) {
	    }
	}
    }

    /**
     * update all of this frame
     */
    protected void update() {
	try {
	    if (pkgList != null) {
		updatePackages();
	    } else if (clsList != null) {
		updateClasses();
	    } else if (mtdList != null) {
		updateMethods();
	    } else {
		updateCurrent();
	    }
	} catch (DejavaException e) {
	    current.classManager((ClassManager)null);
	    update();
	}
    }

    /**
     * project selection has been changed
     * this method is not called by interactive select operation
     */
    protected void updatePackages() throws DejavaException {
	if (pkgList == null) {
	    // no package list area
	} else if (current.projectManager() == null) {
	    // no selected project name, clear package list
	    pkgList.clear();
	} else {
	    // check list same as last displaied list
	    ProjectManager pm = current.projectManager();
	    String pkgName = current.packageName();
	    if (!pkgList.selectItemIfSameList(pm.packageNames(), pkgName)) {
		if (pkgList.listAndSelectItem(pm.packageNames(), pkgName) < 0) {
		    current.packageManager((PackageManager)null);
		}
	    }
	}
	updateClasses();
    }

    /**
     * package selection has been changed
     */
    protected void updateClasses() throws DejavaException {
	if (clsList == null) {
	    // no class list area
	} else if (current.packageManager() == null) {
	    // no selected package name, clear class list
	    clsList.clear();
	} else {
	    // check list same as last displaied list
	    PackageManager pm = current.packageManager();
	    String className = current.className();
	    if (!clsList.selectItemIfSameList(pm.classNames(), className)) {
		if (clsList.listAndSelectItem(pm.classNames(), className) < 0) {
		    current.classManager((ClassManager)null);
		}
	    }
	}
	updateMethods();
    }

    /**
     * class selection has been changed
     */
    protected void updateMethods() throws DejavaException {
	if (mtdList == null) {
	    // no method list area
	} else if (current.classManager() == null) {
	    // no selected class name, clear method list
	    mtdList.clear();
	} else {
	    // check list same as last displaied list
	    ClassManager cm = current.classManager();
	    AccessFlags hideMethods = ((BrowserMenuBar)getMenuBar()).hideMethods();
	    String methodPrototype = current.methodPrototype();
	    if (!mtdList.selectItemIfSameList(
		      cm.methodPrototypes(hideMethods), methodPrototype)) {
		if (mtdList.listAndSelectItem(
		      cm.methodPrototypes(hideMethods), methodPrototype) < 0) {
		    current.methodManager((MethodManager)null); // no select
		}
	    }
	}
	updateCurrent();
    }
}
