/*
 * Copyright (c) 1995 PFU Limited.
 *      author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.sys.*;
import dejava.util.*;

import java.awt.*;
import java.util.*;
import java.io.File;
import java.io.DataInputStream;
import java.io.IOException;

import dejava.gui.ListingCommand;

public class CompileFrame extends ListingCommand {
    private Compile javac;
    private Vector detail = new Vector();
    private TextArea errorMessage;
    private boolean compileUpdated;

    private static final Hashtable frames = new Hashtable();

    static void openOn(CodePath cp, boolean updated) {
        CompileFrame compile = find(cp);
        if (compile == null) {
            compile = new CompileFrame(cp, "Compile frame on: " + cp.printString(), updated);
            compile.start();
            compile.open();
        } else {
            compile.restart();
            compile.show();
	    compile.requestFocus();
        }
    }

    public CompileFrame(CodePath cp, String title, boolean updated) {
	super(cp, title, "compilation");
	compileUpdated = updated;
	frames.put(cp.printString(), this);
    }

    protected DejavaMenuBar createMenuBar() {
	return new CompileMenuBar();
    }

    protected void createDisplayItems() {
	super.createDisplayItems();
	errorMessage = new CodeArea(4, 80);
	errorMessage.setBackground(getBackground());
	errorMessage.setEditable(false);
	listingArea.add(SOUTH, errorMessage);
    }

    void start() {
	start(javac = new Compile(startup, compileUpdated));
    }

    public void start(Command cmd) {
	if (((Compile)cmd).fileNames().trim().length() == 0) {
	    addMessage("No source files to be compiled.\n");
	} else {
	    super.start(cmd);
	}
    }

    void restart() {
	restart(javac = new Compile(startup, compileUpdated));
    }

    protected void doCommand(DataInputStream errors) throws IOException {
	    
	while (commandThread != null) {
	    String errorMessage, errorText, errorPosition;
	    if ((errorMessage  = errors.readLine()) == null) return;
	    int first  = errorMessage.indexOf(':');
	    if ((SystemManager.onWin32) &&
		first == 1 && errorMessage.charAt(2) == '\\') {
		char c = errorMessage.charAt(0);
		if ('A' <= c && c <= 'Z' || 'a' <= c && c <= 'z') {
		    first = errorMessage.indexOf(':', 3);
		}
	    }
	    int second = errorMessage.indexOf(':', first + 1);
	    if (first < 0 || second <= first) {
		addMessage(errorMessage + "\n");
		continue;
	    }
	    if ((errorText     = errors.readLine()) == null ||
		(errorPosition = errors.readLine()) == null) {
		return;
	    }

	    String detailMessage = errorMessage  + "\n"
				 + errorText     + "\n"
				 + errorPosition + "\n";

	    String fileName = errorMessage.substring(0, first);
	    int lineNumber = 0;
	    try {
		lineNumber =
		   Integer.parseInt(errorMessage.substring(first + 1, second));
	    } catch (NumberFormatException e) {
		addMessage(errorMessage + "\n");
		continue;
	    }
	    int separator = fileName.lastIndexOf(File.separatorChar);
	    if (separator >= 0) {
		errorMessage = errorMessage.substring(separator + 1);
	    }
	    
	    int position = errorPosition.indexOf('^');
	    if (position < 0) position = 0;
	    try {
		CodeLocation item = new CodeLocation(fileName, lineNumber,
						     errorMessage, position, 0);
		addItem(item);
		detail.addElement(detailMessage);
		list.reshape();
	    } catch (CodeLocationException e) {
		addMessage("fail to search location " +
			   fileName + ":" + lineNumber + ", sorry.\n");
	    }
	}
    }

    protected boolean syncAndClearListItems() {
	if (super.syncAndClearListItems()) {
	    detail.setSize(0);
	    return true;
	} else {
	    return false;
	}
    }

    protected boolean select(int newSelection) {
	if (super.select(newSelection)) {
	    if (newSelection == -1) {
		errorMessage.setText("compiled files: " + javac.fileNames());
	    } else {
		errorMessage.setText((String)detail.elementAt(newSelection));
	    }
	    return true;
	}
	return false;
    }

    protected void update() {
	super.update();
	((CompileMenuBar)menuBar).updateExecuteMenuActivity(startup);
    }

    CodePath getExecuteCodePath() {
	return startup;
    }

    protected ListingFrame copyCreate(CodePath cp) {
	return new CompileFrame(cp, "Same elements for: " + cp.printString(),
							     compileUpdated);
    }

    protected static CompileFrame find(CodePath cp) {
	if (frames.containsKey(cp.printString())) {
	    return (CompileFrame)frames.get(cp.printString());
	} else {
	    return null;
	}
    }

    void close() {
	frames.remove(startup.printString());
	super.close();
    }
}
