/*
 * Copyright (c) 1995 PFU Limited.
 *	author Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.sys.*;
import dejava.util.StringSorter;
import java.io.File;
import java.io.IOException;
import java.awt.*;
import dejava.gui.DejavaDialog;

class DejavaFileDialog extends DejavaDialog implements Runnable {

    private Thread chooserThread;
    private FocusDialog chooser;
    private boolean chooserListup = false;

    protected static final String CHOOSE = "choose";
    private static final String FILTER = "filter";
    private static final String DIRS   = "dirs";
    private static final String FILES  = "files";
    private static final String CLOSE  = "close";

    protected final TextField path = new TextField(40);
    protected final Button choose = new DejavaButton(CHOOSE);
    private TextField filter;
    private Button filterSet;
    private Panel dirsArea;
    private Panel filesArea;
    private DejavaList dirs;
    private DejavaList files;
    private Button close;
    private int lastDirSelected = -1;
    private int lastFileSelected = -1;

    private static final String CURRENT = "." + File.separator;
    private static final String PARENT = ".." + File.separator;
    private String currentDir  = System.getProperty("user.home");
    private String currentFile = "";
    private String startFile   = "";

    private String chosenFile;

    public DejavaFileDialog(Frame f, String title) {
	this(f, title, true);
    }

    public DejavaFileDialog(Frame f, String title, boolean withDirectory) {
	super(f, title);
	chooser = dejavaFileChooser(f, withDirectory);
	createDisplayItems();
	setDirectory(null);
    }

    public void createDisplayItems() {
	Panel fileArea = new Panel();
	fileArea.setLayout(new BorderLayout());
	fileArea.add(CENTER, path);
	fileArea.add(EAST, choose);
	add(CENTER, fileArea);
    }

    protected String getPath() {
	String path = this.path.getText();
	int top = 1;	// for root directory
	if (SystemManager.onWin32) {
	    top = path.indexOf(':') + 2;
	    if (top < 2) top = 1;
	}
	if (path.length() > top) {
	    while (path.endsWith(File.separator)) {
		path = path.substring(0, path.lastIndexOf(File.separator));
	    }
	}
	return path;
    }

    protected DejavaFileChooser dejavaFileChooser(Frame f, boolean withDirectory) {

	//if (withDirectory) {	// if you want to use FileDialog
	//    return null;	// out this comment
	//} else {		// but it seems do not work correctly
	    DejavaFileChooser chooser = new DejavaFileChooser(f, this);

	    Panel filterArea = new Panel();
	    filterArea.setLayout(new BorderLayout());
	    filterArea.add(WEST, new Label("Postfix:"));
	    filterArea.add(CENTER, filter = new TextField());
	    filterArea.add(EAST, filterSet = new DejavaButton("filter"));
	    filter.setEditable(true);
	    filter.enable();

	    int listLines = DejavaProperties.getInteger("attribute.files.lines");
	    if (withDirectory) {
		dirs = new DejavaList(listLines, false);
		dirsArea = new Panel();
		dirsArea.setLayout(new BorderLayout());
		dirsArea.add(NORTH, new Label("Directories:"));
		dirsArea.add(CENTER, dirs.makePanel());
		chooser.add(WEST, dirsArea);
	    }
	    files = new DejavaList(listLines, false);
	    filesArea = new Panel();
	    filesArea.setLayout(new BorderLayout());
	    filesArea.add(NORTH, new Label("Files:"));
	    filesArea.add(CENTER, files.makePanel());

	    Panel buttonArea = new Panel();
	    buttonArea.add(CLOSE, close = new DejavaButton(CLOSE));

	    chooser.add(NORTH, filterArea);
	    chooser.add(CENTER, filesArea);
	    chooser.add(SOUTH, buttonArea);
	    chooser.pack();
	    chooser.hide();
	    return chooser;
	//}
    }

    public void hide() {
	if (chooser != null && chooser.isShowing()) {
	    chooser.hide();
	    chooserListup = false;
	}
	super.hide();
    }

    public void setDirectory(String dir) {
	select(dir);
    }

    public void setFile(String file) {
	startFile = file;
	currentFile = "";
	select(currentDir + file);
    }

    public void setPostFix(String postFix) {
	filter.setText(postFix);
	select(null);
    }

    public String request() {
	if (!confirm()) {
	    return null;
	}
	String fname = path.getText().trim();
	if (fname.length() == 0) {
	    return null;
	}
	int last = fname.length() - 1;
	if (last > 0 && fname.charAt(last) == File.separatorChar) {
	    return fname.substring(0, last);
	}
	return fname;
    }

    public String request(boolean showChooser) {
	if (showChooser && chooser != null) chooser.show();
	if (!confirm()) {
	    return null;
	}
	String fname = path.getText().trim();
	if (fname.length() == 0) {
	    return null;
	}
	int last = fname.length() - 1;
	if (last > 0 && fname.charAt(last) == File.separatorChar) {
	    return fname.substring(0, last);
	}
	return fname;
    }

    public boolean handleEvent(Event ev) {
	if (ev.target == dirs) {
	    String name = dirs.getSelectedItem();
	    int selectedIndex = dirs.getSelectedIndex();
	    if (ev.id == Event.LIST_DESELECT ||
		selectedIndex == lastDirSelected ||
		name.equals(PARENT)) {
		int sep = currentDir.length() - 2;
		int drive = (SystemManager.onWin32) ?
			    currentDir.indexOf(":") : -1;
		if (sep > drive &&
		    (sep = currentDir.lastIndexOf(File.separator, sep)) >= 0) {
		    select(currentDir.substring(0, sep + 1));
		} else {
		    select(null);	// it is a top directory
		}
	    } else if (name.equals(CURRENT)) {
		select(null);	// redisplay current directory
	    } else {
		select(currentDir + name);
	    }
	    return true;
	} else if (ev.target == files) {
	    int selectedIndex = files.getSelectedIndex();
	    if (ev.id == Event.LIST_SELECT && selectedIndex != lastFileSelected) {
		select(currentDir + ((List)ev.target).getSelectedItem());
	    } else {
		select(currentDir + startFile);
	    }
	    return true;
	}
	return super.handleEvent(ev);
    }

    public void run() {
	if (Thread.currentThread() == chooserThread) {
	    if (chooser != null) {
		// no use FileDIalog
		chooser.show();
	    } else {
		// use FileDialog
		String text = path.getText().trim();
		File f = new File(text);
		FileDialog fd = new FileDialog(this, "chooser");
		if (f.isDirectory()) {
		    String dir = f.getPath();
		    if (dir.endsWith(File.separator)) {
			fd.setDirectory(dir.substring(0, dir.lastIndexOf(File.separator)));
		    } else {
			fd.setDirectory(dir);
		    }
		} else {
		    String dir = f.getParent();
		    if (dir != null) fd.setDirectory(dir);
		    String name = f.getName();
		    if (name != null) fd.setFile(name);
		}
		fd.show();
		String dir = fd.getDirectory();
		String name = fd.getFile();
		if (dir == null || dir.length() == 0) {
		    dir = text;
		} else if (!dir.endsWith(File.separator)) {
		    dir += File.separator;
		}
		path.setText((name == null) ? dir : (dir + name));
	    }
	    chooserThread = null;
	}
    }

    public boolean action(Event ev, Object arg) {
	if (ev.target == filter || ev.target == filterSet) {
	    select(null);
	    return false;
	} else if (ev.target == path) {
	    String name = path.getText().trim();
	    int tail = name.length() - 1;
	    if (tail < 0) {
		select(null);
	    } else if (chooserListup) {
		File file = new File(getPath());
		if (file.isDirectory()) {
		    currentFile = "";
		    setDirectory(getPath());
		} else {
		    setDirectory(file.getParent());
		    setFile(file.getName());
		}
	    } else {
		chosenFile = name;
		actionLeftButton();
	    }
	    return false;
	} else if (ev.target == choose) {
	    if (!chooserListup) {
		chooserThread = new Thread(this);
		chooserThread.start();
	    }
	    chooserListup = true;
	    File file = new File(getPath());
	    if (file.isDirectory()) {
		currentFile = "";
		setDirectory(getPath());
	    } else {
		setDirectory(file.getParent());
		setFile(file.getName());
	    }
	    return true;
	} else if (ev.target == close) {
	    chooser.hide();
	    chooserListup = false;
	    return true;
	}
	return super.action(ev, arg);
    }

    private synchronized void select(String name) {
	lastDirSelected  = -1;
	lastFileSelected = -1;
	if (name == null || name.length() == 0) {
	    name = currentDir;
	}
	File f = new File(name);
	boolean isDirectory = f.isDirectory();
	if (isDirectory) {
	    if (!name.endsWith(File.separator)) name += File.separator;
	    currentDir = name;
	    if (chooserListup) {
		if (dirs != null) {
		    dirs.clear();
		    dirs.addItem(CURRENT);
		    dirs.addItem(PARENT);
		}
		files.clear();
		String filterString = filter.getText(); 
		int wild = filterString.lastIndexOf('*');
		if (wild >= 0) filterString = filterString.substring(wild + 1);
		String list[] = StringSorter.sort(f.list());
		if (list != null)  {
		    for (int i = 0; i < list.length; i++) {
			String nm = list[i];
			f = new File(currentDir + nm);
			if (f.isDirectory()) {
			    if (dirs != null) dirs.addItem(nm + File.separator);
			} else if (f.isFile() && nm.endsWith(filterString)) {
			    files.addItem(nm);
			}
		    }
		}
		if (SystemManager.onSolaris)  {
		    Dimension d = files.size();
		    files.resize(d.width, d.height + 1);
		    files.resize(d.width, d.height);
		    if (dirs != null) {
			d = dirs.size();
			dirs.resize(d.width, d.height + 1);
			dirs.resize(d.width, d.height);
		    }
		}
	    }
	} else {
	    String nm = f.getName();
	    if (nm.length() > 0) {
		currentFile = f.getName();
	    }
	    if (chooserListup) {
		for (int i = files.countItems(); --i >= 0; ) {
		    if (files.getItem(i).equals(currentFile)) {
			if (!files.isSelected(i)) {
			    if (i + 3 < files.countItems()) {
				files.makeVisible(i + 3);
			    } else {
				files.makeVisible(i);
			    }
			    files.select(i);
			    lastFileSelected = i;
			}
			break;
		    }
		}
	    }
	}
	if (dirs == null) {
	    path.setText(currentFile);
	} else {
	    path.setText(currentDir + currentFile);
	}
    }

    public static
    String request(Frame frame, String title, String dir, String file) {
	DejavaFileDialog dialog =  new DejavaFileDialog(frame, title);
	dialog.setDirectory(dir);
	dialog.setFile(file);
	String path = dialog.request();
	dialog.dispose();
	return path;
    }

    public static
    String requestName(Frame frame, String title, String directory, String file) {

	DejavaFileDialog dialog = new DejavaFileDialog(frame, title, false);
	int postfix = file.lastIndexOf('.');
	if (postfix >= 0) {
	    dialog.setPostFix(file.substring(postfix + 1));
	}
	dialog.setDirectory(directory);
	dialog.setFile(file);
	String name = dialog.request();
	dialog.dispose();
	return name;
    }
}

class DejavaFileChooser extends FocusDialog {
    DejavaFileDialog fileDialog;

    public DejavaFileChooser(Frame f, DejavaFileDialog d) {
	super(f, false);
	setFont(d.getFont());
	fileDialog = d;
    }

    public void show() {
	super.show();
	Rectangle parent = fileDialog.bounds();
	Dimension me = size();
	resize(parent.width, me.height);
	move(parent.x, parent.y + parent.height);
	requestFocus();
    }

    public boolean handleEvent(Event ev) {
	return fileDialog.handleEvent(ev);
    }
}
