/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.sys.*;
import dejava.lang.*;
import java.util.*;
import java.io.InputStream;
import java.awt.*;

public class DejavaFrame extends DelegateEventFrame implements EventHandler {

    private static Vector dejavaFrames = new Vector();
    private static DejavaFrame currentDejavaFrame = null;

    protected CodePath	startup;
    protected CodePath	current;

    protected DejavaMenuBar menuBar;
    private TextField	  codePathField;
    private Button	  acceptButton;
    private Button	  resetButton;
    private Button	  syncButton;
    protected CodeView	  codeView;

    private boolean	noUpdateCode = false;

    // strings for BorderLayout
    protected final static String NORTH  = "North";
    protected final static String WEST   = "West";
    protected final static String SOUTH  = "South";
    protected final static String EAST   = "East";
    protected final static String CENTER = "Center";

    private boolean codePathEditable;

    public DejavaFrame(CodePath cp, String title) {
	this(cp, title, false);
    }

    public DejavaFrame(CodePath cp, String title, boolean editable) {
	super(title);
	setFont(DejavaProperties.getFont("font.default"));
	setBackground(Color.lightGray);

	startup = cp;
	current = new CodePath(cp);
	codePathEditable = editable;

	menuBar = createMenuBar();
	setMenuBar(menuBar);
	createDisplayItems();
	menuBar.setEditArea(codeView);
	pack();

	dejavaFrames.addElement(this);
	Transcript.getTranscriptMenuBar().addFrame(this);
    }

    protected DejavaMenuBar createMenuBar() {
	return new DejavaMenuBar();
    }

    protected void createDisplayItems() {
	codePathField = new TextField();
	codePathField.setFont(DejavaProperties.getFont("font.list"));
	codePathField.setText(current.printString());
	codePathField.setEditable(codePathEditable);
	if (codePathEditable) {
	    codePathField.setBackground(new Color(135, 135, 135));
	} else {
	    codePathField.setBackground(Color.lightGray);
	}

	codeView = new CodeView(23, 81); // Why? I'd like to set 24,80
	codeView.setEditable(false);

	Panel buttons = new Panel();
	buttons.setLayout(new FlowLayout(FlowLayout.CENTER, 2, 5));
	if (SystemManager.onSolaris) {
	    Label blink = new Label("");
	    codeView.setBlink(blink);
	    buttons.add(blink);
	}
	buttons.add(acceptButton = new DejavaButton("Accept"));
	buttons.add(  syncButton = new DejavaButton("Sync"  ));
	buttons.add( resetButton = new DejavaButton("Reset" ));

	Panel operation = new Panel();
	operation.setLayout(new BorderLayout());
	operation.add(CENTER,  codePathField);
	operation.add(EAST, buttons);

	Panel code = new Panel();
	code.setLayout(new BorderLayout());
	code.add(CENTER, codeView);

	Panel dejava = new Panel();
	dejava.setLayout(new BorderLayout());
	dejava.add(NORTH, operation);
	dejava.add(CENTER, code);

	add(CENTER, dejava);
    }

    /**
     * update all of this frame
     */
    public static final void updateNotify(CodePath cp) {
	Enumeration en = dejavaFrames.elements();
	while (en.hasMoreElements()) {
	    DejavaFrame frame = (DejavaFrame)en.nextElement();
	    if (frame == currentDejavaFrame) {
		frame.updateExceptCode();
	    } else if (cp.hasInfluenceOn(frame.current)) {
		if (cp.matches(frame.current) || frame.current.classManager() == null) {
		    frame.update();
		} else {
		    frame.updateExceptCode();
		}
	    }
	}
    }

    protected final void updateExceptCode() {
	noUpdateCode = true;
	update();
	noUpdateCode = false;
    }

    /**
     * update all of this frame
     */
    protected void update() {
	updateCurrent();
    }

    /**
     * update CurrentMonitor, Menu and CodeView
     */
    protected final void updateCurrent() {
	String codePathString = current.printString();
	if (!codePathString.equals(codePathField.getText())) {
	    codePathField.setText(codePathString);
	}
	if (!noUpdateCode) {
	    updateCode();
	}
	updateButtonActivity();
	menuBar.updateActivity(this, current);
    }

    /**
     * update CodeView, CurrentMonitor and Menus activity
     */
    protected void updateCode() {
	Manager manager = current.getManager();
	String code = manager.getCode();
	setCode(code, manager.isEditable());
    }

    protected void updateButtonActivity() {
	updateButtonActivity(codeView.isEditable(), codeView.isEditable());
    }

    protected final void updateButtonActivity(boolean acceptable, boolean resetable) {
	if (acceptable) {
	    acceptButton.enable();
	} else {
	    acceptButton.disable();
	}
	if (resetable) {
	    resetButton.enable();
	} else {
	    resetButton.disable();
	}
	if (acceptable && current.getManager().isChanged()) {
	    syncButton.enable();
	} else {
	    syncButton.disable();
	}
    }

    /**
     * get code from CodeView
     */
    protected final String getCode() {
	return codeView.getCode();
    }

    /**
     * get code from CodeView
     */
    protected final String getSelectedCode() {
	return codeView.getSelectedText();
    }

    /**
     * get code from CodeView
     */
    protected final void replaceSelectedCode(String code) {
	codeView.replaceText(code, codeView.getSelectionStart(),
				   codeView.getSelectionEnd());
    }

    /**
     * set the code into CodeView
     */
    protected final void setCode(String code) {
	setCode(code, false);
    }

    /**
     * set the code into CodeView
     */
    protected final void setCode(String code, boolean editable) {
	boolean codeSameAsView = code.equals(codeView.getCode());
	if (!codeView.isChanged()) {
	    if (codeSameAsView) {
		codeView.setEditable(editable);
		return;
	    }
	} else {
	    if (codeSameAsView) {
		codeView.setEditable(editable);
		codeView.replaceOriginal(code);
		return;
	    }
	    if (!reset("This `" + current.printShortString() +
			      " has been changed, too.\nForce reset it?")) {
		codeView.replaceOriginal(code);
		return;
	    }
	}
	codeView.setEditable(editable);
	codeView.setCode(code);
	return;
    }

    /**
     * set cursor position to CodeView
     */
    final void setCursorPos(int pos) {
	codeView.select(pos, pos);
    }

    final void setCursorPos(int beginning, int end) {
	codeView.select(beginning, end);
    }

    public boolean handleEvent(Event e) {
	if (e.id == Event.WINDOW_DESTROY) {
	    close();
	    return true;
	}
	return super.handleEvent(e);
    }

    public final boolean mouseEnter(Event ev, int x, int y) {
	if (SystemManager.onSolaris) {
	    Component comp = locate(x, y);
	    if (comp != null) {
		while (comp instanceof Container) {
		    Point location = comp.location();
		    x -= location.x;
		    y -= location.y;
		    Component inside = comp.locate(x, y);
		    if (inside == null || inside == comp) break;
		    comp = inside;
		}
		comp.requestFocus();
		return true;
	    }
	}
	return super.mouseEnter(ev, x, y);
    }

    /**
     * action handler
     */
    public boolean action(Event ev, Object arg) {
	try {
	    if (ev.target instanceof MenuItem) {
		menuBar.action(ev, arg, this, current);
	    } else if (ev.target == codePathField) {
		String codePathString = codePathField.getText();
		if (codePathString.length() == 0) {
		    codePathField.setText(current.printString());
		    return false;
		}
		String codePathNames[] = CodePath.pathToNames(codePathString);
		CodePath cp = CodePath.complete(codePathNames);
		boolean uncomplete = (cp == null);
		if (uncomplete) cp = new CodePath(codePathNames);
		if (cp.matches(current)) {
		    codePathField.setText(current.printString());
		} else {
		    if (!acceptWithConfirm()) {
			codePathField.setText(current.printString());
			return false;
		    }
		    current = cp;
		    update();
		}
		if (uncomplete) {
		    SystemManager.beep();
		    codePathField.setText(CodePath.printString(codePathNames));
		}
		return false;
	    } else if (ev.target == acceptButton) {
		accept();
		return true;
	    } else if (ev.target == resetButton) {
		reset();
		return true;
	    } else if (ev.target == syncButton) {
		sync();
		return true;
	    }
	} catch (DejavaException e) {
	    doReset();
	    current.classManager((ClassManager)null);
	    DejavaFrame.updateNotify(current);
	    return true;
	}
	return super.action(ev, arg);
    }

    /**
     * accept code
     */
    protected boolean acceptWithConfirm() {
	if (!codeView.isChanged()) {
	    return true;
	} else {
	    int chosen = (new DejavaDialog(this,
				"'" + current.printShortString() +
				"' has been changed.\n" +
				"Accept it ? Or reset it?",
				null, "Accept", "Reset", "Cancel")).choose();
	    if (chosen == 3) {	// Cancel
		return false;
	    } else if (chosen == 2) {
		doReset();
		return true;
	    } else {
		return doAccept();
	    }
	}
    }

    /**
     * accept code
     */
    protected boolean accept() {
	if (!codeView.isChanged()) {
	    return true;
	} else {
	    return doAccept();
	}
    }

    /**
     * accept code
     *     return same CodePath, if success to accept as replaced
     *     return different CodePath, if success to accept as created new class/method
     *     return null, if fail to accept
     */
    protected final boolean doAccept() {

	String code = codeView.getCode();

	Manager oldManager = current.getManager();
	Manager newManager;
	try {
	    newManager = oldManager.setCode(code);
	} catch (DejavaException e) {
	    DejavaDialog.information(this, e.getMessage());
	    return false;
	}
	if (newManager == null) return false;	// fail to accept

	// success to accept
	codeView.replaceOriginal(code);

	if (newManager != oldManager) {
	    current.setManager(newManager);
	}

	DejavaFrame.updateNotify(current);
	return true;
    }

    /**
     * reset code
     */
    protected boolean reset() {
	return reset("`" + current.printShortString() +
		     "' has been changed. Force reset it?");
    }

    /**
     * reset code
     */
    protected final boolean reset(String confirmMessage) {
	if (!codeView.isChanged()) {
	    return true;
	} else if (confirmMessage != null &&
		   !confirm(confirmMessage, "Reset")) {
	    return false;
	} else {
	    doReset();
	    return true;
	}
    }

    /**
     * reset code
     */
    protected final void doReset() {
	codeView.resetCode();
    }

    /**
     * sync code
     */
    protected final void sync() {
	sync(current);
    }

    /**
     * sync code
     */
    void sync(CodePath cp) {
	String saveLabel = syncButton.getLabel();
	//syncButton.setLabel("...");
	cp.getManager().sync();
	//syncButton.setLabel(saveLabel);
	DejavaFrame.updateNotify(cp);
    }

    void rebuild(CodePath cp) {
	String names[] = current.names();
	if (cp.packageManager() != null) {
	    cp.packageManager().rebuild();
	} else if (cp.projectManager() != null) {
	    cp.projectManager().rebuild();
	}
	current = new CodePath(names);	// for keep previous selection
	DejavaFrame.updateNotify(cp);
    }

    /**
     * compile
     */
    void compile(CodePath cp, boolean updated) {
	if (!acceptWithConfirm()) {
	    SystemManager.information("Compilation on " + cp.printString() + " aborted.");
	} else {
	    cp.getManager().sync();
	    CompileFrame.openOn(cp, updated);
	}
    }

    /**
     * grep
     */
    final void grep(CodePath cp) {
	String pattern = codeView.getSelectedText();
	if (pattern.length() == 0) pattern = request("egrep pattern");
	try {
	    grep(cp, pattern, null);
	} catch (DejavaException e) {
	    DejavaDialog.information(this, "grep: " + e.getMessage());
	}
    }

    static final void grep(CodePath cp, String pattern, String title)
    throws DejavaException {
	if (pattern == null || pattern.length() == 0) {
	    return;
	}
	GrepFrame.openOn(cp, pattern, title);
    }

    /**
     * spawn browser
     */
    void spawn(CodePath cp) {
	Browser.openOn(cp, current);
    }

    /**
     * spawn hierarchy browser
     */
    final void spawnHierarchy(CodePath cp) {
	CodePath listCp = new CodePath(cp);
	if (cp.methodManager() != null) {
	    listCp = cp.classCodePath();
	} else {
	    listCp = cp.packageCodePath();
	}
	HierarchyBrowser.openOn(listCp, cp,
			      "HierarchyBrowser on: " + cp.printShortString());
    }

    /**
     * spawn implementers
     */
    final void spawnImplementers(CodePath cp) {
	spawnImplementers(cp, "Implementers of: " + cp.printString());
    }

    /**
     * spawn implementers
     */
    final void spawnImplementers(CodePath cp, String title) {
	String regexp;
	try {
	    if (cp.methodManager() == null) {
		regexp = "implements[ \t]*([a-zA-Z][a-zA-Z0-9]*,[ \t]*)*" + cp.className();
		grep(cp.projectCodePath(), regexp, title);
	    } else {
		ImplementersFrame.openOn(cp.methodManager());
	    }
	} catch (DejavaException e) {
	    DejavaDialog.information(this, "grep: " + e.getMessage());
	}
    }

    /**
     * spawn implementers
     */
    final void spawnImplementers(CodePath cp, IdentifierSpec idspec, String label) {
	try {
	    if (cp.methodManager() == null) {
		// not implemented yet
		String regexp = "implements[ \t]*([a-zA-Z][a-zA-Z0-9]*,[ \t]*)*" + cp.className();
		grep(cp.projectCodePath(), regexp, "Implementers of: " + label);
	    } else {
		ImplementersFrame.openOn(cp, idspec, label);
	    }
	} catch (DejavaException e) {
	    DejavaDialog.information(this, "grep: " + e.getMessage());
	}
    }

    /**
     * spawn senders
     */
    final void spawnSenders(CodePath cp) {
	try {
	    if (cp.methodManager() == null) {
		String regexp = "[^a-zA-Z_0-9]" + cp.className() + "[^a-zA-Z_0-9]";
		grep(cp.projectCodePath(),  regexp, "Senders of: " + cp.className());
	    } else {
		SendersFrame.openOn(cp);
	    }
	} catch (DejavaException e) {
	    DejavaDialog.information(this, "grep: " + e.getMessage());
	}
    }

    /**
     * spawn messages
     */
    final void spawnMessages(CodePath cp, String methodName) {
	try {
	    grep(cp, methodName, "Implementers of: " + methodName);
	} catch (DejavaException e) {
	    DejavaDialog.information(this, "grep: " + e.getMessage());
	}
    }

    /**
     * spawn users
     */
    final void spawnUsers(CodePath cp) {
	try {
	    String title = "Users of: " + cp.printString();
	    if (cp.methodManager() == null) {
		String regexp = "[^a-zA-Z0-9]" + cp.className() + "[^a-zA-Z0-9]";
		grep(cp.projectCodePath(), regexp, title);
	    } else {
		spawnImplementers(cp, title);
	    }
	} catch (DejavaException e) {
	    DejavaDialog.information(this, "grep: " + e.getMessage());
	}
    }

    protected Object clone() {
	try {
	    return super.clone();
	} catch (CloneNotSupportedException e) {
	    return null;
	}
    }

    void open() {
	show();
	update();
    }

    void close() {
	if (acceptWithConfirm()) {
	    menuBar.close();
	    int index = dejavaFrames.indexOf(this);
	    dejavaFrames.removeElementAt(index);
	    Transcript.getTranscriptMenuBar().removeFrameAt(index);
	    hide();
	}
    }

    final boolean tryClose() {
	close();
	return !isVisible();
    }

    final static Enumeration allFrames() {
	return dejavaFrames.elements();
    }

    /**
     * request user input for String
     */
    protected final String request(String message) {
	show();
	return DejavaDialog.request(this, message, "");
    }

    /**
     * request user input for String
     */
    protected final String request(String message, String def) {
	show();
	return DejavaDialog.request(this, message, def);
    }

    /**
     * confirm Ok or Cancel
     */
    protected final boolean confirm(String message) {
	show();
	return DejavaDialog.confirm(this, message);
    }

    /**
     * confirm <label> or Cancel
     */
    protected final boolean confirm(String message, String label) {
	show();
	return DejavaDialog.confirm(this, message, label, DejavaDialog.CANCEL);
    }

    static final Frame getCurrentFrame() {
        return currentDejavaFrame;
    }
}
